/*
 * send.cc: methods for sending netmessages
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include "config.h"
#include "defs.h"
#include "netmessage.h"

int netmessage::sendHeader(transport* socket, uint32 opcode, uint32 commandID = 0)
{
    unsigned char header[HEADER_SIZE];
    uint32 headerFragment;

    headerFragment = htonl(opcode);
    memcpy(header, &headerFragment, 4);
    headerFragment = 0; // reserved for future use
    memcpy(header + 4, &headerFragment, 4);
    headerFragment = htonl(sessionID);
    memcpy(header + 8, &headerFragment, 4);
    headerFragment = htonl(commandID);
    memcpy(header + 12, &headerFragment, 4);
    memset(header + 16, 0, 16);

    return socket->send(&header[0], HEADER_SIZE);
}

int netmessage::send(transport* socket, uint32 opcode)
{
    int rv;

    if (transmitting == false) 
    { 
        currMember = members; 
        transmitting = true;
    }
        
    while (currMember != NULL)
    {
        switch (currMember->dataType)
        {
            case nmMemberList::headerType:
            {
                prepareToSend();
                rv = sendHeader(socket, opcode, commandID);
                break;
            }

            case nmMemberList::charstarType:
            case nmMemberList::unsignedcharstarType:
            case nmMemberList::binaryType:
            {
                uint32 tmpLength;
                char* tmp; 

                tmp = *(char**) currMember->data;
                
                tmpLength = htonl(currMember->length);
                rv = socket->send(&tmpLength, 4);
                rv = socket->send(tmp, currMember->length);
                break;
            }
            
            case nmMemberList::charType:
            case nmMemberList::unsignedcharType:
            case nmMemberList::boolType:
            {
                uint32 tmpLength;
                
                tmpLength = htonl(currMember->length);
                
                rv = socket->send(&tmpLength, 4);
                rv = socket->send(currMember->data, 1);
                break;
            }

            case nmMemberList::shortType:
            {
                memcpy(&tmpShort, currMember->data, SIZEOF_SHORT);
                tmpShort = htons(tmpShort);
                rv = socket->send(&tmpShort, SIZEOF_SHORT);
                break;
            }
 
            case nmMemberList::unsignedshortType:
            {
                memcpy(&tmpUShort, currMember->data, SIZEOF_SHORT);
                tmpShort = htons(tmpUShort);
                rv = socket->send(&tmpUShort, SIZEOF_SHORT);
                break;
            }

            case nmMemberList::intType:
            {
                memcpy(&tmpInt, currMember->data, SIZEOF_INT);
                tmpInt = htonl(tmpInt);
                rv = socket->send(&tmpInt, SIZEOF_INT);
                break;
            }
            
            case nmMemberList::unsignedintType:
            {
                memcpy(&tmpUInt, currMember->data, SIZEOF_INT);
                tmpInt = htonl(tmpUInt);
                rv = socket->send(&tmpUInt, SIZEOF_INT);
                break;
            }
            
            case nmMemberList::longType:
            {
                memcpy(&tmpLong, currMember->data, SIZEOF_LONG);
                tmpLong = htonl(tmpLong);
                rv = socket->send(&tmpLong, SIZEOF_LONG);
                break;
            }
            
            case nmMemberList::unsignedlongType:
            {
                memcpy(&tmpULong, currMember->data, SIZEOF_LONG);
                tmpULong = htonl(tmpULong);
                rv = socket->send(&tmpULong, SIZEOF_LONG);
                break;
            }

            case nmMemberList::longlongType:
            {
                memcpy(&tmpLLong, currMember->data, SIZEOF_LONG_LONG);
#ifndef WORDS_BIGENDIAN
                tmpLLong = bswap(tmpLLong);
#endif
                rv = socket->send(&tmpLLong, SIZEOF_LONG_LONG);
                break;
            }

            case nmMemberList::unsignedlonglongType:
            {
                memcpy(&tmpULLong, currMember->data, SIZEOF_LONG_LONG);
#ifndef WORDS_BIGENDIAN
                tmpULLong = bswap(tmpULLong);
#endif
                rv = socket->send(&tmpULLong, SIZEOF_LONG_LONG);
                break;
            }

            case nmMemberList::floatType:
            {
                memcpy(&tmpFloat, currMember->data, SIZEOF_FLOAT);
#ifndef WORDS_BIGENDIAN
                tmpFloat = bswap(tmpFloat);
#endif
                rv = socket->send(&tmpFloat, SIZEOF_FLOAT);
                break;
            }

            case nmMemberList::doubleType:
            {
                memcpy(&tmpDouble, currMember->data, SIZEOF_DOUBLE);
#ifndef WORDS_BIGENDIAN
                tmpDouble = bswap(tmpDouble);
#endif
                rv = socket->send(&tmpDouble, SIZEOF_DOUBLE);
                break;
            }

            case nmMemberList::fileType:
            {
                fileByteCount = currMember->length;

                // get the file path
                filepath = *(char **) currMember->data;

                rv = sendFile();
                break;
            }
        }

        if (rv < 0)
        {
            return rv;
        }

        currMember = currMember->nextMember;
    }

    if (currMember == NULL)
    {
        transmitting = false;
        return 0;
    }
    return 1;
}
