/*
 * netmessage.h: sends and receives data across a network connection.
 *
 * Designed as a virtual class, objects that inherit from this class will be
 * used as the arguments to the server object on the client.  Since this
 * object knows how to serialize the information for its object, it can use
 * the TCP stream to recreate the object in the other process.
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#ifndef __netmessage_h_
#define __netmessage_h_

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <string>
#include <limits.h>

#include "config.h"
#include "defs.h"
#include "commid.h"
#include "msgs.h"
#include "transport.h"
#include "nmMemberList.h"

class netmessage
{
    public:
        netmessage(transport* connection, uint32 session_id, uint32 command_id);
        virtual ~netmessage(void);
    
        virtual void prepareToSend(void) = 0;
        virtual int activate(void) = 0;
    
        int receive(void);

        int send(transport* socket, uint32 opcode);
        int sendHeader(transport* socket, uint32 opcode, uint32 commandID = 0);
    
        unsigned long getSessionID() { return sessionID; }
        int memberCount() { return members->count(); }

        // exceptions that are thrown
        class nmTempFileError { };
        class nmNameLengthError { };
        class nmFileCreationError { };

    protected:
        uint32 sessionID;
        uint32 commandID;

        transport* socket;
        nmMemberList* members;
    
    private:
        int bytesreqd;              // amount of data required for this object
        int bytesrec;               // amount read so far
        int maxBuffer;              // length of the buffer
    
        // receive state information
        nmMemberList *currMember;
        uint32 amtReceived;
        uint32 dynamicLength;
        bool gotDynamicLength;
        bool transmitting;
    
        // state information for temporary files
        char* filepath;
        int filedesc;
        uint32 fileByteCount;
        bool gotFileHeader;
        bool gotfileopen;
    
        // determine byteorder
        long long bswap(long long sll);
        unsigned long long bswap(unsigned long long ull);
        float bswap(float fv);
        double bswap(double dv);
        void bswap(unsigned int size, char *in, char *out);
        
        /* methods for reading data from the socket */
        int getData(void* buffer, int length);
        int getDynamicData(char** dataPointer);
        int getBinaryData(char** dataPointer);

        int sendFile();

        // overloaded data decoding methods
        int recvMember(char& c);
        int recvMember(unsigned char& c);
        int recvMember(bool& b);
        int recvMember(short int& i);
        int recvMember(unsigned short& i);
        int recvMember(int& i);
        int recvMember(unsigned int& i);
        int recvMember(long& i);
        int recvMember(unsigned long& i);
        int recvMember(long long int& i);
        int recvMember(unsigned long long int& i);
        int recvMember(float& f);
        int recvMember(double& d);
        int recvMember(int& i, char *s); // binary string of length n
        int recvFile(void);
        
        /* temporary variables used in send and receive methods */
        short tmpShort;
        unsigned short tmpUShort;
        int tmpInt;
        unsigned int tmpUInt;
        long tmpLong;
        unsigned long tmpULong;
        long long tmpLLong;
        unsigned long long tmpULLong;
        float tmpFloat;
        double tmpDouble;
};

#endif /* __netmessage_h_ */
