/*
 * database.h: the dual key hash table database
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#ifndef __database_h_
#define __database_h_

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <netinet/in.h>
#include <iostream>
#include <string>
#include <hash_map>
#include <vector>

#include "config.h"
#include "defs.h"

#include "intarray.h"

#define DB_VERSION_MAJOR        0
#define DB_VERSION_MINOR        2

#define DB_HEADER_SIZE          32

#define RECORD_DELETE           0x80000000
#define DELETE_RECORD           0x7fffffff

/* exceptions */
class dbBadFormat {};
class dbBadVersion {};
class dbOutOfSync {};
class dbOpenFail {};
class dbCreateFail {};
class dbReadFail {};
class dbWriteFail{};
class dbStatFail {};
class dbSeekFail {};

struct eqstr
{
    bool operator()(const char* s1, const char* s2) const
    {
        return (strcmp(s1, s2) == 0);
    }
};

class Database
{
    public:
        Database(const char* filename);
        ~Database(void);

        char* read(const char* key1, const char* key2);
        vector<char *>* readAll(const char* key1, const char* key2);
        vector<char *>* readAllKey1(const char* key2);
        vector<char *>* readAllKey2(const char* key1);

        void insert(const char* key1, const char* key2, const char* value);

        void update(const char* key1, const char* key2, const char* value, const char* newvalue);
        void update(const char* key1, const char* key2, const char* newvalue, int pos);

        void remove(const char* key1, const char* key2, const char* value);
        void removeAll(const char* key1, const char* key2);
        void removeKey1(const char* key1);
        void removeKey2(const char* key2);
    
        void defrag(void);
        time_t lastDefrag(void);

        bool hasEntries(void);

    private:

        hash_map<const char*, intarray*, hash<const char*>, eqstr> hash_table1;
        hash_map<const char*, intarray*, hash<const char*>, eqstr> hash_table2;

        char* dbfile;
        int dbfd;
        time_t defragTime;

        /* hash table functions */
        void hashBuild(void);

        void hashAdd(const char* key1, const char* key2, unsigned long pos);
        void hashRemove(const char* key1, const char* key2);
        void hashRemove(const char* key1, const char* key2, unsigned long pos);

        unsigned long hashGet(const char* key1, const char* key2);

        intarray* hashGetAll(const char* key1, const char* key2);

        /* database file functions */
        void fileInit(void);

        unsigned long fileReset(void);

        bool checkOffset(unsigned long offset, const char* key1, const char* key2);

        unsigned long fileSize(void);

        unsigned long fileSeekEmpty(uint32 recordsize);

        void fileDefrag(void);
        void updateDefragTime(int fd);

        char* fileRead(unsigned long pos);
        char* fileReadKey1(unsigned long pos);
        char* fileReadKey2(unsigned long pos);

        int fileReadNext(char*& key1, char*& key2);

        unsigned long fileWrite(unsigned long offset, const char* key1, const char* key2, const char* value);
        unsigned long writeRecord(uint32 recordsize, const char* key1, const char* k2, const char* value);

        void fileRemove(unsigned long pos);
};

#endif /* __database_h_ */
