/*
 * sha.cc: An implementation of SHA-1, a secure hash algorithm.
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include "sha.h"

#define F1(B,C,D)   (D ^ (B & (C ^ D)))         /* rounds  0 - 19 */
#define F2(B,C,D)   (B ^ C ^ D)                 /* rounds 20 - 39 */
#define F3(B,C,D)   ((B & (C | D)) | (C & D))   /* rounds 40 - 59 */
#define F4(B,C,D)   (B ^ C ^ D)                 /* rounds 60 - 79 */

#define K1          0x5a827999L                 /* rounds  0 - 19 */
#define K2          0x6ed9eba1L                 /* rounds 20 - 39 */
#define K3          0x8f1bbcdcL                 /* rounds 40 - 59 */
#define K4          0xca62c1d6L                 /* rounds 60 - 79 */

#define SUBROUND_A(F, K, a, b, c, d, e, f)                      \
    *w  = *(block++) << 24;                                     \
    *w |= *(block++) << 16;                                     \
    *w |= *(block++) <<  8;                                     \
    *w |= *(block++);                                           \
    f = ((a << 5) | (a >> 27)) + F(b, c, d) + e + *(w++) + K;   \
    b = (b << 30) | (b >> 2);

#define SUBROUND_B(F, K, a, b, c, d, e, f)                      \
    f = *(w-3) ^ *(w-8) ^ *(w-14) ^ *(w-16);                    \
    *w = (f << 1) | (f >> 31);                                  \
    f = ((a << 5) | (a >> 27)) + F(b, c, d) + e + *(w++) + K;   \
    b = (b << 30) | (b >> 2);

SHA::SHA(void) {
    ll = hl = 0;
    reset();
}

SHA::~SHA(void) { reset(); }

void SHA::reset(void)
{
    if (ll || hl)
    {
        unsigned char *ptr(buffer);
        int i;

        for (i = 0; i < sizeof(buffer); i++)
            *(ptr++) = '\0';
        ll = hl = 0;
    }
    h0 = 0x67452301L;
    h1 = 0xefcdab89L;
    h2 = 0x98badcfeL;
    h3 = 0x10325476L;
    h4 = 0xc3d2e1f0L;
}

void SHA::transform(const unsigned char *block)
{
    unsigned long a, b, c, d, e, f, *w;
    unsigned long W[SHA_BLOCK_SIZE];
    int i;

    a = h0, b = h1, c = h2, d = h3, e = h4;
    w = W;

    SUBROUND_A(F1, K1, a, b, c, d, e, f);
    SUBROUND_A(F1, K1, f, a, b, c, d, e);
    SUBROUND_A(F1, K1, e, f, a, b, c, d);
    SUBROUND_A(F1, K1, d, e, f, a, b, c);
    SUBROUND_A(F1, K1, c, d, e, f, a, b);
    SUBROUND_A(F1, K1, b, c, d, e, f, a);
    SUBROUND_A(F1, K1, a, b, c, d, e, f);
    SUBROUND_A(F1, K1, f, a, b, c, d, e);
    SUBROUND_A(F1, K1, e, f, a, b, c, d);
    SUBROUND_A(F1, K1, d, e, f, a, b, c);
    SUBROUND_A(F1, K1, c, d, e, f, a, b);
    SUBROUND_A(F1, K1, b, c, d, e, f, a);
    SUBROUND_A(F1, K1, a, b, c, d, e, f);
    SUBROUND_A(F1, K1, f, a, b, c, d, e);
    SUBROUND_A(F1, K1, e, f, a, b, c, d);
    SUBROUND_A(F1, K1, d, e, f, a, b, c); // 15
    SUBROUND_B(F1, K1, c, d, e, f, a, b);
    SUBROUND_B(F1, K1, b, c, d, e, f, a);
    SUBROUND_B(F1, K1, a, b, c, d, e, f);
    SUBROUND_B(F1, K1, f, a, b, c, d, e); // 19
    SUBROUND_B(F2, K2, e, f, a, b, c, d);
    SUBROUND_B(F2, K2, d, e, f, a, b, c);
    SUBROUND_B(F2, K2, c, d, e, f, a, b);
    SUBROUND_B(F2, K2, b, c, d, e, f, a);
    SUBROUND_B(F2, K2, a, b, c, d, e, f);
    SUBROUND_B(F2, K2, f, a, b, c, d, e);
    SUBROUND_B(F2, K2, e, f, a, b, c, d);
    SUBROUND_B(F2, K2, d, e, f, a, b, c);
    SUBROUND_B(F2, K2, c, d, e, f, a, b);
    SUBROUND_B(F2, K2, b, c, d, e, f, a);
    SUBROUND_B(F2, K2, a, b, c, d, e, f);
    SUBROUND_B(F2, K2, f, a, b, c, d, e);
    SUBROUND_B(F2, K2, e, f, a, b, c, d);
    SUBROUND_B(F2, K2, d, e, f, a, b, c);
    SUBROUND_B(F2, K2, c, d, e, f, a, b);
    SUBROUND_B(F2, K2, b, c, d, e, f, a);
    SUBROUND_B(F2, K2, a, b, c, d, e, f);
    SUBROUND_B(F2, K2, f, a, b, c, d, e);
    SUBROUND_B(F2, K2, e, f, a, b, c, d);
    SUBROUND_B(F2, K2, d, e, f, a, b, c); // 39
    SUBROUND_B(F3, K3, c, d, e, f, a, b);
    SUBROUND_B(F3, K3, b, c, d, e, f, a);
    SUBROUND_B(F3, K3, a, b, c, d, e, f);
    SUBROUND_B(F3, K3, f, a, b, c, d, e);
    SUBROUND_B(F3, K3, e, f, a, b, c, d);
    SUBROUND_B(F3, K3, d, e, f, a, b, c);
    SUBROUND_B(F3, K3, c, d, e, f, a, b);
    SUBROUND_B(F3, K3, b, c, d, e, f, a);
    SUBROUND_B(F3, K3, a, b, c, d, e, f);
    SUBROUND_B(F3, K3, f, a, b, c, d, e);
    SUBROUND_B(F3, K3, e, f, a, b, c, d);
    SUBROUND_B(F3, K3, d, e, f, a, b, c); 
    SUBROUND_B(F3, K3, c, d, e, f, a, b);
    SUBROUND_B(F3, K3, b, c, d, e, f, a);
    SUBROUND_B(F3, K3, a, b, c, d, e, f);
    SUBROUND_B(F3, K3, f, a, b, c, d, e);
    SUBROUND_B(F3, K3, e, f, a, b, c, d);
    SUBROUND_B(F3, K3, d, e, f, a, b, c); 
    SUBROUND_B(F3, K3, c, d, e, f, a, b);
    SUBROUND_B(F3, K3, b, c, d, e, f, a); // 59
    SUBROUND_B(F4, K4, a, b, c, d, e, f);
    SUBROUND_B(F4, K4, f, a, b, c, d, e);
    SUBROUND_B(F4, K4, e, f, a, b, c, d);
    SUBROUND_B(F4, K4, d, e, f, a, b, c);
    SUBROUND_B(F4, K4, c, d, e, f, a, b);
    SUBROUND_B(F4, K4, b, c, d, e, f, a);
    SUBROUND_B(F4, K4, a, b, c, d, e, f);
    SUBROUND_B(F4, K4, f, a, b, c, d, e);
    SUBROUND_B(F4, K4, e, f, a, b, c, d);
    SUBROUND_B(F4, K4, d, e, f, a, b, c);
    SUBROUND_B(F4, K4, c, d, e, f, a, b);
    SUBROUND_B(F4, K4, b, c, d, e, f, a);
    SUBROUND_B(F4, K4, a, b, c, d, e, f);
    SUBROUND_B(F4, K4, f, a, b, c, d, e);
    SUBROUND_B(F4, K4, e, f, a, b, c, d);
    SUBROUND_B(F4, K4, d, e, f, a, b, c);
    SUBROUND_B(F4, K4, c, d, e, f, a, b);
    SUBROUND_B(F4, K4, b, c, d, e, f, a);
    SUBROUND_B(F4, K4, a, b, c, d, e, f);
    SUBROUND_B(F4, K4, f, a, b, c, d, e); // 79

    h0 = (h0 + e) & 0xffffffffL;
    h1 = (h1 + f) & 0xffffffffL;
    h2 = (h2 + a) & 0xffffffffL;
    h3 = (h3 + b) & 0xffffffffL;
    h4 = (h4 + c) & 0xffffffffL;
}

void SHA::transform(const unsigned char *block, unsigned char *digest)
{
    transform(block);

    *(digest++) = (h0 >> 24) & 0xff;
    *(digest++) = (h0 >> 16) & 0xff;
    *(digest++) = (h0 >>  8) & 0xff;
    *(digest++) = (h0      ) & 0xff;
    *(digest++) = (h1 >> 24) & 0xff;
    *(digest++) = (h1 >> 16) & 0xff;
    *(digest++) = (h1 >>  8) & 0xff;
    *(digest++) = (h1      ) & 0xff;
    *(digest++) = (h2 >> 24) & 0xff;
    *(digest++) = (h2 >> 16) & 0xff;
    *(digest++) = (h2 >>  8) & 0xff;
    *(digest++) = (h2      ) & 0xff;
    *(digest++) = (h3 >> 24) & 0xff;
    *(digest++) = (h3 >> 16) & 0xff;
    *(digest++) = (h3 >>  8) & 0xff;
    *(digest++) = (h3      ) & 0xff;
    *(digest++) = (h4 >> 24) & 0xff;
    *(digest++) = (h4 >> 16) & 0xff;
    *(digest++) = (h4 >>  8) & 0xff;
    *digest     = (h4      ) & 0xff;
}

void SHA::hash(const unsigned char *data, unsigned char *digest, unsigned long len)
{
    update(data, len);
    final(digest);
}

void SHA::update(const unsigned char *data, unsigned long len)
{
    unsigned long clen, rlen(ll % 64);
    int i;

    if ((ll = ll + len) > ll)
        hl++;

    while (len > 0)
    {
        if (rlen == 0 && len >= 64)
        {
            transform(data);
            data += 64;
            len -= 64;
        }
        else
        {
            if ((clen = 64 - rlen) > 0)
            {
                if (clen > len)
                    clen = len;

                for (i = 0; i < clen; i++, len--)
                    buffer[rlen++] = *(data++);

                if (rlen == 64)
                {
                    transform(buffer);
                    rlen = 0;
                }
            }
        }
    }
}

void SHA::final(unsigned char *digest)
{
    unsigned long rlen(ll % 64), hcount, lcount;
    int i;

    /* calculate bit count */
    hcount = (hl << 3) + (ll >> 29);
    lcount = (ll << 3);

    buffer[rlen++] = 0x80;

    if (rlen > 56)
    {
        for (i = rlen + 1; i < 64; i++)
            buffer[i] = 0;

        rlen = 0;
        transform(buffer);
    }

    for (i = rlen; i < 56; i++)
        buffer[i] = 0;

    buffer[56] = (hcount >> 24) & 0xff;
    buffer[57] = (hcount >> 16) & 0xff;
    buffer[58] = (hcount >>  8) & 0xff;
    buffer[59] = (hcount      ) & 0xff;
    buffer[60] = (lcount >> 24) & 0xff;
    buffer[61] = (lcount >> 16) & 0xff;
    buffer[62] = (lcount >>  8) & 0xff;
    buffer[63] = (lcount      ) & 0xff;

    transform(buffer);

    *(digest++) = (h0 >> 24) & 0xff;
    *(digest++) = (h0 >> 16) & 0xff;
    *(digest++) = (h0 >>  8) & 0xff;
    *(digest++) = (h0      ) & 0xff;
    *(digest++) = (h1 >> 24) & 0xff;
    *(digest++) = (h1 >> 16) & 0xff;
    *(digest++) = (h1 >>  8) & 0xff;
    *(digest++) = (h1      ) & 0xff;
    *(digest++) = (h2 >> 24) & 0xff;
    *(digest++) = (h2 >> 16) & 0xff;
    *(digest++) = (h2 >>  8) & 0xff;
    *(digest++) = (h2      ) & 0xff;
    *(digest++) = (h3 >> 24) & 0xff;
    *(digest++) = (h3 >> 16) & 0xff;
    *(digest++) = (h3 >>  8) & 0xff;
    *(digest++) = (h3      ) & 0xff;
    *(digest++) = (h4 >> 24) & 0xff;
    *(digest++) = (h4 >> 16) & 0xff;
    *(digest++) = (h4 >>  8) & 0xff;
    *digest     = (h4      ) & 0xff;

    reset();
}
