/*
 * random.cc: a pseudo random number generator
 *
 * This code is based (in part) on Peter Gutmann's paper, "Software Generation
 * of Practically Strong Random Numbers."
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include "config.h"
#include "random.h"

CSPRNG::CSPRNG(void)
{
    readpos = writepos = 0;
    mixed = false;
    balance = 0;

    randomPoll(RANDOM_POOL_SIZE);
}

/*
 * this method is only temporary, it will (amongst several other things) use
 * a better random poll in the future.
 */
void CSPRNG::randomPoll(int size)
{
    int temp(balance);

    balance = 0;

    while (balance + temp < size)
        fastPoll();
    balance += temp;
}

unsigned char *CSPRNG::getRandom(int size)
{
    unsigned char *buffer, *ptr;
    int rsize;

    ptr = buffer = new unsigned char[size];

    while (size)
    {
        if (size > RANDOM_POOL_SIZE)
            rsize = RANDOM_POOL_SIZE;
        else rsize = size;

        extractData(ptr, rsize);
        ptr += rsize;
        size -= rsize;
    }
    return buffer; 
}

void CSPRNG::mix(unsigned char *pool)
{
    unsigned char *pend(pool + RANDOM_POOL_SIZE - RANDOM_DIGEST_SIZE);
    unsigned char pond[RANDOM_BLOCK_SIZE];
    unsigned char *p(pond), *start(pool), *end(pool + RANDOM_POOL_SIZE);
    SHA hashfunc;
    int i, j;

    for (i = 0; i < RANDOM_DIGEST_SIZE; i++)
        *(p++) = *(pend++);
    pend = pool;

    for (i = 0; i < RANDOM_POOL_SIZE; i += RANDOM_DIGEST_SIZE)
    {
        if (pend + RANDOM_BLOCK_SIZE - RANDOM_DIGEST_SIZE > end)
        {
            unsigned char *tp(pend);

            for (j = RANDOM_DIGEST_SIZE; j < RANDOM_BLOCK_SIZE; j++)
            {
                if (tp == end)
                    tp = start;
                *(p++) = *(tp++);
            }
        }
        else
        {
            for (j = RANDOM_DIGEST_SIZE; j < RANDOM_BLOCK_SIZE; j++)
                *(p++) = *(pend++);
        }

        hashfunc.transform(pond, pond);
        p = pond;

        for (j = 0; j < RANDOM_DIGEST_SIZE; j++)
            *(pool++) = *(p++);
        pend = &(*pool);
    }
}

void CSPRNG::addRandom(const void *buffer, int size)
{
    const unsigned char *ptr((unsigned char *) buffer);
    unsigned char *pool(&rndpool[writepos]);

    balance = balance + size;

    while (size-- > 0)
    {
        if (writepos == RANDOM_POOL_SIZE)
        {
            mix(rndpool);
            mixed = !(size - 1);
            writepos = 0;
        }
        *(pool++) = *(ptr++);
        writepos++;
    }
}

void CSPRNG::extractData(unsigned char *buffer, int size)
{
    unsigned long *sp((unsigned long *) rndpool);
    unsigned long *dp((unsigned long *) keypool);
    unsigned char *pool(&keypool[readpos]);
    int i;

    if (balance < size)
        randomPoll(size - balance);
    balance = balance - size;

    if (mixed == false)
        mix(rndpool);

    for (i = 0; i < RANDOM_POOL_SIZE; i += SIZEOF_LONG)
        *(dp++) = *(sp++) ^ 0xffffffffL;

    mix(rndpool);
    mix(keypool);

    while (size-- > 0)
    {
        if (readpos == RANDOM_POOL_SIZE)
            readpos = 0;

        *(buffer++) = *(pool++);
        readpos++;
    }
    mixed = false;
}
