/*
 * dh.cc: a simple implementation of Diffie-Hellman key exchange
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include <iostream>
#include "config.h"
#include "bn.h"
#include "germain.h"
#include "random.h"
#include "dh.h"

DH::DH(CSPRNG *random, struct BigNum *prime, struct BigNum *generator)
{
    RNG = random;

    p = new struct BigNum;
    bnBegin(p);
    bnCopy(p, prime);

    g = new struct BigNum;
    bnBegin(g);
    bnCopy(g, generator);

    k = new struct BigNum;
    bnBegin(k);

    K = NULL;

    x = new struct BigNum;
    bnBegin(x);

    init();
}

DH::~DH(void)
{
    if (p != NULL)
    {
        bnEnd(p);
        delete p;
    }

    if (g != NULL)
    {
        bnEnd(g);
        delete g;
    }

    if (k != NULL)
    {
        bnEnd(k);
        delete k;
    }

    if (K != NULL)
    {
        bnEnd(K);
        delete K;
    }

    if (x != NULL)
    {
        bnEnd(x);
        delete x;
    }
}

void DH::init(void)
{
    struct BigNum q, r, n, d;
    unsigned char *buffer;
    int s;

    bnBegin(&q);
    bnBegin(&r);
    bnBegin(&n);
    bnBegin(&d);

    bnSetQ(&d, 2);
    bnCopy(&n, p);
    bnSubQ(&n, 1);

    bnDivMod(&q, &r, &n, &d);

    s = bnBytes(p);
    buffer = RNG->getRandom(s);
    bnInsertLittleBytes(x, buffer, 0, s);
    delete buffer;

    while (bnCmp(&q, x) < 1)
    {
        bnRShift(x, 1);
    }

    bnExpMod(k, g, x, p);

    bnEnd(&q);
    bnEnd(&r);
    bnEnd(&n);
    bnEnd(&d);
}

struct BigNum *DH::publickey(void)
{
    return k;
}

struct BigNum *DH::privatekey(void)
{
    return K;
}

void DH::generate(struct BigNum *f)
{
    K = new struct BigNum;

    bnBegin(K);
    bnExpMod(K, f, x, p);
}
