/*
 * des.cc: algorithms for DES and Triple-DES encryption and decryption.
 *
 * This implementation is derived from the DES implementation by Michael
 * Roth included with GnuPG.
 *
 * Copyright (c) 1999-2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include "des.h" 

/* 
 * The values in this S-box table are permuted according to 'primitive
 * function P' and are rotated one bit to the left.
 */
static unsigned long spbox[8][64] = {
{
    0x01010400, 0x00000000, 0x00010000, 0x01010404,
    0x01010004, 0x00010404, 0x00000004, 0x00010000,
    0x00000400, 0x01010400, 0x01010404, 0x00000400,
    0x01000404, 0x01010004, 0x01000000, 0x00000004,
    0x00000404, 0x01000400, 0x01000400, 0x00010400,
    0x00010400, 0x01010000, 0x01010000, 0x01000404,
    0x00010004, 0x01000004, 0x01000004, 0x00010004,
    0x00000000, 0x00000404, 0x00010404, 0x01000000,
    0x00010000, 0x01010404, 0x00000004, 0x01010000,
    0x01010400, 0x01000000, 0x01000000, 0x00000400,
    0x01010004, 0x00010000, 0x00010400, 0x01000004,
    0x00000400, 0x00000004, 0x01000404, 0x00010404,
    0x01010404, 0x00010004, 0x01010000, 0x01000404,
    0x01000004, 0x00000404, 0x00010404, 0x01010400,
    0x00000404, 0x01000400, 0x01000400, 0x00000000,
    0x00010004, 0x00010400, 0x00000000, 0x01010004
},{
    0x80108020, 0x80008000, 0x00008000, 0x00108020,
    0x00100000, 0x00000020, 0x80100020, 0x80008020,
    0x80000020, 0x80108020, 0x80108000, 0x80000000,
    0x80008000, 0x00100000, 0x00000020, 0x80100020,
    0x00108000, 0x00100020, 0x80008020, 0x00000000,
    0x80000000, 0x00008000, 0x00108020, 0x80100000,
    0x00100020, 0x80000020, 0x00000000, 0x00108000,
    0x00008020, 0x80108000, 0x80100000, 0x00008020,
    0x00000000, 0x00108020, 0x80100020, 0x00100000,
    0x80008020, 0x80100000, 0x80108000, 0x00008000,
    0x80100000, 0x80008000, 0x00000020, 0x80108020,
    0x00108020, 0x00000020, 0x00008000, 0x80000000,
    0x00008020, 0x80108000, 0x00100000, 0x80000020,
    0x00100020, 0x80008020, 0x80000020, 0x00100020,
    0x00108000, 0x00000000, 0x80008000, 0x00008020,
    0x80000000, 0x80100020, 0x80108020, 0x00108000
},{
    0x00000208, 0x08020200, 0x00000000, 0x08020008,
    0x08000200, 0x00000000, 0x00020208, 0x08000200,
    0x00020008, 0x08000008, 0x08000008, 0x00020000,
    0x08020208, 0x00020008, 0x08020000, 0x00000208,
    0x08000000, 0x00000008, 0x08020200, 0x00000200,
    0x00020200, 0x08020000, 0x08020008, 0x00020208,
    0x08000208, 0x00020200, 0x00020000, 0x08000208,
    0x00000008, 0x08020208, 0x00000200, 0x08000000,
    0x08020200, 0x08000000, 0x00020008, 0x00000208,
    0x00020000, 0x08020200, 0x08000200, 0x00000000,
    0x00000200, 0x00020008, 0x08020208, 0x08000200,
    0x08000008, 0x00000200, 0x00000000, 0x08020008,
    0x08000208, 0x00020000, 0x08000000, 0x08020208,
    0x00000008, 0x00020208, 0x00020200, 0x08000008,
    0x08020000, 0x08000208, 0x00000208, 0x08020000,
    0x00020208, 0x00000008, 0x08020008, 0x00020200
},{
    0x00802001, 0x00002081, 0x00002081, 0x00000080,
    0x00802080, 0x00800081, 0x00800001, 0x00002001,
    0x00000000, 0x00802000, 0x00802000, 0x00802081,
    0x00000081, 0x00000000, 0x00800080, 0x00800001,
    0x00000001, 0x00002000, 0x00800000, 0x00802001,
    0x00000080, 0x00800000, 0x00002001, 0x00002080,
    0x00800081, 0x00000001, 0x00002080, 0x00800080,
    0x00002000, 0x00802080, 0x00802081, 0x00000081,
    0x00800080, 0x00800001, 0x00802000, 0x00802081,
    0x00000081, 0x00000000, 0x00000000, 0x00802000,
    0x00002080, 0x00800080, 0x00800081, 0x00000001,
    0x00802001, 0x00002081, 0x00002081, 0x00000080,
    0x00802081, 0x00000081, 0x00000001, 0x00002000,
    0x00800001, 0x00002001, 0x00802080, 0x00800081,
    0x00002001, 0x00002080, 0x00800000, 0x00802001,
    0x00000080, 0x00800000, 0x00002000, 0x00802080
},{
    0x00000100, 0x02080100, 0x02080000, 0x42000100,
    0x00080000, 0x00000100, 0x40000000, 0x02080000,
    0x40080100, 0x00080000, 0x02000100, 0x40080100,
    0x42000100, 0x42080000, 0x00080100, 0x40000000,
    0x02000000, 0x40080000, 0x40080000, 0x00000000,
    0x40000100, 0x42080100, 0x42080100, 0x02000100,
    0x42080000, 0x40000100, 0x00000000, 0x42000000,
    0x02080100, 0x02000000, 0x42000000, 0x00080100,
    0x00080000, 0x42000100, 0x00000100, 0x02000000,
    0x40000000, 0x02080000, 0x42000100, 0x40080100,
    0x02000100, 0x40000000, 0x42080000, 0x02080100,
    0x40080100, 0x00000100, 0x02000000, 0x42080000,
    0x42080100, 0x00080100, 0x42000000, 0x42080100,
    0x02080000, 0x00000000, 0x40080000, 0x42000000,
    0x00080100, 0x02000100, 0x40000100, 0x00080000,
    0x00000000, 0x40080000, 0x02080100, 0x40000100
},{
    0x20000010, 0x20400000, 0x00004000, 0x20404010,
    0x20400000, 0x00000010, 0x20404010, 0x00400000,
    0x20004000, 0x00404010, 0x00400000, 0x20000010,
    0x00400010, 0x20004000, 0x20000000, 0x00004010,
    0x00000000, 0x00400010, 0x20004010, 0x00004000,
    0x00404000, 0x20004010, 0x00000010, 0x20400010,
    0x20400010, 0x00000000, 0x00404010, 0x20404000,
    0x00004010, 0x00404000, 0x20404000, 0x20000000,
    0x20004000, 0x00000010, 0x20400010, 0x00404000,
    0x20404010, 0x00400000, 0x00004010, 0x20000010,
    0x00400000, 0x20004000, 0x20000000, 0x00004010,
    0x20000010, 0x20404010, 0x00404000, 0x20400000,
    0x00404010, 0x20404000, 0x00000000, 0x20400010,
    0x00000010, 0x00004000, 0x20400000, 0x00404010,
    0x00004000, 0x00400010, 0x20004010, 0x00000000,
    0x20404000, 0x20000000, 0x00400010, 0x20004010
},{
    0x00200000, 0x04200002, 0x04000802, 0x00000000,
    0x00000800, 0x04000802, 0x00200802, 0x04200800,
    0x04200802, 0x00200000, 0x00000000, 0x04000002,
    0x00000002, 0x04000000, 0x04200002, 0x00000802,
    0x04000800, 0x00200802, 0x00200002, 0x04000800,
    0x04000002, 0x04200000, 0x04200800, 0x00200002,
    0x04200000, 0x00000800, 0x00000802, 0x04200802,
    0x00200800, 0x00000002, 0x04000000, 0x00200800,
    0x04000000, 0x00200800, 0x00200000, 0x04000802,
    0x04000802, 0x04200002, 0x04200002, 0x00000002,
    0x00200002, 0x04000000, 0x04000800, 0x00200000,
    0x04200800, 0x00000802, 0x00200802, 0x04200800,
    0x00000802, 0x04000002, 0x04200802, 0x04200000,
    0x00200800, 0x00000000, 0x00000002, 0x04200802,
    0x00000000, 0x00200802, 0x04200000, 0x00000800,
    0x04000002, 0x04000800, 0x00000800, 0x00200002
},{
    0x10001040, 0x00001000, 0x00040000, 0x10041040,
    0x10000000, 0x10001040, 0x00000040, 0x10000000,
    0x00040040, 0x10040000, 0x10041040, 0x00041000,
    0x10041000, 0x00041040, 0x00001000, 0x00000040,
    0x10040000, 0x10000040, 0x10001000, 0x00001040,
    0x00041000, 0x00040040, 0x10040040, 0x10041000,
    0x00001040, 0x00000000, 0x00000000, 0x10040040,
    0x10000040, 0x10001000, 0x00041040, 0x00040000,
    0x00041040, 0x00040000, 0x10041000, 0x00001000,
    0x00000040, 0x10040040, 0x00001000, 0x00041040,
    0x10001000, 0x00000040, 0x10000040, 0x10040000,
    0x10040040, 0x10000000, 0x00040000, 0x10001040,
    0x00000000, 0x10041040, 0x00040040, 0x10000040,
    0x10040000, 0x10001000, 0x10001040, 0x00000000,
    0x10041040, 0x00041000, 0x00041000, 0x00001040,
    0x00001040, 0x00040040, 0x10000000, 0x10041000 }
};

/* these two tables are part of the 'permuted choice 1' function. */
static unsigned long lks[16] = {
    0x00000000, 0x00000001, 0x00000100, 0x00000101,
    0x00010000, 0x00010001, 0x00010100, 0x00010101,
    0x01000000, 0x01000001, 0x01000100, 0x01000101,
    0x01010000, 0x01010001, 0x01010100, 0x01010101
};

static unsigned long rks[16] = {
    0x00000000, 0x01000000, 0x00010000, 0x01010000,
    0x00000100, 0x01000100, 0x00010100, 0x01010100,
    0x00000001, 0x01000001, 0x00010001, 0x01010001,
    0x00000101, 0x01000101, 0x00010101, 0x01010101,
};

/*
 * iperm: this performs the "initial permutation" on the data to be encrypted or
 * decrypted.  Additionally, the resulting two words are rotated one bit to the
 * left.
 */
inline void DES::iperm(unsigned long &lb, unsigned long &rb)
{
    unsigned long wb;

    wb  = ((lb >>  4) ^ rb) & 0x0f0f0f0f;
    lb ^= wb <<  4;
    rb ^= wb;
    wb  = ((lb >> 16) ^ rb) & 0x0000ffff;
    lb ^= wb << 16;
    rb ^= wb;
    wb  = ((rb >>  2) ^ lb) & 0x33333333;
    rb ^= wb <<  2;
    lb ^= wb;
    wb  = ((rb >>  8) ^ lb) & 0x00ff00ff;
    rb ^= wb <<  8;
    lb ^= wb;
    rb  = ((rb << 1) | ((rb >> 31) & 1)) & 0xffffffff;
    wb  = (lb ^ rb) & 0xaaaaaaaa;
    lb ^= wb;
    rb ^= wb;
    lb = ((lb << 1) | ((lb >> 31) & 1)) & 0xffffffff;
}

/*
 * fperm: performs the "inverse initial permutation" 
 */
inline void DES::fperm(unsigned long &lb, unsigned long &rb)
{
    unsigned long wb;

    rb  = (rb << 31) | (rb >> 1);
    wb  = (lb ^ rb) & 0xaaaaaaaa;
    rb ^= wb;
    lb ^= wb;
    lb  = (lb << 31) | (lb >> 1);
    wb  = ((lb >> 8) ^ rb) & 0x00ff00ff;
    lb ^= (wb << 8);
    rb ^= wb;
    wb  = ((lb >> 2) ^ rb) & 0x33333333;
    lb ^= (wb << 2);
    rb ^= wb;
    wb  = ((rb >> 16) ^ lb) & 0x0000ffff;
    rb ^= (wb << 16);
    lb ^= wb;
    wb  = ((rb >> 4) ^ lb) & 0x0f0f0f0f;
    rb ^= (wb << 4);
    lb ^= wb;
}

DES::DES(unsigned char *key, int klen)
{
    unsigned char buffer[8];
    unsigned char *ptr;
    int pad, i;

    if (klen < 8)
    {
        pad = 8 - (klen & 7);

        for (i = 0, ptr = buffer; i < klen; i++)
            *(ptr++) = *(key++);
        for (i = 0; i < pad; i++)
            *(ptr++) = pad;

        keySchedule(buffer, &esk[0]);
    }
    else keySchedule(key, &esk[0]);

    for (i = 0; i < 32; i += 2)
    {
        dsk[i] = esk[30-i];
        dsk[i+1] = esk[31-i];
    }
}

DES::DES(unsigned char *key, int klen, unsigned char *ivect, int ivlen)
{
    unsigned char buffer[8];
    unsigned char *ptr;
    int pad, i;

    if (klen < 8)
    {
        pad = 8 - klen;

        for (i = 0, ptr = buffer; i < klen; i++)
            *(ptr++) = *(key++);
        for (i = 0; i < pad; i++)
            *(ptr++) = pad;

        keySchedule(buffer, &esk[0]);
    }
    else keySchedule(key, &esk[0]);

    for (i = 0; i < 32; i += 2)
    {
        dsk[i] = esk[30-i];
        dsk[i+1] = esk[31-i];
    }

    if (ivlen < 8)
    {
        pad = 8 - ivlen;

        for (i = 0, ptr = buffer; i < ivlen; i++)
            *(ptr++) = *(ivect++);
        for (i = 0; i < pad; i++)
            *(ptr++) = pad;
        ptr = buffer;

        iv[0] = *(ptr++) << 24;
        iv[0] |= *(ptr++) << 16;
        iv[0] |= *(ptr++) <<  8;
        iv[0] |= *(ptr++);
        iv[1] = *(ptr++) << 24;
        iv[1] |= *(ptr++) << 16;
        iv[1] |= *(ptr++) <<  8;
        iv[1] |= *ptr;
    }
    else
    {
        iv[0] = *(ivect++) << 24;
        iv[0] |= *(ivect++) << 16;
        iv[0] |= *(ivect++) <<  8;
        iv[0] |= *(ivect++);
        iv[1] = *(ivect++) << 24;
        iv[1] |= *(ivect++) << 16;
        iv[1] |= *(ivect++) <<  8;
        iv[1] |= *ivect;
    }
    oiv[0] = iv[0];
    oiv[1] = iv[1];
}

void DES::keySchedule(unsigned char *key, unsigned long *subkeys)
{
    unsigned long lb, rb, wb, shifts(0x7efc);
    int i;

    lb = *(key++) << 24;
    lb |= *(key++) << 16;
    lb |= *(key++) <<  8;
    lb |= *(key++);
    rb = *(key++) << 24;
    rb |= *(key++) << 16;
    rb |= *(key++) <<  8;
    rb |= *key;

    wb  = ((rb >> 4) ^ lb) & 0x0f0f0f0f, rb ^= wb << 4, lb ^= wb;
    wb  = ((rb >> 0) ^ lb) & 0x10101010, rb ^= wb << 0, lb ^= wb;

    lb  = (lks[(lb      ) & 0xf] << 3) | (lks[(lb >>  8) & 0xf] << 2) |
          (lks[(lb >> 16) & 0xf] << 1) | (lks[(lb >> 24) & 0xf]     ) |
          (lks[(lb >>  5) & 0xf] << 7) | (lks[(lb >> 13) & 0xf] << 6) |
          (lks[(lb >> 21) & 0xf] << 5) | (lks[(lb >> 29) & 0xf] << 4);
    lb &= 0x0fffffff;

    rb  = (rks[(rb >>  1) & 0xf] << 3) | (rks[(rb >>  9) & 0xf] << 2) |
          (rks[(rb >> 17) & 0xf] << 1) | (rks[(rb >> 25) & 0xf]     ) |
          (rks[(rb >>  4) & 0xf] << 7) | (rks[(rb >> 12) & 0xf] << 6) |
          (rks[(rb >> 20) & 0xf] << 5) | (rks[(rb >> 28) & 0xf] << 4);
    rb &= 0x0fffffff;

    for (i = 0; i < 16; i++)
    {
        if (shifts & 1)
        {
            lb = ((lb << 2) | (lb >> 26)) & 0x0fffffff;
            rb = ((rb << 2) | (rb >> 26)) & 0x0fffffff;
        }
        else
        {
            lb = ((lb << 1) | (lb >> 27)) & 0x0fffffff;
            rb = ((rb << 1) | (rb >> 27)) & 0x0fffffff;
        }
        shifts >>= 1;

        *(subkeys++) = ((lb <<  4) & 0x24000000) | ((lb << 28) & 0x10000000) |
                       ((lb << 14) & 0x08000000) | ((lb << 18) & 0x02080000) |
                       ((lb <<  6) & 0x01000000) | ((lb <<  9) & 0x00200000) |
                       ((lb >>  1) & 0x00100000) | ((lb << 10) & 0x00040000) |
                       ((lb <<  2) & 0x00020000) | ((lb >> 10) & 0x00010000) |
                       ((rb >> 13) & 0x00002000) | ((rb >>  4) & 0x00001000) |
                       ((rb <<  6) & 0x00000800) | ((rb >>  1) & 0x00000400) |
                       ((rb >> 14) & 0x00000200) | ((rb      ) & 0x00000100) |
                       ((rb >>  5) & 0x00000020) | ((rb >> 10) & 0x00000010) |
                       ((rb >>  3) & 0x00000008) | ((rb >> 18) & 0x00000004) |
                       ((rb >> 26) & 0x00000002) | ((rb >> 24) & 0x00000001);

        *(subkeys++) = ((lb << 15) & 0x20000000) | ((lb << 17) & 0x10000000) |
                       ((lb << 10) & 0x08000000) | ((lb << 22) & 0x04000000) |
                       ((lb >>  2) & 0x02000000) | ((lb <<  1) & 0x01000000) |
                       ((lb << 16) & 0x00200000) | ((lb << 11) & 0x00100000) |
                       ((lb <<  3) & 0x00080000) | ((lb >>  6) & 0x00040000) |
                       ((lb << 15) & 0x00020000) | ((lb >>  4) & 0x00010000) |
                       ((rb >>  2) & 0x00002000) | ((rb <<  8) & 0x00001000) |
                       ((rb >> 14) & 0x00000808) | ((rb >>  9) & 0x00000400) |
                       ((rb      ) & 0x00000200) | ((rb <<  7) & 0x00000100) |
                       ((rb >>  7) & 0x00000020) | ((rb >>  3) & 0x00000011) |
                       ((rb <<  2) & 0x00000004) | ((rb >> 21) & 0x00000002);
    }
}

unsigned long DES::padLength(int len)
{
    return (len + 8) & ~7;
}

void DES::resetIV(void)
{
    iv[0] = oiv[0];
    iv[1] = oiv[1];
}

unsigned char *DES::encipher(unsigned char *message, int len)
{
    unsigned char *nstr, *nptr;
    int pad, i;

    if ((pad = 8 - (len & 7)) == 0)
        pad = 8;

    nptr = nstr = new unsigned char[len+pad];

    while (len >= 0)
    {
        if (len < 8)
        {
            unsigned char *tmp(&(*nstr));

            for (i = 0; i < len; i++)
                *(tmp++) = *(message++);
            for (i = 0; i < pad; i++)
                *(tmp++) = (char) pad;

            blockCipherCBC(nstr, nstr, DES_ENCRYPT);
        }
        else blockCipherCBC(message, nstr, DES_ENCRYPT);

        nstr += 8;
        message += 8;
        len -= 8;
    }
    return nptr;
}

unsigned long DES::decipher(unsigned char *message, int len)
{
    int mlen(0), i, j;

    if ((len & 7) == 0)
        return 0;

    while (len > 0)
    {
        if (len == 8)
        {
            blockCipherCBC(message, message, DES_DECRYPT);

            for (i = 8; i > 0; i--)
            {
                if (*message == i)
                {
                    for (j = i; i > 0; i--)
                    {
                        if (*message == j)
                            *(message++);
                        else
                        {
                            i = j;
                            message -= 8-i+1;
                            mlen++;
                            break;
                        }
                    }
                }
                else
                {
                    *(message++);
                    mlen++;
                }
            }
        }
        else
        {
            blockCipherCBC(message, message, DES_DECRYPT);
            message += 8;
            mlen += 8;
        }
        len -= 8;
    }
    return mlen;
}

void DES::cipherECB(unsigned char *src, unsigned char *dest, int length, bool encrypt)
{
    int i;

    for (i = 0; i < length; i += 8)
    {
        blockCipherECB(src, dest, encrypt);
        src += 8;
        dest += 8;
    }
}

void DES::cipherCBC(unsigned char *src, unsigned char *dest, int length, bool encrypt)
{
    int i;

    for (i = 0; i < length; i += 8)
    {
        blockCipherCBC(src, dest, encrypt);
        src += 8;
        dest += 8;
    }
}

void DES::blockCipherECB(unsigned char *block, unsigned char *to, bool encrypt)
{
    unsigned long lb, rb, wb, *sk;
    int i;

    /* 0 = decrypt, 1 = encrypt */
    sk = encrypt ? esk : dsk;

    /* pack the block into two 32-bit segments */
    lb = (*(block++) << 24);
    lb |= (*(block++) << 16);
    lb |= (*(block++) <<  8);
    lb |= *(block++);
    rb = (*(block++) << 24);
    rb |= (*(block++) << 16);
    rb |= (*(block++) <<  8);
    rb |= *block;

    /* initial permutation */
    iperm(lb, rb);

    /* 16 rounds of DES */
    for (i = 0; i < 8; i++)
    {
        wb  = rb ^ *(sk++);
        lb ^= spbox[7][(wb      ) & 0x3f];
        lb ^= spbox[5][(wb >>  8) & 0x3f];
        lb ^= spbox[3][(wb >> 16) & 0x3f];
        lb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((rb << 28) | (rb >> 4)) ^ *(sk++);
        lb ^= spbox[6][(wb      ) & 0x3f];
        lb ^= spbox[4][(wb >>  8) & 0x3f];
        lb ^= spbox[2][(wb >> 16) & 0x3f];
        lb ^= spbox[0][(wb >> 24) & 0x3f];

        wb  = lb ^ *(sk++);
        rb ^= spbox[7][(wb      ) & 0x3f];
        rb ^= spbox[5][(wb >>  8) & 0x3f];
        rb ^= spbox[3][(wb >> 16) & 0x3f];
        rb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((lb << 28) | (lb >> 4)) ^ *(sk++);
        rb ^= spbox[6][(wb      ) & 0x3f];
        rb ^= spbox[4][(wb >>  8) & 0x3f];
        rb ^= spbox[2][(wb >> 16) & 0x3f];
        rb ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* final permutation */
    fperm(lb, rb);

    /* reassemble the block */
    *(to++) = (rb >> 24);
    *(to++) = (rb >> 16);
    *(to++) = (rb >>  8);
    *(to++) = (rb      );
    *(to++) = (lb >> 24);
    *(to++) = (lb >> 16);
    *(to++) = (lb >>  8);
    *(to  ) = (lb      );
}

void DES::blockCipherCBC(unsigned char *block, unsigned char *to, bool encrypt)
{
    unsigned long lb, rb, wb, w[2], *sk;
    int i;

    /* pack the block into two 32-bit segments */
    lb = (*(block++) << 24);
    lb |= (*(block++) << 16);
    lb |= (*(block++) <<  8);
    lb |= *(block++);
    rb = (*(block++) << 24);
    rb |= (*(block++) << 16);
    rb |= (*(block++) <<  8);
    rb |= *block;

    /* chain if encrypting */
    if (encrypt == DES_ENCRYPT)
    {
        sk = esk;
        w[0] = lb ^ iv[0];
        w[1] = rb ^ iv[1];
    }
    else
    {
        sk = dsk;
        w[0] = lb;
        w[1] = rb;
    }

    /* initial permutation */
    iperm(w[0], w[1]);

    /* 16 rounds of DES */
    for (i = 0; i < 8; i++)
    {
        wb    = w[1] ^ *(sk++);
        w[0] ^= spbox[7][(wb      ) & 0x3f];
        w[0] ^= spbox[5][(wb >>  8) & 0x3f];
        w[0] ^= spbox[3][(wb >> 16) & 0x3f];
        w[0] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[1] << 28) | (w[1] >> 4)) ^ *(sk++);
        w[0] ^= spbox[6][(wb      ) & 0x3f];
        w[0] ^= spbox[4][(wb >>  8) & 0x3f];
        w[0] ^= spbox[2][(wb >> 16) & 0x3f];
        w[0] ^= spbox[0][(wb >> 24) & 0x3f];

        wb    = w[0] ^ *(sk++);
        w[1] ^= spbox[7][(wb      ) & 0x3f];
        w[1] ^= spbox[5][(wb >>  8) & 0x3f];
        w[1] ^= spbox[3][(wb >> 16) & 0x3f];
        w[1] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[0] << 28) | (w[0] >> 4)) ^ *(sk++);
        w[1] ^= spbox[6][(wb      ) & 0x3f];
        w[1] ^= spbox[4][(wb >>  8) & 0x3f];
        w[1] ^= spbox[2][(wb >> 16) & 0x3f];
        w[1] ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* final permutation */
    fperm(w[0], w[1]);

    /* chain if decrypting, then update the initialization vector */
    if (encrypt == DES_ENCRYPT)
    {
        iv[0] = w[0];
        iv[1] = w[1];
    }
    else
    {
        w[0] ^= iv[1];
        w[1] ^= iv[0];
        iv[0] = rb;
        iv[1] = lb;
    }

    /* reassemble the block */
    *(to++) = (w[1] >> 24);
    *(to++) = (w[1] >> 16);
    *(to++) = (w[1] >>  8);
    *(to++) = (w[1]      );
    *(to++) = (w[0] >> 24);
    *(to++) = (w[0] >> 16);
    *(to++) = (w[0] >>  8);
    *(to  ) = (w[0]      );
}

TripleDES::TripleDES(unsigned char *key, int klen)
{
    unsigned char buffer[24];
    unsigned char *ptr;
    int pad, i;
    
    if (klen < 24)
    {
        pad = 24 - klen;

        for (i = 0, ptr = buffer; i < klen; i++)
            *(ptr++) = *(key++);
        for (i = 0; i < pad; i++)
            *(ptr++) = pad;

        keySchedule(&buffer[0], &esk[0]);
        keySchedule(&buffer[8], &dsk[32]);
        keySchedule(&buffer[16], &esk[64]);
    }
    else
    {
        keySchedule(&key[0], &esk[0]);
        keySchedule(&key[8], &dsk[32]);
        keySchedule(&key[16], &esk[64]);
    }

    for (i = 0; i < 32; i += 2)
    {
        dsk[i] = esk[94-i];
        dsk[i+1] = esk[95-i];
        esk[i+32] = dsk[62-i];
        esk[i+33] = dsk[63-i];
        dsk[i+64] = esk[30-i];
        dsk[i+65] = esk[31-i];
    }
}

TripleDES::TripleDES(unsigned char *key, int klen, unsigned char *ivect, int ivlen)
{
    unsigned char buffer[24];
    unsigned char *ptr;
    int pad, i;

    if (klen < 24)
    {
        pad = 24 - klen;

        for (i = 0, ptr = buffer; i < klen; i++)
            *(ptr++) = *(key++);
        for (i = 0; i < pad; i++)
            *(ptr++) = pad;

        keySchedule(&buffer[0], &esk[0]);
        keySchedule(&buffer[8], &dsk[32]);
        keySchedule(&buffer[16], &esk[64]);
    }
    else
    {
        keySchedule(&key[0], &esk[0]);
        keySchedule(&key[8], &dsk[32]);
        keySchedule(&key[16], &esk[64]);
    }

    for (i = 0; i < 32; i += 2)
    {
        dsk[i] = esk[94-i];
        dsk[i+1] = esk[95-i];
        esk[i+32] = dsk[62-i];
        esk[i+33] = dsk[63-i];
        dsk[i+64] = esk[30-i];
        dsk[i+65] = esk[31-i];
    }

    if (ivlen < 8)
    {
        pad = 8 - ivlen;

        for (i = 0, ptr = buffer; i < ivlen; i++)
            *(ptr++) = *(ivect++);
        for (i = 0; i < pad; i++)
            *(ptr++) = pad;
        ptr = buffer;

        iv[0] = *(ptr++) << 24;
        iv[0] |= *(ptr++) << 16;
        iv[0] |= *(ptr++) <<  8;
        iv[0] |= *(ptr++);
        iv[1] = *(ptr++) << 24;
        iv[1] |= *(ptr++) << 16;
        iv[1] |= *(ptr++) <<  8;
        iv[1] |= *ptr;
    }
    else
    {
        iv[0] = *(ivect++) << 24;
        iv[0] |= *(ivect++) << 16;
        iv[0] |= *(ivect++) <<  8;
        iv[0] |= *(ivect++);
        iv[1] = *(ivect++) << 24;
        iv[1] |= *(ivect++) << 16;
        iv[1] |= *(ivect++) <<  8;
        iv[1] |= *ivect;
    }
    oiv[0] = iv[0];
    oiv[1] = iv[1];
}

unsigned char *TripleDES::encipher(unsigned char *message, int length)
{
    unsigned char *nptr, *nstr;
    int pad, i;

    if ((pad = 8 - (length & 7)) == 0)
        pad = 8;

    nptr = nstr = new unsigned char[length+pad];

    while (length >= 0)
    {
        if (length < 8)
        {
            unsigned char *tmp(&(*nstr));

            for (i = 0; i < length; i++)
                *(tmp++) = *(message++);
            for (i = 0; i < pad; i++)
                *(tmp++) = (char) pad;

            blockCipherCBC(nstr, nstr, DES_ENCRYPT);
        }
        else blockCipherCBC(message, nstr, DES_ENCRYPT);

        nstr += 8;
        message += 8;
        length -= 8;
    }
    return nptr;
}

unsigned long TripleDES::decipher(unsigned char *message, int length)
{
    int mlen(0), i, j;

    if (length & 7)
        return 0;

    while (length > 0)
    {
        if (length == 8)
        {
            blockCipherCBC(message, message, DES_DECRYPT);

            for (i = 8; i > 0; i--)
            {
                if (*message == i)
                {
                    for (j = i; i > 0; i--)
                    {
                        if (*message == j)
                            *(message++);
                        else
                        {
                            i = j;
                            message -= 8-i+1;
                            mlen++;
                            break;
                        }
                    }
                }
                else
                {
                    *(message++);
                    mlen++;
                }
            }
        }
        else
        {
            blockCipherCBC(message, message, DES_DECRYPT);
            message += 8;
            mlen += 8;
        }
        length -= 8;
    }
    return mlen;
}

void TripleDES::cipherECB(unsigned char *src, unsigned char *dest, int length, bool encrypt)
{
    int i;

    for (i = 0; i < length; i += 8)
    {
        blockCipherECB(src, dest, encrypt);
        src += 8;
        dest += 8;
    }
}

void TripleDES::cipherCBC(unsigned char *src, unsigned char *dest, int length, bool encrypt)
{
    int i;

    for (i = 0; i < length; i += 8)
    {
        blockCipherCBC(src, dest, encrypt);
        src += 8;
        dest += 8;
    }
}

void TripleDES::blockCipherECB(unsigned char *block, unsigned char *to, bool encrypt)
{
    unsigned long lb, rb, wb, *sk;
    int i;

    /* 0 = decrypt, 1 = encrypt */
    sk = encrypt ? &esk[0] : &dsk[0];

    /* pack the block into two 32-bit segments */
    lb  = (*(block++) << 24);
    lb |= (*(block++) << 16);
    lb |= (*(block++) <<  8);
    lb |= *(block++);
    rb  = (*(block++) << 24);
    rb |= (*(block++) << 16);
    rb |= (*(block++) <<  8);
    rb |= *block;

    /* initial permutation */
    iperm(lb, rb);

    /* 16 rounds of DES */
    for (i = 0; i < 8; i++)
    {
        wb  = rb ^ *(sk++);
        lb ^= spbox[7][(wb      ) & 0x3f];
        lb ^= spbox[5][(wb >>  8) & 0x3f];
        lb ^= spbox[3][(wb >> 16) & 0x3f];
        lb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((rb << 28) | (rb >> 4)) ^ *(sk++);
        lb ^= spbox[6][(wb      ) & 0x3f];
        lb ^= spbox[4][(wb >>  8) & 0x3f];
        lb ^= spbox[2][(wb >> 16) & 0x3f];
        lb ^= spbox[0][(wb >> 24) & 0x3f];

        wb  = lb ^ *(sk++);
        rb ^= spbox[7][(wb      ) & 0x3f];
        rb ^= spbox[5][(wb >>  8) & 0x3f];
        rb ^= spbox[3][(wb >> 16) & 0x3f];
        rb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((lb << 28) | (lb >> 4)) ^ *(sk++);
        rb ^= spbox[6][(wb      ) & 0x3f];
        rb ^= spbox[4][(wb >>  8) & 0x3f];
        rb ^= spbox[2][(wb >> 16) & 0x3f];
        rb ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* 16 rounds of DES^-1 */
    for (i = 0; i < 8; i++)
    {
        wb  = lb ^ *(sk++);
        rb ^= spbox[7][(wb      ) & 0x3f];
        rb ^= spbox[5][(wb >>  8) & 0x3f];
        rb ^= spbox[3][(wb >> 16) & 0x3f];
        rb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((lb << 28) | (lb >> 4)) ^ *(sk++);
        rb ^= spbox[6][(wb      ) & 0x3f];
        rb ^= spbox[4][(wb >>  8) & 0x3f];
        rb ^= spbox[2][(wb >> 16) & 0x3f];
        rb ^= spbox[0][(wb >> 24) & 0x3f];

        wb  = rb ^ *(sk++);
        lb ^= spbox[7][(wb      ) & 0x3f];
        lb ^= spbox[5][(wb >>  8) & 0x3f];
        lb ^= spbox[3][(wb >> 16) & 0x3f];
        lb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((rb << 28) | (rb >> 4)) ^ *(sk++);
        lb ^= spbox[6][(wb      ) & 0x3f];
        lb ^= spbox[4][(wb >>  8) & 0x3f];
        lb ^= spbox[2][(wb >> 16) & 0x3f];
        lb ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* 16 rounds of DES */
    for (i = 0; i < 8; i++)
    {
        wb  = rb ^ *(sk++);
        lb ^= spbox[7][(wb      ) & 0x3f];
        lb ^= spbox[5][(wb >>  8) & 0x3f];
        lb ^= spbox[3][(wb >> 16) & 0x3f];
        lb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((rb << 28) | (rb >> 4)) ^ *(sk++);
        lb ^= spbox[6][(wb      ) & 0x3f];
        lb ^= spbox[4][(wb >>  8) & 0x3f];
        lb ^= spbox[2][(wb >> 16) & 0x3f];
        lb ^= spbox[0][(wb >> 24) & 0x3f];

        wb  = lb ^ *(sk++);
        rb ^= spbox[7][(wb      ) & 0x3f];
        rb ^= spbox[5][(wb >>  8) & 0x3f];
        rb ^= spbox[3][(wb >> 16) & 0x3f];
        rb ^= spbox[1][(wb >> 24) & 0x3f];

        wb  = ((lb << 28) | (lb >> 4)) ^ *(sk++);
        rb ^= spbox[6][(wb      ) & 0x3f];
        rb ^= spbox[4][(wb >>  8) & 0x3f];
        rb ^= spbox[2][(wb >> 16) & 0x3f];
        rb ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* final permutation */
    fperm(lb, rb);

    /* reassemble the block */
    *(to++) = (rb >> 24);
    *(to++) = (rb >> 16);
    *(to++) = (rb >>  8);
    *(to++) = (rb      );
    *(to++) = (lb >> 24);
    *(to++) = (lb >> 16);
    *(to++) = (lb >>  8);
    *(to  ) = (lb      );
}

void TripleDES::blockCipherCBC(unsigned char *block, unsigned char *to, bool encrypt)
{
    unsigned long lb, rb, wb, w[2], *sk;
    int i;

    /* pack the block into two 32-bit segments */
    lb = (*(block++) << 24);
    lb |= (*(block++) << 16);
    lb |= (*(block++) <<  8);
    lb |= *(block++);
    rb = (*(block++) << 24);
    rb |= (*(block++) << 16);
    rb |= (*(block++) <<  8);
    rb |= *block;

    /* chain if encrypting */
    if (encrypt == DES_ENCRYPT)
    {
        sk = esk;
        w[0] = lb ^ iv[0];
        w[1] = rb ^ iv[1];
    }
    else
    {
        sk = dsk;
        w[0] = lb;
        w[1] = rb;
    }

    /* initial permutation */
    iperm(w[0], w[1]);

    /* 16 rounds of DES */
    for (i = 0; i < 8; i++)
    {
        wb    = w[1] ^ *(sk++);
        w[0] ^= spbox[7][(wb      ) & 0x3f];
        w[0] ^= spbox[5][(wb >>  8) & 0x3f];
        w[0] ^= spbox[3][(wb >> 16) & 0x3f];
        w[0] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[1] << 28) | (w[1] >> 4)) ^ *(sk++);
        w[0] ^= spbox[6][(wb      ) & 0x3f];
        w[0] ^= spbox[4][(wb >>  8) & 0x3f];
        w[0] ^= spbox[2][(wb >> 16) & 0x3f];
        w[0] ^= spbox[0][(wb >> 24) & 0x3f];

        wb    = w[0] ^ *(sk++);
        w[1] ^= spbox[7][(wb      ) & 0x3f];
        w[1] ^= spbox[5][(wb >>  8) & 0x3f];
        w[1] ^= spbox[3][(wb >> 16) & 0x3f];
        w[1] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[0] << 28) | (w[0] >> 4)) ^ *(sk++);
        w[1] ^= spbox[6][(wb      ) & 0x3f];
        w[1] ^= spbox[4][(wb >>  8) & 0x3f];
        w[1] ^= spbox[2][(wb >> 16) & 0x3f];
        w[1] ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* 16 rounds of DES^-1 */
    for (i = 0; i < 8; i++)
    {
        wb    = w[0] ^ *(sk++);
        w[1] ^= spbox[7][(wb      ) & 0x3f];
        w[1] ^= spbox[5][(wb >>  8) & 0x3f];
        w[1] ^= spbox[3][(wb >> 16) & 0x3f];
        w[1] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[0] << 28) | (w[0] >> 4)) ^ *(sk++);
        w[1] ^= spbox[6][(wb      ) & 0x3f];
        w[1] ^= spbox[4][(wb >>  8) & 0x3f];
        w[1] ^= spbox[2][(wb >> 16) & 0x3f];
        w[1] ^= spbox[0][(wb >> 24) & 0x3f];

        wb    = w[1] ^ *(sk++);
        w[0] ^= spbox[7][(wb      ) & 0x3f];
        w[0] ^= spbox[5][(wb >>  8) & 0x3f];
        w[0] ^= spbox[3][(wb >> 16) & 0x3f];
        w[0] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[1] << 28) | (w[1] >> 4)) ^ *(sk++);
        w[0] ^= spbox[6][(wb      ) & 0x3f];
        w[0] ^= spbox[4][(wb >>  8) & 0x3f];
        w[0] ^= spbox[2][(wb >> 16) & 0x3f];
        w[0] ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* 16 rounds of DES */
    for (i = 0; i < 8; i++)
    {
        wb    = w[1] ^ *(sk++);
        w[0] ^= spbox[7][(wb      ) & 0x3f];
        w[0] ^= spbox[5][(wb >>  8) & 0x3f];
        w[0] ^= spbox[3][(wb >> 16) & 0x3f];
        w[0] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[1] << 28) | (w[1] >> 4)) ^ *(sk++);
        w[0] ^= spbox[6][(wb      ) & 0x3f];
        w[0] ^= spbox[4][(wb >>  8) & 0x3f];
        w[0] ^= spbox[2][(wb >> 16) & 0x3f];
        w[0] ^= spbox[0][(wb >> 24) & 0x3f];

        wb    = w[0] ^ *(sk++);
        w[1] ^= spbox[7][(wb      ) & 0x3f];
        w[1] ^= spbox[5][(wb >>  8) & 0x3f];
        w[1] ^= spbox[3][(wb >> 16) & 0x3f];
        w[1] ^= spbox[1][(wb >> 24) & 0x3f];

        wb    = ((w[0] << 28) | (w[0] >> 4)) ^ *(sk++);
        w[1] ^= spbox[6][(wb      ) & 0x3f];
        w[1] ^= spbox[4][(wb >>  8) & 0x3f];
        w[1] ^= spbox[2][(wb >> 16) & 0x3f];
        w[1] ^= spbox[0][(wb >> 24) & 0x3f];
    }

    /* final permutation */
    fperm(w[0], w[1]);

    /* chain if decrypting, update the initialization vector */
    if (encrypt == DES_ENCRYPT)
    {
        iv[0] = w[0];
        iv[1] = w[1];
    }
    else
    {
        w[0] ^= iv[1];
        w[1] ^= iv[0];
        iv[0] = rb;
        iv[1] = lb;
    }

    /* reassemble the block */
    *(to++) = (w[1] >> 24);
    *(to++) = (w[1] >> 16);
    *(to++) = (w[1] >>  8);
    *(to++) = (w[1]      );
    *(to++) = (w[0] >> 24);
    *(to++) = (w[0] >> 16);
    *(to++) = (w[0] >>  8);
    *(to  ) = (w[0]      );
}
