/*
 * cipher.cc
 *
 * Copyright (c) 2000 Mount Linux Inc.
 * Licensed under the terms of the GPL
 */

#include <iostream>
#include <fstream>
#include "config.h"
#include "exceptions.h"
#include "cipher.h"

struct cipherDefinition
{
    char *name;
    int blocklength;
    int keylength;
    int idnum;
} cipherAlgorithms[] = {
  { "des-cbc", 8, 8, CIPHER_DES },
  { "3des-cbc", 8, 24, CIPHER_3DES },
  { "none", 1, 0, CIPHER_NONE },
  { NULL }
};

cipher::cipher(void)
{
    cipherType = CIPHER_NONE;
}

cipher::cipher(int type, unsigned char *key, int klen, unsigned char *iv, int ivlen)
{
    setCipher(type, key, klen, iv, ivlen);
}

cipher::~cipher(void)
{
    switch (cipherType)
    {
        case CIPHER_DES:
            delete context.des;
            break;
        case CIPHER_3DES:
            delete context.des3;
            break;
    }
}

char *cipher::b10to16(unsigned char *ptr, int length)
{
    static char *f("0123456789ABCDEF");
    char *nstr, *nptr;
    int i;

    nstr = nptr = new char[(length << 1) + 1];

    for (i = 0; i < length; i++)
    {
        *(nptr++) = *(f + (*(ptr) >> 4));
        *(nptr++) = *(f + (*(ptr++) & 0xf));
    }
    *(nptr++) = '\0';
    return nstr;
}

unsigned char *cipher::b16to10(char *ptr, int length)
{
    unsigned char *nstr, *nptr;
    int i;

    nstr = nptr = new unsigned char[(length >> 1)];

    for (i = 0; i < length; i += 2)
    {
        if (*ptr > 0x39)
            *nptr = ((*(ptr++) & 0xf) + 9) << 4;
        else *nptr = (*(ptr++) & 0xf) << 4;

        if (*ptr > 0x39)
            *(nptr++) += ((*(ptr++) & 0xf) + 9);
        else *(nptr++) += (*(ptr++) & 0xf);
    }
    return nstr;
}

char *cipher::fileDigest(const char *filename)
{
    unsigned char buffer[4096];
    unsigned char digest[SHA_DIGEST_SIZE];
    ifstream ifs(filename, ios::in);
    SHA hash;

    if (!ifs)
    {
        throw fileOpenError((char *) filename);
    }

    while (!(ifs.eof()))
    {
        ifs.read(buffer, sizeof(buffer));
        hash.update(buffer, ifs.gcount());
    }
    hash.final(digest);

    return b10to16(digest, SHA_DIGEST_SIZE);
}

void cipher::setCipher(int type, unsigned char *key, int klen, unsigned char *iv, int ivlen)
{
    if (type == CIPHER_DES)
    {
        cipherType = CIPHER_DES;
        context.des = new DES(key, klen, iv, ivlen);
    }
    else if (type == CIPHER_3DES)
    {
        cipherType = CIPHER_3DES;
        context.des3 = new TripleDES(key, klen, iv, ivlen);
    }
    else
    {
        cipherType = CIPHER_NONE;
    }
}

void cipher::encrypt(unsigned char *src, unsigned char *dest, int length)
{
    switch (cipherType)
    {
        case CIPHER_NONE:
            memcpy(dest, src, length);
            break;
        case CIPHER_DES:
            context.des->cipherCBC(src, dest, length, DES_ENCRYPT);
            break;
        case CIPHER_3DES:
            context.des3->cipherCBC(src, dest, length, DES_ENCRYPT);
            break;
    }
}

void cipher::decrypt(unsigned char *src, unsigned char *dest, int length)
{
    switch (cipherType)
    {
        case CIPHER_NONE:
            memcpy(dest, src, length);
            break;
        case CIPHER_DES:
            context.des->cipherCBC(src, dest, length, DES_DECRYPT);
            break;
        case CIPHER_3DES:
            context.des3->cipherCBC(src, dest, length, DES_DECRYPT);
            break;
    }
}

char *cipher::cipherList(void)
{
    char *list(NULL);
    int i, len, nlen;

    for (i = 0, len = 0; cipherAlgorithms[i].name; i++)
    {
        if (cipherAlgorithms[i + 1].name)
        {
            len++;
        }
        len += strlen(cipherAlgorithms[i].name);
    }

    list = new char[len + 1];
    list[len] = '\0';

    while (len)
    {
        nlen = strlen(cipherAlgorithms[i].name);
        len = len - nlen;

        memcpy(&list[len], cipherAlgorithms[i].name, nlen);

        if (len != 0)
        {
            len--;
            list[len] = ',';
            i--;
        }
    }
    return list;
}

int cipher::cipherNumber(char *name)
{
    int i;

    for (i = 0; cipherAlgorithms[i].name; i++)
    {
        if (strcasecmp(name, cipherAlgorithms[i].name) == 0)
        {
            return cipherAlgorithms[i].idnum;
        }
    }
    return -1;
}

char *cipher::cipherName(int type)
{
    int i;

    for (i = 0; cipherAlgorithms[i].name; i++)
    {
        if (cipherAlgorithms[i].idnum == type)
        {
            return strdup(cipherAlgorithms[i].name);
        }
    }
    return NULL;
}

int cipher::blocklength(int type)
{
    int i;

    for (i = 0; cipherAlgorithms[i].name; i++)
    {
        if (cipherAlgorithms[i].idnum == type)
        {
            return cipherAlgorithms[i].blocklength;
        }
    }
    return 0;
}

int cipher::keylength(int type)
{
    int i;

    for (i = 0; cipherAlgorithms[i].name; i++)
    {
        if (cipherAlgorithms[i].idnum == type)
        {
            return cipherAlgorithms[i].keylength;
        }
    }
    return 0;
}
