#include <stdlib.h>
#include <stdio.h>

#include <qstackbarbtn.h>

/*!
 * \brief One version of the constructor
 *
 * This constructor allows you to set the name of the button, and tell it
 * what widget it shall contain.  It also sets the default colors to
 * 200,200,200, and the hilight color to 220,220,220.  It sets the font
 * colors to black, and the selection-font to bold.
 */
QStackBarBtn::QStackBarBtn(QString s, QWidget *w)
{
	_label = s;

	QColor base(200,200,200);

	_color = new QColorGroup( base, base, base.light(), base.dark(), base.light(), base, base, base, base );

	base.setRgb(220,220,220);
	_hiColor = new QColorGroup( base, base, base.light(), base.dark(), base.light(), base, base, base, base );
	_selColor = new QColorGroup( *_color );
	pWidget = w;

	_fcolor = _fselcolor = _fhicolor = QColor(0,0,0);

	pPixmap = NULL;

	if( !w )
	{
		fprintf(stderr, "QSTACKBAR ERROR - Cannot set a null widget\n");
		return;
	}

	_selFont.setBold(true);

	w->hide();
}

/*!
 * \brief Another version of the constructor
 *
 * This constructor differs from the above one only in that it allows you
 * to specify a color for the buttons.  It will set the base color to that
 * color, and the hilight color to a lighter version of that color.
 */
QStackBarBtn::QStackBarBtn(QString s, QWidget *w, QColor c)
{
	_label = s;
	pWidget = w;


	QColor base(c);
	_color = new QColorGroup( base, base, base.light(), base.dark(), base.light(), base, base, base, base );
	base = base.light();
	_hiColor = new QColorGroup( base, base, base.light(), base.dark(), base.light(), base, base, base, base );
	_selColor = new QColorGroup( *_color );

	if( !w )
	{
		fprintf(stderr, "QSTACKBAR ERROR - Cannot set a null widget\n");
		return;
	}

	_fcolor = _fselcolor = _fhicolor = QColor(0,0,0);

	_selFont.setBold(true);

	pPixmap = NULL;

	w->hide();
}

/*!
 * \brief Sets the widget this button/page will contain
 *
 * Sets the widget for this button/page to contain
 */
void QStackBarBtn::setWidget( QWidget *w )
{
	if( !w )
	{
		fprintf(stderr, "QSTACKBAR ERROR - Cannot set a null widget\n");
		return;
	}

	pWidget = w;
	w->hide();
}

/*!
 * \brief Destructor
 *
 * Deletes the widget, and the color groups
 */
QStackBarBtn::~QStackBarBtn()
{
	delete pWidget;
	delete _color;
	delete _hiColor;
	delete _selColor;

	if( pPixmap )
		delete pPixmap;
}

/*!
 * \brief Set the base color of the button
 */
void QStackBarBtn::setColor( QColor c )
{
	delete _color;

	_color = new QColorGroup( c, c, c.light(), c.dark(), c.light(), c,c,c,c );
}

/*! 
 *\brief Set the pixmap for the button background
 */
void QStackBarBtn::setPixmap( QPixmap *pPix )
{
	if( pPixmap )
		delete pPixmap;

	pPixmap = pPix;
}

/*!
 * \brief Sets the hilight color of this button
 */
void QStackBarBtn::setHiColor( QColor c )
{
	delete _hiColor;

	_hiColor = new QColorGroup( c, c, c.light(), c.dark(), c.light(), c,c,c,c );
}


/*!
 * \brief Sets the selection color of this button
 */
void QStackBarBtn::setSelColor( QColor c )
{
	delete _selColor;

	_selColor = new QColorGroup( c, c, c.light(), c.dark(), c.light(), c,c,c,c );
}
