#include <stdio.h>
#include <qpainter.h>
#include <qdrawutil.h>

#include <qstackbar.h>
#include <qstackbarbtn.h>

/*!
  \class QStackBar qstackbar.h
  \brief Implements a button-bar similar to that of MS Outlook

  Use this widget to implement multiple pages of widgets in a single vertical
  column.  Each button can have its own unselected, selected, and highlighted
  color, text color, and text font.  They can also be given a pixmap background
  instead (which will not highlight when you put the mouse over it).
  
  The following code demonstrates how to create a stackbar, and add a few pages
  to it with some widgets.
  
  For more examples, see the tests in the test directory of the distribution.
  
\code

	QStackBar *sb;
	QListBox *l1;
	
	sb = new QStackBar(pParent, NULL);
	l1 = new QListBox *l1 = new QListBox( sb );
	
	l1->insertItem("Item 1");
	l1->insertItem("Item 2");
	l1->insertItem("Item 3");
	l1->insertItem("Item 4");
	
	sb.addPage( new QStackBarBtn( "List Box 1", l1 ) );
	sb.addPage( new QStackBarBtn( "A Button", new QPushButton("Hello") ) );
	sb.addPage( new QStackBarBtn( "Another Button", new QPushButton("Yep")));

\endcode
*/


/*!
 * \brief Default constructor
 *
 * Initializes the lists for holding the page-buttons, sets the current page
 * to -1, turns mouse tracking on, sets the stackHeight to the default of
 * 18 pixels.
 */
QStackBar::QStackBar( QWidget *parent, const char *name )
	: QWidget( parent, name )
{
	pButtons = new QList<QStackBarBtn>;
	pButtons->setAutoDelete(true);

	curPage = -1;
	
	setMouseTracking( true );

	curHighlight = -1;
	_stackHeight = 18;

}


/*!
 * \brief Default structor
 *
 * Destroys the list of QStackBarBtns
*/
QStackBar::~QStackBar()
{
	delete pButtons;
}


/*!
 * \brief Handles window resizes
 *
 * Forces the buttons to rearrange themselves inside the widget. It
 * also resizes the active page widget.
*/
void QStackBar::resizeEvent( QResizeEvent *e )
{
	rearrangeButtons(e->size().width(), e->size().height() );
}


/*!
 * \brief Rearranges the buttons to match the dimensions w,h
 *
 * This is called internally by the widget to rearrange the buttons
 * associated with the pages.
*/
void QStackBar::rearrangeButtons( int w, int h )
{
	int topY;
	int bottomY;
	int count = pButtons->count()-1;
	QWidget *pWidget;
	QStackBarBtn *pBtn;

	topY = (curPage+1) * _stackHeight;
	bottomY = h - ((count - curPage)*_stackHeight);
	
	pBtn = pButtons->at( curPage );
	pWidget = pBtn->widget();
	pWidget->setGeometry( 0, topY, w, bottomY-topY );
}


/*!
 * \brief Paints the QStackBar
 *
 * Duh
*/
void QStackBar::paintEvent( QPaintEvent * )
{
	QPainter *pPaint;
	QBrush brush;
	QStackBarBtn *pBtn;

	pPaint = new QPainter(this);

	brush.setStyle( QBrush::SolidPattern );

	
	int i;

	// Draw the buttons before the current button
	pBtn = NULL;
	for( i=0; i<curPage; i++ )
	{
		// Get the QStackBarBtn
		if( !pBtn )
			pBtn = pButtons->first();
		else
			pBtn = pButtons->next();

		brush.setColor( pBtn->color()->base() );
		if( pBtn->pixmap() )
		{
			printf("Pixmap\n");
			brush.setPixmap( *(pBtn->pixmap()) );
		}
		else
			brush.setStyle( QBrush::SolidPattern );

		
		pPaint->setPen( pBtn->fontColor() );
	
		qDrawWinButton( pPaint, 0, i*_stackHeight, width(), _stackHeight, *(pBtn->color()), false, &brush  );

		pPaint->setFont( *(pBtn->font()) );
		pPaint->drawText( 0, i*_stackHeight-1, width(), _stackHeight, AlignHCenter | AlignVCenter, pBtn->label() );
	}


	// Draw the current button
	if( !pBtn )
		pBtn = pButtons->first();
	else
		pBtn = pButtons->next();

	brush.setColor( pBtn->selColor()->base() );
	if( pBtn->pixmap() )
		brush.setPixmap( *(pBtn->pixmap()) );
	else
		brush.setStyle( QBrush::SolidPattern );

	qDrawWinButton( pPaint, 0, i*_stackHeight, width(), _stackHeight, *(pBtn->selColor()), false, &brush  );
	pPaint->setFont( *(pBtn->selFont()) );
	pPaint->setPen( pBtn->fontSelColor());
	pPaint->drawText( 0, i*_stackHeight-1, width(), _stackHeight, AlignHCenter | AlignVCenter, pBtn->label() );


	// Draw the bottom buttons
	pBtn = NULL;
	for( i=pButtons->count()-1; i>curPage; i-- )
	{
		// Get the QStackBarBtn
		if( !pBtn )
			pBtn = pButtons->last();
		else
			pBtn = pButtons->prev();
			
		brush.setColor( pBtn->color()->base() );
		if( pBtn->pixmap() )
			brush.setPixmap( *(pBtn->pixmap()) );
		else
			brush.setStyle( QBrush::SolidPattern );


		qDrawWinButton( pPaint, 0, height()-((pButtons->count()-i)*_stackHeight), width(), _stackHeight, *(pBtn->color()), false, &brush );

		pPaint->setFont( *(pBtn->font()) );
		pPaint->setPen( pBtn->fontColor());

		pPaint->drawText( 0, height()-((pButtons->count()-i)*_stackHeight)-1, width(), _stackHeight, AlignHCenter | AlignVCenter, pBtn->label() );
	}

	delete pPaint;
}


/*!
 * \brief Add a new page to the QStackBar
 *
 * Call this to add a new page to the widget.
*/
void QStackBar::addPage( QStackBarBtn *pBtn )
{
	pButtons->append( pBtn );
	curPage = pButtons->count() - 1;
}


/*!
 * \brief Sets the current page to i
 *
 * Call this to set the current page.  The widget will refresh and
 * rearrange itself.
 */
void QStackBar::setCurPage( int i )
{
	if( i >= (int)pButtons->count() || i < 0 )
		return;

	QStackBarBtn *pBtn;

	if( curPage >= 0 )
	{
		pBtn = pButtons->at( curPage );
		pBtn->widget()->hide();
	}
	
	curPage = i;
	pBtn = pButtons->at( curPage );
	pBtn->widget()->show();


	rearrangeButtons( width(), height() );
	repaint(false);
}


/*!
 * \brief Returns the button at mx, my
 *
 * This is used internally to find out what button is located
 * at a given mouse coordinate.  -1 is returned if nothing
 * is found.
*/
int QStackBar::whichButton( int mx, int my )
{
	int numBottom;	// number of buttons on the bottom stack

	// quick death
	if( curPage < 0 )
		return -1;

	if( mx < 0 || mx >= width() )
		return -1;
	
	numBottom = pButtons->count()-curPage-1;
	
	if( my >=0 && my < (curPage+1)*_stackHeight )
		return my / _stackHeight;
	else if( my >= height()-(numBottom*_stackHeight) && my < height() )
		return curPage + ( (my-(height()-(numBottom*_stackHeight)))/_stackHeight) + 1;

	return -1;
}


/*!
 * \brief Mouse press handler
 *
 * Selects the page under the mouse cursor if it can, and before that
 * it emits the pageSelected signal.
*/
void QStackBar::mousePressEvent( QMouseEvent *e )
{
	int buttonNum = whichButton( e->x(), e->y() );

	if( buttonNum < 0 )
		return;

	emit pageSelected( buttonNum );
	setCurPage( buttonNum );
}


/*!
 * \brief Handles the movement of the mouse
 *
 * Checks if the mouse cursor is over a button, if it is, it highlights
 * it and unhighlights the old button.  Also emits the signal pageHighlighted.
*/
void QStackBar::mouseMoveEvent( QMouseEvent *e )
{
	int buttonNum;
	QBrush brush;
	QStackBarBtn *pBtn;
	QFont *pFont;
	QColorGroup *pCGroup;
	QColor fcolor;

	if( curPage < 0 )
		return;

	brush.setStyle( QBrush::SolidPattern );

	buttonNum = whichButton( e->x(), e->y() );
	
	// This button is already highlighted, leave
	if( buttonNum == curHighlight )
		return;

	QPainter *pPaint = new QPainter(this);
	// Otherwise, undraw the last button if we have one
	if( curHighlight >= 0 )
	{
		pBtn = pButtons->at( curHighlight );

		// If we are the current page, undraw in the selected color & font
		if( curHighlight == curPage )
		{
			brush.setColor( pBtn->selColor()->base() );
			pFont = pBtn->selFont();
			pCGroup = pBtn->selColor();
			fcolor = pBtn->fontSelColor();
		}
		else
		{
			brush.setColor( pBtn->color()->base() );
			pFont = pBtn->font();
			pCGroup = pBtn->color();
			fcolor = pBtn->fontColor();
		}

		if( pBtn->pixmap() )
			brush.setPixmap( *(pBtn->pixmap()) );
		else
			brush.setStyle( QBrush::SolidPattern );
	
		pPaint->setFont( *pFont );
	
		if( curHighlight <= curPage )
		{
			qDrawWinButton( pPaint, 0, curHighlight*_stackHeight, width(), _stackHeight, *pCGroup, false, &brush  );

			pPaint->setPen( fcolor );
			pPaint->drawText( 0, curHighlight*_stackHeight-1, width(), _stackHeight, AlignHCenter | AlignVCenter, pBtn->label() );
		}
		else
		{
			// bottom button
			qDrawWinButton( pPaint, 0, height()-((pButtons->count()-curHighlight)*_stackHeight), width(), _stackHeight, *pCGroup, false, &brush );
			pPaint->setPen( fcolor );
			pPaint->drawText( 0, height()-((pButtons->count()-curHighlight)*_stackHeight)-1, width(), _stackHeight, AlignHCenter | AlignVCenter, pBtn->label() );
		}
	}


	// And draw this onea
	curHighlight = buttonNum;
	if( curHighlight >= 0 )
	{
		pBtn = pButtons->at(curHighlight);
		brush.setColor( pBtn->hiColor()->base() );
		if( pBtn->pixmap() )
			brush.setPixmap( *(pBtn->pixmap()) );
		else
			brush.setStyle( QBrush::SolidPattern );

		pCGroup = pBtn->hiColor();

		if( curHighlight==curPage )
		{
			pPaint->setFont( *(pBtn->selFont()) );
			fcolor = pBtn->fontSelColor();
		}
		else
		{
			pPaint->setFont( *(pBtn->hiFont()) );
			fcolor = pBtn->fontHiColor();
		}

		pPaint->setPen( fcolor );
		if( curHighlight <= curPage )
		{
			qDrawWinButton( pPaint, 0, curHighlight*_stackHeight, width(), _stackHeight, *pCGroup, false, &brush  );	
			pPaint->drawText( 0, curHighlight*_stackHeight-1, width(), _stackHeight, AlignHCenter | AlignVCenter, pBtn->label() );
		}
		else
		{
			// bottom button
			qDrawWinButton( pPaint, 0, height()-((pButtons->count()-curHighlight)*_stackHeight), width(), _stackHeight, *pCGroup, false, &brush );		
			pPaint->drawText( 0, height()-((pButtons->count()-curHighlight)*_stackHeight)-1, width(), _stackHeight, AlignHCenter | AlignVCenter, pBtn->label() );
		}
	}

	emit pageHighlighted( curHighlight );

	delete pPaint;

}

/*!
 * \brief Sets the button height
 *
 * Use this to set the height of all the buttons.  All buttons
 * are the same height.
 *
 * \code
      // make the buttons 30 pixels heigh
	  pStackBar->setButtonHeight(30);
   \endcode
*/
void QStackBar::setButtonHeight( int i )
{
	_stackHeight = i;

	rearrangeButtons( width(), height() );
}


