/*****************************************************************************
*
*   Copyright (c) 2000 Mount Linux Inc.
*   Licensed under the terms of the GPL
*
*   dis_connect.cc 
*
*   the methods to open and close a network connection
*
*****************************************************************************/

#include <qlist.h>
#include <qmessagebox.h>
#include <qsocketnotifier.h>
#include <qtimer.h>
#include "cipher.h"
#include "guisocket.h"
#include "hostObj.h"
#include "nmAlgInit.h"
#include "nmKexInit.h"
#include "nmKexReply.h"
#include "nmUserAuth.h"
#include "nmVersion.h"

void guisocket::connect()
{
    int rv, sd(0);

    switch (connectionState)
    {
        case CLOSED:
            {
                socket = new transport();

                sd = socket->connect((char *)host->getHostname()->data(), host->getPort());
            
                connectionState = CONNECTING;

                if (sd == 0)
                {
                    attempts = MAXATTEMPTS;
                    connect();
                    return;
                }
                else if (sd < 0 && errno != EINPROGRESS)
                {
                    connectError(errno);
                    return;
                }
                else
                {
                    if (socketTimer != NULL)
                    {
                        delete socketTimer;
                    }

                    socketTimer = new QTimer(this);

                    QObject::connect( socketTimer, SIGNAL(timeout()), this, SLOT(connect()));
                    socketTimer->start(1000, true);
                }
            }
        break;

        case CONNECTING:
            {
                fd_set fd_in, fd_out;
                socklen_t length;
                int sockerr;

                FD_ZERO(&fd_in);
                FD_ZERO(&fd_out);
                FD_SET(socket->getsd(), &fd_in);
                FD_SET(socket->getsd(), &fd_out);

                if ((rv = select(socket->getsd() + 1, &fd_in, &fd_out, NULL, NULL)) < 0)
                {
                    if (errno != EINTR)
                    {
                        connectError(errno);
                    }
                    return;
                }
                else if (rv == 0 && MAXATTEMPTS > ++attempts)
                {
                    timeout();
                    attempts = 0;
                    return;
                }

                if (FD_ISSET(socket->getsd(), &fd_in) || FD_ISSET(socket->getsd(), &fd_out))
                {
                    length = sizeof(sockerr);

                    if ((rv = getsockopt(socket->getsd(), SOL_SOCKET, SO_ERROR, &sockerr, &length)) < 0)
                    {
                        connectError(errno);
                        attempts = 0;
                        return;
                    }
                    else if (sockerr != 0)
                    {
                        connectError(sockerr);
                        return;
                    }

                    connectionState = VEXINIT;

                    if (socketTimer != NULL)
                    {
                        socketTimer->stop();
                        delete socketTimer;
                        socketTimer = NULL;
                    }
                
                    if (writeWatcher == NULL)
                    {
                        writeWatcher = new QSocketNotifier(socket->getsd(), QSocketNotifier::Write, 0);
                    }
                    else
                    {
                        writeWatcher->setEnabled(true);
                    }

                    QObject::connect(writeWatcher, SIGNAL(activated(int)), this, SLOT(connect()));
                                 
                    socketTimer = new QTimer( this );
                    QObject::connect(socketTimer, SIGNAL(timeout()), this, SLOT(timeout()));
                    socketTimer->start(5000, true);
                }
            }            
        break;

        case VEXINIT:
            {
                nmVersion * tmpversion;

                socketTimer->stop();

                if (socket == NULL)
                {
                    emit connectionSignal(false);
                    return;
                }

                if (writeWatcher != NULL)
                {
                    writeWatcher->setEnabled(false);
                    QObject::disconnect(writeWatcher,SIGNAL(activated(int)), this, SLOT(connect()));
                }

                tmpversion = new nmVersion(socket, 1, COMMAND_VERSION);
                tmpversion->send(socket, PROTOCOL_VERSION);
                delete tmpversion;

                connectionState = VEXREPLY;

                readWatcher = new QSocketNotifier(socket->getsd(),QSocketNotifier::Read,0);
                QObject::connect(readWatcher,SIGNAL(activated(int)), this, SLOT(connect()));

                socketTimer->start(5000, true);
            }
        break;

        case VEXREPLY:
            {
                if (socket == NULL)
                {
                    emit connectionSignal(false);
                    return;
                }

                if (currentMessage == NULL)
                {
                    if ((rv = readHeader()) == 0)
                    {
                        currentMessage = new nmVersion(socket, 1, COMMAND_VERSION);
                    }
                    else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                    {
                        socketError(errno);
                        return;
                    }
                    else return;
                }

                if ((rv = currentMessage->receive()) == 0)
                {
                    socketTimer->stop();
                    currentMessage->activate();
                    delete currentMessage;

                    currentMessage = NULL;
                    connectionState = ALGINIT;

                    opcode = sessionID = currentObjectID = 0;

                    if (writeWatcher != NULL)
                    {
                        writeWatcher->setEnabled(false);
                        QObject::disconnect(writeWatcher,SIGNAL(activated(int)), this, SLOT(connect()));
                    }
                    socketTimer->start(5000, true);
                }
                else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                {
                    socketError(errno);
                    return;
                }
            }
        break;

        case ALGINIT:
            {
                if (socket == NULL)
                {
                    emit connectionSignal(false);
                    return;
                }

                if (currentMessage == NULL)
                {
                    if ((rv = readHeader()) == 0)
                    {
                        currentMessage = new nmAlgInit(socket, 1, COMMAND_ALGINIT);
                    }
                    else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                    {
                        socketError(errno);
                        return;
                    }
                    else return;
                }

                if ((rv = currentMessage->receive()) == 0)
                {
                    socketTimer->stop();

                    if (currentMessage->activate() != 1)
                    {
                        disconnect();
                        QMessageBox::critical(0, tr("Login Failed"), tr("Logging in to ") + *host->getCommonName() + tr(" failed during algorithm negotiation.\nReceived invalid list of algorithms."));
                    }

                    currentMessage->send(socket, ALG_INIT);
                    delete currentMessage;

                    currentMessage = NULL;
                    connectionState = ALGREPLY;

                    opcode = sessionID = currentObjectID = 0;

                    socketTimer->start(5000, true);
                }
                else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                {
                    socketError(errno);
                    return;
                }
            }
        break;

        case ALGREPLY:
            {
                if ((rv = readHeader()) == 0)
                {
                    socketTimer->stop();

                    if (opcode == ALG_SUCCESS)
                    {
                        connectionState = KEXINIT;
                        opcode = sessionID = currentObjectID = 0;

                        if (writeWatcher != NULL)
                        {
                            writeWatcher->setEnabled(true);
                            QObject::connect(writeWatcher,SIGNAL(activated(int)), this, SLOT(connect()));
                        }
                        socketTimer->start(5000, true);
                    }
                    else if (opcode == ALG_FAILURE)
                    {
                        disconnect();
                        QMessageBox::critical(0, tr("Login Failed"), tr("Logging in to ") + *host->getCommonName() + tr(" failed during algorithm negotiation."));
                    }
                    else
                    {
                        disconnect();
                        QMessageBox::critical(0, tr("Login Failed"), tr("Logging in to ") + *host->getCommonName() + tr(" failed during algorithm negotiation.\nReceived unexpected response from peer."));
                    }
                }
                else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                {
                    socketError(errno);
                    return;
                }
            }
        break;

        case KEXINIT:
            {
                nmKexInit * kexInit;

                socketTimer->stop();

                if (socket == NULL)
                {
                    emit connectionSignal(false);
                    return;
                }

                if (writeWatcher != NULL)
                {
                    writeWatcher->setEnabled(false);
                    QObject::disconnect(writeWatcher,SIGNAL(activated(int)), this, SLOT(connect()));
                }

                kexInit = new nmKexInit(socket, 1, COMMAND_KEXINIT, 512);
                kexInit->send(socket, KEX_DH_INIT);
                delete kexInit;

                connectionState = KEXINIT2;

                socketTimer->start(7000, true);
            }
        break;

        case KEXINIT2:
            {
                if (socket == NULL)
                {
                    emit connectionSignal(false);
                    return;
                }

                if (currentMessage == NULL)
                {
                    if ((rv = readHeader()) == 0)
                    {
                        currentMessage = new nmKexInit(socket, 1, COMMAND_KEXINIT);
                    }
                    else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                    {
                        socketError(errno);
                        return;
                    }
                    else return;
                }

                if ((rv = currentMessage->receive()) == 0)
                {
                    socketTimer->stop();
                    currentMessage->activate();
                    delete currentMessage;

                    currentMessage = NULL;
                    connectionState = KEXREPLY;

                    opcode = sessionID = currentObjectID = 0;

                    if (writeWatcher != NULL)
                    {
                        writeWatcher->setEnabled(true);
                        QObject::connect(writeWatcher,SIGNAL(activated(int)), this, SLOT(connect()));
                    }
                    socketTimer->start(5000, true);
                }
                else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                {
                    socketError(errno);
                    return;
                }
            }
        break;

        case KEXREPLY:
            {
                nmKexReply * kexReply;

                socketTimer->stop();
                writeWatcher->setEnabled(false);
                                                    
                if (socket == NULL)
                {
                    emit connectionSignal(false);
                    return;
                }

                connectionState = SENDAUTH;

                kexReply = new nmKexReply(socket, 1, COMMAND_KEXREPLY);
                kexReply->send(socket, KEX_DH_REPLY);
                delete kexReply;

                socket->setCipher(CIPHER_3DES, true);
               
                writeWatcher->setEnabled(true);
                socketTimer->start(5000, true);
            }
        break;

        case SENDAUTH:
            {
                nmUserAuth * authMessage;

                socketTimer->stop();

                if (socket == NULL) 
                { 
                    emit connectionSignal(false);
                    return; 
                }

                if (writeWatcher != NULL)
                {
                    writeWatcher->setEnabled(false);
                    QObject::disconnect(writeWatcher, SIGNAL(activated(int)), this, SLOT(connect()));
                }

                connectionState = AUTH;    	

                authMessage = new nmUserAuth(host->getUsername(), host->getPassword(), socket, 1, COMMAND_USERAUTH);
                authMessage->send(socket,USERAUTH_REQUEST); 
                delete authMessage;

                socketTimer->start(5000, true);              
            }
        break;
        
        case AUTH:
            {
                if (socketTimer != NULL)
                {
                    socketTimer->stop();
                    delete socketTimer;
                    socketTimer = NULL;
                }
      
                if (socketTimer != NULL)
                {
                    delete socketTimer;
                    socketTimer = NULL;
                }

                if (socket == NULL) 
                { 
                    emit connectionSignal(false);
                    return; 
                }

                if (opcode == 0)
                {
                    if ((rv = readHeader()) == 0)
                    {
                        if (opcode == USERAUTH_SUCCESS)
                        {
                            delete readWatcher;
                            readWatcher = new QSocketNotifier(socket->getsd(),QSocketNotifier::Read,0);
                            QObject::connect(readWatcher,SIGNAL(activated(int)), this, SLOT(readSocket()));

                            exceptionWatcher = new QSocketNotifier(socket->getsd(),QSocketNotifier::Exception,0);
                            QObject::connect(exceptionWatcher,SIGNAL(activated(int)), this, SLOT(disconnect()));

                            if (writeWatcher == NULL)
                            {
                                writeWatcher = new QSocketNotifier(socket->getsd(), QSocketNotifier::Write, 0);
                            }
                            
                            if (! messageQueue->isEmpty())
                            {
                                writeWatcher->setEnabled(true);
                            }

                            QObject::connect(writeWatcher, SIGNAL(activated(int)), this, SLOT(sendNextMessage()));
                            emit connectionSignal(true);

                            connectionState = OPEN;
                            opcode = sessionID = currentObjectID = 0;
                        }
                        else if (opcode == USERAUTH_FAILURE)
                        {
                            host->setPassword("");
                            loginFail();
                        }
                        else
                        {
                            disconnect();
                            QMessageBox::critical(0, tr("Login Failed"), tr("Logging in to ") + *host->getCommonName() + tr(" failed and I have no idea why.\nReceived unexpected response from peer."));
                            host->setPassword("");
                        }
                    }
                    else if (rv < 0 && errno != EINTR && errno != EAGAIN)
                    {
                        socketError(errno);
                        return;
                    }
                }
            }
        break;

        case OPEN:
        break;
    }
}

void guisocket::disconnect()
{
    #ifdef DEBUG_GUICONNECT         
        debug("");
        debug("guisocket::disconnect >> prepping to disconnect");
    #endif

    if (connectionState != CLOSED)
    {
        #ifdef DEBUG_GUICONNECT         
            debug("guisocket::disconnect >> we need to close this, so we are");
        #endif
        
        if (socket != NULL)
        {
            #ifdef DEBUG_GUICONNECT         
                debug("guisocket::disconnect >> \tdeleting socket");
            #endif    
            delete socket;
            socket = NULL;
        }
        
        if (readWatcher != NULL)
        {
            #ifdef DEBUG_GUICONNECT         
                debug("guisocket::disconnect >> \tdeleting readWatcher");
            #endif    
            delete readWatcher;
            readWatcher = NULL;
        }
        
        if (writeWatcher != NULL)
        {
            #ifdef DEBUG_GUICONNECT         
                debug("guisocket::disconnect >> \tdeleting writeWatcher");
            #endif    
            delete writeWatcher;
            writeWatcher = NULL;
        }
        
        if (exceptionWatcher != NULL)
        {
            #ifdef DEBUG_GUICONNECT         
                debug("guisocket::disconnect >> \tdeleting exceptionWatcher");
            #endif    
            QObject::disconnect(exceptionWatcher,SIGNAL(activated(int)));
            delete exceptionWatcher;
            exceptionWatcher = NULL;
        }

        if (socketTimer != NULL)
        {
            #ifdef DEBUG_GUICONNECT         
                debug("guisocket::disconnect >> \tdeleting socketTimer");
            #endif    
            delete socketTimer;
            socketTimer = NULL;
        }
        
        if (!messageQueue->isEmpty())
        {
            #ifdef DEBUG_GUICONNECT         
                debug("guisocket::disconnect >> \tdeleting messageQueue");
            #endif    
            messageQueue->clear();
        }
        
        connectionState = CLOSED;
        emit connectionSignal(false);
    }

    #ifdef DEBUG_GUICONNECT         
        debug("guisocket::disconnect >> finished disconnect\n");
    #endif
}
