#!/usr/local/bin/wish

# stacker.tcl - The Tape Interface
#
# Copyright (c) 1998 Mark Black
#
#
# This file contains:

# List of all the tape devices
set tapeList {}
# The currently selected stacker
set cstacker {}
# List of tape id's in the stacker
set tlist {}

#
# stackadm - Stacker Administration GUI
#            This GUI is responsible for gathering information
#            about the position of tapes in a stacker
#
proc stackadm {} {
    global errno currenthost tapeserv tapeList cstacker tlist
    
    set tapeserv "darkstar"

    if { [winfo exists .stac ] == 0 } {
	toplevel .stac -class Dialog
	wm title .stac "Stacker on $tapeserv"
	wm iconname .stac "Stacker"
	frame .stac.f1 -bd 2 -relief "raised"
	
	label .stac.f1.l1 -text "Tape Devices" 
	pack .stac.f1.l1 -fill x

	frame .stac.f1.f2 -bd 2 -relief "flat"
	scrollbar .stac.f1.f2.vscroll -orient vertical -command ".stac.f1.f2.text yview" 
	listbox .stac.f1.f2.text -selectforeground white -selectbackground #000080 -yscrollcommand {.stac.f1.f2.vscroll set } -height 3 -width 32
	.stac.f1.f2.text config -cursor hand2
	pack .stac.f1.f2.vscroll -padx 5 -side left -fill y
	pack .stac.f1.f2.text -side left 
	button .stac.f1.f2.b1 -text "Select" -foreground red -activeforeground #a00000 -command {
	    if { [.stac.f1.f2.text curselection ] != "" } {
		for { set q 0 } { $q < [llength $tapeList] } { incr q 1 } {
		    set t [lindex $tapeList $q ]
		    if { [lindex [split $t {:} ] 0 ] == [.stac.f1.f2.text get [.stac.f1.f2.text curselection ] ]} {
			set cstacker $q
			set tlist [lrange [split $t {:} ] 3 end ]
			drawStacker .stac.f2.can ts [lindex [split $t {:} ] 2 ]
			drawTapes .stac.f2.can $tlist
		    }
		}
	    }
	}
	label .stac.f1.f2.l1 -text "Select a tape device\nand drag the tapes\nin/out."
 
	pack .stac.f1.f2.b1 -padx 10 -side left
	pack .stac.f1.f2.l1 -padx 20 -side left
	pack .stac.f1.f2 -side top -fill both -expand 1

	frame .stac.f2 -bd 2 -relief "raised"
	canvas .stac.f2.can -height 400 -width 540 -yscrollcommand {.stac.f2.vscroll set } -scrollregion { 0 0 600 2000 }
	scrollbar .stac.f2.vscroll -orient vertical -command ".stac.f2.can yview"
	pack .stac.f2.can .stac.f2.vscroll -side left -expand 1 -fill both

	frame .stac.f6
	button .stac.f6.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	
	}
	button .stac.f6.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .stac
	}
	button .stac.f6.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help stacker
	}
	label .stac.f6.e1 -width 4
	label .stac.f6.e2 -width 4
	pack .stac.f6.b1 .stac.f6.b2 .stac.f6.b3 .stac.f6.e1 .stac.f6.e2 -side left -anchor w

	pack .stac.f1 .stac.f2 .stac.f6 -fill both -expand 1

	# Fill in the tape device list
	getTapeDevsList
	foreach t $tapeList {
	    .stac.f1.f2.text insert end [lindex [split $t {:} ] 0 ]
	}

	# Draw a picture of the stacker
	if { [llength $tapeList ] == 1 } {
	    drawStacker .stac.f2.can ts [lindex [split $tapeList {:} ] 2 ] $tlist
	    drawTapes .stac.f2.can
	}


	# ---------------------- Bindings --------------------------------
	bind .stac.f1.f2.text <Double-1> {
	    .stac.f1.f2.b1 invoke
	}

	bind .stac.f2.can <Any-Motion> { 
	    .stac.f6.e1 configure -text "%x" 
	    .stac.f6.e2 configure -text "%y"
	}  
	
	bind .stac.f2.can <Button-1> {
	    set curX %x
	    set curY %y
	    set oldX %x
	    set oldY %y
	}

	.stac.f2.can bind tvol <B1-Motion> {
	    moveNode [ .stac.f2.can find withtag current ] [expr %x-$curX ] [expr %y-$curY ]
	    moveNode [ .stac.f2.can find below current ] [expr %x-$curX ] [expr %y-$curY ]
	    set curX %x
	    set curY %y
	}

	.stac.f2.can bind tvoli <B1-Motion> {
	    moveNode [ .stac.f2.can find withtag current ] [expr %x-$curX ] [expr %y-$curY ]
	    moveNode [ .stac.f2.can find above current ] [expr %x-$curX ] [expr %y-$curY ]
	    set curX %x
	    set curY %y
	}
	
	.stac.f2.can bind tvoli <ButtonRelease-1> {
	    if { %y > 30 } {
		if { %x > 360 } {
		    set nx 380
		} elseif { %x < 160 } {
		    set nx 20
		} else {
		    set nx 180
		}
		set cords [ .stac.f2.can coords [ .stac.f2.can find withtag current ] ]
		set curX [lindex $cords 0 ]
		set curY [lindex $cords 1 ]
		set ny [expr "( %y - 30 ) / 26 * 26 + 30 + 20" ]
		puts stdout "Slot = [expr ( %y - 30 ) / 26 ]"
		moveNode [ .stac.f2.can find withtag current ] [expr $nx-$curX ] [expr $ny-$curY ]
		moveNode [ .stac.f2.can find above current ] [expr $nx-$curX ] [expr $ny-$curY ]
		set curX %x
		set curY %y
	    }
	}
	
	.stac.f2.can bind tvol <ButtonRelease-1> {
	    if { %y > 30 } {
		if { %x > 360 } {
		    set nx 380
		} elseif { %x < 160 } {
		    set nx 20
		} else {
		    set nx 180
		}
		set cords [ .stac.f2.can coords [ .stac.f2.can find below current ] ]
		set curX [lindex $cords 0 ]
		set curY [lindex $cords 1 ]
		set ny [expr "( %y - 30 ) / 26 * 26 + 30 + 20" ]
		moveNode [ .stac.f2.can find withtag current ] [expr $nx-$curX ] [expr $ny-$curY ]
		moveNode [ .stac.f2.can find below current ] [expr $nx-$curX ] [expr $ny-$curY ]
		set curX %x
		set curY %y
	    }
	}

    }   
}


#
# drawStacker - Draw a diagram depicting the stacker
#
proc drawStacker { win tag ntapes } {
    $win config -cursor watch
    set startX 380
    set startY 30
    set icon_size 48
    set incX 160
    set incY 26
    set cntx 0
    incr startY 20
    set tvol [ image create photo -file "../lib/tvol.gif" ]

    $win delete $tag
    $win create rectangle 340 10 520 [expr $startY + ($incY * $ntapes) ] -width 2 -fill gray -tag $tag
    $win create text [expr $startX + 50] [expr $startY - 25 ] \
	    -justify center -text "Tapedrive Slots" -tag $tag \
	    -font {-*-Helvetica-Medium-R-Bold--*-120-*-*-*-*-*-*} 
    
    for { set p 0 } { $p < $ntapes } { incr p 1 } {
	$win create text [expr $startX - 0.5 * $icon_size] $startY  \
		-justify center -text "$p" -tag $tag \
		-font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*}
	$win create rectangle 365 [expr $startY - ($incY / 2)] 515 [expr $startY + ($incY / 2)] -width 1 -tag $tag
	incr startY $incY 
    }
    $win config -cursor top_left_arrow
    return $startY
}


#
# drawTapes - Draw a diagram of all the available tapes
#             tlist = a list of the stacker tape postiions
#
proc drawTapes { win tlist } {
    $win config -cursor watch
    global tapeserv errno
    # Retrive list of available tapes
    set data [queryHost $tapeserv "TAPE names $tapeserv" 0 ]
    if { $errno == 2 } {
	puts stdout "Cannot contact host $tapeserv"
	puts stdout $data
    } elseif { $errno == 1 } {
	puts stdout $data
    } else {
	set tvol [ image create photo -file "../lib/tvol.gif" ]
	set startX 20
	set startY 20
	set icon_size 48
	set incX 160
	set incY 26
	set cntx 0
	$win delete tl tvoli tvol
	$win create text [expr $startX + 120] $startY  \
		-justify center -text "Available Tapes" -tags tl \
		-font {-*-Helvetica-Medium-R-Bold--*-120-*-*-*-*-*-*} 
	incr startY 30
	for { set p 0 } { $p < [llength $data] } { incr p 1 } {
	    set tmp [lindex $data $p ]
	    set tmp3 [split $tmp {;} ]
	    set volno [lindex $tmp3 1 ]
	    set volnm [lindex $tmp3 6 ]
	    # Check to see where the tape is located
	    set match -1
	    for { set q 0 } { $q < [llength $tlist] } { incr q 1 } {
		if { $volno == [lindex $tlist $q] } {
		    set match $q
		    break
		}
	    }
	    if { $match == -1 } {
		$win create image $startX $startY -image $tvol -tags tvoli 
		$win bind tvol <Double-1> "puts stdout {Hello}"
		$win create text [expr $startX + 12] $startY \
			-anchor w -text $volnm -tags tvol \
			-font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*}
		incr startX $incX 
		incr cntx 1
		if { $cntx == 2 } {
		    set cntx 0
		    set startX 20
		    incr startY $incY 
		}
	    } else {
		set sY [expr $p * 26 + 30 + 20 ]
		$win create image 380 $sY -image $tvol -tags tvoli 
		$win bind tvol <Double-1> "puts stdout {Hello}"
		$win create text [expr 380 + 12] $sY  \
			-anchor w -text $volnm -tags tvol \
			-font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} 
	    }
		
	}
    }
    $win config -cursor top_left_arrow
}


#
# getTapeDevs - Read in a list of the available stackers
#               Returns a list with  {Name} {Device} {Slots} {Tape List}
proc getTapeDevsList { } {
    global tapeList currenthost errno
    if { $errno == 2 } {
	puts stdout "Cannot contact host $tapeserv"
    } elseif { $errno == 1 } {
	puts stdout "Communications Error"
    } else {
	set tapeList {}
	set data [queryHost $currenthost "MON get tapedevs" 0 ]
	for { set p 0 } { $p < [llength $data] } { incr p 1 } {
	    if { [lindex $data $p ] != {} } {
		lappend tapeList [lindex $data $p ]
	    }
	}
    }
}


#
# moveNode - Move Node procedure
#
proc moveNode { node xDist yDist} {
    .stac.f2.can move $node $xDist $yDist
}





stackadm



