# matd.tcl Library
#
# Copyright (c) 1998 Mark Black
#
# This library contains the following functions:
#   taskText   - Fill the text window with the MATd config data
#   readTasks  - Read in the task list and make a 2D array containing the data
#   canMATD    - Fill the Canvas with all the monitored parameter icons
#


##################################################################################################
# taskText  - Fill the text window with the MATd config data                  TEXT WINDOW
#             w = window to display in
##################################################################################################        
proc taskText { data w type } {
    global currenthost
    . config -cursor watch
    $w config -cursor watch
    $w configure -state normal
    $w delete 0.0 end
    # MATd
    set tabbing(151) {6 13 24 }
    set toplabel(151) "ID#   Run Type     Task Type"
                                     
    set bold "-background #c0c000 -relief raised -borderwidth 1"
    set normal "-background {} -relief flat"
    set select "-underline 1"
    set tplabel $w
    append tplabel "label"
    $tplabel configure -state normal
    $tplabel delete 0.0 end
    $tplabel insert end $toplabel($type)
    $tplabel configure -state disabled

    if { $type == 151 } {
	# ------------------------------  MATd  --------------------------------
	for { set pzn 0 } { $pzn < [llength $data] } { incr pzn 1 } {
	    set tmp [lindex $data $pzn ]
	    set tmp4 [lindex $tmp 1 ]
	    set outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 0] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    set tmp4 [lindex $tmp 0 ]
	    append outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 1] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    # Parse data
	    set tmp4 [returnType $tmp ]
	    append outline [string trim $tmp4 {\{\}} ]

	    $w tag delete l$pzn
	    $w insert end "$outline\n" l$pzn
	    $w tag bind l$pzn <Any-Enter> "$w tag configure l$pzn $bold"
	    $w tag bind l$pzn <Any-Leave> "$w tag configure l$pzn $normal"
	    $w tag bind l$pzn <Button-1> "Selecter $w l$pzn"
	    $w tag bind l$pzn <Double-1>  "matdEdit $currenthost [lindex $tmp 1 ]" 
	}
	$w configure -state disabled
    }

}


##################################################################################################
# returnType  - Given a line of of the matd.conf file generate a string
#               describing the type
##################################################################################################
proc returnType { data } {
    if { [lindex $data 0 ] == "timed" } {
	# It's a Timed job, data begins at field 7
	set d 7
    } else {
	set d 3
    }
    set desc {}
    set type [lindex $data $d ]
    if { $type == "1" } {
	# Monitoring Job
	append desc "Monitor "
	set mtype [lindex $data [expr $d + 9 ]]
	append lfile [lindex $data [expr $d + 4 ]]
	if { $mtype == "1" } {
	    # RunQ
	    append desc "Run Queue."
	} elseif { $mtype == "2" } {
	    append desc "CPU Use."
	} elseif { $mtype == "3" } {
	    append desc "Disk Use on [lindex $data [expr $d + 10 ]]"
	} elseif { $mtype == "4" } {
	    append desc "Swap Use."
	} elseif { $mtype == "5" } {
	    append desc "Memory Use."
	} elseif { $mtype == "6" } {
	    append desc "Required Processes"
	} elseif { $mtype == "7" } {
	    append desc "Connectivity to [lindex $data [expr $d + 10 ]]"
	} elseif { $mtype == "8" } {
	    append desc "Network Interfaces"
	} elseif { $mtype == "9" } {
	    append desc "FTP service"
	} elseif { $mtype == "10" } {
	    append desc "Mail daemon"
	} else {
	    # Unknown job type
	    append desc "ERROR:  Unknown type!  Upgrade console"
	}
    } elseif { $type == "2" } {
	append desc "Backup"
    } elseif { $type == "3" } {
	append desc "Replicate"
    } elseif { $type == "4" } {
	append desc "Run Users Script"
    } elseif { $type == "5" } {
	append desc "Clean Log file"
    } elseif { $type == "6" } {
	append desc "Signal Remote task"
    } elseif { $type == "7" } {
	append desc "Synchronize Clock"
    } elseif { $type == "8" } {
	append desc "MATd log cleaner"
    } elseif { $type == "10" } {
	append desc "Auto Discover monitored parameters"
    } else {
	# Unknown job type
    }
    return $desc
}



##################################################################################################
# readTasks  - Read in the task list and make a 2D array containing
#              the id icon number, text files, and log file
##################################################################################################
proc readTasks { } {
    global currenthost
    set id {}
    set icons {}
    set tfield {}
    set lfile {}
    set returndata [queryHost $currenthost "TASK get tasks" 0 ]
    
    for { set pos 0 } { $pos < [llength $returndata] } { incr pos 1 } {
	set tmp [lindex $returndata $pos ]
	if { [lindex $tmp 0 ] == "timed" } {
	    # It's a Timed job, data begins at field 7
	    set d 7
	} else {
	    set d 3
	}
	append id "[lindex $tmp 1 ] "
	# Find out the type of job
	set type [lindex $tmp $d ]
	lappend lfile [lindex $tmp [expr $d + 4 ]]

	if { $type == "1" } {
	    # Monitoring Job
	    set mtype [lindex $tmp [expr $d + 9 ]]
	    if { $mtype == "1" } {
		# RunQ
		lappend tfield {Run Queue}
		append icons "156 "
		# Logins
		lappend tfield Logins
		append icons "504 "
		lappend lfile [lindex $tmp [expr $d + 4 ]]
		append id "[lindex $tmp 1 ] "
	    } elseif { $mtype == "2" } {
		# CPU use
		lappend tfield {CPU Use}
		append icons "154 "
	    } elseif { $mtype == "3" } {
		# Disk use
		set tmp "Disk Use on\n[lindex $tmp [expr $d + 10 ]]"
		lappend tfield $tmp
		append icons "155 "
	    } elseif { $mtype == "4" } {
		# Swap Use
		lappend tfield {Swap Use}
		append icons "152 "
	    } elseif { $mtype == "5" } {
		# Memory
		lappend tfield {Memory Use}
		append icons "153 "
	    } elseif { $mtype == "6" } {
		# Processes
		lappend tfield "Required\nProcesses"
		append icons "157 "
	    } elseif { $mtype == "7" } {
		# Connection
		set tmp "Connectivity\nto [lindex $tmp [expr $d + 10 ]]"
		lappend tfield $tmp
		append icons "158 "
	    } elseif { $mtype == "8" } {
		# Network Interfaces
		set rdata [split [queryHost $currenthost "TASK instance 8" 0 ] ":" ] 
		for { set p 0 } { $p < [lindex $rdata 0] } { incr p 1 } {
		    lappend tfield "Interface\n[lindex $rdata [expr $p + 1]]"
		    append icons "506 "
		    lappend lfile [lindex $tmp [expr $d + 4 ]]
		    append id "[lindex $tmp 1 ] "
		}
		# Remove one of the entries from lfile, and id, because it is added later
		set tmp [lrange $id 0 [expr [llength $id] - 2 ] ]
		set id "$tmp "
		set tmp [lrange $lfile 0 [expr [llength $lfile] - 2 ] ]
		set lfile $tmp
	    } elseif { $mtype == "9" } {
		# FTP daemon
		lappend tfield "FTP daemon"
		append icons "507 "
	    } elseif { $mtype == "10" } {
		# Sendmail
		lappend tfield "EMail"
		append icons "508 "
	    } else {
		# Unknown job type
	    }

	} elseif { $type == "2" } {
	    # Backup Job
	    lappend tfield {Backup}
	    append icons "505 "
	} elseif { $type == "3" } {
	    # Replicate Job
	    lappend tfield {Replicate}
	    append icons "510 "
	} elseif { $type == "4" } {
	    # User Script Job
	    lappend tfield "User Script"
	    append icons "503 "
	} elseif { $type == "5" } {
	    # Cleanup Job
	    lappend tfield {Clean Log}
	    append icons "502 "
	} elseif { $type == "6" } {
	    # Informer Job
	    lappend tfield {-}
	    append icons "159 "
	} elseif { $type == "7" } {
	    # Time Sync Job
	    lappend tfield {-}
	    append icons "159 "
	} elseif { $type == "8" } {
	    # MATD log cleaner
	    lappend tfield "MATd cleaner"
	    append icons "501 "
	} elseif { $type == "10" } {
	    # MATD log cleaner
	    lappend tfield "Discovery"
	    append icons "509 "
	} else {
	    # Unknown job type
	    lappend tfield {-}
	    append icons "159 "
 	}
    }
    set out {}
    lappend out $id
    lappend out $icons
    lappend out $tfield
    lappend out $lfile
    #puts stdout "Out = \n $out"
    return $out
}


##################################################################################################
# canMATD - Build the Display for the MATd task canvas                  CANVAS WINDOW
#
##################################################################################################
proc canMATD {} {
    global currenthost
    global level
    global browse
    set browse 151
    global tapeserv
    .tp.title configure -text "Host($currenthost) --> Config --> MATd "
    set level 150

    # Enable the accelerators
    #bind .tp.host.canvas <Button-3> "rclick .tp.host.canvas"

    set fig(151) [ image create photo -file "../lib/crontab.gif" ]    
    # Matd tasks
    set fig(152) [ image create photo -file "../lib/swap.gif" ]       
    # Swap space
    set fig(153) [ image create photo -file "../lib/mem.gif" ]        
    # Memory space
    set fig(154) [ image create photo -file "../lib/cpu.gif" ]        
    # CPU use
    set fig(155) [ image create photo -file "../lib/diskuse.gif" ]    
    # Disk use
    set fig(156) [ image create photo -file "../lib/runque.gif" ]     
    # Run Queue
    set fig(157) [ image create photo -file "../lib/rprocs.gif" ]     
    # Required Processes
    # Network Connectivity
    set fig(158) [ image create photo -file "../lib/connect.gif" ]
    set fig(159) [ image create photo -file "../lib/sudo40.gif" ]
    set fig(501) [ image create photo -file "../lib/matdclean.gif" ]
    set fig(502) [ image create photo -file "../lib/cleanlog.gif" ]
    set fig(503) [ image create photo -file "../lib/userscrpt.gif" ]
    set fig(504) [ image create photo -file "../lib/userlogs.gif" ]
    set fig(505) [ image create photo -file "../lib/sched.gif" ]
    set fig(506) [ image create photo -file "../lib/moneth.gif" ]
    set fig(507) [ image create photo -file "../lib/monftpd.gif" ]
    set fig(508) [ image create photo -file "../lib/monsmtp.gif" ]
    set fig(509) [ image create photo -file "../lib/discover.gif" ]
    set fig(510) [ image create photo -file "../lib/replicate.gif" ]
    set fig(0) [ image create photo -file "../lib/back.gif" ]        
    set startX 40
    set startY 40
    set icon_size 48
    set incX 90
    set incY 90
    set cntx 0
    set node [ .tp.host.canvas create image $startX $startY -image $fig(0) -tags bk ]
    .tp.host.canvas bind bk <Double-1> "repack -20"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "<- Back" -tags bk \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

    set node [ .tp.host.canvas create image $startX $startY -image $fig(151) -tags "MATd" ]
    .tp.host.canvas bind "MATd" <Double-1> "set browse 151 ;repack 151"
    .tp.host.canvas bind "MATd" <Button-3> "rClickMatd .tp.host.canvas 150"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "MATd" -tags "MATd" \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1
    
    # Draw icons for each monitored parameter
    set rdata [ readTasks ]
    set ids [lindex $rdata 0 ]
    set icons [lindex $rdata 1 ]
    set tfield [lindex $rdata 2 ]
    set logf [lindex $rdata 3 ]
    
    for { set pos 0 } { $pos < [llength $icons] } { incr pos 1 } {
	set itype [lindex $icons $pos ]
	set nlabel [lindex $tfield $pos ]
	set id [lindex $ids $pos ]
	set logfile [string trim [lindex $logf $pos ]]
	if { $nlabel != "-" } {
	    set node [ .tp.host.canvas create image $startX $startY -image $fig($itype) -tags t$pos ]
	    .tp.host.canvas bind t$pos <Button-3> "rClickMatd .tp.host.canvas $id"
	    .tp.host.canvas bind t$pos <Double-1> "monOut \{ $logfile \} $itype \{ $nlabel \}"
	    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
		    -justify center -text "$nlabel" -tags t$pos \
		    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
	    incr startX $incX 
	    if { $cntx == 5 } {
		set startX 40
		incr startY $incY 
		set cntx -1
	    }
	    incr cntx 1
	}
    }
    incr startY $incY 
    set sregion { 0 0 500 }
    lappend sregion $startY
    .tp.host.canvas configure -scrollregion $sregion
    
}


#
# rClickMatd  - Right-Click accelerators for MATd
#
# USE:   rClickMatd {window} {jobid}
#
proc rClickMatd { w id } {
    set mpos [winfo pointerxy $w ]
    if {[string match $w [eval winfo contain $mpos ]]} {
	# Cursor on window
	set n .rcmatd
	if ![winfo exists $n ] {
	    toplevel $n
	    wm override $n 1
	    wm withdraw $n
	    text $n.t -width 10 -height 2 -relief raised
	    pack $n.t
	}
	update idle

	# Create the text box entries and bindings
	set cnt 0
	set bold "-background #c0c000 -relief raised -borderwidth 1"
	set normal "-background {} -relief flat"
	global currenthost
	global currentdomain
	global browse
	if { $id != 150 } {
	    set labels {New Modify}
	    foreach val $labels {
		$n.t insert end "$val\n" tag$cnt
		$n.t tag bind tag$cnt <Any-Enter> "$n.t tag configure tag$cnt $bold"
		$n.t tag bind tag$cnt <Any-Leave> "$n.t tag configure tag$cnt $normal"
		incr cnt 1 
	    }
	    $n.t tag bind tag0 <Button-1> "matdEdit $currenthost -9999"
	    $n.t tag bind tag1 <Button-1> "matdEdit $currenthost $id"
	} else {
	    # This is the MATd icon
	    set labels {New List Discover Status}
	    foreach val $labels {
		$n.t insert end "$val\n" tag$cnt
		$n.t tag bind tag$cnt <Any-Enter> "$n.t tag configure tag$cnt $bold"
		$n.t tag bind tag$cnt <Any-Leave> "$n.t tag configure tag$cnt $normal"
		incr cnt 1 
	    }
	    $n.t tag bind tag0 <Button-1> "matdEdit $currenthost -9999"
	    $n.t tag bind tag1 <Button-1> "set browse 151 ;repack 151"
	    $n.t tag bind tag2 <Button-1> "matdDiscover $currenthost"
	    $n.t tag bind tag3 <Button-1> "matdStatus $currenthost"
	}

	$n.t configure -height [llength $labels]
	bind $n <Any-Leave> "destroy $n"
	set x [expr [lindex $mpos 0 ] - 10 ]
	set y [expr [lindex $mpos 1 ] - 10 ]
	wm geometry $n +$x+$y
	wm deiconify $n
	raise $n
    }
}


##################################################################################################
# monOut - This function outputs the monitored values in an 
#          appropriate form.
#          itype  - The Icon type, from which we can deduce how to deal with this parameter
#          nlabel - The name label for the monitored parameter
#          NOTE:  Data is passed using the upgrade buffer 
##################################################################################################
proc monOut { logfile itype nlabel args } {
    if { [string length $args ] != 0 } {
	append nlabel " $args"
    }
    global currenthost
    global errno
    global upgrade_buff

    set upgrade_buff {}
    set retval [queryHost $currenthost "get raw [string trim $logfile ]" 0 ]
    
    if { $errno == 2 } {
	error 550
	puts stdout "Cannot contact host $currenthost"
	puts stdout $retval
    } elseif { $errno == 1 } {
	error 550
	puts stdout $retval
    } else {
	# remove the last line if it is blank
	if { [string length [lindex $retval end]] < 1 } {
	    set upgrade_buff [ lreplace $retval [expr [llength $retval ] - 2 ] end]
	} else {
	    set upgrade_buff $retval
	}
	if { $itype == 155 } {
	    drawGraph .g1 $currenthost $nlabel {Day/Time} 100.0 0 {"Disk Use" {[expr 100 * [lindex $raw 5] / ( [lindex $raw 5] + [lindex $raw 6] )] }} 
	} elseif { $itype == 152 } {
	    drawGraph .g1 $currenthost $nlabel {Day/Time} 100.0 0 {"Swap Use" {[expr 100 * [lindex $raw 6] / [lindex $raw 5] ] }}
	} elseif { $itype == 153 } {
	    drawGraph .g1 $currenthost $nlabel {Day/Time} 100.0 0 {"Memory Use" {[expr 100 * [lindex $raw 6] / [lindex $raw 5] ] }}
	} elseif { $itype == 154 } {
	    drawGraph .g1 $currenthost $nlabel {Day/Time} 100.0 0 {"Total CPU Use" {[expr [lindex $raw 4] + [lindex $raw 5]]}} {"System CPU Use" {[lindex $raw 5]}} {"User CPU Use" {[lindex $raw 4]}} 
	} elseif { $itype == 156 } {
	    drawGraph .g1 $currenthost $nlabel {Day/Time} 2.0 0 {"15 Min Run Queue" {[lindex $raw 7]}} {"5 Min Run Queue" {[lindex $raw 6]}} {"Run Queue" {[lindex $raw 5]}} 
	} elseif { $itype == 504 } {
	    drawGraph .g1 $currenthost $nlabel {Day/Time} 20 0 {"Users" {[lindex $raw 4]}} 
	} elseif { $itype == 506 } {
	    # The data to graph depends on the interface
	    set iface [lrange $nlabel end end ]
	    set tmp [lindex $upgrade_buff 0 ]
	    set tmp2 [split $tmp {:}]
	    set loc 5
	    for { set pzn 5 } { $pzn < [llength $tmp2 ] } { incr pzn 1 } {
		if { $iface == [split [lindex $tmp2 $pzn ] {,} ] } {
		    set loc $pzn
		}
	    }
	    if { $loc == 5 } {
		drawGraph .g1 $currenthost $nlabel {Day/Time} 20000.0 0.0 {"Collisions" {[lindex $raw 10]}} {"RX" {[lindex $raw 6]}} {"TX" {[lindex $raw 8]}} {"RX errors" {[lindex $raw 7]}} {"TX errors" {[lindex $raw 9]}}
	    } elseif { $loc == 11 } {
		drawGraph .g1 $currenthost $nlabel {Day/Time} 20000.0 0.0 {"Collisions" {[lindex $raw 16]}} {"RX" {[lindex $raw 12]}} {"TX" {[lindex $raw 14]}} {"RX errors" {[lindex $raw 13]}} {"TXerr" {[lindex $raw 15]}}
	    } elseif { $loc == 17 } {
		drawGraph .g1 $currenthost $nlabel {Day/Time} 20000.0 0.0 {"Collisions" {[lindex $raw 22]}} {"RX" {[lindex $raw 18]}} {"TX" {[lindex $raw 20]}} {"RX errors" {[lindex $raw 19]}} {"TXerr" {[lindex $raw 21]}}
		} elseif { $loc == 23 } {
		drawGraph .g1 $currenthost $nlabel {Day/Time} 20000.0 0.0 {"Collisions" {[lindex $raw 28]}} {"RX" {[lindex $raw 24]}} {"TX" {[lindex $raw 26]}} {"RX errors" {[lindex $raw 25]}} {"TXerr" {[lindex $raw 27]}}
	    }
	} else {
	    # Text return (No graph)
	    listData $nlabel 
	}
    }
}


#
# listData - List the Boolean type data in a text window
#
proc listData { nlabel } {
    global upgrade_buff
    . config -cursor watch
    .general.text config -cursor watch
    .general.text configure -state normal
    .general.text delete 0.0 end
    .general.label configure -text $nlabel
    .general.textlabel configure -state normal
    .general.textlabel delete 0.0 end
    .general.textlabel insert end "Date  Time    Output"
    .general.textlabel configure -state disabled
    set tabbing {6 8 24}

    for { set pzn 0 } { $pzn < [llength $upgrade_buff] } { incr pzn 1 } {
	.general.text tag delete l$pzn
	set tmp [lindex $upgrade_buff $pzn ]
	set tmp2 "[lindex [split $tmp {:} ] 0 ]:[lindex [split $tmp {:} ] 1 ]"
	set outline $tmp2
	for { set cnt 0 } { $cnt < ([lindex $tabbing 0] -[string length $tmp2 ]) } { incr cnt 1 } {
	    append outline " "
	}
	set tmp2 "[lindex [split $tmp {:} ] 2 ]:[lindex [split $tmp {:} ] 3 ]"
	append outline $tmp2
	for { set cnt 0 } { $cnt < ([lindex $tabbing 1] -[string length $tmp2 ]) } { incr cnt 1 } {
	    append outline " "
	}
	set tmp2 [string trim [lrange [split $tmp {:} ] 5 end] {\{\}} ]
	append outline "$tmp2\n"
	.general.text insert end $outline l$pzn
	if { [lindex [split $tmp {:} ] 4 ] == "0" } {
	    .general.text tag configure l$pzn -foreground #a00000
	} else {
	    .general.text tag configure l$pzn -foreground #0000a0
	}
    }
    .general.text configure -state disabled
    . config -cursor top_left_arrow
    .general.text config -cursor xterm
}



##################################################################################################
# matdEdit - GUI for editing the MATd.conf file
#
# Set jobid to -9999 for a new task
##################################################################################################
proc matdEdit { currenthost jobid } {
    global errno upgrade_buff
    set host $currenthost
    if { [winfo exists .matd ] == 0 } {
	toplevel .matd -class Dialog
	wm title .matd "Task Editor: $currenthost"
	wm iconname .matd "Task Editor"
	entry .matd.hidden1 
	entry .matd.hidden2
	entry .matd.h3
	entry .matd.h4
	# NOT GOOD
	.matd.hidden1 insert end $currenthost
	.matd.hidden2 insert end $jobid
	.matd.h3 insert end "0"
	.matd.h4 insert end "0"

	frame .matd.f1 -bd 2 -relief "raised"
	label .matd.f1.l1 -text "Task ID#:" -width 14
	entry .matd.f1.e1 -width 24
	pack .matd.f1.l1 .matd.f1.e1 -fill both -side left -expand 1

	frame .matd.f2 -bd 2 -relief "raised"
	button .matd.f2.b1 -text "Run Condition" -padx 0 -width 14 -command {
	    set tmp [ .matd.h3 get ]
	    set upgrade_buff -9999
	    getRunC $tmp
	    tkwait window .matdrc
	    if { $upgrade_buff != {0} } {
		.matd.f2.e1 configure -state normal
		.matd.h3 delete 0 end
		.matd.h3 insert end $upgrade_buff
		.matd.f2.e1 delete 0 end
		.matd.f2.e1 insert end [ lindex $upgrade_buff 0 ]
		.matd.f2.e1 configure -state disabled
	    }
	    # puts stdout "Vwait done:  Return Value = [ .matd.h3 get ]"
	}
	entry .matd.f2.e1 -width 24
	pack .matd.f2.b1 .matd.f2.e1 -fill both -side left -expand 1

	frame .matd.f3 -bd 2 -relief "raised"
	button .matd.f3.b1 -text "Job Type" -padx 0 -width 14 -command {
	    getJobT [.matd.h4 get ]
	    tkwait window .matdjt
	    if { $upgrade_buff != {0} } {
		jobTypeDisplay [lindex $upgrade_buff 0]
		.matd.h4 delete 0 end
		.matd.h4 insert end $upgrade_buff
	    }
	}
	entry .matd.f3.e1 -width 24
	pack .matd.f3.b1 .matd.f3.e1 -fill both -side left -expand 1

	frame .matd.f31 -bd 2 -relief "raised"
	label .matd.f31.l1 -text "Timeout (s)" -width 14
	entry .matd.f31.e1 -width 24
	pack .matd.f31.l1 .matd.f31.e1 -fill both -side left -expand 1

	frame .matd.f4 -bd 2 -relief "raised"
	label .matd.f4.l1 -text "Run As UID:" -width 14
	entry .matd.f4.e1 -width 24
	pack .matd.f4.l1 .matd.f4.e1 -fill both -side left -expand 1
	.matd.f4.e1 delete 0 end
	.matd.f4.e1 insert end "0"

	frame .matd.f5 -bd 2 -relief "raised"
	label .matd.f5.l1 -text "Log to File:" -width 14
	entry .matd.f5.e1 -width 24
	pack .matd.f5.l1 .matd.f5.e1 -fill both -side left -expand 1
	
	frame .matd.f6 -bd 2 -relief "raised"
	label .matd.f6.l1 -text "Lines to retain" -width 14
	entry .matd.f6.e1 -width 24
	pack .matd.f6.l1 .matd.f6.e1 -fill both -side left -expand 1

	frame .matd.f10 -bd 2 -relief "flat"
	button .matd.f10.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    set out "[lindex [.matd.h3 get] 0 ] [.matd.f1.e1 get] [lrange [.matd.h3 get] 1 end] "
	    append out "[lindex [.matd.h4 get] 0 ] [.matd.f31.e1 get] [.matd.f4.e1 get] 0 "
	    append out "[.matd.f5.e1 get ] [.matd.f6.e1 get] [lrange [.matd.h4 get] 1 end]"
	    if { [.matd.hidden2 get] == -9999 } {
		set cmd "TASK add "
	    } else {
		set cmd "TASK mod [.matd.hidden2 get] "
	    }
	    append cmd $out

	    puts "CMD = $cmd"

	    set returndata [queryHost $currenthost "$cmd" 1 ]
	    if { $errno != 0 } {
		error 91
		puts stdout "ERROR:  Unable to update the /etc/group file on $currenthost\n       Check the files permissions"
	    } else {
		repack 151
		destroy .matd
	    }
	}
	button .matd.f10.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .matd
	}
	button .matd.f10.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help matd
	}
	pack .matd.f10.b1 .matd.f10.b2 .matd.f10.b3 -side left -fill x
	# UID of process is disabled, until MATd is ready
	#pack .matd.f1 .matd.f2 .matd.f3 .matd.f31 .matd.f4 .matd.f5 .matd.f6 .matd.f10 -fill both -side top -expand 1
	pack .matd.f1 .matd.f2 .matd.f3 .matd.f31 .matd.f5 .matd.f6 .matd.f10 -fill both -side top -expand 1
    } else {
	# Window exists
	.matd.h3 delete 0 end
	.matd.h3 insert end "0"
	.matd.h4 delete 0 end
	.matd.h4 insert end "0"
	.matd.hidden1 delete 0 end
	.matd.hidden2 delete 0 end
	.matd.hidden1 insert end $currenthost
	.matd.hidden2 insert end $jobid
    }
    
    .matd.f1.e1 delete 0 end
    .matd.f2.e1 delete 0 end
    .matd.f3.e1 delete 0 end
    .matd.f31.e1 delete 0 end
    #.matd.f4.e1 delete 0 end
    .matd.f5.e1 delete 0 end
    .matd.f6.e1 delete 0 end

    # Fill the entries
    if { $jobid != -9999 } {
	.matd.f1.e1 insert end [ .matd.hidden2 get ]
	# Read data from file
	set data [ getTaskdata [ .matd.hidden1 get ] [ .matd.hidden2 get ] ]
	.matd.f2.e1 configure -state normal
	.matd.f2.e1 delete 0 end
	.matd.f2.e1 insert end [ lindex $data 0 ]
	.matd.f2.e1 configure -state disabled
	if { [lindex $data 0 ] == "timed" } {
	# It's a Timed job, data begins at field 7
	    set d 7
	} else {
	    set d 3
	}
	.matd.f3.e1 insert end [lindex $data [expr $d ]]
	.matd.f31.e1 insert end [lindex $data [expr $d + 1 ]]
	.matd.f4.e1 delete 0 end
	.matd.f4.e1 insert end [lindex $data [expr $d + 2 ]]
	.matd.f5.e1 insert end [lindex $data [expr $d + 4 ]]
	.matd.f6.e1 insert end [lindex $data [expr $d + 5 ]]
	.matd.h3 delete 0 end
	.matd.h3 insert end "[ lindex $data 0 ] "
	.matd.h3 insert end "[lrange $data 2 [expr $d - 1 ] ]"
	set type "[lindex $data $d] "
	append type [lrange $data [expr $d + 6 ] end ]
	.matd.h4 delete 0 end
	.matd.h4 insert end "$type"
	jobTypeDisplay [lindex $data $d]
    }
}

#
# jobTypeDisplay - Given the job type display the appropriate descriptor
#
proc jobTypeDisplay { tp } {
    if { $tp == 1 } {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "Monitor"
	.matd.f3.e1 configure -state disabled
    } elseif { $tp == 2 } {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "Backup"
	.matd.f3.e1 configure -state disabled
    } elseif { $tp == 3 } {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "Replicate"
	.matd.f3.e1 configure -state disabled
    } elseif { $tp == 4 } {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "User Script"
	.matd.f3.e1 configure -state disabled
    } elseif { $tp == 5 } {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "Clean Log"
	.matd.f3.e1 configure -state disabled
    } elseif { $tp == 8 } {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "MATd Log Cleaner"
	.matd.f3.e1 configure -state disabled
    } elseif { $tp == 10 } {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "MATd Discovery"
	.matd.f3.e1 configure -state disabled
    } else {
	.matd.f3.e1 configure -state normal
	.matd.f3.e1 delete 0 end
	.matd.f3.e1 insert end "Undefined!"
	.matd.f3.e1 configure -state disabled
    }
}


##################################################################################################
# getTaskdata - Given a jobid fetch the appropriate job
#               data
##################################################################################################
proc getTaskdata { currenthost jobid } {
    global errno
    set returndata [queryHost $currenthost "TASK get tasks" 0 ]
    for { set pos 0 } { $pos < [llength $returndata] } { incr pos 1 } {
	set tmp [lindex $returndata $pos ]
	if { $jobid == [lindex $tmp 1] } {
	    return $tmp
	}
    }
    return 9999
}


##################################################################################################
# getRunC - Gather the the run condition information
# NOTE:  This function returns data in .matd.h3
##################################################################################################
proc getRunC { time } {
    global errno
    if { [winfo exists .matdrc ] == 0 } {
	toplevel .matdrc -class Dialog
	wm title .matdrc "Task Time Editor"
	wm iconname .matdrc "Task Time Editor"
	grab set .matdrc 
	entry .matdrc.h1
	.matdrc.h1 delete 0 end
	frame .matdrc.f1 -relief raised -bd 2
	radiobutton .matdrc.f1.interval -text "Interval" -variable ttype -value 1 -command {
	    if { $ttype == 1 } {
		pack forget .matdrc.f2
		pack forget .matdrc.f10
		pack .matdrc.f3 .matdrc.f10 -side top -fill x
	    } elseif { $ttype == 0 } {
		pack forget .matdrc.f3
		pack forget .matdrc.f10
		pack .matdrc.f2 .matdrc.f10 -side top -fill x
	    }
	}
	radiobutton .matdrc.f1.timed -text "Timed" -variable ttype -value 0 -command {
	    if { $ttype == 1 } {
		pack forget .matdrc.f2
		pack forget .matdrc.f10
		pack .matdrc.f3 .matdrc.f10 -side top -fill x
	    } elseif { $ttype == 0 } {
		pack forget .matdrc.f3
		pack forget .matdrc.f10
		pack .matdrc.f2 .matdrc.f10 -side top -fill x
	    }
	}
	pack .matdrc.f1.interval .matdrc.f1.timed -side left -expand 1 -fill x
	frame .matdrc.f2 -relief flat -bd 0
	
	frame .matdrc.f2.f3 -bd 2 -relief "raised"
	menubutton .matdrc.f2.f3.l1 -text "Minutes:" -width 12 -menu .matdrc.f2.f3.l1.m -relief raised
	menu .matdrc.f2.f3.l1.m -tearoff 0 
	for { set cnt 0 } { $cnt < 61 } { incr cnt 5 } {
	    .matdrc.f2.f3.l1.m add command -label $cnt -command ".matdrc.f2.f3.e1 insert end \"$cnt,\" "
	}
	entry .matdrc.f2.f3.e1 -width 20
	pack .matdrc.f2.f3.l1 .matdrc.f2.f3.e1 -fill x -side left -expand 1 -pady 4

	frame .matdrc.f2.f4 -bd 2 -relief "raised"
	menubutton .matdrc.f2.f4.l1 -text "Hours:" -width 12 -menu .matdrc.f2.f4.l1.m -relief raised
	menu .matdrc.f2.f4.l1.m -tearoff 0 
	set hours {{1} {2} {3} {4} {5} {6} {7} {8} {9} {10} {11} {12} {13} {14} {15} {16} {17} {18} {19} {20} {21} {22} {23} {Any}}
	for { set cnt 0 } { $cnt < 24 } { incr cnt 1 } {
	    .matdrc.f2.f4.l1.m add command -label [lindex $hours $cnt] -command ".matdrc.f2.f4.e1 insert end \"[lindex $hours $cnt],\" "
	}
	entry .matdrc.f2.f4.e1 -width 20
	pack .matdrc.f2.f4.l1 .matdrc.f2.f4.e1 -fill x -side left -expand 1 -pady 4

	frame .matdrc.f2.f5 -bd 2 -relief "raised"
	menubutton .matdrc.f2.f5.l1 -text "Month Day:" -width 12 -menu .matdrc.f2.f5.l1.m -relief raised
	menu .matdrc.f2.f5.l1.m -tearoff 0 
	set monthdays {{1} {2} {3} {4} {5} {6} {7} {8} {9} {10} {11} {12} {13} {14} {15} {16} {17} {18} {19} {20} {21} {22} {23} {24} {25} {26} {27} {28} {29} {30} {31} {Any}}
	for { set cnt 0 } { $cnt < 32 } { incr cnt 1 } {
	    .matdrc.f2.f5.l1.m add command -label [lindex $monthdays $cnt ] -command ".matdrc.f2.f5.e1 insert end \"[lindex $monthdays $cnt ],\" "
	}
	entry .matdrc.f2.f5.e1 -width 20
	pack .matdrc.f2.f5.l1 .matdrc.f2.f5.e1 -fill x -side left -expand 1 -pady 4

	frame .matdrc.f2.f6 -bd 2 -relief "raised"
	menubutton .matdrc.f2.f6.l1 -text "Month:" -width 12 -menu .matdrc.f2.f6.l1.m -relief raised
	menu .matdrc.f2.f6.l1.m -tearoff 0 
	set months {{Jan} {Feb} {Mar} {Apr} {May} {Jun} {Jul} {Sep} {Oct} {Nov} {Dec} {Any}}
	for { set cnt 0 } { $cnt < 12 } { incr cnt 1 } {
	    .matdrc.f2.f6.l1.m add command -label [lindex $months $cnt ] -command ".matdrc.f2.f6.e1 insert end \"[lindex $months $cnt ],\" "
	}
	entry .matdrc.f2.f6.e1 -width 20
	pack .matdrc.f2.f6.l1 .matdrc.f2.f6.e1 -fill x -side left -expand 1 -pady 4

	frame .matdrc.f2.f7 -bd 2 -relief "raised"
	menubutton .matdrc.f2.f7.l1 -text "Week Day:" -width 12 -menu .matdrc.f2.f7.l1.m -relief raised
	menu .matdrc.f2.f7.l1.m -tearoff 0 
	set weekday {{Sun} {Mon} {Tue} {Wed} {Thu} {Fri} {Sat} {Any}}
	for { set cnt 0 } { $cnt < 8 } { incr cnt 1 } {
	    .matdrc.f2.f7.l1.m add command -label [lindex $weekday $cnt ] -command ".matdrc.f2.f7.e1 insert end \"[lindex $weekday $cnt ],\" "
	}
	entry .matdrc.f2.f7.e1 -width 20
	pack .matdrc.f2.f7.l1 .matdrc.f2.f7.e1 -fill x -side left -expand 1 -pady 4
	pack .matdrc.f2.f3 .matdrc.f2.f4 .matdrc.f2.f5 .matdrc.f2.f6 .matdrc.f2.f7 -side top -fill both

	frame .matdrc.f3 -bd 0 -relief "flat"
	frame .matdrc.f3.f1 -bd 2 -relief "raised"
	menubutton .matdrc.f3.f1.l1 -text "Interval (min):" -width 12 -menu .matdrc.f3.f1.l1.m -relief raised
	menu .matdrc.f3.f1.l1.m -tearoff 0 
	set hours {{1} {2} {3} {4} {5} {6} {7} {8} {9} {10} {15} {20} {30} {45} {1h} {1.5h} {2h} {3h} {4h} {6h} {12h} {24h} {48h}}
	for { set cnt 0 } { $cnt < 23 } { incr cnt 1 } {
	    .matdrc.f3.f1.l1.m add command -label [lindex $hours $cnt] -command ".matdrc.f3.f1.e1 delete 0 end ; .matdrc.f3.f1.e1 insert end \"[lindex $hours $cnt]\" "
	}
	entry .matdrc.f3.f1.e1 -width 20
	pack .matdrc.f3.f1.l1 .matdrc.f3.f1.e1 -fill x -side left -expand 1 -pady 4
	pack .matdrc.f3.f1 -fill x -side left -expand 1

	frame .matdrc.f10 -bd 2 -relief "raised"
	button .matdrc.f10.b1 -text "Apply" -foreground red -activeforeground #a00000 \
		-command {
	    global upgrade_buff 
	    grab release .matdrc
	    set upgrade_buff [ makeNewTime $ttype ]
	    destroy .matdrc
	}
	button .matdrc.f10.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    global upgrade_buff
	    grab release .matdrc
	    set upgrade_buff 0
	    after 500 [ destroy .matdrc ]
	}
	pack .matdrc.f10.b1 .matdrc.f10.b2 -anchor w -side left 

	pack .matdrc.f1 .matdrc.f10 -side top -fill x
    } else {
	.matdrc.h1 delete 0 end
    }
    .matdrc.h1 insert end $time
    .matdrc.f3.f1.e1 delete 0 end
    .matdrc.f2.f3.e1 delete 0 end 
    .matdrc.f2.f4.e1 delete 0 end
    .matdrc.f2.f5.e1 delete 0 end
    .matdrc.f2.f6.e1 delete 0 end
    .matdrc.f2.f7.e1 delete 0 end

    # Populate the entries
    if { $time != "0" } {
	if { [lindex [ .matdrc.h1 get ] 0 ] == "interval" } {
	    .matdrc.f1.interval invoke
	    .matdrc.f3.f1.e1 insert end [lindex [ .matdrc.h1 get ] 1 ]
	} else {
	    .matdrc.f1.timed invoke
	    .matdrc.f2.f3.e1 insert end [lindex [ .matdrc.h1 get ] 1 ]
	    set tmp [lindex [ .matdrc.h1 get ] 2 ]
	    if { $tmp == "*" } {
		set tmp "Any,"
	    }
	    .matdrc.f2.f4.e1 insert end $tmp
	    set tmp [lindex [ .matdrc.h1 get ] 3 ]
	    if { $tmp == "*" } {
		set tmp "Any,"
	    }
	    .matdrc.f2.f5.e1 insert end $tmp
	    set tmp [lindex [ .matdrc.h1 get ] 4 ]
	    if { $tmp == "*" } {
		set tmp "Any,"
	    }
	    .matdrc.f2.f6.e1 insert end $tmp
	    set tmp [lindex [ .matdrc.h1 get ] 5 ]
	    if { $tmp == "*" } {
		set tmp "Any,"
	    }
	    .matdrc.f2.f7.e1 insert end $tmp
	}
    }
}


##################################################################################################
# makeNewTime - given the new run conditions build the new run conditions 
#               variable, and return it to the calling function.
##################################################################################################
proc makeNewTime { ttype } {
    if { $ttype == 1 } {
	# It's an interval function
	set intv [.matdrc.f3.f1.e1 get ]
	if { [string range $intv end end ] == "h" } {
	    set sec [ expr [string trimright $intv "h" ] * 60]
	    return "interval $sec"
	} else {
	    set sec "$intv"
	    return "interval $sec"
	}
    } else {
	# It's a timed job
	set nmin [string trimright [.matdrc.f2.f3.e1 get ] "," ]
	set nhour [string trimright [.matdrc.f2.f4.e1 get ] "," ]
	foreach v [split $nhour {,} ] {
	    if { $v == "Any" } {
		set thour "*"
		break
	    } else {
		append thour "[expr $v - 1 ],"
	    }
	}
	set nhour [string trimright $thour "," ]

	set nmday [string trimright [.matdrc.f2.f5.e1 get ] "," ]
	foreach v [split $nmday {,} ] {
	    if { $v == "Any" } {
		set nmday "*"
		break
	    }
	}

	set months {{Jan} {Feb} {Mar} {Apr} {May} {Jun} {Jul} {Sep} {Oct} {Nov} {Dec}}
	set tmths [string trimright [.matdrc.f2.f6.e1 get ] "," ]
	set nmnths {}
	foreach v [split $tmths {,} ] {
	    if { $v == "Any" } {
		set nmnths "*"
		break
	    }
	    for { set p 0 } { $p < [llength $months] } { incr p 1 } {
		if { [lindex $months $p ] == $v } {
		    append nmnths "$p,"
		}
	    }
	}
	set nmnth [string trimright $nmnths "," ]

	set weekday {{Sun} {Mon} {Tue} {Wed} {Thu} {Fri} {Sat} {Any}}
	set twday [string trimright [.matdrc.f2.f7.e1 get ] "," ]
	set nwdays {}
	foreach v [split $twday {,} ] {
	    if { $v == "Any" } {
		set nwdays "*"
		break
	    }
	    for { set p 0 } { $p < [llength $weekday] } { incr p 1 } {
		if { [lindex $weekday $p ] == $v } {
		    append nwdays "$p,"
		}
	    }
	}
	set nwday [string trimright $nwdays "," ]

	return "timed $nmin $nhour $nmday $nmnth $nwday"
    }
}


##################################################################################################
# getJobT - Gather the the Job Type information
##################################################################################################
proc getJobT { type } {
    if { [winfo exists .matdjt ] == 0 } {
	toplevel .matdjt -class Dialog
	wm title .matdjt "Task Type Editor"
	wm iconname .matdjt "Task Type Editor"
	grab set .matdjt
	global currenthost

	frame .matdjt.f1 -relief raised -bd 2
	entry .matdjt.h1
	# Hidden entry for the setected tape id:  .matdjt.h2
	entry .matdjt.h2

	# Don't remember what this was for!!!
	entry .matdjt.h3
	entry .matdjt.h4

	.matdjt.h1 delete 0 end
	.matdjt.h1 insert end $type
	.matdjt.h3 delete 0 end
	.matdjt.h3 insert end "0"

	set jtype 0
	radiobutton .matdjt.f1.t1 -text "Monitor Parameter" -variable jtype -value 1 -command "matdjtHelp \$jtype"
	radiobutton .matdjt.f1.t2 -text "Backup" -variable jtype -value 2 -command "matdjtHelp \$jtype"
	radiobutton .matdjt.f1.t3 -text "Replicate" -variable jtype -value 3 -command "matdjtHelp \$jtype"
	radiobutton .matdjt.f1.t4 -text "User Script" -variable jtype -value 4 -command "matdjtHelp \$jtype"
	radiobutton .matdjt.f1.t5 -text "Clean Log" -variable jtype -value 5 -command "matdjtHelp \$jtype"
	#radiobutton .matdjt.f1.t6 -text "Sync Time" -variable jtype -value 7 -command "matdjtHelp \$jtype"
	radiobutton .matdjt.f1.t7 -text "MATd log cleaner" -variable jtype -value 8 -command "matdjtHelp \$jtype"
	radiobutton .matdjt.f1.t8 -text "Discover" -variable jtype -value 10 -command "matdjtHelp \$jtype"

	pack .matdjt.f1.t1 .matdjt.f1.t4 .matdjt.f1.t2 .matdjt.f1.t3 .matdjt.f1.t5 .matdjt.f1.t7 .matdjt.f1.t8 -side top -anchor w
	#pack .matdjt.f1.t1 .matdjt.f1.t4 .matdjt.f1.t5 .matdjt.f1.t2 -side top -anchor w

	# Monitoring Frame
	frame .matdjt.f2 -relief flat -bd 0
	frame .matdjt.f2.f1 -relief raised -bd 2
	label .matdjt.f2.f1.l1 -text "Monitoring job"
	pack .matdjt.f2.f1.l1 -fill x 
	pack .matdjt.f2.f1 -fill x 

	frame .matdjt.f2.f2 -relief raised -bd 2
	frame .matdjt.f2.f2.f1 -relief raised -bd 0
	menubutton .matdjt.f2.f2.f1.l1 -text "Monitor:" -width 17 -menu .matdjt.f2.f2.f1.l1.m -relief raised
	menu .matdjt.f2.f2.f1.l1.m -tearoff 0 
	# The jobs are defined as follows:
	#define MONrunq   0x00000001       /* Note this also gets the number of users */
	#define MONcpuuse 0x00000002
	#define MONdisk   0x00000003
	#define MONswap   0x00000004
	#define MONmemory 0x00000005
	#define MONprocs  0x00000006       /* Monitor the processes that should run on this host.  If not running alert */
	#define MONconn   0x00000007       /* Monitor the hosts connectivity to the network.  If not running alert */
	#define MONeths   0x00000008       /* Monitor the network interfaces */
	#define MONftp    0x00000009       /* Monitor the FTP service */
	#define MONsmtp   0x0000000A       /* Monitor the SMTP service */

	# ***************  NOTE:  Any changes to the vals or optdata arrays MUST be accompanied with changes to indx2type and type2index
	
	set vals {{Run Queue} {CPU Use} {Disk Use} {Processes} {Connectivity} {NIC collisions} {FTP daemon} {SMTP mail} {Swap Use} {Memory}}
	set optdata {{-} {-} {Filesystem}  {Process List} {Gateway's IP} {-} {-} {-} {-} {-} {-} }
	for { set cnt 0 } { $cnt < 10 } { incr cnt 1 } {
	    .matdjt.f2.f2.f1.l1.m add command -label [lindex $vals $cnt] \
		    -command "monGuiConf $cnt"
	}
	entry .matdjt.f2.f2.f1.e1 -width 16
	pack .matdjt.f2.f2.f1.l1 .matdjt.f2.f2.f1.e1 -side left -anchor w -fill x -padx 2
	
	frame .matdjt.f2.f2.f2 -relief raised -bd 0
	label .matdjt.f2.f2.f2.l1 -text "Warning Threshold:" -width 18
	button .matdjt.f2.f2.f2.b1 -text "<=" -width 2 -command {
	    if { [.matdjt.f2.f2.f2.b1 cget -text] == {<=} } {
		.matdjt.f2.f2.f2.b1 configure -text {>=}
	    } else {
		.matdjt.f2.f2.f2.b1 configure -text {<=}
	    }
	}
	entry .matdjt.f2.f2.f2.e1 -width 10
	pack .matdjt.f2.f2.f2.l1 .matdjt.f2.f2.f2.b1 .matdjt.f2.f2.f2.e1 -side left -anchor w -fill x

	frame .matdjt.f2.f2.f3 -relief raised -bd 0
	label .matdjt.f2.f2.f3.l1 -text "Warning Script:" -width 18
	entry .matdjt.f2.f2.f3.e1 -width 16
	pack .matdjt.f2.f2.f3.l1 .matdjt.f2.f2.f3.e1 -side left -anchor w -fill x

	frame .matdjt.f2.f2.f4 -relief raised -bd 0
	menubutton .matdjt.f2.f2.f4.l1 -text "Warn After #" -width 17 -menu .matdjt.f2.f2.f4.l1.m -relief raised
	menu .matdjt.f2.f2.f4.l1.m -tearoff 0 
	for { set cnt 0 } { $cnt < 32 } { incr cnt 1 } {
	    .matdjt.f2.f2.f4.l1.m add command -label $cnt -command ".matdjt.f2.f2.f4.e1 delete 0 end ;.matdjt.f2.f2.f4.e1 insert end $cnt"
	}
	entry .matdjt.f2.f2.f4.e1 -width 16
	pack .matdjt.f2.f2.f4.l1 .matdjt.f2.f2.f4.e1 -side left -anchor w -fill x -padx 2

	frame .matdjt.f2.f2.f5 -relief raised -bd 0
	label .matdjt.f2.f2.f5.l1 -text "Filesystem:" -width 18
	entry .matdjt.f2.f2.f5.e1 -width 16
	pack .matdjt.f2.f2.f5.l1 .matdjt.f2.f2.f5.e1 -side left -anchor w -fill x

	pack .matdjt.f2.f2.f1 .matdjt.f2.f2.f2 .matdjt.f2.f2.f3 .matdjt.f2.f2.f4 .matdjt.f2.f2.f5 -side top 
	pack .matdjt.f2.f1 .matdjt.f2.f2 -fill x -side top 

	# Backup Frame
	frame .matdjt.f3 -relief flat -bd 0
	frame .matdjt.f3.f1 -relief raised -bd 2
	label .matdjt.f3.f1.l1 -text "Backup Filesystems"
	pack .matdjt.f3.f1.l1 -fill x 
	frame .matdjt.f3.f2 -relief raised -bd 2
	menubutton .matdjt.f3.f2.l1 -text "Job:" -width 18 -menu .matdjt.f3.f2.l1.m -relief raised
	menu .matdjt.f3.f2.l1.m -tearoff 0 
	set rdata [queryHost $currenthost "TAPE get jobids" 0 ]
	foreach th $rdata {
	    .matdjt.f3.f2.l1.m add command -label [lindex [split $th ":" ] 6 ] -command "mkTapeMenu [lindex [split $th {:} ] 0 ]"
	}
	entry .matdjt.f3.f2.e1 -width 16 -state disabled
	pack .matdjt.f3.f2.l1 .matdjt.f3.f2.e1 -side left -anchor w -fill x
	pack .matdjt.f3.f1 .matdjt.f3.f2 -fill x 

	# Replication Frame
	frame .matdjt.f4 -relief flat -bd 0
	frame .matdjt.f4.f1 -relief raised -bd 2
	label .matdjt.f4.f1.l1 -text "Replicate Directory"
	pack .matdjt.f4.f1.l1 -fill x 
	frame .matdjt.f4.f2 -relief raised -bd 2
	menubutton .matdjt.f4.f2.l1 -text "Job:" -width 18 -menu .matdjt.f4.f2.l1.m -relief raised
	menu .matdjt.f4.f2.l1.m -tearoff 0 
	set rdata [queryHost $currenthost "TAPE get repids" 0 ]
	foreach th $rdata {
	    .matdjt.f4.f2.l1.m add command -label [lindex [split $th ":" ] 9 ] -command "mkRepMenu [lindex [split $th {:} ] 0 ]"
	}
	entry .matdjt.f4.f2.e1 -width 16 -state disabled
	pack .matdjt.f4.f2.l1 .matdjt.f4.f2.e1 -side left -anchor w -fill x
	pack .matdjt.f4.f1 .matdjt.f4.f2 -fill x 
	pack .matdjt.f4.f1 -fill x 

	# User Script Frame
	frame .matdjt.f5 -relief flat -bd 0
	frame .matdjt.f5.f1 -relief raised -bd 2
	label .matdjt.f5.f1.l1 -text "Run User Script"
	pack .matdjt.f5.f1.l1 -fill x 
	frame .matdjt.f5.f2 -relief raised -bd 2
	frame .matdjt.f5.f2.f2 -relief raised -bd 0
	label .matdjt.f5.f2.f2.l1 -text "Script Filename:" -width 18
	entry .matdjt.f5.f2.f2.e1 -width 16
	pack .matdjt.f5.f2.f2.l1 .matdjt.f5.f2.f2.e1 -side left -anchor w -fill x
	pack .matdjt.f5.f2.f2 -fill x -side top
	pack .matdjt.f5.f1 .matdjt.f5.f2 -fill x 

	# Cleaner Frame
	frame .matdjt.f6 -relief flat -bd 0
	frame .matdjt.f6.f1 -relief raised -bd 2
	label .matdjt.f6.f1.l1 -text "Clean Log File"
	pack .matdjt.f6.f1.l1 -fill x 
	frame .matdjt.f6.f2 -relief raised -bd 2
	frame .matdjt.f6.f2.f2 -relief raised -bd 0
	label .matdjt.f6.f2.f2.l1 -text "Log Filename:" -width 18
	entry .matdjt.f6.f2.f2.e1 -width 16
	pack .matdjt.f6.f2.f2.l1 .matdjt.f6.f2.f2.e1 -side left -anchor w -fill x

	frame .matdjt.f6.f2.f3 -relief raised -bd 0
	label .matdjt.f6.f2.f3.l1 -text "Max # Lines:" -width 18
	entry .matdjt.f6.f2.f3.e1 -width 16
	pack .matdjt.f6.f2.f3.l1 .matdjt.f6.f2.f3.e1 -side left -anchor w -fill x
	pack .matdjt.f6.f2.f2 .matdjt.f6.f2.f3 -fill x -side top
	pack .matdjt.f6.f1 .matdjt.f6.f2 -fill x 


	# Time Sync Frame
	frame .matdjt.f7 -relief flat -bd 0
	frame .matdjt.f7.f1 -relief raised -bd 2
	label .matdjt.f7.f1.l1 -text "Synchronize Time"
	pack .matdjt.f7.f1.l1 -fill x 
	pack .matdjt.f7.f1 -fill x

	# MATd log cleaner
	frame .matdjt.f8 -relief flat -bd 0
	frame .matdjt.f8.f1 -relief raised -bd 2
	label .matdjt.f8.f1.l1 -text "MATd cleaner"
	pack .matdjt.f8.f1.l1 -fill x 
	pack .matdjt.f8.f1 -fill x

	# MATd Discovery
	frame .matdjt.f9 -relief flat -bd 0
	frame .matdjt.f9.f1 -relief raised -bd 2
	label .matdjt.f9.f1.l1 -text "MATd Discovery"
	pack .matdjt.f9.f1.l1 -fill x 
	pack .matdjt.f9.f1 -fill x

	frame .matdjt.f10 -bd 2 -relief "raised"
	button .matdjt.f10.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    grab release .matdjt
	    set okflag 0
	    set retval "$jtype "
	    if { $jtype == 1 } {
		# Monitoring Jobs:    WarnFilename  Warn_Thres  Gflags  Monitor_What  {Optional Data}
		if { [.matdjt.f2.f2.f3.e1 get ] != {} } {
		    append retval "[.matdjt.f2.f2.f3.e1 get ] "
		    set okflag 1
		} 
		if { [.matdjt.f2.f2.f2.e1 get ] != {} } {
		    append retval "[.matdjt.f2.f2.f2.e1 get ] "
		    set okflag [expr $okflag ? 1 : 0 ]
		} 
		if { [.matdjt.f2.f2.f4.e1 get ] != {} } {
		    set wv [.matdjt.f2.f2.f4.e1 get ]
		    if { [.matdjt.f2.f2.f2.b1 cget -text] == {<=} } {
			append retval "$wv "
		    } else {
			append retval "[expr $wv + 32] "
		    }
		    set okflag [expr $okflag ? 1 : 0 ]
		}
		# *************************************************  Change this when adding new parameters  *****************************************************************
		set indx2type {1 2 3 6 7 8 9 10 4 5}
		append retval "[lindex $indx2type [.matdjt.h4 get ]]"

		if { [.matdjt.f2.f2.f4.e1 get ] != {} } {
		    # Optional data section
		    append retval " [.matdjt.f2.f2.f5.e1 get ]"
		} 

	    } elseif { $jtype == 2 } {
		if { [.matdjt.f3.f2.e1 get ] != {} & [.matdjt.h3 get ] != {} } {
		    append retval "[.matdjt.f3.f2.e1 get ] [.matdjt.h3 get ] /bin/false"
		    set okflag 1
		} 
	    } elseif { $jtype == 3 } {
		if { [.matdjt.f4.f2.e1 get ] != {} & [.matdjt.h3 get ] != {} } {
		    append retval "[.matdjt.f4.f2.e1 get ] [.matdjt.h3 get ] /bin/false"
		    set okflag 1
		} 
	    } elseif { $jtype == 4 } {
		if { [.matdjt.f5.f2.f2.e1 get ] != {} } {
		    append retval "[.matdjt.f5.f2.f2.e1 get ]"
		    set okflag 1
		} 
	    } elseif { $jtype == 5 } {
		if { [.matdjt.f6.f2.f2.e1 get ] != {} & [.matdjt.f6.f2.f3.e1 get ] != {} } {
		    append retval "[.matdjt.f6.f2.f2.e1 get ] "
		    append retval "[.matdjt.f6.f2.f3.e1 get ]"
		    set okflag 1
		}
	    } elseif { $jtype == 8 } {
		set okflag 1
	    } elseif { $jtype == 10 } {
		set okflag 1
	    } else {
		puts stdout "Job type not yet supported "
	    }

	    if { $okflag == 1 } {
		global upgrade_buff
		set upgrade_buff $retval
		destroy .matdjt
	    } else {
		error 551
	    }
	}
	button .matdjt.f10.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    grab release .matdjt
	    global upgrade_buff
	    set upgrade_buff 0
	    destroy .matdjt
	}
	pack .matdjt.f10.b1 .matdjt.f10.b2 -anchor w -side left 

	pack .matdjt.f1 .matdjt.f10 -side top -fill x
    } else {
	# window exists!
	.matdjt.h1 delete 0 end
	.matdjt.h1 insert end $type
    }

    # Fill the appropriate entries
    set tp [lindex $type 0 ] 
    if { $tp == 1 } {
	# It's a Monitoring Job
	.matdjt.f1.t1 invoke
	set jtype [ expr [lindex $type 4 ] - 1 ]
	set type2index {99 0 1 2 99 99 3 4 5 6 7 99 99}
	monGuiConf [lindex [split $type2index { } ] [lindex $type 4 ] ]
	.matdjt.f2.f2.f3.e1 delete 0 end
	.matdjt.f2.f2.f3.e1 insert end [lindex $type 1 ]
	.matdjt.f2.f2.f2.e1 delete 0 end
	.matdjt.f2.f2.f2.e1 insert end [lindex $type 2 ]
	.matdjt.f2.f2.f4.e1 delete 0 end
	.matdjt.f2.f2.f4.e1 insert end [expr [lindex $type 3 ] & 31 ]
	if { [expr [lindex $type 3 ] & 32 ] == 32 } {
	    .matdjt.f2.f2.f2.b1 configure -text {>=}
	} else {
	    .matdjt.f2.f2.f2.b1 configure -text {<=}
	}
	.matdjt.f2.f2.f5.e1 delete 0 end
	.matdjt.f2.f2.f5.e1 insert end [lindex $type 5 ]

    } elseif { $tp == 2 } {
	# Backup Job
	.matdjt.f1.t2 invoke
	.matdjt.f3.f2.e1 configure -state normal
	.matdjt.f3.f2.e1 delete 0 end
	.matdjt.f3.f2.e1 insert end [lindex $type 1 ]
	.matdjt.f3.f2.e1 configure -state disabled
    } elseif { $tp == 3 } {
	# Replicate Job
	.matdjt.f1.t3 invoke
	.matdjt.f4.f2.e1 configure -state normal
	.matdjt.f4.f2.e1 delete 0 end
	.matdjt.f4.f2.e1 insert end [lindex $type 1 ]
	.matdjt.f4.f2.e1 configure -state disabled
    } elseif { $tp == 4 } {
	# User Script
	.matdjt.f1.t4 invoke
	.matdjt.f5.f2.f2.e1 delete 0 end
	.matdjt.f5.f2.f2.e1 insert end [lindex $type 1 ]
    } elseif { $tp == 5 } {
	# Clean Log
	.matdjt.f1.t5 invoke
	.matdjt.f6.f2.f2.e1 delete 0 end
	.matdjt.f6.f2.f2.e1 insert end [lindex $type 1 ]
	.matdjt.f6.f2.f3.e1 delete 0 end
	.matdjt.f6.f2.f3.e1 insert end [lindex $type 2 ]
    } elseif { $tp == 8 } {
	# User Script
	.matdjt.f1.t7 invoke
    } elseif { $tp == 10 } {
	# User Script
	.matdjt.f1.t8 invoke
    } 
}


#
# mkTapeMenu - construct the tape menu, and fill entries
#
proc mkTapeMenu { ts } {
    global tapeserv
    set tapeserv $ts
    .matdjt.f3.f2.e1 configure -state normal
    .matdjt.f3.f2.e1 delete 0 end
    .matdjt.f3.f2.e1 insert end $tapeserv
    .matdjt.f3.f2.e1 configure -state disabled
}

#
# mkRepMenu - construct the replicateion menu, and fill entries
#
proc mkRepMenu { ts } {
    global tapeserv
    set tapeserv $ts
    .matdjt.f4.f2.e1 configure -state normal
    .matdjt.f4.f2.e1 delete 0 end
    .matdjt.f4.f2.e1 insert end $tapeserv
    .matdjt.f4.f2.e1 configure -state disabled
}


#
# getFNames - get the names of files and directories to backup
#
proc getFNames { } {
    grab release .matdjt
    global currenthost
    fileSelect $currenthost [.matdjt.h2 get] 1
}


#
# getTapeNames - get a list of tape id and tape name pairs
#                
proc getTapeNames { tapeserv } {
    if { $tapeserv != {} } {
	set data [queryHost $tapeserv "TAPE names $tapeserv" 0 ]
	set out {}
	for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	    set tmp [lindex $data $position ]
	    set tmp3 [split $tmp {;} ]
	    lappend out "[lindex $tmp3 1 ] [lindex $tmp3 6 ]"
	}
	return $out
    } else {
	error 2004
	return {}
    }
}


#
# monGuiConf - configures the Job Type monitoring section
#              USE:  monGuiConf type
#              type = index of $vals array used to build the menu button which calls this function
#
proc monGuiConf { index } {
    # *********** NOTE:  If the order of these is changed, the if's expressions have to be changed
    set vals {{Run Queue} {CPU Use} {Disk Use} {Processes} {Connectivity} {NIC collisions} {FTP daemon} {SMTP mail} {Swap Use} {Memory}}
    set optdata {{-} {-} {Filesystem}  {Process List} {Gateway's IP} {-} {-} {-} {-} {-} {-}}
    .matdjt.f2.f2.f5.l1 configure -text "[lindex $optdata $index]"
    .matdjt.h4 delete 0 end
    .matdjt.h4 insert end $index

    # Disable the bottom entry 
    if { $index == 0 | $index == 1 | $index == 5 | $index == 6 | $index == 7 } {
	.matdjt.f2.f2.f5.e1 configure -state normal
	.matdjt.f2.f2.f5.e1 delete 0 end
	.matdjt.f2.f2.f5.e1 configure -state disabled
    } else {
	.matdjt.f2.f2.f5.e1 configure -state normal
    }
    .matdjt.f2.f2.f1.e1 configure -state normal
    .matdjt.f2.f2.f1.e1 delete 0 end
    .matdjt.f2.f2.f1.e1 insert end "[lindex $vals $index]"
    .matdjt.f2.f2.f1.e1 configure -state disabled

    # Put in the default threshold
    if { $index == 3 | $index == 4 | $index == 6 | $index == 7 } {
	.matdjt.f2.f2.f2.e1 delete 0 end
	.matdjt.f2.f2.f2.e1 insert end "0"
	.matdjt.f2.f2.f2.e1 configure -state disabled
	.matdjt.f2.f2.f2.b1 configure -text {<=}
    } else {
	.matdjt.f2.f2.f2.e1 configure -state normal
	.matdjt.f2.f2.f2.e1 delete 0 end
    }
}


##################################################################################################
# matdjtHelp - Window repacker for matdjt
##################################################################################################
proc matdjtHelp { jtype } {
    pack forget .matdjt.f2 .matdjt.f3 .matdjt.f4 .matdjt.f5 .matdjt.f6 .matdjt.f7 .matdjt.f8 .matdjt.f9 .matdjt.f10
    if { $jtype == 1 } {
	pack .matdjt.f2 .matdjt.f10 -side top -fill x
    } elseif { $jtype == 2 } {
	pack .matdjt.f3 .matdjt.f10 -side top -fill x
    } elseif { $jtype == 3 } {
	pack .matdjt.f4 .matdjt.f10 -side top -fill x
    } elseif { $jtype == 4 } {
	pack .matdjt.f5 .matdjt.f10 -side top -fill x
    } elseif { $jtype == 5 } {
	pack .matdjt.f6 .matdjt.f10 -side top -fill x
    } elseif { $jtype == 7 } {
	pack .matdjt.f7 .matdjt.f10 -side top -fill x
    } elseif { $jtype == 8 } {
	pack .matdjt.f8 .matdjt.f10 -side top -fill x
    } elseif { $jtype == 10 } {
	pack .matdjt.f9 .matdjt.f10 -side top -fill x
    } else {
	puts stdout "Unknown Type $jtype"
    }
}


##################################################################################################
# matdStatus  - Check the status of the MATd jobs
##################################################################################################
proc matdStatus { host } {
    global currenthost
    set returndata [queryHost $currenthost "TASK get status" 1 ]

}

##################################################################################################
# matdDiscover  - Discover all the types of monitored parameters available for a host
##################################################################################################
proc matdDiscover { host } {
    global currenthost

    set returndata [queryHost $currenthost "TASK discover 1" 1 ]
}