# Lams.tcl Library
#
# Copyright (c) 1997, 1998, 2000 Mark Black
#
# Rebuild index using "auto_mkindex /home/mark/mat/lib *.tcl"
#
# This library contains the following functions:
#   AutoProbe  - Ping the hosts to see if they respond
#   MatHost    - Add/Delete Hosts from the MAT hosts file
#   MatUser    - Add/Delete MAt users, and set the permissions
#   MatLic     - Add/Delete MAT licenses
#   Sudo       - Sudo update GUI
#   DnsClient  - DNS Clint GUI
#   gethostent - Get a hosts entry from the MAT hosts file
#   -------------- Task Procedures ----------------

set DEBUG 1

proc AutoProbe { } {
    if { [winfo exists .mprobe ] == 0 } {
	# Window does not exist:  Create it
	toplevel .mprobe -class Dialog
	wm title .mprobe "MAT Probe"
	wm iconname .mprobe "MAT Probe"
	global active_hosts
	set slen [llength $active_hosts ]
	label .mprobe.l1 -text "Probing hosts" 
	scale .mprobe.scale -orient horizontal -length 250 -from 0 -to $slen \
	-command "" -tickinterval 50
	button .mprobe.b1 -text "Cancel" -command "global currentProbe ;set currentProbe 9999 ;destroy .mprobe"
	pack .mprobe.l1 .mprobe.scale .mprobe.b1 -padx 10
	AutoProbe2
    } else {
	# Window exists
	global currentProbe
	.mprobe.scale set $currentProbe
	AutoProbe2
    }
}

#
# AutoProbe - Ping the hosts to see if they respond
#             The main Host display canvas uses the hostname
#             as the tag, so use the tag to modify the hosts
#             icon.
#             Set currentProbe to 9999 to stop it
proc AutoProbe2 { } {
    global currentProbe
    if { $currentProbe != 9999 } {
	# Get name of host to probe
	global active_hosts
	set probeHost [ lindex [split [lindex $active_hosts $currentProbe ] ":" ] 0 ]
	set tp [ lindex [split [lindex $active_hosts $currentProbe ] ":" ] 1 ]
	if { $tp != "cont" } {
	    puts stdout "Probing $probeHost"
	    set dataline [lindex $active_hosts $currentProbe ]
	    set pingval [pingHost $probeHost ]
	    if { $pingval == 0 } {
		# Ping failed
		if { [lindex [split $dataline ":" ] 4 ] != 0 } {
		    set t [split $dataline ":" ]
		    set dataline "[lindex $t 0 ]:[lindex $t 1 ]:[lindex $t 2 ]:[lindex $t 3 ]:0:"
		    append dataline "[lindex $t 5 ]:[lindex $t 6 ]:"
		    set newlist [ lreplace $active_hosts $currentProbe $currentProbe $dataline ]
		    set active_hosts $newlist
		} 
		global level
		if { $level == 10 } {
		    set hostoff [ image create photo -file "../lib/host_off.gif" ]
		    # Due to a bug in wish this has to be in a catch statement
		    catch ".tp.host.canvas itemconfigure $probeHost -image $hostoff"
		}
	    } else {
		set hosttype [lindex [split $pingval ] 2 ]
		if { [lindex [split $dataline ":" ] 1 ] != $hosttype } {
		    set t [split $dataline ":" ]
		    set dataline "[lindex $t 0 ]:$hosttype:[lindex $t 2 ]:[lindex $t 3 ]:"
		    append dataline "1:[lindex $t 5 ]:[lindex $t 6 ]:"
		    set active_hosts [ lreplace $active_hosts $currentProbe $currentProbe $dataline ]
		}
	    }
	}
	# Start another
	after 200 { AutoProbe }
	incr currentProbe 1
	if { $currentProbe > [expr [llength $active_hosts ] - 1] } {
	    set currentProbe 9999
	    set fid [ open "../hosts" w ]
	    foreach line $active_hosts {
		puts $fid "$line"
	    }
	    close $fid
	}
    } else {
	destroy .mprobe
	fillCan 10
    }
}


#
# MatHost - Procedure for updating the MAT host file
#           If an argument is given then modify that host    
# The hosts file is resident on the management station
#
proc MatHost { {edithost "-" } args } {
    set host_list {}
    if { [string length $args ] != 0 } {
	append edithost " $args"
    }
    if { [winfo exists .mhst ] == 0 } {
	toplevel .mhst -class Dialog
	wm title .mhst "MAT Hosts Editor"
	wm iconname .mhst "MAT Hosts Editor"
	entry .mhst.hidden1
	.mhst.hidden1 insert end $edithost

	# Define main frames 
	frame .mhst.f1 -bd 2 -relief "flat"
	frame .mhst.f2 -bd 2 -relief "flat"
	frame .mhst.f5 -bd 2 -relief "flat"

	frame .mhst.f10 -bd 0 -relief flat
	pack .mhst.f10
	frame .mhst.f10.f1 -bd 2 -relief flat
	grid .mhst.f10.f1
	button .mhst.hb1


	button .mhst.f10.f1.b1 -relief groove -text "General" -command {
	    set c1 [ .mhst.hb1 cget -activebackground ]
	    set c2 [ .mhst.hb1 cget -bg ]
	    .mhst.f10.f1.b1 configure -relief flat -bg $c2 -activebackground $c1
	    .mhst.f10.f1.b2 configure -relief groove -bg $c1 -activebackground $c2
	    pack forget .mhst.f1 .mhst.f2 .mhst.f5
	    pack .mhst.f1 .mhst.f5 -anchor w -fill x
	}
	button .mhst.f10.f1.b2 -relief groove -text "Properties" -command {
	    set c1 [ .mhst.hb1 cget -activebackground ]
	    set c2 [ .mhst.hb1 cget -bg ]
	    .mhst.f10.f1.b1 configure -relief groove -bg $c1 -activebackground $c2
	    .mhst.f10.f1.b2 configure -relief flat -bg $c2 -activebackground $c1
	    pack forget .mhst.f1 .mhst.f2 .mhst.f5
	    pack .mhst.f2 .mhst.f5 -anchor w -fill x
	}
	grid .mhst.f10.f1.b1 .mhst.f10.f1.b2
	.mhst.f10.f1.b1 invoke 

	# General Frame
	frame .mhst.f1.f1 -bd 2 -relief "groove"
	label .mhst.f1.f1.l1 -text "Name:"
	entry .mhst.f1.f1.e1 -width 16
	pack .mhst.f1.f1.l1 .mhst.f1.f1.e1 -fill both -side left -expand 1

	frame .mhst.f1.f2 -bd 2 -relief "groove"
	label .mhst.f1.f2.l1 -text "Type:" 
	radiobutton .mhst.f1.f2.b1 -text "Box" -variable cntn -value 1 -command " \
		pack forget .mhst.f2.f1 .mhst.f2.f2 ; \
		pack .mhst.f2.f2 -fill both -expand 1 " 
	radiobutton .mhst.f1.f2.b2 -text "Host" -variable cntn -value 0 -command " \
		pack forget .mhst.f2.f1 .mhst.f2.f2 ; \
		pack .mhst.f2.f1 .mhst.f2.f2 -fill both -expand 1 " 
	
	pack .mhst.f1.f2.l1 .mhst.f1.f2.b1 .mhst.f1.f2.b2 -side top -anchor w

	pack .mhst.f1.f1 .mhst.f1.f2 -side top -fill both -expand 1

	frame .mhst.f2.f1 -bd 2 -relief "groove"
	label .mhst.f2.f1.l1 -text "Services offered by Host:"
	checkbutton .mhst.f2.f1.ck1 -text "DNS Master" -onvalue 1 -offvalue 0 -variable DNSyes
	checkbutton .mhst.f2.f1.ck2 -text "NIS Master" -onvalue 1 -offvalue 0 -variable NISyes
	checkbutton .mhst.f2.f1.ck3 -text "NFS Server" -onvalue 1 -offvalue 0 -variable NFSyes
	checkbutton .mhst.f2.f1.ck4 -text "Replicate Server" -onvalue 1 -offvalue 0 -variable REPyes
	checkbutton .mhst.f2.f1.ck5 -text "Backup Server" -onvalue 1 -offvalue 0 -variable TAPEyes
	checkbutton .mhst.f2.f1.ck6 -text "User Defined" -onvalue 1 -offvalue 0 -variable USER1yes
	pack .mhst.f2.f1.l1 .mhst.f2.f1.ck1 .mhst.f2.f1.ck2 .mhst.f2.f1.ck3 .mhst.f2.f1.ck4 .mhst.f2.f1.ck5 .mhst.f2.f1.ck6 -anchor w
	
	frame .mhst.f2.f2 -bd 2 -relief "groove"
	label .mhst.f2.f2.l0 -text "- Optional -"
	label .mhst.f2.f2.l1 -text "X position:"
	entry .mhst.f2.f2.e1 -width 5
	label .mhst.f2.f2.l2 -text "Y position:"
	entry .mhst.f2.f2.e2 -width 5
	label .mhst.f2.f2.l3 -text "Container:"
	entry .mhst.f2.f2.e3 -width 5
	grid .mhst.f2.f2.l0 -columnspan 2
	grid .mhst.f2.f2.l1 .mhst.f2.f2.e1
	grid .mhst.f2.f2.l2 .mhst.f2.f2.e2
	grid .mhst.f2.f2.l3 .mhst.f2.f2.e3

	pack .mhst.f2.f1 .mhst.f2.f2 -fill both -expand 1 

	# Bottom frame
	button .mhst.f5.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [ .mhst.f1.f1.e1 get ] != {} } {
		if { [ .mhst.hidden1 get ] == "-" } {
		    # This is a new one
		    # Data format is:    hostname:OS:Container:Services:status:Xposition:Yposition:
		    if { $cntn == 0 } {
			set c {}
			if { $DNSyes == 1 } { append c "DNS," }
			if { $NISyes == 1 } { append c "NIS," }
			if { $NFSyes == 1 } { append c "NFS," }
			if { $REPyes == 1 } { append c "REP," }
			if { $TAPEyes == 1 } { append c "TAPE," }
			if { $USER1yes == 1 } { append c "USER1" }
			global box
			set pingval [pingHost [ .mhst.f1.f1.e1 get ] ]
			if { $pingval == 0 } {
			    # Ping failed
			    set cmd "[ .mhst.f1.f1.e1 get ]::$box:"
			    append cmd "$c"
			    append cmd ":0:::"
			} else {
			    set hosttype [lindex [split $pingval ] 2 ]
			    set cmd "[ .mhst.f1.f1.e1 get ]:"
			    append cmd "$hosttype"
			    append cmd ":$box:$c"
			    append cmd ":1:::"
			}
			global mb_hosts
			set fid [ open $mb_hosts a ]
			puts $fid $cmd
			close $fid
		    } else {
			# It's a box
			# Containname:cont:this-level:next-level:0:Xposition:Yposition:
			global box active_hosts
			
			# Replace spaces
			set th [ .mhst.f1.f1.e1 get ]
			set cp [string first { } $th ]
			while { $cp > 0 } {
			    set t1 [string range $th 0 [ expr $cp - 1 ] ]
			    set t2 [string range $th [ expr $cp + 1 ] end ]
			    set th "$t1\_$t2" ;
			    set cp [string first { } $th ]
			} 

			set c "$th:cont:$box:"
			# Scan through the active_hosts for a free box
			# NOTE: The "-" is required below, because of the search algorithm
			set avl "-abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0-"
			set used {}
			foreach line $active_hosts {
			    set t [split $line {:} ]
			    if { [lindex $t 1 ] == "cont" && [lindex $t 2 ] == $box } {
				lappend used [string range [lindex $t 3 ] end end ]
			    }
			}
			puts stdout "Used = $used"
			# Now look for a free letter
			foreach ltr $used {
			    set p [string first $ltr $avl ]
			    set t1 [string range $avl 0 [ expr $p - 1 ] ]
			    set t2 [string range $avl [ expr $p + 1 ] end ]
			    set avl "$t1$t2" ;
			    puts stdout "New avl = $avl"
			}
			set nf [string range $avl 1 1 ]
			append c "$box$nf:0:"
			if { [.mhst.f2.f2.e1 get ] != {} } {
			    append c "[.mhst.f2.f2.e1 get ]"
			}
			append c ":"
			if { [.mhst.f2.f2.e1 get ] != {} } {
			    append c "[.mhst.f2.f2.e2 get ]"
			}
			append c ":"
			global mb_hosts
			set fid [ open $mb_hosts a ]
			puts $fid $c
			close $fid
		    } 
		} else {
		    # Get the hosts original data
		    if { $cntn == 0 } {
			# NOT a Container
			set host_list {}
			set fid [ open $mb_hosts r ]
			while { [ gets $fid fout ] > 0 } {
			    lappend host_list $fout
			}
			close $fid
			set edit_host [ lindex [split [ .mhst.hidden1 get ] ":" ] 0 ]
			# Write out new file
			set fid [ open $mb_hosts w ]
			foreach data $host_list {
			    if { $edit_host == [lindex [ split $data ":" ] 0 ] } {
				# This is the one to replace
				set t [ split $data ":" ]
				set cmd "[ .mhst.f1.f1.e1 get ]:[lindex $t 1 ]:"
				if { [.mhst.f2.f2.e1 get ] != {} } {
				    append cmd "[.mhst.f2.f2.e3 get ]:"
				} else {
				    append cmd "[lindex $t 2 ]:"
				}
				if { $DNSyes == 1 } { append cmd "DNS," }
				if { $NISyes == 1 } { append cmd "NIS," }
				if { $NFSyes == 1 } { append cmd "NFS," }
				if { $REPyes == 1 } { append cmd "REP," }
				if { $TAPEyes == 1 } { append cmd "TAPE," }
				if { $USER1yes == 1 } { append cmd "USER1" }
				append cmd ":[lindex $t 4 ]:"
				if { [.mhst.f2.f2.e1 get ] != {} } {
				    append cmd "[.mhst.f2.f2.e1 get ]:"
				} else {
				    append cmd "[lindex $t 5 ]:"
				}
				if { [.mhst.f2.f2.e1 get ] != {} } {
				    append cmd "[.mhst.f2.f2.e2 get ]:"
				} else {
				    append cmd "[lindex $t 6 ]:"
				}
				puts $fid $cmd
			    } else {
				puts $fid $data
			    }
			}
			close $fid
		    } else {
			# It's a box:  Only change the name and or position
			# Containname:cont:this-level:next-level:0:Xposition:Yposition:
			set host_list {}
			set fid [ open $mb_hosts r ]
			while { [ gets $fid fout ] > 0 } {
			    lappend host_list $fout
			}
			close $fid
			set edit_host [ lindex [split [ .mhst.hidden1 get ] ":" ] 0 ]
			# Replace spaces in name
			set th [ .mhst.f1.f1.e1 get ]
			set cp [string first { } $th ]
			while { $cp > 0 } {
			    set t1 [string range $th 0 [ expr $cp - 1 ] ]
			    set t2 [string range $th [ expr $cp + 1 ] end ]
			    set th "$t1\_$t2" ;
			    set cp [string first { } $th ]
			} 
			# Write out new file
			set fid [ open $mb_hosts w ]
			foreach data $host_list {
			    if { $edit_host == [lindex [ split $data ":" ] 0 ] } {
				# This is the one to replace
				set t [ split $data ":" ]
				set cmd "$th:cont:[lindex $t 2 ]:[lindex $t 3 ]:0:"
				if { [.mhst.f2.f2.e1 get ] != {} } {
				    append cmd "[.mhst.f2.f2.e1 get ]:"
				} else {
				    append cmd "[lindex $t 5 ]:"
				}
				if { [.mhst.f2.f2.e1 get ] != {} } {
				    append cmd "[.mhst.f2.f2.e2 get ]:"
				} else {
				    append cmd "[lindex $t 6 ]:"
				}
				puts $fid $cmd
			    } else {
				puts $fid $data
			    }
			}
			close $fid
		    }
		}
		ReadHosts
		fillCan 10
		repack 90
		destroy .mhst
	    }
	}
	button .mhst.f5.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .mhst
	}
	button .mhst.f5.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help mathost
	}
	pack .mhst.f5.b1 .mhst.f5.b2 .mhst.f5.b3 -side left -fill x 

    } else {
	# Window exists
	.mhst.hidden1 delete 0 end
	.mhst.hidden1 insert end $edithost
    }

    #------------------- Routine GUTS ------------------------------
    .mhst.f1.f2.b2 invoke
    .mhst.f2.f1.ck1 deselect
    .mhst.f2.f1.ck2 deselect
    .mhst.f2.f1.ck3 deselect
    .mhst.f2.f1.ck4 deselect
    .mhst.f2.f1.ck5 deselect
    .mhst.f2.f1.ck6 deselect
    if { $edithost != "-" } {
	.mhst.f1.f1.e1 delete 0 end
	set t [split $edithost ":" ]
	.mhst.f1.f1.e1 insert end [ lindex $t 0 ]
	if { [lindex $t 1 ] == "cont" } {
	    .mhst.f1.f2.b1 invoke
	}
	.mhst.f2.f2.e1 delete 0 end
	.mhst.f2.f2.e2 delete 0 end
	.mhst.f2.f2.e3 delete 0 end
	.mhst.f2.f2.e1 insert end [lindex $t 5 ]
	.mhst.f2.f2.e2 insert end [lindex $t 6 ]
	.mhst.f2.f2.e3 insert end [lindex $t 2 ]
	foreach data [split [lindex $t 3 ] "," ] {
	    if { $data == "DNS" } {
		.mhst.f2.f1.ck1 select
	    } elseif { $data == "NIS" } {
		.mhst.f2.f1.ck2 select
	    } elseif { $data == "NFS" } {
		.mhst.f2.f1.ck3 select
	    } elseif { $data == "REP" } {
		.mhst.f2.f1.ck4 select
	    } elseif { $data == "TAPE" } {
		.mhst.f2.f1.ck5 select
	    } elseif { $data == "USER1" } {
		.mhst.f2.f1.ck6 select
	    } 
	}
    }
}


#
# MatUser - Procedure for updating the MAT user/password file
#     
# USE MatUser CurrentHost CurrentLine SelectedData
#             CurrentHost = host to update
#             CurrentLine = Line number of selected line
#                           If 9999 then it is a new entry
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc MatUser { currenthost currentline line args } {
    global errno
    set host $currenthost
    if { [winfo exists .matuser ] == 0 } {
	toplevel .matuser -class Dialog
	wm title .matuser "MAT User Editor: $currenthost"
	wm iconname .matuser "MAT User Editor"
	entry .matuser.hidden1 
	entry .matuser.hidden2
	entry .matuser.hidden3
	# NOT GOOD
	.matuser.hidden1 insert end $currenthost
	.matuser.hidden2 insert end $currentline
	.matuser.hidden3 insert end $line
	frame .matuser.f1 -bd 2 -relief "raised"
	label .matuser.f1.l1 -text "Login:" -width 16
	entry .matuser.f1.e1 -width 16
	pack .matuser.f1.l1 .matuser.f1.e1 -fill x -side left -expand 1
	frame .matuser.f2 -bd 2 -relief "raised"
	label .matuser.f2.l1 -text "Password:" -width 16
	entry .matuser.f2.e1 -width 16 -show "*"
	pack .matuser.f2.l1 .matuser.f2.e1 -fill x -side left -expand 1

	frame .matuser.f25 -bd 2 -relief "raised"
	label .matuser.f25.l1 -text "Confirm Password:" -width 16
	entry .matuser.f25.e1 -width 16 -show "*"
	pack .matuser.f25.l1 .matuser.f25.e1 -fill x -side left -expand 1

	frame .matuser.f3 -bd 2 -relief "raised"
	frame .matuser.f3.f1 -bd 2 -relief "flat"

	frame .matuser.f3.f1.f1 -bd 0 -relief "flat"
	label .matuser.f3.f1.f1.l1 -text "Available"
	pack .matuser.f3.f1.f1.l1 -fill x -side top
	scrollbar .matuser.f3.f1.f1.vscroll -orient vertical -command ".matuser.f3.f1.f1.text yview" 
	listbox .matuser.f3.f1.f1.text -selectforeground white -selectbackground #000080 -yscrollcommand {.matuser.f3.f1.f1.vscroll set } -height 5 -width 20
	.matuser.f3.f1.f1.text config -cursor hand2
	pack .matuser.f3.f1.f1.vscroll -side left -fill y
	pack .matuser.f3.f1.f1.text -side left -expand 1 -fill both

	frame .matuser.f3.f1.f2 -bd 0 -relief "flat"
	label .matuser.f3.f1.f2.l1 -text "Joined"
	pack .matuser.f3.f1.f2.l1 -fill x -side top
	scrollbar .matuser.f3.f1.f2.vscroll -orient vertical -command ".matuser.f3.f1.f2.text yview" 
	listbox .matuser.f3.f1.f2.text -selectforeground white -selectbackground #000080 -yscrollcommand {.matuser.f3.f1.f2.vscroll set } -height 5 -width 20
	.matuser.f3.f1.f2.text config -cursor hand2
	pack .matuser.f3.f1.f2.vscroll -side left -fill y
	pack .matuser.f3.f1.f2.text -side left -expand 1 -fill both
	pack .matuser.f3.f1.f1 .matuser.f3.f1.f2 -fill both -side left -expand 1

	frame .matuser.f3.f2 -bd 2 -relief "flat"
	button .matuser.f3.f2.b1 -text "Add" -command {
	    if { [.matuser.f3.f1.f1.text curselection ] != "" } {
		.matuser.f3.f1.f2.text insert end [.matuser.f3.f1.f1.text get [.matuser.f3.f1.f1.text curselection ]]
		.matuser.f3.f1.f1.text delete [.matuser.f3.f1.f1.text curselection ]
	    }
	}
	button .matuser.f3.f2.b2 -text "Delete" -command {
	    if { [.matuser.f3.f1.f2.text curselection ] != "" } {
		.matuser.f3.f1.f1.text insert end [.matuser.f3.f1.f2.text get [.matuser.f3.f1.f2.text curselection ]]
		.matuser.f3.f1.f2.text delete [.matuser.f3.f1.f2.text curselection ]
	    }
	}
	pack .matuser.f3.f2.b1 .matuser.f3.f2.b2 -fill x -side left -expand 1
	pack .matuser.f3.f1 .matuser.f3.f2 -fill both -side top -expand 1

	frame .matuser.f4 -bd 2 -relief "raised"
	button .matuser.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.matuser.f1.e1 get] != "" } {
		set fail 0
		if { [.matuser.hidden2 get ] == 9999 } {
		    # New MAT password entry
		    set cmd "add matpass "
		    if { [.matuser.f2.e1 get] == {} } {
			error 341
			set fail 1
		    } else {
			if { [.matuser.f2.e1 get] == [.matuser.f25.e1 get] } {
			    # Build command arguements
			    append cmd "[ string tolower [.matuser.f1.e1 get]]:[.matuser.f2.e1 get]:"
			} else {
			    error 342
			    .matuser.f2.e1 delete 0 end
			    .matuser.f25.e1 delete 0 end
			    focus .matuser.f2.e1
			    set fail 1
			}
		    }
		} else {
		    # Modify an old entry
		    set cmd "mod matpass "

		    if { [.matuser.f2.e1 get] == {} } {
			# Do not change password
			append cmd "[ string tolower [.matuser.f1.e1 get]]::"
		    } else {
			# Change the password
			if { [.matuser.f2.e1 get] == [.matuser.f25.e1 get] } {
			    # Build command arguements
			    append cmd "[ string tolower [.matuser.f1.e1 get]]:[.matuser.f2.e1 get]:"
			} else {
			    error 342
			    .matuser.f2.e1 delete 0 end
			    .matuser.f25.e1 delete 0 end
			    set fail 1
			}
		    }
		}
		if {$fail == 0 } {
		    append cmd [ list2Perms .matuser.f3.f1.f2.text $feature_list ]
		    set returnval [queryHost $currenthost $cmd 1 ]
		    if { $errno != 0 } {
			error 91
			puts stdout "ERROR:  Unable to update the MAT password file on $currenthost\n       Check the files permissions"
		    } else {
			repack 91
			destroy .matuser
		    }
		}
	    } else {
		error 340
	    }
	}
	button .matuser.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    repack 91
	    destroy .matuser
	}
	button .matuser.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help sudo
	}
	pack .matuser.f4.b1 .matuser.f4.b2 .matuser.f4.b3 -side left -fill x 
	pack .matuser.f1 .matuser.f2 .matuser.f25 .matuser.f3 .matuser.f4 -fill both -side top -expand 1
    } else {
	# Window Exists already

    }
    # Initialize windows
    global feature_list_file
    global feature_list
    set feature_list {}
    set fid [ open $feature_list_file r ]
    while { [ gets $fid fout ] > 0 } {
	lappend feature_list $fout
    }
    close $fid
    for { set position 0 } { $position < [llength $feature_list] } { incr position 1 } {
	set tmp [lindex [split [lindex $feature_list $position ] ":" ] 1 ]
	.matuser.f3.f1.f1.text insert end $tmp
    }
    .matuser.f1.e1 delete 0 end
    .matuser.f2.e1 delete 0 end
    .matuser.f25.e1 delete 0 end
    .matuser.f3.f1.f2.text delete 0 end
    if { $currentline == 9999 } {
	# New entry


    } else {
	# Remove permission groups from the available box
	.matuser.f1.e1 insert end [lindex [split [.matuser.hidden3 get] ":" ] 0 ]
	foreach data [ perms2list [lindex [split [.matuser.hidden3 get] ":" ] 2 ] $feature_list ] {
	    .matuser.f3.f1.f2.text insert end $data
	    # Delete lines from other list
	    for { set position 0 } { $position < [.matuser.f3.f1.f1.text index end ] } { incr position 1 } {
		set tmp [ .matuser.f3.f1.f1.text get $position ]
		if { $tmp == $data } {
		    .matuser.f3.f1.f1.text delete $position
		}
	    }
	}
    }
    # ------------------ Bindings ----------------------------
    bind .matuser.f1.e1 <Return> {
	focus .matuser.f2.e1
    }
    bind .matuser.f2.e1 <Return> {
	focus .matuser.f25.e1
    }
    bind .matuser.f3.f1.f1.text <Double-1> {
	.matuser.f3.f2.b1 invoke
    }
    bind .matuser.f3.f1.f2.text <Double-1> {
	.matuser.f3.f2.b2 invoke
    }
}

#
# Convert the entries in a listbox to their
# numerical permission numbers
# Returns a permission string for the password file
proc list2Perms { window features } {
    set perms {}
    foreach permLine [ $window get 0 end ] {
	for { set position 0 } { $position < [llength $features] } { incr position 1 } {
	    set tmp [lindex [split [lindex $features $position ] ":" ] 1 ]
	    if { $tmp == $permLine } {
		append perms [lindex [split [lindex $features $position ] ":" ] 0 ]
	    }
	}
    }
    return $perms
}

#
# Convert a password perms field into a string list
# Returns a list with the matching strings for the permissions
proc perms2list { perms features } {
    set retlist {}
    foreach permVal [ split $perms {} ] {
	for { set position 0 } { $position < [llength $features] } { incr position 1 } {
	    set tmp [lindex [split [lindex $features $position ] ":" ] 0 ]
	    if { $tmp == $permVal } {
		lappend retlist [lindex [split [lindex $features $position ] ":" ] 1 ]
	    }
	}
    }
    return $retlist
}
#
# Sudoers update procedure
#
# USE Sudo CurrentHost CurrentLine SelectedData
#             CurrentHost = host to update
#             CurrentLine = Line number of selected line
#                           If 9999 then it is a new entry
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc Sudo { currenthost currentline line args } {
    global errno
    set host $currenthost
    if { [string length $args ] != 0 } {
	append line " $args"
    }
    if { [winfo exists .sudo ] == 0 } {
	toplevel .sudo -class Dialog
	wm title .sudo "Sudo Editor: $currenthost"
	wm iconname .sudo "Sudo Editor"
	entry .sudo.hidden1 
	entry .sudo.hidden2
	# NOT GOOD
	.sudo.hidden1 insert end $currenthost
	.sudo.hidden2 insert end $currentline
	frame .sudo.f1 -bd 2 -relief "raised"
	label .sudo.f1.l1 -text "User:" -width 6
	entry .sudo.f1.e1 -width 16
	pack .sudo.f1.l1 .sudo.f1.e1 -fill both -side left -expand 1
	frame .sudo.f2 -bd 2 -relief "raised"
	label .sudo.f2.l1 -text "Access:" -width 6
	entry .sudo.f2.e1 -width 16
	pack .sudo.f2.l1 .sudo.f2.e1 -fill both -side left -expand 1
	
	frame .sudo.f4 -bd 2 -relief "raised"
	button .sudo.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.sudo.f2.e1 get] != "" } {
		if { [.sudo.f1.e1 get] != "" } {
		    set newline "[ string tolower [.sudo.f1.e1 get]]    [ string tolower [.sudo.f2.e1 get]]:"
		    puts stdout "Newline = $newline"
		    set currentline [.sudo.hidden2 get]
		    if { $currentline == 9999 } {
			set cmd "add sudo $newline"
		    } else {
			set cmd "mod sudo $currentline $newline"
		    }
		    set returnval [queryHost $currenthost $cmd 1 ]
		    if { $errno != 0 } {
			error 91
			puts stdout "ERROR:  Unable to update the /etc/group file on $currenthost\n       Check the files permissions"
		    } else {
			repack 11
			destroy .sudo
		    }
		}
	    }
	}
	button .sudo.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    repack 11
	    destroy .sudo
	}
	button .sudo.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help sudo
	}
	pack .sudo.f4.b1 .sudo.f4.b2 .sudo.f4.b3 -side left -fill x 
	pack .sudo.f1 .sudo.f2 .sudo.f4 -fill both -expand 1
	# ----- Bindings -----
	bind .sudo.f1.e1 <Return> {
	    focus .sudo.f2.e1
	}
    }
}


#
# DnsClient update procedure
#
# USE DnsClient CurrentHost CurrentLine SelectedData
#               CurrentHost = host to update
#               CurrentLine = Not Used
#               SelectedData = Not used
# NOTE: Data passed to this routine will be ignored.  Only the data from the client will be used
#
proc DnsClient { currenthost currentline line } {
    global errno
    set host $currenthost
    if { [winfo exists .dnsC ] == 0 } {
	toplevel .dnsC -class Dialog
	wm title .dnsC "DNS Client Editor: $currenthost"
	wm iconname .dnsC "Groups Editor"
	entry .dnsC.hidden1 
	entry .dnsC.hidden2
	# NOT GOOD
	.dnsC.hidden1 insert end $currenthost
	frame .dnsC.f1 -bd 2 -relief "raised"
	label .dnsC.f1.l1 -text "Domain:" -width 6
	entry .dnsC.f1.e1 -width 20
	pack .dnsC.f1.l1 .dnsC.f1.e1 -fill both -side left -expand 1

	frame .dnsC.f2 -bd 2 -relief "raised"
	label .dnsC.f2.ltop -text "DNS Name Servers" 
	pack .dnsC.f2.ltop -fill x
	frame .dnsC.f2.f1 -bd 2 -relief "flat"
	entry .dnsC.f2.f1.e1 -width 3
	label .dnsC.f2.f1.l1 -text "."
	entry .dnsC.f2.f1.e2 -width 3
	label .dnsC.f2.f1.l2 -text "."
	entry .dnsC.f2.f1.e3 -width 3
	label .dnsC.f2.f1.l3 -text "."
	entry .dnsC.f2.f1.e4 -width 3
	button .dnsC.f2.f1.b1 -text "Add" -command {
	    if { [.dnsC.f2.f1.e1 get] < 256 && [.dnsC.f2.f1.e1 get] > -1  } {
		if { [.dnsC.f2.f1.e2 get] < 256 && [.dnsC.f2.f1.e2 get] > -1  } {
		    if { [.dnsC.f2.f1.e4 get] < 256 && [.dnsC.f2.f1.e3 get] > -1  } {
			if { [.dnsC.f2.f1.e4 get] < 255 && [.dnsC.f2.f1.e4 get] > 0  } {
			    .dnsC.f2.f2.text insert end "[.dnsC.f2.f1.e1 get].[.dnsC.f2.f1.e2 get].[.dnsC.f2.f1.e3 get].[.dnsC.f2.f1.e4 get]"
			    .dnsC.f2.f1.e1 delete 0 end
			    .dnsC.f2.f1.e2 delete 0 end
			    .dnsC.f2.f1.e3 delete 0 end
			    .dnsC.f2.f1.e4 delete 0 end
			    focus .dnsC.f2.f1.e1
			} else {
			    focus .dnsC.f2.f1.e4
			    if { [.dnsC.f2.f1.e4 get] < 255 } { 
				error 202
			    } elseif { [.dnsC.f2.f1.e4 get] > 0  } {
				error 201
			    } else { error 200 }
			}
		    } else {
			focus .dnsC.f2.f1.e3
			error 200
		    }
		} else {
		    focus .dnsC.f2.f1.e2
		    error 200
		}
	    } else {
		focus .dnsC.f2.f1.e1
		error 200
	    }
	    
	}
	pack .dnsC.f2.f1.e1 .dnsC.f2.f1.l1 .dnsC.f2.f1.e2 .dnsC.f2.f1.l2 .dnsC.f2.f1.e3 .dnsC.f2.f1.l3 .dnsC.f2.f1.e4 .dnsC.f2.f1.b1 -fill x -side left -expand 1
	frame .dnsC.f2.f2 -bd 2 -relief "flat"
	listbox .dnsC.f2.f2.text -selectforeground white -selectbackground #000080 -height 3 -width 16
	.dnsC.f2.f2.text config -cursor hand2
	pack .dnsC.f2.f2.text -side left -expand 1 -fill both
	button .dnsC.f2.f2.b1 -text "Delete" -foreground red -activeforeground #a00000 -command {
	    if { [.dnsC.f2.f2.text curselection ] != "" } {
		.dnsC.f2.f2.text delete [.dnsC.f2.f2.text curselection ]
	    }
	}
	pack .dnsC.f2.f2.b1 -side left
	pack .dnsC.f2.f1 .dnsC.f2.f2 -side top -fill both -expand 1

	frame .dnsC.f3 -bd 2 -relief "raised"
	label .dnsC.f3.l1 -text "Additional Search Domains" 
	pack .dnsC.f3.l1 -fill x
	frame .dnsC.f3.f1 -bd 2 -relief "flat"
	entry .dnsC.f3.f1.e1 -width 10
	button .dnsC.f3.f1.b1 -text "Add" -command {
	    if { [.dnsC.f3.f1.e1 get] != "" } {
		.dnsC.f3.f2.text insert end [.dnsC.f3.f1.e1 get]
	    }
	    .dnsC.f3.f1.e1 delete 0 end
	}
	pack .dnsC.f3.f1.e1 .dnsC.f3.f1.b1 -fill both -side left -expand 1
	frame .dnsC.f3.f2 -bd 2 -relief "flat"
	scrollbar .dnsC.f3.f2.vscroll -orient vertical -command ".dnsC.f3.f2.text yview" 
	listbox .dnsC.f3.f2.text -selectforeground white -selectbackground #000080 -yscrollcommand {.dnsC.f3.f2.vscroll set } -height 4 -width 16
	.dnsC.f3.f2.text config -cursor hand2
	pack .dnsC.f3.f2.vscroll -side left -fill y
	pack .dnsC.f3.f2.text -side left -expand 1 -fill both
	button .dnsC.f3.f2.b1 -text "Delete" -foreground red -activeforeground #a00000 -command {
	    if { [.dnsC.f3.f2.text curselection ] != "" } {
		.dnsC.f3.f2.text delete [.dnsC.f3.f2.text curselection ]
	    }
	}
	pack .dnsC.f3.f2.b1 -side left
	pack .dnsC.f3.f1 .dnsC.f3.f2 -side top -fill both -expand 1
	frame .dnsC.f4 -bd 2 -relief "flat"
	button .dnsC.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [ .dnsC.f1.e1 get ] != "" } {
		# Check for at least one nameserver
		if { [ .dnsC.f2.f2.text size ] > 0 } {
		    # The arguemnets to the Perl script will include "!" and "+".  The "!" seperates fields,
		    # while the "+" seperates values field values
		    set cmd "CON dnscset [ .dnsC.f1.e1 get ]!"
		    for { set position 0 } { $position < [ .dnsC.f2.f2.text size ] } { incr position 1 } {
			append cmd "[ .dnsC.f2.f2.text get $position]+"
		    }
		    set cmd2 [string trimright $cmd "+" ]
		    append cmd2 "!"
		    if { [ .dnsC.f3.f2.text size ] > 0 } {
			for { set position 0 } { $position < [ .dnsC.f3.f2.text size ] } { incr position 1 } {
			    append cmd2 "[ .dnsC.f3.f2.text get $position]+"
			}
		    }
		    set cmd [string trimright $cmd2 "+" ]
		    if $DEBUG { puts stdout "Command = $cmd" } 
		    set returnval [queryHost [.dnsC.hidden1 get] $cmd 1 ]
		    if { $errno != 0 } {
			error 205
			puts stdout "ERROR:  Unable to update the /etc/resolv.conf file on [.dnsC.hidden1 get]\n       Check the files permissions"
		    } else {
			repack 6
			destroy .dnsC
		    }
		} else {
		    error 204
		}
	    } else {
		error 203
	    }
	}
	button .dnsC.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    repack 6
	    destroy .dnsC
	}
	button .dnsC.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help dnsc
	}
	pack .dnsC.f4.b1 .dnsC.f4.b2 .dnsC.f4.b3 -side left -fill x 
	pack .dnsC.f1 .dnsC.f2 .dnsC.f3 .dnsC.f4 -fill both -expand 1
	# ----- Bindings -----
	bind .dnsC.f2.f1.e1 <Return> {
	    focus .dnsC.f2.f1.e2
	}
	bind .dnsC.f2.f1.e1 . {
	    after 250 {
		.dnsC.f2.f1.e1 delete [expr [string length [.dnsC.f2.f1.e1 get]] -1] end
	    }
	    focus .dnsC.f2.f1.e2 
	}
	bind .dnsC.f2.f1.e2 <Return> {
	    focus .dnsC.f2.f1.e3
	}
	bind .dnsC.f2.f1.e2 . {
	    after 250 {
		.dnsC.f2.f1.e2 delete [expr [string length [.dnsC.f2.f1.e2 get]] -1] end
	    }
	    focus .dnsC.f2.f1.e3 
	}
	bind .dnsC.f2.f1.e3 <Return> {
	    focus .dnsC.f2.f1.e4
	}
	bind .dnsC.f2.f1.e3 . {
	    after 250 {
		.dnsC.f2.f1.e3 delete [expr [string length [.dnsC.f2.f1.e3 get]] -1] end
	    }
	    focus .dnsC.f2.f1.e4 
	}
	bind .dnsC.f2.f1.e4 <Return> {
	    .dnsC.f2.f1.b1 invoke
	}
	bind .dnsC.f2.f1.e4 . {
	    after 250 {
		.dnsC.f2.f1.e4 delete [expr [string length [.dnsC.f2.f1.e4 get]] -1] end
	    }
	    .dnsC.f2.f1.e1 delete 0 end
	    .dnsC.f2.f1.e2 delete 0 end
	    .dnsC.f2.f1.e3 delete 0 end
	    .dnsC.f2.f1.e4 delete 0 end
	}

	bind .dnsC.f3.f1.e1 <Return> {
	    if { [.dnsC.f3.f1.e1 get] != "" } {
		.dnsC.f3.f2.text insert end [.dnsC.f3.f1.e1 get]
	    }
	    .dnsC.f3.f1.e1 delete 0 end
	}
    } else {
	.dnsC.hidden1 delete 0 end
	.dnsC.hidden1 insert end $currenthost
	wm title .dnsC "DNS Client Editor: $currenthost"
    }

    # Fill boxes with data
    .dnsC.f2.f2.text delete 0 end
    .dnsC.f3.f2.text delete 0 end
    set args ""
    set returndata [queryHost $currenthost "CON get dnsc" 0 ]
    for { set position 0 } { $position < [llength $returndata] } { incr position 1 } {
	set tmp [lindex $returndata $position ]
	set arg1 [lindex $tmp 0 ]
	set arg2 [lindex $tmp 1 ]
	if { $arg1 == "domain" } {
	    .dnsC.f1.e1 delete 0 end
	    .dnsC.f1.e1 insert end $arg2
	} elseif { $arg1 == "nameserver" } {
	    .dnsC.f2.f2.text insert end $arg2
	} elseif { $arg1 == "search" } {
	    set args [ split $tmp {, } ]
	    for { set position2 1 } { $position2 < [llength $args] } { incr position2 1 } {
		if { [lindex $args $position2 ] != "" } {
		    .dnsC.f3.f2.text insert end [lindex $args $position2 ]
		}
	    }
	}
    }
}




#
# --------------------------------------------------------------------------------------------------------
# --------------------------                  TASK  PROCEDURES           ---------------------------------
# --------------------------------------------------------------------------------------------------------
#

#
# -- Procedure to read tasks from raw data and fill list box window --
#
proc gettask { data window } {  
    $window delete 0 end
    for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	set tmp [split [lindex $data $position ]]
	set task [lindex $tmp 1 ]
	$window insert end $task
    }
}


#
# ----- Procedure for getting the name of a new task -----
#
proc getnewtask { } {
    global stask
    toplevel .newtask -class Dialog
    wm title .newtask "Query?"
    wm iconname .newtask Dialog
    label .newtask.label -text "Please enter the name\nof the new task"
    entry .newtask.entry -width 30
    pack .newtask.label .newtask.entry -side top -fill both -expand 1
    focus .newtask.entry
    bind .newtask.entry <Return> {
	set stask [.newtask.entry get]
	destroy .newtask
    }
}




#
# Task update procedure
#
# USE Tasks   CurrentHost CurrentLine SelectedData
#            CurrentHost = host to update
#            CurrentLine = line number 
#            SelectedData = data
## NOTE: Data passed to this routine will be ignored.  Only the data from the client will be used

proc Tasks { currenthost jobid line args } {
    global errno
    if { [string length $args ] != 0 } {
	append line " $args"
    }
    set host $currenthost
    if { [winfo exists .task ] == 0 } {
	toplevel .task -class Dialog
	wm title .task "Task Editor: $currenthost"
	wm iconname .task "Task Editor"
	entry .task.hidden1 
	entry .task.hidden2
	# NOT GOOD
	.task.hidden1 insert end $currenthost

	frame .task.time -relief raised -bd 2
	set stask [lindex $line 1 ]
	label .task.time.label -text "Host: $currenthost  Task: $stask"
	pack .task.time.label
	frame .task.time.type -relief flat -bd 0
	checkbutton .task.time.type.interval -text "Interval" -variable ttype -onvalue 1 -offvalue 0 -command {
	    if { $ttype == 1 } {
		.task.time.timeval.label1 configure -text "Minutes"
		.task.time.timeval.label2 configure -foreground grey
		.task.time.timeval.entry2 configure -state disabled
	    } elseif { $ttype == 0 } {
		.task.time.timeval.label1 configure -text "Hour(s)"
		.task.time.timeval.label2 configure -foreground black
		.task.time.timeval.entry2 configure -state normal
	    }
	}
	checkbutton .task.time.type.timed -text "Timed" -variable ttype -onvalue 0 -offvalue 1 -command {
	    if { $ttype == 1 } {
		.task.time.timeval.label1 configure -text "Minutes"
		.task.time.timeval.label2 configure -foreground grey
		.task.time.timeval.entry2 configure -state disabled
	    } elseif { $ttype == 0 } {
		.task.time.timeval.label1 configure -text "Hour(s)"
		.task.time.timeval.label2 configure -foreground black
		.task.time.timeval.entry2 configure -state normal
	    }
	}
	pack .task.time.type.interval .task.time.type.timed -side left -expand 1 -fill x
	frame .task.time.timeval -relief flat -bd 0
	label .task.time.timeval.label1 -text "Hour(s)"
	entry .task.time.timeval.entry1 -width 10
	label .task.time.timeval.label2 -text "Minute(s)"
	entry .task.time.timeval.entry2 -width 10
	pack .task.time.timeval.label1 .task.time.timeval.entry1 .task.time.timeval.label2 .task.time.timeval.entry2 -side left -fill x -expand 1
	pack .task.time.type .task.time.timeval -side top -fill x
	pack .task.time -fill x
	
	frame .task.outfile -relief raised -bd 2
	label .task.outfile.label -text "Task Output History"
	pack .task.outfile.label -side top -fill x
	frame .task.outfile.frame -relief flat -bd 0
	label .task.outfile.frame.filelabel -text "File name"
	entry .task.outfile.frame.filename 
	pack .task.outfile.frame.filelabel .task.outfile.frame.filename -side left -fill x -expand 1
	pack .task.outfile.frame -side top -fill x
	scale .task.outfile.histval -from 0 -to 35 -resolution 1 -tickinterval 0 -orient horizontal -label "Retention Period (Days)" -variable retain 
	pack .task.outfile.histval -side top -fill x
	pack .task.outfile -fill x
	
	frame .task.warn -relief raised -bd 2
	label .task.warn.label -text "Warning Threshold"
	entry .task.warn.val 
	pack .task.warn.label .task.warn.val -side left -fill x -expand 1
	pack .task.warn -fill x
	
	frame .task.alarm -relief raised -bd 2
	label .task.alarm.label -text "Alarm Threshold   "
	entry .task.alarm.val 
	pack .task.alarm.label .task.alarm.val -side left -fill x -expand 1
	pack .task.alarm -fill x
	
	frame .task.recover -relief raised -bd 2
	label .task.recover.label -text "Recovery Script  "
	entry .task.recover.val 
	pack .task.recover.label .task.recover.val -side left -fill x -expand 1
	pack .task.recover -fill x
	
	frame .task.bottm -relief raised -bd 2
	button .task.bottm.apply -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { $current != 9999 } {
		set cmdline "cdadmin modifynum $current "
	    } else {
		set cmdline "cdadmin add "
	    }
	    if { $ttype == 0 } {
		append cmdline "T[ .time.timeval.entry1 get ]"
		append cmdline ":[ .time.timeval.entry2 get ]"
	    } else {
		append cmdline [ .time.timeval.entry1 get ]
	    }
	    append cmdline " "
	    append cmdline $stask
	    append cmdline " [expr $retain * 1440 / $interval]"
	    append cmdline " [.outfile.frame.filename get]"
	    append cmdline " [.warn.val get]"
	    append cmdline " [.alarm.val get]"
	    append cmdline " [.recover.val get]"
	    set returnval [queryHost $currenthost $cmdline 1 ]
	    if { $errno != 0 } {
		error 1
	    }
	    puts stdout $cmdresult
	    set returndata [queryHost $currenthost cdadmin 0 ]
	    gettask $returndata .tp.out.text
	}
	
	pack .task.bottm.apply 
	pack .task.bottm -anchor w
    } else {
	.task.outfile.frame.filename delete 0 end
	.task.warn.val delete 0 end
	.task.alarm.val delete 0 end
	.task.recover.val delete 0 end
	.task.time.timeval.entry2 delete 0 end
	.task.time.timeval.entry1 delete 0 end
    }



    #
    #  Task Bindings
    #
    
    bind .tp.out.text <Double-1> { 
	if { [.tp.out.text curselection ] != "" } {
	    set stask [ .tp.out.text get [ .tp.out.text curselection ]]
	    set current [ .tp.out.text curselection ]
	    newstate normal
	    .time.label configure -text "HOST: $currenthost  Task: $stask"
	    set tmp [split [lindex $returndata $current ]]
	    set ttime [lindex $tmp 0 ]
	    set cttype [lindex [split $ttime {} ] 0 ]
	    if { $cttype == "T" | $cttype == "t" } {
		.task.time.label configure -foreground black
		.task.time.label configure -text "Host: $currenthost  Task: $stask"
		.task.time.timeval.label1 configure -text "Hour(s)"
		.task.time.timeval.label2 configure -foreground black
		.task.time.timeval.entry2 configure -state normal
		.task.time.timeval.entry2 delete 0 end
		.task.time.timeval.entry1 delete 0 end
		.task.time.timeval.entry1 insert end [string range [lindex [split $ttime ":"] 0] 1 end]
		.task.time.timeval.entry2 insert end [lindex [split $ttime ":"] 1]
		set ttype 0
	    } else {
		.task.time.timeval.label1 configure -text "Minutes"
		.task.time.timeval.label2 configure -foreground grey
		.task.time.timeval.entry2 delete 0 end
		.task.time.timeval.entry2 configure -state disabled
		.task.time.timeval.entry1 delete 0 end
		.task.time.timeval.entry1 insert end $ttime
		set ttype 1
	    }
	    set task [lindex $tmp 1 ]
	    set history [lindex $tmp 2 ]
	    set outfile [lindex $tmp 3 ]
	    set warn [lindex $tmp 4 ]
	    set alarm [lindex $tmp 5 ]
	    set recover [lindex $tmp 6 ]
	    if { $history == "-" } {
		set retain 0
	    } else {
		set retain [ expr ($interval * $history) / 1440 ]
	    }
	    .task.outfile.frame.filename delete 0 end
	    .task.warn.val delete 0 end
	    .task.alarm.val delete 0 end
	    .task.recover.val delete 0 end
	    .task.outfile.frame.filename insert end $outfile
	    .task.warn.val insert end $warn
	    .task.alarm.val insert end $alarm
	    .task.recover.val insert end $recover
	}
    } 
    
    bind .task.time.timeval.entry1 <Return> {
	if { $ttype == 0 } {
	    focus .task.time.timeval.entry2
	} else {
	    focus .task.outfile.frame.filename
	}
    }
    
    bind .task.time.timeval.entry2 <Return> {
	focus .task.outfile.frame.filename
    }
    
    bind .task.outfile.frame.filename <Return> {
	focus .task.outfile.histval
    }
    
    bind .task.warn.val <Return> {
	focus .task.alarm.val
    }
    
    bind .task.alarm.val <Return> {
	focus .task.recover.val
    }
    
    bind .task.recover.val <Return> {
	focus .task.bottm
    }
}


#
# Procedure to Enable/Disable Entry
#
proc newstate { state } {
    if { $state == "disabled" } {
	.task.time.label configure -foreground grey
	.task.time.timeval.label1 configure -foreground grey
	.task.time.timeval.label2 configure -foreground grey
	.task.outfile.label configure -foreground grey
	.task.outfile.frame.filelabel configure -foreground grey
	.task.outfile.histval configure -foreground grey
	.task.warn.label configure -foreground grey
	.task.alarm.label configure -foreground grey
	.task.recover.label configure -foreground grey

    } else {
	.task.time.label configure -foreground black
	.task.time.label configure -text "Host:    Task:"
	.task.time.timeval.label1 configure -foreground black
	.task.time.timeval.label2 configure -foreground black
	.task.outfile.label configure -foreground black
	.task.outfile.frame.filelabel configure -foreground black
	.task.outfile.histval configure -foreground black
	.task.warn.label configure -foreground black
	.task.alarm.label configure -foreground black
	.task.recover.label configure -foreground black
    }
    .task.time.type.interval configure -state $state
    .task.time.type.timed configure -state $state
    .task.time.timeval.entry1 configure -state $state
    .task.time.timeval.entry2 configure -state $state
    .task.outfile.frame.filename configure -state $state
    .task.outfile.histval configure -state $state
    .task.warn.val configure -state $state
    .task.alarm.val configure -state $state
    .task.recover.val configure -state $state
    .task.bottm.apply configure -state $state
}

#
#   gethostent - Get a hosts entry from the MAT hosts file
#   USE:   set hostsline [ gethostent hostname ]
#   Returns the appropriate line from the hosts file
#
proc gethostent { h } {
    global active_hosts
    foreach l $active_hosts {
	if { $h == [ lindex [split $l ":" ] 0 ] } {
	    return $l
	}
    }
    return {}
}



#
# MatLic - Procedure for updating the MAT license file
#     
# USE MatLic args
#             {} = New entry
#
proc MatLic { currenthost args } {
    global errno textd
    set host $currenthost
    if { [winfo exists .matl ] == 0 } {
	toplevel .matl -class Dialog
	wm title .matl "MAT License Editor: $currenthost"
	wm iconname .matl "License Editor"
	entry .matl.h1 
	# NOT GOOD
	.matl.h1 insert end $args

	frame .matl.f1 -bd 2 -relief "raised"
	label .matl.f1.l1 -text "License:" -width 8
	entry .matl.f1.e1 -width 32 -font $textd
	pack .matl.f1.l1 .matl.f1.e1 -fill x -side left -expand 1

	frame .matl.f4 -bd 2 -relief "raised"
	button .matl.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.matl.f1.e1 get] != "" } {
		if { [.matl.h1 get ] == {} } {
		    # New MAT license entry
		    set cmd "CON add matlic [.matl.f1.e1 get]"
		} else {
		    set oname [lindex [split [.matl.h1 get ] {:} ] 0 ]
		    set cmd "CON mod matlic $oname [.matl.f1.e1 get]"
		}
		set returnval [queryHost $currenthost $cmd 1 ]
		if { $errno != 0 } {
		    error 7
		    puts stdout "ERROR:  Unable to update the MAT license file on $currenthost\n       Check the files permissions"
		} else {
		    repack 94
		    destroy .matl
		}
	    }
	}
	button .matl.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    repack 94
	    destroy .matl
	}
	button .matl.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help matlic
	}
	pack .matl.f4.b1 .matl.f4.b2 .matl.f4.b3 -side left -fill x 
	pack .matl.f1 .matl.f4 -fill both -side top -expand 1
    } else {
	# Window Exists already

    }
    # Initialize windows
    .matl.f1.e1 delete 0 end
    if { $args != {} } {
	.matl.f1.e1 insert end $args
    }
}
