# archive.tcl - The Tape Interface
#
# Copyright (c) 1997, 1998, 1999 Mark Black
#
#
# This file contains:
#   tapeText    - Fill the text window with data
#   canBACKUP   - Draw the canvas for the backup tools
#   canRESTORE  - Draw the canvas for the restore tools
#   canMEDIA    - Draw the canvas for the tape list
#   canTSCHED   - Draw the canvas for the tape scheduling tools
#   canARCHIVE  - Draw the canvas for the backup/restore entry point
#   getTapeHost - Returns a list of tape servers
#   newTape     - Add a new backup tape to the tape server


#
# -- Tape Specific Procedure to read data from raw data and fill list window --
#    w = window to display it in
proc tapeText { data w type } {
    global currenthost
    global tapeserv
    . config -cursor watch
    $w config -cursor watch
    $w configure -state normal
    $w delete 0.0 end

    # Backup
    set tabbing(121) {6 16 24 }
    set toplabel(121) "ID    Name            Contents"
    # Backup Group
    set tabbing(123) {7 }
    set toplabel(123) "GID    Name"
    # Backup Job
    # job#:group#:retain:drive#:pool#:Descriptor
    set tabbing(125) {6 8 }
    set toplabel(125) "Job   Group   Name"
    # Backup Devices
    set tabbing(126) {4 8 }
    set toplabel(126) "ID  Slots   Name"
    # Media Maanagement
    set tabbing(127) {6 10 }
    set toplabel(127) "Vol#  DD/MM/YY  Contents"
    # Restore 
    set tabbing(131) {6 10 10 20 }
    set toplabel(131) "Type  Size(K)   Date                Description"

    set bold "-background #c0c000 -relief raised -borderwidth 1"
    set normal "-background {} -relief flat"
    set select "-underline 1"
    set tplabel $w
    append tplabel "label"
    $tplabel configure -state normal
    $tplabel delete 0.0 end
    $tplabel insert end $toplabel($type)
    $tplabel configure -state disabled

    if { $type == 121 } {
	# ------------------------------  Backup  --------------------------------
	for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	    set tmp [lindex $data $position ]
	    set tmp3 [split $tmp {;} ]
#	    puts stdout $tmp3
	    set tmp4 [lindex $tmp3 1 ]
	    set outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 0] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    set tmp4 [lindex $tmp3 6 ]
	    append outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 1] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }

	    set tmp4 [string trim [lrange $tmp3 11 end ]]
	    append outline $tmp4
	    $w insert end "$outline\n" l$position
	    $w tag bind l$position <Any-Enter> "$w tag configure l$position $bold"
	    $w tag bind l$position <Any-Leave> "$w tag configure l$position $normal"
	    $w tag bind l$position <Button-1> "Selecter $w l$position"
	    $w tag bind l$position <Double-1>  "fileSelect $currenthost [lindex $tmp3 1 ] 0" 
	}
	$w configure -state disabled
    } elseif { $type == 123 } {
	# ------------------------------  Backup Group --------------------------------
	for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	    set tmp [lindex $data $position ]
	    set tmp3 [split $tmp {:} ]
	    set tmp4 [lindex $tmp3 0 ]
	    set outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 0] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }

	    set tmp4 [string trim [lindex $tmp3 2]]
	    append outline $tmp4
	    $w insert end "$outline\n" l$position
	    $w tag bind l$position <Any-Enter> "$w tag configure l$position $bold"
	    $w tag bind l$position <Any-Leave> "$w tag configure l$position $normal"
	    $w tag bind l$position <Button-1> "Selecter $w l$position"
	    $w tag bind l$position <Double-1>  "tapeGroup $currenthost [lindex $tmp3 0 ] $tmp" 
	    $w tag configure l$position -foreground #000000
	}
	$w configure -state disabled
    } elseif { $type == 125 } {
	# ------------------------------  Backup Job --------------------------------
	for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	    set tmp [lindex $data $position ]
	    set tmp3 [split $tmp {:} ]
#	    puts stdout $tmp3
	    set tmp4 [lindex $tmp3 0 ]
	    set outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 0] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    set tmp4 [lindex $tmp3 1 ]
	    append outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 1] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }

	    set tmp4 [string trim [lindex $tmp3 6 ]]
	    append outline $tmp4
	    $w insert end "$outline\n" l$position
	    $w tag bind l$position <Any-Enter> "$w tag configure l$position $bold"
	    $w tag bind l$position <Any-Leave> "$w tag configure l$position $normal"
	    $w tag bind l$position <Button-1> "Selecter $w l$position"
	    $w tag bind l$position <Double-1>  "tapeJobs $currenthost [lindex $tmp3 0 ] $tmp" 
	    $w tag configure l$position -foreground #000000
	}
	$w configure -state disabled
    } elseif { $type == 126 } {
	# ------------------------------  Backup Devices --------------------------------
	for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	    set tmp [lindex $data $position ]
	    set tmp3 [split $tmp {:} ]
#	    puts stdout $tmp3
	    set tmp4 [lindex $tmp3 0 ]
	    set outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 0] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    set tmp4 [lindex $tmp3 4 ]
	    append outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 1] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }

	    set tmp4 [string trim [lindex $tmp3 1]]
	    append outline $tmp4
	    $w insert end "$outline\n" l$position
	    $w tag bind l$position <Any-Enter> "$w tag configure l$position $bold"
	    $w tag bind l$position <Any-Leave> "$w tag configure l$position $normal"
	    $w tag bind l$position <Button-1> "Selecter $w l$position"
	    $w tag bind l$position <Double-1>  "tapeDevs $currenthost [lindex $tmp3 0 ]" 
	    $w tag configure l$position -foreground #000000
	}
	$w configure -state disabled
    } elseif { $type == 127 } {
	# ------------------------------  Tape lists  --------------------------------
	# The data format of the tape lists is determined by the getTapeInfo function
	# eg:   2:First Tape to write                -  TapeID:Tape name
	#       0:27/12/98:1:Backup Job to tape 3    -  Volume#:Backup date:Expired:Job description
	#
	for { set pzn 0 } { $pzn < [llength $data] } { incr pzn 1 } {
	    set tmp [lindex $data $pzn ]
	    # We will need a simple state machine to parse the data correctly
	    set tmp3 [split $tmp {:} ]

	    # It's a tape volume
	    set tmp4 [lindex $tmp3 0 ]
	    set outline "$tmp4"
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 0] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    set tmp4 [lindex $tmp3 1 ]
	    append outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 1] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    set tmp4 [lrange $tmp3 3 end ]
	    append outline [string trim $tmp4 {\{\}} ]

	    $w insert end "$outline\n" l$pzn
	    $w tag bind l$pzn <Any-Enter> "$w tag configure l$pzn $bold"
	    $w tag bind l$pzn <Any-Leave> "$w tag configure l$pzn $normal"
	    $w tag bind l$pzn <Button-1> "Selecter $w l$pzn"
	    if { [lindex $tmp3 2 ] == 1 } {
		$w tag configure l$pzn -foreground #006000
	    } else {
		$w tag configure l$pzn -foreground #000000
	    }
		    
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 1] -[string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }

	}
	$w configure -state disabled
    } elseif { $type == 131 } {
	# ------------------------------  Restore  --------------------------------
	# Get the job descriptions and create an array
	set cmd "TAPE get jobids"
	global currenthost
	global errno
	set host $currenthost
	set retval [queryHost $host $cmd 1 ]
	if { $errno != 0 } {
	    error 2005
	    puts stdout "ERROR:  Unable to retrieve backup information from $host"
	}

	# jobnum, groupid, level, status, retain, drive, tape, vol, sizek, timestr
	for { set position 0 } { $position < [llength $data] } { incr position 1 } {
	    set tmp [lindex $data $position ]
	    set tmp3 [split $tmp {,} ]
#	    puts stdout $tmp3
	    set tmp4 [lindex $tmp3 2 ]
	    if { [lindex $tmp3 2 ] == 0 } {
		set outline "Full"
	    } elseif { [lindex $tmp3 2 ] == -1 } {
		set outline "Diff"
	    } else {
		set outline "Incr"
	    }
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 0] - 4) } { incr cnt 1 } {
		append outline " "
	    }

	    # Size
	    set tmp4 [lindex $tmp3 8 ]
	    append outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 2 ] - [string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }
	    # Date
	    set tmp4 [string range [lindex $tmp3 9 ] 0 15 ]
	    append outline $tmp4
	    for { set cnt 0 } { $cnt < ([lindex $tabbing($type) 3 ] - [string length $tmp4 ]) } { incr cnt 1 } {
		append outline " "
	    }

	    # Job Description:  Scan through the backup list for matches
	    set tmp4 [lindex $tmp3 0 ]
	    set des ""
	    for { set p 0 } { $p < [llength $retval] } { incr p 1 } {
		set tp [split [lindex $retval $p ] {:} ]
		if { [lindex $tp 0 ] == $tmp4 } {
		    set des [string trim [lrange $tp 6 end ] {\{\}} ]
		    append outline $des
		}
	    }

	    $w insert end "$outline\n" l$position
	    $w tag bind l$position <Any-Enter> "$w tag configure l$position $bold"
	    $w tag bind l$position <Any-Leave> "$w tag configure l$position $normal"
	    $w tag bind l$position <Button-1> "Selecter $w l$position"
	    $w tag bind l$position <Double-1> "resSelect $tmp,$des" 
	}
	$w configure -state disabled
    } 
}


# getDir  - Read a directory from the remote machine and
#           populate the listbox $lb.  If $dir is NOT "/"
#           then include ".." in the listbox.
proc getDir { host dir lb } {
    global errno
    $lb delete 0 end
 
    set cmd "TAPE get dir $dir"
#   puts stdout "Cmd = $cmd"
    set retval [queryHost $host $cmd 0 ]
    if { $errno != 0 } {
	error 402
	puts stdout "ERROR:  Unable to retrieve directory information from $host"
    }
    
    if { $dir != "/" } {
	$lb insert end ".."
    }
    foreach e $retval {
	$lb insert end $e
    }
}



#
# canBACKUP  - Fill canvas with tape backup icons
#
proc canBACKUP {} {
    # ---------------------  Tape Backup Window  ---------------------
    global currenthost
    global level
    global browse
    global tapeserv
    .tp.title configure -text "Host($currenthost) --> Config --> Archives --> Backup "
    set level 120
    # Enable the accelerators
    bind .tp.host.canvas <Button-3> "rclick .tp.host.canvas"
    # Put in back button
    set fig(121) [ image create photo -file "../lib/new.gif" ]

    set fig(123) [ image create photo -file "../lib/tapegrp.gif" ]
    set fig(125) [ image create photo -file "../lib/tapejob.gif" ]
    set fig(126) [ image create photo -file "../lib/tapedevs.gif" ]
    set fig(127) [ image create photo -file "../lib/tapes.gif" ]
    set fig(140) [ image create photo -file "../lib/sched.gif" ]

    set fig(0) [ image create photo -file "../lib/back.gif" ]
    set startX 40
    set startY 40
    set icon_size 48
    set incX 70
    set incY 80
    set cntx 0

    set node [ .tp.host.canvas create image $startX $startY -image $fig(0) -tags "<-Back" ]
    .tp.host.canvas bind "<-Back" <Double-1> "repack -60"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "<- Back" -tags nameLabel \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

    # Draw an icon for the Tape drives
    set node [ .tp.host.canvas create image $startX $startY -image $fig(126) -tags tdevs ]
    .tp.host.canvas bind tdevs <Double-1> "set browse 126 ;repack 126"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "Backup\nDevices" -tags tdevs \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

    # Draw an icon for the backup host groups
    set node [ .tp.host.canvas create image $startX $startY -image $fig(123) -tags tgrp ]
    .tp.host.canvas bind tgrp <Double-1> "set browse 123 ;repack 123"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "Backup\nGroup" -tags tgrp \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

    # Draw an icon for the backup jobids
    set node [ .tp.host.canvas create image $startX $startY -image $fig(125) -tags tjob ]
    .tp.host.canvas bind tjob <Double-1> "set browse 125 ;repack 125"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "Backup\nJob" -tags tjob \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

    
    # Draw an icon for the Tapes
    set node [ .tp.host.canvas create image $startX $startY -image $fig(127) -tags tapes ]
#    .tp.host.canvas bind tapes <Double-1> "set browse 127 ;repack 127"
    .tp.host.canvas bind tapes <Double-1> "fillCan 127"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "Media\nManagement" -tags tapes \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

   
    # Draw an icon for the backup schedule
    set node [ .tp.host.canvas create image $startX $startY -image $fig(140) -tags ttime ]
    .tp.host.canvas bind ttime <Double-1> "fillCan 150"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "Backup\nSchedule" -tags ttime \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

}


#
# canRESTORE  - Fill canvas with tape restore icons
#
proc canRESTORE {} {
    # ---------------------  Tape Restore Window  ---------------------
    global currenthost
    global level
    global browse
    global tapeserv
    .tp.title configure -text "Host($currenthost) --> Config --> Archives --> Restore "
    set level 130
    # Enable the accelerators
    bind .tp.host.canvas <Button-3> "rclick .tp.host.canvas"
    # Put in back button
    set fig(132) [ image create photo -file "../lib/tapehost.gif" ]
    set fig(0) [ image create photo -file "../lib/back.gif" ]
    set startX 40
    set startY 40
    set icon_size 48
    set incX 100
    set incY 80
    set cntx 0
    set node [ .tp.host.canvas create image $startX $startY -image $fig(0) -tags "<-Back" ]
    .tp.host.canvas bind "<-Back" <Double-1> "repack -60"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "<- Back" -tags nameLabel \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1
    
    # Get a list of all the groups, and compile a list of all the hosts
    set rlist {}
    set rdata [queryHost $currenthost "TAPE get group" 0 ]
    foreach l $rdata {
	set tmp [lindex [split $l ":" ] 1 ]
	set tmp2 [split $tmp "," ]
	append rlist "$tmp2 "
    }

    # Look for doubles
    set cnt 0
    set hlist {}
    for { set pn 0 } { $pn < [llength $rlist] } { incr pn 1 } {
	set h [lindex $rlist $pn ]
	set flag 0
	for { set hpn 0 } { $hpn < [llength $hlist] } { incr hpn 1 } {
	    if { [lindex $hlist $hpn ] == $h } {
		# Match!!
		set flag 1
	    }
	}
	if { $flag == 0 } {
	    append hlist "$h "
	}
    }

    # Draw icons for each Tape server
    foreach tapeS $hlist {
	set node [ .tp.host.canvas create image $startX $startY -image $fig(132) -tags $tapeS ]
	.tp.host.canvas bind $tapeS <Double-1> "set tapeserv $tapeS ;repack 131 "
	set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
		    -justify center -text "$tapeS" -tags nameLabel \
		    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
	incr startX $incX
	incr cntx 1
	if { $cntx == 6 } {
	    set startX 40
	    incr startY $incY 
	    set cntx 0
	}
    }
}



#
# canMEDIA  - Fill canvas with tape lists
#
proc canMEDIA {} {
    # ---------------------  Tape Media Window  ---------------------
    global currenthost
    global level
    global browse
    global tapeserv
    .tp.title configure -text "Host($currenthost) --> Config --> Archives --> Backup --> Media "
    set level 127
    set browse 127
    # Enable the accelerators
    # bind .tp.host.canvas <Button-3> "rclick .tp.host.canvas"
    # Put in back button
    set fig(127) [ image create photo -file "../lib/tvol.gif" ]
    set fig(0) [ image create photo -file "../lib/back.gif" ]
    set startX 40
    set startY 40
    set icon_size 48
    set node [ .tp.host.canvas create image $startX $startY -image $fig(0) -tags "<-Back" ]
    .tp.host.canvas bind "<-Back" <Double-1> "repack -120"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "<- Back" -tags nameLabel \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    
    set startX 96
    set startY 40
    set X2 128

    # Get a list of all the tapes
    set rdata {}
    set pzn 0
    global tapeserv
    foreach i [queryHost $currenthost "TAPE get drives" 0 ] {
	set drv [lindex [split $i {:} ] 0 ]
	puts stdout "Quering drive $drv"
	append rdata [queryHost $currenthost "TAPE get tapes $drv" 0 ]
	set rdata [queryHost $currenthost "TAPE get tapes $drv" 0 ]
	
	# Draw an Icon for each tape
	for { set position 0 } { $position < [llength $rdata] } { incr position 1 } {
	    set tmp [lindex $rdata $position ]
	
	    set tmp3 [split $tmp {:} ]
	    
	    # This is a new tape
	    incr pzn 1
	    set name "[lindex $tmp3 0 ]   [lindex $tmp3 1 ]"
	    set node [ .tp.host.canvas create image $startX $startY -image $fig(127) -tags tp$pzn ]
	    .tp.host.canvas bind tp$pzn <Double-1> "repack 127 $drv [lindex $tmp3 0 ]"
	    set nameLabel [ .tp.host.canvas create text $X2 $startY \
		    -anchor w \
		    -justify left -text "$name" -tags tp$pzn \
		    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
	    incr startY 20
	    
	}
    }
}


#  ***************************************  NOT USED !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
# canTSCHED  - Tape scheduling window
#
proc canTSCHED {} {
# ---------------------  Tape Restore Window  ---------------------
    global currenthost
    global level
    global browse
    global tapeserv
    .tp.title configure -text "Host($currenthost) --> Config --> Archives --> Restore "
    set level 130
    # Enable the accelerators
    bind .tp.host.canvas <Button-3> "rclick .tp.host.canvas"
    # Put in back button
    set fig(132) [ image create photo -file "../lib/tapehost.gif" ]
    set fig(0) [ image create photo -file "../lib/back.gif" ]
    set startX 40
    set startY 40
    set icon_size 48
    set incX 70
    set incY 80
    set cntx 0
    set node [ .tp.host.canvas create image $startX $startY -image $fig(0) -tags "<-Back" ]
    .tp.host.canvas bind "<-Back" <Double-1> "repack -60"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "<- Back" -tags nameLabel \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1
    
    # Draw icons for each Tape server
    foreach tapeS [ getTapeHost ] {
	set node [ .tp.host.canvas create image $startX $startY -image $fig(132) -tags $tapeS ]
	.tp.host.canvas bind $tapeS <Double-1> "set tapeserv $tapeS ;repack 131 "
	set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
		    -justify center -text "$tapeS" -tags nameLabel \
		    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
	incr startX $incX 
	if { $cntx == 8 } {
	    set startX 40
	    incr startY $incY 
	    incr cntx 1
	}
    }
}


#
# getTapeHost - Read the MAT hosts file and return a list
#               of all tape servers.
#
proc getTapeHost {} {
    global active_hosts
    set retval {}
    foreach l $active_hosts {
	set tmp [ lindex [split $l ":" ] 3 ]
	foreach e [ split $tmp "," ] {
	    if { $e == "TAPE" } {
		# It's a tape host
		lappend retval [ lindex [split $l ":" ] 0 ]
	    }
	}
    }
    return $retval
}



#
# tapeJobs - Manage the tape job requests.  This GUI allows the user to set the
#            the following:   Job ID#   - Used to schedule the job
#                             Backup Gruop  - The hosts to backup
#                             Retention     - How long to hold a tape
#                             Drive         - Desired backup device
#                             Pool          - Pool of tapes available for the backup
#            Set jobnum to 9999 for new jobs
#
set sd {}
proc tapeJobs { tapeserv jobnum data args } {
    if { [string length $args ] != 0 } {
        append line " $args"
    }
    global errno
    global sd
    global textd
    set sd "/"
    set host $tapeserv
    if { [winfo exists .tjob ] != 0 } {
	destroy .tjob
    }

    toplevel .tjob -class Dialog
    wm title .tjob "Backup Job"
    wm iconname .tjob "Backup Job"
    entry .tjob.hidden1 
    entry .tjob.hidden2
    .tjob.hidden1 insert end $tapeserv
    .tjob.hidden2 insert end $jobnum
    

    frame .tjob.f1 -bd 2 -relief "flat"
    frame .tjob.f2 -bd 2 -relief "flat"
    frame .tjob.f3 -bd 2 -relief "flat"
    frame .tjob.f4 -bd 2 -relief "flat"
    # Browse frame
    frame .tjob.f5 -bd 2 -relief "flat"
    # Bottom Buttons
    frame .tjob.f7 -bd 2 -relief "raised"

    frame .tjob.f20 -bd 0 -relief "flat"
    pack .tjob.f20 -anchor w
    frame .tjob.f20.f1 -bd 2 -relief "flat"
    grid .tjob.f20.f1
    button .tjob.hb1
    button .tjob.f20.f1.b1 -relief groove -text "Description" -command {
	set c1 [ .tjob.hb1 cget -activebackground ]
	set c2 [ .tjob.hb1 cget -background ]
	.tjob.f20.f1.b1 configure -relief flat -bg $c2 -activebackground $c1
	.tjob.f20.f1.b2 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b3 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b4 configure -relief groove -bg $c1 -activebackground $c2
	pack forget .tjob.f1 .tjob.f2 .tjob.f3 .tjob.f7 .tjob.f5
	pack .tjob.f1 .tjob.f7
    }
    button .tjob.f20.f1.b2 -relief groove -text "Group" -command {
	set c1 [ .tjob.hb1 cget -activebackground ]
	set c2 [ .tjob.hb1 cget -background ]
	.tjob.f20.f1.b1 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b2 configure -relief flat -bg $c2 -activebackground $c1
	.tjob.f20.f1.b3 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b4 configure -relief groove -bg $c1 -activebackground $c2
	pack forget .tjob.f1 .tjob.f2 .tjob.f3 .tjob.f7 .tjob.f5
	pack .tjob.f2 .tjob.f7
    }
    button .tjob.f20.f1.b3 -relief groove -text "Target" -command {
	set c1 [ .tjob.hb1 cget -activebackground ]
	set c2 [ .tjob.hb1 cget -background ]
	.tjob.f20.f1.b1 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b2 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b3 configure -relief flat -bg $c2 -activebackground $c1
	.tjob.f20.f1.b4 configure -relief groove -bg $c1 -activebackground $c2
	pack forget .tjob.f1 .tjob.f2 .tjob.f3 .tjob.f7 .tjob.f5
	pack .tjob.f3 .tjob.f7
    }
    button .tjob.f20.f1.b4 -relief groove -text "Files" -command {
	set c1 [ .tjob.hb1 cget -activebackground ]
	set c2 [ .tjob.hb1 cget -background ]
	.tjob.f20.f1.b1 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b2 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b3 configure -relief groove -bg $c1 -activebackground $c2
	.tjob.f20.f1.b4 configure -relief flat -bg $c2 -activebackground $c1
	pack forget .tjob.f1 .tjob.f2 .tjob.f3 .tjob.f7 .tjob.f5
	pack .tjob.f5 .tjob.f7
    }
    grid .tjob.f20.f1.b1 .tjob.f20.f1.b2 .tjob.f20.f1.b3 .tjob.f20.f1.b4 -sticky w
    .tjob.f20.f1.b1 invoke

    # ------------  Description Frame  -------------------------------
    label .tjob.f1.l1 -text "Job #" -width 6
    entry .tjob.f1.e1 -width 4
    label .tjob.f1.l2 -text "Description:" -width 12
    entry .tjob.f1.e2 -width 32
    grid .tjob.f1.l1 .tjob.f1.e1 -sticky w
    grid .tjob.f1.l2 .tjob.f1.e2 -sticky w

    # ------------  Groups Frame  ------------------------------------
    label .tjob.f2.l1 -text "Backup Group" -width 15
    menubutton .tjob.f2.b1 -text "..." -menu .tjob.f2.b1.m -relief raised -width 2
    menu .tjob.f2.b1.m -tearoff 0 
    set rdata [queryHost $tapeserv "TAPE get groups" 0 ]
    foreach l $rdata {
	set val [lindex [split $l ":"] 0]
	.tjob.f2.b1.m add command -label $val -command ".tjob.f2.e1 configure -state normal ; .tjob.f2.e1 delete 0 end ; .tjob.f2.e1 insert end \"$val\" ; .tjob.f2.e1 configure -state disabled "
    }
    entry .tjob.f2.e1 -width 7
    grid .tjob.f2.l1 .tjob.f2.b1 .tjob.f2.e1

    # ------------  Target Frame  -----------------------------------
    label .tjob.f3.l1 -text "Backup Device:" -width 18
    menubutton .tjob.f3.b1 -text "..." -menu .tjob.f3.b1.m -relief raised -width 3
    menu .tjob.f3.b1.m -tearoff 0 
    set vals {}
    set rdata [queryHost $tapeserv "TAPE get drives" 0 ]
    for { set p 0 } { $p < [llength $rdata] } { incr p 1 } {
	set tmp [lindex $rdata $p ]
	set t1 [lindex [split [lindex $rdata $p ] {:} ] 1 ]
	if { $t1 != {} } {
	    lappend vals $t1
	}
    }
    foreach val $vals {
	.tjob.f3.b1.m add command -label $val -command ".tjob.f3.e1 configure -state normal ; .tjob.f3.e1 delete 0 end ; .tjob.f3.e1 insert end \"$val\" ; .tjob.f3.e1 configure -state disabled "
    }
    entry .tjob.f3.e1 -width 16
    grid .tjob.f3.l1 .tjob.f3.b1 .tjob.f3.e1

    label .tjob.f3.l3 -text "Retention (weeks):" -width 18
    menubutton .tjob.f3.b3 -text "..." -menu .tjob.f3.b3.m -relief raised -width 3
    menu .tjob.f3.b3.m -tearoff 0 
    set vals { 1 2 4 8 12 16 26 52 }
    foreach val $vals {
	.tjob.f3.b3.m add command -label $val -command ".tjob.f3.e3 delete 0 end ; .tjob.f3.e3 insert end \"$val\" "
    }
    entry .tjob.f3.e3 -width 7
    grid .tjob.f3.l3 .tjob.f3.b3 .tjob.f3.e3

    label .tjob.f3.l2 -text "Backup Pool:" -width 18
    entry .tjob.f3.e2 -width 7
    grid .tjob.f3.l2 -columnspan 2
    grid .tjob.f3.e2 -row 2 -column 2


    # -------------------   Browse Host and File Selection  ---------------------------
    label .tjob.f5.l1 -text "Requested files to include in backup"
    frame .tjob.f5.f1 -bd 2 -relief "flat"
    label .tjob.f5.f1.l1 -text "Browse Host:" -width 12
    button .tjob.f5.f1.b1 -text "..." -padx 0 -command "FillGrpList .tjob $host .tjob.f2.e1 .tjob.f5.f1.e1"
    entry .tjob.f5.f1.e1 -width 14 -font $textd
    .tjob.f5.f1.e1 delete 0 end
    .tjob.f5.f1.e1 insert end $tapeserv
    grid .tjob.f5.f1.l1 .tjob.f5.f1.b1 .tjob.f5.f1.e1
    
    label .tjob.f5.f1.l2 -text "Current Dir:" -width 12
    entry .tjob.f5.f1.e2 -width 32 -font $textd
    grid .tjob.f5.f1.l2
    grid .tjob.f5.f1.e2 -row 1 -column 1 -columnspan 2
    #pack .tjob.f5.f1.l1 .tjob.f5.f1.b1 .tjob.f5.f1.e1 .tjob.f5.f1.l2 .tjob.f5.f1.e2 -fill x -side left
    pack .tjob.f5.f1 -side top
 
    frame .tjob.f5.f2 -bd 2 -relief "flat"
    scrollbar .tjob.f5.f2.vscroll -orient vertical -command ".tjob.f5.f2.text yview"
    scrollbar .tjob.f5.f2.hscroll -orient horizontal -command ".tjob.f5.f2.text xview"
    listbox .tjob.f5.f2.text -selectforeground white -selectbackground #000080 \
	    -yscrollcommand {.tjob.f5.f2.vscroll set } \
	    -xscrollcommand {.tjob.f5.f2.hscroll set } -height 8 -width 42 -font $textd
    .tjob.f5.f2.text config -cursor hand2
    pack .tjob.f5.f2.vscroll -side right -fill y
    pack .tjob.f5.f2.hscroll -side bottom -fill x
    pack .tjob.f5.f2.text -side left -expand 1 -fill both

    frame .tjob.f5.f4 -bd 2 -relief "flat"
    button .tjob.f5.f4.b1 -text "\\/ Add \\/" -command {
	if { [.tjob.f5.f2.text curselection ] != "" } {
	    .tjob.f5.f6.text insert end [string trimright "[.tjob.f5.f1.e2 get ][.tjob.f5.f2.text get [.tjob.f5.f2.text curselection ]]" {.} ]
	    add2backup [.tjob.f1.e1 get ] [string trimright "[.tjob.f5.f1.e2 get ][.tjob.f5.f2.text get [.tjob.f5.f2.text curselection ]]" {.} ]
	}
    }
    button .tjob.f5.f4.b2 -text "/\\ Remove /\\" -command {
	if { [.tjob.f5.f6.text curselection ] != "" } {
	    delbackup [.tjob.f1.e1 get ] [.tjob.f5.f6.text get [.tjob.f5.f6.text curselection ] ]
	    .tjob.f5.f6.text delete [.tjob.f5.f6.text curselection ]
	}
    }
    pack .tjob.f5.f4.b1 .tjob.f5.f4.b2 -side left -fill x
    
    frame .tjob.f5.f6 -bd 2 -relief "flat"
    scrollbar .tjob.f5.f6.vscroll -orient vertical -command ".tjob.f5.f6.text yview" 
    scrollbar .tjob.f5.f6.hscroll -orient horizontal -command ".tjob.f5.f6.text xview"
    listbox .tjob.f5.f6.text -selectforeground white -selectbackground #000080 \
	    -yscrollcommand {.tjob.f5.f6.vscroll set } \
	    -xscrollcommand {.tjob.f5.f6.hscroll set } -height 6 -width 42 -font $textd
    .tjob.f5.f6.text config -cursor hand2
    pack .tjob.f5.f6.vscroll -side right -fill y
    pack .tjob.f5.f6.hscroll -side bottom -fill x
    pack .tjob.f5.f6.text -side left -expand 1 -fill both
    
    pack .tjob.f5.f2 .tjob.f5.f4 .tjob.f5.f6 -side top -fill both
    

    # -----------------  Bottom Buttons  ---------------------------------------------------
    button .tjob.f7.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	if { [ .tjob.hidden2 get ] != 9999 } {
	    set cmd "TAPE mod jobids [ .tjob.hidden2 get ] "
	} else {
	    set cmd "TAPE add jobids "
	}
	append cmd "[ .tjob.f1.e1 get ]:[ .tjob.f2.e1 get ]:[ week2sec [ .tjob.f3.e3 get ] ]:"
	# Will have to go and get the drive info from the machine
	set mdata [queryHost [ .tjob.hidden1 get ] "TAPE get drives" 0 ]
	foreach m $mdata {
	    if { [ .tjob.f3.e1 get ] == [lindex [split $m ":"] 1] } {
		append cmd "[lindex [split $m ":"] 0]:"
	    }
	}
	append cmd "[ .tjob.f3.e2 get ]:0:[ .tjob.f1.e2 get ]"
	set returnval [queryHost [.tjob.hidden1 get] $cmd 1 ]
	if { $errno != 0 } {
	    error 2001
	    puts stdout "ERROR:  Unable to add new job to [.tjob.hidden1 get]"
	} else {
	    repack 125
	    destroy .tjob
	}
    }
    button .tjob.f7.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	destroy .tjob
    }
    button .tjob.f7.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	help tapejob
    }
    button .tjob.f7.b4 -text "Backup Now" -command {
	if { [ .tjob.f1.e1 get ] != {} } {
	    set cmd "TAPE backup [ .tjob.f1.e1 get ] 0"
	    set returnval [queryHost [.tjob.hidden1 get] $cmd 1 ]
	}
	if { $errno != 0 } {
	    error 2502
	    puts stdout "ERROR:  Backup job [.tjob.hidden1 get] did not start"
	} 
	destroy .tjob
    }
    pack .tjob.f7.b1 .tjob.f7.b2 .tjob.f7.b3 .tjob.f7.b4 -side left -fill x 
    

    # ------------ Bindings --------------------
    bind .tjob.f5.f2.text <Button-3> {
	.tjob.f5.f6.text insert end [string trimright "[.tjob.f5.f1.e2 get ][.tjob.f5.f2.text get @%x,%y ]" {.} ]
	add2backup [.tjob.f1.e1 get ] [string trimright "[.tjob.f5.f1.e2 get ][.tjob.f5.f2.text get @%x,%y ]" {.} ]
    }
    bind .tjob.f5.f6.text <Button-3> {
	delbackup [.tjob.f1.e1 get ] [string trimright "[.tjob.f5.f6.text get @%x,%y ]" {.} ]
	.tjob.f5.f6.text delete @%x,%y
    }
    bind .tjob.f5.f6.text <Double-1> {
	if { [.tjob.f5.f6.text curselection ] != "" } {
	    .tjob.f5.f6.text delete [.tjob.f5.f6.text curselection ]
	}
    }
    bind .tjob.f5.f2.text <Double-1> {
	if { [.tjob.f5.f2.text curselection ] != {} } {
	    set t [.tjob.f5.f2.text get [.tjob.f5.f2.text curselection ] ]
	    if { $t == ".." } {
		# Need to go down a directory
		global sd
		set tmp [string trimright $sd {/} ]
		set ldir [string last {/} $tmp ]
		if { $ldir <= 0 } {
		    set sd "/"
		} else {
		    set tmp [string range $sd 0 $ldir ]
		    set sd $tmp
		}
		getDir [.tjob.f5.f1.e1 get] $sd .tjob.f5.f2.text
	    } elseif { [string range $t end end ] == {/} } {
		global sd
		#set sd [string trimright [.tjob.f5.f2.text get [.tjob.f5.f2.text curselection ] ] {/} ]
		set sd "[.tjob.f5.f1.e2 get ][.tjob.f5.f2.text get [.tjob.f5.f2.text curselection ] ]"
		#set sd [.tjob.f5.f2.text get [.tjob.f5.f2.text curselection ] ]
		getDir [.tjob.f5.f1.e1 get] $sd .tjob.f5.f2.text
	    }
	}
	.tjob.f5.f1.e2 delete 0 end
	.tjob.f5.f1.e2 insert end "$sd"

    }


    # ---------- Fill boxes with data --------------------------
    global sd
    set sd "/"
    .tjob.f5.f1.e2 delete 0 end
    .tjob.f5.f1.e2 insert end "$sd"
    .tjob.f1.e1 configure -state normal
    .tjob.f1.e1 delete 0 end
    .tjob.f2.e1 configure -state normal
    .tjob.f2.e1 delete 0 end
    .tjob.f3.e3 delete 0 end
    .tjob.f3.e1 delete 0 end
    .tjob.f3.e2 delete 0 end
    .tjob.f1.e2 delete 0 end
    .tjob.f5.f6.text delete 0 end
    if { $jobnum != 9999 } {
	# Go get the jobdata from the tape server 
	set rdata [queryHost [ .tjob.hidden1 get ] "TAPE get jobids" 0 ]
	foreach l $rdata {
	    if { $jobnum == [lindex [split $l ":"] 0] } {
		set tmp [split $l ":"]
		.tjob.f1.e1 insert end [lindex $tmp 0]
		.tjob.f2.e1 insert end [lindex $tmp 1]
		.tjob.f3.e2 insert end [lindex $tmp 4]
		.tjob.f1.e2 insert end [lindex $tmp 6]
		# Populate the retention
		.tjob.f3.e3 insert end [ sec2week [lindex $tmp 2] ]
		# Will have to go and get the group info from the machine
		set mdata [queryHost [ .tjob.hidden1 get ] "TAPE get drives" 0 ]
		foreach m $mdata {
		    if { [lindex $tmp 3] == [lindex [split $m ":"] 0] } {
			.tjob.f3.e1 insert end [lindex [split $m ":"] 1]
		    }
		}
	    }
	}
	.tjob.f1.e1 configure -state disabled
	.tjob.f2.e1 configure -state disabled

	# ---- It's an old job, so populate the selected list
	set mdata [queryHost [ .tjob.hidden1 get ] "TAPE get backuplist [ .tjob.hidden2 get ]" 0 ]
	foreach m $mdata {
	    .tjob.f5.f6.text insert end $m
	}
    }
    getDir [.tjob.f5.f1.e1 get] $sd .tjob.f5.f2.text
}


#
# add2backup - Add an entry to the backup list
#
proc add2backup { jobnum entry args } {
    global currenthost
    global errno
    if { [string length $args ] != 0 } {
        append entry " $args"
    }
    if { ( $jobnum != {} ) & ( $entry != {} ) } {
	set mdata [queryHost $currenthost "TAPE add backuplist $jobnum $entry" 0 ]
	if { $errno != 0 } {
	    error 501
	    puts stdout "ERROR:  Can't update job backup list for job# $jobnum"
	}
    }
}
 

#
# delbackup - Delete an entry from the backup list
#
proc delbackup { jobnum entry args } {
    global currenthost
    global errno
    if { [string length $args ] != 0 } {
        append entry " $args"
    }
    if { ( $jobnum != {} ) & ( $entry != {} ) } {
	set mdata [queryHost $currenthost "TAPE del backuplist $jobnum $entry" 0 ]
	if { $errno != 0 } {
	    error 402
	    puts stdout "ERROR:  Can't update job backup list for job# $jobnum"
	}
    }
}


#
# week2sec - Convert the number of weeks to seconds ad return it
#
proc week2sec { week } {
    return [expr $week * 604800 ]
}


#
# sec2week - Convert the number of seconds to weeks and return it
#
proc sec2week { sec } {
    return [expr $sec / 604800 ]
}


#
# tapeGroup - Manage the tape host groups.  The group controls which
#             hosts are included in a backup job.
#
# USE: tapeGroup CurrentHost CurrentGroup SelectedData
#             CurrentHost = host to update
#             CurrentGroup = The current selected group
#                           If 9999 then it is a new entry
#             SelectedData = line of data to modify
# NOTE: Data must be seperated by a single space or a singel tab
#
proc tapeGroup { currenthost currentgroup line args } {
    if { [string length $args ] != 0 } {
        append line " $args"
    }
    global errno
    set host $currenthost
    if { [winfo exists .tgrp ] == 0 } {
	toplevel .tgrp -class Dialog
	wm title .tgrp "Backup Group Editor: $currenthost"
	wm iconname .tgrp "Backup Group"
	entry .tgrp.hidden1 
	entry .tgrp.hidden2
	# NOT GOOD
	.tgrp.hidden1 insert end $currenthost
	.tgrp.hidden2 insert end $currentgroup

	frame .tgrp.f1 -bd 2 -relief "raised"
	label .tgrp.f1.l1 -text "Group:" -width 6
	entry .tgrp.f1.e1 -width 24
	pack .tgrp.f1.l1 .tgrp.f1.e1 -fill both -side left -expand 1

	frame .tgrp.f2 -bd 2 -relief "raised"
	label .tgrp.f2.l1 -text "GID #:" -width 6
	entry .tgrp.f2.e1 -width 10
	pack .tgrp.f2.l1 .tgrp.f2.e1 -fill both -side left -expand 1

	frame .tgrp.f3 -bd 2 -relief "raised"
	label .tgrp.f3.l1 -text "Member Hosts" 
	pack .tgrp.f3.l1 -fill x
	frame .tgrp.f3.f1 -bd 2 -relief "flat"
	button .tgrp.f3.f1.b0 -text "..." -padx 0 -command "FillTapeList .tgrp $currenthost .tgrp.f3.f1.e1"
	entry .tgrp.f3.f1.e1 -width 10
	button .tgrp.f3.f1.b1 -text "Add" -command {
	    if { [.tgrp.f3.f1.e1 get] != "" } {
		.tgrp.f3.f2.text insert end [.tgrp.f3.f1.e1 get]
	    }
	    .tgrp.f3.f1.e1 delete 0 end
	}
	pack .tgrp.f3.f1.b0 .tgrp.f3.f1.e1 .tgrp.f3.f1.b1 -fill both -side left -expand 1
	frame .tgrp.f3.f2 -bd 2 -relief "flat"
	scrollbar .tgrp.f3.f2.vscroll -orient vertical -command ".tgrp.f3.f2.text yview" 
	listbox .tgrp.f3.f2.text -selectforeground white -selectbackground #000080 -yscrollcommand {.tgrp.f3.f2.vscroll set } -height 5 -width 10
	.tgrp.f3.f2.text config -cursor hand2
	pack .tgrp.f3.f2.vscroll -side left -fill y
	pack .tgrp.f3.f2.text -side left -expand 1 -fill both
	button .tgrp.f3.f2.b1 -text "Delete" -foreground red -activeforeground #a00000 -command {
	    if { [.tgrp.f3.f2.text curselection ] != "" } {
		.tgrp.f3.f2.text delete [.tgrp.f3.f2.text curselection ]
	    }
	}
	pack .tgrp.f3.f2.b1 -side left
	pack .tgrp.f3.f1 .tgrp.f3.f2 -side top -fill both -expand 1

	frame .tgrp.f4 -bd 2 -relief "flat"
	button .tgrp.f4.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.tgrp.f2.e1 get] > -2 || [.tgrp.f2.e1 get] < 65536 } {
		if { [.tgrp.f1.e1 get] != "" } {
		    set tmpline "[ .tgrp.f2.e1 get]:"
		    foreach hst [.tgrp.f3.f2.text get 0 end] {
			append tmpline "[string tolower [string trimright $hst]],"
		    }
		    set newline [string trimright $tmpline "," ] 
		    append newline ":[ .tgrp.f1.e1 get]"
		    set currentgroup [.tgrp.hidden2 get]
		    if { $currentgroup == 9999 } {
			set cmd "TAPE add group $newline"
		    } else {
			set cmd "TAPE mod group $currentgroup $newline"
		    }
		    set returnval [queryHost $currenthost $cmd 1 ]
		    if { $errno != 0 } {
			error 91
			puts stdout "ERROR:  Unable to update tape groups on $currenthost\n       Check the files permissions"
		    } else {
			repack 123
			destroy .tgrp
		    }
		}
	    }
	}
	button .tgrp.f4.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .tgrp
	}
	button .tgrp.f4.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help tapegroup
	}
	pack .tgrp.f4.b1 .tgrp.f4.b2 .tgrp.f4.b3 -side left -fill x 
	pack .tgrp.f2 .tgrp.f1 .tgrp.f3 .tgrp.f4 -fill both -expand 1
	# ----- Bindings -----
	bind .tgrp.f1.e1 <Return> {
	    focus .tgrp.f2.e1
	}
	bind .tgrp.f2.e1 <Return> {
	    focus .tgrp.f3.f1.e1
	}
	bind .tgrp.f3.f1.e1 <Return> {
	    if { [.tgrp.f3.f1.e1 get] != "" } {
		.tgrp.f3.f2.text insert end [.tgrp.f3.f1.e1 get]
	    }
	    .tgrp.f3.f1.e1 delete 0 end
	}
    } else {
	.tgrp.hidden1 delete 0 end
	.tgrp.hidden2 delete 0 end
	.tgrp.hidden1 insert end $currenthost
	.tgrp.hidden2 insert end $currentgroup
	wm title .tgrp "Groups Editor: $currenthost"
    }

    # Fill boxes with data
    if { $currentgroup != 9999 } {
	.tgrp.f1.e1 delete 0 end
	.tgrp.f2.e1 delete 0 end
	.tgrp.f3.f2.text delete 0 end
	# Will have to go and get the group info from the machine
	set rdata [queryHost $currenthost "TAPE get groups" 0 ]
	foreach l $rdata {
	    if { $currentgroup == [lindex [split $l ":"] 0] } {
		.tgrp.f1.e1 insert end [lindex [split $l ":"] 2]
		.tgrp.f2.e1 insert end [lindex [split $l ":"] 0]
		set tmp [lindex [split $l ":"] 1]
		foreach user [split $tmp ","] {
		    .tgrp.f3.f2.text insert end $user
		}
	    }
	}
    }
}


#
# tapeDevs - Add and delete tape devices.
#            Set drivenum to 9999 for new drives
#
proc tapeDevs { host drivenum } {
    global errno
    if { [winfo exists .bdev ] == 0 } {
	toplevel .bdev -class Dialog
	wm title .bdev "Backup Device Editor: $host"
	wm iconname .bdev "Backup Devices"
	entry .bdev.hidden1
	entry .bdev.hidden2
	# NOT GOOD
	.bdev.hidden1 insert end $host
	.bdev.hidden2 insert end $drivenum

	frame .bdev.f1 -bd 2 -relief "raised"
	label .bdev.f1.l1 -text "ID # :" -width 7
	entry .bdev.f1.e1 -width 4
	pack .bdev.f1.l1 .bdev.f1.e1 -fill x -side left

	frame .bdev.f2 -bd 2 -relief "raised"
	label .bdev.f2.l1 -text "Name:" -width 6
	entry .bdev.f2.e1 -width 20
	pack .bdev.f2.l1 .bdev.f2.e1 -fill x -side left -expand 1

	frame .bdev.f3 -bd 2 -relief "raised"
	label .bdev.f3.l1 -text "Path:" -width 6
	entry .bdev.f3.e1 -width 20
	pack .bdev.f3.l1 .bdev.f3.e1 -fill x -side left -expand 1

	frame .bdev.f4 -bd 2 -relief "raised"
	frame .bdev.f4.f1 -bd 2 -relief "raised"
	label .bdev.f4.f1.l1 -text "Picker Path:" -width 13
	entry .bdev.f4.f1.e1 -width 20
	pack .bdev.f4.f1.l1 .bdev.f4.f1.e1 -fill x -side left -expand 1

	frame .bdev.f4.f2 -bd 2 -relief "raised"
	label .bdev.f4.f2.l1 -text "Number of slots:" -width 17
	entry .bdev.f4.f2.e1 -width 4
	pack .bdev.f4.f2.l1 .bdev.f4.f2.e1 -fill x -side left

	pack .bdev.f4.f1 .bdev.f4.f2 -side top -fill x -expand 1

	frame .bdev.f8 -bd 2 -relief "flat"
	button .bdev.f8.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.bdev.f1.e1 get ] > 0 || [.bdev.f1.e1 get] < 999 } {
		if { [.bdev.f3.e1 get] != "" } {
		    set newline "[.bdev.f1.e1 get ]:[.bdev.f2.e1 get ]:[.bdev.f3.e1 get ]:[.bdev.f4.f1.e1 get ]:[.bdev.f4.f2.e1 get ]"
		    set drive [.bdev.hidden2 get]
		    if { [ .bdev.hidden2 get ] == 9999 } {
			set cmd "TAPE add drive $newline"
		    } else {
			set cmd "TAPE mod drive $drive $newline"
		    }
		    set returnval [queryHost [.bdev.hidden1 get] $cmd 1 ]
		    if { $errno != 0 } {
			error 502
			puts stdout "ERROR:  Can't add backup device or file.\n       Check the files permissions"
		    } else {
			repack 126
			destroy .bdev
		    }
		} else {
		    puts stdout "ERROR:  Must provide a path for the backup file or device"
		    error 503
		}
	    } else {
		puts stdout "Pick a reasonable number like 2 or 6"
		error 504
	    }
	}
	button .bdev.f8.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .bdev
	}
	button .bdev.f8.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help tapedevice
	}
	pack .bdev.f8.b1 .bdev.f8.b2 .bdev.f8.b3 -side left -fill x 


	pack .bdev.f1 .bdev.f2 .bdev.f3 .bdev.f4 .bdev.f8 -side top -fill x -expand 1
    } else {
	# Window exists
	.bdev.hidden1 delete 0 end
	.bdev.hidden2 delete 0 end
	.bdev.hidden1 insert end $host
	.bdev.hidden2 insert end $drivenum
    }
    
    # --------  Populate the widgets  --------------
    .bdev.f1.e1 configure -state normal
    .bdev.f1.e1 delete 0 end
    .bdev.f2.e1 delete 0 end
    .bdev.f3.e1 delete 0 end
    .bdev.f4.f1.e1 delete 0 end
    .bdev.f4.f2.e1 configure -state normal
    .bdev.f4.f2.e1 delete 0 end
    if { $drivenum != "9999" } {
	# Get the tape info
	set rdata [queryHost $host "TAPE get drives" 0 ]
	foreach l $rdata {
	    set val [lindex [split $l ":"] 0]
	    if { $val == $drivenum } {
		# This is the backup device to work with
		.bdev.f1.e1 insert end [lindex [split $l ":"] 0]
		.bdev.f2.e1 insert end [lindex [split $l ":"] 1]
		.bdev.f3.e1 insert end [lindex [split $l ":"] 2]
		.bdev.f4.f1.e1 insert end [lindex [split $l ":"] 3]
		.bdev.f4.f2.e1 insert end [lindex [split $l ":"] 4]
	    }
	}
	.bdev.f1.e1 configure -state disabled
	.bdev.f4.f2.e1 configure -state disabled
    }
}


#   FillTapeList  - Fill a listbox with all the hosts where MAT is installed.  
#                   Return the selected host
#                      p = parent window
#                      chost = currenthost
#                      ent  = entry widget to operate on
proc FillTapeList {p chost ent } {
    $p config -cursor watch
    global active_hosts

    set w .hList
    if ![winfo exists $w] {
        toplevel $w 
        wm withdraw $w
        frame $w.f1
        scrollbar $w.f1.vscroll -orient vertical -command "$w.f1.text yview" 
        listbox $w.f1.text -selectforeground white -selectbackground #000080 -yscrollcommand "$w.f1.vscroll set " -height 10 -width 20
        pack $w.f1.vscroll -side left -fill y
        pack $w.f1.text -side left -expand 1 -fill both
        pack $w.f1
        entry $w.hide
    }
    $w.f1.text delete 0 end
    $w.hide delete 0 end
    $w.hide insert end $ent
    foreach line $active_hosts {
	# The line of data contains numerous fields
	set hostname [ lindex [split $line ":" ] 0 ]
	$w.f1.text insert end $hostname
    }
    set x [winfo pointerx $p ]
    set y [winfo pointery $p ]
    wm geometry $w +$x+$y
    wm deiconify $w
    raise $w
    $p config -cursor xterm
    bind $w.f1.text <Double-1> {
        set tmp [ .hList.f1.text curselection ]
        if { $tmp != {} } { 
            [.hList.hide get] delete 0 end
            [.hList.hide get] insert end [.hList.f1.text get $tmp ]
            destroy .hList
        }
    }
}


#   FillGrpList  -  Fill a listbox with all the hosts in a given group.
#                   Return the selected host
#                      p = parent window
#                      chost = currenthost
#                      grp = widget containing the group
#                      ent  = entry widget to operate on
proc FillGrpList {p chost grp ent } {
    $p config -cursor watch

    set w .gList
    if ![winfo exists $w] {
        toplevel $w 
        wm withdraw $w
        frame $w.f1
        scrollbar $w.f1.vscroll -orient vertical -command "$w.f1.text yview" 
        listbox $w.f1.text -selectforeground white -selectbackground #000080 -yscrollcommand "$w.f1.vscroll set " -height 10 -width 20
        pack $w.f1.vscroll -side left -fill y
        pack $w.f1.text -side left -expand 1 -fill both
        pack $w.f1
        entry $w.hide
    }
    $w.f1.text delete 0 end
    $w.hide delete 0 end
    $w.hide insert end $ent
    
    set rdata [queryHost $chost "TAPE get groups" 0 ]
    foreach l $rdata {
	set val [lindex [split $l ":"] 0]
	if { $val == [ $grp get ] } {
	    # This is the group list to work with
	    set tmp [split [lindex [split $l ":"] 1] "," ]
	    foreach h $tmp {
		$w.f1.text insert end $h
	    }
	}
    }

    set x [winfo pointerx $p ]
    set y [winfo pointery $p ]
    wm geometry $w +$x+$y
    wm deiconify $w
    raise $w
    $p config -cursor xterm
    bind $w.f1.text <Double-1> {
        set tmp [ .gList.f1.text curselection ]
        if { $tmp != {} } { 
            [.gList.hide get] delete 0 end
            [.gList.hide get] insert end [.gList.f1.text get $tmp ]
            destroy .gList
        }
    }
}


#
# editTape - Add, delete, and modify backup media properties.
#            Set drivenum to 9999 for new drives
#
proc editTape { host tapeid } {
    global errno
    if { [winfo exists .etp ] == 0 } {
	toplevel .etp -class Dialog
	wm title .etp "Backup Media Editor: $host"
	wm iconname .etp "Backup Media"
	entry .etp.hidden1
	entry .etp.hidden2
	# NOT GOOD
	.etp.hidden1 insert end $host
	.etp.hidden2 insert end $tapeid

	frame .etp.f1 -bd 2 -relief "raised"
	label .etp.f1.l1 -text "ID # :" -width 7
	entry .etp.f1.e1 -width 4 -relief "flat"
	pack .etp.f1.l1 .etp.f1.e1 -fill x -side left

	frame .etp.f2 -bd 2 -relief "raised"
	label .etp.f2.l1 -text "Name:" -width 6
	entry .etp.f2.e1 -width 20
	pack .etp.f2.l1 .etp.f2.e1 -fill x -side left -expand 1

	frame .etp.f3 -bd 2 -relief "raised"
	label .etp.f3.l1 -text "Backup Pool#:" -width 14
	entry .etp.f3.e1 -width 4
	pack .etp.f3.l1 .etp.f3.e1 -fill x -side left -expand 1

	frame .etp.f4 -bd 2 -relief "raised"
	label .etp.f4.l1 -text "Backup Device:" -width 15
	menubutton .etp.f4.b1 -text "..." -menu .etp.f4.b1.m -relief raised -width 3
	menu .etp.f4.b1.m -tearoff 0 
	set vals {}
	set rdata [queryHost [ .etp.hidden1 get ] "TAPE get drives" 0 ]
	for { set p 0 } { $p < [llength $rdata] } { incr p 1 } {
	    set tmp [lindex $rdata $p ]
	    set t1 [lindex [split [lindex $rdata $p ] {:} ] 1 ]
	    if { $t1 != {} } {
		lappend vals $t1
	    }
	}
	foreach val $vals {
	    .etp.f4.b1.m add command -label $val -command ".etp.f4.e1 configure -state normal ; .etp.f4.e1 delete 0 end ; .etp.f4.e1 insert end \"$val\" ; .etp.f4.e1 configure -state disabled "
	}
	entry .etp.f4.e1 -width 16
	pack .etp.f4.l1 .etp.f4.b1 .etp.f4.e1 -fill x -side left
	

	frame .etp.f5 -bd 2 -relief "raised"
	label .etp.f5.l2 -text "Raw Capacity (KB):" -width 20
	menubutton .etp.f5.b2 -text "..." -menu .etp.f5.b2.m -relief raised -width 3
	menu .etp.f5.b2.m -tearoff 0 
	set vals { 120 150 240 1300 2000 2300 2600 4000 7000 20000 35000 40000 }
	foreach val $vals {
	    .etp.f5.b2.m add command -label $val -command ".etp.f5.e2 delete 0 end ; .etp.f5.e2 insert end \"$val\" "
	}
	entry .etp.f5.e2 -width 7
	pack .etp.f5.l2 .etp.f5.b2 .etp.f5.e2 -fill both -side left
	

	frame .etp.f8 -bd 2 -relief "flat"
	button .etp.f8.b1 -text "Apply" -foreground red -activeforeground #a00000 -command {
	    if { [.etp.f4.e1 get] != "" } {

		# Will have to go and get the drive info from the machine
		set mdata [queryHost [ .etp.hidden1 get ] "TAPE get drives" 0 ]
		foreach m $mdata {
		    if { [ .etp.f4.e1 get ] == [lindex [split $m ":"] 1] } {
			set line "[lindex [split $m ":"] 0]:"
		    }
		}
		append line "[ .etp.f3.e1 get ]:[ .etp.f5.e2 get ]::[ .etp.f2.e1 get ]"
		set cmd "TAPE add tape $line"
		set retval [ actionItem . "Insert the media in the drive and click OK to continue." "OK Cancel" red ]
		if { $retval == 0 } {
		    set returnval [queryHost [.etp.hidden1 get] $cmd 1 ]
		    if { $errno != 0 } {
			error 502
			puts stdout "ERROR:  Can't add backup media."
		    } else {
			#repack 127
			canMEDIA
			destroy .etp
		    }
		}
		
	    } else {
		puts stdout "ERROR:  Must provide a backup device"
		error 505
	    }

	}
	button .etp.f8.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	    destroy .etp
	}
	button .etp.f8.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	    help tapemedia
	}
	pack .etp.f8.b1 .etp.f8.b2 .etp.f8.b3 -side left -fill x 


	pack .etp.f1 .etp.f2 .etp.f3 .etp.f4 .etp.f5 .etp.f8 -side top -fill x -expand 1
    } else {
	# Window exists
	.etp.hidden1 delete 0 end
	.etp.hidden2 delete 0 end
	.etp.hidden1 insert end $host
	.etp.hidden2 insert end $tapeid
    }
    
    # --------  Populate the widgets  --------------
    .etp.f1.e1 configure -state normal
    .etp.f1.e1 delete 0 end
    .etp.f2.e1 delete 0 end
    .etp.f3.e1 delete 0 end
    .etp.f4.e1 delete 0 end
    .etp.f5.e2 configure -state normal
    .etp.f5.e2 delete 0 end
    if { $tapeid != "9999" } {
	# Get the tape info
	set rdata [queryHost $host "TAPE get tapes" 0 ]
	foreach l $rdata {
	    set val [lindex [split $l ":"] 0]
	    if { $val == $drivenum } {
		# This is the backup device to work with
		.etp.f1.e1 insert end [lindex [split $l ":"] 0]
		.etp.f2.e1 insert end [lindex [split $l ":"] 1]
		.etp.f3.e1 insert end [lindex [split $l ":"] 2]
		.etp.f4.e1 insert end [lindex [split $l ":"] 3]
		.etp.f5.e1 insert end [lindex [split $l ":"] 4]
	    }
	}
	.etp.f1.e1 configure -state disabled
	.etp.f5.e2 configure -state disabled
    }
}


#
# Build the Backup/Restore Screen
#
proc canARCHIVE {} {
    global currenthost
    global currentdomain
    global level
    global browse
    global upgrade_buff
    global tapeserv
    .tp.title configure -text "Host($currenthost) --> Config --> Archives "
    set level 60
    # Enable the accelerators
    bind .tp.host.canvas <Button-3> "rclick .tp.host.canvas"
    # Put in back button
    set fig(61) [ image create photo -file "../lib/backup.gif" ]
    set fig(62) [ image create photo -file "../lib/restore.gif" ]
    set fig(0) [ image create photo -file "../lib/back.gif" ]
    set startX 40
    set startY 40
    set icon_size 48
    set incX 70
    set incY 80
    set cntx 0
    set node [ .tp.host.canvas create image $startX $startY -image $fig(0) -tags "<-Back" ]
    .tp.host.canvas bind "<-Back" <Double-1> "repack -20"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "<- Back" -tags "<-Back" \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1
    set node [ .tp.host.canvas create image $startX $startY -image $fig(61) -tags "Backup" ]
    .tp.host.canvas bind "Backup" <Double-1> "fillCan 120"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "Backup" -tags "Backup" \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1
    set node [ .tp.host.canvas create image $startX $startY -image $fig(62) -tags "Restore" ]
    .tp.host.canvas bind "Restore" <Double-1> "fillCan 130"
    set nameLabel [ .tp.host.canvas create text $startX [expr $startY + 0.8 * $icon_size] \
	    -justify center -text "Restore" -tags "Restore" \
	    -font {-*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*} ]
    incr startX $incX 
    incr cntx 1

    .tp.host.canvas create text 270 100  -justify center -text "Beta version.  See release notes." -fill red -font {-*-Helvetica-Medium-R-Normal--*-180-*-*-*-*-*-*}
}


# getBDir - Read a tape catalog from the tape server, and
#           populate the listbox $lb.  If $dir is NOT "/"
#           then include ".." in the listbox.
proc getBDir { host drive tape vol bhost dir lb lb2 } {
    global errno
    $lb delete 0 end
    $lb2 delete 0 end

    set cmd "TAPE get cat $drive $tape $vol $bhost $dir"
    set retval [queryHost $host $cmd 1 ]
    if { $errno != 0 } {
	error 2006
	puts stdout "ERROR:  Unable to retrieve backup catalog from $host"
    }
    
    if { $dir != "/" } {
	$lb insert end ".."
	set e [lindex $retval 0 ]
	$lb2 insert end [lindex [split $e {:} ] 0 ]
    }
    foreach e $retval {
	set l [lindex [split $e {:} ] 0 ]
	$lb insert end [string range $e [expr 1 + [string length $l] ] end ]
	$lb2 insert end $l
    }
}


# resSelect  - Restore GUI
#
proc resSelect { args } {
    global currenthost tapeserv errno sd textd
    set data $args
    
    set sd "/"
    if { [winfo exists .rest ] != 0 } {
	destroy .rest
    }
    toplevel .rest -class Dialog
    wm title .rest "Restore: $tapeserv"
    wm iconname .rest "Restore"
    entry .rest.h_drv 
    entry .rest.h_tape
    entry .rest.h_vol
    set tmp [split $data {,} ]
    .rest.h_drv insert end [lindex $tmp 5 ]
    .rest.h_tape insert end [lindex $tmp 6 ]
    .rest.h_vol insert end [lindex $tmp 7 ]

    # Description frame
    frame .rest.f1 -bd 2 -relief "flat"
    # Target frame
    frame .rest.f3 -bd 2 -relief "flat"
    # Browse frame
    frame .rest.f5 -bd 2 -relief "flat"
    # Bottom Buttons
    frame .rest.f7 -bd 2 -relief "raised"

    frame .rest.f20 -bd 0 -relief "flat"
    pack .rest.f20 -anchor w 
    button .rest.hb1
    frame .rest.f20.f1 -bd 2 -relief "flat"
    grid .rest.f20.f1
    button .rest.f20.f1.b1 -relief groove -text "Description" -command {
	set c1 [ .rest.hb1 cget -activebackground ]
	set c2 [ .rest.hb1 cget -background ]
	.rest.f20.f1.b1 configure -relief flat -bg $c2 -activebackground $c1
	.rest.f20.f1.b3 configure -relief groove -bg $c1 -activebackground $c2
	.rest.f20.f1.b4 configure -relief groove -bg $c1 -activebackground $c2
	pack forget .rest.f1 .rest.f3 .rest.f7 .rest.f5
	pack .rest.f1 .rest.f7
    }
    button .rest.f20.f1.b3 -relief groove -text "Target" -command {
	set c1 [ .rest.hb1 cget -activebackground ]
	set c2 [ .rest.hb1 cget -background ]
	.rest.f20.f1.b1 configure -relief groove -bg $c1 -activebackground $c2
	.rest.f20.f1.b3 configure -relief flat -bg $c2 -activebackground $c1
	.rest.f20.f1.b4 configure -relief groove -bg $c1 -activebackground $c2
	pack forget .rest.f1 .rest.f3 .rest.f7 .rest.f5
	pack .rest.f3 .rest.f7
    }
    button .rest.f20.f1.b4 -relief groove -text "Files" -command {
	set c1 [ .rest.hb1 cget -activebackground ]
	set c2 [ .rest.hb1 cget -background ]
	.rest.f20.f1.b1 configure -relief groove -bg $c1 -activebackground $c2
	.rest.f20.f1.b3 configure -relief groove -bg $c1 -activebackground $c2
	.rest.f20.f1.b4 configure -relief flat -bg $c2 -activebackground $c1
	pack forget .rest.f1 .rest.f3 .rest.f7 .rest.f5
	pack .rest.f5 .rest.f7
    }
    grid .rest.f20.f1.b1 .rest.f20.f1.b3 .rest.f20.f1.b4 -sticky w
    .rest.f20.f1.b1 invoke

    # ------------  Description Frame  -------------------------------
    label .rest.f1.l1 -text "Job #"
    entry .rest.f1.e1
    label .rest.f1.l2 -text "Description:"
    entry .rest.f1.e2 -width 32
    label .rest.f1.l3 -text "Type:"
    entry .rest.f1.e3 -width 32
    label .rest.f1.l4 -text "Created:"
    entry .rest.f1.e4 -width 32
    label .rest.f1.l5 -text "Size:"
    entry .rest.f1.e5 -width 32
    label .rest.f1.l6 -text "Status:"
    entry .rest.f1.e6 -width 32
    label .rest.f1.l7 -text "Drive #:"
    entry .rest.f1.e7 -width 32
    label .rest.f1.l8 -text "Tape #:"
    entry .rest.f1.e8 -width 32
    label .rest.f1.l9 -text "Volume #:"
    entry .rest.f1.e9 -width 32
    grid .rest.f1.l1 .rest.f1.e1 -sticky w
    grid .rest.f1.l2 .rest.f1.e2 -sticky w
    grid .rest.f1.l3 .rest.f1.e3 -sticky w
    grid .rest.f1.l4 .rest.f1.e4 -sticky w
    grid .rest.f1.l5 .rest.f1.e5 -sticky w
    grid .rest.f1.l6 .rest.f1.e6 -sticky w
    grid .rest.f1.l7 .rest.f1.e7 -sticky w
    grid .rest.f1.l8 .rest.f1.e8 -sticky w
    grid .rest.f1.l9 .rest.f1.e9 -sticky w


    # ------------  Target Frame  --------------------
    frame .rest.f3.f1 -bd 2 -relief "flat"
    label .rest.f3.f1.l1 -text "Host:" -width 12
    entry .rest.f3.f1.e1 -width 32
    .rest.f3.f1.e1 delete 0 end
    .rest.f3.f1.e1 insert end $tapeserv
    label .rest.f3.f1.l2 -text "Location:" -width 12
    entry .rest.f3.f1.e2 -width 32
    .rest.f3.f1.e2 delete 0 end
    .rest.f3.f1.e2 insert end "/"
    grid .rest.f3.f1.l1 .rest.f3.f1.e1 -sticky w
    grid .rest.f3.f1.l2 .rest.f3.f1.e2 -sticky w
    pack .rest.f3.f1

    frame .rest.f3.f2 -bd 2 -relief "groove"
    label .rest.f3.f2.l1 -text "When the file already exists:"
    radiobutton .rest.f3.f2.b1 -text "Replace it." -variable coll -value 4
    radiobutton .rest.f3.f2.b2 -text "Replace if time differs." -variable coll -value 2
    radiobutton .rest.f3.f2.b3 -text "Skip it." -variable coll -value 1
    radiobutton .rest.f3.f2.b4 -text "Rename it (Add .###) " -variable coll -value 0
    pack .rest.f3.f2.l1 .rest.f3.f2.b1 .rest.f3.f2.b2 .rest.f3.f2.b3 .rest.f3.f2.b4 -anchor w
    pack .rest.f3.f2 -fill both

    # 0 - Rename all files where the name conflicts
    # 1 - Skip all files where the name conflicts
    # 2 - Replace all files were the time differs
    # 4 - Replace all files even if there is no difference
    # 8 - Replace all newer files.
	     


    # -------------------   Browse Host and File Selection  ---------------------------
    label .rest.f5.l1 -text "Requested files to restore"
    frame .rest.f5.f1 -bd 2 -relief "flat"
    label .rest.f5.f1.l1 -text "Backup Host:" -width 12
    entry .rest.f5.f1.e1 -width 32 -font $textd
    .rest.f5.f1.e1 delete 0 end
    .rest.f5.f1.e1 insert end $tapeserv
    .rest.f5.f1.e1 configure -state disabled
    label .rest.f5.f1.l2 -text "Current Dir:" -width 12
    entry .rest.f5.f1.e2 -width 32 -font $textd
    .rest.f5.f1.e2 delete 0 end
    .rest.f5.f1.e2 insert end $sd
    grid .rest.f5.f1.l1 .rest.f5.f1.e1
    grid .rest.f5.f1.l2 .rest.f5.f1.e2

    pack .rest.f5.f1 -side top
 
    frame .rest.f5.f2 -bd 2 -relief "flat"
    scrollbar .rest.f5.f2.vscroll -orient vertical -command ".rest.f5.f2.text yview"
    scrollbar .rest.f5.f2.hscroll -orient horizontal -command ".rest.f5.f2.text xview"
    listbox .rest.f5.f2.text -selectforeground white -selectbackground #000080 \
	    -yscrollcommand {.rest.f5.f2.vscroll set } \
	    -xscrollcommand {.rest.f5.f2.hscroll set } -height 8 -width 42 -font $textd
    listbox .rest.f5.f2.text2 -yscrollcommand {.rest.f5.f2.vscroll set } \
	    -xscrollcommand {.rest.f5.f2.hscroll set } -height 8 -width 6 -font $textd
    .rest.f5.f2.text config -cursor hand2
    pack .rest.f5.f2.vscroll -side right -fill y
    pack .rest.f5.f2.hscroll -side bottom -fill x
    # pack .rest.f5.f2.text2 .rest.f5.f2.text -side left -expand 1 -fill both
    pack .rest.f5.f2.text -side left -expand 1 -fill both
    frame .rest.f5.f4 -bd 2 -relief "flat"
    button .rest.f5.f4.b1 -text "\\/ Add \\/" -command {
	if { [.rest.f5.f2.text curselection ] != "" } {
	    set i [.rest.f5.f2.text curselection ]
	    .rest.f5.f6.text insert end [string trimright "[.rest.f5.f1.e2 get ][.rest.f5.f2.text get $i ]" {.} ]
	    .rest.f5.f6.text2 insert end [.rest.f5.f2.text2 get $i ]
	}
    }
    button .rest.f5.f4.b2 -text "/\\ Remove /\\" -command {
	if { [.rest.f5.f6.text curselection ] != "" } {
	    set i [.rest.f5.f6.text curselection ]
	    .rest.f5.f6.text delete $i
	    .rest.f5.f6.text2 delete $i
	}
    }
    pack .rest.f5.f4.b1 .rest.f5.f4.b2 -side left -fill x
    
    frame .rest.f5.f6 -bd 2 -relief "flat"
    scrollbar .rest.f5.f6.vscroll -orient vertical -command ".rest.f5.f6.text yview" 
    scrollbar .rest.f5.f6.hscroll -orient horizontal -command ".rest.f5.f6.text xview"
    listbox .rest.f5.f6.text -selectforeground white -selectbackground #000080 \
	    -yscrollcommand {.rest.f5.f6.vscroll set } \
	    -xscrollcommand {.rest.f5.f6.hscroll set } -height 6 -width 42 -font $textd
    listbox .rest.f5.f6.text2 -yscrollcommand {.rest.f5.f6.vscroll set } \
	    -xscrollcommand {.rest.f5.f6.hscroll set } -height 6 -width 6 -font $textd
    .rest.f5.f6.text config -cursor hand2
    pack .rest.f5.f6.vscroll -side right -fill y
    pack .rest.f5.f6.hscroll -side bottom -fill x
    # pack .rest.f5.f6.text2 .rest.f5.f6.text -side left -expand 1 -fill both
    pack .rest.f5.f6.text -side left -expand 1 -fill both
    pack .rest.f5.f2 .rest.f5.f4 .rest.f5.f6 -side top -fill both
    

    # -----------------  Bottom Buttons  ---------------------------------------------------
    button .rest.f7.b4 -text "Restore Now" -command {
	global currenthost tapeserv errno
	if { [ .rest.f1.e1 get ] != {} } {
	    # Command Format:   drivenum tapenum volname hostname restoredir action index1,index2,...
	    set cmd "TAPE restore [.rest.h_drv get ] [.rest.h_tape get ] [.rest.h_vol get ] [.rest.f3.f1.e1 get ] [.rest.f3.f1.e2 get ] $coll "
	    set new {}
	    regsub -all -- " " [ .rest.f5.f6.text2 get 0 end ] "," new
	    append cmd $new
	    puts stdout "Sending to $currenthost cmd: $cmd"

	    set returnval [queryHost $currenthost $cmd 1 ]
	}
	if { $errno != 0 } {
	    error 2503
	    puts stdout "ERROR:  Restore job did not start"
	} 
	destroy .rest
    }
    button .rest.f7.b2 -text "Cancel" -foreground blue -activeforeground #000080 -command {
	destroy .rest
    }
    button .rest.f7.b3 -text "Help" -foreground #008000 -activeforeground #004000 -command {
	help tapejob
    }
    pack .rest.f7.b4 .rest.f7.b2 .rest.f7.b3 -side left -fill x 

    # -----------------  Bindings  ----------------------
    bind .rest.f5.f2.text <Button-3> {
	# .rest.f5.f6.text insert end [string trimright "[.rest.f5.f1.e2 get ][.rest.f5.f2.text get @%x,%y ]" {.} ]
	set i [.rest.f5.f2.text index @%x,%y ]
	set st [.rest.f5.f2.text get $i ]
	.rest.f5.f6.text insert end [string trimright "[.rest.f5.f1.e2 get ]$st" {.} ]
	.rest.f5.f6.text2 insert end [.rest.f5.f2.text2 get $i ]
    }
    bind .rest.f5.f6.text <Button-3> {
	set i [.rest.f5.f6.text index @%x,%y ]
	.rest.f5.f6.text delete @%x,%y
	.rest.f5.f6.text2 delete $i
    }
    bind .rest.f5.f6.text <Double-1> {
	if { [.rest.f5.f6.text curselection ] != "" } {
	    set i [.rest.f5.f6.text curselection ]
	    .rest.f5.f6.text delete $i
	    .rest.f5.f6.text2 delete $i
	}
    }
    bind .rest.f5.f2.text <Double-1> {
	if { [.rest.f5.f2.text curselection ] != {} } {
	    set t [.rest.f5.f2.text get [.rest.f5.f2.text curselection ] ]
	    if { $t == ".." } {
		# Need to go down a directory
		global currenthost tapeserv errno sd
		set tmp [string trimright $sd {/} ]
		set ldir [string last {/} $tmp ]
		if { $ldir <= 0 } {
		    set sd "/"
		} else {
		    set tmp [string range $sd 0 $ldir ]
		    set sd $tmp
		}
		getBDir $currenthost [.rest.h_drv get ] [.rest.h_tape get ] [.rest.h_vol get ] $tapeserv $sd .rest.f5.f2.text .rest.f5.f2.text2
	    } elseif { [string range $t end end ] == {/} } {
		global currenthost tapeserv errno sd
		set sd "[.rest.f5.f1.e2 get ][.rest.f5.f2.text get [.rest.f5.f2.text curselection ] ]"
		getBDir $currenthost [.rest.h_drv get ] [.rest.h_tape get ] [.rest.h_vol get ] $tapeserv $sd .rest.f5.f2.text .rest.f5.f2.text2
	    }
	}
	.rest.f5.f1.e2 delete 0 end
	.rest.f5.f1.e2 insert end "$sd"

    }

    # -----------------  Populate Description  -----------------------
    # jobnum, groupid, level, status, retain, drive, tape, vol, sizek, timestr, description
    set tmp [split $data {,} ]
    .rest.f1.e1 insert end [lindex $tmp 0 ]
    .rest.f1.e1 configure -state disabled
    .rest.f1.e2 insert end [lindex $tmp 10 ]
    .rest.f1.e2 configure -state disabled
    if { [lindex $tmp 2 ] == 0 } {
	.rest.f1.e3 insert end "Full"
    } elseif { [lindex $tmp 2 ] == -1 } {
	.rest.f1.e3 insert end "Differential"
    } else {
	.rest.f1.e3 insert end "Incremental"
    }
    .rest.f1.e3 configure -state disabled
    .rest.f1.e4 insert end [lindex $tmp 9 ]
    .rest.f1.e4 configure -state disabled
    .rest.f1.e5 insert end "[lindex $tmp 8 ] KBytes"
    .rest.f1.e5 configure -state disabled
    if { [lindex $tmp 2 ] == 0 } {
	.rest.f1.e6 insert end "Good"
    } else {
	.rest.f1.e6 insert end "Failed"
    }
    .rest.f1.e6 configure -state disabled
    .rest.f1.e7 insert end [lindex $tmp 5 ]
    .rest.f1.e7 configure -state disabled
    .rest.f1.e8 insert end [lindex $tmp 6 ]
    .rest.f1.e8 configure -state disabled
    .rest.f1.e9 insert end [lindex $tmp 7 ]
    .rest.f1.e9 configure -state disabled

    # ------- Get initial directory ------
    # global currenthost tapeserv errno sd
    getBDir $currenthost [.rest.h_drv get ] [.rest.h_tape get ] [.rest.h_vol get ] $tapeserv $sd .rest.f5.f2.text .rest.f5.f2.text2


}


