#include <unistd.h>
#include <stdlib.h>
#include <pwd.h>
#include <limits.h>
#include <sys/stat.h>
#include "../paths.h"
#include "internal.h"
#include "userconf.h"
#include "userconf.m"

/* #Specification: userconf / etc/passwd
	/etc/passwd is the user database. Its permission flags are
	always set to 0644 when rewritten.
*/
#define ETC_PTMP	"/etc/ptmp"
static USERCONF_HELP_FILE help_users ("users");
static CONFIG_FILE f_passwd (ETC_PASSWD,help_users,CONFIGF_MANAGED
	,"root","root",0644);

PRIVATE void USERS::readusers()
{
	/* #Specification: /etc/passwd / strategy
		/etc/passwd is read "by hand" instead of using getpwent() to avoid
		getting all those NIS entries. This is done when editing local
		user account.
	*/
	nisentry = NULL;
	nis_at_end = 1;
	FILE *fin = configf->fopen ("r");
	if (fin != NULL){
		char line[1000];
		while (fgets(line,sizeof(line)-1,fin)!=NULL){
			strip_end (line);
			if (line[0] != '\0'){
				USER *usr = new USER(line);
				if (strcmp(usr->getname(),"+")==0){
					delete nisentry;
					nisentry = usr;
					if (getnb()==0) nis_at_end = 0;
				}else{
					add (usr);
				}
			}
		}
		fclose (fin);
	}
	shadows = NULL;
	rstmodified();
}

PUBLIC USERS::USERS()
{
	home.setfrom ("/home");
	baseuid = 1;
	configf = &f_passwd;
	readusers();
	if (shadow_exist()) shadows = new SHADOWS;
}

PUBLIC USERS::USERS(CONFIG_FILE &_file, const char *_home, int _baseuid)
{
	home.setfrom (_home);
	baseuid = _baseuid;
	configf = &_file;
	readusers();
}
/*
	Return the recommend standard HOME base directory for users.
*/
PUBLIC const char *USERS::getstdhome()
{
	return home.get();
}

/*
	This object will contain a copy (only pointers).
	It is used as a temporary holder for the normal object, allowing doing
	some sort.
*/
PRIVATE USERS::USERS(USERS *users)
{
	configf = NULL;
	shadows = NULL;
	nisentry = NULL;
	nis_at_end = 1;
	int n = users->getnb();
	neverdelete();
	for (int i=0; i<n; i++) add (users->getitem(i));
}


PUBLIC USERS::~USERS()
{
	delete shadows;
	delete nisentry;
}
/*
	Get one USER specification of the table or NULL
*/
PUBLIC USER *USERS::getitem(int no)
{
	return (USER*)ARRAY::getitem(no);
}
/*
	Get one USER specification of the table or NULL from his login name
*/
PUBLIC USER *USERS::getitem(const char *name, USER *exclude)
{
	USER *ret = NULL;
	int nbu = getnb();
	for (int i=0; i<nbu; i++){
 		USER *usr = getitem(i);
		if (usr != exclude && strcmp(usr->getname(),name)==0){
			ret = usr;
			break;
		}
	}
	return ret;
}

/*
	Get one USER specification of the table or NULL from his login name
*/
PUBLIC USER *USERS::getitem(const char *name)
{
	return getitem (name,NULL);
}
/*
	Get one SHADOW specification of the table or NULL from his login name
*/
PUBLIC SHADOW *USERS::getshadow(USER *usr)
{
	SHADOW *ret = NULL;
	if (shadows != NULL) ret = shadows->getitem(usr->getname());
	return ret;
}
PUBLIC void USERS::addshadow (SHADOW *shadow)
{
	shadows->add (shadow);
}
/*
	Get one USER specification of the table or NULL from his UID
*/
PUBLIC USER *USERS::getfromuid(int uid, USER *exclude)
{
	USER *ret = NULL;
	int nbu = getnb();
	for (int i=0; i<nbu; i++){
 		USER *usr = getitem(i);
		if (usr != exclude && usr->getuid() == uid){
			ret = usr;
			break;
		}
	}
	return ret;
}
/*
	Get one USER specification of the table or NULL from his UID
*/
PUBLIC USER *USERS::getfromuid(int uid)
{
	return getfromuid(uid,NULL);
}
/*
	Get one unused User ID base of the group name.
	This function try to organise user id group wize, allocating 500
	entry per group.
*/
PUBLIC int USERS::getnewuid(int )	// gid
{
	/* #Specification: userconf / automatic allocaion of uid
		We multiply gid by 500. From there we search in all
		user id and allocate the first uid in the range.

		We don't allocate into holes (unused uid between used one)
		to avoid uid reuse (and a security hole).

		This has shown problematic. Now linuxconf allocated the a new
		UID as the next one to the one with the highest uid number.
	*/
	#if 0
	int base = gid * 500;
	int maxu = base + 500;
	#else
	int base = baseuid;
	int maxu = 65530;	// Some special UID exist at the end of the rande
						// such as nobody which is often 65535
	#endif
	int ret = base;
	int nbu = getnb();
	for (int i=0; i<nbu; i++){
 		USER *usr = getitem(i);
		int uid = usr->getuid();
		if (uid >= base && uid < maxu){
			if (uid >= ret) ret = uid + 1;
		}
	}
	return ret;	
}
/*
	Write the /etc/passwd file with proper locking
*/
PUBLIC int USERS::write(PRIVILEGE *priv)
{
	int ret = -1;
	//sortbygid();
	if (configf != NULL){
		const char *path = configf->getpath();
		char path_tmp[PATH_MAX];
		if (configf == &f_passwd){
			strcpy (path_tmp,ETC_PTMP);
		}else{
			sprintf (path_tmp,"%s.TMP",path);
		}
		FILE *fout = configf->fopen (priv,path_tmp,"w");
		if (fout != NULL){
			int nbu = getnb();
			if (nisentry != NULL && !nis_at_end) nisentry->write(fout);
			for (int i=0; i<nbu; i++){
				getitem(i)->write(fout);
			}
			if (nisentry != NULL && nis_at_end) nisentry->write(fout);
			fclose(fout);
			char path_old[PATH_MAX];
			sprintf (path_old,"%s.OLD",path);
			unlink(path_old);
			link(path, path_old);
			unlink(path);
			link(path_tmp, path);
			unlink(path_tmp);
			if (shadows != NULL) shadows->write(priv);
			ret = 0;
		}
	}
	return ret;
}

/*
	Select one user from the list.
	May return NULL if no valid selection was done (escape). See code.
*/
PRIVATE USER *USERS::select_sorted(
	USER *like,	// Used to select which user to pick.
			// the function USER::islike() is called for
			// each.
	int may_add,	// Set the delete and add button
	MENU_STATUS &code,
	int &choice)	// Will contain the selection. Not so useful
			// but help for the reentrancy of the list
			// (It reedit on the last item selected).
{
	int nbu = getnb();
	/* #Specification: userconf / user account / root bin ...
		Some special account are simply left out of the configuration
		menu. These account are never edited. They make the list larger
		for no reason.

		Also account with special shells are not shown. This include
		accounts like uucp and slip. Theu are show in a separate menu.

		The same functionnality is used to edit those accounts, but
		the editition is trigerred from different menus.
	*/
	DIALOG dia;
	for (int i=0; i<nbu; i++){
		USER *usr = getitem(i);
		if (usr->is_like(like)){
			dia.new_menuitem (usr->getname(),usr->getgecos());
		}
	}
	if (may_add){
		dia.addwhat (MSG_R(I_TOADD));
	}
	code = dia.editmenu (MSG_U(T_USERACCT,"Users accounts")
		,may_add
			? MSG_U(I_CANEDIT,"You can edit, add, or delete users")
			: MSG_U(I_SELECT,"You must select one of those users")
		,help_users
		,choice,0);
	USER *ret = NULL;
	// Locate the selected user in the list, given that not all
	// user where displayed.
	int nou = 0;
	for (int j=0; j<nbu; j++){
		USER *usr = getitem(j);
		if (usr->is_like(like)){
			if (nou == choice){
				ret = usr;
				break;
			}
			nou++;
		}
	}
	return ret;
}

/*
	Select one user from the list.
	May return NULL if no valid selection was done (escape). See code.
*/
PUBLIC USER *USERS::select(
	USER *like,	// Used to select which user to pick.
			// the function USER::islike() is called for
			// each.
	int may_add,	// Set the delete and add button
	MENU_STATUS &code,
	int &choice)	// Will contain the selection. Not so useful
			// but help for the reentrancy of the list
			// (It reedit on the last item selected).
{
	USERS sorted(this);
	sorted.sortbyname();
	return sorted.select_sorted(like,may_add,code,choice);
}
/*
	Add one new user
	Return -1 if the user was not added.
*/
PUBLIC int USERS::addone (
	USER *special,
	const char *name,	// Proposed login name
	GROUPS &groups,
	const char *fullname,	// gecos field suggested
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	int editprivi)
{
	int ret = -1;
	USER *user = new USER;
	user->setlike (special);
	user->setname (name);
	user->setgecos (fullname);
	ret = user->edit(*this,groups,1,priv,editprivi);
	if (ret==0){
		add (user);
		write (priv);
	}else{
		delete user;
	}
	return ret;
}

PUBLIC void USERS::remove_del (USER *usr)
{
	SHADOW *sha = getshadow(usr);
	if (sha != NULL) shadows->remove_del (sha);
	ARRAY::remove_del (usr);
}
/*
	General edition (addition/deletion/correction) of /etc/passwd
*/
PUBLIC int USERS::edit(
	USER *special,		// Template for user creation
						// and selection.
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	int editprivi)
{
	int ret = -1;
	int choice = 0;
	GROUPS groups;
	while (1){
		MENU_STATUS code;
		USER *usr = select (special,1,code,choice);
		if (code == MENU_ESCAPE || code == MENU_QUIT){
			break;
		}else if (code == MENU_OK){
			if (usr != NULL){
				int status = usr->edit(*this,groups,0,priv,editprivi);
				if (status != -1){
					if (status == 1) remove_del (usr);
					write(priv);
					ret = 0;
				}
			}
		}else if (perm_access(priv,MSG_U(P_USERDB
				,"to maintain the user database"))){
			if (code == MENU_ADD){
				addone (special,NULL,groups,NULL,priv,editprivi);
			}
		}
	}
	return ret;
}
static int cmpbyname (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	USER *g1 = (USER*) o1;
	USER *g2 = (USER*) o2;
	return strcmp(g1->getname(),g2->getname());
}
/*
	Sort the array of group by name
*/
PUBLIC void USERS::sortbyname()
{
	sort (cmpbyname);
}
static int cmpbygid (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	USER *g1 = (USER*) o1;
	USER *g2 = (USER*) o2;
	int ret = g1->getgid() - g2->getgid();
	if (ret == 0){
		ret = strcmp(g1->getname(),g2->getname());
	}
	return ret;
}
/*
	Sort the array of group by gid, and name
*/
PUBLIC void USERS::sortbygid()
{
	sort (cmpbygid);
}
#if 0
/*
	Edition of users password.
	Return 0 if at least one change was done.
*/
PUBLIC int USERS::editpass(
	USER *special)		// see USERS::select()
{
	int ret = -1;
	int choice = 0;
	while (1){
		MENU_STATUS code;
		USER *usr = select (special,0,code,choice);
		if (code == MENU_ESCAPE || code == MENU_QUIT){
			break;
		}else{
			SHADOW *shadow = getshadow (usr);
			if (usr->editpass(1,shadow,1) != -1){
				write();
				ret = 0;
			}
		}
	}
	return ret;
}
#endif
/*
	General edition of users account and special account
	See USERS::edit()
*/
void users_edit(
	USER *special,	// Template for a special account.
					// Or NULL.
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	int editprivi)
{
	USERS users;
	users.edit(special,priv,editprivi);
}

/*
	Add one new special user.
	Return -1 if the user was not created
*/
int users_addone(
	const char *name,
	const char *group,		// Which group to use or NULL.
	const char *fullname)	// gecos field suggested
{
	
	int ret = -1;
	PRIVILEGE *priv = special_getpriv (group);
	if (perm_access(priv,MSG_R(P_USERDB))){
		USER *special;
		if (special_init (group,special) != -1){
			USERS users;
			GROUPS groups;
			ret = users.addone(special,name,groups,fullname,priv,0);
		}
		delete special;	
	}
	return ret;
}

/*
	Return != 0 if a user account exist.
*/
int user_exist (const char *name)
{
	return getpwnam(name)!=NULL;
}
	

/*
	Edit one user spec.
	If the user does not exist, ask if we want to create it.

	Return -1 if the account was not created
*/
int users_editone(
	const char *name,
	const char *group,		// Group to use (or NULL) if the user
							// is created
	const char *fullname)	// gecos field suggested when creating a new
							// account
{
	int ret = -1;
	USERS users;
	PRIVILEGE *priv = special_getpriv (group);
	if (perm_access(priv,MSG_R(P_USERDB))){
		USER *user = users.getitem(name);
		if (user == NULL){
			char buf[300];
			sprintf (buf,MSG_U(I_USERCREATE,"User account %s does not exist\n"
				 "Do you want to create it"),name);
			if (xconf_yesno(MSG_U(Q_USERCREATE,"Account creation")
				,buf,help_users)==MENU_YES){
				ret = users_addone	(name,group,fullname);
			}
		}else{
			GROUPS groups;
			ret = user->edit(users,groups,0,priv,0);
			if (ret!=-1) users.write(priv);
		}
	}
	return ret;
}


#ifdef TEST

int main (int argc, char *argv[])
{
	users_edit(NULL);
	users_edit("-");
	users_edit("/usr/lib/uucp/uucico");
}

#endif


