#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include "../paths.h"
#include <misc.h>
#include "userconf.h"
#include "userconf.m"
#include "internal.h"

static SSTRINGS tbuser;
static SSTRINGS tbppp;
static SSTRINGS tbslip;

static USERCONF_HELP_FILE help_shells ("shells");
static CONFIG_FILE f_shells (ETC_SHELLS,help_shells
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL);


static void shells_readuser ()
{
	if (tbuser.getnb() == 0){
		/* #Specification: userconf / /etc/shells & getusershell()
			Shells available to configure user accounts are defined
			by getusershell() (which reads /etc/shells optionnally).

			It is assumed that the first entry in /etc/shells is the default
			shell (When the field is empty in /etc/passwd).
		*/
		setusershell();
		char *pt;
		while ((pt=getusershell())!=NULL){
				tbuser.add (new SSTRING(pt));
		}
		endusershell();
	}
}

/*
	Return != 0 if a path is an accepted shell
*/
int shells_isok(const char *path)
{
	shells_readuser();
	int ret = 0;
	if (path[0] == '\0'){
		ret = 1;
	}else{
		ret = tbuser.lookup(path)!=-1;
	}
	return ret;
}

int shells_exist (const char *path)
{
	struct stat sstat;
	int ret = 0;
	if (path[0] == '\0'){
		ret = 1;
	}else{
		if (stat(path,&sstat)!=-1
			&& S_ISREG(sstat.st_mode)
			&& sstat.st_mode & 1){
			ret = 1;
		}
	}
	return ret;
}


/*
	Edit the list of shells available
	Return 0 if the list has to be saved
*/
static int shell_edit(SSTRINGS &tb, const char *title)
{
	int ret = -1;
	DIALOG dia;
	tb.add (new SSTRING);
	tb.add (new SSTRING);
	int i;
	for (i=0; i<tb.getnb(); i++){
		dia.newf_str ("",*tb.getitem(i));
	}
	dia.addwhat (MSG_U(I_EMPTYLINE,"an empty line"));
	int nof = 0;
	while (1){
		MENU_STATUS code = dia.edit (title
			,""
			,help_shells
			,nof
			,MENUBUT_ADD|MENUBUT_ACCEPT|MENUBUT_CANCEL);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			break;
		}else if (code == MENU_ADD){
			SSTRING *s = new SSTRING;
			tb.add (s);
			dia.newf_str ("",*s);
		}else if (code == MENU_ACCEPT){
			for (i=0; i<tb.getnb(); i++){
				SSTRING *s = tb.getitem(i);
				if (!s->is_empty() && !shells_exist(s->get())){
					nof = i;
					xconf_error (MSG_U(E_SHELLUNKNOWN,"Shell %s does not exist")
						,s->get());
					break;
				}
			}
			if (i == tb.getnb()){
				ret = 0;
				break;
			}
		}
	}
	return ret;
}

void shells_edituser()
{
	shells_readuser();
	if (shell_edit(tbuser,MSG_U(T_USERSHELL,"Standard user shells"))==0){
		FILE *fout = f_shells.fopen("w");
		if (fout != NULL){
			for (int i=0; i<tbuser.getnb(); i++){
				SSTRING *s = tbuser.getitem(i);
				if (!s->is_empty()){
					fprintf (fout,"%s\n",s->get());
				}else{
					tbuser.remove_del(s);
					i--;
				}
			}
			fclose (fout);
		}
	}
}

static const char K_SHELLS[]="shells";
static const char K_PPP[]="ppp";
static const char K_SLIP[]="slip";

static void shells_readothers(const char *key, SSTRINGS &tb)
{
	if (tb.getnb()==0) linuxconf_getall (K_SHELLS,key,tb,1);
}

static void shells_editothers(const char *key, SSTRINGS &tb, const char *title)
{
	shells_readothers(key,tb);
	if (shell_edit(tb,title)==0){
		linuxconf_removeall (K_SHELLS,key);
		for (int i=0; i<tb.getnb(); i++){
			SSTRING *s = tb.getitem(i);
			if (!s->is_empty()){
				linuxconf_add (K_SHELLS,key,s->get());
			}else{
				tb.remove_del(s);
				i--;
			}
		}
		linuxconf_save();
	}else{
		tb.remove_all();
		shells_readothers(key,tb);
	}
}

void shells_editppp ()
{
	shells_editothers(K_PPP,tbppp,MSG_U(T_PPPSHELL,"Available PPP login shells"));
}
void shells_editslip ()
{
	shells_editothers(K_SLIP,tbslip,MSG_U(T_SLIPSHELL,"Available SLIP login shells"));
}

/*
	Return != 0 if path is an accepted login shell for PPP account
*/
int shells_isppp (const char *path)
{
	shells_readothers(K_PPP,tbppp);
	return tbppp.lookup (path)!=-1;
}

/*
	Return != 0 if path is an accepted login shell for SLIP account
*/
int shells_isslip (const char *path)
{
	shells_readothers(K_SLIP,tbslip);
	return tbslip.lookup (path)!=-1;
}

/*
	Get the list of available shells for normal users
*/
const SSTRINGS *shells_getuserlist()
{
	return &tbuser;
}
/*
	Get the list of available shells for PPP accounts
*/
const SSTRINGS *shells_getppplist()
{
	return &tbppp;
}
/*
	Get the list of available shells for SLIP accounts
*/
const SSTRINGS *shells_getsliplist()
{
	return &tbslip;
}

/*
	Return the path of the default shell for normal user.
*/
const char *shells_getdefault()
{
	shells_readuser ();
	return tbuser.getnb() == 0 ? "/bin/sh" : tbuser.getitem(0)->get();
}
/*
	Return the path of the default shell for ppp users.
*/
const char *shells_getpppdefault()
{
	shells_readothers(K_PPP,tbppp);
	return tbppp.getnb() == 0 ? USR_LIB_PPP_PPPLOGIN : tbppp.getitem(0)->get();
}
/*
	Return the path of the default shell for slip users.
*/
const char *shells_getslipdefault()
{
	shells_readothers(K_SLIP,tbslip);
	return tbslip.getnb() == 0 ? SBIN_SLIPLOGIN : tbslip.getitem(0)->get();
}

