#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include <misc.h>
#include <netconf.h>
#include "userconf.h"
#include "internal.h"
#include "userconf.m"

static time_t lasttime_root = 0;
static time_t lasttime_user = 0;

/*
	Force a one minute "no question asked" root autorization for
	one minute.

	Normally linuxconf does nothing priviledged unless the user
	has provided some password (generally root). Some operation
	are indeed priviledge but are accepted, for example, changing
	his/her password.
*/
void perm_forceok()
{
	lasttime_root = lasttime_user = time(&lasttime_root);
}

#define VALIDATION_TIME	(2*60)

/*
	Get the crypt password of a user
*/
static int perm_getupass (
	USERS &users,
	const char *username,
	char password[])
{
	/* #Specification: password / strategy / by hand
		To support transparently standard and shadow password without
		recompiling, linuxconf read manually the /etc/passwd and
		/etc/shadow files.

		This is causing problem for NIS user though. This will
		have to be fixed.
	*/
	USER *usr = users.getitem(username);
	int ret = -1;
	if (usr != NULL){
		SHADOW *shadow = users.getshadow(usr);
		const char *pwd = usr->getpwd();
		if (shadow != NULL){
			pwd = shadow->getpwd();
		}
		strcpy (password,pwd);
		ret = 0;
	}
	return ret;
}

/*
	Verify if a user/password combination is valid
*/
int perm_validpass (const char *username, const char *pass)
{
	int ret = 0;
	USERS users;
	char crypt_pass[100];
	if (perm_getupass(users,username,crypt_pass)!=-1
		&& strcmp(crypt(pass, crypt_pass),crypt_pass)==0){
		ret = 1;
	}
	return ret;
}


static int html_uid=-1;
/*
	Sent by the html mode to identify the user.
	Currently, only root is accepted as a password.

	Ultimatly, www administrator could be defined.
*/
void perm_setaccess (const char *username, const char *password)
{
	html_uid = -1;
	char upass[100];
	USERS users;
	USER *user = users.getitem(username);
	if (perm_getupass(users,username,upass)!=-1){
		if (upass[0] != '\0'){
			if (strcmp(crypt(password, upass),upass)==0){
				html_uid = user->getuid();
			}
		}else if (password[0] == '\0'){
			html_uid = user->getuid();
		}
	}
}

/*
	Ask the password for root or optionnally the user.
	Return 1 if a good password was supplied
*/
static int perm_askpass (
	USERS &users,
	const char *uname,
	time_t valid_until)
{
	int ret = 0;
	/* #Specification: root access / password validation
		When the admin/user must provide the root
		password, he has 3 chances.
	*/
	struct {
		char root[100];
		char user[100];
	}crypt_pass;
	if (perm_getupass (users,"root",crypt_pass.root) == -1){
		xconf_error (MSG_U(E_NOROOT
			,"No root user in /etc/passwd\n"
			 "Better to let you in"));
			ret = 1;
	}else if (crypt_pass.root[0] != '\0'){
		if (uname != NULL
			&& perm_getupass (users,uname,crypt_pass.user) == -1){
			xconf_error (MSG_U(E_NOUSER
				,"No user %s in /etc/passwd\n"
				 "privilege denied")
				,uname);
		}
		for (int i=0; i<3; i++){
			DIALOG dia;
			SSTRING rootpass,userpass;
			dia.newf_pass (MSG_U(F_ROOTPASS,"root password"),rootpass);
			const char *intro = MSG_U(I_ENTERPASS
				 ,"Enter password for root\n"
				  "Only the superuser is allowed to perform\n"
				  "configuration task.");
			if (uname != NULL){
				dia.newf_pass (MSG_U(F_YOURPASS,"Your password"),userpass);
				intro = MSG_U(I_ENTERONEPASS
					 ,"Enter password for root or your password\n"
					  "The superuser and you are allowed to perform\n"
					  "this configuration task.");
			}
			if (dia.edit (MSG_U(T_PASSREQ,"Password requiered")
				,intro
				,help_nil) != MENU_ACCEPT){
				ret = 0;
				break;
			}else if (strcmp(crypt(userpass.get(), crypt_pass.user)
				,crypt_pass.user)==0){
				lasttime_user = valid_until;
				ret = 1;
				break;
			}else if (strcmp(crypt(rootpass.get(), crypt_pass.root)
				,crypt_pass.root)!=0){
				ret = 0;
				xconf_error (MSG_U(E_IVLDPASS,"Invalid password"));
			}else{
				lasttime_root = valid_until;
				ret = 1;
				break;
			}
		}
	}
	return ret;
}

/*
	Verify if there is password for root. If so ask the user
	to enter it and validate it. So askrunlevel is "safe". No
	one will be allowed to reconfigured the network if he don't
	know the root password.

	Return != 0 if user if allowed to get in.

	If priv != NULL, the user password is also checked for.
*/
static int perm_checkpass (PRIVILEGE *priv)
{
	time_t curtime = time(&curtime);
	/* #Specification: root access / timeout
		When the user select a configuration task, the password for root
		must be supplied. This "validation" is good for 2 minute.
		It means that the user may do several configuration in one
		minutes without being asked for the root password every time.
		
		If the user wait a minute or more, the password will be
		asked again. Look safe and user friendly to me.
	*/
	int ret = 1;
	if (dialog_mode == DIALOG_HTML){
		if (html_uid == -1){
			html_needpasswd();
			ret = 0;
		}else if (priv == NULL){
			if (html_uid != 0){
				html_needpasswd();
				ret = 0;
			}
		}else if (html_uid != 0){
			USERS users;
			USER *user = users.getfromuid (html_uid);
			PRIVILEGE_DATA *data = priv->getdata (user->getname());
			if (!data->has_priv()){
				html_needpasswd();
				ret = 0;
			}
			delete data;
		}
	}else if (curtime - lasttime_root > VALIDATION_TIME){
		time_t valid_until = curtime + VALIDATION_TIME;
		USERS users;
		if (priv == NULL){
			// We need the root password
			ret = perm_askpass(users,NULL,valid_until);
		}else{
			USER *user = users.getfromuid (getuid());
			if (user == NULL){
				xconf_error (MSG_U(E_NOUSERFROMID
					,"Can't identify yourself from your user ID (%d)")
					,getuid());
				ret = perm_askpass(users,NULL,valid_until);
			}else{
				PRIVILEGE_DATA *data = priv->getdata (user->getname());
				if (data->has_priv()){
					if (data->mustident()
						&& curtime - lasttime_user > VALIDATION_TIME){
						ret = perm_askpass (users,user->getname()
							,valid_until);
					}
				}else{
					ret = perm_askpass (users,NULL,valid_until);
				}
			}
		}
	}
	return ret;
}
/*
	Verify if there is password for root. If so ask the user
	to enter it and validate it. So askrunlevel is "safe". No
	one will be allowed to reconfigured the network if he don't
	know the root password.

	Return != 0 if user if allowed to get in.
*/
int perm_checkpass ()
{
	return perm_checkpass (NULL);
}

static int perm_html_mode = 0;
/*
	Check if the user is really root. If it is not, but the effective ID
	is root, then ask for the root password.
	Return != 0 if the real UID is root or the user knows the root password.

	It prints an informative message about the action which will occur.
*/
static int perm_v_access(PRIVILEGE *priv, const char *buf)
{
	/* #Specification: configurator / setuid root
		The configurator (anything-conf) can be set setuid root.
		It will allows normal users to get in and then will ask
		for the root password at the proper time. This strategy
		make the system friendlier. It allows normal user to
		inspect the configuration (if allowed) and when finding
		something odd, use the root password (if known) to fix
		things, The idea here is that we generally think first
		about getting somewhere and later about the permissions
		needed to get there.

		The nice thing about this scheme is that this program
		will deny root access automaticly after some time.

		If you don't like this, then don't set it setuid root. It
		will operate correctly and won't bug you with password.
	*/
	int ret = perm_html_mode ? 0 : getuid()==0;
	if (geteuid()==0){
		if (!ret){
			ret = perm_checkpass(priv);
		}
	}else{
		xconf_error (MSG_U(E_MUSTBEROOT,"You must be root to\n%s"),buf);
	}
	return ret;
}

/*
	Check if the user is really root. If it is not, but the effective ID
	is root, then ask for the root password.
	Return != 0 if the real UID is root or the user knows the root password.

	It prints an informative message about the action which will occur.
*/
int perm_rootaccess(const char *ctl, ...)
{
	va_list list;
	va_start (list,ctl);
	char buf[1000];
	vsprintf (buf,ctl,list);
	va_end (list);
	return perm_v_access (NULL,buf);
}
/*
	Check if the user is really root or has the privilege to do a task
	If it is not, but the effective ID is root, then ask
	for the root password or his own password.

	Return != 0 if the real UID is root or the user knows the root password
	or has enough privilege.

	It prints an informative message about the action which will occur.
*/
int perm_access(PRIVILEGE *priv, const char *ctl, ...)
{
	va_list list;
	va_start (list,ctl);
	char buf[1000];
	vsprintf (buf,ctl,list);
	va_end (list);
	return perm_v_access (priv,buf);
}

/*
	Check if the user has a privilege.
	If the user has supplied the root password once, it has the privilege

	Return != 0 if so.
*/
int perm_checkpriv (PRIVILEGE *priv)
{
	int ret = 0;
	if (lasttime_root > 0 || getuid()==0){
		ret = 1;
	}else if (priv != NULL){
		USERS users;
		USER *user = users.getfromuid (getuid());
		if (user != NULL){
			PRIVILEGE_DATA *data = priv->getdata (user->getname());
			ret = data->has_priv();
			delete data;
		}
	}
	return ret;
}

/*
	Set the html mode for granting priviledge.
	In html mode we are executing as root all the time. Normally
	no question is asked to root. Now we must ask for password before
	going through.
*/
void perm_sethtml (int _mode)
{
	perm_html_mode = _mode;
}

