#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <limits.h>
#include "internal.h"

#if defined (unix)
	#include <sys/mman.h>
	#define O_BINARY	0
	static const char *translat_readall(int fd, int size)
	{
		return (const char*)mmap(0,size,PROT_READ,MAP_SHARED,fd,0);
	}

#else
	#if defined (UNIX)
		// Just to test the DOS/Windows mode in Unix.
		#include <unistd.h>
		#define O_BINARY	0
	#else
		#include <io.h>
	#endif
	static const char *translat_readall(int fd, int size)
	{
		char *ptm = (char*)malloc(size);
		if (ptm != NULL){
			if (read(fd,ptm,size)!=size){
				fprintf (stderr,"Can't read messages database\n");
			}
		}else{
			fprintf (stderr,"Can't allocate memory for messages\n");
		}
		close (fd);
		return ptm;
	}

#endif
#include "translat.h"

static TRANSLATE_SYSTEM_REQ *first = NULL;

PUBLIC TRANSLATE_SYSTEM_REQ::TRANSLATE_SYSTEM_REQ(
	const char *_sysname,
	const char **&_global_var,	// Will point to an array
					// of message (char*)
	int _nb_expected,
	int _version)
	: global_var (_global_var)
{
	nb_expected = _nb_expected;
	version = _version;
	/* #Specification: translate / dictionnary request
		A dictionnary request is done simply by declaring
		a static variable of type TRANSLATE_SYSTEM_REQ.
		We generally defined this variable in the
		source module which is always used within a system.

		Sometime, there is no such a module but indeed a few
		module (A library) which are candidate to be included
		in every program which use the librairy.

		It is possible to declare several static
		TRANSLATE_SYSTEM_REQ, one in every module we need. These
		request should use obviously the same name and
		the same global variable.

		Special macros are automagically defined by
		msgscan to insure consistency.
	*/
	TRANSLATE_SYSTEM_REQ *pt = first;
	while (pt != NULL){
		if (strcmp(pt->sysname,_sysname)==0)break;
		pt = pt->next;
	}
	if (pt == NULL){
		next = first;
		first = this;
		sysname = _sysname;
	}
}
static char *pterr = NULL;
/*
	Print an error message and increment the err counter
*/
static void translat_err (const char *msg, ...)
{
	va_list list;
	va_start (list,msg);
	pterr += vsprintf (pterr,msg,list);
	va_end (list);
}



static void translat_alloc (
	BDICT_SYSTEM *tbsys,
	int nbsys,
	const char *ptm)
{
	TRANSLATE_SYSTEM_REQ *req = first;
	while (req != NULL){
		long offset = sizeof(BDICT_HEADER)+nbsys * sizeof(BDICT_SYSTEM);
		BDICT_SYSTEM *ptsys = tbsys;
		int i;
		for (i=0; i<nbsys; i++, ptsys++){
			int nbmsg = ptsys->nbmsg;
			if (strcmp(req->sysname,ptsys->name)==0){
				if (req->version > ptsys->version){
					translat_err ("sub-dictionnary %s: Invalid version %d < %d\n"
						,req->sysname,ptsys->version,req->version);
				}else if (req->nb_expected > ptsys->nbmsg){
					translat_err ("sub-dictionnary %s: Not enough messages %d < %d\n"
						,req->sysname,ptsys->nbmsg,req->nb_expected);
				}else{
					req->global_var = (const char**)malloc(nbmsg*sizeof(char*));
					const char **ptvar = req->global_var;
					long *ptoff = (long*)(ptm + offset);
					for (int m=0; m<nbmsg; m++, ptvar++, ptoff++){
						*ptvar = ptm + *ptoff;
					}
				}
				break;
			}else{
				offset += nbmsg * sizeof(long);
			}
		}
		if (i==nbsys){
			translat_err ("Unknown dictionnary %s\n",req->sysname);
		}
		req = req->next;
	}
}


/*
	Load the message dictionnary.
	Terminate the application if any error.
*/
int translat_load (
	const char *basepath,	// Directory holding dictionnaries
	const char *basename,	// base name of the dictionnary
				// the language selection will supply
				// the extension.
	char *errmsg)		// Contiendra le message d'erreur
{
	errmsg[0] = '\0';
	pterr = errmsg;
	char path[128];
	sprintf (path,"%s/%s",basepath,basename);
	int fd  = open (path,O_BINARY|O_RDONLY);
	if (fd == -1){
		translat_err ("Can't load dictionnary file %s (%s)\n"
			,path,strerror (errno));
	}else{
		struct stat st;
		stat (path,&st);
		const char *ptm = translat_readall (fd,st.st_size);
		if (ptm != NULL){
			BDICT_HEADER *hd = (BDICT_HEADER*)ptm;
			if (hd->magic != BDICT_MAGIC){
				translat_err ("Invalid dictionnary magic word\n");
			}else if (hd->version != BDICT_VERSION){
				translat_err ("Invalid dictionnary version\n");
			}else{
				BDICT_SYSTEM *tbsys
					= (BDICT_SYSTEM*)(ptm+sizeof(BDICT_HEADER));
				translat_alloc (tbsys,hd->nbsys,ptm);
			}
		}
	}
	first = NULL;	// This is done to allow module (dlls) loaded at
					// runtime to have their own dictionnary
	return pterr > errmsg ? -1 : 0;
}

/*
	Load the message dictionnary.
	Terminate the application if any error.
*/
void translat_load (
	const char *basepath,	// Directory holding dictionnaries
	const char *basename)	// base name of the dictionnary
				// the language selection will supply
				// the extension.
{
	char errmsg[2000];
	if (translat_load (basepath,basename,errmsg)==-1){
		fprintf (stderr,"%s",errmsg);
		exit (-1);
	}
}

/*
	Load the message dictionnary.
	Terminate the application if any error.
*/
void translat_load (
	const char *basepath,	// Default directory holding dictionnaries
	const char *envdirvar,	// Environment variable to override basepath
	const char *basename,	// base name of the dictionnary
							// the language selection will supply
							// the extension.
	const char *envlangvar)	// Environment variable to override the
							// language
{
	/* #Specification: messages dictionnary / location / strategy
		The application specify the location and the default language
		for the message dictionnary. This is supplied as a basepath and a
		basename. The effective path of the dictionnary is

		#
		basepath/basename.lang
		#

		An environnement variable let you override the base path and another
		environnement variable let you override the language. For the
		linuxconf project, the environnement variables are for example.
	
		#
		LINUXCONF_DICT: base path
		LINUXCONF_LANG: language extension
		#

		The basepath may be overriden only if "geteuid() == getuid()"
		for security reason.
	*/
	if (geteuid() == getuid()){
		char *en = getenv (envdirvar);
		if (en != NULL) basepath = en;
	}
	const char *lang = "eng";
	const char *lang_env = getenv(envlangvar);
	if (lang_env != NULL) lang = lang_env;

	char bdict[30];
	sprintf (bdict,"%s.%s",basename,lang);
	translat_load (basepath,bdict);
}

