#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "misc.h"
#include "confdb.h"
#include "../paths.h"

/* #Specification: /etc/conf.linuxconf / permissions
	For security reasons, /etc/conf.linuxconf is unreadable for all
	users except root. There is a lot of information in it allowing
	potential intruders to spot information that they really don't need to
	know.

	For exemple, /etc/conf.linuxconf tells which users have which
	administrative privilege. This is not much for an intruder, but it
	certainly tells which user passwords have move value than others.
*/
static HELP_FILE helpf ("misc","linuxconf");

static CONFIG_FILE f_linuxconf (ETC_CONF_LINUXCONF
	,helpf
	,CONFIGF_OPTIONNAL|CONFIGF_MANAGED
	,"root","root",0600);



static CONFDB *tb = NULL;

static void linuxconf_init()
{
	if (tb == NULL) tb = new CONFDB (f_linuxconf);
}

/*
	Locate one configuration parameter.
	Return NULL if not found.
*/
const char *linuxconf_getval (const char *prefix, const char *key)
{
	linuxconf_init();
	return tb->getval (prefix,key);
}
/*
	Locate one numeric configuration parameter.
	Return defval if not found.
*/
int linuxconf_getvalnum (const char *prefix, const char *key, int defval)
{
	linuxconf_init();
	return tb->getvalnum(prefix,key,defval);
}

/*
	Locate all configuration parameter with the same key.
	Return the number found.
*/
int linuxconf_getall (
	const char *prefix,
	const char *key,
	SSTRINGS &lst,
	int copy)	// Take a copy of the values
{
	linuxconf_init();
	return tb->getall (prefix,key,lst,copy);
}

/*
	Remove all entry with a given key.
*/
void linuxconf_removeall (const char *prefix, const char *key)
{
	if (tb != NULL) tb->removeall (prefix,key);
}

/*
	Forget the in memory copy of /etc/conf.linuxconf
*/
void linuxconf_forget()
{
	delete tb;
	tb = NULL;
}
/*
	Save the configuration parameters
	Return -1 if any error.
*/
int linuxconf_save(PRIVILEGE *priv)
{
	int ret = 0;
	if (tb != NULL){
		ret = tb->save(priv);
		if (ret == -1 && errno == EPERM){
			// Avoid keeping in memory records potentially entered
			// by a user who don't know the root password.
			linuxconf_forget();
		}
	}
	return ret;
}
/*
	Save the configuration parameters
	Return -1 if any error.
*/
int linuxconf_save()
{
	return linuxconf_save (NULL);
}
/*
	Add one record to the configuration file
*/
void linuxconf_add (const char *prefix, const char *key, const char *val)
{
	linuxconf_init();
	tb->add (prefix,key,val);
}
/*
	Add one record in the configuration file
*/
void linuxconf_add (const char *prefix, const char *key, const SSTRING &val)
{
	linuxconf_init();
	tb->add (prefix,key,val);
}
/*
	Add one record in the configuration file
*/
void linuxconf_add (const char *prefix, const char *key, int val)
{
	linuxconf_init();
	tb->add (prefix,key,val);
}
/*
	Replace one record in the configuration file
*/
void linuxconf_replace (const char *prefix, const char *key, const char *val)
{
	linuxconf_init();
	tb->replace (prefix,key,val);
}
/*
	Replace one record in the configuration file
*/
void linuxconf_replace (const char *prefix, const char *key, int val)
{
	linuxconf_init();
	tb->replace (prefix,key,val);
}
/*
	Replace one record in the configuration file
*/
void linuxconf_replace (const char *prefix, const char *key, const SSTRING &val)
{
	linuxconf_init();
	tb->replace (prefix,key,val);
}

/*
	Replace all records to the configuration file for a key
*/
void linuxconf_replace (
	const char *prefix,
	const char *key,
	const SSTRINGS &vals)
{
	linuxconf_init();
	tb->replace (prefix,key,vals);
}


