#pragma implementation
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include "misc.h"
#include <userconf.h>
#include "../paths.h"
#include "misc.m"


static HELP_FILE help_configf ("misc","configf");
static CONFIG_FILE *first = NULL;
static int nbconfig=0;

PRIVATE void CONFIG_FILE::init(
	const char *_path,
	int _status,
	const char *_owner,
	const char *_group,
	int _perm)
{
	owner = _owner;
	group = _group;
	perm = _perm;
	stdpath = _path;
	realpath = NULL;	// fixpath() will look it up
				// later when more is initialised
				// This also avoid a massive query
				// to the linuxconf_getval() function
				// at startup time.
	status = _status;
	next = first;
	first = this;
	nbconfig++;
}
/*
	Record the spec of a config file that is maintain (or used) by linuxconf
*/
PUBLIC CONFIG_FILE::CONFIG_FILE(
	const char *_path,
	HELP_FILE &_helpfile,
	int _status) : helpf(_helpfile)
{
	/* #Specification: configuration files / default permissions
		Unless explicitly stated, the owner and permissions
		of every configuration file produced and maintained
		by linuxconf are

		#
		rw-r--r--	root	root
		#
	*/
	init (_path,_status,"root","root",0644);
}

/*
	Record the spec of a config file that is maintain (or used) by linuxconf
*/
PUBLIC CONFIG_FILE::CONFIG_FILE(
	const char *_path,
	HELP_FILE &_helpfile,
	int _status,
	const char *_owner,
	const char *_group,
	int _perm) : helpf(_helpfile)
{
	init (_path,_status,_owner,_group,_perm);
}
PUBLIC CONFIG_FILE::~CONFIG_FILE()
{
	CONFIG_FILE **ptpt = &first;
	while (*ptpt != NULL){
		if (*ptpt == this){
			*ptpt = next;
			break;
		}
		ptpt = &(*ptpt)->next;
	}
}

/*
	Set the permissions and owner of a file to the same value as
	the configuration file, if it has such a requirement.

	Return -1 if any error.
*/
PUBLIC int CONFIG_FILE::setperm (const char *fpath) const
{
	int ret = 0;
	if (owner != NULL){
		struct passwd *pw = getpwnam(owner);
		struct group *gr = getgrnam(group);
		ret = -1;
		if (pw == NULL){
			xconf_error (MSG_U(E_SETOWNER
				,"Can't set ownership of file\n"
				 "%s\n\n"
				 "User %s does not exist\n")
				,fpath,owner);
		}else if (gr == NULL){
			xconf_error (MSG_U(E_SETGROUP
				,"Can't set group ownership of file\n"
				 "%s\n\n"
				 "Group %s does not exist\n")
				,fpath,group);
		}else if (chown (fpath,pw->pw_uid,gr->gr_gid) != -1
			&& chmod (fpath,perm) != -1){
			ret = 0;
		}
	}
	return ret;
}

static char CONFIG[]="config";

PRIVATE void CONFIG_FILE::fixpath() const
{
	if (realpath == NULL){
		/* #Specification: /etc/conf.linuxconf / can't be moved
			Given that /etc/conf.linuxconf is used to store
			almost everything which do not have a standard
			home (configuration file), including the
			the corrected path of the configuration themselves
			it is not possible to change the location of
			/etc/conf.linuxconf.
		*/
		CONFIG_FILE *pt = (CONFIG_FILE*)this;
		if (strcmp(stdpath,ETC_CONF_LINUXCONF)==0){
			pt->realpath = stdpath;
		}else{
			const char *path = linuxconf_getval (CONFIG,stdpath);
			if (path == NULL) path = stdpath;
			pt->realpath = strdup(path);
		}
	}
}
/*
	Open the configuration file without permission checking.
*/
PUBLIC FILE *CONFIG_FILE::fopen_ok(const char *mode) const
{
	FILE *ret = NULL;
	fixpath();
	if (strcmp(mode,"r")==0 && (status & CONFIGF_OPTIONNAL) != 0){
		ret = ::fopen (realpath,mode);
	}else{
		ret = xconf_fopen(realpath,mode);
		setperm(realpath);
	}
	return ret;
}
/*
	Open the configuration file with permission checking.
	It may even ask the user for the root password or
	a user password if priv != NULL.
*/
PUBLIC FILE *CONFIG_FILE::fopen(PRIVILEGE *priv, const char *mode) const
{
	FILE *ret = NULL;
	fixpath();
	if (strcmp(mode,"r")==0 && (status & CONFIGF_OPTIONNAL) != 0){
		ret = ::fopen (realpath,mode);
	}else{
		ret = xconf_fopencfg(priv,realpath,mode);
		setperm(realpath);
	}
	return ret;
}
/*
	Open the configuration file with permission checking.
	It may even ask the user for the root password.
*/
PUBLIC FILE *CONFIG_FILE::fopen(const char *mode) const
{
	return fopen ((PRIVILEGE*)NULL,mode);
}
/*
	Open the temp configuration file with permission checking.
	It may even ask the user for the root password.
*/
PUBLIC FILE *CONFIG_FILE::fopen(PRIVILEGE *priv, const char *temp, const char *mode) const
{
	FILE *ret = xconf_fopencfg(priv,temp,mode);
	setperm(temp);
	return ret;
}

/*
	Open the temp configuration file with permission checking.
	It may even ask the user for the root password.
*/
PUBLIC FILE *CONFIG_FILE::fopen(const char *temp, const char *mode) const
{
	return fopen (NULL,temp,mode);
}
/*
	Return the path of the configuration file
*/
PUBLIC const char *CONFIG_FILE::getpath() const
{
	fixpath();
	return realpath;
}
/*
	Return the standard path of the configuration file
*/
PUBLIC const char *CONFIG_FILE::getstdpath() const
{
	fixpath();
	return stdpath;
}
/*
	Return the path of the help for that configuration file
*/
PUBLIC const char *CONFIG_FILE::gethelp() const
{
	return helpf.getpath();
}
/*
	Return != 0 if the configuration file is managed by linuxconf.
	Some configuration file are only read (expect to be there
	and probably never edited by the user.
*/
PUBLIC int CONFIG_FILE::is_managed() const
{
	return status & CONFIGF_MANAGED;
}
/*
	Return != 0 if the configuration file is optionnal.
*/
PUBLIC int CONFIG_FILE::is_optionnal() const
{

	return status & CONFIGF_OPTIONNAL;
}
/*
	Return != 0 if the configuration file is generated by linuxconf.
*/
PUBLIC int CONFIG_FILE::is_generated() const 
{
	return status & CONFIGF_GENERATED;
}
/*
	Return != 0 if the configuration file is erased at boot time.
*/
PUBLIC int CONFIG_FILE::is_erased() const 
{
	return status & CONFIGF_ERASED;
}
/*
	Return != 0 if the configuration file is probed by linuxconf.
	probing is simply to check its modification time.
*/
PUBLIC int CONFIG_FILE::is_probed() const
{
	return status & CONFIGF_PROBED;
}

/*
	Check if the file or directory exist.
	Return != 0 if yes.
*/
int file_exist (const char *path)
{
	struct stat st;
	return stat(path,&st)!=-1;
}
/*
	Check if the configuration file do exist.
	Return != 0 if yes.
*/
PUBLIC int CONFIG_FILE::exist() const
{
	fixpath();
	return file_exist (realpath);
}

/*
	Return the modification date of a file
*/
long file_date (const char *path)
{
	struct stat buf;
	long ret = -1;
	if (stat(path,&buf)!=-1){
		ret = buf.st_mtime;
	}
	return ret;
}

/*
	Get the modification time of a configuration file.
*/
PUBLIC long CONFIG_FILE::getdate() const
{
	fixpath();
	return file_date (realpath);
}
/*
	Get the modification time of a configuration file.
*/
PUBLIC int CONFIG_FILE::unlink() const
{
	fixpath();
	return ::unlink (realpath);
}


PUBLIC void CONFIG_FILE::editpath()
{
	SSTRING p(getpath());
	while (1){
		DIALOG dia;
		dia.newf_str (MSG_U(F_CORRPATH,"Correct path"),p);
		int nof = 0;
		MENU_STATUS code = dia.edit (
			MSG_U(T_MODCONFPATH,"Modifying a config file path")
			,MSG_U(I_MODCONFPATH
			 ,"You can redefined the path of a\n"
			 "configuration file. If you do so\n"
			 "Linuxconf will use the new path from now on.\n"
			 "\n"
			 "Be advise that other utilities are on there\n"
			 "own and may forget to notice. Unless you\n"
			 "really knows what you are doing, don't play\n"
			 "here.\n")
			,helpf
			,nof
			,MENUBUT_RESET|MENUBUT_ACCEPT|MENUBUT_CANCEL);
		if (code == MENU_ESCAPE || code == MENU_CANCEL){
			break;
		}else if (code == MENU_RESET){
			p.setfrom (stdpath);
		}else if (code == MENU_ACCEPT){
			if (p.is_empty()) p.setfrom (stdpath);
			free ((char*)realpath);
			realpath = NULL;
			if (p.cmp(stdpath)==0){
				linuxconf_removeall (CONFIG,stdpath);
			}else{
				linuxconf_replace (CONFIG,stdpath,p);
			}
			linuxconf_save();
			break;
		}
	}
}

static int cmp_config(const void *pt1, const void *pt2)
{
	CONFIG_FILE *p1 = *(CONFIG_FILE**)pt1;
	CONFIG_FILE *p2 = *(CONFIG_FILE**)pt2;
	return strcmp(p1->getstdpath(),p2->getstdpath());
}

/*
	List all config file managed by this system
*/
void configf_show()
{
	CONFIG_FILE *tb[200];
	{
		CONFIG_FILE *f = first;
		int no = 0;
		while (f != NULL){
			tb[no++] = f;
			f = f->next;
		}
		qsort (tb,nbconfig,sizeof(CONFIG_FILE*),cmp_config);
	}
	int choice = 0;
	while (1){
		DIALOG dia;
		for (int i=0; i<nbconfig; i++){
			CONFIG_FILE *f = tb[i];
			char type[6] = "     ";
			if (f->is_erased())    type[0] = 'E';
			if (f->is_generated()) type[1] = 'G';
			if (f->is_managed())   type[2] = 'M';
			if (f->is_optionnal()) type[3] = 'O';
			if (f->is_probed())    type[4] = 'P';
			const char *path = f->getpath();
			const char *stdp = f->getstdpath();
			char buf[80];
			if (strcmp(path,stdp)==0){
				strcpy (buf,stdp);
			}else{
				sprintf (buf,"%s (%s)",stdp,path);
			}
			dia.new_menuitem (type,buf);
		}
		MENU_STATUS code = dia.editmenu (
			MSG_U(T_LISTCONF,"List of configuration files")
			,MSG_U(I_LISTCONF
			 ,"This is the list of all file managed\n"
			  "by linuxconf. For each file, you can access\n"
			  "directly a help file describing its purpose.\n"
			  "The letters preceding the file name indicate\n"
			  "how this file is managed by Linuxconf. Press help\n"
			  "for more info")
			,help_configf
			,choice,MENUBUT_EDIT);
		if (code == MENU_QUIT || code == MENU_ESCAPE){
			break;
		}else{
			CONFIG_FILE *cfgf = tb[choice];
			if (code == MENU_EDIT){
				if (perm_rootaccess (
					MSG_U(P_MODCONF,"modify a configuration file path"))){
					cfgf->editpath();
				}
			}else{
				dialog_textbox (cfgf->getpath(),cfgf->gethelp());
			}
		}
	}
}

/*
	Erase some config file at boot time.
*/
void configf_booterase()
{
	CONFIG_FILE *f = first;
	while (f != NULL){
		if (f->is_erased()) f->unlink();
		f = f->next;
	}
}
