/* #Specification: mail / virtual domain / concept
	The concept of the virtual email domain is simply the same
	as for virtual web hosting.

	From the outside, one computer looks like several independant one.
	In the same way as a Web server may deliver different page depending
	on the virtual host being queried, the virtual email server will
	have the following:

	#
		-One independant user list per virtual email domain
		 These users are really "virtual". The only service they
		 can get is the POP service (need a special POP server)
		-One independant folder directory
		-Potentially an independant aliases file.
		-Potentially (one day) a special hierarchy for HOME directory
		 which may be used by a virtualised IMAP server.
	#
	To make it simple, if you define two virtual email domain, say
	virtual1.com and virtual2.com, then joe@virtual1.com and
	joe@virtual2.com are two different users with different password
	and different messages.
*/
#include <limits.h>
#include <userconf.h>
#include <fixperm.h>
#include "internal.h"
#include "mailconf.h"
#include "mailconf.m"
#include "../paths.h"

static MAILCONF_HELP_FILE help_vdomain("vdomain");

PUBLIC VDOMAIN::VDOMAIN(const char *line)
{
	const char *pt = domain.copyword (line);
	aliases.copyword (pt);
}

PUBLIC VDOMAIN::VDOMAIN()
{
}

PUBLIC int VDOMAIN::edit (VDOMAINS &vs)
{
	int ret = -1;
	DIALOG dia;
	dia.newf_str (MSG_U(F_VDOMAIN,"Virtual domain (fqdn)"),domain);
	dia.last_noempty();
	dia.newf_str (MSG_U(F_ALIASFILE,"Aliases file"),aliases);
	int nof = 0;
	while (1){
		MENU_STATUS code = dia.edit (
			MSG_U(T_ONEVDOMAIN,"One vdomain definition")
			,MSG_U(I_DEFVDOMAIN,"")
			,help_vdomain
			,nof
			,MENUBUT_CANCEL|MENUBUT_ACCEPT|MENUBUT_DEL);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			break;
		}else if (code == MENU_DEL){
			if (xconf_delok()){
				ret = 1;
				break;
			}
		}else{
			VDOMAIN *other = vs.getitem(domain.get());
			if (other != NULL && other != this){
				xconf_error (MSG_U(E_VDOMEXIST
					,"This virtual domain is already defined"));
				nof = 0;
			}else{
				ret = 0;
				break;
			}
		}
	}
	if (ret != 0) dia.restore();
	return ret;
}

static int cmp_by_domain (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	VDOMAIN *d1 = (VDOMAIN*)o1;
	VDOMAIN *d2 = (VDOMAIN*)o2;
	return d1->domain.cmp(d2->domain.get());
}

PUBLIC void VDOMAINS::sort()
{
	ARRAY::sort (cmp_by_domain);
}


PUBLIC VDOMAIN *VDOMAINS::getitem (int no)
{
	return (VDOMAIN*)ARRAY::getitem(no);
}

PUBLIC VDOMAIN *VDOMAINS::getitem (const char *domain)
{
	VDOMAIN *ret = NULL;
	int n = getnb();
	for (int i=0; i<n; i++){
		VDOMAIN *d = getitem(i);
		if (d->domain.cmp(domain)==0){
			ret = d;
			break;
		}
	}
	return ret;
}

static const char K_VDOMAIN[] = "vdomain";
static const char K_INDEX[] = "index";

PUBLIC VDOMAINS::VDOMAINS()
{
	SSTRINGS tb;
	linuxconf_getall (K_VDOMAIN,K_INDEX,tb,0);
	int n = tb.getnb();
	for (int i=0; i<n; i++){
		add (new VDOMAIN(tb.getitem(i)->get()));
	}
}

PUBLIC int VDOMAINS::write ()
{
	linuxconf_removeall (K_VDOMAIN,K_INDEX);
	int n = getnb();
	for (int i=0; i<n; i++){
		VDOMAIN *d = getitem(i);
		char data[300];
		sprintf (data,"%s %s",d->domain.get(),d->aliases.get());
		linuxconf_add (K_VDOMAIN,K_INDEX,data);
	}
	return linuxconf_save();
}

/*
	Edit the definition of virtual email domains.
	Return != 0 if the has been one modification.
*/
PUBLIC MENU_STATUS VDOMAINS::select(
	const char *title,
	const char *intro,
	int mayadd,
	int &nof)
{
	sort();
	DIALOG dia;
	for (int i=0; i<getnb(); i++){
		VDOMAIN *d = getitem(i);
		dia.new_menuitem(" ",d->domain.get());
	}
	if (mayadd) dia.addwhat (MSG_U(I_ONEVDOMAIN,"one virtual domain"));
	MENU_STATUS code = dia.editmenu (title,intro,help_vdomain,nof,0);
	return code;
}
static char privi_changed = 0;
/*
	Edit the definition of virtual email domains.
	Return != 0 if the has been one modification.
*/
PUBLIC int VDOMAINS::edit()
{
	int ret = 0;
	int nof = 0;
	while (1){
		MENU_STATUS code = select (
			MSG_U(T_VDOMAINS,"Virtual email domains")
			,MSG_U(I_VDOMAINS
				,"You can define new virtual email domain\n"
				 "virtual email hosting is a new concept.\n"
				 "It is not required for most mail server configuration.\n")
			,1
			,nof);
		if (code == MENU_QUIT || code == MENU_ESCAPE){
			break;
		}else if (code == MENU_ADD){
			VDOMAIN *d = new VDOMAIN;
			if (manage_edit (d,d->edit(*this)) >= 0) ret = 1;
		}else if (nof >= 0 && nof < getnb()){
			VDOMAIN *d = getitem(nof);
			if (manage_edit (d,d->edit(*this)) >= 0) ret = 1;
		}
	}
	if (ret == 1){
		checkperm(false);
		privi_changed = 1;
	}
	return ret;
}

class FIXPERM_VDOMAIN: public FIXPERM_TASK{
	int check (bool boottime);
};

static FIXPERM_VDOMAIN perm;

int FIXPERM_VDOMAIN::check (bool boottime)
{
	VDOMAINS conf;
	return conf.checkperm(boottime);
}

PUBLIC int VDOMAINS::checkperm(bool )
{
	int n = getnb();
	FIXPERM_SPECS specs;
	if (n > 0){
		specs.addline ("/etc/vmail root mail d 700 required");
		specs.addline (VHOME " root root d 755 required");
		specs.addline ("/var/spool/vmail root root d 755 required");
	}
	for (int i=0; i<n; i++){
		VDOMAIN *d = getitem(i);
		char buf[2*PATH_MAX];
		sprintf (buf,"%s/%s root mail d 1777 required"
			,VAR_SPOOL_VMAIL,d->domain.get());
		specs.addline (buf);
		sprintf (buf,"%s/%s root root d 755 required"
			,VHOME,d->domain.get());
		specs.addline (buf);
	}
	return specs.check();
}

static PRIVILEGES tb;

static void vdomain_setprivi()
{
	if (tb.getnb()==0 || privi_changed){
		privi_changed = 0;
		tb.remove_all();
		VDOMAINS vdom;
		int n = vdom.getnb();
		for (int i=0; i<n; i++){
			VDOMAIN *d = vdom.getitem(i);
			const char *dname = d->domain.get();
			char id[PATH_MAX];
			sprintf (id,"vdomain_%s",dname);
			tb.add (new PRIVILEGE (id,dname
				,MSG_U(T_VIRTDOMPRIV,"z-Virtual email domains")));
		}
	}
}

static PRIVILEGE_DECLARATOR vdom_decl(vdomain_setprivi);

void vdomain_editusers (USER *like)
{
	VDOMAINS vdom;
	if (vdom.getnb()==0){
		xconf_error (MSG_U(E_NOVDOM,"No virtual email domain defined"));
	}else{
		int nof = 0;
		while (1){
			MENU_STATUS code = vdom.select(
				MSG_U(T_PICKVDOM,"Pick the domain")
				,""
				,0,nof);
			if (code == MENU_QUIT || code == MENU_ESCAPE){
				break;
			}else{
				VDOMAIN *v = vdom.getitem(nof);
				if (v != NULL){
					const char *dname = v->domain.get();
					char path[PATH_MAX];
					sprintf (path,"%s/passwd.%s",ETC_VMAIL,dname);
					CONFIG_FILE file (path,help_nil
						,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
						,"root","mail",0640);
					/* #Specification: virtual domain / users / uid
						UID for users of virtual email domain are
						receiving UIDs starting at 60000 to avoid
						clash with normal users. This is done so a normal
						won't be able to read the folder of any
						virtual users. For technical reason and compatibility
						with barely modified POP server, the spool
						directory for virtual account is world readable.
					*/
					char pathhome[PATH_MAX];
					sprintf (pathhome,"%s/%s",VHOME,dname);
					vdomain_setprivi();
					PRIVILEGE *priv = privilege_lookup (dname);
					USERS users (file,pathhome,60000);
					users.edit(like,priv,0);
				}
			}
		}
	}
}

