# =======================================================================
# TEX Application Module 
# for Hamster Font Manager
#
# $Date: 1998/10/30 17:22:54 $ 
# $Revision: 1.36 $
#
# =======================================================================
#
# Copyright (C) 1998 The Hamster Project Team [GB]
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# =======================================================================

# REQUIRED-FILES: mod_tex.map afmmaker.ps tex.xbm

# Append to list of known applications
lappend AppMods "TEX"

putdebug "LOADING APP: TEX"

# Configuration information
set tex_config(active) 0
set tex_config(fontdir) ""
set tex_config(fontmapfile) ""
set tex_config(texhash) ""

# virtual fonts already available on the system
# tex_vfonts 
# tex_tfms

# Mark in config file where data for TeX begins
set tex_config(mark) "<TEX>"

# If the config info is loaded from the config-file, set this to 1
set tex_config(loaded) 0

set tex_config(icon) [image create bitmap -file [file join $HFMDirectory tex.xbm]]

# make tex-userinput global:
set tex_userinput_fontdir ""
set tex_userinput_fontmapfile ""
set tex_userinput_texhash ""
set tex_userinput_encfile ""
set tex_userinput_enccmds ""
#the following two variables are used in the font-prefs-dialog
set tex_currentfont ""
set tex_currentfontfile ""

# Font-Aliases:   tex_fonts (fontname) : { alias1 alias2 ... }
# Aliases KB:     tex_aliases (fontname) : stdalias
# BackupAliases:  tex_oldaliases (fontname) : { alias1 alias2 ... }
# Encoding-file:  tex_encfile  (fontname) : .enc-file
# Encoding-Cmd:   tex_enccmd   (fontname) : "cmd"     
# Font-Embedding: tex_fontembed
# Font-Encoding:  tex_fontencoding

# ========================================================================
# GLOBAL PROCEDURES - Following the HFM Module standard
# ========================================================================

# ------------------------------------------------------------------------
# TEX_Init
# ------------------------------------------------------------------------
#  Search for the section of this module in the configuration file
#  and read the information stored there if any.
#
#  Initialize module if necessary.
#
#  Read the configuration file of the associated application to determine  
#  which fonts are currently installed.
# ------------------------------------------------------------------------
# Config items:
#  fontdir <dir>
#  fontmapfile <filepath>

proc TEX_Init { configfile } {
    global tex_config tex_stdaliases tex_aliases HFMDirectory tex_vfonts tex_tfms
    
    seek $configfile 0 start

    # seek for TEX-Config-section

    while {[gets $configfile line] >= 0} {
	if {[string compare $line $tex_config(mark)] == 0} {

	    # Read my part of the config-file:

	    set tex_config(loaded) 1
	    gets $configfile buf
	    set tex_config(fontdir) [string range $buf 1 end]
	    gets $configfile buf
	    set tex_config(fontmapfile) [string range $buf 1 end]
	    gets $configfile buf
	    set tex_config(texhash) [string range $buf 1 end]

	    while {[gets $configfile line] >= 0} {
		if {[string compare $line "--- ALIASES KNOWLEDGE-BASE END ---"]==0} {
		    break;
		}

		set tex_aliases([lindex $line 1]) [lindex $line 0]
	    }

	    #
	    # Read mod_tex.map:
	    #

	    if {[catch {set in [open [file join $HFMDirectory mod_tex.map] r]}]} {
		puts "error reading mod_tex.map file."
		exit
	    }

	    while {[gets $in line] >= 0} {
		set tex_stdaliases([lindex $line 1]) [lindex $line 0]
	    }

	    close $in

	    #
	    # Find vfs and tfms on this system:
	    #

	    catch {
		set pipe [open "|find $tex_config(fontdir) -name \"*.vf\" -print" r]
		while {[gets $pipe line] >= 0} {
		    set tex_vfonts([file tail $line]) $line
		}
		close $pipe
	    }
	    catch {
		set pipe [open "|find $tex_config(fontdir) -name \"*.tfm\" -print" r]
		while {[gets $pipe line] >= 0} {
		    set tex_tfms([file tail $line]) $line
		}
		close $pipe
	    }

	    if {$tex_config(fontdir) != ""} {
		set tex_config(active) 1
		# read psfonts.map:
		return [ tex_readfontmap $tex_config(fontmapfile) ]
	    } {
		set tex_config(active) 0
		# Well...*sigh*
		return 1 
	    }
	}
    }

    # didn't find my section:
    return 0
}

# ------------------------------------------------------------------------
# TEX_CheckConfigInput
# ------------------------------------------------------------------------
# User input must be checked
# ------------------------------------------------------------------------
# Variables bound to configuration form:
#   tex_userinput_fontdir
#   tex_userinput_fontmapfile
#   tex_userinput_texhash

proc TEX_CheckConfigInput { } {
    global tex_userinput_fontmapfile tex_userinput tex_userinput_texhash tex_userinput_fontdir tex_config

    set errors ""

    if {$tex_userinput_fontdir != ""} {
	# OK, user wants TeX to be active
	set tex_userinput(active) 1

	if { [file isdirectory $tex_userinput_fontdir] == 0} {
	    lappend errors "No such directory: $tex_userinput_fontdir"
	}
	set fontdir [file join $tex_userinput_fontdir "fonts"]
	if { [file isdirectory $fontdir] == 0} {
	    lappend errors "No such directory: $fontdir"
	} else {
	    if { [file writable $fontdir] == 0} {
		lappend errors "Not writeable: $fontdir"
	    } else {
		if { [file isdirectory [file join $fontdir "tfm"]] == 0} {
		    if {[file mkdir [file join $fontdir "tfm"]] == 0} {
			lappend errors "$fontdir/tfm couldnt be created."
		    }
		} 
		if {[file writable [file join $fontdir "tfm"]] == 0} {
		    lappend errors "$fontdir/tfm is not writeable."
		}

		if { [file isdirectory [file join $fontdir "vf"]] == 0} {
		    if {[file mkdir [file join $fontdir "vf"]] == 0} {
			lappend errors "$fontdir/vf couldnt be created."
		    }
		}
		if {[file writable [file join $fontdir "vf"]] == 0} {
		    lappend errors "$fontdir/vf is not writeable."
		}
	    }
	}
	if { [file isfile $tex_userinput_fontmapfile] == 0} {
	    lappend errors "File specified for psfonts.map does not exist:\
		    $tex_userinput_fontmapfile"
	} else {
	    if {[file writable $tex_userinput_fontmapfile] == 0} {
		lappend errors "File specified for psfonts.map is not writeable: $tex_userinput_fontmapfile"
	    } else {
		# Reads the Fontmap file
		if {[tex_readfontmap $tex_userinput_fontmapfile] != 1} {
		    lappend errors "Could not read the psfont.map file:\
			    $tex_userinput_fontmapfile"
		}
	    }
	}
	
	if { $tex_userinput_texhash != "" } {
	    if {![file executable $tex_userinput_texhash]} {
		lappend errors "Texhash $tex_userinput_texhash is not executable!"
	    }
	}
    } {
	# Tex is not active.
	set tex_userinput(active) 0
    }

    if {$errors == ""} {
	# No errors: Replace configuration items
	set tex_config(fontdir) $tex_userinput_fontdir
	set tex_config(fontmapfile) $tex_userinput_fontmapfile
	set tex_config(texhash) $tex_userinput_texhash
	set tex_config(active) $tex_userinput(active)
    }

    return $errors
}

# ------------------------------------------------------------------------
# TEX_QueryInstalled
# ------------------------------------------------------------------------
# Answers if TeX is installed and active
# ------------------------------------------------------------------------
proc TEX_QueryInstalled { } {
    global tex_config
    
    return $tex_config(active)
}

# ------------------------------------------------------------------------
# TEX_QueryNewFonts
# ------------------------------------------------------------------------
# Returns List of new fonts
# ------------------------------------------------------------------------
proc TEX_QueryNewFonts { } {
    return {}
}

# ------------------------------------------------------------------------
# TEX_QueryConfigFrame
# ------------------------------------------------------------------------
# Returns a frame in which the configuration items for this modules
# are to be edited.  Bind input fields to known variable names.
# ------------------------------------------------------------------------
# Variables bound to configuration form:
#   tex_userinput_fontdir
#   tex_userinput_fontmapfile
#   tex_userinput_texhash

proc TEX_QueryConfigFrame { parent } {
    global tex_userinput_fontdir tex_userinput_fontmapfile
    global tex_userinput_texhash tex_config tex_dlg
    
    set tex_userinput_fontdir $tex_config(fontdir)
    set tex_userinput_fontmapfile $tex_config(fontmapfile)
    set tex_userinput_texhash $tex_config(texhash)

    if {!$tex_config(loaded)} {
	# First time config, try to find the paths
	tex_GuessConfig
    }

    set texframe $parent.texframe

    tixLabelFrame $texframe -label "TeX"  -labelside acrosstop -options {
	label.padX 5
    }

    set f [$texframe subwidget frame]

    frame $f.texmf

    label $f.texmf.l -text "texmf directory:" -width 30
    entry $f.texmf.e -width 30 -textvariable tex_userinput_fontdir
    button $f.texmf.b -text "Browse..." -command {
	global tex_userinput_fontdir
	UTIL_RequestDir "Select texmf directory" "texmf-Directory" tex_userinput_fontdir
    }
    pack $f.texmf.l -in $f.texmf -side left -padx 10
    pack $f.texmf.b -in $f.texmf -side right -padx 10
    pack $f.texmf.e -in $f.texmf -side right -padx 10 -expand yes -fill x

    frame $f.texmap

    label $f.texmap.l -text "psfonts.map:" -width 30
    entry $f.texmap.e -width 30 -textvariable tex_userinput_fontmapfile
    button $f.texmap.b -text "Browse..." -command {
	global tex_userinput_fontmapfile
	UTIL_RequestFile "Select psfonts.map" tex_userinput_fontmapfile {
	    {{psfonts.map}		{psfonts.map}}
	    {{*}	                {*     All files}}
	}
    }
    
    pack $f.texmap.l -in $f.texmap -side left -padx 10 
    pack $f.texmap.b -in $f.texmap -side right -padx 10
    pack $f.texmap.e -in $f.texmap -side right -padx 10 -expand yes -fill x

    frame $f.texhash 

    label $f.texhash.l -text "Texhash command (if any):" -width 30
    entry $f.texhash.e -width 30 -textvariable tex_userinput_texhash
    button $f.texhash.b -text "Browse..." -command {
	global tex_userinput_fontmapfile
	UTIL_RequestFile "Select texhash command (if any)" tex_userinput_texhash {
	    {{texhash}		{texhash}}
	    {{*}	                {*     All files}}
	} 
    }
    pack $f.texhash.l -in $f.texhash -side left -padx 10 
    pack $f.texhash.b -in $f.texhash -side right -padx 10
    pack $f.texhash.e -in $f.texhash -side right -padx 10 -expand yes -fill x 

    pack $f.texmf $f.texmap $f.texhash -in $f -side top -anchor e -expand yes -fill both

    return $texframe
}



# ------------------------------------------------------------------------
# TEX_QueryTypeSupported
# ------------------------------------------------------------------------
# Find out if a specific font type is supported by TeX / dvips
# ------------------------------------------------------------------------

proc TEX_QueryTypeSupported { fonttype } {
    if { [string compare [string tolower $fonttype] "ps"] == 0} {
	return 1
    } {
	return 0
    }
}

# ------------------------------------------------------------------------
# TEX_QueryPrefsFrame
# ------------------------------------------------------------------------
# Returns a frame in which font preferences fort his font
# are to be edited.  Bind input fields to known variable names.
# ------------------------------------------------------------------------

proc TEX_QueryPrefsFrame { parent fontinfo } {
    global tex_fontembed tex_userinput_fontembed tex_currentfont tex_userinput_enccmds tex_userinput_encfile tex_encfile tex_enccmd tex_currentfontfile tex_config

    if {$tex_config(active) == 1} {

	set texframe $parent.texframe

	tixLabelFrame $texframe -label "dvips options"  -labelside acrosstop -options {
	    label.padX 5
	}

	putdebug "Fontinfo : $fontinfo"

	set f [$texframe subwidget frame]
	set fontname [lindex $fontinfo 0]
	set tex_currentfont $fontname
	set tex_currentfontfile [lindex $fontinfo 1]

	if {[info exists tex_fontembed($fontname)]} {
	    set tex_userinput_fontembed 1
	} else {
	    set tex_userinput_fontembed 0
	}

	checkbutton $f.cb -text "embed font" -anchor w -variable tex_userinput_fontembed

	if {[info exists tex_enccmd($fontname)]} {
	    set tex_userinput_enccmds $tex_enccmd($fontname)
	} else {
	    set tex_userinput_enccmds ""
	}

	frame $f.bottom1
	pack $f.bottom1 -in $f -side bottom -expand yes -fill x

	label $f.bottom1.l2 -text "Encoding commands: " -width 20
	entry $f.bottom1.e2 -width 30 -textvariable tex_userinput_enccmds

	pack $f.bottom1.l2 -in $f.bottom1 -side left -anchor w -pady 4
	pack $f.bottom1.e2 -in $f.bottom1 -side right -expand yes -fill both -pady 4

	if {[info exists tex_encfile($fontname)]} {
	    set tex_userinput_encfile $tex_encfile($fontname)
	} else {
	    set tex_userinput_encfile ""
	}

	frame $f.bottom2
	pack $f.bottom2 -in $f -side bottom -expand yes -fill x

	label $f.bottom2.l1 -text "Encoding file: " -width 20
	entry $f.bottom2.e1 -width 30 -textvariable tex_userinput_encfile

	pack $f.bottom2.l1 -in $f.bottom2 -side left -anchor w -pady 4
	pack $f.bottom2.e1 -in $f.bottom2 -side right -expand yes -fill both -pady 4

	pack $f.cb -in $f -side top -expand no -fill both

	return $texframe
    } else {
	return ""
    }
}

# ------------------------------------------------------------------------
# TEX_CheckPrefsInput
# ------------------------------------------------------------------------
# Check if font entered preferences are valid
# ------------------------------------------------------------------------

proc TEX_CheckPrefsInput { } {
    global tex_currentfont tex_currentfontfile tex_fontembed tex_fontencoding tex_userinput_fontembed tex_fontembed tex_encfile tex_enccmd tex_userinput_encfile tex_userinput_enccmds tex_config

    if {$tex_config(active) == 1} {

	if {$tex_userinput_fontembed == 1} {
	    set tex_fontembed($tex_currentfont) $tex_currentfontfile
	} else {
	    catch {unset tex_fontembed($tex_currentfont)}
	}

	if {$tex_userinput_encfile == ""} {
	    catch { unset tex_encfile($tex_currentfont) }
	} else {
	    set tex_encfile($tex_currentfont) $tex_userinput_encfile
	}

	if {$tex_userinput_enccmds == ""} {
	    catch {unset tex_enccmd($tex_currentfont)}
	} else {
	    set tex_enccmd($tex_currentfont) $tex_userinput_enccmds
	}
    }
    return ""
}


# ------------------------------------------------------------------------
# TEX_QueryIcon
# ------------------------------------------------------------------------
# Returns a nice TeX - icon
# ------------------------------------------------------------------------

proc TEX_QueryIcon {} {
    global tex_config
    return $tex_config(icon)
}

# ------------------------------------------------------------------------
# TEX_QueryAppName
# ------------------------------------------------------------------------
# Returns the name of this application
# ------------------------------------------------------------------------

proc TEX_QueryAppName { } {
    return "TeX"
}

# ------------------------------------------------------------------------
# TEX_WriteConfig
# ------------------------------------------------------------------------
# - Writes module configuration to HFM's config file
# - Writes current Fontmap
# ------------------------------------------------------------------------

proc TEX_WriteConfig { file } {
    global tex_config tex_aliases tex_fontembed
    
    # Write to config file
    puts $file $tex_config(mark)
    puts $file "\#$tex_config(fontdir)"
    puts $file "\#$tex_config(fontmapfile)"
    puts $file "\#$tex_config(texhash)"

    foreach font [array names tex_aliases] {
	puts $file "$tex_aliases($font) $font"
    }

    puts $file "--- ALIASES KNOWLEDGE-BASE END ---"

    # Save new psfonts.map file
    if {$tex_config(active)==1} {
	tex_writefontmap $tex_config(fontmapfile)
    }

    return ""
}

# ------------------------------------------------------------------------
# TEX_ActivateFont
# ------------------------------------------------------------------------
# Adds a new font to the TeX System
# ------------------------------------------------------------------------
# Type:
#  fontinfo - list in form  { fontname filename type fondry weight ... }

proc TEX_ActivateFont { fontinfo oldfile {stdname ""}} {
    global tex_config tex_fonts tex_oldaliases tex_fontembed HFMDirectory tex_enccmd tex_encfile tex_global_cancelled

    set fontname [lindex $fontinfo 0]
    set filename [lindex $fontinfo 1]

    if {$stdname == ""} {
	if {[info exists tex_oldaliases($fontname)]} {
	    # lappend tex_fonts($fontname) $tex_oldaliases($fontname)
	    set tex_fonts($fontname) $tex_oldaliases($fontname)

	    foreach fontalias $tex_fonts($fontname) {
		set error [tex_GenerateVfTfm $fontinfo $fontalias]
		if {$error != ""} {
		    return $error
		}
	    }

	    return ""
	} else {
	    set stdname [tex_generatestdname $fontname]
	}
    }

    set err [tex_valid_alias $fontname $stdname]

    if {$err != ""} {
	return $err
    }

    # Make sure we've got .tfms and .vfs for that font

    set error [tex_GenerateVfTfm $fontinfo $stdname]
    if {$error != ""} {
	return $error
    }

    if {$tex_global_cancelled == 1} {
	return ""
    }

    # finally: activate the new font

    lappend tex_fonts($fontname) $stdname


    #
    # Embed this font?
    #

    if {![info exists tex_fontembed($fontname)]} {
	set tex_fontembed($fontname) $filename
    }

    #
    # Set standard re-encoding?
    #

    if {![info exists tex_enccmd($fontname)]} {
	set tex_enccmd($fontname) " TeXBase1Encoding ReEncodeFont "
	set tex_encfile($fontname) "8r.enc"
    }

    return ""
}

# ------------------------------------------------------------------------
# TEX_DeactivateFont
# ------------------------------------------------------------------------
# Remove a font from the TeX System
# ------------------------------------------------------------------------

proc TEX_DeactivateFont { fontinfo } {
    global tex_fonts tex_oldaliases

    set fontname [lindex $fontinfo 0]

    catch { set tex_oldaliases($fontname) $tex_fonts($fontname) }
    catch { unset tex_fonts($fontname) }
    return ""
}

# ------------------------------------------------------------------------
# TEX_QueryFontActive
# ------------------------------------------------------------------------
# Check if specified font is active
# ------------------------------------------------------------------------

proc TEX_QueryFontActive { fontinfo } {
    global tex_fonts

    set fontname [lindex $fontinfo 0]
    return [info exists tex_fonts($fontname)]
}

# ------------------------------------------------------------------------
# TEX_AddAlias
# ------------------------------------------------------------------------
# Set the alias for an specific font
# ------------------------------------------------------------------------

proc TEX_AddAlias { fontinfo aliasname } {
    global tex_aliases tex_fonts tex_oldaliases

    return [TEX_ActivateFont $fontinfo "" $aliasname ]
}

# ------------------------------------------------------------------------
# tex_valid_alias
# ------------------------------------------------------------------------
# check if alias name is legal
# ------------------------------------------------------------------------

proc tex_valid_alias { fontname aliasname } {
    global tex_aliases tex_fonts tex_oldaliases

    foreach font [array names tex_aliases] {
	if {$fontname != $font} {
	    if {$tex_aliases($font)==$aliasname} {
		return "Alias \"$aliasname\" is stdalias for $font"
	    }
	}
    }
    
    foreach font [array names tex_oldaliases] {
	if {$fontname != $font} {
	    if {[lsearch $tex_oldaliases($font) $aliasname]>-1} {
		return "Alias \"$aliasname\" is a known alias for $font"
	    }
	}
    }
    
    foreach font [array names tex_fonts] {
	if {[lsearch $tex_fonts($font) $aliasname ]>-1} {
	    if {$font == $fontname} {
		return "Alias \"$aliasname\" is already in use for this font"
	    } else {
		return "Alias \"$aliasname\" is already in use for $font"
	    }
	}
    }

    if {[regexp "^\[A-Za-z\]\[A-Za-z0-9_\-\]*$" $aliasname]} {
	if {![info exists tex_aliases($fontname)]} {
	    set tex_aliases($fontname) $aliasname
	}
	return ""
    } else {
	return "Illegal alias \"$aliasname\" for TeX (dvips)"
    }
}



# ------------------------------------------------------------------------
# TEX_RenameAlias
# ------------------------------------------------------------------------
# Rename alias for an specific font
# ------------------------------------------------------------------------

proc TEX_RenameAlias { fontinfo oldalias newalias } {
    set fontname [lindex $fontinfo 0]

    set erg [tex_valid_alias $fontname $newalias]

    if {$erg == ""} {
	set erg [TEX_RemoveAlias $fontinfo $oldalias]
	if {$erg == ""} {
	    set erg [TEX_AddAlias $fontinfo $newalias]
	}
    }

    return $erg
}

# ------------------------------------------------------------------------
# TEX_RemoveAlias
# ------------------------------------------------------------------------
# Alias for TEX_DeactivateFont
# ------------------------------------------------------------------------
proc TEX_RemoveAlias { fontinfo aliasname } {
    global tex_fonts

    set fontname [lindex $fontinfo 0]

    if {![info exists tex_fonts($fontname)]} {
	return "Internal error: $fontname is not active in TeX!!"
    }

    foreach alias $tex_fonts($fontname) {
	if {$alias != $aliasname} {
	    lappend newaliaslist $alias
	}
    }
    
    if {[info exists newaliaslist]} {
	set tex_fonts($fontname) $newaliaslist
    } else {
	unset tex_fonts($fontname)
    }

    return ""
}

# ------------------------------------------------------------------------
# TEX_QueryAliases
# ------------------------------------------------------------------------
# Get the aliases for that font
# ------------------------------------------------------------------------

proc TEX_QueryAliases { fontinfo } {
    global tex_fonts

    set fontname [lindex $fontinfo 0]
    if {[info exist tex_fonts($fontname)]} {
	return $tex_fonts($fontname)
    }
    return ""
}

# ------------------------------------------------------------------------
# TEX_QueryMAS
# ------------------------------------------------------------------------
# Now we support multiple aliases, even for TeX
# ------------------------------------------------------------------------

proc TEX_QueryMAS {} {
    return 1
}

# ========================================================================
# PRIVATE PROCEDURES - For internal use only!
# ========================================================================

# ------------------------------------------------------------------------
# tex_readfontmap
# ------------------------------------------------------------------------
# Read dvips fontmap file to find out which fonts are currently installed
# ------------------------------------------------------------------------

proc tex_readfontmap { filename } {
    global tex_fonts tex_config tex_aliases tex_encfile tex_enccmd tex_fontembed

    if {[catch {set in [open $filename r]}]} {
	# Could not read Fontmap (permission denied, does not exist...)
	return 0
    }

    # Reset all global variables
    catch { unset tex_fonts }

    while {[gets $in line] >= 0} {
	
	set fontname [lindex $line 1]
	set texalias [lindex $line 0]

	set alreadythere 0
	foreach font [array names tex_fonts] {
	    if {[lsearch $tex_fonts($font) $texalias]>-1} {
		putdebug "Alias $texalias is already in use for $font"
		set alreadythere 1
	    }
	}

	if {$alreadythere == 0} {
	    if {! [info exists tex_aliases($fontname)]} {
		# remember alias
		set tex_aliases($fontname) $texalias
	    }
	    
	    lappend tex_fonts($fontname) $texalias
	    
	    # Read additional information if present:
	    
	    foreach item [lrange $line 2 end] {
		if [string match "<*.enc" $item] {
		    
		    set tex_encfile($fontname) [string range $item 1 end]
		    
		} else {
		    if [string match "<*" $item] {
			
			set tex_fontembed($fontname) [string range $item 1 end]
			
		    } else {
			
			set tex_enccmd($fontname) $item
			
		    }
		}
	    }
	}
    }

    close $in

    return 1
}

# ------------------------------------------------------------------------
# tex_writefontmap
# ------------------------------------------------------------------------
# Write new psfonts.map
# ------------------------------------------------------------------------

proc tex_writefontmap { filename } {
    global tex_fonts tex_config tex_fontembed tex_encfile tex_enccmd
    set out [open $filename w]

    foreach font [array names tex_fonts] {
	if {[info exists tex_fontembed($font)]} {
	    set do_embed $tex_fontembed($font)
	} else {
	    set do_embed 0
	}

	if {[info exists tex_encfile($font)]} {
	    set encfile "<$tex_encfile($font)"
	} else {
	    set encfile ""
	}

	if {[info exists tex_enccmd($font)]} {
	    set enccmd "\"$tex_enccmd($font)\""
	} else {
	    set enccmd ""
	}

	foreach alias $tex_fonts($font) {
	    if {$do_embed!="0"} {
		puts $out "$alias $font  $enccmd $encfile <$do_embed "
	    } else {
		puts $out "$alias $font  $enccmd $encfile"
	    }
	}
    }

    close $out
    
    #
    # run texhash, if any
    #
    
    if {$tex_config(texhash)!=""} {
	catch { exec $tex_config(texhash) }
    }
}

# ------------------------------------------------------------------------
# tex_generatestdname
# ------------------------------------------------------------------------
# returns standard up-to-8-character-name for this font
# ------------------------------------------------------------------------

proc tex_generatestdname { fontinfo } {
    global tex_stdaliases tex_aliases tex_alias
    
    set fontname [lindex $fontinfo 0]

    set tex_alias [string tolower $fontname]

    if {[catch {set tex_alias $tex_aliases($fontname)}]} {
	if {[catch {set tex_alias $tex_stdaliases($fontname)}]} {
	    # Ask user - sorry!
	    set fenster ".texaskuser"
	    
	    toplevel $fenster -border 1 -relief raised 
	    wm title $fenster "Enter TeX alias"
	    
	    label $fenster.label -text "Enter TeX alias for $fontname:"
	    entry $fenster.entry -width 40 -textvariable tex_alias
	    
	    pack $fenster.label $fenster.entry -padx 2m -pady 1m -side top -expand yes -fill both
	    
	    tixButtonBox $fenster.box -orientation horizontal
	    $fenster.box add ok    -text OK    -underline 0 -command "destroy $fenster" -width 5
	    
	    pack $fenster.box -side bottom -fill x -expand yes
	    tkwait visibility $fenster
	    grab set $fenster
	    tkwait window $fenster
	}
    } 

    return $tex_alias
}

# ------------------------------------------------------------------------
# tex_AskUserForAfmFile
# ------------------------------------------------------------------------
# Display a dialog asking the user for an afm-File
# ------------------------------------------------------------------------

set tex_userinput_afmfile ""
set tex_global_fontname ""
set tex_global_dlgreturn 0
set tex_global_window ".texaskuser"
set tex_global_cancelled 0

proc tex_AskUserForAfmFile { fontname } {
    global tex_userinput_afmfile tex_global_fontname tex_global_window tex_global_dlgreturn tex_global_cancelled

    set window ".texaskuser"
    set tex_global_fontname $fontname
    set tex_global_cancelled 0

    toplevel $window -border 1 -relief raised 
    wm title $window "Afm file needed"
    
    label $window.label1 -text "Do You have an .afm-file for $fontname?"
    label $window.label2 -text "(Simply enter nothing here to make HFM generate one)"

    frame $window.entryframe
    set tex_userinput_afmfile ""
    entry $window.entryframe.entry -width 40 -textvariable tex_userinput_afmfile

    button $window.entryframe.browse -text "Browse..." -command {
	global tex_userinput_afmfile tex_global_fontname
	UTIL_RequestFile "Afm-file for $tex_global_fontname" tex_userinput_afmfile {
	    {{*}		{*     -- All files}}
	    {{*.afm}	{*.afm -- Afm files}}
	}
    }


    pack $window.entryframe.entry -in $window.entryframe -side left -expand yes -fill both
    pack $window.entryframe.browse -in $window.entryframe -side left -fill both

    pack $window.label1 $window.label2 $window.entryframe -padx 2m -pady 1m -side top -expand yes -fill both
    
    tixButtonBox $window.box -orientation horizontal
    $window.box add ok    -text OK    -width 8 -underline 0 -command {
	global tex_global_dlgreturn tex_global_window
	destroy $tex_global_window
	set tex_global_dlgreturn 1
    }
    $window.box add close -text Cancel -underline 0 -width 8 -command {
	global tex_global_dlgreturn tex_global_window tex_global_cancelled 

	destroy $tex_global_window
	set tex_global_dlgreturn 0
	set tex_global_cancelled 1
    }
    
    pack $window.box -side bottom -fill x -expand yes

    tkwait visibility $window
    grab set $window
    tkwait window $window

    if {$tex_global_dlgreturn == 1} {
	return $tex_userinput_afmfile
    } else {
	return ""
    }
}

# ------------------------------------------------------------------------
# tex_GuessConfig
# ------------------------------------------------------------------------
# Try to guess TeX-configuration on this machine
# ------------------------------------------------------------------------

proc tex_GuessConfig { } {
    global tex_userinput_fontdir tex_userinput_fontmapfile tex_userinput_texhash env

    set tex_userinput_fontdir ""

    #
    # Try to find any kind of TeX on this system
    #
    
    if { [info exists env(TETEXDIR)]==1 } {
	putdebug "teTeX detected"
	set tex_userinput_fontdir [file join $env(TETEXDIR) texmf]
	set search_texhash $env(TETEXDIR)
    } else {
	if { [file isdirectory /usr/lib/teTeX] } {
	    putdebug "teTeX detected, TETEXDIR not set."
	    set tex_userinput_fontdir "/usr/lib/teTeX/texmf"
	    set search_texhash "/usr/lib/teTeX"
	} else {
	    if { [file isdirectory /usr/local/lib/teTeX] } {
		putdebug "teTeX detected, TETEXDIR not set."
		set tex_userinput_fontdir "/usr/local/lib/teTeX/texmf"
		set search_texhash "/usr/local/lib/teTeX"
	    } else {
		if { [file isdirectory /usr/local/teTeX] } {
		    putdebug "teTeX detected, TETEXDIR not set."
		    set tex_userinput_fontdir "/usr/local/teTeX/texmf"
		    set search_texhash "/usr/local/teTeX"
		} else {
		    if { [file isdirectory /usr/teTeX] } {
			putdebug "teTeX detected, TETEXDIR not set."
			set tex_userinput_fontdir "/usr/teTeX/texmf"
			set search_texhash "/usr/teTeX"
		    } else {
			if {[file isdirectory "/usr/lib/texmf"]} {
			    set tex_userinput_fontdir "/usr/lib/texmf"
			    putdebug "Slakware/cTeX/nTeX detected"
			} else {
			    if {[file isdirectory "/usr/local/lib/texmf"]} {
				set tex_userinput_fontdir "/usr/local/lib/texmf"
				putdebug "Slakware/cTeX/nTeX detected"
			    }
			}
		    }
		}
	    }
	}
    }	
    

    if {$tex_userinput_fontdir != ""} {
	#
	# Search for psfonts.map:
	#
	
	set pipe [open "|find $tex_userinput_fontdir -name psfonts.map -print" r]
	if {[gets $pipe line] >= 0} {
	    set tex_userinput_fontmapfile $line
	} else {
	    set tex_userinput_fontmapfile "error: psfonts.map not found"
	}
	close $pipe
    }

    if {[info exists search_texhash]} {
	#
	# Search for texhash:
	#

	set pipe [open "|find $search_texhash -name texhash -print" r]
	if {[gets $pipe line] >= 0} {
	    set tex_userinput_texhash $line
	} else {
	    set tex_userinput_texhash "error: teTeX without texhash?"
	}
	close $pipe
    } 
}

# ------------------------------------------------------------------------
# tex_LinkVfIfNeeded
# ------------------------------------------------------------------------
# If there is more than one alias, make sure there is
# a link to the standardnamed vf for each alias 
# ------------------------------------------------------------------------

proc tex_LinkVfIfNeeded { fontname } {
    
    global tex_fonts

    

}

# ------------------------------------------------------------------------
# tex_GenerateVfTfmFromAfm
# ------------------------------------------------------------------------
# Generate stdname.vf and stdname.tfm for the given font from the
# given afm-file
# ------------------------------------------------------------------------

proc tex_GenerateVfTfmFromAfm { afmfile stdname fontinfo } {
    global tex_config tex_fonts tex_oldaliases tex_fontembed tex_userreply HFMDirectory tex_vfonts tex_tfms

    putdebug "Trying to Generate VfTfm from $afmfile"

    if {[catch {exec afm2tfm $afmfile -v /tmp/$stdname /tmp/r$stdname} error]} {
	putdebug "afm2tfm failed: $error"
	return "afm2tfm failed: $error"
    }
    
    if {[catch {exec vptovf /tmp/$stdname.vpl /tmp/$stdname.vf /tmp/$stdname.tfm} error]} {
	catch {file delete -force /tmp/$stdname.vpl}
	catch {file delete -force /tmp/r$stdname.tfm}
	return "vptovf failed: $error"
    }
    
    # Uff - we got all the files we need. Move them:
    
    if {[catch {exec mv /tmp/r$stdname.tfm /tmp/$stdname.tfm $tex_config(fontdir)/fonts/tfm} error ]} {
	catch {exec rm /tmp/hfm-tmp[pid].afm /tmp/$stdname.vpl /tmp/r$stdname.tfm /tmp/$stdname.tfm /tmp/$stdname.vf}
	return "Couldn't move tfm files: $error"
    }
    
    if {[catch {exec mv /tmp/$stdname.vf [file join $tex_config(fontdir) "fonts/vf"]} error ]} {
	return "Couldn't move vf file: $error"
    }

    set tex_vfonts($stdname.vf) [file join $tex_config(fontdir) "fonts/vf/$stdname.vf"]
    set tex_tfms($stdname.tfm) [file join $tex_config(fontdir) "fonts/tfm/$stdname.tfm"]
    set tex_tfms(r$stdname.tfm) [file join $tex_config(fontdir) "fonts/tfm/r$stdname.tfm"]

    catch { file delete -force /tmp/$stdname.vpl }

    return ""
}

# ------------------------------------------------------------------------
# tex_tryToCopyVfFrom
# ------------------------------------------------------------------------
# See if realname.tfm rrealname.tfm and realname.vf exist
# and copy them if so
# ------------------------------------------------------------------------

proc tex_tryToCopyVfFrom { real_alias newname } {
    global tex_vfonts tex_tfms tex_config

    putdebug "trying to copy $real_alias..."

    if {[info exist tex_vfonts($real_alias.vf)]} {
	putdebug "vfont exists..."
	if {[info exist tex_tfms($real_alias.tfm)]} {
	    putdebug "tfm exists..."
	    if {[info exist tex_tfms(r$real_alias.tfm)]} {
		putdebug "rtfm exists..."
		#
		# Got it!
		#
		
		set source $tex_vfonts($real_alias.vf)
		set dest [file join $tex_config(fontdir) "fonts/vf/$newname.vf"]
		putdebug "Copying from $source to $dest"
		catch {file copy -force $source $dest}
		set tex_vfonts($newname.vf) $dest
		
		set source $tex_tfms($real_alias.tfm)
		set dest [file join $tex_config(fontdir) "fonts/tfm/$newname.tfm"]
		putdebug "Copying from $source to $dest"
		catch {file copy -force $source $dest}
		set tex_tfms($newname.tfm) $dest
		
		set source "r$tex_tfms($real_alias.tfm)"
		set dest [file join $tex_config(fontdir) "fonts/tfm/r$newname.tfm"]
		putdebug "Copying from $source to $dest"
		catch {file copy -force $source $dest}
		set tex_tfms(r$newname.tfm) $dest
		return 1
	    }
	}
    }
    return 0
}

# ------------------------------------------------------------------------
# tex_GenerateVfTfm
# ------------------------------------------------------------------------
# Generate stdname.vf and stdname.tfm for the given font if necessary
# ------------------------------------------------------------------------

set tex_userreply 0

proc tex_GenerateVfTfm { fontinfo {stdname ""}} {
    global tex_config tex_fonts tex_oldaliases tex_fontembed tex_userreply HFMDirectory tex_global_cancelled tex_vfonts tex_tfms tex_stdaliases tex_aliases

    set fontname [lindex $fontinfo 0]
    set filename [lindex $fontinfo 1]
    set addfiles [lindex $fontinfo 11]

    if {$stdname == ""} {
	set stdname [tex_generatestdname $fontname]
    }

    if {$stdname == ""} {
	return "Invalid alias given for $fontname"
    }

    #
    # Are there already the virtual fonts?
    #

    putdebug "Checking if $stdname.vf exists..."
    
    if {[info exist tex_vfonts($stdname.vf)]} {
	if {[info exist tex_tfms($stdname.tfm)]} {
	    if {[info exist tex_tfms(r$stdname.tfm)]} {
		putdebug "yes."
		return ""
	    }
	}
    }

    putdebug "...no"

    #
    # Perhaps $stdname is a strange alias and the
    # virtual fonts exists for the standard alias of
    # this font? Lets see...
    #
    
    if {![catch {set real_alias $tex_stdaliases($fontname)}]} {

	if {[tex_tryToCopyVfFrom $real_alias $stdname]} {
	    return ""
	}
    }

    if {![catch {set real_alias $tex_aliases($fontname)}]} {
	if {[tex_tryToCopyVfFrom $real_alias $stdname]} {
	    return ""
	}
    }

    if {![catch {set alias_list $tex_fontss($fontname)}]} {
	foreach real_alias $alias_list {
	    if {[tex_tryToCopyVfFrom $real_alias $stdname]} {
		return ""
	    }
	}
    }

    #
    # OK, we need an afm-file here.
    #
    
    putdebug "AddFiles for $fontname: $addfiles"
    
    #
    # perhaps there is one in $addfiles ?
    #
    
    foreach addfile $addfiles {
	set error [tex_GenerateVfTfmFromAfm $addfile $stdname $fontinfo ]  
	if {$error == ""} {
	    return ""
	}
    }
    
    # perhaps the user has an appropriate .afmfile ?
    
    set afmfile [tex_AskUserForAfmFile $fontname]
    
    if {$tex_global_cancelled == 1} {
	return ""
    }
    
    if {$afmfile != ""} {
	set error [tex_GenerateVfTfmFromAfm $afmfile $stdname $fontinfo ]  
	if {$error == ""} {
	    return ""
	}
    }	    
    
    #
    # Well - generate afm file
    #

    UTIL_ShowHamster -metertext "afmmaker.ps" -label "Generating vf/tfm..." -metervalue 0
    
    # include ps-fonts in afmmaker to make sure GS has it

    set tmpFontmap /tmp/hfm-Fontmap.[pid]

    # Creates an temporary Fontmap file
    set out [open $tmpFontmap w]
    puts $out "/$fontname	($filename)	;"
    close $out

#    catch { file delete -force /tmp/hfm-font[pid].pfa }
#    if {[catch {exec pfbtops $filename >/tmp/hfm-font[pid].pfa} error ]} {
#	set in [open $filename r]
#    } else {
#	set in [open /tmp/hfm-font[pid].pfa r]
#    }
#    
#    while {[gets $in line] >= 0} {
#	puts $out $line
#    }
#    
#    close $in
    
    UTIL_ShowHamster -metervalue 0.1
    
#    catch { file delete -force /tmp/hfm-font[pid].pfa }
    
    # generate the afmmaker.ps file
    
    set out [open /tmp/hfm-afmmaker[pid].ps w]
    puts $out "/fontName /$fontname def"
    set in [open [file join $HFMDirectory afmmaker.ps] r]
    while {[gets $in line] >= 0} {
	puts $out $line
    }
    close $out
    close $in
    
    catch { file delete -force /tmp/hfm-tmp[pid].afm }
    
    #
    # Not all gs-version quit safely...
    #
    
    set bError [catch {
	
	UTIL_ShowHamster -metervalue 0.2 -metertext "Running gs..."
	
	set in [open "|gs -q -dNODISPLAY -dBATCH -sFONTMAP=$tmpFontmap /tmp/hfm-afmmaker[pid].ps" r]
puts "gs -q -dNODISPLAY -dBATCH -sFONTMAP=$tmpFontmap /tmp/hfm-afmmaker[pid].ps"
	set out [open /tmp/hfm-tmp[pid].afm w]

	set line ""

	while {[string compare $line EndFontMetrics] != 0} {
	    if {[gets $in line]==0} {
		break
	    }
	    if {[string match "Error:*" $line]} {
		catch {close $in}
		catch {close $out}
		
		catch {file delete -force /tmp/hfm-afmmaker[pid].ps }
		error $line
	    }
	    puts $out $line
	}
	
	catch {close $in}
	catch {close $out}
	
	catch {file delete -force /tmp/hfm-afmmaker[pid].ps }
	
    } error]

    catch { file delete -force $tmpFontmap }

	UTIL_ShowHamster -metervalue 0.7 -metertext "generating vf tfm"
	
	if {$bError==1} {
	    destroy .texprogress
	    UTIL_ShowHamster -done
	    return "Error producing .afm file. GS failed: $error"
	}
	
	set afmfile /tmp/hfm-tmp[pid].afm
	set error [tex_GenerateVfTfmFromAfm $afmfile $stdname $fontinfo ]  
	catch {file delete -force /tmp/hfm-tmp[pid].afm}
	if {$error == ""} {
	    UTIL_ShowHamster -done
	    return ""
	} else {
	    UTIL_ShowHamster -done
	    return $error
	}
    }

    UTIL_ShowHamster -done
    
    return "Internal TeX-Module error : Couldn't read generated .afm-file"
}


