# =======================================================================
# Graphical User Interface Module
# for HAMSTER Font Manager.
#
# $Date: 1998/11/04 10:02:20 $
# $Revision: 1.36 $
#
# =======================================================================
#
# Copyright (C) 1998 The Hamster Project Team [AM]
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# =======================================================================

# REQUIRED-FILES: Hamster.gif manual.ps.gz

#****************************************************************************
#
# PRIVATE VARIABLES
#
#****************************************************************************

set gui_ConfigMsg "Enter configuration information for your applications:"
set gui_AboutText "HAMSTER Font Manager\nVersion $HFMVersion\n\n\
(c) 1998 The Hamster Project Team:\n\
G. Bartsch\n\
E. Baschny\n\
M. Fischer\n\
A.Miller\n\n\
HFM comes with ABSOLUTELY NO WARRANTY,\n\
See the LICENSE file for details."

set gui_ManualFile [ file join $HFMDirectory "manual.ps" ]

# gui_Widgets
# Array containing the pathnames of widgets used by multiple procedures
set gui_Widgets(MessageWindow) .topMessage
set gui_Widgets(DialogWindow) .topDialog

# gui_Applications
# Integer-indexed array containing lists in the format { prefix name icon }.
# First index is 1! gui_Applications(count) contains the number of entries.
# Values are set by gui_Configure and GUI_Init.
set gui_Applications(count) 0

# gui_Images
# Array containing some images. Set by gui_GetImages.

# gui_Values
# Array which is used to store the input of modal dialogs.

#****************************************************************************
#
# PRIVATE PROCEDURES
#
#****************************************************************************

#-------------------------------------------------------------------------
#
# gui_String2Name
#
#-------------------------------------------------------------------------

# Convert any string into a name containing only letters, digits, and "_".
# Parameter:    string (string): the string to convert.
# Return value: the converted string.

proc gui_String2Name { string } {
    regsub -all _ $string _[ scan _ %c ch ; format %02x $ch ] string
    regsub -all \[\{\] $string _[ scan \{ %c ch ; format %02x $ch ] string
    regsub -all \[\}\] $string _[ scan \} %c ch ; format %02x $ch ] string
    regsub -all {[\\]} $string _[ scan \\ %c ch ; format %02x $ch ] string
    regsub -all {[^a-zA-Z0-9_]} $string {_[ scan {&} %c ch ; format %02x $ch ]} string
    eval return \"$string\"
}

#-------------------------------------------------------------------------
#
# gui_Name2String
#
#-------------------------------------------------------------------------

# Convert a string created by gui_String2Name back to the original value.
# Parameter:    string (string): the name to convert.
# Return value: the re-converted string.

proc gui_Name2String { string } {
    regsub -all {_([a-fA-F0-9][a-fA-F0-9])} $string {[format %c 0x\1]} string
    eval return \"$string\"
}

#-------------------------------------------------------------------------
#
# gui_CreateWindow
#
#-------------------------------------------------------------------------

# Create a new window with HFM's default style.
# Parameters:   window (string): the window identifier.
#               title (string): window title.
# Return value: none

proc gui_CreateWindow { window title } {
    catch { destroy $window }
    toplevel $window -relief raised -border 1
    wm title $window $title
}

#-------------------------------------------------------------------------
#
# gui_ShowMessage
#
#-------------------------------------------------------------------------

# Open a window and show a message.
# Parameters:   message (string): the message to display.
#               title (string, optional): window title.
#               image (string, optional): the image to show or "".
# Return value: none

proc gui_ShowMessage { message { title Message } { image "" } } {
    global gui_Widgets
    set window $gui_Widgets(MessageWindow)
    gui_CreateWindow $window $title
    frame $window.f
    pack $window.f -side top
    if [ string length $image ] then {
	label $window.f.image -image $image
	pack $window.f.image -side left -padx 20 -pady 10
    }
    label $window.f.label -text $message -justify left
    pack $window.f.label -side left -padx 20 -pady 10 -expand 1 -fill x
    tixButtonBox $window.box -orientation horizontal -relief flat
    $window.box add close -text "Dismiss" -width 10 -underline 0 \
	    -command "destroy $window"
    pack $window.box -side bottom -expand 1 -fill x
    set close [ $window.box subwidget close ]
    bind $window <Key-Return> "$close invoke"
    bind $window <Key-Escape> "$close invoke"
    bind $window <Key-space> "$close invoke"
    bind $window <Key-d> "$close invoke"
}

#-------------------------------------------------------------------------
#
# gui_ErrorMessage
#
#-------------------------------------------------------------------------

# Show an error message if necessary.
# Parameters:   message (string): the message to display or "".
#               modale (bool, optional): whether dialog should be modale.
# Return value: "1" if the message has been displayed, "0" if it was empty.

proc gui_ErrorMessage { message { modale 0 } } {
    global gui_Widgets
    if [ string length $message ] then {
	gui_ShowMessage $message "Error"
	if $modale then {
	    tixPushGrab $gui_Widgets(MessageWindow)
	    tkwait window $gui_Widgets(MessageWindow)
	    tixPopGrab
	}
	return 1
    } else {
	return 0
    }
}

#-------------------------------------------------------------------------
#
# gui_WriteConfig
#
#-------------------------------------------------------------------------

# Write config file and display an eventual error message.
# Parameters:   none
# Return value: 1 if config file could be saved, 0 otherwise.

proc gui_WriteConfig { } {
    return [ expr ! [ gui_ErrorMessage [ FM_WriteConfig ] ] ]
}

#-------------------------------------------------------------------------
#
# gui_GetApps
#
#-------------------------------------------------------------------------

# Set gui_Applications array and file patterns for file selection dialog.
# Parameters:   none
# Return value: none

proc gui_GetApps { } {
    global gui_Applications gui_Widgets
    # set array "gui_Applications".
    set i 0
    catch { unset gui_Applications }
    foreach app [ FM_QueryApps ] {
	incr i
	set gui_Applications($i) \
		[ list $app [ FM_QueryAppName $app ] [ FM_QueryIcon $app ] ]
    }
    set gui_Applications(count) $i
    # Set file patterns of file selection dialog.
    set fsbox [ $gui_Widgets(FileSelectDialog) subwidget fsbox ]
    $fsbox configure -filetypes [ FM_QueryFilePatterns ]
    [ $fsbox subwidget types ] pick 0
}

#-------------------------------------------------------------------------
#
# gui_ShowConfigDialog
#
#-------------------------------------------------------------------------

# Display configuration-dialog.
# Parameters:   none
# Return value: "1" if the user confirmed the configuration dialog,
#               "0" if she pressed "Cancel" instead.

proc gui_ShowConfigDialog { } {
    global gui_Widgets gui_Values gui_ConfigMsg
    
    putdebug "gui_ShowConfigDialog"

    # Delete values from old dialogs.
    catch { unset gui_Values }
    set w $gui_Widgets(DialogWindow)
    gui_CreateWindow $w "Config"
    # Create label.
    label $w.gui_l -justify left -text $gui_ConfigMsg
    # Command substrings for buttons.
    set c1 "set gui_Values(button)"
    set c2 "; destroy $w"
    set gui_Values(button) 0
    # Create buttons.
    tixButtonBox $w.gui_box -orientation horizontal -relief flat
    $w.gui_box add ok -text OK -underline 0 -command "$c1 1 $c2" -width 7
    $w.gui_box add ca -text Cancel -underline 0 -command "$c1 0 $c2" -width 7
    # Key bindings for buttons.
    bind $w <Key-Return> "[ $w.gui_box subwidget ok ] invoke"
    bind $w <Control-Key-o> "[ $w.gui_box subwidget ok ] invoke"
    bind $w <Meta-Key-o> "[ $w.gui_box subwidget ok ] invoke"
    bind $w <Key-Escape> "[ $w.gui_box subwidget ca ] invoke"
    bind $w <Control-Key-c> "[ $w.gui_box subwidget ca ] invoke"
    bind $w <Meta-Key-c> "[ $w.gui_box subwidget ca ] invoke"
    # Query frames from plugins
    set cfg [ FM_QueryConfigFrame $w ]
    # Pack all.
    pack $w.gui_box -side bottom -fill x -expand yes
    pack $w.gui_l $cfg -side top -fill x -padx 5 -pady 5
    # Wait until dialog is closed.
    tixPushGrab $w
    tkwait window $w
    tixPopGrab
    # Now we can return which button was pressed.
    return $gui_Values(button)
}

#-------------------------------------------------------------------------
#
# gui_Configure
#
#-------------------------------------------------------------------------

# Display configuration-dialog until input is valid or the user cancels it.
# Parameters:   none
# Return value: "1" if the user confirmed the configuration dialog,
#               "0" if she pressed "Cancel" instead.

proc gui_Configure { } {
    while { [ gui_ShowConfigDialog ] } {
	# User confirmed dialog. Check if input is valid.
	set result [ FM_OnConfigClosed ]
	gui_ErrorMessage [ lindex $result 1 ] 1
	if [ lindex $result 0 ] then {
	    # Input is valid, configuration is saved, plugins are reconfigured.
	    # Reconfigure GUI module.
	    gui_GetApps
	    gui_SetTrees
	    return 1
	}
	# There were errors. Try again.
    }
    # Cancelled.
    return 0
}

#-------------------------------------------------------------------------
#
# gui_Quit
#
#-------------------------------------------------------------------------

# Terminate program.
# Parameters:   none
# Return value: none

proc gui_Quit { } {
    global gui_Widgets
    # update all config files
    gui_ErrorMessage [ FM_WriteConfig ] 1
    # Terminate program
    exit
}

#-------------------------------------------------------------------------
#
# gui_ShowAbout
#
#-------------------------------------------------------------------------

# Display about box.
# Parameters:   none
# Return value: none

proc gui_ShowAbout { } {
    global gui_AboutText gui_Images
    set img $gui_Images(Hamster:0)
    gui_ShowMessage $gui_AboutText "About" $img
}

#-------------------------------------------------------------------------
#
# gui_ShowHelp
#
#-------------------------------------------------------------------------

# Display user manual.
# Parameters:   none
# Return value: none

proc gui_ShowHelp { } {
    global gui_ManualFile gui_ShowHelpUseGV

    if { ! [ info exist gui_ShowHelpUseGV ] } then {
	set viewer ""
	catch {
	    set inp [ open "|which gv" ]
	    set viewer [ read -nonewline $inp ]
	    close $inp
	}
	if [ file executable $viewer ] then {
	    set gui_ShowHelpUseGV 1
	} else {
	    set gui_ShowHelpUseGV 0
	    # not used #set viewer "ghostview"
	}
    }

    if { ! [ file readable $gui_ManualFile ] } then {

	# Postscript manual file not found. Look for a compressed one.
	foreach ext { {.gz} {.Z} {.zip} } {
	    if [ file readable ${gui_ManualFile}$ext ] then {
		set manfile ${gui_ManualFile}$ext
		break
	    }
	}

	if { ! [ info exist manfile ] } then {
	    # No manual there.
	    gui_ErrorMessage "Manual not found."

	} elseif [ catch { 
	    # Compressed manual found. Show it.
	    if $gui_ShowHelpUseGV then {
		exec gv $manfile >& /dev/null &
	    } else {
		exec gzip -dc $manfile > $gui_ManualFile
		exec sh -c "ghostview $gui_ManualFile ; rm $gui_ManualFile" &
	    } 
	} ] then {
	    gui_ErrorMessage "Can't show manual file ${manfile}: $err."
	} 

    } elseif [ catch { 
	# Postscript file found. Show it.
	if $gui_ShowHelpUseGV then {
	    exec gv $gui_ManualFile >& /dev/null &
	} else {
	    exec ghostview $gui_ManualFile >& /dev/null &
	} 
    } ] then {
	gui_ErrorMessage "Can't run ghostview."
    } 
}

#-------------------------------------------------------------------------
#
# gui_FileSelDlgCmd
#
#-------------------------------------------------------------------------

# Called when the user presses "Ok" or <Return> in the file selection dialog.
# Parameter:    file (string): the contents of the file entry.
# Return value: none

proc gui_FileSelDlgCmd { file } {
    global gui_Widgets gui_Values
    if [ file isfile $file ] then {
	# Regular file selected; accept.
	set gui_Values(files) [ list $file ]
    } elseif [ file isdirectory $file ] then {
	# Directory name typed. Show dialog again.
	[ $gui_Widgets(FileSelectDialog) subwidget fsbox ] configure -dir $file
	$gui_Widgets(FileSelectDialog) popup
    } else {
	# Value isn't a valid file name. Show dialog again.
	$gui_Widgets(FileSelectDialog) popup
    }
}

#-------------------------------------------------------------------------
#
# gui_FileSelDlgOk
#
#-------------------------------------------------------------------------

# Called when the user presses "Ok" in the file selection dialog.
# Parameter:    oldcmd (string): the original command of the button.
# Return value: none

proc gui_FileSelDlgOk { oldcmd } {
    global gui_Widgets gui_Values
    set fs [ $gui_Widgets(FileSelectDialog) subwidget fsbox ]
    eval $oldcmd
    if [ string length [ $fs cget -value ] ] then {
	# No wildcards, default procedure accepts value.
	# Handled by gui_FileSelDlgCmd, nothing to do here.
    } else {
	# Wildcards in file name. Get matching file names.
	set dir [ $fs cget -dir ]
	set filespecs [ [ $fs subwidget file ] cget -value ]
	set gui_Values(files) ""
	foreach fs $filespecs {
	    set gui_Values(files) [ concat $gui_Values(files) \
		    [ glob -nocomplain [ file join $dir $fs ] ] ]
	}
	$gui_Widgets(FileSelectDialog) popdown
    }
}

#-------------------------------------------------------------------------
#
# gui_FileSelDlgCancel
#
#-------------------------------------------------------------------------

# Called when the user presses "Cancel" in the file selection dialog.
# Parameter:    oldcmd (string): the original command of the button.
# Return value: none

proc gui_FileSelDlgCancel { oldcmd } {
    global gui_Values
    eval $oldcmd
    set gui_Values(files) ""
}

#-------------------------------------------------------------------------
#
# gui_CreateFileSelectDialog
#
#-------------------------------------------------------------------------

# Create Dialog window used by gui_AddFont.
# Parameters:   none
# Return value: none

proc gui_CreateFileSelectDialog { } {
    global gui_Widgets
    set dlg .fileselect
    set gui_Widgets(FileSelectDialog) $dlg
    tixExFileSelectDialog $dlg -command gui_FileSelDlgCmd \
	    -title "Select font(s) to add"
    set bbox [ $dlg subwidget fsbox ]
    [ [ $dlg subwidget fsbox ] subwidget ok ] config -command \
	    "gui_FileSelDlgOk [ list [ [ $bbox subwidget ok ] cget -command ] ]"
    [ [ $dlg subwidget fsbox ] subwidget cancel ] config -command \
	    "gui_FileSelDlgCancel [ list [ [ $bbox subwidget cancel ] cget -command ] ]"
    tixLabelFrame $dlg.addaction -labelside acrosstop -label "How should the file(s) be added?"
    pack $dlg.addaction -side bottom -fill x
    set f [ $dlg.addaction subwidget frame ]
    radiobutton $f.r1 -value "cp" -variable gui_Values(action) -text "Copy"
    radiobutton $f.r2 -value "mv" -variable gui_Values(action) -text "Move"
    radiobutton $f.r3 -value "ln" -variable gui_Values(action) -text "Hard link"
    radiobutton $f.r4 -value "ln -s" -variable gui_Values(action) -text "Symbolic link"
    pack $f.r1 $f.r2 $f.r3 $f.r4 -side left -fill x -expand 1 -padx 5 -pady 5
}

#-------------------------------------------------------------------------
#
# gui_AddFontGroupItem
#
#-------------------------------------------------------------------------

# Adds an item to the right position in the fonts tree list, but doesn't 
# update font/group count variables or group's checkboxes.
# Parameters:   group (string): the name of the group.
#               font (string): the name of the font, or "" if it's a group.
# Return value: none

proc gui_AddFontGroupItem { group font } {
    global gui_Widgets gui_Applications gui_Images
    set hlist $gui_Widgets(FontsTreeList)

    if [ string length $font ] {
	# Item is a font.
	set image [ FM_QueryIcon [ lindex [ FM_QueryFontInfo $font ] 1 ] ]
	set parent [ gui_String2Name $group ]
	set newitem ${parent}.[ gui_String2Name $font ]
	set itemtext $font
    } else {
	# The item is a group.
	set image $gui_Images(folder)
	set parent ""
	set newitem [ gui_String2Name $group ]
	set itemtext $group
    }

    # Add the item to fonts tree list.
    foreach item [ $hlist info children $parent ] {
	set itemtxt [ gui_Name2String [ lindex [ split $item . ] end ] ]
	if { [ string compare $itemtxt $itemtext ] > 0 } then {
	    $hlist add $newitem -text $itemtext -image $image -before $item
	    break
	}
    }
    if { ! [ $hlist info exist $newitem ] } then {
	$hlist add $newitem -text $itemtext -image $image
    }


    # Create checkboxes.
    for { set i 1 } { $i <= $gui_Applications(count) } { incr i } {
	set app [ lindex $gui_Applications($i) 0 ]

	if [ string length $font ] {
	    # Item is a font.
	    set image  [ FM_QueryFontActive $font $app ]
	} else {
	    # The item is a group.
	    set image N/A
	}

	$hlist item create $newitem $i -itemtype image \
		-image $gui_Images($image)
    }
}

#-------------------------------------------------------------------------
#
# gui_AddFont
#
#-------------------------------------------------------------------------

# Prompt the user for files and add the fonts to the selected group.
# Parameters:   none
# Return value: none

proc gui_AddFont { } {
    global gui_Widgets gui_Values gui_Applications gui_Images
    set gitem [ lindex [ split [ $gui_Widgets(FontsTreeList) \
	    info selection ] . ] 0 ]
    set group [ gui_Name2String $gitem ]
    set hlist $gui_Widgets(FontsTreeList)
    # Delete values from old dialogs.
    catch { unset gui_Values }
    set gui_Values(action) "cp"
    set dlg $gui_Widgets(FileSelectDialog)
    # Show dialog.
    $dlg popup
    tixPushGrab $dlg
    tkwait variable gui_Values(files)
    tixPopGrab
    # Process user input.
    if { ! [ string length $gui_Values(files) ] } then { return }
    set errors ""
    foreach file $gui_Values(files) {
	putdebug "Add `$file' using `$gui_Values(action)'."

	set result [ FM_AddFont $file $group $gui_Values(action) ]
	if [ string length [ lindex $result 0 ] ] then {
	    # There was an error.
	    lappend errors [ lindex $result 0 ]
	} else {
	    # The font could be added.
	    set font [ lindex $result 1 ]
	    if [ string length $group ] then {
		gui_AddFontGroupItem $group $font
		incr gui_Widgets(FontCount)
	    }
	}
    }
    if [ info exist font ] then {
	# At least one font has been added.
	if [ string length $group ] then {
	    # Update the group's checkboxes
	    for { set i 1 } { $i <= $gui_Applications(count) } { incr i } {
		gui_UpdateCheckbox {} $group [ lindex $gui_Applications($i) 0 ]
	    }

	    # Select the last of the inserted fonts.
	    set newitem $gitem.[ gui_String2Name $font ]
	    $hlist selection clear
	    $hlist selection set $newitem
	    $hlist anchor set $newitem
	    $hlist see $newitem
	    $gui_Widgets(FontsTree) autosetmode
	    gui_OnItemSelected $newitem
	    # Set status bar.
	    $gui_Widgets(StatusBarLabel) configure -text ""
	} else {
	    # We don't know the group name. Redraw the whole tree.
	    gui_SetTrees
	}
    }
    gui_ErrorMessage [ join $errors \n ]
}

#-------------------------------------------------------------------------
#
# gui_ShowPrefsDialog
#
#-------------------------------------------------------------------------

# Display preferences dialog.
# Parameter:    font (string): the name of the font for which the preferences
#                    should be displayed.
# Return value: "1" if the user confirmed the preferences dialog,
#               "0" if he pressed "Cancel" instead.

proc gui_ShowPrefsDialog { font } {
    global gui_Widgets gui_Values
    if { ! [ string length $font ] } then { return 0 }
    # Delete values from old dialogs.
    catch { unset gui_Values }

    set w $gui_Widgets(DialogWindow)
    gui_CreateWindow $w "Properties"
    # Create label.
    label $w.gui_l -justify left -text "Properties for font `$font':"

    # Command substrings for buttons.
    set c1 "set gui_Values(button)"
    set c2 "; destroy $w"
    set gui_Values(button) 0

    # Create buttons.
    tixButtonBox $w.gui_box -orientation horizontal -relief flat
    $w.gui_box add ok -text OK -underline 0 -command "$c1 1 $c2" -width 7
    $w.gui_box add ca -text Cancel -underline 0 -command "$c1 0 $c2" -width 7

    # Key bindings for buttons.
    bind $w <Key-Return> "[ $w.gui_box subwidget ok ] invoke"
    bind $w <Control-Key-o> "[ $w.gui_box subwidget ok ] invoke"
    bind $w <Meta-Key-o> "[ $w.gui_box subwidget ok ] invoke"
    bind $w <Key-Escape> "[ $w.gui_box subwidget ca ] invoke"
    bind $w <Control-Key-c> "[ $w.gui_box subwidget ca ] invoke"
    bind $w <Meta-Key-c> "[ $w.gui_box subwidget ca ] invoke"

    # Query frames from plugins
    set cfg [ FM_QueryPrefsFrame $w $font ]

    # Pack all.
    pack $w.gui_box -side bottom -fill x -expand yes
    pack $w.gui_l $cfg -side top -fill x -padx 5 -pady 5

    # Wait until dialog is closed.
    tixPushGrab $w
    tkwait window $w
    tixPopGrab

    # Now we can return which button was pressed.
    return $gui_Values(button)
}

#-------------------------------------------------------------------------
#
# gui_FontPreferences
#
#-------------------------------------------------------------------------

# Display preferences dialog until input is valid or the user cancels it.
# Parameters:   none
# Return value: none

proc gui_FontPreferences { } {
    global gui_Widgets
    set item [ $gui_Widgets(FontsTreeList) info selection ]
    set font [ gui_Name2String [ lindex [ split $item . ] 1 ] ]

    while { [ gui_ShowPrefsDialog $font ] } {
	# User confirmed dialog.

	# Update alias tree, it may have changed.
	gui_SetAliasTree $font

	# Check if input is valid.
	if { ! [ gui_ErrorMessage [ FM_OnPrefsClosed ] 1 ] } then {
	    # Input is valid.
	    return
	}
	# There were errors. Try again.
    }
    # Cancelled.
}

#-------------------------------------------------------------------------
#
# gui_PreviewFont
#
#-------------------------------------------------------------------------

# Show preview image of the selected font.
# Parameters:   none
# Return value: none

proc gui_PreviewFont { } {
    global gui_Widgets gui_Images
    set fontname [ gui_Name2String [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 1 ] ]
    if { ! [ string length $fontname ] } then {
	gui_ErrorMessage "No font selected.\nYou've found a bug."
	return
    }
    set img gui_Images(Preview)
    if [ gui_ErrorMessage [ FM_CreatePreviewImage $fontname $img ] ] then {
	# Could not create image.
	return
    }
    gui_ShowMessage "Preview of `$fontname':" Preview
    label $gui_Widgets(MessageWindow).preview -image $img
    pack $gui_Widgets(MessageWindow).preview -side top -padx 20
}

#-------------------------------------------------------------------------
#
# gui_ShowContextMenu
#
#-------------------------------------------------------------------------

# Create and show a context menu.
# Parameters:   group (string): the name of the selected group.
#               font (string): the name of the selected font or "".
#               aliasentry (string): the entry path of the selected alias tree
#                    entry, or "" if the user clicked on a font/group.
# Return value: none

proc gui_ShowContextMenu { group font aliasentry } {
    global gui_Widgets
    set menu $gui_Widgets(ContextMenu)
    set submenu $menu.moveto

    # Destroy old context menu and create a new one.
    catch { destroy $menu }
    menu $menu -tearoff 0

    # Create the menu entries.
    if [ string length $aliasentry ] then {
	# User clicked in the alias tree.

	$menu add command -label "New Alias..." -command gui_AddAlias
	if [ string length [ lindex [ split $aliasentry . ] 2 ] ] then {
	    # An alias is selected.
	    $menu add separator
	    $menu add command -label "Rename..." -command gui_RenameAlias
	    $menu add command -label "Delete" -command gui_DeleteAlias
	}
    } elseif [ string length $font ] then {
	# The user clicked on a font.

	# Create moveto submenu.
	menu $submenu -tearoff 0
	foreach grp [ lsort [ FM_QueryGroups ] ] {
	    if [ string compare $group $grp ] then {
		set state normal
	    } else {
		set state disabled
	    }

	    # Add group to submenu.
	    $submenu add command -label $grp -state $state \
		    -command "$gui_Widgets(MoveToBox) configure -value $grp ; gui_MoveFont"
	}

	# Create menu entries for a font.
	$menu add command -label "Properties..." -command gui_FontPreferences
	$menu add command -label "View Font" -command gui_PreviewFont
	$menu add separator
	$menu add cascade -label "Move To" -menu $submenu
	$menu add command -label "Delete" -command gui_DeleteFont
	$menu add separator
	$menu add command -label "New Alias..." -command gui_AddAlias
    } else {
	# The user clicked on a group's item.

	# Create menu entries for a group.
	$menu add command -label "Add Font..." -command gui_AddFont
	$menu add separator
	$menu add command -label "Rename..." -command gui_RenameGroup
	$menu add command -label "Delete" -command gui_DeleteGroup
    }

    # Show menu.
    tk_popup $menu [ winfo pointerx . ] [ winfo pointery . ]
}

#-------------------------------------------------------------------------
#
# gui_OnAddAliasInputChanged
#
#-------------------------------------------------------------------------

# Update state of OK-button of "Add Alias" dialog on user input.
# Parameters:   none
# Return value: none

proc gui_OnAddAliasInputChanged { } {
    global gui_Values gui_Applications
    if [ string length $gui_Values(entry) ] then {
	for { set i 1 } { $i <= $gui_Applications(count) } { incr i } {
	    if { $gui_Values(app_$i) } then {
		# An application is selected and a name is given.
		$gui_Values(okbuttonpath) configure -state normal
		return
	    }
	}
    }
    # No alias name or no application selected.
    $gui_Values(okbuttonpath) configure -state disabled
}

#-------------------------------------------------------------------------
#
# gui_AddAlias
#
#-------------------------------------------------------------------------

# Add a new alias for the selected font.
# Parameters:   none
# Return value: none

proc gui_AddAlias { } {
    global gui_Widgets gui_Values gui_Applications
    set font [ gui_Name2String [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 1 ] ]
    set group [ gui_Name2String [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 0 ] ]
    set selectedapp [ lindex [ split [ $gui_Widgets(AliasTreeList) info \
	    selection ] . ] 1 ]

    # Delete values from old dialogs.
    catch { unset gui_Values }

    # -*- Create dialog window. -*-

    set w $gui_Widgets(DialogWindow)
    gui_CreateWindow $w "New Alias"

    tixLabelFrame $w.appframe -label "Applications" -labelside acrosstop \
	    -options { label.padX 5 }
    label $w.label -text "Alias Name for font `$font':"
    entry $w.entry -width 30 -textvariable gui_Values(entry)

    set c1 "set gui_Values(button)"
    set c2 "; destroy $w"
    set gui_Values(button) 0
    tixButtonBox $w.box -orientation horizontal -relief flat
    $w.box add ok -text OK -underline 0 -command "$c1 1 $c2" -width 7 \
	    -state disabled
    set gui_Values(okbuttonpath) [ $w.box subwidget ok ]
    $w.box add ca -text Cancel -underline 0 -command "$c1 0 $c2" -width 7

    pack $w.appframe -padx 2m -pady 1m -side top -expand yes -fill both
    pack $w.box $w.entry $w.label -padx 2m -pady 1m -side bottom -fill x

    set f [ $w.appframe subwidget frame ]

    # Compute number of columns for checkbuttons
    if { $gui_Applications(count) > 15 } then {
	set columns 4
    } elseif { $gui_Applications(count) > 4 } then {
	set columns 3
    } elseif { $gui_Applications(count) == 4 } then {
	set columns 2
    } else {
	set columns $gui_Applications(count)
    }
    for { set i 0 } { $i < $columns } { incr i } {
	frame $f.$i
	pack $f.$i -side left -fill both -expand yes
    }

    # Create checkbuttons.
    for { set i 1 } { $i <= $gui_Applications(count) } { incr i } {
	set col [ expr ( $i - 1 ) % $columns ]

	# Preselect application which is selected in alias tree.
	if { ! [ string compare [ lindex $gui_Applications($i) 0 ] \
		$selectedapp ] } then {
	    set gui_Values(app_$i) 1
	} else {
	    set gui_Values(app_$i) 0
	}

	checkbutton $f.$col.$i -anchor w -variable gui_Values(app_$i) \
		-text [ lindex $gui_Applications($i) 1 ] \
		-command gui_OnAddAliasInputChanged

	if { [ $gui_Widgets(AliasTreeList) info exist \
		.[ lindex $gui_Applications($i) 0 ] ] && \
		! [ FM_QueryMAS [ lindex $gui_Applications($i) 0 ] ] } then {
	    # Can't add alias for this application.
	    $f.$col.$i configure -state disabled
	}

	pack $f.$col.$i -side top -fill x
    }

    # Key bindings
    bind $w <Key> gui_OnAddAliasInputChanged
    bind $w <Control-Key-o> "[ $w.box subwidget ok ] invoke"
    bind $w <Meta-Key-o> "[ $w.box subwidget ok ] invoke"
    bind $w <Key-Return> "[ $w.box subwidget ok ] invoke"
    bind $w <Control-Key-c> "[ $w.box subwidget ca ] invoke"
    bind $w <Meta-Key-c> "[ $w.box subwidget ca ] invoke"
    bind $w <Key-Escape> "[ $w.box subwidget ca ] invoke"

    # -*- Show dialog. -*-

    focus $w.entry
    tixPushGrab $w
    tkwait window $w
    tixPopGrab

    # -*- Process user input -*-

    if { ! $gui_Values(button) } then {
	# Cancelled.
	return
    }
    for { set i 1 } { $i <= $gui_Applications(count) } { incr i } {
	if { $gui_Values(app_$i) } then {
	    lappend apps [ lindex $gui_Applications($i) 0 ]
	}
    }

    # Try to add the alias.
    gui_ErrorMessage [ FM_AddAlias $font $apps $gui_Values(entry) ]

    # Update trees.
    gui_SetAliasTree $font
    foreach app $apps {
	gui_UpdateCheckbox $font $group $app
	gui_UpdateCheckbox "" $group $app
    }
}

#-------------------------------------------------------------------------
#
# gui_ShowDeleteDialog
#
#-------------------------------------------------------------------------

# Show a dialog which asks the user for confirmation.
# Parameters:   font (string): the name of the font that will be deleted.
#               multi (int): 0=only one font, -1=last font, 1=multiple fonts.
# Return value: "YES", "ALL", "NO", or "NONE" depending on the button the
#               user presses.

proc gui_ShowDeleteDialog { font multi } {
    global gui_Widgets gui_Values gui_Images
    # Delete values from old dialogs.
    catch { unset gui_Values }
    # -*- Create dialog window. -*-
    set w $gui_Widgets(DialogWindow)
    gui_CreateWindow $w "Delete font"
    frame $w.f
    label $w.f.label -text "About to delete font `$font'.\nAre you sure?" -justify left
    label $w.f.image -image $gui_Images(warning)
    pack $w.f -side top
    pack $w.f.image $w.f.label -side left -pady 5m -padx 1m -expand yes -fill both
    set c1 "set gui_Values(button)"
    set c2 "; destroy $w"
    set gui_Values(button) "NONE"
    tixButtonBox $w.box -orientation horizontal -relief flat
    $w.box add yes -text Yes -underline 0 -command "$c1 YES $c2" -width 7
    if $multi then {
	if { $multi == -1 } then {
	    set state disabled
	} else {
	    set state normal
	}
	$w.box add all -text "Yes, all" -underline 5 -command "$c1 ALL $c2" \
		-state $state -width 7
    }
    $w.box add no -text No -underline 0 -command "$c1 NO $c2" -width 7
    if $multi then {
	$w.box add none -text "Cancel" -underline 0 -command "$c1 NONE $c2" \
		-state $state -width 7
    }
    pack $w.box -side bottom -fill x -expand yes
    # Key bindings
    bind $w <Key-y> "[ $w.box subwidget yes ] invoke"
    bind $w <Key-Return> "[ $w.box subwidget yes ] invoke"
    bind $w <Key-n> "[ $w.box subwidget no ] invoke"
    if $multi then {
	bind $w <Key-a> "[ $w.box subwidget all ] invoke"
	bind $w <Key-c> "[ $w.box subwidget none ] invoke"
	bind $w <Key-Escape> "[ $w.box subwidget none ] invoke"
    } else {
    bind $w <Key-Escape> "[ $w.box subwidget no ] invoke"
    }
    # -*- Show dialog. -*-
    tixPushGrab $w
    tkwait window $w
    tixPopGrab
    # -*- Process user input -*-
    return $gui_Values(button)
}

#-------------------------------------------------------------------------
#
# gui_ShowRenameDialog
#
#-------------------------------------------------------------------------

# Show a dialog where the user can edit a name.
# Parameters:   title (string): the string to show as window title.
#               label (string): the label string to show above the entry.
#               oldname (string): the old name to be edited.
# Return value: the new name, or "" if the user pressed "cancel".

proc gui_ShowRenameDialog { title label oldname } {
    global gui_Widgets gui_Values
    # Delete values from old dialogs.
    catch { unset gui_Values }
    # -*- Create dialog window. -*-
    set w $gui_Widgets(DialogWindow)
    gui_CreateWindow $w $title
    label $w.label -text $label
    entry $w.entry -width 40 -textvariable gui_Values(entry)
    set gui_Values(entry) $oldname
    pack $w.label $w.entry -pady 1m -padx 2m -side top -expand yes -fill both
    set c1 "set gui_Values(button)"
    set c2 "; destroy $w"
    set gui_Values(button) 0
    tixButtonBox $w.box -orientation horizontal -relief flat
    $w.box add ok -text OK -underline 0 -command "$c1 1 $c2" -width 7
    $w.box add cancel -text Cancel -underline 0 -command "$c1 0 $c2" -width 7
    pack $w.box -side bottom -fill x -expand yes
    # Key bindings
    bind $w <Control-Key-o> "[ $w.box subwidget ok ] invoke"
    bind $w <Meta-Key-o> "[ $w.box subwidget ok ] invoke"
    bind $w <Key-Return> "[ $w.box subwidget ok ] invoke"
    bind $w <Control-Key-c> "[ $w.box subwidget cancel ] invoke"
    bind $w <Meta-Key-c> "[ $w.box subwidget cancel ] invoke"
    bind $w <Key-Escape> "[ $w.box subwidget cancel ] invoke"
    # -*- Show dialog. -*-
    focus $w.entry
    tixPushGrab set $w
    tkwait window $w
    tixPopGrab
    # -*- Process user input -*-
    if { ! $gui_Values(button) } then {
	# Cancelled.
	return ""
    }
    return $gui_Values(entry)
}


#-------------------------------------------------------------------------
#
# gui_RenameAlias
#
#-------------------------------------------------------------------------

# Rename the selected alias.
# Parameters:   none
# Return value: none

proc gui_RenameAlias { } {
    global gui_Widgets gui_Images
    set hlist $gui_Widgets(AliasTreeList)
    set aliasentry [ $hlist info selection ]
    set font [ gui_Name2String [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 1 ] ]
    set alias [ gui_Name2String [ lindex [ split $aliasentry . ] 2 ] ]
    set app [ lindex [ split $aliasentry . ] 1 ]
    if { ! [ string length $alias ] } then {
	# No alias selected.
	gui_ErrorMessage "No alias selected. That's a bug!"
	return
    }
    # Show dialog.
    set newname [ gui_ShowRenameDialog "Rename Alias" "Alias Name:" $alias ]
    if { ! [ string length $newname ] } then {
	# Cancelled. Do nothing.
	return
    }
    if { ! [ string compare $alias $newname ] } then {
	# Name has not changed. nothing to do.
	return
    }
    # Try to rename the alias.
    if [ gui_ErrorMessage [ FM_RenameAlias $app $font $alias $newname ] ] then {
	# There were errors.
	return
    }
    # Replace the entry in the alias tree with a new one.
# We could redraw the tree instead.
    set newentry .$app.[ gui_String2Name $newname ]
    $hlist add $newentry -text $newname -image $gui_Images(alias) \
	    -before $aliasentry
    $hlist delete entry $aliasentry
    $hlist selection clear
    $hlist selection set $newentry
}

#-------------------------------------------------------------------------
#
# gui_DeleteAlias
#
#-------------------------------------------------------------------------

# Delete the selected alias.
# Parameters:   none
# Return value: none

proc gui_DeleteAlias { } {
    global gui_Widgets
    set hlist $gui_Widgets(AliasTreeList)
    set aliasentry [ $hlist info selection ]
    set font [ gui_Name2String [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 1 ] ]
    set group [ gui_Name2String [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 0 ] ]
    set alias [ gui_Name2String [ lindex [ split $aliasentry . ] 2 ] ]
    set app [ lindex [ split $aliasentry . ] 1 ]
    if { ! [ string length $alias ] } then {
	# No alias selected.
	gui_ErrorMessage "No alias selected. That's a bug!"
	return
    }
    # Try to remove the alias
    if [ gui_ErrorMessage [ FM_RemoveAlias $app $font $alias ] ] then {
	# There were errors. (??)
	return
    }
    # Update alias tree.
    set newselection [ $hlist info next $aliasentry ]
    if [ $hlist info exists $newselection ] then { 
	$hlist selection clear
	$hlist selection set $newselection
	$hlist anchor set $newselection
	$hlist see $newselection
    }
    gui_OnAliasSelected $newselection
    gui_SetAliasTree $font
    gui_UpdateCheckbox $font $group $app
    gui_UpdateCheckbox "" $group $app
}

#-------------------------------------------------------------------------
#
# gui_RenameGroup
#
#-------------------------------------------------------------------------

# Rename the selected group.
# Parameters:   none
# Return value: none

proc gui_RenameGroup { } {
    global gui_Widgets
    set group [ gui_Name2String [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 0 ] ]
    if { ! [ string length $group ] } then {
	# No group selected.
	gui_ErrorMessage "No group selected. That's a bug!"
	return
    }
    # Show dialog.
    set newname [ gui_ShowRenameDialog "Rename Group" "Group Name:" $group ]
    if { ! [ string length $newname ] } then {
	# Cancelled. Do nothing.
	return
    }
    if { ! [ string compare $group $newname ] } then {
	# Name has not changed. nothing to do.
	return
    }
    # Try to rename the group.
    if [ gui_ErrorMessage [ FM_RenameGroup $group $newname ] ] then {
	# There were errors.
	return
    }
    # Create new fonts tree.
    gui_SetTrees
}

#-------------------------------------------------------------------------
#
# gui_NewGroup
#
#-------------------------------------------------------------------------

# Create a new group.
# Parameters:   none
# Return value: none

proc gui_NewGroup { } {
    global gui_Widgets gui_Images gui_Applications
    # Show dialog.
    set groupname [ gui_ShowRenameDialog "New Group" "Group Name:" "" ]
    if { ! [ string length $groupname ] } then {
	# Cancelled. Do nothing.
	return
    }
    # Try to create the group.
    if [ gui_ErrorMessage [ FM_NewGroup $groupname ] ] then {
	# There were errors.
	return
    }
    # Update fonts tree.
    set hlist $gui_Widgets(FontsTreeList)
    gui_AddFontGroupItem $groupname ""
    set newitem [ gui_String2Name $groupname ]
    gui_SetMoveToList
    $hlist selection clear
    $hlist selection set $newitem
    $hlist anchor set $newitem
    $hlist see $newitem
    gui_OnItemSelected $newitem
    incr gui_Widgets(GroupCount)
    $gui_Widgets(StatusBarLabel) configure -text ""
}

#-------------------------------------------------------------------------
#
# gui_DeleteGroup
#
#-------------------------------------------------------------------------

# Delete the selected group. Ask user for confirmation when deleting fonts.
# Parameters:   none
# Return value: none

proc gui_DeleteGroup { } {
    global gui_Widgets
    set hlist $gui_Widgets(FontsTreeList)
    set group [ gui_Name2String [ lindex [ split \
	    [ $hlist info selection ] . ] 0 ] ]
    if { ! [ string length $group ] } then {
	# No group selected.
	gui_ErrorMessage "No group selected. That's a bug!"
	return
    }
    set retval ""
    set err ""
    set errors ""
    foreach font [ lsort [ FM_QueryFonts $group ] ] {
	set fontname [ lindex $font 0 ]
	if [ string compare $retval ALL ] then {
	    # Show Message if there were errors.
	    gui_ErrorMessage $err
	    set errors ""
	    # Ask user for confirmation.
	    set retval [ gui_ShowDeleteDialog $fontname 1 ]
# 2nd parameter should be -1 for last font.
	    if { ! [ string compare $retval NONE ] } then {
		# don't continue.
		return
	    }
	}
	if [ string compare $retval NO ] then {
	    # "Yes" or "Yes, all" selected.
	    set err [ FM_RemoveFont $fontname $group ]
	    if [ string length $err ] {
		lappend errors $err
	    } else {
		$hlist delete entry [ gui_String2Name $group ].[ gui_String2Name $fontname ]
	    }
	}
    }
    if { ! [ llength [ FM_QueryFonts $group ] ] } then {
	# Group is empty. Try to delete it.
	set err [ FM_RemoveGroup $group ]
	if [ string length $err ] {
	    lappend errors $err
	} else {
	    $hlist delete entry [ gui_String2Name $group ]
	    gui_SetMoveToList
	    incr gui_Widgets(GroupCount) -1
	    $gui_Widgets(StatusBarLabel) configure -text ""
	    gui_OnItemSelected ""
	}
    }
    # Show eventually error messages.
    set err [ join $errors "\n" ]
    gui_ErrorMessage $err
}

#-------------------------------------------------------------------------
#
# gui_DeleteFont
#
#-------------------------------------------------------------------------

# Delete the selected font. Ask user for confirmation first.
# Parameters:   none
# Return value: none

proc gui_DeleteFont { } {
    global gui_Widgets
    set hlist $gui_Widgets(FontsTreeList)
    set item [ $hlist info selection ]
    set group [ gui_Name2String [ lindex [ split $item . ] 0 ] ]
    set font [ gui_Name2String [ lindex [ split $item . ] 1 ] ]
    if { ! [ string length $font ] } then {
	# No font selected.
	gui_ErrorMessage "No font selected. That's a bug!"
	return
    }
    # Ask user for confirmation.
    if [ string compare [ gui_ShowDeleteDialog $font 0 ] YES ] then {
	# Dont't delete font.
	return
    }
    # Try to remove the font.
    if [ gui_ErrorMessage [ FM_RemoveFont $font $group ] ] then {
	# There where Errors. Font is not removed.
	return
    }
    # Update fonts tree.
    set newselection [ $hlist info next $item ]
    if [ $hlist info exists $newselection ] then { 
	$hlist selection clear
	$hlist selection set $newselection
	$hlist anchor set $newselection
	$hlist see $newselection
    }
    gui_OnItemSelected $newselection
    $hlist delete entry $item
    $gui_Widgets(FontsTree) autosetmode
    incr gui_Widgets(FontCount) -1
    $gui_Widgets(StatusBarLabel) configure -text ""
}

#-------------------------------------------------------------------------
#
# gui_MoveFont
#
#-------------------------------------------------------------------------

# Move the selected font to the group selected as destination.
# Parameters:   none
# Return value: none

proc gui_MoveFont { } {
    global gui_Widgets gui_Applications gui_Images
    set hlist $gui_Widgets(FontsTreeList)
    set item [ $hlist info selection ]
    set group [ gui_Name2String [ lindex [ split $item . ] 0 ] ]
    set font [ gui_Name2String [ lindex [ split $item . ] 1 ] ]
    set newgroup [ $gui_Widgets(MoveToBox) cget -value ]
    if { ! [ string length $font ] } then {
	# No font selected.
	gui_ErrorMessage "No font selected. That's a bug!"
	return
    }
    if { ! [ string length $newgroup ] } then {
	# No destination selected.
	gui_ErrorMessage "No destination selected. That's a bug!"
	return
    }
    if [ gui_ErrorMessage [ FM_MoveFont $font $group $newgroup ] ] then {
	# There werer Errors. Font is not moved.
	return
    }
    # Update fonts tree.
    set newselection [ $hlist info next $item ]
    if [ $hlist info exists $newselection ] then { 
	$hlist selection clear
	$hlist selection set $newselection
	$hlist anchor set $newselection
	$hlist see $newselection
    }
    gui_OnItemSelected $newselection
    # Delete old entry.
    $hlist delete entry $item
    # Add new entry.
    gui_AddFontGroupItem $newgroup $font
    for { set i 1 } { $i <= $gui_Applications(count) } { incr i } {
	set app [ lindex $gui_Applications($i) 0 ]
	gui_UpdateCheckbox {} $group $app
	gui_UpdateCheckbox {} $newgroup $app
    }
    $gui_Widgets(FontsTree) autosetmode
}

#-------------------------------------------------------------------------
#
# gui_UpdateCheckbox
#
#-------------------------------------------------------------------------

# Draw a new checkbox image for a font/group for an application.
# Parameters:   font (string): the name of the font, or "" if it's a group.
#               group (string): the name of the group (the font is in).
#               app (string): the application's prefix.
# Return value: none

proc gui_UpdateCheckbox { font group app } {
    global gui_Widgets gui_Applications gui_Images
    # Get column number for $app.
    set idx 1
    while { [ string compare [ lindex $gui_Applications($idx) 0 ] $app ] } {
	incr idx
    }
    # Get the right image and the item's path name.
    if [ string length $font ] then {
	# It's a font.
	set img [ FM_QueryFontActive $font $app ]
	set item [ gui_String2Name $group ].[ gui_String2Name $font ]
    } else {
	# It's a group.
	set img [ FM_QueryGroupActive $group $app ]
	set item [ gui_String2Name $group ]
    }
    $gui_Widgets(FontsTreeList) item configure $item $idx -image $gui_Images($img)
}

#-------------------------------------------------------------------------
#
# gui_(De)activate
#
#-------------------------------------------------------------------------

# Activates or deactivates a font or group for an application.
# Parameters:   font (string): the name of the font which should be activated,
#                              or "" if a group is selected.
#               group (string): the name of the group (the font is in).
#               app (string): the application's prefix.
# Return value: none

proc gui_(De)activate { font group app } {
    if [ string length $font ] {
	# It's a font.
	if [ string compare [ FM_QueryFontActive $font $app ] "ON" ] then {
	    if [ string compare [ FM_QueryFontActive $font $app ] "N/A" ] then {
		# Font is inactive. Activate.
		gui_ErrorMessage [ FM_ActivateFont $font $app ]
	    } else {
		# Font can't be activated.
		return
	    }
	} else {
	    # Font is active. Deactivate.
		gui_ErrorMessage [ FM_DeactivateFont $font $app ]
	}
	gui_UpdateCheckbox $font $group $app
	gui_UpdateCheckbox "" $group $app
    } else {
	# It's a group.
	if [ string compare [ FM_QueryGroupActive $group $app ] "ON" ] then {
	    if [ string compare [ FM_QueryGroupActive $group $app ] "N/A" ] then {
		# Group is (partly) inactive. Activate.
		set result [ FM_ActivateGroup $group $app ]
		gui_ErrorMessage [ lindex $result 0 ]
	    } else {
		# Group can't be activated.
		return
	    }
	} else {
	    # Group is active. Deactivate.
	    set result [ FM_DeactivateGroup $group $app ]
	    gui_ErrorMessage [ lindex $result 0 ]
	}
	foreach font [ lindex $result 1 ] {
	    gui_UpdateCheckbox $font $group $app
	}
	gui_UpdateCheckbox "" $group $app
    }
}

#-------------------------------------------------------------------------
#
# gui_OnFontsTreeClicked
#
#-------------------------------------------------------------------------

# Handles clicks on the checkboxes of the fonts tree.
# Parameters:   button (integer): the number of the button that was pressed.
#               x_pos (integer): the horizontal position of the mouse pointer.
#               y_pos (integer): the vertical position of the mouse pointer.
# Return value: none

proc gui_OnFontsTreeClicked { button x_pos y_pos } {
    global gui_Widgets gui_Applications
    set hlist $gui_Widgets(FontsTreeList)

    # Process all columns until x-position has been found
    set col_pos 0
    for { set col 0 } { $col < [ $hlist cget -columns ] } { incr col } {
	# Next column
	incr col_pos [ $hlist column width $col ]
	# Position found?
	if { $x_pos < $col_pos } { break }
    }
    if { $col > $gui_Applications(count) } { return }
    # $x_pos is in column $col.

    # Determine selected item.
    set selitem [ $hlist nearest $y_pos ]
    set selected [ split $selitem . ]

    # Process events depending on which button was pressed.
    if { $button == 1 } then {
	if { $col == 0 } then { return }
	set group [ gui_Name2String [ lindex $selected 0 ] ]
	set font [ gui_Name2String [ lindex $selected 1 ] ]

	# (De)activate font or group.
	gui_(De)activate $font $group [ lindex $gui_Applications($col) 0 ]

	# Alias tree may be updated or not. Redraw it. %-(
	gui_SetAliasTree $font
    } elseif { $button == 2 } then {
	set group [ gui_Name2String [ lindex $selected 0 ] ]
	if { ! [ string length $group ] } then { return }

	# Use group of B2-clicked item as destination group.
	$gui_Widgets(MoveToBox) configure -value $group
    } elseif { $button == 3 } then {
	# Select the item the user clicked on.
	$hlist selection clear
	$hlist selection set $selitem
	$hlist anchor set $selitem

	# Update other widgets.
	gui_OnItemSelected $selitem

	set group [ gui_Name2String [ lindex $selected 0 ] ]
	set font [ gui_Name2String [ lindex $selected 1 ] ]

	# Show a contex menu.
	gui_ShowContextMenu $group $font ""
    }
}

#-------------------------------------------------------------------------
#
# gui_OnAliasTreeClicked
#
#-------------------------------------------------------------------------

# Handles clicks on the alias tree, currently only used to show context menu.
# Parameters:   button (integer): the number of the button that was pressed.
#               x_pos (integer): the horizontal position of the mouse pointer.
#               y_pos (integer): the vertical position of the mouse pointer.
# Return value: none

proc gui_OnAliasTreeClicked { button x_pos y_pos } {
    global gui_Widgets
    set hlist $gui_Widgets(AliasTreeList)

    # Determine the selected item.
    set selected [ $hlist nearest $y_pos ]

    # Process events depending on which button was pressed.
    if { $button == 3 } then {
	# Select the item the user clicked on.
	$hlist selection clear
	$hlist selection set $selected
	$hlist anchor set $selected

	# Update other widgets.
	gui_OnAliasSelected $selected

	set fontitem [ $gui_Widgets(FontsTreeList) info selection ]
	set group [ gui_Name2String [ lindex $fontitem 0 ] ]
	set font [ gui_Name2String [ lindex $fontitem 1 ] ]

	# Show a contex menu.
	gui_ShowContextMenu $group $font $selected
    }
}

#-------------------------------------------------------------------------
#
# gui_SetAliasTree
#
#-------------------------------------------------------------------------

# Called when a font or group has been selected. Displays alias tree.
# Parameter:    font (string): the name of the font for which the aliases
#                    should be displayed, or "" if a group has been selected.
# Return value: none

proc gui_SetAliasTree { font } {
    global gui_Widgets gui_Applications gui_Images FM_Fonts
    set hlist $gui_Widgets(AliasTreeList)
    $hlist delete all
    if [ string length $font ] {
	# A font is selected.
	set aliasimg $gui_Images(alias)
	$hlist add . -text $font -image \
		[ FM_QueryIcon [ lindex [ FM_QueryFontInfo $font ] 1 ] ]
	set i 1
	while { $i <= $gui_Applications(count) } {
	    set appid [ lindex $gui_Applications($i) 0 ]
	    set aliaslist [ FM_QueryAliases $font $appid ]
	    if [ string length $aliaslist ] then {
		$hlist add .$appid -text [ lindex $gui_Applications($i) 1 ] \
			-image [ lindex $gui_Applications($i) 2 ]
		foreach alias $aliaslist {
		    set aliasid [ gui_String2Name $alias ]
		    $hlist add .$appid.$aliasid -text $alias -image $aliasimg
		}
	    }
	    incr i
	}
	$gui_Widgets(AddAliasButton) configure -state normal
	$gui_Widgets(AliasTree) autosetmode
    } else {
	# No font selected.
	$gui_Widgets(AddAliasButton) configure -state disabled
    }
    $gui_Widgets(RenameAliasButton) configure -state disabled
    $gui_Widgets(DeleteAliasButton) configure -state disabled
#blubber
# We have to do nothing if alias tree is up to date!!
}

#-------------------------------------------------------------------------
#
# gui_OnItemSelected
#
#-------------------------------------------------------------------------

# Called when a font or group has been selected.
# Parameter:    item (string): the entry path of the selected item.
# Return value: none

proc gui_OnItemSelected { item } {
    global gui_Widgets
    set group [ gui_Name2String [ lindex [ split $item . ] 0 ] ]
    set font [ gui_Name2String [ lindex [ split $item . ] 1 ] ]
    # Sets alias tree and buttons.
    gui_SetAliasTree $font
    # Set button states.
    if [ string length $group ] then {
	if [ string length $font ] then {
	    # font selected.
	    $gui_Widgets(DeleteButton) configure -state normal \
		    -text "Delete Font" -command gui_DeleteFont
	    $gui_Widgets(RenameGroupButton) configure -state disabled
	    if [ FM_QueryHasPreview $font ] then {
		$gui_Widgets(PreviewFontButton) configure -state normal
	    } else {
		$gui_Widgets(PreviewFontButton) configure -state disabled
	    }
	    $gui_Widgets(PreferencesButton) configure -state normal
	    if [ string length [ $gui_Widgets(MoveToBox) cget -value ] ] then {
		$gui_Widgets(MoveFontButton) configure -state normal
	    } else {
		$gui_Widgets(MoveFontButton) configure -state disabled
	    }
	} else {
	    # group selected.
	    $gui_Widgets(PreviewFontButton) configure -state disabled
	    $gui_Widgets(PreferencesButton) configure -state disabled
	    $gui_Widgets(DeleteButton) configure -state normal \
		    -text "Delete Group" -command gui_DeleteGroup
	    $gui_Widgets(RenameGroupButton) configure -state normal
	    $gui_Widgets(MoveFontButton) configure -state disabled
	}
    } else {
	# no item selected.
	$gui_Widgets(PreviewFontButton) configure -state disabled
	$gui_Widgets(PreferencesButton) configure -state disabled
	$gui_Widgets(RenameGroupButton) configure -state disabled
	$gui_Widgets(DeleteButton) configure -state disabled -text "Delete"
	$gui_Widgets(MoveFontButton) configure -state disabled
    }
}

#-------------------------------------------------------------------------
#
# gui_OnAliasSelected
#
#-------------------------------------------------------------------------

# Called when an alias has been selected.
# Parameter:    item (string): the entry path of the selected item.
# Return value: none

proc gui_OnAliasSelected { item } {
    global gui_Widgets
    set alias [ lindex [ split $item . ] 2 ]
    # Set button states.
    if [ string length $alias ] then {
	# alias selected.
	$gui_Widgets(RenameAliasButton) configure -state normal
	$gui_Widgets(DeleteAliasButton) configure -state normal
    } else {
	# no alias selected.
	$gui_Widgets(RenameAliasButton) configure -state disabled
	$gui_Widgets(DeleteAliasButton) configure -state disabled
    }
}

#-------------------------------------------------------------------------
#
# gui_OnDestGroupSelected
#
#-------------------------------------------------------------------------

# Called when a destination group has been selected.
# Parameter:    value (string): the name of the selected group.
# Return value: none

proc gui_OnDestGroupSelected { value } {
    global gui_Widgets
    if { ! [ string length [ lindex [ split \
	    [ $gui_Widgets(FontsTreeList) info selection ] . ] 1 ] ] } then {
	# No font selected. Moving not possible.
	set value ""
    }
    # Set button states.
    if [ string length $value ] then {
	$gui_Widgets(MoveFontButton) configure -state normal
    } else {
	$gui_Widgets(MoveFontButton) configure -state disabled
    }
}

#-------------------------------------------------------------------------
#
# gui_OnFontsTreeResized
#
#-------------------------------------------------------------------------

# Sets column widths of fonts tree.
# Parameters:   none
# Return value: none

proc gui_OnFontsTreeResized { } {
    global gui_Widgets gui_Images gui_Applications
    set hlist $gui_Widgets(FontsTreeList)
    set restwidth [ winfo width $hlist ]
    # Compute space needed for checkboxes.
    set ckwidth 0
    foreach img { ON OFF N/A SOME } {
	if [ catch { set imgwidth [ image width $gui_Images($img) ] } ] then {
	    set imgwidth 0
	}
	if { $imgwidth > $ckwidth } then { set chwidth $imgwidth }
    }
    set i 1
    while { $i <= $gui_Applications(count) } {
	# Compute space needed for current application.
	if [ catch { set imgwidth [ image width [ lindex \
		$gui_Applications($i) 2 ] ] } ] then { set imgwidth 0 }
	if { $imgwidth < $ckwidth } then { set imgwidth $ckwidth }
	# Add space for border. 
	incr imgwidth 9
	# Set column width.
	$hlist column width $i $imgwidth
	# Decrease remaining space
	incr restwidth -$imgwidth
	# Next application.
	incr i
    }
    if { $restwidth <= 50 } then { set restwidth "" }
    $hlist column width 0 $restwidth
}

#-------------------------------------------------------------------------
#
# gui_OnStatusBarChanged
#
#-------------------------------------------------------------------------

# Called when text of statusbar has changed.
# Parameters:   none
# Return value: none

proc gui_OnStatusBarChanged { } {
    global gui_Widgets
    set label $gui_Widgets(StatusBarLabel)
    set text [ $label cget -text ]
    # A change of the label's text will call the procedure again.
    # Be careful that you don't create loops here!
    if { ! [ string length $text ] } then {
	$label configure -text "$gui_Widgets(FontCount) fonts in $gui_Widgets(GroupCount) groups."
    }
}

#-------------------------------------------------------------------------
#
# gui_SetMoveToList
#
#-------------------------------------------------------------------------

# Sets entries of moveto list.
# Parameters:   none
# Return value: none

proc gui_SetMoveToList { } {
    global gui_Widgets
    set movetolist $gui_Widgets(MoveToBoxList)
    set oldvalue [ $gui_Widgets(MoveToBox) cget -value ]
    # Delete old moveto list.
    $movetolist delete 0 end
    $gui_Widgets(MoveToBox) configure -value ""
    foreach group [ lsort [ FM_QueryGroups ] ] {
	# Add group to moveto list.
	$movetolist insert end $group
	if { ! [ string compare $oldvalue $group ] } then {
	    $gui_Widgets(MoveToBox) configure -value $group
	}
    }
}

#-------------------------------------------------------------------------
#
# gui_SetTrees
#
#-------------------------------------------------------------------------

# Sets values of fonts tree, aliases tree, moveto list.
# Parameters:   none
# Return value: none

proc gui_SetTrees { } {
    global gui_Widgets gui_Applications gui_Images
    set fontstree $gui_Widgets(FontsTree)
    set movetolist $gui_Widgets(MoveToBoxList)
    # Create new fonts tree widget with right number of columns.
    destroy $fontstree
    tixTree $fontstree -options "hlist.header true \
	    hlist.columns [ expr $gui_Applications(count) + 1 ]" \
	    -browsecmd gui_OnItemSelected
    pack $fontstree -side top -expand 1 -fill both -padx 2 -pady 2
    set fontstreelist [ $fontstree subwidget hlist ]
    # Set header of subwidget hlist
    set gui_Widgets(FontsTreeList) $fontstreelist
    $fontstreelist header create 0 -itemtype text -text Fonts:
    for { set i 1 } { $i <= $gui_Applications(count) } { incr i } {
	##$fontstreelist header create $i -itemtype image \
	##	-image [ lindex $gui_Applications($i) 2 ]
	label $fontstreelist.appimg$i -image [ lindex $gui_Applications($i) 2 ]
	$gui_Widgets(BalloonWindow) bind $fontstreelist.appimg$i \
		-balloonmsg [ lindex $gui_Applications($i) 1 ] \
		-statusmsg "This column shows which fonts are available in [ lindex $gui_Applications($i) 1 ]."
	$fontstreelist header create $i -itemtype window \
		-window $fontstreelist.appimg$i
    }
    # Delete old moveto list.
    $movetolist delete 0 end
    $gui_Widgets(MoveToBox) configure -value ""
    set gcount 0
    set fcount 0
    # -*- Process all Groups in alphabetical order. -*-
    foreach group [ lsort [ FM_QueryGroups ] ] {
	# Add group to moveto list.
	$movetolist insert end $group
	# Add group to tree
	set groupid [ gui_String2Name $group ]
	$fontstreelist add $groupid -text $group -image $gui_Images(folder)
	set i 1
	while { $i <= $gui_Applications(count) } {
	    set img [ FM_QueryGroupActive $group \
		    [ lindex $gui_Applications($i) 0 ] ]
	    $fontstreelist item create $groupid $i -itemtype image \
		    -image $gui_Images($img)
	    incr i
	}
	# -- Process Fonts in alphabetical order. --
	foreach fontl [ lsort [ FM_QueryFonts $group ] ] {
	    set font [ lindex $fontl 0 ]
	    # Add font to the tree.
	    set fontid [ gui_String2Name $font ]
	    set img [ FM_QueryIcon [ lindex $fontl 1 ] ]
	    $fontstreelist add $groupid.$fontid -text $font -image $img
	    set i 1
	    while { $i <= $gui_Applications(count) } {
		set img [ FM_QueryFontActive $font \
			[ lindex $gui_Applications($i) 0 ] ]
		$fontstreelist item create $groupid.$fontid $i \
			-itemtype image -image $gui_Images($img)
		incr i
	    }
	    incr fcount
	}
	incr gcount
    }
    $fontstree autosetmode
    set gui_Widgets(FontCount) $fcount
    set gui_Widgets(GroupCount) $gcount
    $gui_Widgets(StatusBarLabel) configure -text \
	    "$fcount fonts in $gcount groups."
    bind $gui_Widgets(FontsTreeList) <Button> { 
	gui_OnFontsTreeClicked %b %x %y
    }
    $gui_Widgets(FontsTreeList) configure -sizecmd "gui_OnFontsTreeResized
        [ $gui_Widgets(FontsTreeList) cget -sizecmd ]"
    # Set button states and aliases tree.
    gui_OnItemSelected ""
#blubber
# We should look for selected items and restore the selection if possible.
}

#-------------------------------------------------------------------------
#
# gui_CMWin:AddButtonBinding
#
#-------------------------------------------------------------------------

# Adds bindings for a button.
# Parameters:   button (string): the index name of the widget in gui_Widgets.
#               key (string): the key symbol to bind to the button
#               statusmsg (string): the string to show in the status bar when
#                          the mouse is over the button.
#               balloonmsg (string, optional): string to show as ballon help.
# Return value: none

proc gui_CMWin:AddButtonBinding { button key statusmsg { balloonmsg "" } } {
    global gui_Widgets

    bind . <Control-$key> "$gui_Widgets($button) invoke"
    bind . <Meta-$key> "$gui_Widgets($button) invoke"

    if [ string length $balloonmsg ] then {
	$gui_Widgets(BalloonWindow) bind $gui_Widgets($button) \
		-statusmsg $statusmsg -balloonmsg $balloonmsg
    } else {
	$gui_Widgets(StatusOnlyBalloon) bind $gui_Widgets($button) \
		-statusmsg $statusmsg
    }
}

#-------------------------------------------------------------------------
#
# gui_CreateMainWindow:Bindings
#
#-------------------------------------------------------------------------

# Adds bindings for the widgets in the main window.
# Parameters:   none
# Return value: none

proc gui_CreateMainWindow:Bindings { } {
    global gui_Widgets
    # Key and balloon bindings for all buttons.
    gui_CMWin:AddButtonBinding ConfigButton c \
	    "Set configuration information for plugin modules."
    gui_CMWin:AddButtonBinding AboutButton b \
	    "About HAMSTER Font Manager."
    gui_CMWin:AddButtonBinding WriteConfigButton y \
	    "Save changes to disk and apply them to the supported applications."
    gui_CMWin:AddButtonBinding ExitButton x \
	    "Save changes and exit the program."
    gui_CMWin:AddButtonBinding HelpButton h \
	    "Show postscript manual with ghostview."

    gui_CMWin:AddButtonBinding AddFontButton a \
	    "Add one or more new font files to HFM."
    gui_CMWin:AddButtonBinding PreviewFontButton v \
	    "Show a preview image for the selected font."
    gui_CMWin:AddButtonBinding PreferencesButton p \
	    "Change preferences for the selected font."
    gui_CMWin:AddButtonBinding DeleteButton d \
	    "Delete the selected font or group."
    gui_CMWin:AddButtonBinding NewGroupButton n \
	    "Create a new group."
    gui_CMWin:AddButtonBinding RenameGroupButton r \
	    "Rename the selected group."
    gui_CMWin:AddButtonBinding MoveFontButton m \
	    "Move the selected font into the group showed after the `Move to' label."

    gui_CMWin:AddButtonBinding AddAliasButton w \
	    "Define a new alias name of the selected font for some applications."
    gui_CMWin:AddButtonBinding RenameAliasButton e \
	    "Change the name of the selected alias."
    gui_CMWin:AddButtonBinding DeleteAliasButton l \
	    "Delete the selected alias name."

    # Trap changes of status bar text.
    bind $gui_Widgets(StatusBarLabel) <Configure> gui_OnStatusBarChanged

    # Mouse clicks on alias tree (fonts tree is bound in gui_SetTrees).
    bind $gui_Widgets(AliasTreeList) <Button> {
	gui_OnAliasTreeClicked %b %x %y
    }
}

#-------------------------------------------------------------------------
#
# gui_CreateMainWindow
#
#-------------------------------------------------------------------------

# Creates the widgets in the main window and sets variables to acces them.
# The trees and list will be empty and must be filled after initialisation.
# Parameters:   none
# Return value: none

proc gui_CreateMainWindow { } {
    global gui_Widgets
    set buttonwidth 15

    wm title . HFM

    # Create menu bar.
    set f .menubar
    frame $f -relief raised -bd 2

    button $f.config -text "Config..." -underline 0 -relief flat -command gui_Configure
    button $f.about -text About -underline 1 -relief flat -command gui_ShowAbout
    button $f.wrcfg -text Apply -underline 4 -relief flat -command gui_WriteConfig
    button $f.exit -text Exit -underline 1 -relief flat -command gui_Quit
    button $f.help -text Help -underline 0 -relief flat -command gui_ShowHelp

    pack $f -side top -fill x
    pack $f.config $f.about $f.wrcfg $f.exit -side left
    pack $f.help -side right

    set gui_Widgets(ConfigButton) $f.config
    set gui_Widgets(AboutButton) $f.about
    set gui_Widgets(WriteConfigButton) $f.wrcfg
    set gui_Widgets(ExitButton) $f.exit
    set gui_Widgets(HelpButton) $f.help

    # Create paned window.
    tixPanedWindow .panedwin -width 500
    .panedwin add win1 -size 300 -min 245 -expand 2
    .panedwin add win2 -size 150 -min 105 -expand 1
    pack .panedwin -side top -fill both -expand 1

    set win1 [ .panedwin subwidget win1 ]
    set win2 [ .panedwin subwidget win2 ]

    # Create status bar.
    frame .statusbar -relief raised -bd 2
    label .statusbar.label -justify left -text \
	    {Initializing HAMSTER Font Manager...}
    pack .statusbar -side bottom -fill x
    pack .statusbar.label -side left
    set gui_Widgets(StatusBarLabel) .statusbar.label
    UTIL_ShowHamster -statuswidget .statusbar.label

    # -*- Fill 1st window. -*-

    set f $win1.leftframe
    frame $f
    pack $f -expand 1 -fill both -side left

    # Create empty fonts tree. It will be replaced later.
    tixTree $f.fontstree
    pack $f.fontstree -side top -expand 1 -fill both -padx 2 -pady 2
    set gui_Widgets(FontsTree) $f.fontstree

    # Create destination box.
    tixComboBox $f.moveto -label "Move to" -labelside left \
	    -command gui_OnDestGroupSelected
    pack $f.moveto -side bottom -fill x -padx 2 -pady 2
    set gui_Widgets(MoveToBox) $f.moveto
    set gui_Widgets(MoveToBoxList) [ $f.moveto subwidget listbox ]

    # Create buttons.
    set f $win1.buttonframe
    frame $f
    pack $f -fill y -side right

    button $f.addfont -text "Add Font..." -underline 0 -width $buttonwidth \
	    -command gui_AddFont
    button $f.viewfont -text "View Font" -underline 0 -width $buttonwidth \
	    -command gui_PreviewFont -state disabled
    button $f.preferences -text "Properties..." -underline 0 \
	    -width $buttonwidth -command gui_FontPreferences -state disabled
    button $f.delete -text "Delete" -underline 0 -width $buttonwidth \
	    -state disabled
    button $f.newgroup -text "New Group..." -underline 0 -width $buttonwidth \
	    -command gui_NewGroup
    button $f.renamegroup -text "Rename Group..." -underline 0 \
	    -width $buttonwidth -state disabled -command gui_RenameGroup
    button $f.move -text "Move Font" -underline 0 -width $buttonwidth \
	    -state disabled -command gui_MoveFont

    pack $f.addfont $f.viewfont $f.preferences $f.delete $f.newgroup \
	    $f.renamegroup -padx 2 -pady 2 -side top
    pack $f.move -padx 2 -pady 2 -side bottom

    set gui_Widgets(AddFontButton) $f.addfont
    set gui_Widgets(PreviewFontButton) $f.viewfont
    set gui_Widgets(PreferencesButton) $f.preferences
    set gui_Widgets(DeleteButton) $f.delete
    set gui_Widgets(NewGroupButton) $f.newgroup
    set gui_Widgets(RenameGroupButton) $f.renamegroup
    set gui_Widgets(MoveFontButton) $f.move

    # -*- Fill 2nd window. -*-

    # Create empty aliases tree.
    tixTree $win2.aliastree -options { hlist.header true } \
	    -browsecmd gui_OnAliasSelected
    pack $win2.aliastree -side left -expand 1 -fill both -padx 2 -pady 2
    set gui_Widgets(AliasTree) $win2.aliastree
    set gui_Widgets(AliasTreeList) [ $win2.aliastree subwidget hlist ]
    $gui_Widgets(AliasTreeList) header create 0 -itemtype text -text "Font names:"

    # Create buttons.
    set f $win2.buttonframe
    frame $f
    pack $f -side right -fill y

    button $f.newalias -text "New Alias..." -underline 2 \
	    -width $buttonwidth -state disabled -command gui_AddAlias
    button $f.renamealias -text "Rename Alias..." -underline 1 \
	    -width $buttonwidth -state disabled -command gui_RenameAlias
    button $f.deletealias -text "Delete Alias" -underline 2 \
	    -width $buttonwidth -state disabled -command gui_DeleteAlias

    pack $f.deletealias $f.renamealias $f.newalias -side bottom -padx 2 -pady 2

    set gui_Widgets(AddAliasButton) $f.newalias
    set gui_Widgets(RenameAliasButton) $f.renamealias
    set gui_Widgets(DeleteAliasButton) $f.deletealias

    # Create balloon windows
    tixBalloon .balloon -statusbar $gui_Widgets(StatusBarLabel)
    set gui_Widgets(BalloonWindow) .balloon
    tixBalloon .stballoon -statusbar $gui_Widgets(StatusBarLabel) \
	    -state status -initwait 100
    set gui_Widgets(StatusOnlyBalloon) .stballoon

    # Add bindings.
    gui_CreateMainWindow:Bindings

    # Initialize vars.
    set gui_Widgets(FontCount) 0
    set gui_Widgets(GroupCount) 0
    set gui_Widgets(ContextMenu) .context
}

#-------------------------------------------------------------------------
#
# gui_GetImages
#
#-------------------------------------------------------------------------

# Creates some images and stores them in gui_Images.
# Parameters:   none
# Return value: none

proc gui_GetImages { } {
    global gui_Images HFMDirectory

    # Load images used by HAMSTER GUI.

    set gui_Images(Preview) previewimage
    # (Preview image will be created later.)

    set gui_Images(folder) [ tix getimage folder ]
    set gui_Images(alias) [ tix getimage file ]

    set gui_Images(ON) [ tix getimage ck_on ]
    set gui_Images(OFF) [ tix getimage ck_off ]
    set gui_Images(SOME) [ tix getimage ck_def ]
    set gui_Images(N/A) {}

    set gui_Images(warning) [ tix getimage warning ]
    if [ catch { set gui_Images(Hamster:0) [ image create photo -file \
	    [ file join $HFMDirectory "Hamster.gif" ] ] } ] then {
	set gui_Images(Hamster:0) ""
    }
}


#****************************************************************************
#
# PUBLIC PROCEDURE
#
#****************************************************************************

#-------------------------------------------------------------------------
#
# GUI_Init
#
#-------------------------------------------------------------------------

# Initialize HAMSTER and finish main window.
# Parameters:   none
# Return value: none

proc GUI_Init { } {
    if { ! [ FM_Init ] } then {
	# No valid config file found.
	if { ! [ gui_Configure ] } then {
	    # User cancelled configuration dialog.
	    exit
	}
	# FM_WriteConfig, FM_Init, and gui_SetTrees already called by
	# the configuring procedures.
    } else {
	# Scan for installed applications.
	gui_GetApps
	# Show fonts/groups/aliases in main window's trees.
	gui_SetTrees
    }
}


#****************************************************************************
#
# STARTUP CODE
#
#****************************************************************************

# Load images.
gui_GetImages

# Create the main window with empty trees. GUI_Init will be called by the
# main module later.
gui_CreateMainWindow
gui_CreateFileSelectDialog

update
