# =======================================================================
# GhostScript Application Module 
# for Hamster Font Manager
#
# $Date: 1998/10/30 11:33:13 $
# $Revision: 1.29 $
#
# =======================================================================
#
# Copyright (C) 1998 The Hamster Project Team [EB]
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# =======================================================================

# REQUIRED-FILES: gs.xbm

# Append itself to list of known applications
lappend AppMods "GS"

putdebug "LOADING APP: GS II - The revenge"

# Fonttypes that this GS-module currently supports
set gs_fonttypesupported "ps tt gs"

# Configuration information
set gs_config(fontdir) ""
set gs_config(fontmapfile) ""

# Mark in config file where data for GS begins
set gs_config(mark) "<GS>"

# If the config info is loaded from the config-file, set this to 1
set gs_config(loaded) 0

# Files that have already been returned as "new fonts"
set gs_config(donenewfont) ""
# Fonts that we have deactivated this session
set gs_deletedaliases ""

set gs_config(icon) [image create bitmap -file [file join $HFMDirectory "gs.xbm"]]

set gs_debug 0

# ========================================================================
# GLOBAL PROCEDURES - Following the HFM Module standard
# ========================================================================

# ------------------------------------------------------------------------
# GS_Init
# ------------------------------------------------------------------------
# Initializes this module. Called when HFM is started
# ------------------------------------------------------------------------
# Config items:
#  fontdir <dir>
#  fontmapfile <filepath>
#  donenewfont <filepath>

proc GS_Init { configfile } {
	global gs_config

gs_putdebug "Init $configfile"

	seek $configfile 0 start
	# If we are in the <GS> section, in_section becomes 1
	set in_section 0
	while {[gets $configfile line] >= 0} {
		if {$in_section && [string match "<*>" $line]} {
			# My section is over
			break
		}
		if {$in_section} {
			# We are in the <GS> section
			set item [lindex $line 0]
			set data [lindex $line 1]
			switch -- $item {
			  "fontdir"     { set gs_config(fontdir) $data }
			  "fontmapfile" { set gs_config(fontmapfile) $data }
			  "donenewfont" { lappend gs_config(donenewfont) $data }
			  default { }
			}
		}
		if {!$in_section && [string match "$gs_config(mark)*" $line]} {
			# My /GS section just begun
			set in_section 1
		}
	}
	if {$in_section == 0} {
		# No config info for GS found...
		return 1
	}
	set gs_config(loaded) 1
	if {$gs_config(fontdir) == "" && $gs_config(fontmapfile) == ""} {
		# Nothing entered -> User does not have GS installed
		return 1
	} {
		if {![file isdirectory $gs_config(fontdir)]} {
			set gs_config(fontdir) ""
		}
		if {![file writable $gs_config(fontmapfile)]} {
			# Fontmap must be writeable for this to make sense
			set gs_config(fontmapfile) ""
			return 0
		} {
			# Reads the Fontmap file
			if {![gs_ReadFontmap $gs_config(fontmapfile)]} {
				set gs_config(fontmapfile) ""
				return 0
			}
		}
	}
	return 1
}

# ------------------------------------------------------------------------
# GS_CheckConfigInput
# ------------------------------------------------------------------------
# User input must be checked
# ------------------------------------------------------------------------
# Variables bound to configuration form:
#   gs_userinput(fontdir)
#   gs_userinput(fontmapfile)

proc GS_CheckConfigInput { } {
	global gs_userinput gs_config

gs_putdebug "GS_CheckConfigInput"
	set errors ""
	if {$gs_userinput(fontdir) == "" && $gs_userinput(fontmapfile) == ""} {
		set gs_config(fontdir) ""
		set gs_config(fontmapfile) ""
		return ""
	}
	if {![file isdirectory $gs_userinput(fontdir)]} {
		lappend errors "No such directory: $gs_userinput(fontdir)"
	}
	if {![file writable $gs_userinput(fontmapfile)]} {
		lappend errors "File specified for Fontmap is not writable:\
				$gs_userinput(fontmapfile)"
	} {
		# Reads the Fontmap file
		if {![gs_ReadFontmap $gs_userinput(fontmapfile)]} {
			lappend errors "Could not read the Fontmap file:\
					$gs_userinput(fontmapfile)"
		}
	}
	if {$errors == ""} {
		# No errors: Replace configuration items
		set gs_config(fontdir) $gs_userinput(fontdir)
		set gs_config(fontmapfile) $gs_userinput(fontmapfile)
	}
	return $errors
}

# ------------------------------------------------------------------------
# GS_QueryNewFonts
# ------------------------------------------------------------------------
# Returns a list of filenames that are new in the "fonts" directory.
# ------------------------------------------------------------------------

proc GS_QueryNewFonts { } {
	global gs_config

gs_putdebug "QueryNewFonts"

	if {$gs_config(fontdir) == ""} { return "" }
	set output ""
	foreach file [glob -nocomplain [file join $gs_config(fontdir) "*"]] {
		if {[lsearch -exact $gs_config(donenewfont) $file] == -1} {
			# A new file! We must return it
			lappend output $file
			lappend gs_config(donenewfont) $file
		}
	}
	return $output
}

# ------------------------------------------------------------------------
# GS_QueryInstalled
# ------------------------------------------------------------------------
# Answers if Ghostscript is installed (i.e. Fontmap file is found)
# ------------------------------------------------------------------------

proc GS_QueryInstalled { } {
	global gs_config

	return [expr {$gs_config(fontmapfile) != ""}]
}

# ------------------------------------------------------------------------
# GS_QueryConfigFrame
# ------------------------------------------------------------------------
# Returns a frame in which the configuration items for this modules
# are to be edited.  Bind input fields to known variable names.
# ------------------------------------------------------------------------
# Variables bound to configuration form:
#   gs_userinput(fontdir)
#   gs_userinput(fontmapfile)

proc GS_QueryConfigFrame { parent } {
	global gs_userinput gs_config gs_dlg

	set gs_userinput(fontdir) $gs_config(fontdir)
	set gs_userinput(fontmapfile) $gs_config(fontmapfile)

	if {!$gs_config(loaded)} {
		gs_GuessConfig
	}

	set gs_frame $parent.gs_frame

	tixLabelFrame $gs_frame -label "GS" -labelside acrosstop -options {
		label.padX 5
	}

	set f [$gs_frame subwidget frame]

	# Fontmap file

	frame $f.fontmap
	label $f.fontmap.l -text "Fontmap file:" -width 30
	entry $f.fontmap.e -width 30 -textvariable gs_userinput(fontmapfile)
	button $f.fontmap.b -text "Browse..." -command {
		set input $gs_userinput(fontmapfile)
		UTIL_RequestFile "Select Fontmap file" input {
			{{Fontmap}	{Fontmap file}}
			{{*}		{All files}}
		}
		set gs_userinput(fontmapfile) $input
	}
	pack $f.fontmap.l -in $f.fontmap -side left -padx 10
	pack $f.fontmap.b -in $f.fontmap -side right -padx 10
	pack $f.fontmap.e -in $f.fontmap -side right -padx 10 -expand yes -fill x

	# Fonts dir

	frame $f.fonts
	label $f.fonts.l -text "Fonts directory:" -width 30
	entry $f.fonts.e -width 30 -textvariable gs_userinput(fontdir)
	button $f.fonts.b -text "Browse..." -command {
		UTIL_RequestDir "Fonts directory" "Select fonts directory" gs_userinput(fontdir)
	}

	pack $f.fonts.l -in $f.fonts -side left -padx 10
	pack $f.fonts.b -in $f.fonts -side right -padx 10
	pack $f.fonts.e -in $f.fonts -side right -padx 10 -expand yes -fill x

	pack $f.fontmap $f.fonts -in $f -side top -anchor e -expand yes -fill both

	return $gs_frame
}


# ------------------------------------------------------------------------
# GS_QueryTypeSupported
# ------------------------------------------------------------------------
# Answers if a specific font type is supported by Ghostscript.
# ------------------------------------------------------------------------
# Variable:
#  gs_fonttypesupported - A list of supported font types

proc GS_QueryTypeSupported { fonttype } {
	global gs_fonttypesupported

	if {[lsearch -exact $gs_fonttypesupported [string tolower $fonttype]] < 0} {
		# Not found in list of supported fonttypes
		return 0
	} {
		return 1
	}
}

# ------------------------------------------------------------------------
# GS_QueryIcon
# ------------------------------------------------------------------------
# Returns the filename to the icon
# ------------------------------------------------------------------------

proc GS_QueryIcon { } {
	global gs_config
	return $gs_config(icon)
}

# ------------------------------------------------------------------------
# GS_QueryAppName
# ------------------------------------------------------------------------
# Returns the name of this application
# ------------------------------------------------------------------------

proc GS_QueryAppName { } {
	return "Ghostscript"
}

# ------------------------------------------------------------------------
# GS_WriteConfig
# ------------------------------------------------------------------------
# - Writes module configuration to HFM's config file
# - Writes current Fontmap
# ------------------------------------------------------------------------

proc GS_WriteConfig { file } {
	global gs_config

gs_putdebug "WriteConfig"
	# Save HFM Config
	puts $file $gs_config(mark)
	puts $file [list "fontdir" $gs_config(fontdir)]
	puts $file [list "fontmapfile" $gs_config(fontmapfile)]
	foreach filename $gs_config(donenewfont) {
		puts $file [list "donenewfont" $filename]
	}
	puts $file ""
	# Save my Fontmap file
	set err_msg ""
	if {$gs_config(fontmapfile) != ""} {
		set err_msg [gs_WriteFontmap $gs_config(fontmapfile)]
	}
	return $err_msg
}

# ------------------------------------------------------------------------
# GS_ActivateFont
# ------------------------------------------------------------------------
# Adds a new font to the Ghostscript configuration (do no write Fontmap)
# ------------------------------------------------------------------------
# Type:
#  fontinfo - list in form  { fontname filename type fondry ... }

proc GS_ActivateFont { fontinfo oldname } {
	global gs_aliases gs_oldaliases

	set fontname [lindex $fontinfo 0]
	set filename [lindex $fontinfo 1]
	set filetail [file tail $filename]
	set fonttype [lindex $fontinfo 2]
gs_putdebug "ActivateFont $filename -> $fontname"

	# Can we handle the given font type?
	if {![GS_QueryTypeSupported $fonttype]} {
		return "Fonts of type $fonttype are not supported by Ghostscript"
	}
	# Use AddAlias to add this font (it will check for conflicts, etc)
	set add_err [GS_AddAlias $fontinfo $fontname]
	if {$add_err != ""} {
		# Something went wrong while adding the font
		return $add_err
	}
	# If this font was removed recently, we still may have aliases
	if {[info exist gs_oldaliases($filetail)]} {
		foreach alias $gs_oldaliases($filetail) {
			# Checks if the old alias was created somewhere else
			if {[gs_AliasExist $alias] != ""} { continue }
			# Removes the alias from the list of deleted aliases
			gs_UndeleteAlias $alias
			if {$alias == $fontname} { continue }
			lappend gs_aliases($filetail) $alias
		}
		unset gs_oldaliases($filetail)
	}
	return ""
}

# ------------------------------------------------------------------------
# GS_RenameAlias
# ------------------------------------------------------------------------
# Rename alias for an specific font
# ------------------------------------------------------------------------

proc GS_RenameAlias { fontinfo oldalias newalias } {
	global gs_aliases

	set fontname [lindex $fontinfo 0]
	set filename [lindex $fontinfo 1]
	set filetail [file tail $filename]

	set i [lsearch $gs_aliases($filetail) $oldalias]
	if {$i == -1} {
		return "Old alias '$oldalias' doesn't exist!"
	}
	if {![gs_ValidAlias $newalias]} {
		return "New name '$newalias' contains illegal characters!"
	}
	set exists [gs_AliasExist $newalias]
	if {$exists != ""} {
		return "New name '$newalias' already exists for font '$exists'"
	}
	set gs_aliases($filetail) [lreplace $gs_aliases($filetail) $i $i $newalias]
	# Add the old alias to the list of deleted aliases
	gs_DeleteAlias $oldname
	return ""
}

# ------------------------------------------------------------------------
# GS_DeactivateFont
# ------------------------------------------------------------------------
# Removes the information about this font.
# Store alias information in temporary variable for eventual restauration
# Store deactivated font into the gs_deactivated list
# ------------------------------------------------------------------------

proc GS_DeactivateFont { fontinfo } {
	global gs_aliases gs_oldaliases gs_fontfiles

	set fontname [lindex $fontinfo 0]
	set filename [lindex $fontinfo 1]
	set filetail [file tail $filename]

gs_putdebug "Deactivated $fontname"
	if {[info exist gs_aliases($filetail)]} {
		# Cache aliases in gs_oldaliases for eventual restauration
		set gs_oldaliases($filetail) $gs_aliases($filetail)
		foreach alias $gs_aliases($filetail) {
			# Mark all aliases to this font as deleted
			gs_DeleteAlias $alias
		}
		unset gs_aliases($filetail)
	}
	if {[info exist gs_fontfiles($filetail)]} {
		unset gs_fontfiles($filetail)
	}
	return ""
}

# ------------------------------------------------------------------------
# GS_QueryFontActive
# ------------------------------------------------------------------------
# Returns 1 if font is currently active in this Ghostscript module
# ------------------------------------------------------------------------

proc GS_QueryFontActive { fontinfo } {
	global gs_aliases

	set filename [lindex $fontinfo 1]
	set filetail [file tail $filename]
	return [info exist gs_aliases($filetail)]
}

# ------------------------------------------------------------------------
# GS_AddAlias
# ------------------------------------------------------------------------
# Adds an alias to a font.  If font is not currently active, activates
# it using the supplied aliasname as "primary fontname". This might happen
# when an alias is added to a font not currently active, or if the font
# is activated (through the GS_FontActivate proc)
# ------------------------------------------------------------------------
# Type:
#  fontinfo - list in form  { fontname filename type fondry ... }

proc GS_AddAlias { fontinfo aliasname } {
	global gs_aliases gs_fontfiles

	set fontname [lindex $fontinfo 0]
	set filename [lindex $fontinfo 1]
	set filetail [file tail $filename]
	set fonttype [lindex $fontinfo 2]
gs_putdebug "Adding alias for $fontname ($filetail): $aliasname"

	if {![gs_ValidAlias $aliasname]} {
		return "The alias '$aliasname' is not valid within Ghostscript!"
	}

	# Check if aliasname collides with already existing alias name
	foreach thisfile [array names gs_aliases] {
		if {$thisfile == $filetail} {
			# Do not check in the font itself!
			# (ActivateFont would not work otherwise)
			continue
		}
		for {set i 0} {$i < [llength $gs_aliases($thisfile)]} {incr i} {
			set thisalias [lindex $gs_aliases($thisfile) $i]
			if {$thisalias == $aliasname} {
				if {$aliasname == [lindex $gs_aliases($thisfile) 0]} {
					return "Name '$aliasname' is already in use in Ghostscript"
				} {
					return "Name '$aliasname' is already in use in Ghostscript \n\
						as an alias to '[lindex $gs_aliases($thisfile) 0]'"
				}
			}
		}
	}
	# Remove this alias from the list of deleted aliases
	gs_UndeleteAlias $aliasname
	# Refreshes the reference to the full path of this file
gs_putdebug "gs_fontfiles($filetail) -> $filename"
	set gs_fontfiles($filetail) $filename
	# Add the font to the activated fonts list
	lappend gs_aliases($filetail) $aliasname
	return ""
}

# ------------------------------------------------------------------------
# GS_RemoveAlias
# ------------------------------------------------------------------------
# Find the aliasname and remove it from the corresponding font
# ------------------------------------------------------------------------

proc GS_RemoveAlias { fontinfo aliasname } {
	global gs_aliases

	set fontname [lindex $fontinfo 0]
	set filename [lindex $fontinfo 1]
	set filetail [file tail $filename]

	if {![info exist gs_aliases($filetail)]} { return "" }
	set pos [lsearch -exact $gs_aliases($filetail) $aliasname]
	if {$pos != -1} {
		# We found the alias! Nuke it
		set gs_aliases($filetail) [lreplace $gs_aliases($filetail) $pos $pos]
	}
	if {$gs_aliases($filetail) == ""} {
		# Last alias deleted
		unset gs_aliases($filetail)
	}
	# Add this alias to the list of deleted aliases
	gs_DeleteAlias $aliasname
	return ""
}

# ------------------------------------------------------------------------
# GS_QueryAliases
# ------------------------------------------------------------------------
# Returns a list of known aliases for a specified font
# ------------------------------------------------------------------------

proc GS_QueryAliases { fontinfo } {
	global gs_aliases

	set filename [lindex $fontinfo 1]
	set filetail [file tail $filename]

	if {[info exist gs_aliases($filetail)]} {
		return $gs_aliases($filetail)
	}
	return ""
}

# ------------------------------------------------------------------------
# GS_QueryMAS
# ------------------------------------------------------------------------
# Does this application supports multiple aliases
# ------------------------------------------------------------------------

proc GS_QueryMAS { } {
	# GS supports multiple aliases
	return 1
}

# ------------------------------------------------------------------------
# GS_QueryPrefsFrame
# ------------------------------------------------------------------------
# Returns a frame in which font preferences fort his font
# are to be edited.  Bind input fields to known variable names.
# ------------------------------------------------------------------------

proc GS_QueryPrefsFrame { parent fontinfo } {
    return ""
}

# ------------------------------------------------------------------------
# GS_CheckPrefsInput
# ------------------------------------------------------------------------
# Check if font entered preferences are valid
# ------------------------------------------------------------------------

proc GS_CheckPrefsInput { } {
    return ""
}


# ========================================================================
# PRIVATE PROCEDURES - For internal use only!
# ========================================================================

proc gs_putdebug { text } {
	global gs_debug
	if {$gs_debug} { puts "GS: $text" }
}

# ------------------------------------------------------------------------
# gs_AliasExist
# ------------------------------------------------------------------------
# Returns:
#  Fontname - Alias already exists for font "Fontname"
#  ""       - Alias does not exist
# ------------------------------------------------------------------------

proc gs_AliasExist { aliasname } {
	global gs_aliases

	foreach font [array names gs_aliases] {
		foreach alias $gs_aliases($font) {
			if {$aliasname == $alias} { return "[lindex $gs_aliases($font) 0]" }
		}
	}
	return ""
}

# ------------------------------------------------------------------------
# gs_ValidAlias
# ------------------------------------------------------------------------
# Checks if an alias name is valid or not
# Valid aliases contains only a-Z, 0-9, '-' and '_'
# ------------------------------------------------------------------------

proc gs_ValidAlias { alias } {
	return [regexp {^[A-Za-z][A-Za-z0-9_\-]*$} $alias]
}

# ------------------------------------------------------------------------
# gs_DeleteAlias
# ------------------------------------------------------------------------
# Add this alias to the list of "deleted" aliases
# ------------------------------------------------------------------------
# Global:
#  gs_deletedaliases { alias1 alias2 ... }

proc gs_DeleteAlias { alias } {
	global gs_deletedaliases

	if {[lsearch -exact $gs_deletedaliases $alias] == -1} {
		# Mark this alias as deleted if it isn't there already
		lappend gs_deletedaliases $alias
	}
}

# ------------------------------------------------------------------------
# gs_UndeleteAlias
# ------------------------------------------------------------------------
# Removes this alias from the list of "deleted" aliases
# ------------------------------------------------------------------------
# Global: 
#  gs_deletedaliases { alias1 alias2 ... }

proc gs_UndeleteAlias { alias } {
	global gs_deletedaliases

	set pos [lsearch -exact $gs_deletedaliases $alias]
	if {$pos != -1} {
		# Remove this alias from the list of deleted aliases
		set gs_deletedaliases [lreplace $gs_deletedaliases $pos $pos]
	}
	return ""
}

# ------------------------------------------------------------------------
# gs_GuessConfig
# ------------------------------------------------------------------------
# Tries to find the GhostScript application and sets a reasonable default
# for the	Fontmap file and fonts directory.  Only leave those blank if
# we *really* cannot find GS on this system
# ------------------------------------------------------------------------
# Global Variables:
#  gs_config(fontmapfile)
#  gs_config(fontdir)

proc gs_GuessConfig { } {
	global env
	global gs_config gs_userinput

	# Where can Ghostscript possibly be?
	set searchpath {
		/usr/local/lib/ghostscript
		/usr/local/share/ghostscript
		/usr/lib/ghostscript
		/usr/share/ghostscript
	}

	foreach path $searchpath {
		foreach fontmap [concat \
							[glob -nocomplain [file join $path "Fontmap"]] \
							[glob -nocomplain [file join $path "*" "Fontmap"]] \
						] {
			lappend possible $fontmap
		}
	}
	if {![info exist possible]} { return "" }

	set last(time) 0
	set last(file) ""
	foreach path $possible {
		if {[file mtime $path] > $last(time)} {
			set last(time) [file mtime $path]
			set last(file) $path
		}
	}
	set gs_userinput(fontmapfile) "$last(file)"

	set fontdir [file join [file dirname $last(file)] "fonts"]
	if {[file isdir $fontdir]} {
		# A subdirectory of the actual version of Ghostscript
		set gs_userinput(fontdir) $fontdir
gs_putdebug "GhostScript found: $last(file) Fonts: $fontdir"
		return ""
	}

	set fontdir [file join [file dirname [file dirname $last(file)]] "fonts"]
	if {[file isdir $fontdir]} {
		# A subdirectory of the main Ghostscript directory
		set gs_userinput(fontdir) $fontdir
gs_putdebug "GhostScript found: $last(file) Fonts: $fontdir"
	}

	return ""
}

# ------------------------------------------------------------------------
# gs_ReadFontmap
# ------------------------------------------------------------------------
# Reads the GS Fontmap file, and stores the found fonts into
# global variables.  If the file is not GS's Fontmap, return 0.
# ------------------------------------------------------------------------
# Global Variables:
#  gs_aliases(filename) {alias1} {alias2} ...

proc gs_ReadFontmap { filename } {
	global gs_aliases gs_fontfiles

	if {[catch {set in [open $filename r]}]} {
		# Could not read Fontmap (permission denied, does not exist...)
		return 0
	}
gs_putdebug "ReadFontmap $filename"
	# Reset all global variables
	catch { unset gs_aliases }
	set fonts ""
	while {[gets $in input] >= 0} {
		set input [string trim $input]
		if {[string match "%*" $input] || $input == ""} {
			# A comment
			continue
		}
		if {[string match "/*" $input]} {
			# A fontname or alias was found
			#  Font:  /CharterBT-Roman   (bchr.gsf)  ;
			# Alias:  /Charter-Roman     /CharterBT-Roman    ;
			set fontname [string trimleft [lindex $input 0] "/"]
			set value [string trimright [lindex $input 1] ";\t "]
			if {[scan $value {(%[^)])} fontfile] == 1} {
				# Its a fontfile
				lappend fonts [list $fontfile [list $fontname]]
			} elseif {[scan $value {/%s} original] == 1} {
				# Its an alias
				lappend aliases($original) $fontname
			}
			# Else: its trash
		}
	}
	close $in
	global HFMDirectory
	# A list of fontnames/aliases already used
	set namesdone ""
	foreach font $fonts {
		# font is an item: { file fontname }
		set file [lindex $font 0]
		set filetail [file tail $file]
		set fontname [lindex $font 1]
		if {[lsearch -exact $namesdone $fontname] != -1} {
			putdebug "GS: Fontname '$fontname' is repeated! Ignoring!"
			continue
		}
		set gs_fontfiles($filetail) $file
		set gs_aliases($filetail) $fontname
		# This name is taken
		lappend namesdone $fontname
		if {[info exist aliases($fontname)]} {
			# This font has more aliases!
			foreach thisalias $aliases($fontname) {
				if {[lsearch -exact $namesdone $thisalias] != -1} {
					putdebug "GS: Alias '$thisalias' for Fontname '$fontname' is repeated! Ignoring!"
					continue
				}
				lappend gs_aliases($filetail) $thisalias
				lappend namesdone $thisalias
			}
		}
	}
	return 1
}

# ------------------------------------------------------------------------
# gs_WriteFontmap
# ------------------------------------------------------------------------
# Writes the fonts from global variables back to the GS Fontmap file
# ------------------------------------------------------------------------
# Global Variables:
#  gs_aliases(filename[A) {Alias1} {Alias2} ...
#  gs_deletedfont { Fontname1 Fontname2 ... }
#  gs_deletedaliases { { FontName1 Alias1 } { FontName2 Alias2 } ... }

proc gs_WriteFontmap { filename } {
	global gs_aliases gs_deletedaliases gs_fontfiles

	set tmpfile [file join "/tmp" "Fontmap.[pid].[clock seconds].tmp"]
	if {[catch {set out [open $tmpfile w+]}]} { 
		# Could not write to this file
 		return "Could not write to file '$tmpfile'"
	}
	set in [open $filename r]
#	puts $out "% Fontmap for Ghostscript"
#	puts $out "% --------------------------------------------------------------------------"
#	puts $out "% Automatically generated by HFM on\
#				[clock format [clock seconds]]"
#	puts $out "% Edit on your own risk!"
#	puts $out ""
	set donefonts ""
	while {[gets $in line] >= 0} {
		switch -glob -- $line {
		  "/*" {
			# A fontname or alias was found
			#  Font:  /CharterBT-Roman   (bchr.gsf)  ;
			# Alias:  /Charter-Roman     /CharterBT-Roman    ;
			set fontname [string trimleft [lindex $line 0] "/"]
			set value [string trimright [lindex $line 1] ";\t "]
			# Mark this name as DONE
			if {[lsearch -exact $gs_deletedaliases $fontname] != -1} {
				# This font or alias was deactivated
				puts $out "%removed: $line"
			} {
				# Active font is still active
				if {[scan $value {(%[^)])} fontfile] == 1} {
					# Its a fontfile  /$fontname ($fontfile) ;
					set filetail [file tail $fontfile]
					if {[info exist gs_fontfiles($filetail)]} {
						puts $out "/[lindex $gs_aliases($filetail) 0]	\($gs_fontfiles($filetail)\)	;"
						lappend donealiases [lindex $gs_aliases($filetail) 0]
					} {
						puts $out "let: $line"
						lappend donealiases $fontname
					}
				} elseif {[scan $value {/%s} original] == 1} {
					# Its an alias    /$fontname  /$original ;
					lappend donealiases $fontname
					foreach file [array names gs_aliases] {
						# Search all aliases if one name match
						set pos [lsearch -exact $gs_aliases($file) $fontname]
						if {$pos == 0} {
							# Alias is the firstname of a font
							puts $out "/$fontname	\($gs_fontfiles($file)\)	;"
							break
						} elseif {$pos > 0} {
							# Alias references to the main name
							puts $out "/$fontname	/[lindex $gs_aliases($file) 0]	;"
							break
						}
					}
					if {$pos == -1} { puts $out "$line" }
				}
			}
		  }
		  "%removed: *" {
			# A deactivated font or alias was found
			set uncommented [string range $line 10 end]
			set fontname [string trimleft [lindex $uncommented 0] "/"]
			set value [string trimright [lindex $uncommented 1] ";\t "]
			if {[scan $value {(%[^)])} fontfile] == 1} {
				# Its a fontfile  /$fontname ($fontfile) ;
				lappend donealiases $fontname
				set filetail [file tail $fontfile]
				if {[info exist gs_aliases($filetail)]} {
					puts $out "/$fontname	\($gs_fontfiles($filetail)\)	;"
				} {
					puts $out "$line"
				}
			} elseif {[scan $value {/%s} original] == 1} {
				# Its an alias    /$fontname  /$original ;
				foreach file [array names gs_aliases] {
					# Search all aliases if one name match
					set pos [lsearch -exact $gs_aliases($file) $fontname]
					if {$pos == 0} {
						# Alias is the firstname of a font
						puts $out "/$fontname	\($gs_fontfile($file)\)	;"
						lappend donealiases $fontname
						break
					} elseif {$pos > 0} {
						# Alias Reference to the first name
						puts $out "/$fontname	/[lindex $gs_aliases($file) 0]	;"
						lappend donealiases $fontname
						break
					}
				}
				if {$pos == -1} { puts $out "$line" } 
			}
		  }
		  default {
			# Leave comments alone
			puts $out $line
		  }
		}
	}

	# Add all new fonts
	set first 1
	foreach fontfile [lsort [array names gs_aliases]] {
		for {set i 0} {$i < [llength $gs_aliases($fontfile)]} {incr i} {
			set font [lindex $gs_aliases($fontfile) $i]
			if {[lsearch -exact $donealiases $font] != -1} { continue }
			# Write the new information
			if {$first} {
				puts $out ""
				puts $out "% New information from: [clock format [clock seconds]]"
				puts $out ""
				set first 0
			}
			if {$i == 0} {
				puts $out "/$font	\($gs_fontfiles($fontfile)\)	;"
			} {
				puts $out "/$font	/[lindex $gs_aliases($fontfile) 0]	;"
			}
		}
	}
	close $out
	if {[catch { file rename -force $tmpfile $filename } error]} {
gs_putdebug "WriteFontMap error: $error"
		return "Error moving '$tmpfile' to '$filename': $error"
	}
	return ""
}

# ========================================================================

putdebug "Loaded GS MOD"
