# =======================================================================
# Font Management Module
# for HAMSTER Font Manager.
#
# $Date: 1998/11/03 09:37:21 $
# $Revision: 1.33 $
#
# =======================================================================
#
# Copyright (C) 1998 The Hamster Project Team [AM]
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# =======================================================================

# REQUIRED-FILES: ani0.xpm ani1.xpm ani2.xpm ani3.xpm

#*************************************************************************
#
# GLOBAL VARIABLES
#
#*************************************************************************

# fm_Groups
# Array

# fm_Fonts
# Array

set fm_DefaultGroup Default
# Name of the group where new fonts are automatically added to.

# fm_AnimationFrames
# Array containing images; load ani*.xpm:
set fm_AnimationFrames(count) 4
for { set fm_i 0 } { $fm_i < $fm_AnimationFrames(count) } { incr fm_i } {
    if [ catch {
	set fm_AnimationFrames($fm_i) [ image create pixmap -file [ file join $HFMDirectory ani${fm_i}.xpm ] ]
    } ] then {
	set fm_AnimationFrames($fm_i) ""
    }
}
unset fm_i

#*************************************************************************
#
# PRIVATE PROCEDURES
#
#*************************************************************************

# Returns file name of the HFM configuration file.
proc fm_QueryConfigFileName { } {
    global HFMDirectory
    return [ file join $HFMDirectory "hfm.conf" ]
    # HFMDirectory is defined in the main module
}

# Returns the path name of the directory HFM's fonts are in.
proc fm_QueryFontsDir { } {
    global HFMDirectory
    return [ file join $HFMDirectory "fonts" ]
}

# Returns an image which is a frame from an animation. The next call with the
# same variable will return the next frame.
proc fm_AnimationFrame { indexvar } {
    global fm_AnimationFrames
    upvar $indexvar i
    if [ catch { 
	set i [ expr ( $i + 1 ) % $fm_AnimationFrames(count) ] 
    } ] then {
	set i 0
    }
    return $fm_AnimationFrames($i)
}

#*************************************************************************
#
# PUBLIC PROCEDURES
#
#*************************************************************************

#-------------------------------------------------------------------------
#
# FM_Init
#
#-------------------------------------------------------------------------

# Try to read configuration file, determine which fonts exist.
# Parameters:   none
# Return value: "1" if a config-file was found and could be read,
#               "0" otherwise.

proc FM_Init { } {
    global AppMods FontTypeMods fm_Groups fm_Fonts fm_DefaultGroup
    set fontsdir [ fm_QueryFontsDir ]
    set retfalse 0
    set extensions ""
    set relatedfiles ""
    catch { unset fm_Groups }
    # Make array fm_Fonts empty
    catch { unset fm_Fonts }

    if { ! [ file exists [ fm_QueryConfigFileName ] ] } then {
	# Config file doesn't exist. HFM is called for the first time.

	# Show the license.
	UTIL_ShowLicense

	# Try to create an empty config file.
	catch { exec touch [ fm_QueryConfigFileName ] }
    }

    if [ catch { open [ fm_QueryConfigFileName ] r } conffile ] then {
	# Can't open config file.
	return 0
    }

    # read config file
    UTIL_ShowHamster -metervalue 0.0 -metertext "Reading config file" \
	    -title "Initialize HFM"
    while 1 {
	if [ eof $conffile ] then break
	set line [ string trim [ gets $conffile ] ]
	if [ string match "<*>" $line ] then break

	if [ string match "font *" $line ] then {
	    set fonts([ lindex $line 3 ]) $line
	    set relatedfiles [ concat $relatedfiles [ lindex $line 13 ] ]
	} elseif [ string match "group *" $line ] then {
	    set fm_Groups([ lindex $line 1 ]) ""
	} elseif [ string length $line ] then {
	    # syntax error
	    UTIL_ShowHamster -done
	    return 0
	}
    }

    set count [ expr [ llength $FontTypeMods ] + [ llength $AppMods ] ]
    set j 0

    # Initialize font type plugin modules
    UTIL_ShowHamster -metertext "Initialize plugins"

    foreach module $FontTypeMods {
	UTIL_ShowHamster -metervalue [ expr 0.1 + ${j}.0 / $count * 0.4 ]
	incr j
	set erg [${module}_Init $conffile]
	putdebug "${module}_Init returned <$erg>"
	if { ! $erg } then {
	    set retfalse 1
	}
    }

    # Initialize application plugin modules
    foreach module $AppMods {
	UTIL_ShowHamster -metervalue [ expr 0.1 + ${j}.0 / $count * 0.4 ]
	incr j
	set erg [${module}_Init $conffile]
	putdebug "${module}_Init returned <$erg>"
	if { ! $erg } then {
	    set retfalse 1
	}
    }

    close $conffile
    if $retfalse then {
	UTIL_ShowHamster -done
	return 0
    }

    # Scan fonts directory
    UTIL_ShowHamster -metervalue 0.5 -metertext "Scanning fonts"
    set fontfiles ""
    if { ![file isdirectory $fontsdir]} {
	exec mkdir $fontsdir
    }

    foreach ffile [ glob -nocomplain [ file join $fontsdir * ] ] {
	if { [ lsearch -exact $fontfiles $ffile ] == -1 } then {
	    lappend fontfiles $ffile
	}
    }

    set count [ llength $fontfiles ]
    set j 0
    UTIL_ShowHamster -metertext "Processing fonts"

    # Process font files
    foreach ffile $fontfiles {
	UTIL_ShowHamster -metervalue [ expr 0.6 + ${j}.0 / $count * 0.4 ]
	incr j

	if { [ lsearch -exact [ array names fonts ] $ffile ] >= 0 } then {
	    # File is registered in config file.
	    set fontname [ lindex $fonts($ffile) 1 ]
	    if { [ lsearch -exact [ array names fm_Fonts ] $fontname ] < 0 } then {
		# No other font with the same name. Add it.
		set fm_Fonts($fontname) [ lrange $fonts($ffile) 3 end ]
		lappend fm_Groups([ lindex $fonts($ffile) 2 ]) $fontname
	    }
	    # If another font with the same name has already been found,
	    # then ignore the new one, even if it has been used before.
	} elseif { [ lsearch -exact $relatedfiles $ffile ] >=0 } then {
	    # It's a file related to a font file. Ignore it.
	} else {
	    # Unknown file. Test it.
	    foreach module $FontTypeMods {
		set result [ ${module}_ReadFontFile $ffile ]
		if [ string length $result ] then {
		    set fontname [ lindex $result 0 ]
		    if { [ lsearch -exact [ array names fm_Fonts ] $fontname ] < 0 } then {
			lappend fm_Groups($fm_DefaultGroup) $fontname
			set fm_Fonts($fontname) [ lrange $result 1 end ]
			set relatedfiles [ concat $relatedfiles [ lindex $result 11 ] ]
		    }
		    # Ignore file if a font with the same name is installed. 
		    break
		}
	    }
	}
    }
    if { ! [ array exist fm_Groups ] } then {
	set fm_Groups($fm_DefaultGroup) ""
    }

    UTIL_ShowHamster -done
    return 1
}

#-------------------------------------------------------------------------
#
# FM_WriteConfig
#
#-------------------------------------------------------------------------

# Writes the configuration file.
# Parameters:   none
# Return value: an empty string if there was no error in writing the file,
#               an error message in form of a string otherwise.

proc FM_WriteConfig { } {
    global AppMods FontTypeMods fm_Groups fm_Fonts
    set errmsg ""

    if [ catch { open [ fm_QueryConfigFileName ] w } conffile ] then {
	return "Can't open config file: $conffile"
    }

    foreach group [ array names fm_Groups ] {
	if [ catch { puts $conffile [ list group $group ] } err ] then {
	    return "Error writing config file: $err"
	}

	foreach font $fm_Groups($group) {
	    if [ catch { puts $conffile [ concat [ list font $font $group ] \
		    $fm_Fonts($font) ] } err ] then {
		return "Error writing config file: $err"
	    }
	}
    }

    foreach module $FontTypeMods {
	set err [ ${module}_WriteConfig $conffile ]
	if [ string length $err ] then { lappend errmsg $err }
    }

    foreach module $AppMods {
	set err [ ${module}_WriteConfig $conffile ]
	if [ string length $err ] then { lappend errmsg $err }
    }

    close $conffile
    return [ join $errmsg \n ]
}

#-------------------------------------------------------------------------
#
# FM_QueryFontInfo
#
#-------------------------------------------------------------------------

# Return fontinfo list stored in fm_Fonts for a font.
# Parameters:   fontname (string): the name of the font.
# Return value: list of font informations.

proc FM_QueryFontInfo { fontname } {
    global fm_Fonts
    return $fm_Fonts($fontname)
}

#-------------------------------------------------------------------------
#
# FM_QueryApps
#
#-------------------------------------------------------------------------

# Which applications are installed on the system?
# Parameters:   none
# Return value: a list of prefixes of the installed app-plugins.

proc FM_QueryApps { } {
    global AppMods
    set retval ""
    foreach app $AppMods {
	if [ ${app}_QueryInstalled ] then {
	    lappend retval $app
	}
    }
    return $retval
}

#-------------------------------------------------------------------------
#
# FM_QueryConfigFrame
#
#-------------------------------------------------------------------------

# Gathers children configuration frames from plugin modules
# Parameter:    parent (string): identifier of the parent container frame.
# Return value: Identifier of the created frame,which can be packed by the GUI.

proc FM_QueryConfigFrame { parent } {
    global AppMods FontTypeMods
    frame $parent.cfgframe
    foreach module [ concat $FontTypeMods $AppMods ] {
	set result [ ${module}_QueryConfigFrame $parent.cfgframe ]
	if [ string length $result ] then {
	    pack $result -in $parent.cfgframe -side top -fill x
	}
    }
    return $parent.cfgframe
}

#-------------------------------------------------------------------------
#
# FM_OnConfigClosed
#
#-------------------------------------------------------------------------

# Called when the configuration dialog is closed.
# Parameters:   none
# Return value: a list of two items, where the first one is 1 if the user
#               input was valid, and 0 otherwise, and the second one is an
#               error message.

proc FM_OnConfigClosed { } {
    global AppMods FontTypeMods fm_DefaultGroup

    foreach module [ concat $FontTypeMods $AppMods ] {
	set result [ ${module}_CheckConfigInput ]
	if [ string length $result ] then {
	    foreach line $result {
		append retval \n$line
	    }
	}
    }

    if [ info exist retval ] then {
	# Input is not valid, return error message
	return [ list 0 [ string trimleft $retval ] ]
    }
    set retval ""

    # Input is ok. Check if new fonts have to be added.
    foreach module $AppMods {
	UTIL_ShowHamster -label "Scanning font files for $module..." \
		-image [ fm_AnimationFrame frameno ]

	foreach fontfile [ ${module}_QueryNewFonts ] {
	    # try to add font to default group, use default action.
	    set result [ FM_AddFont $fontfile $fm_DefaultGroup ]

	    UTIL_ShowHamster -image [ fm_AnimationFrame frameno ]

	    if { [ lindex $result 1 ] != "" } then {
		# It is a valid font file, so try to activate it.
		# If it couldn't be added it may exist anyway in HFM.
		FM_ActivateFont [ lindex $result 1 ] $module $fontfile

		if { [ lindex $result 0 ] != "" } then {
		    # It's a font file, but it couldn't be added.
		    append retval \n[ lindex $result 0]
		}
	    }
	    # Not a font file. Just ignore it and do nothing.
	}
    }

    UTIL_ShowHamster -done

    # now reinitialize the modules to make all changes take effect.
    set result [ FM_WriteConfig ]
    if [ string length $result ] then { 
	append retval "\nCouldn't write config file: $result"
	return [ list 1 $retval ]
    }

    if { ! [ FM_Init ] } then {
	append retval "\nReinitialisation of the modules failed.\n That's a bug."
    }

    return [ list 1 $retval ]
}

#-------------------------------------------------------------------------
#
# FM_QueryFonts
#
#-------------------------------------------------------------------------

# Which fonts are available in a specific group?
# Parameter:    groupname (string): the groupname for which to list the fonts.
# Return value: A list that should contain one list item per font.  Each item 
#               is another list in the format {fontname icon} where "icon" is
#               the identifier of the icon for this type of font.

proc FM_QueryFonts { groupname } {
    global fm_Groups fm_Fonts
    set retval ""
    foreach font $fm_Groups($groupname) {
	set icon [ lindex $fm_Fonts($font) 1 ]
	lappend retval [ list $font $icon ]
    }
    return $retval
}

#-------------------------------------------------------------------------
#
# FM_AddFont
#
#-------------------------------------------------------------------------

# Try to add a new font to a group.
# Parameters:   fontfile (path): full path of the file that is to be added.
#               group (string): name of the group where to add the font.
# Return value: A list containing an error message (which will be empty if
#               the font could be added) and the fontname (empty if no font).

proc FM_AddFont { fontfile group { action "ln -s" } } {
    global FontTypeMods fm_Groups fm_Fonts fm_DefaultGroup

    # Check if the file is a valid font file.
    foreach module $FontTypeMods {
	set result [ ${module}_ReadFontFile $fontfile ]
	set fontname [ lindex $result 0 ]
	if [ string length $result ] then break
    }
    if { ! [ string length $result ] } then {
	return [ list "File is not a valid font file: $fontfile" {} ]
    }

    # Check if the file already exists in HFM fonts directory.
    set destfile [ file join [ fm_QueryFontsDir ] [ file tail $fontfile ] ]
    if [ file exist $destfile ] then {
	return [ list "File already exists: $destfile." $fontname ]
    }

    if { ! [ string length $group ] } then {
	# Add font th default group if no group name is given.
	set group $fm_DefaultGroup
    }

    if { [ lsearch -exact [ array names fm_Fonts ] $fontname ] < 0 } then {

	# Font doesn't exist yet. Copy/move it.
	if [ catch "exec $action $fontfile $destfile" ] then {
	    return [ list "Can't put file into HFM fonts directory ($action failed): $fontfile" {} ]
	}

	# Add related files.
	foreach relfile [ lindex $result 11 ] {
	    set reldest [ file join [ fm_QueryFontsDir ] \
		    [ file tail $relfile ] ]
	    catch "exec $action $relfile $reldest"
	}

	# Get fontinfo with the right pathname. 
	set result [ ${module}_ReadFontFile $destfile ]
	##set result [ lreplace $result 1 1 $destfile ]

	# Add font to internal lists.
	lappend fm_Groups($group) $fontname
	set fm_Fonts($fontname) [ lrange $result 1 end ]
	# Font is added, nothing left to do.
	return [ list "" $fontname ]
    } else {

	# Font already exists.
	return [ list "File $fontfile: There is already a font called $fontname." $fontname ]
    }
}

#-------------------------------------------------------------------------
#
# FM_RemoveFont
#
#-------------------------------------------------------------------------

# Removes a font file from the system.
# Parameter:    fontname (string): the font to be removed.
#               groupname (string): the group the font is in.
# Return value: An empty string if the font could be removed. Otherwise,
#               returns a string containing an error message.

proc FM_RemoveFont { fontname groupname } {
    global AppMods fm_Fonts fm_Groups
    if { ! [ info exist fm_Fonts($fontname) ] } then {
	# This error should never occur!
	return "Font doesn't exist: $fontname.\nYou have found a bug."
    }

    # Deactivate font first.
    foreach app $AppMods {
	set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
	if [ ${app}_QueryFontActive $fontinfo ] then {
	    ${app}_DeactivateFont $fontinfo
	}
    }

    set filename [ lindex $fm_Fonts($fontname) 0 ]
    # Remove the font file if possible.
    # Missing font files may cause problems in other applications.
    # But thats not our problem ;-) ...
    if [ catch { file delete -force -- $filename } ] then {
	return "Can't remove font file: $filename."
    }

    # Remove related files.
    foreach relfile [ lindex $fm_Fonts($fontname) 10 ] {
	catch { file delete -force -- $relfile }
    }

    # Remove font from internal arrays.
    unset fm_Fonts($fontname)
    set group $groupname
    set idx [ lsearch -exact $fm_Groups($group) $fontname ]
    set fm_Groups($group) [ lreplace $fm_Groups($group) $idx $idx ]

    return ""
}

#-------------------------------------------------------------------------
#
# FM_MoveFont
#
#-------------------------------------------------------------------------

# Moves a font from one group to another.
# Parameters:   fontname (string): the font to be moved.
#               oldgroup (string): group the font is in.
#               newgroup (string): destination group.
# Return value: An empty string if the font could be removed. Otherwise,
#               returns a string containing an error message.
## will always return "" if font exists.

proc FM_MoveFont { fontname oldgroup newgroup } {
    global fm_Fonts fm_Groups

    if { ! [ info exist fm_Fonts($fontname) ] } then {
	# This error should never occur!
	return "Font doesn't exist: $fontname.\nYou have found a bug."
    }

    set idx [ lsearch -exact $fm_Groups($oldgroup) $fontname ]

    # Remove it from the old group.
    set fm_Groups($oldgroup) [ lreplace $fm_Groups($oldgroup) $idx $idx ]

    # Insert it in the new group.
    lappend fm_Groups($newgroup) $fontname
    # This will work even for newgroup == oldgroup.

    return ""
}

#-------------------------------------------------------------------------
#
# FM_ActivateFont
#
#-------------------------------------------------------------------------

# Activates a font in a specific application.
# Parameters:   fontname (string): the font to be activated.
#               app (string): application prefix.
#               oldfile (string, optional): the old filename if the font has
#                   been added from app's directory, "" otherwise.
# Return value: The same string that the plugin module returned.

proc FM_ActivateFont { fontname app { oldfile "" } } {
    global fm_Fonts

    if { ! [ info exist fm_Fonts($fontname) ] } then {
	# font doesn't exist.
	return "Font doesn't exist: `$fontname'."
    }

    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
    return [ ${app}_ActivateFont $fontinfo $oldfile ]
}

#-------------------------------------------------------------------------
#
# FM_DeactivateFont
#
#-------------------------------------------------------------------------

# Deactivates a font in a specific application.
# Parameters:   fontname (string): the font to be deactivated.
#               app (string): application prefix.
# Return value: Returns the same string that the plugin module returned.

proc FM_DeactivateFont { fontname app } {
    global fm_Fonts
    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
    # Just call the module's DeactivateFont proc.
    return [ ${app}_DeactivateFont $fontinfo ]
}

#-------------------------------------------------------------------------
#
# FM_QueryFontActive
#
#-------------------------------------------------------------------------

# Is a font active in a specific application?
# Parameters:   fontname (string): the font to be queried.
#               app (string): application prefix.
# Return value: "ON" if the font is active in this application,
#               "OFF" if the font is inactive in this application,
#               "N/A" if the font could not be activated in this application.

proc FM_QueryFontActive { fontname app } {
    global fm_Fonts
    set fonttype [ lindex $fm_Fonts($fontname) 1 ]
    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]

    if [ ${app}_QueryFontActive $fontinfo ] then {
	return "ON"
    } elseif [ ${app}_QueryTypeSupported $fonttype ] then {
	return "OFF"
    } else {
	return "N/A"
    }
}


#-------------------------------------------------------------------------
#
# FM_QueryGroups
#
#-------------------------------------------------------------------------

# Which groups are available to store fonts in?
# Parameters:   none
# Return value: a list of available groups.

proc FM_QueryGroups { } {
    global fm_Groups
    return [ array names fm_Groups ]
}

#-------------------------------------------------------------------------
#
# FM_NewGroup
#
#-------------------------------------------------------------------------

# Create a new, initially empty group.
# Parameter:    groupname (string): name of the group to be created.
# Return value: Returns an empty string if the group could be created.
#               Otherwise, returns an string containing the error that avoided
#               the group to be created.

proc FM_NewGroup { groupname } {
    global fm_Groups
    if [ info exist fm_Groups($groupname) ] then {
	return "Group `$groupname' already exists."
    } else {
	set fm_Groups($groupname) ""
	return ""
    }
}

#-------------------------------------------------------------------------
#
# FM_RenameGroup
#
#-------------------------------------------------------------------------

# Renames a group to another name.
# Parameters:   oldgroup (string): name of the old group.
#               newgroup (string): name of the new group.
# Return value: Returns an empty string if the group could be renamed.
#               Otherwise, returns an string containing the error that avoided
#               the group to be renamed.

proc FM_RenameGroup { oldgroup newgroup } {
    global fm_Groups
    if [ info exist fm_Groups($newgroup) ] then {
	return "Group `$newgroup' already exists."
    } else {
	set fm_Groups($newgroup) $fm_Groups($oldgroup)
	unset fm_Groups($oldgroup)
	return ""
    }
}

#-------------------------------------------------------------------------
#
# FM_RemoveGroup
#
#-------------------------------------------------------------------------

# Removes a group of fonts.
# Parameter:    groupname (string): name of the group to be removed.
# Return value: Returns an empty string if the group could be removed.
#               Otherwise, returns an string containing the error that avoided
#               the group to be removed.

proc FM_RemoveGroup { groupname } {
    global fm_Groups
    if [ llength $fm_Groups($groupname) ] then {
	return "Group is not empty. It can't be removed."
    }
    unset fm_Groups($groupname)
    return ""
}

#-------------------------------------------------------------------------
#
# FM_ActivateGroup
#
#-------------------------------------------------------------------------

# Activates all fonts in a group for a specific application.
# Parameters:   groupname (string): the group to be activated.
#               app (string): application prefix.
# Return value: A list containing two elements {errormsg fontnames}:
#               errormsg: A string with eventual error messages returned
#                 from the FM_ActivateFont calls. This will be an empty string
#                 if there were no errors.
#               fontnames: A list of fontnames that could be activated.
#                 This will be used by the GUI to change the state of the
#                 checkboxes in the font tree.

proc FM_ActivateGroup { groupname app } {
    global fm_Groups
    set errormsg ""
    set fontnames ""

    putdebug "FM_ActivateGroup $groupname $app"

    foreach fontname $fm_Groups($groupname) {
	if { [ FM_QueryFontActive $fontname $app ] == "OFF" } then {
	    set result [ FM_ActivateFont $fontname $app ]
	    if [ string length $result ] then {
		if [ string length $errormsg ] then {		
		    append errormsg \n $result
		} else {
		    set errormsg $result
		}
	    } else {
		lappend fontnames $fontname
	    }
	}
    }
    return [ list $errormsg $fontnames ]
}

#-------------------------------------------------------------------------
#
# FM_DeactivateGroup
#
#-------------------------------------------------------------------------

# Deactivates all fonts in a group in a specific application.
# Parameters:   groupname (string) the group to be deactivated.
#               app (string): application prefix.
# Return value: A list containing two elements {errormsg fontnames}:
#               errormsg: A string with eventual error messages returned
#                 from the FM_DeactivateFont calls. This will be an empty
#                 string if there were no errors.
#               fontnames: A list of fontnames that could be deactivated.
#                 This will be used by the GUI to change the state of the
#                 checkboxes in the font tree.

proc FM_DeactivateGroup { groupname app } {
    global fm_Groups
    set errormsg ""
    set fontnames ""

    foreach fontname $fm_Groups($groupname) {
	if { [ FM_QueryFontActive $fontname $app ] == "ON" } then {
	    set result [ FM_DeactivateFont $fontname $app ]
	    if [ string length $result ] then {
		if [ string length $errormsg ] then {		
		    append errormsg \n $result
		} else {
		    set errormsg $result
		}
	    } else {
		lappend fontnames $fontname
	    }
	}
    }
    return [ list $errormsg $fontnames ]
}

#-------------------------------------------------------------------------
#
# FM_QueryGroupActive
#
#-------------------------------------------------------------------------

# Are all, none or some fonts active in this group in this application?
# Parameters:   groupname (string): the group to scan.
#               app (string) the application prefix.
# Return value: "ON" if ALL fonts in this group are active in this application,
#               "OFF" if NO fonts in this group are active,
#               "SOME" if SOME fonts in this group are active,
#               "N/A" if there is no font which could be activated.

proc FM_QueryGroupActive { groupname app } {
    global fm_Fonts fm_Groups
    set active 0
    set not_act 0
    set available 0

    foreach fontname $fm_Groups($groupname) {
	set result [ FM_QueryFontActive $fontname $app ]

	if { ! [ string compare $result "ON" ] } then {
	    incr active
	    incr available
	} elseif { ! [ string compare $result "OFF" ] } then {
	    incr not_act
	    incr available
	} else {
	    # incr not_act
	    # Uncomment the line if this proc should return
	    # "ON" only if really ALL fonts this group are active.
	}

	if { $active && $not_act } then { 
	    return "SOME"
	}
    }

    if $active then {
	return "ON"
    } elseif $available then {
	return "OFF"
    } else {
	return "N/A"
    }
}


#-------------------------------------------------------------------------
#
# FM_QueryAliases
#
#-------------------------------------------------------------------------

# The list of aliases for a font.
# Parameters:   fontname (string): the desired font.
#               app (string): application prefix.
# Return value: a list of alias names for this font in this application.

proc FM_QueryAliases { fontname app } {
    global fm_Fonts
    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
    return [ ${app}_QueryAliases $fontinfo ]
}

#-------------------------------------------------------------------------
#
# FM_AddAlias
#
#-------------------------------------------------------------------------

# Adds an alias name to an installed font.
# Parameters:   fontname (string): the font for which to add the alias.
#               apps (list): a list of application prefixes where to add the alias.
#               aliasname (string): the desired alias name.
# Return value: Eventual error messages returned by the underlying modules.
#               An empty string if no errors were returned.

proc FM_AddAlias { fontname apps aliasname } {
    global fm_Fonts
    set retval ""

    foreach app $apps {
	set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
	set msg [ ${app}_AddAlias $fontinfo $aliasname ]

	if [ string length $msg ] then {
	    if [ string length $retval ] then {		
		append retval \n $msg
	    } else {
		set retval $msg
	    }
	}
    }
    return $retval
}

#-------------------------------------------------------------------------
#
# FM_RemoveAlias
#
#-------------------------------------------------------------------------

# Removes an alias from an installed font.
# Parameters:   app (string): application prefix.
#               fontname (string): name of the font to which the alias belongs.
#	        aliasname (string): the alias name to be removed.
# Return value: Eventual error messages returned by the underlying modules.
#               An empty string if no errors were returned.

proc FM_RemoveAlias { app fontname aliasname } {
    global fm_Fonts
    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
    return [ ${app}_RemoveAlias $fontinfo $aliasname ]
}

#-------------------------------------------------------------------------
#
# FM_RenameAlias
#
#-------------------------------------------------------------------------

# Renames an alias to another one.
# Parameters:   app (string): application prefix.
#               fontname (string): the font for which to rename the alias.
#		oldalias (string): the old alias name.
#               newalias (string): the new alias name.
# Return value: Eventual error messages returned by the underlying modules.
#               An empty string if no errors were returned.

proc FM_RenameAlias { app fontname oldalias newalias } {
    global fm_Fonts
    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
    return [ ${app}_RenameAlias $fontinfo $oldalias $newalias ]
}

#-------------------------------------------------------------------------
#
# FM_QueryMAS
#
#-------------------------------------------------------------------------

# Are multiple aliases supported in the specified application?
# Parameter:    app (string): application prefix.
# Return value: "1" if the application supports multiple aliases,
#               "" otherwise.

proc FM_QueryMAS { app } {
    return [ ${app}_QueryMAS ]
}

#-------------------------------------------------------------------------
#
# FM_QueryAppName
#
#-------------------------------------------------------------------------

# Get the name of the specified application.
# Parameter:    app (string): application prefix.
# Return value: The name of the Application.

proc FM_QueryAppName { app } {
    return [ ${app}_QueryAppName ]
}

#-------------------------------------------------------------------------
#
# FM_QueryIcon
#
#-------------------------------------------------------------------------

# Get an icon to use for the specicied application
# Parameter:    app (string): application prefix.
# Return value: The image identifier or "".

proc FM_QueryIcon { app } {
    return [ ${app}_QueryIcon ]
}

#-------------------------------------------------------------------------
#
# FM_QueryHasPreview
#
#-------------------------------------------------------------------------

# Is a preview available the specified font?
# Parameter:    fontname (string): the name of the font.
# Return value: "1" if the preview is available, "0" otherwise.

proc FM_QueryHasPreview { fontname } {
    global fm_Fonts
    set app [ lindex $fm_Fonts($fontname) 1 ]
    return [ ${app}_QueryHasPreview ]
}

#-------------------------------------------------------------------------
#
# FM_QueryFilePatterns
#
#-------------------------------------------------------------------------

# Return file name patterns for font files
# Parameters:   none
# Return value: a list which the elements of are lists with two elements
#               {patterns text}. "patterns" is a list of file patterns, and
#               "text" is a string that describes these patterns.

proc FM_QueryFilePatterns { } {
    global FontTypeMods
    set allpatterns ""

    foreach app $FontTypeMods {
	set patterns [ ${app}_QueryFilePatterns ]
	set text "[ FM_QueryAppName $app ] font files"
	set allpatterns [ concat $allpatterns $patterns ]
	lappend patternlist [ list $patterns $text ]
    }

    lappend patternlist [ list {*} "All files" ]

    return [ concat [ list [ list [ lsort $allpatterns ] "All font files" ] ] \
	    $patternlist ]
}

#-------------------------------------------------------------------------
#
# FM_CreatePreviewImage
#
#-------------------------------------------------------------------------

# Show a preview image for the specified font.
# Parameters:   fontname (string): the name of the font.
#               imageid (string): the id of the image that should be created.
# Return value: Eventual error message returned by the underlaying module.
#               An empty string if no errors were returned.

proc FM_CreatePreviewImage { fontname imageid } {
    global fm_Fonts
    set app [ lindex $fm_Fonts($fontname) 1 ]
    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
    return [ ${app}_CreatePreviewImage $fontinfo $imageid ]
}

#-------------------------------------------------------------------------
#
# FM_QueryPrefsFrame
#
#-------------------------------------------------------------------------

# Gathers children preferences frames from plugin modules
# Parameter:    parent (string): identifier of the parent container frame.
#               fontname (string): the font the prefs should be changed of.
# Return value: Identifier of the created frame,which can be packed by the GUI.

proc FM_QueryPrefsFrame { parent fontname } {
    global AppMods fm_Fonts
    set fontinfo [ concat [ list $fontname ] $fm_Fonts($fontname) ]
    frame $parent.cfgframe
    foreach module $AppMods {
	set result [ ${module}_QueryPrefsFrame $parent.cfgframe $fontinfo ]
	if [ string length $result ] then {
	    pack $result -in $parent.cfgframe -side top -fill x
	}
    }
    return $parent.cfgframe
}

#-------------------------------------------------------------------------
#
# FM_OnPrefsClosed
#
#-------------------------------------------------------------------------

# Called when the Preferences dialog is closed.
# Parameters:   none
# Return value: all strings returned by the app_CheckPrefsInput should be 
#               appended together and returned.

proc FM_OnPrefsClosed { } {
    global AppMods fm_DefaultGroup
    foreach module $AppMods {
	set result [ ${module}_CheckPrefsInput ]
	if [ string length $result ] then {
	    foreach line $result {
		append retval \n$line
	    }
	}
    }
    if [ info exist retval ] then {
	# Input is not valid, return error message
	return [ string trimleft $retval ]
    }
    return ""
}
