/* $Copyright: $
 * Copyright (c) 1997 by Steve Baker (ice@mama.indstate.edu)
 * All Rights reserved
 *
 * This software is provided as is without any express or implied
 * warranties, including, without limitation, the implied warranties
 * of merchant-ability and fitness for a particular purpose.
 */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <ctype.h>
#include <linux/quota.h>
#include <sys/file.h>
#include <pwd.h>
#include <limits.h>
#include <time.h>

#define PATH_DHLOGS	"/var/log/dh"

static char *version = "$Version: $ diskhogs v0.5 (c) 1997 by Steve Baker $";

#define scopy(s)  (strcpy(malloc(strlen(s)+1),s))
enum {FALSE=0, TRUE=1};
enum {IGNORE_USERS, ONLY_FS, EXCLUDE_FS};

struct fs {
  char *device;
  char *mountpt;
  char *uqfile;
  char *gqfile;
  char usrquota, grpquota;
  struct user *usrs, *usorted;
  u_long nusrs, ublks;
  struct grp *grps, *gsorted;
  u_long ngrps, gblks;
  struct fs *nxt;
};

struct user {
  u_short uid;
  u_long blks;
  u_long soft, hard;
  int expire;
  struct user *nxt, *snxt;
};

struct grp {
  u_short gid;
  u_long blks;
  u_long soft, hard;
  int expire;
  struct grp *nxt, *snxt;
};

/* excluded users */
struct xuser {
  u_short uid;
  struct xuser *nxt;
} *xusers = NULL;

/* prefered/excluded filesystems */
struct pfs {
  char *mountpt;
  struct pfs *nxt;
} *pfs = NULL, *xfs = NULL;

/* utmp like struct for disk usage over time functions */
struct dtmp {
  time_t time;
  u_short uid;
  u_long blks;
};

struct ltmp {
  u_short uid;
  u_long blks;
  struct ltmp *nxt;
} *last = NULL;

/* powers of 2 please */
#define HASH_SIZE	256
#define hash(u)		(u&(HASH_SIZE-1))
struct usr {
  char *uname;
  u_short uid;
  struct usr *nxt;
} *users[HASH_SIZE];


u_long top = 10;
char *cmd, path[1025];

char **split(), *itos();
int isnum(), main();
struct fs *get_filesystems(), *merge(), *printit();
void sizeup_filesystem(), usage(), do_record(), do_list();
void size_user(), size_group();
char *change(), *date(), *username(u_short uid);

/*
 * diskhogs [-i users] [-x fslist] [-12] [-t #] [-r] [[-f] fslist] [-l]
 */
int main(int argc,char **argv)
{
  struct fs *hfs, *fs, *p;
  int i, j, n, maxcols;
  char listtype = ONLY_FS, record = FALSE, list = FALSE, *t;
  struct xuser *xu;
  struct passwd *pw;
  struct pfs *pf, *xf;

  for(i=0;i<HASH_SIZE;i++) users[i] = NULL;

  maxcols = 3;

  t = strrchr(argv[0],'/');
  if (t == NULL) t = argv[0];
  else t++;
  cmd = scopy(t);

  for(n=i=1;argv[i];i=n) {
    n++;
    if (argv[i][0] == '-') {
      for(j=1;argv[i][j];j++) {
	switch(argv[i][j]) {
	  case 'i':
	    listtype = IGNORE_USERS;
	    break;
	  case 'f':
	    listtype = ONLY_FS;
	    break;
	  case 'x':
	    listtype = EXCLUDE_FS;
	    break;
	  case 't':
	    if (!argv[n]) usage();
	    top = atoi(argv[n++]);
	    break;
	  case '1':
	    maxcols = 1;
	    break;
	  case '2':
	    maxcols = 2;
	    break;
	  case 'r':
	    record = TRUE;
	    break;
	  case 'l':
	    list = TRUE;
	    break;
	  default:
	    usage();
	}
      }
    } else {
      switch(listtype) {
	case ONLY_FS:
	  pf = malloc(sizeof(struct pfs));
	  pf->mountpt = scopy(argv[i]);
	  pf->nxt = pfs;
	  pfs = pf;
	  break;
	case EXCLUDE_FS:
	  xf = malloc(sizeof(struct pfs));
	  xf->mountpt = scopy(argv[i]);
	  xf->nxt = xfs;
	  xfs = xf;
	  break;
	case IGNORE_USERS:
	  xu = malloc(sizeof(struct xuser));
	  if (isnum(argv[i])) xu->uid = atoi(argv[i]);
	  else {
	    pw=getpwnam(argv[i]);
	    if (!pw) {
	      free(xu);
	      continue;
	    }
	    xu->uid = pw->pw_uid;
	  }
	  xu->nxt = xusers;
	  xusers = xu;
	  break;
      }
    }
  }

  hfs = fs = get_filesystems(pfs,xfs);

  if (fs == NULL) {
    printf("diskhogs: no filesystems with quotas.\n");
    exit(1);
  }

  for(p=fs; p; p=p->nxt) {
    sizeup_filesystem(p);
    if (record) do_record(p);
    else if (list) do_list(p);
  }
  if (record || list) exit(0);

  printf("Top %ld diskhogs:\n\n",top);

  while((fs = printit(fs,maxcols)));

  fs = merge(hfs);
  exit(0);
}

void usage()
{
  printf("usage: %s [[-f] fslist] [-x fslist] [-i userlist] [-12r] [-t #] [-l]\n",cmd);
  exit(1);
}

int isnum(char *s)
{
  while(isdigit(*s)) s++;
  if (*s) return FALSE;
  return TRUE;
}

/*
 * Format output in up to 3 columns for each filesystem.
 * Probably ought to do uid->pw lookup in the
 * size_{user|group} functions to avoid the slow output.
 */
struct fs *printit(struct fs *fs, int maxcols)
{
  struct fs *f[3];
  struct user *u[3];
/*  struct passwd *pw; */
  struct xuser *xu;
  int l, i, j, n, h;
  char c1, c2, tbuf[80];
  u_long tblks[3], usrs[3];

  for(n=0;n<maxcols && fs;n++,fs=fs->nxt) {
    f[n] = fs;
    u[n] = fs->usrs;
    tblks[n] = usrs[n] = 0;
  }
  h = n < 3? 18: 13;

  for(i=0;i<n;i++) {
    l = strlen(f[i]->mountpt)/2;
    printf("%*s%-*s",h-l,"",h+l,f[i]->mountpt);
  }
  printf("\n");

  for(i=0;i<top;i++) {
    for(j=0;j<n;j++) {
      for(xu=xusers;xu && u[j];) {
	if (xu->uid == u[j]->uid) {
	  u[j] = u[j]->nxt;
	  xu=xusers;
	} else xu=xu->nxt;
      }
      if (u[j]) {
/*	pw = getpwuid(u[j]->uid); */
	c2 = ' ';
	c1 = (u[j]->expire)? ((u[j]->hard && u[j]->blks > u[j]->hard)? c2='!' : '!') : ((u[j]->soft && u[j]->blks > u[j]->soft)? '^' : (u[j]->soft? '+' : ' '));
	printf("%*lu%c%c %-*s",h-2,u[j]->blks,c1,c2,j+1==n?1:h-1,username(u[j]->uid)/*pw==NULL?itos(u[j]->uid):pw->pw_name*/);
	tblks[j] += u[j]->blks;
	usrs[j]++;
	u[j] = u[j]->nxt;
      } else printf("%*s   %-*s",h-2,"-",j+1==n?1:h-1,"-");
    }
    printf("\n");
  }
  printf("\n");
  for(i=0;i<n;i++) {
    sprintf(tbuf,"(%5.2f MB/hog)",usrs[i]?((float)tblks[i]/(float)usrs[i])/1024:(float)0);
    printf("%*lu %-*s",h-5,tblks[i],h+4,tbuf);
  }
  printf("\n\n");

  return fs;
}

char *itos(int i)
{
  static char ibuf[50];

  sprintf(ibuf,"%d",i);
  return ibuf;
}

void sizeup_filesystem(struct fs *fs)
{
  if (fs->usrquota) {
    fs->nusrs = 0;
    fs->ublks = 0;
    size_user(fs);
  }
  if (fs->grpquota) {
    fs->ngrps = 0;
    fs->gblks = 0;
    size_group(fs);
  }
}

void size_user(struct fs *fs)
{
  struct dqblk q;
  struct user *u, *pp, *cp;
  int fd, uid;

  quotactl(QCMD(Q_SYNC,USRQUOTA),fs->device,0,0);

  if ((fd = open(fs->uqfile,O_RDONLY)) < 0) {
    fprintf(stderr,"%s: error opening user quota file for fs %s [%s]\n",cmd,fs->mountpt,fs->uqfile);
    return;
  }
  
  for (uid=0;read(fd,&q,sizeof(struct dqblk)) == sizeof(struct dqblk);uid++) {
    if (!q.dqb_curblocks) continue;
    u = malloc(sizeof(struct user));
    u->uid = uid;
    u->blks = q.dqb_curblocks;
    u->soft = q.dqb_bsoftlimit;
    u->hard = q.dqb_bhardlimit;
    u->nxt = u->snxt = NULL;
    u->expire = 0;
    if (u->hard && u->blks > u->hard) u-> expire = 1;
    else if (u->soft && u->blks > u->soft) u->expire = (time(0) > q.dqb_btime)? 1 : 0;

    if (fs->usrs == NULL) fs->usrs = u;
    else {
      for(pp=cp=fs->usrs;cp;cp=cp->nxt) {
	if (cp->blks < u->blks) {
	  u->nxt = cp;
	  if (cp == fs->usrs) fs->usrs = u;
	  else pp->nxt = u;
	  break;
	}
	pp = cp;
      }
      if (!cp) pp->nxt = u;
    }

    if (fs->usorted == NULL) fs->usorted = u;
    else {
      for(pp=cp=fs->usorted;cp;cp=cp->snxt) {
	if (u->uid < cp->uid) {
	  u->snxt = cp;
	  if (cp == fs->usorted) fs->usorted = u;
	  else pp->snxt = u;
	  break;
	}
	pp = cp;
      }
      if (!cp) pp->snxt = u;
    }

    fs->nusrs++;
    fs->ublks += u->blks;
  }
  close(fd);
}

void size_group(struct fs *fs)
{
}

struct fs *merge(struct fs *fs)
{
  return NULL;
}

struct fs *get_filesystems(struct pfs *pfs, struct pfs *xfs)
{
  static char *qfnames[] = INITQFNAMES
  struct fs *tfs, *hfs = NULL, *p;
  char buf[1025];
  FILE *fd;
  char **s, **o, *uqfile, *gqfile;
  int i, n, uq, gq;
  struct pfs *pf;

  if ((fd = fopen("/etc/fstab","r")) == NULL) {
    fprintf(stderr,"%s: couldn't open fstab.\n",cmd);
    exit(1);
  }
  while (fgets(buf,1024,fd) != NULL) {
    if (buf[0] == '#') continue;
    s = split(buf,&n," \t");
    if (n < 4) {
      free(s);
      continue;      
    }
    uqfile = gqfile = NULL;
    o = split(s[3],&n,",");

    if (pfs) {
      for(pf=pfs;pf;pf=pf->nxt)
	if (!strcmp(pf->mountpt,s[1])) break;
      if (pf == NULL) continue;
    }

    for(pf=xfs;pf;pf=pf->nxt)
      if (!strcmp(pf->mountpt,s[1])) break;
    if (pf != NULL) continue;

    for(uq=gq=i=0;i<n;i++) {
      if (!strncmp("usrquota",o[i],8)) {
	uq = TRUE;
	if (o[i][8] == '=' && o[i][9]) uqfile = scopy(o[i]+9);
      }
      if (!strncmp("grpquota",o[i],8)) {
	gq = TRUE;
	if (o[i][8] == '=' && o[i][9]) gqfile = scopy(o[i]+9);
      }
    }
    if (uq || gq) {
      tfs = malloc(sizeof(struct fs));
      tfs->device = scopy(s[0]);
      tfs->mountpt = scopy(s[1]);
      if (!uqfile) {
	sprintf(path,"%s/%s.%s",tfs->mountpt,QUOTAFILENAME,qfnames[USRQUOTA]);
	tfs->uqfile = scopy(path);
      } else tfs->uqfile = uqfile;
      if (!gqfile) {
	sprintf(path,"%s/%s.%s",tfs->mountpt,QUOTAFILENAME,qfnames[GRPQUOTA]);
	tfs->gqfile = scopy(path);
      } else tfs->gqfile = gqfile;

      tfs->usrquota = uq;
      tfs->grpquota = gq;
      tfs->usrs = tfs->usorted = NULL;
      tfs->grps = tfs->gsorted = NULL;
      tfs->nxt = NULL;

/* put it at the end of the list, just like in the fstab */

      if (hfs == NULL) hfs = tfs;
      else {
	for(p=hfs;p->nxt;p=p->nxt);
	p->nxt = tfs;
      }
    }
    free(s);
    free(o);
  }
  return hfs;
}

char **split(char *s,int *n,char *b)
{
  char **arg;

  *n = 0;
  arg = malloc(sizeof(char *) * 100);

  arg[*n] = strtok(s,b);
  while ((arg[++*n] = strtok(NULL,b)));
  return arg;
}

void do_record(struct fs *fs)
{
  char path[PATH_MAX+1], fname[PATH_MAX+1], *s;
  struct user *u;
  struct dtmp d,t;
  int last, dlog;
  time_t ct = time(0);

  strcpy(path,fs->mountpt);
  if (!strcmp(path,"/")) strcpy(path,"root");
  else
    for(s=path;(s=strchr(s,'/'));s++) *s = ':';

  sprintf(fname,"%s/%s.last",PATH_DHLOGS,path);
  last = open(fname,O_RDWR|O_CREAT,0600);
  if (last < 0) {
    fprintf(stderr,"%s: error opening last report file [%s/%s.last].\n",cmd,PATH_DHLOGS,path);
    exit(1);
  }
  sprintf(fname,"%s/%s.dlog",PATH_DHLOGS,path);
  dlog = open(fname,O_WRONLY|O_CREAT|O_APPEND,0600);
  if (dlog < 0) {
    fprintf(stderr,"%s: error opening report log [%s/%s.dlog].\n",cmd,PATH_DHLOGS,path);
    exit(1);
  }

  t.time = ct;
  for(u=fs->usorted;read(last,&d,sizeof(struct dtmp)) == sizeof(struct dtmp);) {
    while(u && u->uid < d.uid) {
      t.uid = u->uid;
      t.blks = u->blks;
      write(dlog,&t,sizeof(struct dtmp));
      u=u->snxt;
    }
    if (u->uid == d.uid) {
      if (u->blks != d.blks) {
	t.uid = u->uid;
	t.blks = u->blks;
	write(dlog,&t,sizeof(struct dtmp));
      }
      u=u->snxt;
    } else {
      t.uid = d.uid;
      t.blks = 0;
      write(dlog,&t,sizeof(struct dtmp));
    }
  }
  for(;u;u=u->snxt) {
    t.uid = u->uid;
    t.blks = u->blks;
    write(dlog,&t,sizeof(struct dtmp));
  }

  lseek(last,0,SEEK_SET);
  ftruncate(last,0);

  for(u=fs->usorted;u;u=u->snxt) {
    t.uid = u->uid;
    t.blks = u->blks;
    write(last,&t,sizeof(struct dtmp));
  }
  
  close(last);
  close(dlog);
}

void do_list(struct fs *f)
{
  int fd;
  struct dtmp d;
  char path[PATH_MAX+1], fname[PATH_MAX+1], *s;

  strcpy(path,f->mountpt);
  if (!strcmp(path,"/")) strcpy(path,"root");
  else
    for(s=path;(s=strchr(s,'/'));s++) *s = ':';

  sprintf(fname,"%s/%s.dlog",PATH_DHLOGS,path);

  fd = open(fname,O_RDONLY);
  if (fd < 0) {
    fprintf(stderr,"%s: error opening file '%s'.\n",cmd,fname);
    return;
  }
  while(read(fd,&d,sizeof(struct dtmp)) == sizeof(struct dtmp)) {
    printf("%s: %8s [%5d]  %10ld blocks, %s\n",date(d.time),username(d.uid),d.uid,d.blks,change(d));
  }
  close(fd);
}

char *date(time_t t)
{
  static char *month[] = {
    "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"
  };
  static char dbuf[40];
  struct tm *tm;

  tm = localtime(&t);
  sprintf(dbuf,"%s %02d %02d, %02d:%02d:%02d",month[tm->tm_mon],tm->tm_mday,
	  tm->tm_year%100,tm->tm_hour,tm->tm_min,tm->tm_sec);
  return dbuf;
}

char *username(u_short uid)
{
  struct usr *u, *p, *n;
  char nbuf[20];
  short h = hash(uid);
  struct passwd *pwd;

  for(u=p=users[h];u;u=u->nxt) {
    if (u->uid == uid) return u->uname;
    if (u->uid > uid) break;
    p = u;
  }
  n = malloc(sizeof(struct usr));
  pwd = getpwuid(uid);
  if (pwd == NULL) {
    sprintf(nbuf,"%d",uid);
    n->uname = scopy(nbuf);
  } else n->uname = scopy(pwd->pw_name);
  n->uid = uid;
  n->nxt = u;
  if (u == users[h]) users[h] = n;
  else p->nxt = n;
  return n->uname;
}

char *change(struct dtmp d)
{
  struct ltmp *l, *p, *t;
  static char lbuf[40];

  for (l=p=last;p;p=p->nxt) {
    if (p->uid == d.uid) {
      sprintf(lbuf,"%+ld blocks", d.blks - p->blks);
      p->blks = d.blks;
      return lbuf;
    }
    if (p->uid > d.uid) break;
    l = p;
  }
  t=malloc(sizeof(struct ltmp));
  t->uid = d.uid;
  t->blks = d.blks;
  t->nxt = p;
  if (p == last) last = t;
  else l->nxt = t;
  sprintf(lbuf,"initial size");
  return lbuf;
}
