/****************************************************************************
    AwReader.cpp 
        2 Aug 1996 Created by LD.
        5 Aug 1996 Added Support for accumulating nacctd statistic by LD
*****************************************************************************/

#include "memdata.h"
#include "utmpdata.h"
#include "handshakes.h"

#pragma implementation
#include "awreader.h"
#include "anreader.h"

#include <stdio.h>

char ModuleName [] = "awreader";


static void PrintUtmp ( RootWtmp* rw_pr );
static void FixSums  (RootWtmp* rw_pr );

RootBase UserRoot;
RootWtmp WtmpToday;
struct Handshake_st ModuleSig = { AWREADER_SIG, VERSION };
struct Handshake_st RespSig;

struct AWReaderRequest Request;

time_t  LastWtmpCheck; // moment of last wtmp checking 
RootWtmp WtmpOverall; // overall statistic

time_t StartTime, EndTime, DayIndex;


int /* OwlMain */ main ( int, char** )
{	
  FILE *PipeIn=stdin, *PipeOut=stdout;
	
  // sending and checking the signature
  if ( 1 != fwrite ( (void*) &ModuleSig, 
    sizeof ( ModuleSig), 1, PipeOut ))
  {
    LOG ( 1, "awreader:broken pipe, exiting\n" );
    exit ( 1 );
  }
  fflush ( PipeOut );
  
#ifndef NO_DEBUG
  LOG ( 4, "awreader: signature sent, waiting for response\n" );
#endif

  if ( 1 != fread ( (void* ) &RespSig,
    sizeof ( RespSig ), 1, PipeIn ))
  {
    LOG ( 1, "awreader:broken pipe, exiting\n" );
    exit ( 1 );
  }
  
  // checking the signature
  if ( memcmp ( (void* ) RespSig.ModuleSig, ADMIND_SIG, 2) )
  {
    LOG ( 1, "awreader: wrong signature received, exiting\n");
    exit ( 1 );
  }
  
  if ( RespSig. version != VERSION )
  {
    LOG ( 1,"awreader: wrong admind version, exiting\n" );
    exit ( 1 );
  }

#ifndef NO_DEBUG 
  LOG ( 4, "awreader: admind signature received and checked okay.\n" );
#endif
  
  // reading AWReader Request 
  if ( 1 != fread ( (void* ) &Request, sizeof ( Request ), 1, PipeIn ))
  {
    LOG (1, "awreader:broken pipe while reading request\n" );
    exit ( 1 );
  }
  
  // reading UserRoot
  if ( UserRoot. ReadFromStream ( PipeIn) )
  {
    LOG (1, "awreader:reading UserRoot failed (broken pipe?)\n" );
    exit ( 1 );
  }

#ifndef NO_DEBUG
  LOG (4, "awreader: request and UserRoot are read okay\n" );
#endif     
  
  
  LastWtmpCheck = time ( NULL );
  
  // decoding request   
  if ( Request. OpCode == OC_TODAY )
    DetermineCurrentDay ( LastWtmpCheck, StartTime, EndTime, DayIndex );
  else if ( Request . OpCode == OC_STAT )
  {
    StartTime = Request . Start;
    EndTime = Request . End;
  }
  else 
  {
    LOG ( 1, "awreader: wrong Operation Code in request\n" );
    exit ( 1 );
  }
  
#ifndef NO_DEBUG
  LOG ( 4, "awreader: request decoded okay, starting to collect statistic\n" );
#endif

  CollectWtmpData ( &WtmpToday, StartTime, EndTime );
  
#ifdef NACCTD_SUPPORT  
  CollectNacctd ( &WtmpToday, StartTime, EndTime );
#endif  

  FixSums ( &WtmpToday );
  
#ifndef NO_DEBUG
  LOG ( 4, "awreader:done collecting wtmp data\n" );
  LOG ( 4, "awreader:starting transfer of stat requested\n" );
#endif

  if ( WtmpToday . SendToStream ( PipeOut ) )
  {
    LOG ( 1, "awreader: error while transferring stat requested\n" );
    exit ( 1 );
  }
  
#ifndef NO_DEBUG
  LOG ( 4, "awreader: done transferring WtmpToday\n" );
#endif

  // transferring LastWtmpCheck, and what else ?
  if ( 1 != fwrite ( (void* )&LastWtmpCheck,
    sizeof ( LastWtmpCheck ),1 ,PipeOut ) )
  {
    LOG ( 1, "awreader: broken pipe while transferring bonus\n" );
    exit ( 1 );
  }
  
#ifndef NO_DEBUG
  LOG ( 4, "awreader:bonus transferred okay, starting to transfer Overall\n" );
#endif

  if ( WtmpOverall . SendToStream ( PipeOut) )
  {
    LOG ( 1, "awreader:just a warning:broken pipe while transerring WtmpOverall\n");
    exit ( 1 );  
  }
#ifndef NO_DEBUG
  LOG ( 4, "awreader: all done okay, exiting\n" );
#endif

  fflush ( PipeOut );
  fclose ( PipeOut );
  fclose ( PipeIn );
      
  return 0;
}


/****************************************************************
              utmp handling 
*****************************************************************/


static int OverallFlushed =0; // if Overall ( before StartTime ) flushed

// debug
static unsigned inconsistence_count=0;

static void FlushAll ( RootWtmp* rw_pr, time_t TimeBorder, int CrashYes );
/* static void FlushOverall ( void ); */
static void FlushAllMakeTails ( RootWtmp* flush_rw_pr, RootWtmp* open_rw_pr,
  time_t TimeBorder );
static void UserProcess ( RootWtmp* rw_pr, struct utmp* utmp_pr );
static void UserLogout ( RootWtmp* rw_pr, struct utmp* utmp_pr );
// static void CheckAintUsersReallyLogged ( void );
static void NewTime( RootWtmp* rw_pr, struct utmp* utmp_pr );

// this function assumes we have already structure under UserRoot built.
// the function will accumulate overall and today's time usage for 
// users pointed out in UserRoot.
// WARNING: UserRoot SHOULD NOT contain predefined system users 
//          as 'shutdown' or 'reboot' :)

int CollectWtmpData (RootWtmp* rw_pr, time_t StartTime, time_t EndTime ) 
{

  inconsistence_count = 0;

  //TodayPreempted = 0;

  // 1. clear current WtmpRoot & given RootWtmp
  while ( WtmpOverall . tail ) delete WtmpOverall . tail ;
  while ( rw_pr -> tail ) delete rw_pr -> tail;
  
  //while ( WtmpToday . tail ) delete WtmpToday . tail;

  // 1.1 determine current day, mainly
  // time_t CurrMoment = time ( NULL ); 
  LastWtmpCheck = time ( NULL );

  // 2. read wtmp
  struct utmp* utmp_pr;
  struct User_st* user_pr;
  RootWtmp* curr_rw_pr = &WtmpOverall;
  
#ifndef WTMP_PATH
  utmpname ( _PATH_WTMP );
#ifndef NO_DEBUG
  Log2 ( 5, "awreader:Using wtmp file %s\n", _PATH_WTMP );
#endif
#else
  utmpname ( WTMP_PATH );
#ifndef NO_DEBUG
  Log2 (5, "awreader:Using wtmp file %s\n", WTMP_PATH );
#endif
#endif
  setutent ();

  for ( 
    utmp_pr = getutent();
    utmp_pr;
    utmp_pr = getutent ()
  ) 
  { 
    if ( OverallFlushed == 0 && utmp_pr -> ut_time > StartTime )
    {
      FlushAllMakeTails ( &WtmpOverall, rw_pr, StartTime );
      OverallFlushed =1;
      curr_rw_pr = rw_pr;
    } 
    if ( OverallFlushed && utmp_pr -> ut_time > EndTime )
    {
      FlushAll ( rw_pr, /* utmp_pr -> ut_time */ EndTime, 0 );
      break;
    }
    // here analysis begins  
    // 2.1. check if it is shutdown or boot: 
    if ( ! strcmp ( utmp_pr -> ut_user, "shutdown" ) )
      FlushAll ( curr_rw_pr, utmp_pr -> ut_time,0 ); // flush all open users statiscs without a crash warning 
    if ( ( ! strcmp ( utmp_pr -> ut_user, "reboot" )) ||
         utmp_pr -> ut_type == BOOT_TIME )
      FlushAll ( curr_rw_pr,utmp_pr -> ut_time,1 ); // flush all with a crash warning: system was probably crashed
    
    // 2.2 or, maybe it is a new user login? :) 
    if ( utmp_pr -> ut_type == USER_PROCESS )
      UserProcess ( curr_rw_pr, utmp_pr );
    
    // 2.3. or, maybe, it is a user logout ? :)
    if ( utmp_pr -> ut_type == DEAD_PROCESS )
      UserLogout ( curr_rw_pr, utmp_pr );

    // 2.4. or, finally, maybe it was a time correction?
    // so far is not implemented, though :)
    if ( utmp_pr -> ut_type == NEW_TIME ) 
      NewTime ( curr_rw_pr, utmp_pr );
    
  }
  
  LOG (5, "CollectWtmpData: Main loop passed \n" );
  sprintf (LogBuff, "CollectWtmpData: Inconsistence Count=%u\n", 
    inconsistence_count );
  Log ( 4 );
  
  if ( OverallFlushed == 0 /* && CurrMoment >= StartTime */ )
  {
    FlushAllMakeTails ( &WtmpOverall, rw_pr, StartTime);
    OverallFlushed = 1;
    curr_rw_pr = rw_pr;
  }
  
  // close access to wtmp
  endutent();
  // CheckAintUsersReallyLogged ();
  return 0;
}

// closes flush_pr and preempts 'tails' to open_pr 
// using TimeBorder
static void FlushAllMakeTails ( RootWtmp* flush_rw_pr, 
  RootWtmp* open_rw_pr, time_t TimeBorder )
{

  // if ( TodayPreempted ) return; // already preemted :)
  
  // check whether Overall has USER_PROCESS 'tails' open
  // and preemt them to 
  UserLimitInfo *flush_uli_pr, *open_uli_pr ;
  for (
    flush_uli_pr = flush_rw_pr -> base ;
    flush_uli_pr ;
    flush_uli_pr = flush_uli_pr -> next
  )
  {
    TtyUsageInfo *tui_pr, *open_tui_pr;
    for (
      tui_pr = flush_uli_pr -> TtyBase . base;
      tui_pr;
      tui_pr = tui_pr -> next
    )
    if ( tui_pr -> UsageProps & TUI_LOGGED_NOW )
    {
      // close the usage with low boundary as TimeBorder
      if ( TimeBorder > tui_pr -> LastLog )
        tui_pr -> Usage += TimeBorder - tui_pr -> LastLog;
      tui_pr -> LastLogout = TimeBorder;    
      tui_pr -> UsageProps &= ~TUI_LOGGED_NOW;
      // create new usage in WtmpToday
      open_uli_pr = open_rw_pr -> FindUserLimitInfo ( flush_uli_pr -> UserName );
      if ( open_uli_pr == NULL )
        open_uli_pr = open_rw_pr -> AddUserLimitInfo ( flush_uli_pr -> UserName );
      open_tui_pr = open_uli_pr -> FindTtyUsageInfo ( tui_pr -> Line );
      if ( open_tui_pr == NULL )
      {
        open_tui_pr = open_uli_pr -> AddTtyUsageInfo ( tui_pr -> Line );
        strcpy ( open_tui_pr -> AbbrevLine, tui_pr -> AbbrevLine );
      }
      // fill in the new usage
      open_tui_pr -> LastLog = TimeBorder; // simulate user logging in at today_start
      open_tui_pr -> LastPid = tui_pr -> LastPid;
      open_tui_pr -> UsageProps = tui_pr -> UsageProps | TUI_LOGGED_NOW;
      // clean user reputation for today if he had spots on one earlier :):
      open_tui_pr -> UsageProps &= ~TUI_HAS_CRASHES;

      // dat's all, seemingly :)
    }
  }  

#ifndef NO_DEBUG  
  // debug
  LOG(5, "FlushAllMakeTales passed.\n" );   
#endif  

  // TodayPreempted = 1;    

}

static void FlushAll ( RootWtmp* rw_pr, time_t TimeBorder, int IsCrash )
{

  // debug 
  // printf ( "Entering FlushAll\n" );

//  RootWtmp* rw_pr;
//  // 1.1 check -- is it today's login ? 
//  if ( utmp_pr -> ut_time >= DayStart && utmp_pr -> ut_time <= DayEnd )
//  {
//    rw_pr = &WtmpToday ;
//    FlushOverall (); // make sure overall is closed :)
//  }
//  else 
//    rw_pr = &WtmpOverall ;

  // 2. flush all users
  UserLimitInfo* uli_pr;
  for (
    uli_pr = rw_pr -> base;
    uli_pr ;
    uli_pr = uli_pr -> next 
  ) 
  {
    TtyUsageInfo* tui_pr;
    for ( 
      tui_pr = uli_pr -> TtyBase . base;
      tui_pr;
      tui_pr = tui_pr -> next 
    )
    if ( tui_pr -> UsageProps & TUI_LOGGED_NOW )
    {
      // flush it !
      // hmm.. probably DEAD_PROCESS could be encountered after shutdown...
      // probably will need some rectifications if have problems with it.
      if ( TimeBorder > tui_pr -> LastLog 
#ifndef COUNT_CRASHTIME
        && IsCrash == 0
#endif              
      )
      {
        tui_pr -> Usage += TimeBorder - tui_pr -> LastLog; 
        tui_pr -> LastLogout = TimeBorder;
      }
      tui_pr -> UsageProps &= ~TUI_LOGGED_NOW;
      if ( IsCrash )
      {
      	tui_pr -> UsageProps |= TUI_HAS_CRASHES;
      	inconsistence_count ++;
#ifndef NO_DEBUG
        LOG (4, "FlushAll: invoked with IsCrash equal True\n" );
#endif      	
      }
      
    }
  }
}

static void UserProcess ( RootWtmp* rw_pr, struct utmp* utmp_pr )
{

//  RootWtmp* rw_pr;

  // 1.1 check if it is a user we need log at all --:)
  // ( here we rely on the fact the configuration already loaded 
  //   and users to look after are already known. )
  
  if ( NULL == UserRoot. FindUser ( utmp_pr -> ut_user ) )
    return;

//  // 1.2 check -- is it today's login ? 
//  if ( utmp_pr -> ut_time >= DayStart && utmp_pr -> ut_time <= DayEnd )
//  {
//    rw_pr = &WtmpToday ;
//    FlushOverall (); // make shure overall is closed :)
//  }
//  else 
//  {
//    rw_pr = &WtmpOverall ;
    
    // debug
    // printf ( "UserProcess: it is an Overall entry\n" );
//  }

  // 2. okay, log it
  UserLimitInfo* uli_pr;
  TtyUsageInfo* tui_pr;
  
  uli_pr = rw_pr -> FindUserLimitInfo ( utmp_pr -> ut_user );
  if ( uli_pr == NULL )
    uli_pr = rw_pr -> AddUserLimitInfo ( utmp_pr -> ut_user );
  tui_pr = uli_pr -> FindTtyUsageInfo ( utmp_pr -> ut_line );
  if ( tui_pr == NULL )
  {
    tui_pr = uli_pr -> AddTtyUsageInfo ( utmp_pr -> ut_line );
    memcpy ( tui_pr -> AbbrevLine, utmp_pr -> ut_id, 2 );
    tui_pr -> AbbrevLine [ 2 ] = 0;
  }
  
  // check for consistency -- user should not have TUI_LOGGED NOW flag
  if ( tui_pr -> UsageProps & TUI_LOGGED_NOW ) 
  {

#ifdef COUNT_CRASHTIME
    // flush user with warning
    if ( utmp_pr -> ut_time > tui_pr -> LastLog )
    	tui_pr -> Usage += utmp_pr -> ut_time - tui_pr -> LastLog;
    tui_pr -> LastLogout = utmp_pr -> ut_time;

#endif /* COUNT_CRASHTIME */
    	
    tui_pr -> UsageProps |= TUI_HAS_CRASHES;
    

#ifndef NO_DEBUG   
    // debug 
    sprintf ( LogBuff, "UserProcess:Inconsistence encountered for user %s, tty %s\n",
      uli_pr -> UserName, tui_pr -> Line );
    Log (4 );
#ifdef COUNT_CRASHTIME
   // debug
   sprintf ( LogBuff, "UserProcess:as COUNT CRASHTIME is set, overall now is %lu\n",
     tui_pr -> Usage );
   Log ( 4 );
#endif /* COUNT_CRASHTIME */
#endif /* NO_DEBUG */   
    inconsistence_count ++;
  }

  // fill in info
  tui_pr -> LastLog = utmp_pr -> ut_time;
  tui_pr -> LastPid = utmp_pr -> ut_pid;
  tui_pr -> UsageProps |= TUI_LOGGED_NOW;

#ifndef NO_DEBUG
  LOG ( 4, "UserProcess: Logon " );
  sprintf ( LogBuff, "%s, tty %s with pid %lu on %s",
    uli_pr -> UserName, tui_pr -> Line, tui_pr -> LastPid, 
    ctime ( &tui_pr -> LastLog ) );
  Log ( 4 );
  sprintf ( LogBuff, "UserProcess: Usage till now was %lu\n",
    tui_pr -> Usage );
  Log ( 4 );
#endif 

}

static void UserLogout (RootWtmp* rw_pr, struct utmp* utmp_pr )
{

//  RootWtmp* rw_pr;
  // 1.1 check -- is it today's login ? 
//  if ( utmp_pr -> ut_time >= DayStart && utmp_pr -> ut_time <= DayEnd )
// {
//    rw_pr = &WtmpToday ;
//    FlushOverall (); // make sure overall is closed :)
//  }
//  else 
//    rw_pr = &WtmpOverall ;
  
  // 2. hmm... Unfortunately
  // we can find out what user is refered to  with two
  // parameters at wtmp entry: pid and (or) tty. so we 
  // need to search through _all_ users currently searched for
  // to determine what user and what tty usage info to use for that 
  // entry ;) it could slow down all process though :)
  // but could you imagine somewhat less complicated or that is faster?  
  // Besides, there is another problem: DEAD_PROCESS entries 
  // could be ran into _after_ shutdown entry ( when all user usages are
  // flushed as it is :) Especially if your shutdown is graceful enough
  // (i.e. it has considerable grace time for users to logout ...)
  // if it is essential for your system, this code may need some 
  // rectifications .....
  
  // for logout process to determine what user is logging out, 
  // it can rely upon : 1) only pid; 2) only tty ; 
  // 3) pid _or_ tty; 4) pid _and_ tty. 
  // After long scrubbing my skull, i've decided to use 4).
  // The reason is it is the most reliable, though it is 
  // the most dependent on wtmp consistence. 
  // Probably 3) is good also :). But all 1-3 are somewhat 
  // ones that trust WTMP too much, and could produce 
  // rather big mistakes while accumulating tty usage without a notice,
  // in case WTMP consistence proves to be broken on your system.
  // On the other hand, 4) would only issue a warning for inconsistence,
  // and will not accumulate inconsistent usages. see also 'man utmp'.
  
  UserLimitInfo* uli_pr ;
  for (
    uli_pr = rw_pr -> base;
    uli_pr;
    uli_pr = uli_pr -> next 
  )
  {
    TtyUsageInfo* tui_pr = uli_pr -> FindTtyUsageInfo ( utmp_pr -> ut_line );
    if ( tui_pr == NULL || 0==( tui_pr -> UsageProps & TUI_LOGGED_NOW ) ) continue;

// this is for method 2) (see above ) in case CHECK_LOGOUT_PID is undefined.
#ifdef CHECK_LOGOUT_PID  
    // found. check for PID 
    if ( tui_pr -> LastPid != utmp_pr -> ut_pid ) 
    {
#ifndef NO_DEBUG  
      LOG ( 5, "LOGOUT:rejected due pid incompatibility " );
      sprintf ( LogBuff, "UserName:%s\n", uli_pr -> UserName );
      Log ( 5 );
#endif    
      return;
    }
#endif /* CHECK_LOGOUT_PID */

    // all, hopefully, is okay, flush this usage 
    if ( utmp_pr -> ut_time > tui_pr -> LastLog ) 
      tui_pr -> Usage += utmp_pr -> ut_time - tui_pr -> LastLog ;
    tui_pr -> LastLogout = utmp_pr -> ut_time;
    tui_pr -> UsageProps &= ~TUI_LOGGED_NOW;
  
#ifndef NO_DEBUG
    // debug 
    LOG ( 5, "LogoutProcess detected okay \n" );
    sprintf ( LogBuff, "for user %s tty %s; overall now is %lu\n",
      uli_pr -> UserName, tui_pr -> Line, tui_pr -> Usage );
    Log ( 4 );

#endif  
    return;
  }

#ifndef NO_DEBUG
  sprintf ( LogBuff, "LogoutProcess: no suitable struct found for pid %lu"
    "tty %s\n", utmp_pr -> ut_pid, utmp_pr -> ut_line );
  Log ( 5 );

#endif  

  
}

// implementation for NEW_TIME type of utmp_pr -- 
// so far have no a slightly idea what to do with it :)
// it is just a placeholder for now :)

static void NewTime ( RootWtmp* rw_pr, struct utmp* utmp_pr )
{
}

// to be implemented later

//static void CheckAintUsersReallyLogged ( void ) 
//{
//  struct utmp* utmp_pr;
//  utmpname ( _PATH_UTMP );
//  setutent ();
//  
//  endutent();
//}

void DetermineCurrentDay ( time_t CurrMoment, time_t &DayStart, time_t &DayEnd, 
  long &AbsoluteDayIndex )
{
  AbsoluteDayIndex = CurrMoment / ( 24 * 3600 );
  struct tm CurrTime;
  memcpy ( &CurrTime, localtime ( &CurrMoment ), sizeof ( struct tm ));
  // 1.2 determine start and end moment of the current day: 
  CurrTime . tm_sec = 0;
  CurrTime . tm_min = 0;
  CurrTime . tm_hour = 0; // start 
  DayStart = mktime ( &CurrTime );
  DayEnd = DayStart + ( 24 * 60 * 60 ); // I guess it is so :)


  return;
}

/**********************************************************************
         5 Aug 1996 
         Added CollectNacctd() -- support for collecting nacctd statistic
**********************************************************************/
#ifdef NACCTD_SUPPORT

int CollectNacctd ( RootWtmp* rw_pr, time_t StartTime, time_t EndTime)
{
  FILE* nacct_log;
  char NacctEntry [ MAXNACCTENTRYLEN + 1];
  NacctEntry [ MAXNACCTENTRYLEN ] = 0;
  NacctdEntry* ne_pr;
  
  RootWtmp* curr_rw_pr;

#ifndef NO_DEBUG
  Log2 (5, "Entering CollectNacctd\n" );
#endif

  if ( NULL == ( nacct_log = fopen ( NET_ACCT_PATH, "r" )))
  {
    Log2 ( 1, "awreader fatal: Cannot open nacctd log file\n" );
    return -1;
  }
  
  while ( ! feof ( nacct_log ) )
  {
    if ( (char*)EOF == fgets ( NacctEntry, MAXNACCTENTRYLEN, nacct_log ) )
      break;
    
    // make its parsing
    ne_pr = new NacctdEntry;
    stemStream ss ( NacctEntry, 0, strlen (NacctEntry ));
    ss = ss >> &ComSep() >> ne_pr >> &ComSep();
    if ( !ss. is_valid () || (! ( ss == 0 )) )
    {
      Log2 ( 1, "awreader:Erroneous nacctd log entry, ignored\n" );
      delete ne_pr;
      continue; // ignore it
    }
    // parse okay
    
    // now determine what account period it is relevant to
    if ( (time_t) ne_pr -> TimeStamp -> value < StartTime )
      curr_rw_pr = &WtmpOverall;
    else if ( (time_t) ne_pr -> TimeStamp -> value >= StartTime &&
      (time_t) ne_pr -> TimeStamp -> value <= EndTime ) 
      curr_rw_pr = rw_pr;
    else 
    {
#ifndef NO_DEBUG    
      Log2 ( 3,"awreader:nacct entry is ignored due to timestamp being out of range\n" );
#endif      
      delete ne_pr;
      continue;
    }
    
    // determine username -- whether it is a user we need log at all
    char uname_tmp [ UT_NAMESIZE + 1 ];
    memcpy ( uname_tmp, &NacctEntry [ ne_pr -> UserName -> sp ],
      ne_pr -> UserName -> np - ne_pr -> UserName -> sp );
    uname_tmp [ ne_pr -> UserName -> np - ne_pr -> UserName -> sp ] =0;
    
    User_st* usr_pr = UserRoot.FindUser ( uname_tmp );
    if ( usr_pr == NULL )
    {
#ifndef NO_DEBUG
      Log2 ( 4, "awreader:nacct entry is ignored due to user stamp not being in UserRoot\n" );
#endif
      delete ne_pr;
      continue;
    }
    // user stamp okay, log it 
    UserLimitInfo* uli_pr = curr_rw_pr -> FindUserLimitInfo ( uname_tmp );
    if ( uli_pr == NULL )
      uli_pr = curr_rw_pr -> AddUserLimitInfo ( uname_tmp );
    
    char devname_tmp [ MAXDEVICELEN + 1 ];
    memcpy ( devname_tmp, &NacctEntry [ ne_pr -> Device -> sp ],
      ne_pr -> Device -> np - ne_pr -> Device -> sp );
    devname_tmp [ ne_pr -> Device -> np - ne_pr -> Device -> sp ] = 0; //fix
      
    DeviceUsageInfo* dui_pr = uli_pr -> FindDeviceUsageInfo (devname_tmp);
    if ( dui_pr == NULL )
      dui_pr = uli_pr -> AddDeviceUsageInfo ( devname_tmp );
      
    // so far i have no idea how could we determine traffic direction
    // so so far i merely add it value to both In and Out
    dui_pr -> InTraffic += ne_pr -> Size -> value;
    dui_pr -> OutTraffic += ne_pr -> Size -> value;
    
    
    delete ne_pr;
  }
  
  fclose ( nacct_log );
  return 0;
}
    
    
#endif /* NACCTD_SUPPORT  */


/**********************************************************************
     12 Aug Added sums
     calculates sums for traffic and time usages and places em to
     UserLimitInfos 
**********************************************************************/
static void FixSums ( RootWtmp* rw_pr )
{
  // traffic sums 
  UserLimitInfo* uli_pr;
  TtyUsageInfo* tui_pr;
  DeviceUsageInfo* dui_pr;
  for ( uli_pr = rw_pr -> base;
    uli_pr ;
    uli_pr = uli_pr ->next 
  )
  {
     time_t TimeSum = 0;
     unsigned long TrafficSum = 0;
     
     for ( tui_pr = uli_pr -> TtyBase .base;
       tui_pr;
       tui_pr = tui_pr -> next
     )
       TimeSum += tui_pr -> Usage;
     
     for  (dui_pr = uli_pr -> DeviceBase . base;
       dui_pr;
       dui_pr = dui_pr -> next 
     )
       TrafficSum += dui_pr -> InTraffic;
     
     uli_pr -> InTraffic=uli_pr -> OutTraffic = TrafficSum;
     uli_pr -> TotalUsage = TimeSum;
  } 
}
/**********************************************************************
**********************************************************************/
/**********************************************************************
**********************************************************************/
/**********************************************************************
**********************************************************************/

/*
static void PrintUtmp ( RootWtmp* rw_pr )
{

	// print out all statistic 
	UserLimitInfo* uli_pr;
	for ( uli_pr = rw_pr -> base;
	  uli_pr;
	  uli_pr = uli_pr -> next
	)
	{
	  printf ( "---> Statistic for user %s <---\n", uli_pr -> UserName );
	  TtyUsageInfo* tui_pr;
	  for ( tui_pr = uli_pr -> TtyBase .base;
	    tui_pr;
	    tui_pr = tui_pr -> next
	  )
	  {
	    printf ( "Tty Name: %s, Abbreviated name :%s\n",
	      tui_pr -> Line, tui_pr -> AbbrevLine );
	    printf ( "   -- Usage :%d days %d hours %d minutes %d seconds (%lu seconds overall)\n",
	      tui_pr -> Usage / (24 * 60 * 60 ),
	      (unsigned )( tui_pr -> Usage % ( 24 * 60 * 60 )) / 3600,
	      (unsigned )( tui_pr -> Usage % ( 60 * 60 )) / 60,
	      (unsigned )( tui_pr -> Usage % 60 ),
	      tui_pr -> Usage
	    );
	    if ( tui_pr -> UsageProps & TUI_LOGGED_NOW )
	    {
	      printf ( "   -- Still Logged in");
	      printf ( " with Pid = %u at %s\n", tui_pr -> LastPid,
	        ctime ( &tui_pr -> LastLog ) );
	    }
	    if ( tui_pr -> UsageProps & TUI_HAS_CRASHES )
	    {
	      printf ( "   -- wtmp inconsistence detected for this user&tty\n" );
	    }
	  }
	}
}
*/
