
#include "admind.h"


#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <sys/time.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>

// the following has to be moved into config
#define MAX_SERVERS_SPARE 2

int MaxServers = MAX_SERVERS_SPARE, ServersCount = 0;
pid_t ServerPids [ MAX_SERVERS_NO ];

char ModuleName[] = "admind";

struct Handshake_st ModuleSig = { ADMIND_SIG, VERSION };
struct Handshake_st RespSig;

RootBase UserRoot;
TagBase TtyTagsBase;

RootWtmp WtmpToday;
RootWtmp WtmpOverall;
time_t LastWtmpCheck =0, LastCheck = 0;
struct tm LastCheck_tm;

unsigned short Port;
unsigned CheckGranularity;
unsigned RereadGranularity;
unsigned GranCount = 0;

static int CallAParser ( void );
static int CallAWReader ( RootWtmp* rw_pr, int OpCode, 
  time_t StartTime =0, time_t EndTime = 0 );
static int ForkWithPipe ( char* process );

static   int ReadPipe [ 2 ];
static   int WritePipe [ 2 ];

/** Signal handlers **/
static void SetSignals ( void );
static void TreatSighup(int);
static void TreatSigterm ( int );
static void TreatSigio ( int );
static void TreatSock (void);
//static void TreatAlarm (int );
static void TreatChild ( int );

static void BuildBlacks ( void );
static void RecheckUtmp ( time_t, time_t );
static void TreatLoginRequest ( FILE* fsock);

/* the following function is for debug only */
#ifndef NO_DEBUG
static void PrintUtmp ( RootWtmp* );
#endif

static struct sockaddr_in sa,ca; // server addr, client addr
static int sock = -1,nsock;
static fd_set io_mask;

int main ( int, char** )
{

  CallAParser();
  CallAWReader( &WtmpToday, OC_TODAY);
  
#ifndef NO_DEBUG
  LOG ( 4, "admind: all configuration is read okay\n" );  
#endif

  // here the real deal starts!
#ifndef DO_NOT_FORK
  switch ( fork () )
  {
    case 0: // child
      break;
    case -1: // error
      Log2 ( 1, "admind fatal: fork() failed.\n" );
      exit ( 1 );
    default: // parent
      exit ( 0 );
  }

  // become a leader 
  if ( -1 == setsid () ) 
  {
    Log2 ( 1, "admind fatal: setsid() failed, aborting..\n" );
    exit ( 1 );
  }
#ifndef NO_DEBUG
  Log2 ( 4, "admind: fork() passed okay -- hi from child\n" ); 
#endif  
#endif /* Do_not_fork */

#ifndef NO_DEBUG
  Log2 ( 3, "admind:--------------- stat for today ----------\n" );
  PrintUtmp ( &WtmpToday );
  Log2 ( 3, "admind:--------------- Overall stat ------------\n" );
  PrintUtmp ( &WtmpOverall );
  Log2 ( 3, "admind:---------------   done stat  ------------\n" );
#endif 

  
  // making some arrangements here
  close (0); close ( 1 ); // close low fds
  SetSignals();
  
#ifndef NO_DEBUG
#endif  

  memset ( (void*)&sa, 0, sizeof (sa));
  sa.sin_family=AF_INET;
  sa.sin_addr.s_addr=INADDR_ANY;
  sa.sin_port = htons (Port );
  sock=socket( AF_INET, SOCK_STREAM, 0 );
  if ( -1 == bind (sock, (struct sockaddr*)&sa,sizeof ( sa )))
  {
    if ( errno == EINVAL )
      Log2 ( 1, "admind fatal: address already in use.\n" );
    Log2 ( 1, "admind fatal: bind() returned error.\n" );
    exit ( 1 );
  }
      
    
  // async
  fcntl ( sock,F_SETOWN, getpid());
  // FASYNC is needed for SIGIO to be sent at all
  // O_NONBLOCK is needed, as expirements had shown, for proper 
  // socket handling ( by accept , etc. )
  fcntl ( sock,F_SETFL,  O_NONBLOCK | FASYNC );
  FD_ZERO ( &io_mask );
  FD_SET ( sock, &io_mask );

  listen ( sock,3); // 3 is size of queue of pending requests

  {
    time_t t = time ( NULL );
    Log2 ( 1, "admind %d.%d%d started on %s", VERSION >> 8, 
      ( VERSION & 0xff ) >> 4, VERSION & 0xf,
      ctime ( &t) );
  }
    

#ifndef NO_DEBUG  
  // debug
  Log2 ( 3, "Admind:about entering sleep loop\n" );
#endif  

  LastCheck = LastWtmpCheck; 
  unsigned left;
  for  ( GranCount=0,left=CheckGranularity;;
    GranCount++,left=CheckGranularity )
  {
    while ( left ) 
    { 
      left = sleep ( left ); 
      SetSignals(); 
#ifndef NO_DEBUG
      Log2 ( 3, "\n\nAdmind: SERVERS COUNT=%d\n", ServersCount );
#endif            
    }
    // block some signals that could cause interference while working 
    // with in -mems.
    sigset_t mask;
    sigemptyset (&mask );
    sigaddset (& mask, SIGHUP );
    sigaddset (&mask, SIGIO );
    sigprocmask ( SIG_BLOCK, &mask, NULL );
    
    if ( GranCount > RereadGranularity )
    {
       GranCount = 0;
       CallAWReader( &WtmpToday, OC_TODAY);
    } else 
    // Build Black lists && check current logs 
    BuildBlacks();
    // CheckUtmp();
    // unblock all
    sigprocmask ( SIG_UNBLOCK, &mask, NULL );
    //SetSignals ();
  }     
  
  return 0;
}

static int ForkWithPipe ( char* Process )
{
   
   if ( pipe ( ReadPipe ) || pipe ( WritePipe ) ) 
   {
     LOG(1, "Admind:Fatal: cannot create pipe\n" );
     exit ( 1 );
   }
#ifndef NO_DEBUG
   sprintf ( LogBuff, "admind: pipe returned descs %d %d for ReadPipe\n",
     ReadPipe [0], ReadPipe [ 1] );
   Log ( 5 );
   sprintf ( LogBuff, "admind: pipe returned descs %d %d for WritePipe\n",
     WritePipe [ 0 ], WritePipe [1 ] );
   Log ( 5 );
#endif   
   
   switch ( fork ( ) )
   {
   
     case -1 :
       LOG ( 1, "admind fatal: fork() failed\n" );
       exit ( 1 );
     case 0: // child 
       close ( ReadPipe [ 0 ] );
       close ( WritePipe [ 1 ] );
       if ( -1 == dup2 ( WritePipe [ 0 ], 0 ) || /*listen on write pipe */
         -1 ==dup2 ( ReadPipe [ 1], 1 ) ) // write to read pipe 
       {
         LOG ( 1, "admind child: fatal: dup2 returned error\n" );
         exit ( 1 );
       }
       close ( WritePipe [ 0 ] );
       close ( ReadPipe [ 1 ] ); // close unnecessary descriptors 
       // execute aparser
       exit ( execl ( Process, NULL ) );
       
       
       
     default : // admind
       close ( ReadPipe [ 1 ] );
       close ( WritePipe [ 0 ] );
   }
       


   return 0;
}

static int CallAParser (void ) 
{

//   ---------- option handling to be placed here

//  ---------- calling parser 
   FILE *ReadParser, *WriteParser;
   
   ForkWithPipe ( APARSER_PATH );
   
   if ( NULL == ( ReadParser = fdopen ( ReadPipe [ 0 ], "r" ) ) ||
     NULL == ( WriteParser = fdopen ( WritePipe [ 1 ], "w" )) )
   {
     LOG ( 1, "admind fatal: fdopen () failed\n" );
     exit ( 1 );
   }

#ifndef NO_DEBUG 
   LOG (4, "admind: fopen gone okay, sending signature\n" );
       

#endif
   // sending our signature
   if ( 1 != fwrite ( (void* ) &ModuleSig, sizeof ( ModuleSig ), 1,
     WriteParser ) )
   {
     LOG ( 1, "admind fatal : broken pipe to aparser\n" );
     exit ( 1 );
   }

   fflush ( WriteParser );
   
#ifndef NO_DEBUG
  LOG (4, "admind: signature sent, waiting for parser's signature\n");
#endif
       
   // reading &checking aparser sig
       
   if ( 1 != fread ( (void* ) &RespSig, sizeof ( RespSig ), 1, 
     ReadParser ) )
   {
     LOG ( 1, "admind fatal : broken pipe to aparser\n" );
     exit ( 1 );
   }
       
   if ( memcmp ( (void* ) RespSig. ModuleSig, APARSER_SIG, 2 ) )
   {
     LOG ( 1, "admind fatal : wrong aparser signature\n" );
     exit ( 1 );
   }
   if ( VERSION != RespSig. version )
   {
     LOG( 1, "admind fatal : wrong aparser version\n" );
     exit ( 1 );
   }

#ifndef NO_DEBUG
  LOG( 4, "admind: signature is read and checked okay\n" );
#endif
         
   // All is okay, read parsed configuration 
       
   // Reading parse results 
   if ( UserRoot . ReadFromStream ( ReadParser ) ||
//     UserTagsBase . ReadFromStream ( ReadParser ) ||
     TtyTagsBase . ReadFromStream ( ReadParser ) || 
     1 != fread ( (void* ) &MaxServers, sizeof (MaxServers),1,ReadParser )||
     1 != fread ( (void* ) &Port, sizeof ( Port ) ,1 , ReadParser ) ||
     1 != fread ( ( void* ) &CheckGranularity,
       sizeof ( CheckGranularity ), 1, ReadParser ) ||
     1 != fread ( ( void* ) &RereadGranularity,
       sizeof ( RereadGranularity ), 1, ReadParser ) )
   {
     LOG ( 1, "admind fatal: parser failed to read configuration" );
     exit ( 1 );
   }
       
   // configuration read okay 
   fclose ( WriteParser );
   fclose ( ReadParser );
       
#ifndef NO_DEBUG        
   // debug 
   LOG (3 , "Admind: configuration is read okay\n" );
   User_st* usr_pr;
   for ( usr_pr = UserRoot . base;
     usr_pr;
     usr_pr = usr_pr -> next
   )
   {
     sprintf ( LogBuff, "User %s is detected in configuration.\n",
       usr_pr -> UserName );
     Log ( 4 );
   }
#endif       
       
	return 0;
}


static int CallAWReader ( RootWtmp* rw_pr, int OpCode, 
  time_t StartTime, time_t EndTime) 
{
  FILE *WriteAWReader, *ReadAWReader;
  
  ForkWithPipe ( AWREADER_PATH );

  if ( NULL == ( ReadAWReader = fdopen ( ReadPipe [ 0 ], "r" ) ) ||
    NULL == ( WriteAWReader = fdopen ( WritePipe [ 1 ], "w" )) )
  {
    LOG ( 1, "admind fatal: fdopen () failed\n" );
    exit ( 1 );
  }

#ifndef NO_DEBUG 
   LOG (4, "admind: fopen gone okay, sending signature\n" );
       

#endif
   // sending our signature
   if ( 1 != fwrite ( (void* ) &ModuleSig, sizeof ( ModuleSig ), 1,
     WriteAWReader ) )
   {
     LOG ( 1, "admind fatal : broken pipe to awreader\n" );
     exit ( 1 );
   }

   fflush ( WriteAWReader );
   
#ifndef NO_DEBUG
  LOG (4, "admind: signature sent, waiting for awreader's signature\n");
#endif
       
   // reading &checking aparser sig
       
   if ( 1 != fread ( (void* ) &RespSig, sizeof ( RespSig ), 1, 
     ReadAWReader ) )
   {
     LOG ( 1, "admind fatal : broken pipe to awreader\n" );
     exit ( 1 );
   }
       
   if ( memcmp ( (void* ) RespSig. ModuleSig, AWREADER_SIG, 2 ) )
   {
     LOG ( 1, "admind fatal : wrong awreader signature\n" );
     exit ( 1 );
   }
   if ( VERSION != RespSig. version )
   {
     LOG( 1, "admind fatal : wrong awreader version\n" );
     exit ( 1 );
   }

#ifndef NO_DEBUG
  LOG( 4, "admind: signature is read and checked okay\n" );
#endif
         
   // All is okay, read parsed configuration 
   
   AWReaderRequest Request;
   
   Request . OpCode = OpCode;
   Request . Start = StartTime;
   Request . End = EndTime;
   
   // Sending request 
   
   if ( 1 != fwrite ( (void* )&Request, sizeof ( Request ), 1, 
     WriteAWReader ) )
   {
     LOG ( 1, "admind: broken pipe while sending request to awreader\n" );
     exit ( 1 );
   }
   
   // Sending UserRoot
   if ( UserRoot. SendToStream ( WriteAWReader ))
   {
     LOG ( 1, "admind: error while sending UserRoot to awreader (broken pipe?)\n");
     exit ( 1 );
   }
   
   fflush ( WriteAWReader );
   
   // receiving WtmpToday
   
   if ( rw_pr -> ReadFromStream ( ReadAWReader ))
   {
     LOG ( 1, "admind: error while reading stat wtmp (broken pipe?)\n" );
     exit ( 1 );
   }
   
   if ( 1 != fread ( (void* ) &LastWtmpCheck, 
     sizeof ( LastWtmpCheck ), 1, ReadAWReader ) )
   {
     LOG ( 1, "admind: broken pipe while reading bonus from awreader\n" );
     exit ( 1 );
   }
    
   // reading WtmpOverAll 
   if  ( WtmpOverall . ReadFromStream ( ReadAWReader ))
   {
     LOG ( 1, "admind: error while reading WtmpOverall (broken pipe?)\n" );
     exit ( 1 );
   }
   
#ifndef NO_DEBUG
  LOG( 4, "admind: communicating with awreader done okay\n" );
#endif      
     
  fclose ( ReadAWReader );
  fclose ( WriteAWReader ); 



  return 0;
}

/****************************************************************************
     Signal handlers . Started to add 5 aug 1996
*****************************************************************************/
static void SetSignals ( void )
{
  signal ( SIGHUP, TreatSighup);
  signal ( SIGTERM, TreatSigterm );
  signal ( SIGIO, TreatSigio );
  signal ( SIGCHLD, TreatChild );
}


static void TreatSighup ( int )
{     
  // the following is to prevent zombies 
  signal ( SIGCHLD, TreatChild );

  // just re-read configurations 
  CallAParser();
  CallAWReader( &WtmpToday, OC_TODAY);
  Log2 ( 1, "admind: received SIGHUP, configuration reread.\n" );
#ifndef NO_DEBUG
  Log2 ( 3, "admind:--------------- stat for today ----------\n" );
  PrintUtmp ( &WtmpToday );
  Log2 ( 3, "admind:--------------- Overall stat ------------\n" );
  PrintUtmp ( &WtmpOverall );
  Log2 ( 3, "admind:---------------   done stat  ------------\n" );
#endif 
  // restore handler 
  return;
}

// child handler is to prevent child servers running out without
// a control 

static void TreatChild ( int )
{
  pid_t server_pid; int status, i;
  // wait for server 
  server_pid = waitpid ( -1, &status, WNOHANG);
  // try to find this process at our list 
  for ( i = 0; i < ServersCount; i++ )
    if ( ServerPids [ i ] == server_pid )
  {
    // found our server termination.
    int j;
    ServersCount --;
    for ( j = i; j < ServersCount; j++ )
      ServerPids [ j ] = ServerPids [ j+1 ];
  }
  // rearm
  signal ( SIGCHLD, TreatChild );
  return;
}
  

static void TreatSigterm ( int )
{
  // terminate
  Log2 ( 1, "admind: received SIGTERM, exiting...\n" );
  if ( sock != -1 ) // free it
   close ( sock );
  exit ( 0 );
}

static void TreatSigio ( int )
{
#ifndef NO_DEBUG
  static char Nested = 0;
  if ( Nested )
    Log2 ( 3, "Admind: NESTED SIGIO!\n" );
  Nested = 1;
#endif

  fd_set mask = io_mask;
  if ( -1 == select ( sock+1, &mask, 0, 0, 0 ) )
  {
     Log2 ( 1, "select EINTR or error\n" );
     return;
   }
   if ( ! FD_ISSET ( sock, &mask ) )
   {
#ifndef NO_DEBUG   
     Log2 ( 3, "TreatSigio:non-socket sigio received.\n" );
#endif     
     return ;
   }
   // socket connection query ( probably:) received 
   for ( ;; )
   {
     nsock = accept ( sock, (struct sockaddr*) &ca, sizeof ( ca ));
     if ( nsock == -1 )
     if ( errno != EWOULDBLOCK )
     {
       Log2 (1, "TreatSigio:accept returned error\n" );
       return ;
     }
     else 
     {
#ifndef NO_DEBUG
       Log2 ( 3, "Admind: TreatSigio: Accept returned no more connections\n" );
#endif
       break;
     }                  
     TreatSock();
#ifndef NO_DEBUG
     Log2 ( 3,"Connection accepted okay\n" );
     Log2 ( 3, "Query has gone from %s\n", inet_ntoa ( ca.sin_addr ) );
#endif
   }
   // rearm
#ifndef NO_DEBUG
   Nested = 0;
#endif
   signal ( SIGIO, TreatSigio );   
   return;
}

/*******************************************************************
   this procedure is called from sigio handler.
   therefore, it should be kept in mind that 
   all other signals are prohibited (and would be lost)
   unless set explicitly.
********************************************************************/
static char BrokenMess[] =
    "admind:TreatSock:connection is broken\n";

static void TreatSock()
{
  static char BrokenMess[] =
    "admind:TreatSock:connection is broken\n";
  pid_t server_pid;

  // for ZOMBIES. 
  signal ( SIGCHLD, TreatChild );
  {
    int ka_val = 1;
    if ( -1 == setsockopt ( nsock, SOL_SOCKET, SO_KEEPALIVE, &ka_val,
      sizeof  (ka_val ) ) )
    {
      Log2 (1, "admind:TreatSock():setsockopt failed.\n" );
      return;
    }
  }
 
#ifndef NO_DEBUG
  Log2 ( 3, "Have Entered TreatSock\n" );
#endif

  switch ( server_pid =fork () )
  {
    case 0: // child
      break;
    case -1: // error
      Log2 ( 1, "admind:TreatSock: fork failed.\n" );
      close ( nsock );
      return;
    default: 
      // server launched, count it:
      ServerPids [ ServersCount ++ ] = server_pid;
      close ( nsock );
      return;
  }
 
  // check the peer name
  int sz=sizeof ( ca );
  if ( -1 == getpeername ( nsock, (struct sockaddr*) &ca, &sz ))
  {
    Log2 ( 1, "admind: getpeername failed for nsock.\n" );
    close ( nsock );
    exit ( 1  );
  }
  // okay, got peername.
  {
    struct in_addr addr; addr.s_addr=ca.sin_addr.s_addr;
#ifndef NO_DEBUG
    Log2 ( 3,"TreatSock:query has gone from %s\n",
      inet_ntoa ( addr ) );
#endif
    if ( NULL == UserRoot . FindAllow ( addr.s_addr ))
    {
      Log2 (1,"admind:query from %s denied:not an allowed address\n",
        inet_ntoa(addr));
      close (nsock);
      exit ( 1 );
    }
  }
     
      
  
    
  FILE* fsock;
  if ( NULL == ( fsock= fdopen ( nsock, "r+" ) ) )
  {
    Log2 ( 1, "admind: fdopen() failed for nsock.\n" );
    close ( nsock );
    exit ( 1 );
  }

  // request handling here 
  // send our signature 
  if ( 1 != fwrite ( (void*)&ModuleSig, sizeof ( ModuleSig ), 1, fsock ))
  {
    Log2 ( 2, BrokenMess );
    exit ( 1 );
  }
  fflush ( fsock);
#ifndef NO_DEBUG
  Log2 ( 4, "admind-TreatSock:signature is sent, waiting for reply\n" );
#endif
  if ( 1 != fread ( (void*)&RespSig, sizeof ( ModuleSig ),1 ,fsock))
  {
    Log2 ( 2, BrokenMess );
    exit ( 1 );
  }
  // check the signature 
  if ( memcmp ( (void*)RespSig .ModuleSig, CGI_SIG, 2 ) &&
    memcmp ( (void* ) RespSig . ModuleSig, LOGIN_SIG,2 ) )
  {
    Log2 ( 2, "admind-TreatSock:wrong request signature, closing connection.\n");
    fclose ( fsock );
    exit ( 1 );
  }
  if ( RespSig.version != VERSION ) 
  {
    Log2 ( 2,"admind-TreatSock:wrong cgi or login utility version.\n" );
    fclose ( fsock );
    exit (1 );
  }
  if ( ! memcmp (  (void* ) RespSig . ModuleSig, LOGIN_SIG,2 ))
  {
    TreatLoginRequest( fsock );
    exit ( 0 );
  }
  
  // Login requests have the highest priority, and , therefore,
  // are treated unconditionally. 
  
  // in opposite to such requests, all other types of ones 
  // ( cgi requests currently ) should be checked against maximum
  // servers that could be spared. 
  
  if ( ServersCount >= MaxServers )
  {
    // refuse the connection. send notification of refusal.
    // char AcknowledgeSig [ 2 ] = "RE";
    if ( 1 != fwrite ( ( void* ) "RE" /* AcknowledgeSig */, 2, 1, fsock ) )
    { 
      Log2 ( 2, BrokenMess );
    }
    exit ( 1 );
  } 
  else 
  {
    // otherwise accept the query. Send signature of acceptance. 
    // char AcknowledgeSig [2 ] = "OK";
    if ( 1 != fwrite ( (void* ) "OK" /* AcknowledgeSig */, 2, 1, fsock ))
    {
      Log2 ( 2, BrokenMess );
      exit ( 1 );
    }
  }
  
  struct AWReaderRequest request;
  // reading request 
  if ( 1 != fread ( (void*) &request, sizeof ( request ),1, fsock ) )
  {
     Log2( 2, BrokenMess );
     exit ( 1 );
  }
  
  // checking kind of request.
  if  ( request . OpCode == OC_STAT )
  {
    // redirect it to awreader.
    
    RootWtmp awr_answer;
    if ( CallAWReader ( &awr_answer, request.OpCode, request.Start,
      request.End ) )
    {
      Log2 ( 1, "admind-TreatSock: awreader request failed.\n" );
      fclose ( fsock );
      exit ( 1 );
    }
    
    // sending answer && LastWtmpCheck && ttyTags && UserRoot
    if ( awr_answer.SendToStream (fsock ) || 
      1 != fwrite ( (void*) &LastWtmpCheck, 
        sizeof ( LastWtmpCheck ),1, fsock ) ||
      TtyTagsBase.SendToStream ( fsock ) ||
      UserRoot.SendToStream ( fsock ))
    {
      Log2 ( 2, BrokenMess );
      exit ( 1 );
    }

  }
  
  if ( request. OpCode = OC_USER_ROOT ) // request for conf. info
  {
    // send the answer 
    if ( UserRoot. SendToStream ( fsock )|| 
      TtyTagsBase. SendToStream ( fsock ))
    Log2 ( 2, BrokenMess );
    exit ( 1 );
  }
    
  
      
  sleep ( 2 );
   
  fclose ( fsock );
  exit ( 0);
}

/*********************************************************************
    Handles login request 
*********************************************************************/
static void TreatLoginRequest ( FILE* fsock )
{
  struct LoginRequest request;
  struct LoginResponse response;
  
  if ( 1 != fread ( (void* )&request, sizeof ( request ), 1, fsock ))
  {
    Log2 ( 2, BrokenMess );
    exit ( 1 );
  }
  // fix request .. for all may happen otherwise
  request . UserName [ UT_NAMESIZE ] = 0;
  request . Line [ UT_LINESIZE ] = 0;
  
  User_st* usr_pr = UserRoot.FindUser ( request . UserName );
  Day_st* day_pr;
  Tty_st* tty_pr;
  
  if ( usr_pr  )
  {
    response .  UserProps = usr_pr -> UserProps;
    // check also, 'cuz it is not noticed in UserProps,
    // whether it is a valid tty for request 
    if ( NULL != ( day_pr = usr_pr -> FindDay ( LastCheck_tm . tm_wday )) &&
      NULL == day_pr -> FindTty ( request . Line ) &&
      0== (day_pr -> DayProps & DAY_ANYTTY))
    response. UserProps |= US_NOTVALID_TTY;
  }
  else
    // user not found, but refuse him only if tty is subjected to draconian rules
  if ( UserRoot. FindDraconianTty ( request . Line ) )
      // found
    response. UserProps |= US_UNKNOWN;

  else // consider as if all is okay for him
    response. UserProps = 0;
    
  // sending response
  if ( 1 != fwrite ( (void* )& response, sizeof ( response ), 1, fsock))
  {
    Log2 ( 2, BrokenMess );
    return ;
  }
}
      
    
    
    
  

/*********************************************************************
  builds black lists 12 aug 
**********************************************************************/
static void BuildBlacks ( void )
{
  UserLimitInfo* uli_pr;
  User_st* usr_pr;
  Day_st* day_pr;
  Interval_st* interval_pr;
  TtyUsageInfo* tui_pr;
  
  // remember old checktime
  time_t PrevCheck = LastCheck;
 
  LastCheck = time (NULL ); // label this moment 
  memcpy ( (void*) &LastCheck_tm, (void*) localtime ( &LastCheck),
    sizeof ( LastCheck_tm ));
    
#ifndef NO_DEBUG
  Log2 ( 3, "DayTag for today is %d\n", LastCheck_tm . tm_wday );
#endif       
    
  time_t TimeOfDay = LastCheck_tm . tm_sec + LastCheck_tm . tm_min *60 +
    LastCheck_tm .tm_hour * 3600;
  {
    struct tm PrevTimeOfDay_tm;
    memcpy ( (void*) &PrevTimeOfDay_tm, (void*) localtime ( &PrevCheck),
      sizeof ( LastCheck_tm ));
    if ( PrevTimeOfDay_tm . tm_mday != LastCheck_tm . tm_mday ) // day change detected.
    {
      CallAWReader (&WtmpToday, OC_TODAY); // reread wtmp
      // and what else ?
      return;
    }
  }
    
  if ( PrevCheck && LastCheck > PrevCheck ) // if time has been shifted, skip it
    RecheckUtmp ( LastCheck , PrevCheck );
  
  for ( usr_pr = UserRoot. base;
    usr_pr ;
    usr_pr = usr_pr ->next
  )
  {
    uli_pr = WtmpToday . FindUserLimitInfo ( usr_pr -> UserName );
    day_pr = usr_pr -> FindDay ( LastCheck_tm . tm_wday );
    
    // 1. check whether current time is valid for user 
    // interval_pr = NULL;
    if  (day_pr && ( day_pr -> DayProps & DAY_ANYTIME ) )
      usr_pr -> UserProps &= ~US_NOTVALID_INTERVAL ;
    else 
     if ( day_pr)
    {  
     
      for ( interval_pr = day_pr -> IntervalBase. base;
        interval_pr ;
        interval_pr = interval_pr -> next 
      )
      if ( TimeOfDay >= interval_pr -> start &&
        TimeOfDay <= interval_pr -> end )
      break;
        
    
      if ( interval_pr ) // found a valid interval
      {
        if ( usr_pr -> UserProps & US_NOTVALID_INTERVAL )
        {
#ifndef NO_DEBUG
          Log2 ( 3, "Valid interval started for user %s on %s",
            usr_pr -> UserName, ctime ( &LastCheck ) );
#endif          
          usr_pr -> UserProps &= ~US_NOTVALID_INTERVAL;
        }
      } 
      else // phuh..
        usr_pr -> UserProps |= US_NOTVALID_INTERVAL;
        
    }  else // no days found for usr_pr
      usr_pr -> UserProps |= US_NOTVALID_INTERVAL;
    
    
    // 2. check whether the user has reached one's soft/hard limits
    
    // 2.1 Time limits 
    if ( uli_pr && day_pr )
    {
      time_t OverallUsage = uli_pr -> TotalUsage;
      
      // Correct OverallUsage -- till present moment 
      // ( should be added usages from last log till now 
      // for users those who are on now )
      for ( tui_pr = uli_pr-> TtyBase .base;
        tui_pr;
        tui_pr = tui_pr -> next )
      if ( ( tui_pr -> UsageProps & TUI_LOGGED_NOW ) &&
        tui_pr -> LastLog < LastCheck )
        OverallUsage+= LastCheck - tui_pr -> LastLog;
#ifndef NO_DEBUG
      Log2 ( 4, "admind:OverallUsage for user %s is %lu\n",
        usr_pr -> UserName, OverallUsage );
#endif        
      
      if ( day_pr -> DayProps & DAY_UNLIMITED )
        usr_pr -> UserProps &= ~ ( US_HARD_TIME_EXPIRED | US_SOFT_TIME_EXPIRED);
      else
      { 
           
        if ( day_pr-> DayLimit <= OverallUsage )
        {
          if ( (usr_pr -> UserProps & US_HARD_TIME_EXPIRED) == 0 )
          {
            usr_pr -> UserProps |= US_HARD_TIME_EXPIRED;
            Log2 ( 1, "admind:User %s has reached one's hard time limit on %s.\n",
              usr_pr -> UserName, ctime (&LastCheck) );
          }
        }
        else 
          usr_pr -> UserProps &= ~US_HARD_TIME_EXPIRED;
      
        if ( (float )day_pr -> DayLimit * SOFT_TIME_KOEFF <= 
          (float) OverallUsage )
        {
          if ( (usr_pr -> UserProps & US_SOFT_TIME_EXPIRED) == 0 )
          {
            usr_pr -> UserProps |= US_SOFT_TIME_EXPIRED;
            Log2 ( 1, "admind:warning: User %s has reached one's soft time limit on %s.\n",
              usr_pr -> UserName, ctime ( &LastCheck) );
          }
        }
        else usr_pr -> UserProps &= ~US_SOFT_TIME_EXPIRED;
      } /* if ( day_pr -> DayProps & DAY_UNLIMITED ) */

#ifdef NACCTD_SUPPORT    
      // 2.2 Traffic limits 
      if ( day_pr -> DayProps & DAY_TRAFFIC_UNLIMITED )
        usr_pr -> UserProps &= ~ ( US_HARD_TRAFFIC_EXPIRED | US_SOFT_TRAFFIC_EXPIRED );
      else 
      {
        if ( day_pr -> TrafficLimit <= uli_pr -> InTraffic )
        {
          if ( (usr_pr -> UserProps & US_HARD_TRAFFIC_EXPIRED) == 0 )
          {
            usr_pr -> UserProps |= US_HARD_TRAFFIC_EXPIRED;
            Log2 ( 1, "admind:User %s has reached one's hard traffic limit on %s.\n",
              usr_pr -> UserName, ctime ( &LastCheck ));
          }
        }
        else 
          usr_pr -> UserProps &= ~US_HARD_TRAFFIC_EXPIRED;
      
        if ( (float ) day_pr -> TrafficLimit * SOFT_TRAFFIC_KOEFF <=
          ( float ) uli_pr -> InTraffic )
        {
          if ( (usr_pr -> UserProps & US_SOFT_TRAFFIC_EXPIRED) == 0 )
          {
            usr_pr -> UserProps |= US_SOFT_TRAFFIC_EXPIRED;
            Log2 ( 1, "admind:User %s has reached one's soft traffic limit on %s.\n",
              usr_pr -> UserName, ctime ( &LastCheck ) );
          }
        }
        else 
          usr_pr -> UserProps &= ~US_SOFT_TRAFFIC_EXPIRED;
      } /* if DAY_TRAFFIC_UNLIMITED */

#endif /* NACCTD_SUPPORT */
        
    }  /* if (uli_pr && day_pr) */
    
    // check for valid terminals and all above -- explicit kills section
    if ( uli_pr )
      for ( tui_pr = uli_pr -> TtyBase . base;
        tui_pr;
        tui_pr = tui_pr -> next
      )
    {
      if ( ( tui_pr -> UsageProps & TUI_LOGGED_NOW) == 0 ) continue;
      
      // 1. check for US_NOTVALID_INTERVAL
      if ( usr_pr -> UserProps & US_NOTVALID_INTERVAL )
      {
        Log2 ( 1, "admind:%s:Not a valid interval for user %s on %s\n",
          ctime ( &LastCheck ), usr_pr -> UserName, tui_pr -> Line );
#ifdef EXPLICIT_KILLS
        kill ( tui_pr -> LastPid, SIGKILL );
        Log2 ( 1, "admind:%s:User %s on %s was killed due to time permissions\n",
          ctime ( &LastCheck), usr_pr -> UserName, tui_pr -> Line );
        continue;
#endif           
      }
#ifdef NACCTD_SUPPORT
#ifdef EXPLICIT_KILLS      
      // check for hard limits
      if (( usr_pr -> UserProps & US_HARD_TRAFFIC_EXPIRED) ||
        ( usr_pr -> UserProps & US_HARD_TIME_EXPIRED ))
      {
        kill ( tui_pr -> LastPid, SIGKILL );
        Log2 (1, "admind %s:User %s on %s was killed due to hard limits being expired.\n",
          ctime ( &LastCheck ), usr_pr -> UserName, tui_pr -> Line );
        continue;
      }
#endif 
#endif /* NACCTD_SUPPORT */
      // finally, check whether it is a valid line for today
      if  (day_pr && ( day_pr -> DayProps & DAY_ANYTTY) == 0 && 
        NULL == day_pr -> FindTty ( tui_pr -> Line ))
      {
        Log2 ( 1, "admind %s:Line %s is not a valid one for user %s.\n",
          ctime ( &LastCheck ), tui_pr -> Line, usr_pr -> UserName );
#ifdef EXPLICIT_KILLS
        kill ( tui_pr -> LastPid, SIGKILL );
        Log2 ( 1, "admind %s:User %s on %s was killed -- not a valid line.\n",
          ctime ( &LastCheck ), usr_pr -> UserName, tui_pr -> Line );
#endif          
      }
        
    }
    
  } /* for ( usr_pr...) */  

  return;
}    
  
static void RecheckUtmp ( time_t LastCheck, time_t PrevCheck )
{
  struct utmp* ut_pr;
  User_st* usr_pr;
  UserLimitInfo* uli_pr;
  TtyUsageInfo* tui_pr;
  
  utmpname ( _PATH_UTMP );
  setutent ();
  
  for ( ut_pr = getutent();
    ut_pr;
    ut_pr = getutent()
  )
  {
    if ( ut_pr -> ut_type != USER_PROCESS )
      continue;
    // -, -,      ? 
    
    usr_pr = UserRoot.FindUser( ut_pr -> ut_user );
    if ( usr_pr == NULL ) // okay, live him alone then
#ifndef EXPLICIT_KILLS
      continue;
#else 
    {
      // check if this is a draconian tty
      if ( UserRoot.FindDraconianTty ( ut_pr -> ut_line ))
      {
        // !       , !
        kill ( ut_pr -> ut_pid, SIGKILL );
        Log2 ( 1, "admind %s:User %s on %s was killed due to draconian "
          "rules applied to this tty\n",
          ctime ( &LastCheck ), ut_pr -> ut_user, ut_pr -> ut_line );
      }
      continue;
    }
#endif      
    
      
    
    uli_pr = WtmpToday . FindUserLimitInfo ( ut_pr -> ut_user );
    if ( uli_pr == NULL )
    {
#ifndef NO_DEBUG
      Log2 ( 3, "RecheckWtmp:failed to fund uli for user %s\n",
        ut_pr -> ut_user );    
#endif
      uli_pr = WtmpToday . AddUserLimitInfo  ( ut_pr -> ut_user );
    }
    tui_pr = uli_pr -> FindTtyUsageInfo ( ut_pr -> ut_line );
    if ( tui_pr == NULL )
    {
#ifndef NO_DEBUG
      Log2 ( 3, "RecheckWtmp:failed to find tui for user %s, line %s\n",
        ut_pr -> ut_user, ut_pr -> ut_line );
#endif
      tui_pr = uli_pr -> AddTtyUsageInfo ( ut_pr -> ut_line );
    }
    
    // correct pid, if necessary
    tui_pr -> LastPid = ut_pr -> ut_pid;
    
    if ( tui_pr -> UsageProps & TUI_LOGGED_NOW )
    {
      // make a stamp he exists
      tui_pr -> UsageProps |= TUI_LOGGED_RECENTLY;
      continue;
    }
    
    // else, he has logged in just now! make a note of the fact.
    tui_pr -> LastLog = ut_pr -> ut_time;
    tui_pr -> UsageProps |= TUI_LOGGED_NOW | TUI_LOGGED_RECENTLY;
    
#ifndef NO_DEBUG    
    Log2 ( 3, "RecheckUtmp:User %s login detected on %s.\n",
    ut_pr -> ut_user, ut_pr -> ut_line );
#endif    
  }
  // now we should seek for recent logouts...    
  for ( uli_pr = WtmpToday . base;
    uli_pr;
    uli_pr = uli_pr -> next
  )
    for ( tui_pr = uli_pr -> TtyBase . base;
      tui_pr;
      tui_pr = tui_pr -> next )
  {
    if ( tui_pr -> UsageProps & TUI_LOGGED_RECENTLY )
    { tui_pr -> UsageProps &= ~TUI_LOGGED_RECENTLY; continue; }
    if ( tui_pr -> UsageProps & TUI_LOGGED_NOW ) // Aha. logout found.
    // those who was not in utmp cannot gain tui_logged_recently flag.
    // therefore, it is recent logout.
    {
      if ( LastCheck > tui_pr -> LastLog )
        tui_pr -> Usage += LastCheck - tui_pr -> LastLog;
      tui_pr -> LastLogout = LastCheck;
      tui_pr -> UsageProps &= ~TUI_LOGGED_NOW;
#ifndef NO_DEBUG
      Log2 ( 3, "RecheckUtmp:User %s  logout detected on line %s\n",
        uli_pr -> UserName, tui_pr -> Line );
#endif        
      
    }
  }
  endutent();
  return;
}
   
   
    

/***************************************************************************
              dumps current in-mem wtmps ( wtmp & naccts stats )
               for debugging purposes only!
****************************************************************************/


#ifndef NO_DEBUG

static void PrintUtmp ( RootWtmp* rw_pr )
{

	// print out all statistic 
	UserLimitInfo* uli_pr;
	for ( uli_pr = rw_pr -> base;
	  uli_pr;
	  uli_pr = uli_pr -> next
	)
	{
	  Log2 (3, "---------------> Statistic for user %s (Tag: %s) <--------------\n", 
	    uli_pr -> UserName,
            UserRoot.FindUser( uli_pr -> UserName ) -> UserTag );
	  TtyUsageInfo* tui_pr;
	  for ( tui_pr = uli_pr -> TtyBase .base;
	    tui_pr;
	    tui_pr = tui_pr -> next
	  )
	  {
	    Log2 (3, "---> Tty Name: %s, Abbreviated name :%s <---\n",
	      tui_pr -> Line, tui_pr -> AbbrevLine );
	    Log2 (3, "   -- Usage :%u days %u hours %u minutes %u seconds (%lu seconds overall)\n",
	      tui_pr -> Usage / (24 * 60 * 60 ),
	      (unsigned )( tui_pr -> Usage % ( 24 * 60 * 60 )) / 3600,
	      (unsigned )( tui_pr -> Usage % ( 60 * 60 )) / 60,
	      (unsigned )( tui_pr -> Usage % 60 ),
	      tui_pr -> Usage
	    );
	    if ( tui_pr -> UsageProps & TUI_LOGGED_NOW )
	    {
	      Log2 (3, "   -- Still Logged in");
	      Log2 (3, " with Pid = %u at %s\n", tui_pr -> LastPid,
	        ctime ( &tui_pr -> LastLog ) );
	    }
	    if ( tui_pr -> UsageProps & TUI_HAS_CRASHES )
	    {
	      Log2 (3, "   -- wtmp inconsistence detected for this user&tty\n" );
	    }
	  }
	  DeviceUsageInfo* dui_pr;
	  for ( dui_pr = uli_pr -> DeviceBase .base;
	    dui_pr;
	    dui_pr = dui_pr -> next
	  )
	  {
	    Log2 ( 3, "---> Traffic stat for device %s <---\n",
	      dui_pr ->DeviceName );
	    Log2 ( 3, "   -- InTraffic:%lu\n", dui_pr -> InTraffic );
	    Log2 ( 3, "   -- OutTraffic:%lu\n", dui_pr -> OutTraffic );
	  }
	}
}
#endif /* NO_DEBUG */
