int arrowHead3d(double x,  double y, double z,
                double nx, double ny, double nz,
                double headLen,
                char color=1)
{

   dx=nx-x;
   dy=ny-y;
   dz=nz-z;

   vc={dx, dy};
   r=sqrt(dx*dx+dy*dy);
   if (r == 0)
     sc=0.0;
   else 
     sc=headLen/r;

   D={ { 0.866025 , 0.5 }, { -0.5, 0.866025 } };
   dn1=D#vc           *sc;
   dn2=transpose(D)#vc*sc;
   dn3=dz*sc;

   line3d(nx,ny,nz,nx-dn1[0],ny-dn1[1],nz-dn3,\color=color);
   line3d(nx,ny,nz,nx-dn2[0],ny-dn2[1],nz-dn3,\color=color);

   return 0;
}


/************************************************************************
*Name: velocity_field3d
*Type: LibraryFunction
*Section: 522 3D Plots
*Parameters: (double[]) u, (double[]) v, (double[]) w,
             (double[]) x = {-1}, (double[]) y = {-1}, (double[]) z = {-1},
             (double[]) startX = {-1},
             (double[]) startY = {-1}, (double[]) startZ = {-1},
             (char[]) colors = {-1},
             (int) nstep = 1, (double) stepSize = 0.1, 
             (double) headScale = 0.5, 
             (int) noaxis = 0
*Return: -1
*DescriptionShort: draws a velocity_field
*DescriptionLong:
\e\velocity_field3d\e\  plots a velocity field with 
arrows pointing in the direction of the given velocity field. 
\e\u\e\, \e\v\e\ and \e\w\e\ give the X, Y and Z components of the
3 dimensional velocity field.
If the parameters \e\x\e\, \e\y\e\ and \e\z\e\ are given the should
be arrays. The size of \e\z\e\ must be equal to the first
dimension of u, v, w and y must be equal to the second dimension
of u, v, w. At least the size of \x\  
must be equal to the third dimension of u, v, w.
\e\startX\e\, \e\startY\e\ and \e\startZ\e\ set the start points
and the array \e\color\e\ sets the colors  of each arrow.
\e\nsteps\e\ changes the number of segments of each arrow and
\e\stepSize\e\ determine the number of segments.
The parameter \e\headScale\e\ is the proportional factor between
the length of the arrow and sides of the arrow-head.
Corresponding to the \e\plot\e\ command the flag \e\noaixs\e\
suppresses the axis.
*Example:
>u=replicate(cos(dincarr(4,4)/5.),4);
>v=replicate(transpose(sin(dincarr(4,4)/5.)),4);
>w=darr(4,4,4)+0.4
>window(0,\t3d);
>velocity_field3d(u,v,w);
*SeeAlso: velocity_field
*Reference:
*End:
*************************************************************************/


int velocity_field3d(double u[], double v[], double w[],
                     double x[]     ={-1},
                     double y[]     ={-1},
                     double z[]     ={-1},
		     double startX[]={-1},
                     double startY[]={-1},
                     double startZ[]={-1},
                     char   colors[] ={-1},
                     int    nsteps   =1,
                     double stepSize =0.1,
                     double headScale=0.5,
                     int noaxis=0)
{
   nsteps++;
   su=size(u);
   sv=size(v);
   sw=size(w);

   xu=su[5]; xv=sv[5]; xw=sw[5];
   yu=su[4]; yv=sv[4]; yw=sw[4];   
   zu=su[3]; zv=su[3]; zw=sw[3];

   xl=su[5];
   yl=su[4];
   zl=su[3];

   if (su[1] != 3 || sv[1] != 3 || sw[1] != 3)
    {
      error("velocity_field","Arrays u, v and w must have two dimensions");
      return -1;
    }
   if (xu != xv || xu != xw || yu != yv || yu != yw || zu != zv || zu != zw )
    {
      error("velocity_field","Arrays u, v and w  must have the same range");
      return -1;
    }

   s3=size(x);
   if (s3[2+s3[1]] == 1)
     {
       x=dincarr(xl);
       s3=size(x);
     }
   else
     if (s3[1] != 1 || s3[3] != xl)
       {
         error("velocity_field","Array x have a wrong dimension or range");
         return -1;
       }

   s4=size(y);
   if (s4[2+s4[1]] == 1)
     {
       y=dincarr(yl);
       s4=size(y);
     }
   else
     if (s4[1] != 1 || s4[3] != yl)
       {
         error("velocity_field","Array y have a wrong dimension or range");
         return -1;
       }

   s5=size(z);
   if (s5[2+s5[1]] == 1)
     {
       z=dincarr(zl);
       s5=size(z);
     }
   else
     if (s5[1] != 1 || s5[3] != yl)
       {
         error("velocity_field","Array z have a wrong dimension or range");
         return -1;
       }

   s6=size(startX);
   s7=size(startY);
   s8=size(startZ);

   if (s6[2+s6[1]] == 1)
     {
       startX=darr(xl*yl*zl);
       for (i=0; i<xl; i++)
         for (j=0; j<yl; j++)
         startX[i+j*xl:xl*yl*zl-1:xl*yl]=x[i];
       s6=size(startX);
     }

   if (s7[2+s7[1]] == 1)
     {
       startY=darr(xl*yl*zl);
       for (i=0; i<yl; i++)
         for (j=0; j<zl; j++)
           startY[i*xl+j*xl*yl:i*xl+j*xl*yl+xl-1]=y[i];
       s7=size(startY);
     }

   if (s8[2+s8[1]] == 1)
     {
       startZ=darr(xl*yl*zl);
       for (i=0; i<zl; i++)
         startZ[i*xl*yl:(i+1)*xl*yl-1]=z[i];
       s8=size(startZ);
     }

   if (s6[2] != s7[2] || s6[2] != s8[2])
    {
      error("velocity_field","Arrays startX, startY and startZ must have the same dimension");
      return -1;
    }

   s9=size(colors);
   if (s9[2+s9[1]] == 1)
        colors=carr(s6[2])+1;
   else
    {
      colors=reshape(colors,s9[2]);
      s9=size(colors);
       if (s9[2] != s6[2])
        {
          error("velocity_field","Array color must have the same dimension as startX, startY and startZ");
          return -1;
        }
    }

   if (noaxis == 0)
    {
      lx=(max(x)-min(x))/(2*s3[3]);
      ly=(max(y)-min(y))/(2*s4[3]);
      lz=(max(z)-min(z))/(2*s5[3]);
      plot3d(\xrange={min(x)-lx,max(x)+lx},
             \yrange={min(y)-ly,max(y)+ly},
             \zrange={min(z)-lz,max(z)+lz});
    }

   li=s6[2];

   dx=u*stepSize;
   dy=v*stepSize;
   dz=w*stepSize;

   px=dpx=darr(nsteps);
   py=dpy=darr(nsteps);
   pz=dpz=darr(nsteps);

   double r, t, s;
   int    aj, ak, al;

   startbuffer();

   for (i=0; i<li; i++)
    {
       px[0]=startX[i];
       py[0]=startY[i];
       pz[0]=startZ[i];
       len  =0.0;         

       for (k=1; k<nsteps; k++)
         {

           for (m=0; m < xl-1; m++)
             if (x[m] <= px[k-1] && px[k-1] < x[m+1])
               {
                 aj=m;
                 t=(px[k-1]-x[aj])/(x[aj+1]-x[aj]);
               }

           if (px[k-1] < x[0])
             { aj=0; t=0.0; }
           else
             if (px[k-1] >= x[xl-1])
               { aj=xl-2; t=1.0; }

	   for (m=0; m < yl-1; m++)
             if (y[m] <= py[k-1] && py[k-1] < y[m+1])
               {
                 ak=m;
                 s=(py[k-1]-y[ak])/(y[ak+1]-y[ak]);
               }

	   if (py[k-1] < y[0])
             { ak=0; s=0; }
           else
             if (py[k-1] >= y[yl-1])
               { ak=yl-2; s=1; }


          for (m=0; m < zl-1; m++)
             if (z[m] <= pz[k-1] && pz[k-1] < z[m+1])
               {
                 al=m;
                 r=(pz[k-1]-z[al])/(z[al+1]-z[al]);
               }

	   if (pz[k-1] < z[0])
             { al=0; s=0; }
           else
             if (pz[k-1] >= z[zl-1])
               { al=zl-2; s=1; }


           dpx[k]=  (1-s)*(1-t)*(1-r)*dx[al,ak,aj]
                  + s*(1-t)*(1-r)    *dx[al+1, ak, aj]
                  + (1-s)*t*(1-r)    *dx[al, ak+1, aj]
                  + (1-s)*(1-t)*r    *dx[al, ak, aj]
                  + (1-s)*t*r        *dx[al, ak+1, aj+1]
                  + s*(1-t)*r        *dx[al+1, ak, aj+1]
                  + s*t*(1-r)        *dx[al+1, ak+1, aj]
                  + s*t*r            *dx[al+1, ak+1, aj+1];

           dpy[k]=  (1-s)*(1-t)*(1-r)*dy[al,ak,aj]
                  + s*(1-t)*(1-r)    *dy[al+1, ak, aj]
                  + (1-s)*t*(1-r)    *dy[al, ak+1, aj]
                  + (1-s)*(1-t)*r    *dy[al, ak, aj]
                  + (1-s)*t*r        *dy[al, ak+1, aj+1]
                  + s*(1-t)*r        *dy[al+1, ak, aj+1]
                  + s*t*(1-r)        *dy[al+1, ak+1, aj]
                  + s*t*r            *dy[al+1, ak+1, aj+1];

           dpz[k]=  (1-s)*(1-t)*(1-r)*dz[al,ak,aj]
                  + s*(1-t)*(1-r)    *dz[al+1, ak, aj]
                  + (1-s)*t*(1-r)    *dz[al, ak+1, aj]
                  + (1-s)*(1-t)*r    *dz[al, ak, aj]
                  + (1-s)*t*r        *dz[al, ak+1, aj+1]
                  + s*(1-t)*r        *dz[al+1, ak, aj+1]
                  + s*t*(1-r)        *dz[al+1, ak+1, aj]
                  + s*t*r            *dz[al+1, ak+1, aj+1];

           px[k]=px[k-1]+dpx[k];
           py[k]=py[k-1]+dpy[k];
           pz[k]=pz[k-1]+dpz[k];
          }

         plot3d(px,py,pz,\line,\color=colors[i]);
	 arrowHead3d(px[nsteps-2], py[nsteps-2], pz[nsteps-2],
                     px[nsteps-1], py[nsteps-1], pz[nsteps-1],
                     headScale*total(sqrt(dpx*dpx+dpy*dpy)),
                     colors[i]);
      }

   endbuffer();
   return -1;
}


