int arrowHead(double x,  double y,
              double nx, double ny,
              double headLen, char color)
{
   dx=nx-x;
   dy=ny-y;

   vc={dx, dy};
   r=sqrt(dx*dx+dy*dy);
   if (r == 0)
     sc=0.0;
   else 
     sc=headLen/r;

   D={ { 0.866025 , 0.5 }, { -0.5, 0.866025 } };
   dn1=D#vc           *sc;
   dn2=transpose(D)#vc*sc;

   line(nx,ny,nx-dn1[0],ny-dn1[1]);
   line(nx,ny,nx-dn2[0],ny-dn2[1]);

   return 0;
}

/************************************************************************
*Name: velocity_field
*Type: LibraryFunction
*Section: 521 2D Plots
*Parameters: (double[]) u, (double[]) v,
             (double[]) x = {-1}, (double[]) y = {-1},
             (double[]) startX = {-1}, (double[]) startY= {-1},
             (double[]) colors= {-1},
             (int) nstep = 1, (double) stepSize = 0.1, 
             (double) headScale = 0.5, 
             (int) noaxis = 0, (int) curve = 0
*Return: -1
*DescriptionShort: draws a velocity_field
*DescriptionLong: \e\velocity_field\e\  plots a velocity field with arrows pointing in the
direction of the given velocity field. \e\u\e\ and \e\v\e\ give the
X and Y components of the velocity field.

If the parameters \e\x\e\ and \e\y\e\ are given the should be arrays.
The size of \e\y\e\ must be equal to the first dimension of u, v and x
must be equal to the secound dimension of u, v. 
\e\startX\e\ and \e\startY\e\ set the start points and the
 array \e\color\e\ sets the colors  of each arrow of each arrow.
\e\nsteps\e\ changes the number of segments of each arrow and
\e\stepSize\e\ determine the number of segments.
The parameter \e\headScale\e\ is the proportional factor between
the length of the arrow and sides of the arrow-head.
Corresponding to the \e\plot\e\ command the flag \e\noaixs\e\
suppresses the axis.
If the segments of the arrows should be connected with splines the
flag \e\curve\e\ will do the job for you.
*Example:
>u=cos(dincarr(10,15)/20.);
>v=transpose(sin(dincarr(15,10)/20.));
>contour(q,\nlevel=7,\curve);
*SeeAlso: velocity_field3d
*Reference:
*End:
*************************************************************************/


int velocity_field(double u[], double v[],
                   double x[]     ={-1}, double y[]     ={-1},
                   double startX[]={-1}, double startY[]={-1},
                   char   colors[]={-1},
		   int    nsteps   =1,
                   double stepSize =0.1,
                   double headScale=0.5,
                   int noaxis=0, int curve=0)
{
   nsteps++;
   s1=size(u);
   s2=size(v);
   x1=s1[4]; x2=s2[4];
   y1=s1[3]; y2=s2[3]; 

   if (s1[1] != 2 || s2[1] != 2)
    {
      error("velocity_field","Arrays u, v must have two dimensions");
      return -1;
    }
   if (x1 != x2 || y1 != y2)
    {
      error("velocity_field","Arrays u, v  must have the same range");
      return -1;
    }

   s3=size(x);
   if (s3[2+s3[1]] == 1)
     {
       x=dincarr(x1);
       s3=size(x);
     }
   else
     if (s3[1] != 1 || s3[3] != x1)
       {
         error("velocity_field","Array x have a wrong dimension or range");
         return -1;
       }

   s4=size(y);
   if (s4[2+s4[1]] == 1)
     {
       y=dincarr(y1);
       s4=size(y);
     }
   else
     if (s4[1] != 1 || s4[3] != y1)
       {
         error("velocity_field","Array y have a wrong dimension or range");
         return -1;
       }

   s5=size(startX);
   if (s5[2+s5[1]] == 1)
     {
       startX=darr(x1*y1);
       for (i=0; i<x1; i++)
         startX[i:x1*y1-1:x1]=x[i];
       s5=size(startX);
     }

   s6=size(startY);
   if (s6[2+s6[1]] == 1)
     {
       startY=darr(x1*y1);
       for (i=0; i<y1; i++)
         startY[i*x1:(i+1)*x1-1]=y[i];
       s6=size(startY);
     }

   if (s5[2] != s6[2])
    {
      error("velocity_field","Array startX have not the same range as startY");
      return -1;
    }
   li=s5[2];

   s9=size(colors);
   if (s9[2+s9[1]] == 1)
        colors=carr(s6[2])+1;
   else
     { 
       colors=reshape(colors,s9[2]);
       s9=size(colors);
       if (s9[2] != s6[2])
        {
          error("velocity_field","Array color must have the same total number of entries as u and v");
          return -1;
        }
     }
   
   if (noaxis == 0)
    {
      lx=(max(x)-min(x))/(2*s3[3]);
      ly=(max(y)-min(y))/(2*s4[3]);
      plot(\xrange={min(x)-lx,max(x)+lx}, \yrange={min(y)-ly,max(y)+ly});
    }

   dx=u*stepSize;
   dy=v*stepSize;
   px=dpx=darr(nsteps);
   py=dpy=darr(nsteps);
   double t, s;
   int    ak, aj;
   startbuffer();

   for (i=0; i<li; i++)
     {
       px[0]=startX[i];
       py[0]=startY[i];
       len  =0.0;

       for (k=1; k<nsteps; k++)
         {
           for (m=0; m < x1-1; m++)
             if (x[m] <= px[k-1] && px[k-1] < x[m+1])
               {
                 aj=m;
                 t=(px[k-1]-x[aj])/(x[aj+1]-x[aj]);
               }

           if (px[k-1] < x[0])
             { aj=0; t=0.0; }
           else
             if (px[k-1] >= x[x1-1])
               { aj=x1-2; t=1.0; }

           for (m=0; m < y1-1; m++)
             if (y[m] <= py[k-1] && py[k-1] < y[m+1])
               {
                 ak=m;
                 s=(py[k-1]-y[ak])/(y[ak+1]-y[ak]);
               }

	   if (py[k-1] < y[0])
             { ak=0; s=0; }
           else
             if (py[k-1] >= y[y1-1])
               { ak=y1-2; s=1; }

           dpx[k]= (1-s)*(1-t)*dx[ak,aj] + s*(1-t)*dx[ak+1,aj]
                  +s*t*dx[ak+1,aj+1]     + (1-s)*t*dx[ak, aj+1];
           dpy[k]= (1-s)*(1-t)*dy[ak,aj] + s*(1-t)*dy[ak+1,aj]
                  +s*t*dy[ak+1,aj+1]     + (1-s)*t*dy[ak, aj+1];

           px[k]=px[k-1]+dpx[k];
           py[k]=py[k-1]+dpy[k];	    
          }

       if (curve)
         plot(px,py,\curve,\color=colors[i]);
       else
         plot(px,py,\line,\color=colors[i]);

       arrowHead(px[nsteps-2], py[nsteps-2],
                 px[nsteps-1], py[nsteps-1],
                 headScale*total(sqrt(dpx*dpx+dpy*dpy)),
		 colors[i]); 
     }

   endbuffer();
   return -1;
}









