import java.awt.*;
import java.awt.peer.*;
import java.awt.image.*;
import java.io.*;
import java.lang.*;
import java.util.*;


/**
 * The embedding window for the java graph device.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 * @see     XiFrame
 */
public class XiWindow extends Panel implements Runnable {
  static Thread thread;
  static XiWindow xiWindow;
  public Frame frame;
  Image img;
  MenuBar menuBar;
  Menu menu;
  public MenuItem menuItem;
  public XiPanel panel;

  public long lastUpdate;

  static String server;
  static int port;
  XiGraphDev device;

  public XiWindow() {
  }

  public XiWindow(String server, int port) {
    this.server=server;
    this.port=port;
    xiWindow=this;

    thread=new Thread(this);
    thread.start();
  }

  public static void main(String argv[]) {
    server=argv[0];
    port=Integer.parseInt(argv[1]);

    thread=new Thread(xiWindow=new XiWindow());
    thread.start();
  }

  public void stop() {
    device.stop();
    Thread.currentThread().stop();
  }

  public void run() {
    try {
      device=new XiGraphDev(xiWindow,24,server,port);
    } catch(Throwable t) {
      System.exit(0);
    }
    
    frame=new XiFrame("XiWindow",xiWindow);
    frame.add("Center",xiWindow);
    frame.resize(400,256);
    menuBar=new MenuBar();
    menu=new Menu("File");
    menuItem=menu.add(new MenuItem("Quit"));
    menuItem.enable();
    menuBar.add(menu);
    frame.setMenuBar(menuBar);
    panel=new XiPanel(xiWindow);
    frame.add("South",panel);
    frame.show();
    frame.toFront();

    //lastUpdate=System.currentTimeMillis();
    
    device.start();
  }

  /**
   * Handles the button events.
   * @param  e the Button or MenuItem event.
   */
  public boolean handleEvent(Event e) {
    String label;
    if (e.target instanceof MenuItem) {
      System.exit(0);                                 // Quit
      //device.send(menuItem.getLabel());
      //return true;
    }
    else
      if (e.target instanceof MenuWindow) {
	label=(String)e.arg;
	if (label.equals(panel.editItem[0])) {
	  //System.err.println(panel.editItem[0]);    // Properties
	  device.send(panel.editItem[0]);
	  return true;
	}
	if (label.equals(panel.editItem[1])) {
	  //System.err.println(panel.editItem[1]);    // Layout
	  device.send(panel.editItem[1]);
	  return true;
	}
	if (label.equals(panel.editItem[2])) {
	  //System.err.println(panel.editItem[2]);    // Clear All
	  device.send(panel.editItem[2]);
	  return true;
	}
	if (label.equals(panel.outputItem[0])) {
	  //System.err.println(panel.outputItem[0]);  // EPS Output
	  device.send(panel.outputItem[0]);
	  return true;
	}
	if (label.equals(panel.outputItem[1])) {
	  //System.err.println(panel.outputItem[1]);  // PPM Output
	  device.send(panel.outputItem[1]);
	  return true;
	}
	return false;
      }
      else
	if (e.target instanceof Button) {
	  label=(String)e.arg;
	  if (label.equals(((Button)panel.comp[0]).getLabel())) {
	    System.exit(0);                                            // Close
	    //device.send(((Button)panel.comp[0]).getLabel());
	    //return true;
	  }
	  if (label.equals(((Button)panel.comp[1]).getLabel())) {
	    //System.err.println(((Button)panel.comp[1]).getLabel());    // Undo
	    if (device.cmdCnt > 0) {
	      device.cmdCnt--;
	      device.update();
	      ((BlackGrayButton)panel.comp[2]).setBlack();
	    }
	    else
	      ((BlackGrayButton)panel.comp[1]).setGray();
	    return true;
	  }
	  else
	    if (label.equals(((Button)panel.comp[2]).getLabel())) {
	      //System.err.println(((Button)panel.comp[2]).getLabel());  // Redo
	      if (device.cmdCnt < device.maxCmdCnt) {
		device.cmdCnt++;
		device.update();
		((BlackGrayButton)panel.comp[1]).setBlack();
	      }
	      else
		((BlackGrayButton)panel.comp[2]).setGray();
	      return true;
	    }
	  return false;
	}
	else
	  switch (e.id) {
	  case Event.WINDOW_DESTROY:
	    System.exit(0);
	  }
    return false;
  }

  /*public void paint(Graphics g) {
    Rectangle r=frame.bounds();
    int width=r.width; int height=r.height;
    int pix[]=new int[width*height];
    int index=0;
    for (int y=0; y<height; y++) {
      int red=(y*255)/(height-1);
      for (int x=0; x<width; x++) {
	int blue=(x*255)/(width-1);
	pix[index++]=(255<<24)|(red<<16)|blue;
      }
    }
    img=createImage(new MemoryImageSource(width,height,pix,0,width));
    g.drawImage(img,0,0,null);
  }*/

  public void paint(Graphics g) {
    if (device != null)
      device.update();
    lastUpdate=System.currentTimeMillis();
  }
  
  /**
   * Updates the graphics window.
   * @param  g the graphics window.
   * @see    XiGraphDev#update
   */
  public void update(Graphics g) {
    if (System.currentTimeMillis()-lastUpdate > 100 &&
	device != null)
      if (!MenuButton.isShown)
	device.update();
      else
	MenuButton.isShown=false;
    lastUpdate=System.currentTimeMillis();
  }
}


/**
 * The embedding frame class for XiWindow.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 * @see     XiWindow
 */
class XiFrame extends Frame {
  XiWindow window;
  Rectangle lastBounds;
  
  public XiFrame(String title, XiWindow window) {
    super(title);
    this.window=window;
  }
  
  /**
   * Handles a MenuButton event.
   * @param  e the event.
   * @see    MenuButton
   * @see    MenuWindow
   */
  public boolean handleEvent(Event e) {
    if (e.target instanceof MenuItem || e.target instanceof MenuWindow) {
      window.postEvent(e);
      return true;
    }
    if (e.target instanceof XiFrame) {
      Rectangle r=bounds();
      if (lastBounds != null) {
	if (r.x != lastBounds.x || r.y != lastBounds.y ||
	    r.width != lastBounds.width || r.height != lastBounds.height) {
	  lastBounds=new Rectangle(r.x,r.y,r.width,r.height);
	  if (window.panel.comp[3] != null)
	    ((MenuButton)window.panel.comp[3]).hideMenu();
	  if (window.panel.comp[4] != null)
	    ((MenuButton)window.panel.comp[4]).hideMenu();
	}
      }
      else
	lastBounds=new Rectangle(r.x,r.y,r.width,r.height);
    }
    return false;
  }
}


/**
 * XiPanel contains all buttons of the XiFrame class.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 * @see     MenuButton
 * @see     WhiteTextField
 */
class XiPanel extends Panel {
  XiWindow window;
  public Component[] comp;
  public String[] editItem, outputItem;
  GridBagLayout layout;
  GridBagConstraints constraints;
  ButtonGroup group;
  
  public XiPanel(XiWindow window) {
    this.window=window;
    comp=new Component[6];
    editItem=new String[3];
    outputItem=new String[2];
    
    layout=new GridBagLayout();
    setLayout(layout);
    constraints=new GridBagConstraints();
    
    comp[0]=new Button("Close");
    comp[1]=new BlackGrayButton("Undo",false);
    comp[2]=new BlackGrayButton("Redo",false);
    comp[3]=new MenuButton(window.frame,"Edit");
    comp[4]=new MenuButton(window.frame,"Output");
    comp[5]=new WhiteTextField("0 kB",10);

    group=new ButtonGroup();
    for (int i=0; i<5; i++)
      group.addElement((Button)comp[i]);
    ((MenuButton)comp[3]).setButtonGroup(group);
    ((MenuButton)comp[4]).setButtonGroup(group);
    
    editItem[0]=((MenuButton)comp[3]).add("Properties");
    editItem[1]=((MenuButton)comp[3]).add("Layout");
    ((MenuButton)comp[3]).addSeparator();
    editItem[2]=((MenuButton)comp[3]).add("Clear All");

    outputItem[0]=((MenuButton)comp[4]).add("EPS Output");
    outputItem[1]=((MenuButton)comp[4]).add("PPM Output");
    
    constraints.fill=GridBagConstraints.VERTICAL;
    constraints.gridheight=GridBagConstraints.REMAINDER;
    constraints.weightx=1;
  
    layout.setConstraints(comp[0],constraints);
    add(comp[0]);

    layout.setConstraints(comp[1],constraints);
    add(comp[1]);

    layout.setConstraints(comp[2],constraints);
    add(comp[2]);

    layout.setConstraints(comp[3],constraints);
    add(comp[3]);

    layout.setConstraints(comp[4],constraints);
    add(comp[4]);
    
    layout.setConstraints(comp[5],constraints);
    add(comp[5]);
  }

  public boolean action(Event evt, Object what) {
    if (evt.target instanceof Button) {
      ((MenuButton)comp[3]).hideMenu();
      ((MenuButton)comp[4]).hideMenu();
      window.postEvent(evt);
      return true;
    }
    return false;
  }
}


/**
 * WhiteTextField shows the actual size of the command buffer.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 * @see     XiPanel
 */
class WhiteTextField extends TextField {
  String text;
  
  public WhiteTextField(String text, int cols) {
    super(cols);
    setEditable(false);
    setBackground(Color.white);
    setText(text);
  }

  public void setText(String t) {
    text=t;
    super.setText(t);
  }
}


class BlackGrayButton extends Button {
  Color color;
  
  public BlackGrayButton(String label, boolean blackColor) {
    super(label);
    if (blackColor)
      color=Color.black;
    else
      color=Color.lightGray;
  }

  public void setBlack() {
    if (isGray()) {
      color=Color.black;
      setLabel(getLabel());
    }
  }

  public void setGray() {
    if (isBlack()) {
      color=Color.gray;
      setLabel(getLabel());
    }
  }

  public boolean isBlack() {
    return color.equals(Color.black);
  }

  public boolean isGray() {
    return color.equals(Color.gray);
  }

  public void paint(Graphics g) {
    g.setColor(color);
    super.paint(g);
  }

  public void update(Graphics g) {
    g.setColor(color);
    super.update(g);
  }
}


/**
 * MenuButton generates a menu window on click.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 * @see     XiPanel
 * @see     MenuWindow
 * @see     ButtonGroup
 */
class MenuButton extends Button {
  String label;
  MenuWindow menu;
  public Frame parent;
  ButtonGroup group;
  public static volatile boolean isShown=false;
  
  public MenuButton(Frame parent, String label) {
    super(label);
    this.label=label;
    this.parent=parent;
    menu=new MenuWindow(this);
  }

  public synchronized String add(String mi) {  // mi = MenuItem
    return menu.add(mi);
  }

  public void addSeparator() {
    menu.addSeparator();
  }

  public void hideMenu() {
    menu.hide();
  }

  public boolean handleEvent(Event e) {
    if (e.target instanceof MenuButton && e.arg != null)
      if (e.arg.equals(label)) {
	isShown=true;
	if (group != null) {
	  Button button;
	  for (int i=0; i<group.size(); i++)
	    if ((button=group.elementAt(i)) instanceof MenuButton &&
		!button.getLabel().equals(label))
	      ((MenuButton)button).hideMenu();
	}
	menu.show();
	return true;
      }
    return false;
  }

  public void setButtonGroup(ButtonGroup group) {
    this.group=group;
  }
}


/**
 * MenuWindow is a pull down window, which is shown on clicking a MenuButton.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 * @see     MenuButton
 * @see     ButtonGroup
 */
class MenuWindow extends Window {
  MenuButton button;
  public Vector item;
  String separator;
  Event lastEvent;
  FontMetrics metrics;
  Graphics g;
  int index, lastIndex, fontSize;
  Dimension menuDim;
  
  public MenuWindow(MenuButton button) {
    super(button.parent);
    this.button=button;
    item=new Vector();
    separator="-";
    Font font=new Font("Dialog",Font.PLAIN,12);
    setFont(font);
    metrics=getFontMetrics(font);
    fontSize=font.getSize()+3;
    menuDim=new Dimension();
  }

  public synchronized String add(String mi) {  // mi = MenuItem
    item.addElement(mi);
    menuDim.width=getMaxWidth()+5;
    menuDim.height=item.size()*fontSize+5;
    return mi;
  }

  public void addSeparator() {
    setSeparator();
    add(separator);
  }

  protected void setSeparator() {
    int maxWidth=getMaxWidth();
    while (metrics.stringWidth(separator) < maxWidth)
      separator+="-";
  }
  
  protected int getMaxWidth() {
    int maxWidth=0, width;
    for (int i=0; i<item.size(); i++)
      if ((width=metrics.stringWidth((String)item.elementAt(i))) > maxWidth)
	maxWidth=width;
    return maxWidth;
  }
  
  public synchronized void show() {
    Rectangle r1=button.parent.bounds(), r2=button.bounds();
    Dimension screenSize=getToolkit().getScreenSize();
    int x=r1.x+r2.x, y=r1.y+r1.height+r2.y-r2.height;
    if (x < 0)
      x=0;
    else
      if (x+menuDim.width > screenSize.width)
	x=screenSize.width-menuDim.width;
    if (y < 0)
      y=0;
    else
      if (y+menuDim.height > screenSize.height)
	y=screenSize.height-menuDim.height;
    reshape(x,y,menuDim.width,menuDim.height);
    super.show();
    toFront();
    repaint();
  }

  public void paint(Graphics g) {
    g.setPaintMode();
    g.clearRect(0,0,menuDim.width,menuDim.height);
    for (int i=0; i<item.size();)
      g.drawString((String)item.elementAt(i),2,++i*fontSize);
    g.drawLine(0,0,0,menuDim.height-1);
    g.drawLine(0,menuDim.height-1,menuDim.width-1,menuDim.height-1);
    g.drawLine(menuDim.width-1,menuDim.height-1,menuDim.width-1,0);
    g.drawLine(menuDim.width-1,0,0,0);
  }

  /**
   * Inverts an item entry in a MenuWindow.
   * @param  e the mouse event.
   */
  public boolean handleEvent(Event e) {
    g=getGraphics();
    g.setXORMode(getBackground());
    switch (e.id) {
    case Event.MOUSE_MOVE:
      if (lastEvent != null) {
	lastIndex=index;
	index=(e.y+fontSize-2)/fontSize-1;
	if (lastIndex != -1  && index != lastIndex)
	  g.fillRect(1,lastIndex*fontSize+2,menuDim.width-2,fontSize);
	if (index >= 0 && index < item.size() &&
	    e.x >= 0 && e.y < menuDim.width) {
	  if (index != lastIndex)
	    if (!item.elementAt(index).equals(separator))
	      g.fillRect(1,index*fontSize+2,menuDim.width-2,fontSize);
	    else
	      index=-1;
	}
	else
	  index=-1;
      }
      else {
	index=(e.y+fontSize-2)/fontSize-1;
	if (index >= 0 && index < item.size() &&
	    e.x >= 0 && e.y < menuDim.width)
	  if (!item.elementAt(index).equals(separator))
	    g.fillRect(1,index*fontSize+2,menuDim.width-2,fontSize);
	  else
	    index=-1;
	else
	  index=-1;
      }
      lastEvent=e;
      return true;
    case Event.MOUSE_DOWN:
      hide();
      if (lastEvent != null && index != -1) {
	hide();
	lastIndex=index;
	for (int i=0; i<item.size(); i++)
	  if (index >=i &&
	      item.elementAt(i).equals(separator))
	    index--;
	button.parent.postEvent(new Event(this,
					  e.when,
					  index,
					  e.x,e.y,
					  e.key,
					  e.modifiers,
					  item.elementAt(lastIndex)));
      }
      lastEvent=null;
      return true;
    case Event.MOUSE_EXIT:
      if (lastEvent != null)
	if (index != -1)
	  if (!item.elementAt(index).equals(separator))
	    g.fillRect(1,index*fontSize+2,menuDim.width-2,fontSize);
      lastEvent=null;
      return true;
    }
    return false;
  }
}


/**
 * ButtonGroup contains all buttons in the XiPanel. It's necessary if
 * a MenuWindow has to hide again.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 * @see     MenuButton
 * @see     MenuWindow
 */
class ButtonGroup {
  Vector button;

  public ButtonGroup() {
    button=new Vector();
  }

  public void addElement(Button b) {
    button.addElement(b);
  }

  public int size() {
    return button.size();
  }

  public Button elementAt(int i) {
    return (Button)button.elementAt(i);
  }
}
