import java.lang.*;
import java.util.*;
import java.net.*;
import java.io.*;


/**
 * Handles the communication between xi and the XiGraphDev class.
 * @version 0.6, 23 Jun 1996
 * @author  Bernd Nottelmann
 * @see     XiGraphDev
 */
public class XiPipe {
  protected Socket xiSocket, SIGUSR1socket;
  protected InetAddress addr;
  
  public    InputStream input;
  public    DataInputStream dataInput;
  public    OutputStream output;
  public    DataOutputStream dataOutput;
  protected InputStream SIGUSR1Input;
  protected OutputStream SIGUSR1Output;

  protected char[] cBuff=new char[1];
  protected short[] sBuff=new short[1];
  protected int[] iBuff=new int[1];
  protected float[] fBuff=new float[1];
  protected double[] dBuff=new double[1];
  protected char singleCBuff;
  
  protected Signal SIGUSR1signal;
  protected SIGUSR1Handler SIGUSR1handler;
  protected Handler handler;

  public static final int MAGIC=3144;  // Default port number
  
  public static final int SIGUSR1=Handler.SIGUSR1;
  public static final int LOOKUP=12345;
  public static final int TRANSFER_INTERRUPT=31527;
  
  int i;


  public XiPipe() throws Throwable {
    this(InetAddress.getLocalHost());
  }

  public XiPipe(String host) throws Throwable {
    this(host,MAGIC);
  }

  public XiPipe(InetAddress addr) throws Throwable {
    this(addr,MAGIC);
  }

  public XiPipe(int port) throws Throwable {
    this(InetAddress.getLocalHost(),port);
  }

  public XiPipe(String host, int port) throws Throwable {
    this(InetAddress.getByName(host),port);
  }
  
  public XiPipe(InetAddress addr, int port) throws Throwable {
    this(addr,port,SIGUSR1Handler.MAGIC);
  }
  
  public XiPipe(String host, int port, int SIGUSR1port) throws Throwable {
    this(InetAddress.getByName(host),port,SIGUSR1port);
  }
  
  public XiPipe(InetAddress addr, int port, int SIGUSR1port) throws Throwable {
    this.addr=addr;

    // Open data stream socket.
    xiSocket=new Socket(addr,port);
    input=xiSocket.getInputStream();
    dataInput=new DataInputStream(input);
    output=xiSocket.getOutputStream();
    dataOutput=new DataOutputStream(output);

    // Open signal stream socket.
    SIGUSR1socket=new Socket(addr,SIGUSR1port);
    SIGUSR1Input=SIGUSR1socket.getInputStream();
    SIGUSR1Output=SIGUSR1socket.getOutputStream();
  }

  /**
   * Looks up if the pipe between xi and XiPipe still exists.
   * @return true if it exists else false.
   */
  public boolean lookup() {
    try {
      new DataOutputStream(SIGUSR1Output).writeInt(LOOKUP);
    } catch (IOException exc) {
      return false;
    }
    return true;
  }

  /**
   * Sends an signal to xi if new data from this class arrive.
   * @exception IOException If an I/O error has occurred.
   */
  public void attention() throws IOException {
    Signal.kill(new DataOutputStream(SIGUSR1Output),SIGUSR1);
  }

  /**
   * Sets the attention handler.
   * @param  handler the attention handler which handles signals from xi.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void onAttention(Handler handler) throws Exception {
    if (SIGUSR1signal != null)
      SIGUSR1signal.stop();
    this.handler=handler;
    SIGUSR1handler=new SIGUSR1Handler(this);
    SIGUSR1signal=new ExpSignal(SIGUSR1,SIGUSR1handler,
				SIGUSR1Input,SIGUSR1Output);
  }

  /**
   * Flushes the output stream of this pipe.
   * @exception IOException If an I/O error has occurred.
   */
  public void flush() throws IOException {
    dataOutput.flush();
  }

  /**
   * Tests if data from inputstream of this pipe are available.
   * @return true if data are readable.
   * @exception IOException If an I/O error has occurred.
   */
  public boolean available() throws IOException {
    return dataInput.available() > 0;
  }

  /**
   * Reads characters from input stream.
   * @param  ptr  reference to character array.
   * @param  size counts the characters to read.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void read(char[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      ptr[i++]=(char)dataInput.readByte();
  }

  /**
   * Writes characters to output stream.
   * @param  ptr  reference to character array.
   * @param  size counts the characters to write.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void write(char[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      dataOutput.writeByte(new Character(ptr[i++]).hashCode());
  }

  /**
   * Reads short integers from input stream.
   * @param  ptr  reference to short integer array.
   * @param  size counts the short integers to read.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void read(short[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      ptr[i++]=dataInput.readShort();
  }

  /**
   * Writes short integers to output stream.
   * @param  ptr  reference to short integer array.
   * @param  size counts the short integers to write.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void write(short[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      dataOutput.writeShort(ptr[i++]);
  }

  /**
   * Reads integers from input stream.
   * @param  ptr  reference to integer array.
   * @param  size counts the integers to read.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void read(int[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      ptr[i++]=dataInput.readInt();
  }

  /**
   * Writes integers to output stream.
   * @param  ptr  reference to integer array.
   * @param  size counts the integers to write.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void write(int[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      dataOutput.writeInt(ptr[i++]);
  }

  /**
   * Reads floating point numbers (single precision) from input stream.
   * @param  ptr  reference to float array.
   * @param  size counts the floating point numbers to read.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void read(float[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      ptr[i++]=dataInput.readFloat();
  }

  /**
   * Writes floating point numbers (single precision) to output stream.
   * @param  ptr  reference to float array.
   * @param  size counts the floating point numbers to write.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void write(float[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      dataOutput.writeFloat(ptr[i++]);
  }

  /**
   * Reads floating point numbers (double precision) from input stream.
   * @param  ptr  reference to double array.
   * @param  size counts the floating point numbers to read.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void read(double[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      ptr[i++]=dataInput.readDouble();
  }

  /**
   * Writes floating point numbers (double precision) to output stream.
   * @param  ptr  reference to double array.
   * @param  size counts the floating point numbers to write.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void write(double[] ptr, long size) throws Exception {
    i=0;
    while (size-- > 0)
      dataOutput.writeDouble(ptr[i++]);
  }

  /**
   * Writes a whole character array to output stream.
   * @param  ptr  the character array.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(char[] c) throws IOException {
    try {
      iBuff[0]=c.length;
      write(iBuff,1);
      write(c,(long)c.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a string to output stream.
   * @param  ptr  the string.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(String S) throws IOException {
    try {
      out(S.toCharArray());
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a single character to output stream.
   * @param  ptr  the character.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(char c) throws IOException {
    cBuff[0]=c;
    try {
      write(cBuff,1);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a whole short integer array to output stream.
   * @param  ptr  the short array.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(short[] s) throws IOException {
    try {
      write(s,(long)s.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a single short integer number to output stream.
   * @param  ptr  the short integer number.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(short s) throws IOException {
    sBuff[0]=s;
    try {
      write(sBuff,1);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a whole integer array to output stream.
   * @param  ptr  the integer array.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(int[] i) throws IOException {
    try {
      write(i,(long)i.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a single integer number to output stream.
   * @param  ptr  the integer number.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(int i) throws IOException {
    iBuff[0]=i;
    try {
      write(iBuff,1);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a whole float array to output stream.
   * @param  ptr  the float array.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(float[] f) throws IOException {
    try {
      write(f,(long)f.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }
  
  /**
   * Writes a single precision floating point number to output stream.
   * @param  ptr  the float number.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(float f) throws IOException {
    fBuff[0]=f;
    try {
      write(fBuff,1);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a whole double array to output stream.
   * @param  ptr  the double array.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(double[] d) throws IOException {
    try {
      write(d,(long)d.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Writes a double precision floating point number to output stream.
   * @param  ptr  the double number.
   * @exception IOException If an I/O error has occurred.
   */
  public void out(double d) throws IOException {
    dBuff[0]=d;
    try {
      write(dBuff,1);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Reads a whole character array from input stream.
   * @param  ptr  reference to character array.
   * @exception Exception If an I/O or another error has occurred.
   */
  public void in(char[] c) throws Exception {
    read(iBuff,1);
    read(c,iBuff[0]);
  }

  /**
   * Reads a whole short integer array from input stream.
   * @param  ptr  reference to short array.
   * @exception IOException If an I/O error has occurred.
   */
  public void in(short[] s) throws IOException {
    try {
      read(s,(long)s.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Reads a whole integer array from input stream.
   * @param  ptr  reference to integer array.
   * @exception IOException If an I/O error has occurred.
   */
  public void in(int[] i) throws IOException {
    try {
      read(i,(long)i.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Reads a whole float array from input stream.
   * @param  ptr  reference to float array.
   * @exception IOException If an I/O error has occurred.
   */
  public void in(float[] f) throws IOException {
    try {
      read(f,(long)f.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Reads a whole double array from input stream.
   * @param  ptr  reference to double array.
   * @exception IOException If an I/O error has occurred.
   */
  public void in(double[] d) throws IOException {
    try {
      read(d,(long)d.length);
    } catch (Exception e) {
      throw new IOException();
    }
  }

  /**
   * Closes the pipe.
   * @exception Throwable If an error has occurred.
   */
  public void close() throws Throwable {
    SIGUSR1signal.stop();
    SIGUSR1socket.close();
    xiSocket.close();
  }

  /**
   * Necessary for garbage collection.
   * @exception Throwable If an error has occurred.
   */
  protected void finalize() throws Throwable {
    close();
  }
}


abstract class Handler implements Runnable {
  public final static int SIGUSR1=10;

  public abstract void onAttentionProc();
  
  public void run() {
    onAttentionProc();
  }
}


class SIGUSR1Handler extends Handler {
  public final static int MAGIC=2720;
  Handler handler;
  
  public SIGUSR1Handler(XiPipe pipe) {
    handler=pipe.handler;
  }

  public void onAttentionProc() {
    if (handler != null)
      handler.onAttentionProc();
  }
}


/**
 * Class Task provides a multi-threading-environment
 * guaranting an even time distribution among all threads
 * of this class.
 * @version 0.5, 9 May 1996
 * @author  Bernd Nottelmann
 * @see     Signal
 */
class Task extends Thread {
  Runnable runnable;

  /**
   * Instanciates the parent class and sets parameter runnable.
   * @param  r a class with method run() which will be executed
   *           every time when task is switched on.
   */
  public Task(Runnable r) {
    super();
    runnable=r;
  }

  public synchronized void exec() {
    runnable.run();
  }

  public void run() {
    while (true) {
      exec();
      yield();
    }
  }
}


abstract class Signal implements Runnable {
  protected int signum;
  protected Runnable runnable;
  protected DataInputStream in;
  protected DataOutputStream out;
  protected final static int sizeOfInt=4;
  public final static int SIGTERM=15;
  
  public DataInputStream getInputStream() {
    return in;
  }

  public DataOutputStream getOutputStream() {
    return out;
  }

  public synchronized void kill(int signum) throws IOException {
    out.writeInt(signum);
  }
  
  public static synchronized void kill(Signal signal, int signum)
    throws IOException {
    signal.getOutputStream().writeInt(signum);
  }

  public static synchronized void kill(DataOutputStream out, int signum)
    throws IOException {
    out.writeInt(signum);
  }

  abstract public void stop();
}


/**
 * Class SoftSignal installs a method which will be
 * executed every time when an InputStream
 * delivers a definit integer signal.
 * @version 0.5, 9 May 1996
 * @author  Bernd Nottelmann
 * @see     Task
 */
final class SoftSignal extends Signal {
  private Task task;
  private int timeout=0;

  public SoftSignal(int signum, Runnable proc,
		   InputStream in, OutputStream out)
    throws Exception {
    this(signum,proc,in,out,Task.NORM_PRIORITY);
  }
  
  /**
   * Inits the signal and starts the corresponding "daemon".
   * @param  signum   the controlling signal.
   * @param  proc     the class with the reaction run() when
   *                  signal signum is delivered.
   * @param  in       the corresponding input stream.
   * @param  out      the output stream.
   * @param  priority the task priority.
   * @exception Exception If an error has occured.
   */
  public SoftSignal(int signum, Runnable proc,
		    InputStream in, OutputStream out,
		    int priority)
    throws Exception {
    this.signum=signum;
    runnable=proc;
    task=new Task(this);
    this.in=new DataInputStream(in);
    this.out=new DataOutputStream(out);
    task.setPriority(priority);
    task.start();
  }

  public void run() {
    try {
      if (in.available() >= sizeOfInt)
	if (in.readInt() == signum)
	  runnable.run();
    } catch (IOException exc) {}
  }

  public void stop() {
    if (task.isAlive()) {
      task.stop();
      try {
	task.join();
      } catch (InterruptedException exc) {}
    }
  }
}


/**
 * Class ExpSignal installs a method which will be
 * executed with an exponentially decaying frequency,
 * if no signal is delivered by the InputStream.
 * @version 0.5, 15 May 1996
 * @author  Bernd Nottelmann
 * @see     Signal
 */
final class ExpSignal extends Signal {
  private Thread thread;
  private double initInterval, finalInterval;
  private int steps;
  private double expDecay;
  private double timeout;

  public int intBuff;
  public static final int LOOKUP=XiPipe.LOOKUP;
  
  public ExpSignal(int signum, Runnable proc,
		   InputStream in, OutputStream out)
    throws Exception {
    this(signum,proc,in,out,Thread.NORM_PRIORITY+1,
	 20,1000,50);
  }
  
  /**
   * Inits the signal and starts the corresponding "deamon".
   * @param  signum        the controlling signal.
   * @param  proc          the class with the reaction run() when
   *                       signal signum is delivered.
   * @param  in            the corresponding input stream.
   * @param  out           the output stream.
   * @param  priority      the thread priority.
   * @param  initInter     the initial time interval between two "run()s".
   * @param  finalInterval the "final" time interval.
   * @param  steps         counts the steps, in which the final interval
   *                       will be reached.
   * @exception Exception If an error has occured.
   */
  public ExpSignal(int signum, Runnable proc,
		   InputStream in, OutputStream out,
		   int priority,
		   long initInterval, long finalInterval,
		   int steps)
    throws Exception {
    this.signum=signum;
    runnable=proc;
    thread=new Thread(this);
    this.in=new DataInputStream(in);
    this.out=new DataOutputStream(out);
    thread.setPriority(priority);
    
    if (initInterval <= 0 || initInterval > finalInterval || steps <= 0)
      throw new IllegalArgumentException();
    this.initInterval=(double)initInterval;
    this.finalInterval=(double)finalInterval;
    this.steps=steps;
    expDecay=Math.exp(Math.log(this.finalInterval/this.initInterval)/steps);
    timeout=(double)initInterval;
    
    thread.start();
  }

  public void run() {
    while (true) {
      try {
	if (in.available() >= sizeOfInt)
	  if ((intBuff=in.readInt()) == signum) {
	    runnable.run();
	    timeout=initInterval;
	  }
	  else
	    if (intBuff == SIGTERM)
	      System.exit(0);
	    else
	      if (intBuff != LOOKUP)
		timeout=initInterval;
	      else
		timeout=(timeout < finalInterval) ?
		        timeout*expDecay : finalInterval;
	else
	  timeout=(timeout < finalInterval) ?
	          timeout*expDecay : finalInterval;
      } catch (IOException exc) {}
      try {
	Thread.currentThread().sleep((long)timeout);
      } catch (InterruptedException exc) {}
    }
  }

  public void stop() {
    if (thread.isAlive()) {
      thread.stop();
      try {
	thread.join();
      } catch (InterruptedException exc) {}
    }
  }
}

