import java.awt.*;
import java.awt.image.*;
import java.lang.*;
import java.io.*;


/**
 * Graphical device class for receiving graphical commands from xi.
 * @version 0.6, 18 Jun 1996
 * @author  Bernd Nottelmann
 */
public class XiGraphDev implements Runnable {
  int bpp;
  String server;
  int port, signalPort;
  XiPipe pipe;
  XiGraphCmdHandler handler;
  XiGraphCmdId cmdId;
  final int msgReady=0;
  XiWindow window;
  Graphics g;
  int width, height;
  double xScale, yScale;
  Rectangle r, frameR;
  Thread thread;
  int[] instruction;
  int[] intParam;
  char[] strParam;
  StringBuffer strBuff;
  XiByteArrayInputStream in;
  XiByteArrayOutputStream out;
  DataInputStream din;
  DataOutputStream dout;
  int outSize, lastOutSize;
  int[] newPixels=new int[1];
  boolean isReceiving;
  public int cmdCnt, maxCmdCnt;  // For Undo and Redo Buttons.
  
  public XiGraphDev(XiWindow window,
		    int bpp,
		    String server,
		    int port) throws Throwable {
    this(window,0,0,200,200,bpp,server,port);
  }

  /**
   * Full constructor of this class.
   * @param  window the embedding window.
   * @param  minx   the bottom x coordinate of the draw bounds.
   * @param  miny   the bottom y coordinate.
   * @param  maxx   the top x coordinate.
   * @param  maxy   the top y coordinate.
   * @param  server the hostname of the server which gives the commands.
   * @param  port   the corresponding port number.
   * @exception Throwable If an error has occured.
   */
  public XiGraphDev(XiWindow window,
		    int minx, int miny, int maxx, int maxy,
		    int bpp,
		    String server,
		    int port) throws Throwable {
    this.window=window; 
    this.bpp=bpp;
    this.server=server;
    this.port=port;
    signalPort=port+1;
    try {
      pipe=new XiPipe(server,port,signalPort);
      handler=new XiGraphCmdHandler();
      pipe.onAttention(handler);
    } catch (Throwable t) {
      throw t;
    }
    r=window.bounds();
    width=r.width; height=r.height;
    xScale=1; yScale=1;
    instruction=new int[1];
    intParam=new int[4];
    strParam=new char[1024];
    out=new XiByteArrayOutputStream();
    dout=new DataOutputStream(out);
    outSize=lastOutSize=0;
    cmdCnt=maxCmdCnt=0;
    isReceiving=false;
  }

  public void start() {
    thread=new Thread(this);
    thread.start();
  }

  public void stop() {
    thread.stop();
  }

  /**
   * Cuts the last undone commands from the rest.
   * @see    #run
   */
  protected void cutDout() {
    if (cmdCnt < maxCmdCnt) {
      byte[] buf;
      int len;
      maxCmdCnt=cmdCnt;
      din=new DataInputStream(new XiByteArrayInputStream(out.toByteArray()));
      try {
	// Find last not undone command
	while (din.available() > 0 && cmdCnt > 0)
	  switch(din.readInt()) {
	  case cmdId.cmdEndList:
	    break;
	  case cmdId.cmdPSetDrawColor:
	    din.skipBytes(12);
	    break;
	  case cmdId.cmdPErase:
	    din.mark(din.available());
	    --cmdCnt;
	    break;
	  case cmdId.cmdPPoint:
	    din.skipBytes(8);
	    --cmdCnt;
	    break;
	  case cmdId.cmdPPoints:
	    din.skipBytes(8*din.readInt());
	    --cmdCnt;
	    break;
	  case cmdId.cmdPLine:
	    din.skipBytes(16);
	    --cmdCnt;
	    break;
	  case cmdId.cmdPLines:
	    din.skipBytes(8*din.readInt());
	    --cmdCnt;
	    break;
	  case cmdId.cmdDevJAVABackDraw:
	    din.skipBytes(4);
	    break;
	  case cmdId.cmdDevJAVAMap:
	    --cmdCnt;
	    break;
	  case cmdId.cmdDevJAVADrawPixels:
	    din.skipBytes(8);
	    din.skipBytes(4*din.readInt()*din.readInt());
	    --cmdCnt;
	  }
	out.reset(out.size()-din.available());
	cmdCnt=maxCmdCnt;
      } catch (IOException e) {}
    }
  }
  
  /**
   * Waits for commands at the pipe, stores and executes them.
   * @see    #update
   */
  public void run() {
    int i, n;
    int[] x=new int[1], y=new int[1];
    int[] pixels=new int[1];
    
    g=window.getGraphics();
    window.setBackground(Color.black);
    erase();

    try {
      pipe.onAttention(handler);
      pipe.out(msgReady);
      pipe.flush();
      for (;;) {
	cutDout();  // Undo is irreversible when new commands are executed!
	
	isReceiving=true;
	while (pipe.available()) {
	  pipe.read(instruction,1);
	  switch(instruction[0]) {
	  case cmdId.cmdEndList:
	    dout.writeInt(instruction[0]);
	    break;
	  case cmdId.cmdPSetDrawColor:
	    dout.writeInt(instruction[0]);
	    pipe.read(intParam,3);
	    dout.writeInt(intParam[0]);
	    dout.writeInt(intParam[1]);
	    dout.writeInt(intParam[2]);
	    setDrawColor(intParam[0],intParam[1],intParam[2]);
	    break;
	  case cmdId.cmdPErase:
	    dout.writeInt(instruction[0]);
	    erase();
	    ++cmdCnt;
	    break;
	  case cmdId.cmdPPoint:
	    dout.writeInt(instruction[0]);
	    pipe.read(intParam,2);
	    dout.writeInt(intParam[0]);
	    dout.writeInt(intParam[1]);
	    point(intParam[0],intParam[1]);
	    ++cmdCnt;
	    break;
	  case cmdId.cmdPPoints:
	    dout.writeInt(instruction[0]);
	    pipe.read(intParam,1);
	    dout.writeInt(intParam[0]);
	    if (intParam[0] > x.length) {
	      x=new int[intParam[0]];
	      y=new int[intParam[0]];
	    }
	    pipe.read(x,intParam[0]);
	    pipe.read(y,intParam[0]);
	    for (i=0; i<intParam[0]; i++)
	      dout.writeInt(x[i]);
	    for (i=0; i<intParam[0]; i++)
	      dout.writeInt(y[i]);
	    points(intParam[0],x,y);
	    ++cmdCnt;
	    break;
	  case cmdId.cmdPLine:
	    dout.writeInt(instruction[0]);
	    pipe.read(intParam,4);
	    dout.writeInt(intParam[0]);
	    dout.writeInt(intParam[1]);
	    dout.writeInt(intParam[2]);
	    dout.writeInt(intParam[3]);
	    line(intParam[0],intParam[1],intParam[2],intParam[3]);
	    ++cmdCnt;
	    break;
	  case cmdId.cmdPLines:
	    dout.writeInt(instruction[0]);
	    pipe.read(intParam,1);
	    dout.writeInt(intParam[0]);
	    if (intParam[0] > x.length) {
	      x=new int[intParam[0]];
	      y=new int[intParam[0]];
	    }
	    pipe.read(x,intParam[0]);
	    pipe.read(y,intParam[0]);
	    for (i=0; i<intParam[0]; i++)
	      dout.writeInt(x[i]);
	    for (i=0; i<intParam[0]; i++)
	      dout.writeInt(y[i]);
	    lines(intParam[0],x,y);
	    ++cmdCnt;
	    break;
	  case cmdId.cmdDevJAVATitle:
	    pipe.in(strParam);
	    setTitle(strParam);
	    break;
	  case cmdId.cmdDevJAVASize:
	    pipe.read(intParam,2);
	    if (intParam[0] > 0 && intParam[1] > 0) {
	      setSize(intParam[0],intParam[1]);
	    }
	    break;
	  case cmdId.cmdDevJAVAPosition:
	    pipe.read(intParam,2);
	    setPosition(intParam[0],intParam[1]);
	    break;
	  case cmdId.cmdDevJAVABackDraw:
	    dout.writeInt(instruction[0]);
	    pipe.read(intParam,1);
	    dout.writeInt(intParam[0]);
	    backDraw(intParam[0]);
	    break;
	  case cmdId.cmdDevJAVAMap:
	    dout.writeInt(instruction[0]);
	    map();
	    ++cmdCnt;
	    break;
	  case cmdId.cmdDevJAVADrawPixels:
	    dout.writeInt(instruction[0]);
	    pipe.read(intParam,4);
	    dout.writeInt(intParam[0]);
	    dout.writeInt(intParam[1]);
	    dout.writeInt(intParam[2]);
	    dout.writeInt(intParam[3]);
	    if ((n=intParam[2]*intParam[3]) > pixels.length)
	      pixels=new int[n];
	    pipe.read(pixels,n);
	    for (i=0; i<n; i++)
	      dout.writeInt(pixels[i]);
	    drawPixels(pixels,intParam[0],intParam[1],intParam[2],intParam[3]);
	    ++cmdCnt;
	    break;
	  default:
	    System.err.println("Unknown graphic command: "+instruction[0]);
	  }
	  do
	    pipe.read(instruction,1);
	  while(instruction[0] != msgReady);
	}
	maxCmdCnt=cmdCnt;
	isReceiving=false;
	
	if (maxCmdCnt > 0)
	  ((BlackGrayButton)window.panel.comp[1]).setBlack();
	((BlackGrayButton)window.panel.comp[2]).setGray();
	
	if ((outSize=out.size()/1024) != lastOutSize) {
	  ((WhiteTextField)window.panel.comp[5]).setText(new String(outSize+" kb"));
	  lastOutSize=outSize;
	}
	window.lastUpdate=System.currentTimeMillis();
	do {
	  window.repaint();
	  thread.yield();
	} while (!pipe.available());

	cutDout();
	
	r=window.bounds();
	if (width != r.width || height != r.height) {
	  xScale=(double)r.width/width; yScale=(double)r.height/height;
	  isReceiving=true;
	  while (pipe.available()) {
	    pipe.read(instruction,1);
	    switch(instruction[0]) {
	    case cmdId.cmdEndList:
	      dout.writeInt(instruction[0]);
	      break;
	    case cmdId.cmdPSetDrawColor:
	      dout.writeInt(instruction[0]);
	      pipe.read(intParam,3);
	      dout.writeInt(intParam[0]);
	      dout.writeInt(intParam[1]);
	      dout.writeInt(intParam[2]);
	      setDrawColor(intParam[0],intParam[1],intParam[2]);
	      break;
	    case cmdId.cmdPErase:
	      dout.writeInt(instruction[0]);
	      erase();
	      ++cmdCnt;
	      break;
	    case cmdId.cmdPPoint:
	      dout.writeInt(instruction[0]);
	      pipe.read(intParam,2);
	      dout.writeInt(intParam[0]);
	      dout.writeInt(intParam[1]);
	      point((int)(xScale*intParam[0]),(int)(yScale*intParam[1]));
	      ++cmdCnt;
	      break;
	    case cmdId.cmdPPoints:
	      dout.writeInt(instruction[0]);
	      pipe.read(intParam,1);
	      dout.writeInt(intParam[0]);
	      if (intParam[0] > x.length) {
		x=new int[intParam[0]];
		y=new int[intParam[0]];
	      }
	      pipe.read(x,intParam[0]);
	      pipe.read(y,intParam[0]);
	      for (i=0; i<intParam[0]; i++) {
		dout.writeInt(x[i]);
		x[i]=(int)(xScale*x[i]);
	      }
	      for (i=0; i<intParam[0]; i++) {
		dout.writeInt(y[i]);
		y[i]=(int)(yScale*y[i]);
	      }
	      points(intParam[0],x,y);
	      ++cmdCnt;
	      break;
	    case cmdId.cmdPLine:
	      dout.writeInt(instruction[0]);
	      pipe.read(intParam,4);
	      dout.writeInt(intParam[0]);
	      dout.writeInt(intParam[1]);
	      dout.writeInt(intParam[2]);
	      dout.writeInt(intParam[3]);
	      line((int)(xScale*intParam[0]),(int)(yScale*intParam[1]),
		   (int)(xScale*intParam[2]),(int)(yScale*intParam[3]));
	      ++cmdCnt;
	      break;
	    case cmdId.cmdPLines:
	      dout.writeInt(instruction[0]);
	      pipe.read(intParam,1);
	      dout.writeInt(intParam[0]);
	      if (intParam[0] > x.length) {
		x=new int[intParam[0]];
		y=new int[intParam[0]];
	      }
	      pipe.read(x,intParam[0]);
	      pipe.read(y,intParam[0]);
	      for (i=0; i<intParam[0]; i++) {
		dout.writeInt(x[i]);
		x[i]=(int)(xScale*x[i]);
	      }
	      for (i=0; i<intParam[0]; i++) {
		dout.writeInt(y[i]);
		y[i]=(int)(yScale*y[i]);
	      }
	      lines(intParam[0],x,y);
	      ++cmdCnt;
	      break;
	    case cmdId.cmdDevJAVATitle:
	      dout.writeInt(instruction[0]);
	      pipe.in(strParam);
	      dout.writeChars(new String(strParam));
	      setTitle(strParam);
	      break;
	    case cmdId.cmdDevJAVASize:
	      pipe.read(intParam,2);
	      if (intParam[0] > 0 && intParam[1] > 0) {
		setSize(intParam[0],intParam[1]);
	      }
	      break;
	    case cmdId.cmdDevJAVAPosition:
	      pipe.read(intParam,2);
	      setPosition(intParam[0],intParam[1]);
	      break;
	    case cmdId.cmdDevJAVABackDraw:
	      dout.writeInt(instruction[0]);
	      pipe.read(intParam,1);
	      dout.writeInt(intParam[0]);
	      backDraw(intParam[0]);
	      break;
	    case cmdId.cmdDevJAVAMap:
	      dout.writeInt(instruction[0]);
	      map();
	      ++cmdCnt;
	      break;
	    case cmdId.cmdDevJAVADrawPixels:
	      dout.writeInt(instruction[0]);
	      pipe.read(intParam,4);
	      dout.writeInt(intParam[0]);
	      dout.writeInt(intParam[1]);
	      dout.writeInt(intParam[2]);
	      dout.writeInt(intParam[3]);
	      if ((n=intParam[2]*intParam[3]) > pixels.length)
		pixels=new int[n];
	      pipe.read(pixels,n);
	      for (i=0; i<n; i++)
		dout.writeInt(pixels[i]);
	      drawPixels(pixels,
			 intParam[0],intParam[1],intParam[2],intParam[3],
			 xScale,yScale);
	      ++cmdCnt;
	      break;
	    default:
	      System.err.println("Unknown graphic command: "+instruction[0]);
	    }
	    do
	      pipe.read(instruction,1);
	    while(instruction[0] != msgReady);
	  }
	  maxCmdCnt=cmdCnt;
	  isReceiving=false;
	}
      }
    } catch (Throwable t) {
      close();
      System.exit(-1);
    }
    close();
    System.exit(0);
  }

  /**
   * Reads a string from an input stream.
   * @param  in The input stream.
   * @return The read string.
   */
  protected char[] readChars(DataInputStream in) {
    strBuff=new StringBuffer();
    char c;
    try {
      while (in.available() > 0 && (c=in.readChar())!='\0')
	strBuff.append(in.readChar());
    } catch (IOException e) {};
    strBuff.append('\0');
    return strBuff.toString().toCharArray();
  }

  /**
   * This function is called when the window has to be repainted.
   * It reads the stored commands from a buffer and executes them.
   * @see    #run
   */
  public synchronized void update() {
    try {
      if (pipe.available()) return;
    } catch (IOException e) {}
    
    int i=1, n;
    int[] pixels=new int[1];
    int imgX, imgY, imgWidth, imgHeight;
    int upCmdCnt=0, lastUpCmdCnt;
    int[] x=new int[i], y=new int[i];

    while (isReceiving)
      Thread.currentThread().yield();
    
    in=new XiByteArrayInputStream(out.toByteArray(),out.size());
    din=new DataInputStream(in);
    try {
      // Find last erase
      din.mark(din.available());
      lastUpCmdCnt=upCmdCnt;
      while (din.available() > 0 && upCmdCnt < cmdCnt)
	switch(din.readInt()) {
	case cmdId.cmdEndList:
	  break;
	case cmdId.cmdPSetDrawColor:
	  din.skipBytes(12);
	  break;
	case cmdId.cmdPErase:
	  din.mark(din.available());
	  ++upCmdCnt;
	  lastUpCmdCnt=upCmdCnt;
	  break;
	case cmdId.cmdPPoint:
	  din.skipBytes(8);
	  ++upCmdCnt;
	  break;
	case cmdId.cmdPPoints:
	  din.skipBytes(8*din.readInt());
	  ++upCmdCnt;
	  break;
	case cmdId.cmdPLine:
	  din.skipBytes(16);
	  ++upCmdCnt;
	  break;
	case cmdId.cmdPLines:
	  din.skipBytes(8*din.readInt());
	  ++upCmdCnt;
	  break;
	case cmdId.cmdDevJAVABackDraw:
	  din.skipBytes(4);
	  break;
	case cmdId.cmdDevJAVAMap:
	  ++upCmdCnt;
	  break;
	case cmdId.cmdDevJAVADrawPixels:
	  din.skipBytes(8);
	  din.skipBytes(4*din.readInt()*din.readInt());
	  ++upCmdCnt;
	}
      din.reset();
      erase();
      upCmdCnt=lastUpCmdCnt;

      Thread.currentThread().yield();
      try {
	if (pipe.available()) return;
      } catch (IOException e) {}
      
      r=window.bounds();
      if (width==r.width && height==r.height)
	while (din.available() > 0 && upCmdCnt < cmdCnt)
	  switch(din.readInt()) {
	  case cmdId.cmdEndList:
	    break;
	  case cmdId.cmdPSetDrawColor:
	    setDrawColor(din.readInt(),din.readInt(),din.readInt());
	    break;
	  case cmdId.cmdPErase:
	    erase();
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPPoint:
	    point(din.readInt(),din.readInt());
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPPoints:
	    if ((n=din.readInt()) > x.length) {
	      x=new int[n];
	      y=new int[n];
	    }
	    for (i=0; i<n; i++)
	      x[i]=din.readInt();
	    for (i=0; i<n; i++)
	      y[i]=din.readInt();
	    points(n,x,y);
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPLine:
	    line(din.readInt(),din.readInt(),din.readInt(),din.readInt());
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPLines:
	    if ((n=din.readInt()) > x.length) {
	      x=new int[n];
	      y=new int[n];
	    }
	    for (i=0; i<n; i++)
	      x[i]=din.readInt();
	    for (i=0; i<n; i++)
	      y[i]=din.readInt();
	    lines(n,x,y);
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdDevJAVABackDraw:
	    backDraw(din.readInt());
	    break;
	  case cmdId.cmdDevJAVAMap:
	    map();
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdDevJAVADrawPixels:
	    imgX=din.readInt();
	    imgY=din.readInt();
	    imgWidth=din.readInt();
	    imgHeight=din.readInt();
	    n=imgWidth*imgHeight;
	    if (n > pixels.length)
	      pixels=new int[n];
	    for (i=0; i<n; i++)
              pixels[i]=din.readInt();
	    drawPixels(pixels,imgX,imgY,imgWidth,imgHeight);
	    ++upCmdCnt;
	  }
      else {
	xScale=(double)r.width/width; yScale=(double)r.height/height;
	while (din.available() > 0 && upCmdCnt < cmdCnt)
	  switch(din.readInt()) {
	  case cmdId.cmdEndList:
	    break;
	  case cmdId.cmdPSetDrawColor:
	    setDrawColor(din.readInt(),din.readInt(),din.readInt());
	    break;
	  case cmdId.cmdPErase:
	    erase();
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPPoint:
	    point((int)(xScale*din.readInt()),(int)(yScale*din.readInt()));
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPPoints:
	    if ((n=din.readInt()) > x.length) {
	      x=new int[n];
	      y=new int[n];
	    }
	    for (i=0; i<n; i++)
	      x[i]=(int)(xScale*din.readInt());
	    for (i=0; i<n; i++)
	      y[i]=(int)(yScale*din.readInt());
	    points(n,x,y);
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPLine:
	    line((int)(xScale*din.readInt()),(int)(yScale*din.readInt()),
		 (int)(xScale*din.readInt()),(int)(yScale*din.readInt()));
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdPLines:
	    if ((n=din.readInt()) > x.length) {
	      x=new int[n];
	      y=new int[n];
	    }
	    for (i=0; i<n; i++)
	      x[i]=(int)(xScale*din.readInt());
	    for (i=0; i<n; i++)
	      y[i]=(int)(yScale*din.readInt());
	    lines(n,x,y);
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdDevJAVABackDraw:
	    backDraw(din.readInt());
	    break;
	  case cmdId.cmdDevJAVAMap:
	    map();
	    ++upCmdCnt;
	    break;
	  case cmdId.cmdDevJAVADrawPixels:
	    imgX=din.readInt();
	    imgY=din.readInt();
	    imgWidth=din.readInt();
	    imgHeight=din.readInt();
	    n=imgWidth*imgHeight;
	    if (n > pixels.length)
	      pixels=new int[n];
	    for (i=0; i<n; i++)
              pixels[i]=din.readInt();
	    drawPixels(pixels,imgX,imgY,imgWidth,imgHeight,xScale,yScale);
	    ++upCmdCnt;
	  }
      }
    } catch (IOException e) {}
  }
  
  public void setDrawColor(int red, int green, int blue) {
    g.setColor(new Color((float)(red&0xFFFF)/0x10000,
			 (float)(green&0xFFFF)/0x10000,
			 (float)(blue&0xFFFF)/0x10000));
  }
  
  /**
   * Clears the whole output window.
   */
  public synchronized void erase() {
    r=window.bounds();
    if (g != null)
      g.clearRect(0,0,r.width,r.height);
  }

  public void point(int x, int y) {
    g.drawLine(x,y,x,y);
  }

  public void points(int n, int[] x, int[] y) {
    int i;
    for (i=0; i<n; ++i)
      g.drawLine(x[i],y[i],x[i],y[i]);
  }
  
  public void line(int x1, int y1, int x2, int y2) {
    g.drawLine(x1,y1,x2,y2);
  }

  public void lines(int n, int[] x, int[] y) {
    g.drawPolygon(x,y,n);
  }

  public void setTitle(char[] title) {
    window.frame.setTitle(new String(title));
  }
  
  public synchronized void setSize(int width, int height) {
    this.width=width; this.height=height;
    xScale=1; yScale=1;
    r=window.bounds();
    frameR=window.frame.bounds();
    window.frame.reshape(frameR.x,frameR.y,
			 frameR.width-r.width+width,
			 frameR.height-r.height+height);
    window.frame.show();
  }

  public synchronized void setPosition(int x, int y) {
    frameR=window.frame.bounds();
    window.frame.reshape(x,y,frameR.width,frameR.height);
    window.frame.show();
  }

  public void backDraw(int flag) {
  }

  public void map() {
  }

  public void close() {
    try {
      pipe.close();
    } catch (Throwable t) {}
  }

  /**
   * Draws an image of integer pixels.
   * @param  pixels The image as integer array (24 bit true color).
   * @param  x      The corresponding x position.
   * @param  y      The y position.
   * @param  width  The width of the image.
   * @param  height The height.
   */
  public void drawPixels(int[] pixels, int x, int y, int width, int height) {
    Image img=window.createImage(new MemoryImageSource(width,height,
						       pixels,0,width));
    g.drawImage(img,x,y,null);
  }

  /**
   * Draws an image of integer pixels depending on the window size.
   * @param  pixels The image as integer array (24 bit true color).
   * @param  x      The corresponding x position.
   * @param  y      The y position.
   * @param  width  The width of the image.
   * @param  height The height.
   * @param  xscale The scaling factor in x direction.
   * @param  yscale The scaling factor in y direction.
   */
  public void drawPixels(int[] pixels, int x, int y, int width, int height,
			 double xScale, double yScale) {
    int newWidth, newHeight, newSize;
    int dw, dh;
    int i, j, k, l, iCnt, jCnt, iAdd, jAdd;
    boolean wLEnW, hLEnH;
    
    newWidth=(int)(xScale*width); newHeight=(int)(yScale*height);
    if ((newSize=newWidth*newHeight) > newPixels.length)
      newPixels=new int[newSize];
    dw=(wLEnW=(width <= newWidth)) ? width-newWidth : width;
    dh=(hLEnH=(height <= newHeight)) ? height-newHeight : height;
    k=l=jCnt=0;
    for (j=0; j<height;) {
      iCnt=0; l=j*width;
      for (i=0; i<width;) {
	newPixels[k++]=pixels[l+i];
	iCnt+=(iCnt < newWidth) ? width : dw;
	iAdd=(wLEnW) ? 1 : iCnt/newWidth-i;
	i+=(iCnt >= newWidth) ? iAdd : 0;
      }
      jCnt+=(jCnt < newHeight) ? height : dh;
      jAdd=(hLEnH) ? 1 : jCnt/newHeight-j;
      j+=(jCnt >= newHeight) ? jAdd : 0;
    }
    Image img=window.createImage(new MemoryImageSource(newWidth,newHeight,
						       newPixels,0,newWidth));
    g.drawImage(img,(int)(xScale*x),(int)(yScale*y),null);
  }

  public void send(String s) {
    try {
      pipe.out(s); pipe.flush();
    } catch (IOException e) {}
  }
}


class XiGraphCmdHandler extends Handler {
  public boolean interrupt=false;

  public void onAttentionProc() {
    interrupt=true;
  }
}
