/*
rotate.c Version 0.4.0. Geodesic Dome Rotation
Copyright (C) 2001-2010  dondalah721@yahoo.com (Dondalah)

Xdome is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

Xdome is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

 	Free Software Foundation, Inc.
 	51 Franklin St, Fifth Floor
 	Boston, MA  02110-1301, USA
*/

/* The rotation in this program is based upon: */

/* Peter Duffett-Smith */
/* Practical astronomy with your calculator */
/* 3rd Edition */
/* Cambridge University Press, 1979, 1988 */
/* ISBN 0-521-35629-6  hardback  */
/* ISBN 0-521-35699-7  paperback */
/* Section 28, Equatorial to Eliptic Coordinate Conversion */
/* Section 31, Generalised Coordinate Transformations */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define LEFT     0
#define RIGHT    1
#define UP       2
#define DOWN     3
#define CCLKWISE 4
#define CLKWISE  5

void rotate(int direction,
   double epsilon,
   double exx,
   double wyy,
   double zee,
   double *newexx,
   double *newwyy,
   double *newzee)
   {
   /* epsilon is the rotation in radians */
   /*        Rotation table              */
   /* direction          rotation        */
   /* ---------       ---------------    */
   /*    0            horizontal left    */
   /*    1            horizontal right   */
   /*    2            vertical   up      */
   /*    3            vertical   down    */
   /*    4            counter clockwise  */
   /*    5            clockwise          */
   /* Section 31 Table 4 Example C */
   /* equatorial to eliptic conversion */
   /* Matrix A  (3x3) variables */
   /* a b c */
   /* d e f */
   /* g h i */
   /* values are */
   /* 1       0           0        */
   /* 0  cos(epsilon) sin(epsilon) */
   /* 0 -sin(epsilon) cos(epsilon) */
   /* Vector v (1x3) variables */
   /* These are the input cartesian coordinates */
   /* x */
   /* y */
   /* z */
   /* Matrix product: w = A * v */
   /* Vector w (1x3) variables */
   /* These are the output cartesian coordinates */
   /* m */
   /* n */
   /* p */
   /* Convert the equatorial coordinates */
   /* alpha and delta */
   /* to the eliptical coordinates */
   /* lambda and beta */
   /* as in Section 28 using the */
   /* generalized method in Section 31 */
   /* from Duffett-Smith */
   /* matrix A */
   double ma,mb,mc,md,me,mf,mg,mh,mi;
   /* vector v */
   double mx,my,mz;
   /* vector w */
   double mm,mn,mp;
   if (direction < 0 || direction > 5)
      {
      fprintf(stderr,"rotate: invalid "
	 "direction %d ",
	 direction);
      exit(1);
      } /* if invalid direction */
   /* define matrix A */
   ma = 1.0;
   mb = 0.0;
   mc = 0.0;
   md = 0.0;
   me = mi = cos(epsilon);
   mf = sin(epsilon);
   mg = 0.0;
   mh = -mf;
   /* mi = cos(epsilon) */
   /* define vector v */
   mx = exx;
   my = wyy;
   mz = zee;
   /* matrix multiplication */
   /* w = A * v */
   mm = (mx * ma) + (my * mb) + (mz * mc);
   mn = (mx * md) + (my * me) + (mz * mf);
   mp = (mx * mg) + (my * mh) + (mz * mi);
   /* depending on the direction of rotation */
   /* define the new cartesian coordinates */
   /* define new exx */
   switch (direction)
      {
      case UP:
      case DOWN:
         *newwyy = mm;
	 break;
      case CCLKWISE:
      case RIGHT:
         *newwyy = mn;
	 break;
      case CLKWISE:
      case LEFT:
         *newwyy = mp;
	 break;
      default:
         fprintf(stderr,"rotate: invalid "
	    "direction %d ",
	    direction);
         exit(1);
      } /* switch direction */
   /* define new zee */
   switch (direction)
      {
      case LEFT:
      case RIGHT:
         *newzee = mm;
	 break;
      case CLKWISE:
      case DOWN:
         *newzee = mn;
	 break;
      case UP:
      case CCLKWISE:
         *newzee = mp;
	 break;
      default:
         fprintf(stderr,"rotate: invalid "
	    "direction %d ",
	    direction);
         exit(1);
      } /* switch direction */
   /* define new exx */
   switch (direction)
      {
      /* counter clockwise */
      /* clockwise */
      case CCLKWISE:
      case CLKWISE:
         *newexx = mm;
	 break;
      /* horizontal left */
      /* vertical up */
      case LEFT:
      case UP:
         *newexx = mn;
	 break;
      /* horizontal right */
      /* vertical down */
      case RIGHT:
      case DOWN:
         *newexx = mp;
	 break;
      default:
         fprintf(stderr,"rotate: invalid "
	    "direction %d ",
	    direction);
         exit(1);
      } /* switch direction */
   } /* rotate */
