/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"

#define WIDTH  100
#define HEIGHT 160

static char * color_xpm[] = {
"40 20 4 1",
"       c #FFFFFFFFFFFF",
".      c #CCCCCCCCCCCC",
"X      c #616161616161",
"o      c #000000000000",
"                                        ",
" ......................................X",
" .XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .X                                   .X",
" ......................................X",
" XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"};

static GtkWidget *main_box;
static GtkWidget *notebook = NULL;
static GtkWidget *dataset_combo;
static GtkWidget *symbol_type_combo;
static GtkWidget *symbol_style_combo;
static GtkWidget *line_style_combo;
static GtkWidget *connector_combo;
static GtkWidget *symbol_color_combo;
static GtkWidget *line_color_combo;
static GtkWidget *border_color_combo;
static GtkWidget *line_width_spin;
static GtkWidget *err_width_spin;
static GtkWidget *err_caps_spin;
static GtkWidget *symbol_size_spin;
static GtkWidget *symbol_width_spin;
static GtkWidget *width_entry;
static GtkWidget *yerr_check;
static GtkWidget *xerr_check;
static GtkWidget *zerr_check;
static GtkWidget *fill_check;
static GtkWidget *arrow_combo;
static GtkWidget *arrow_length_spin;
static GtkWidget *arrow_width_spin;
static GtkWidget *arrow_center_check;
static GtkWidget *text_entry;
static GtkWidget *xline_combo;
static GtkWidget *yline_combo;
static GtkWidget *zline_combo;
static GtkWidget *xline_color_combo;
static GtkWidget *xline_width_spin;
static GtkWidget *font_combo;
static GtkWidget *font_color_combo;
static GtkWidget *labels_check;
static GtkWidget *gradient;
static GtkWidget *gradient_check;
static GtkWidget *button_min, *button_max;
static GtkWidget *min_spin, *max_spin, *levels_spin;
static GtkWidget *color_pixmap;
static GtkWidget *csd;
static GtkWidget *h_check, *s_check, *v_check;
static GdkColor color_min, color_max;
static GtkWidget *grid_check;
static GtkWidget *mesh_check;
static GtkWidget *project_check;
static GtkWidget *lines_check;
static GtkWidget *fg_color_combo;
static GtkWidget *bg_color_combo;
static GList *columns;
static SGlayer *the_layer;


static void update_default		();
static void update_surface		();
static void update_gradient		();
static void init_default_dialog		();
static void init_surface_dialog		();
static void init_gradient_dialog	();
static void build_default_dialog	();
static void build_surface_dialog	();
static void build_gradient_dialog	();
static void open_color_selection	(GtkWidget *widget, gpointer data);

static gboolean
mw_destroy(GtkWidget *widget)
{
  sg_dialog_kill(widget);

  /* This is needed to get out of gtk_main */
  gtk_main_quit ();

  return FALSE;
}

void
sg_style_dialog_update_plot(GtkWidget *widget)
{
  SGplot *plot;
  gchar *legend_text;

  if(active_dataset->type != SG_DATA_SURFACE &&
     active_dataset->type != SG_DATA_CSURFACE &&
     active_dataset->type != SG_DATA_CONTOUR) {
         update_default();
         init_default_dialog();
  } else {
         update_surface();
         init_surface_dialog();
  }

  if(active_dataset->type == SG_DATA_CSURFACE ||
     active_dataset->type == SG_DATA_CONTOUR || 
     active_dataset->type == SG_DATA_COLORS || 
     active_dataset->type == SG_DATA_FLUX){ 
         update_gradient();
         init_gradient_dialog();
  }

  /* Legends */
  legend_text = gtk_entry_get_text(GTK_ENTRY(text_entry));

  gtk_plot_data_set_legend(active_dataset->real_data, legend_text);


  plot = (SGplot *)the_layer->parent;
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
update_default()
{
  GtkPlotData *data;
  GtkWidget *child;
  GdkColormap *colormap;
  GdkColor line_color, symbol_color, font_color;
  gint line_style; 
  gfloat line_width;
  gint symbol_style, symbol_type, symbol_size;
  gfloat symbol_width; 
  gint connector;
  gint row, col;
  gint font_height;
  gchar *name;
  gchar *font_name;
  gboolean state;

  data = GTK_PLOT_DATA(active_dataset->real_data); 
  colormap = gdk_colormap_get_system(); 

  if(active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_CONTOUR){

      child = GTK_LIST(GTK_COMBO(line_style_combo)->list)->selection->data;
      line_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(line_style_combo)->list), child);

      child = GTK_LIST(GTK_COMBO(connector_combo)->list)->selection->data;
      connector = gtk_list_child_position(GTK_LIST(GTK_COMBO(connector_combo)->list), child);

      line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(line_width_spin));

      data->fill_area = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(fill_check));

      row = GTK_COLOR_COMBO(line_color_combo)->row;
      col = GTK_COLOR_COMBO(line_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(line_color_combo), row, col);
      gdk_color_parse(name, &line_color);
      gdk_color_alloc(colormap, &line_color);

      gtk_plot_data_set_connector(data, connector);

      gtk_plot_data_set_line_attributes(data,
                                        (GtkPlotLineStyle) line_style,
                                        line_width,
                                        &line_color);
  }
 
  /* Symbols */

  if(active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_CONTOUR &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_HBARS){
      GdkColor border;

      if(active_dataset->type != SG_DATA_VBOXES &&
         active_dataset->type != SG_DATA_HBOXES)
           symbol_type = sg_toggle_symbol_get_selection(symbol_type_combo);

      child = GTK_LIST(GTK_COMBO(symbol_style_combo)->list)->selection->data;
      symbol_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(symbol_style_combo)->list), child);

      symbol_size = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(symbol_size_spin));
      symbol_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(symbol_width_spin));

      row = GTK_COLOR_COMBO(symbol_color_combo)->row;
      col = GTK_COLOR_COMBO(symbol_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(symbol_color_combo), row, col);
      gdk_color_parse(name, &symbol_color);
      gdk_color_alloc(colormap, &symbol_color);

      row = GTK_COLOR_COMBO(border_color_combo)->row;
      col = GTK_COLOR_COMBO(border_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(border_color_combo), row, col);
      gdk_color_parse(name, &border);
      gdk_color_alloc(colormap, &border);

      gtk_plot_data_set_symbol(data,
                               (GtkPlotSymbolType) symbol_type,
                               (GtkPlotSymbolStyle) symbol_style,
                               symbol_size,
                               symbol_width,
                               &symbol_color,
                               &border);
  }


  /* X-Y-Z lines */

  if(the_layer->type != SG_PLOT_POLAR &&
     active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_CONTOUR){
      child = GTK_LIST(GTK_COMBO(xline_combo)->list)->selection->data;
      line_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(xline_combo)->list), child);

      line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(xline_width_spin));

      row = GTK_COLOR_COMBO(xline_color_combo)->row;
      col = GTK_COLOR_COMBO(xline_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(xline_color_combo), row, col);
      gdk_color_parse(name, &line_color);
      gdk_color_alloc(colormap, &line_color);

      gtk_plot_data_set_x_attributes(data, 
                                     line_style,
                                     line_width,
                                     &line_color);

      child = GTK_LIST(GTK_COMBO(yline_combo)->list)->selection->data;
      line_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(yline_combo)->list), child);

      gtk_plot_data_set_y_attributes(data, 
                                     line_style,
                                     line_width,
                                     &line_color);

      if(the_layer->type == SG_PLOT_3D){
        child = GTK_LIST(GTK_COMBO(zline_combo)->list)->selection->data;
        line_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(zline_combo)->list), child);

        gtk_plot_data_set_z_attributes(data, 
                                       line_style,
                                       line_width,
                                       &line_color);
      }
  }

  /* Error bars */

  if(active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_FLUX &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_BUBBLES &&
     active_dataset->type != SG_DATA_CONTOUR){
       if(the_layer->type == SG_PLOT_3D){
         data->show_zerrbars = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(zerr_check));
         data->zerrbar_caps = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(err_caps_spin));
         data->zerrbar_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(err_width_spin));
       } else {
         if(active_dataset->type != SG_DATA_VBOXES)
           data->show_xerrbars = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(xerr_check));
         if(active_dataset->type != SG_DATA_HBOXES)
           data->show_yerrbars = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(yerr_check));

         data->xerrbar_caps = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(err_caps_spin));
         data->yerrbar_caps = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(err_caps_spin));
                                        
         data->xerrbar_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(err_width_spin));
         data->yerrbar_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(err_width_spin));

       }
  }

  /* Arrows */

  if(active_dataset->type == SG_DATA_FLUX){
    child = GTK_LIST(GTK_COMBO(arrow_combo)->list)->selection->data;
    GTK_PLOT_FLUX(data)->arrow_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(arrow_combo)->list), child);

    GTK_PLOT_FLUX(data)->arrow_length = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(arrow_length_spin));
    GTK_PLOT_FLUX(data)->arrow_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(arrow_width_spin));
    GTK_PLOT_FLUX(data)->centered = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(arrow_center_check));
  }


  if(active_dataset->type == SG_DATA_LPOINTS ||
     active_dataset->type == SG_DATA_CONTOUR){
    data->show_labels = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(labels_check));

    font_name = GTK_FONT_COMBO(font_combo)->psfont->psname;
    font_height = GTK_FONT_COMBO(font_combo)->height;
  
    row = GTK_COLOR_COMBO(xline_color_combo)->row;
    col = GTK_COLOR_COMBO(xline_color_combo)->column;
    name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(font_color_combo), row, col);
    gdk_color_parse(name, &font_color);
    gdk_color_alloc(colormap, &font_color);

    gtk_plot_data_labels_set_attributes(data,
                                        font_name, font_height, 90, 
                                        &font_color, NULL);
  }

  /* Bars */

  if(active_dataset->type == SG_DATA_HBARS ||
     active_dataset->type == SG_DATA_VBARS){
      GdkColor border;
      gchar *text;

      text = gtk_entry_get_text(GTK_ENTRY(width_entry));
      GTK_PLOT_BAR(data)->width = atof(text);

      child = GTK_LIST(GTK_COMBO(symbol_style_combo)->list)->selection->data;
      symbol_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(symbol_style_combo)->list), child);

      symbol_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(symbol_width_spin));

      row = GTK_COLOR_COMBO(symbol_color_combo)->row;
      col = GTK_COLOR_COMBO(symbol_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(symbol_color_combo), row, col);
      gdk_color_parse(name, &symbol_color);
      gdk_color_alloc(colormap, &symbol_color);

      row = GTK_COLOR_COMBO(border_color_combo)->row;
      col = GTK_COLOR_COMBO(border_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(border_color_combo), row, col);
      gdk_color_parse(name, &border);
      gdk_color_alloc(colormap, &border);

      gtk_plot_data_set_symbol(data,
                               (GtkPlotSymbolType) symbol_type,
                               (GtkPlotSymbolStyle) symbol_style,
                               0,
                               symbol_width,
                               &symbol_color,
                               &border);
  }
}

static void
update_surface()
{
  GtkPlotSurface *data;
  GtkWidget *child;
  GdkColormap *colormap;
  GdkColor color;
  gint line_style, line_width;
  gint symbol_style, symbol_type, symbol_size, symbol_width, connector;
  gint row, col;
  gint font_height;
  gchar *name;
  gboolean state;

  data = GTK_PLOT_SURFACE(active_dataset->real_data); 
  colormap = gdk_colormap_get_system(); 

  if(active_dataset->type == SG_DATA_SURFACE){
      row = GTK_COLOR_COMBO(fg_color_combo)->row;
      col = GTK_COLOR_COMBO(fg_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(fg_color_combo), row, col);
      gdk_color_parse(name, &color);
      gdk_color_alloc(colormap, &color);

      data->color = color;

      row = GTK_COLOR_COMBO(bg_color_combo)->row;
      col = GTK_COLOR_COMBO(bg_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(bg_color_combo), row, col);
      gdk_color_parse(name, &color);
      gdk_color_alloc(colormap, &color);

      data->shadow = color;

      row = GTK_COLOR_COMBO(xline_color_combo)->row;
      col = GTK_COLOR_COMBO(xline_color_combo)->column;
      name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(xline_color_combo), row, col);
      gdk_color_parse(name, &color);
      gdk_color_alloc(colormap, &color);

      data->grid_background = color;
  }
  row = GTK_COLOR_COMBO(line_color_combo)->row;
  col = GTK_COLOR_COMBO(line_color_combo)->column;
  name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(line_color_combo), row, col);
  gdk_color_parse(name, &color);
  gdk_color_alloc(colormap, &color);

  data->grid_foreground = color;

  data->show_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(grid_check));
  data->show_mesh = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mesh_check));

  if(active_dataset->type == SG_DATA_CSURFACE ||
     active_dataset->type == SG_DATA_CONTOUR){
      GtkPlotCSurface *contour = GTK_PLOT_CSURFACE(data); 
      contour->lines_visible = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lines_check));
      contour->project_xy = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(project_check));
  }

}
 
static void
update_gradient()
{
  GtkPlotData *data;
  GtkWidget *child;
  GdkColormap *colormap;
  GdkColor line_color, symbol_color, font_color;
  gint line_style, line_width;
  gint symbol_style, symbol_type, symbol_size, symbol_width, connector;
  gint row, col;
  gint font_height;
  gchar *name;
  gchar *font_name;
  gboolean state;

  data = GTK_PLOT_DATA(active_dataset->real_data); 
  colormap = gdk_colormap_get_system(); 

  data->gradient_mask = 0;
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(h_check));
  data->gradient_mask |= state * GTK_PLOT_GRADIENT_H;
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(s_check));
  data->gradient_mask |= state * GTK_PLOT_GRADIENT_S;
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(v_check));
  data->gradient_mask |= state * GTK_PLOT_GRADIENT_V;

  data->show_gradient = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gradient_check));

  data->color_max = color_max;
  data->color_min = color_min;

  data->gradient.begin = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(min_spin));
  data->gradient.end = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(max_spin));
  data->gradient.nmajorticks = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(levels_spin));

  gtk_plot_data_set_gradient(data, data->gradient.begin, data->gradient.end, 
                             data->gradient.nmajorticks);

}


static void
init_default_dialog()
{
  GtkPlotData *data;
  GdkColor line_color, symbol_color, border_color, color;
  GdkGC *gc;
  GtkPlotLineStyle line_style;
  SGdataset *dataset;
  gfloat line_width;
  GtkPlotSymbolStyle symbol_style;
  GtkPlotSymbolType symbol_type; 
  GtkPSFont *font;
  gfloat symbol_width; 
  gint symbol_size, connector;
  gint y, level;
  gdouble h, step;

  dataset = active_dataset;
  data = GTK_PLOT_DATA(dataset->real_data);

  if(active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_CONTOUR){
      gtk_plot_data_get_line_attributes(active_dataset->real_data,
                                        &line_style, &line_width, &line_color);
 
      connector = gtk_plot_data_get_connector(active_dataset->real_data);

      gtk_list_select_item(GTK_LIST(GTK_COMBO(line_style_combo)->list), line_style); 
      gtk_list_select_item(GTK_LIST(GTK_COMBO(connector_combo)->list), connector); 

      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fill_check), 
                                gtk_plot_data_area_is_filled(active_dataset->real_data));

      gtk_spin_button_set_value(GTK_SPIN_BUTTON(line_width_spin), line_width);
      sg_color_combo_init(GTK_COLOR_COMBO(line_color_combo), line_color);
  }

  /* Symbols */

  gtk_plot_data_get_symbol(active_dataset->real_data,
                           &symbol_type, &symbol_style, 
                           &symbol_size, &symbol_width, 
                           &symbol_color, &border_color);

  if(active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_CONTOUR &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_HBARS){

      if(active_dataset->type != SG_DATA_VBOXES &&
         active_dataset->type != SG_DATA_HBOXES)
           sg_toggle_symbol_init(symbol_type_combo, symbol_type);

      gtk_list_select_item(GTK_LIST(GTK_COMBO(symbol_style_combo)->list), symbol_style); 

      gtk_spin_button_set_value(GTK_SPIN_BUTTON(symbol_size_spin), symbol_size);

      gtk_spin_button_set_value(GTK_SPIN_BUTTON(symbol_width_spin), symbol_width);

      sg_color_combo_init(GTK_COLOR_COMBO(symbol_color_combo), symbol_color);
      sg_color_combo_init(GTK_COLOR_COMBO(border_color_combo), border_color);
  }



  /* X-Y-Z lines */

  if(the_layer->type != SG_PLOT_POLAR &&
     active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_CONTOUR){
      gtk_list_select_item(GTK_LIST(GTK_COMBO(xline_combo)->list), 
                           dataset->real_data->x_line.line_style); 
      gtk_list_select_item(GTK_LIST(GTK_COMBO(yline_combo)->list), 
                           dataset->real_data->y_line.line_style); 
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(xline_width_spin),
                                dataset->real_data->x_line.line_width);
      sg_color_combo_init(GTK_COLOR_COMBO(xline_color_combo), 
                          dataset->real_data->x_line.color);
      if(the_layer->type == SG_PLOT_3D){
           gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(zerr_check), 
                                 dataset->real_data->show_zerrbars);
      }
  }

  /* Error bars */

  if(active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_FLUX &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_BUBBLES &&
     active_dataset->type != SG_DATA_CONTOUR){
      if(the_layer->type == SG_PLOT_3D){
         gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(zerr_check), 
                               dataset->real_data->show_zerrbars);
      } else {
         gtk_spin_button_set_value(GTK_SPIN_BUTTON(err_width_spin),
                                     dataset->real_data->xerrbar_width);
         gtk_spin_button_set_value(GTK_SPIN_BUTTON(err_caps_spin),
                                     dataset->real_data->xerrbar_caps);
         if(active_dataset->type != SG_DATA_VBOXES)
           gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(xerr_check), 
                                        dataset->real_data->show_xerrbars);
         if(active_dataset->type != SG_DATA_HBOXES)
           gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(yerr_check), 
                                        dataset->real_data->show_yerrbars);
      }
  }

  /* Labels */

  if(active_dataset->type == SG_DATA_LPOINTS){
    font = gtk_psfont_get_font(dataset->real_data->labels_attr.font);
    gtk_font_combo_select(GTK_FONT_COMBO(font_combo),
                          font->family,
                          font->bold,
                          font->italic,
                          dataset->real_data->labels_attr.height);

    sg_color_combo_init(GTK_COLOR_COMBO(font_color_combo),
                        dataset->real_data->labels_attr.fg);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(labels_check),
				 dataset->real_data->show_labels);
  }

  /* Bars */

  if(active_dataset->type == SG_DATA_HBARS ||
     active_dataset->type == SG_DATA_VBARS){
       gchar text[100];

       sprintf(text, "%f", GTK_PLOT_BAR(data)->width);
       gtk_entry_set_text(GTK_ENTRY(width_entry), text);

       gtk_list_select_item(GTK_LIST(GTK_COMBO(symbol_style_combo)->list), symbol_style); 

       gtk_spin_button_set_value(GTK_SPIN_BUTTON(symbol_width_spin), symbol_width);

       sg_color_combo_init(GTK_COLOR_COMBO(symbol_color_combo), symbol_color);
       sg_color_combo_init(GTK_COLOR_COMBO(border_color_combo), border_color);
  }        

}

static void
init_surface_dialog()
{
  SGdataset *dataset;
  GtkPlotSurface *data;

  dataset = active_dataset;
  data = GTK_PLOT_SURFACE(dataset->real_data);

  if(dataset->type == SG_DATA_SURFACE){
    sg_color_combo_init(GTK_COLOR_COMBO(fg_color_combo), data->color);
    sg_color_combo_init(GTK_COLOR_COMBO(bg_color_combo), data->shadow);
    sg_color_combo_init(GTK_COLOR_COMBO(xline_color_combo), data->grid_background);
  }  
  sg_color_combo_init(GTK_COLOR_COMBO(line_color_combo), data->grid_foreground);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(grid_check), data->show_grid);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mesh_check), data->show_mesh);

  if(dataset->type == SG_DATA_CSURFACE ||
     dataset->type == SG_DATA_CONTOUR){
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(project_check), 
                                  GTK_PLOT_CSURFACE(data)->project_xy);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lines_check), 
                                  GTK_PLOT_CSURFACE(data)->lines_visible);
  }

}

static void
init_gradient_dialog()
{
  GtkPlotData *data;
  GdkColor line_color, symbol_color, border_color, color;
  GdkGC *gc;
  GtkPlotLineStyle line_style;
  SGdataset *dataset;
  gint line_width;
  GtkPlotSymbolStyle symbol_style;
  GtkPlotSymbolType symbol_type; 
  GtkPSFont *font;
  gint symbol_width, symbol_size, connector;
  gint y, level;
  gdouble h, step;

  dataset = active_dataset;
  data = GTK_PLOT_DATA(dataset->real_data);


  /* Gradient */
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(min_spin), data->gradient.begin);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(max_spin), data->gradient.end);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(levels_spin), data->gradient.nmajorticks);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(h_check),
			       data->gradient_mask & GTK_PLOT_GRADIENT_H);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(s_check),
			       data->gradient_mask & GTK_PLOT_GRADIENT_S);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(v_check),
			       data->gradient_mask & GTK_PLOT_GRADIENT_V);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gradient_check),
			       data->show_gradient);

  gdk_draw_rectangle(GTK_PIXMAP(gradient)->pixmap, 
                     iconlist1->style->white_gc, TRUE, 0, 0, -1, -1);

  y = 0;
  gc = gdk_gc_new(iconlist1->window);
  step = (data->gradient.end - data->gradient.begin) / HEIGHT;

  for(h = data->gradient.end; h >= data->gradient.begin; h -= step){
    gtk_plot_data_get_gradient_level(data, h, &color);
    gdk_color_alloc(gdk_colormap_get_system(), &color);
    gdk_gc_set_foreground(gc, &color);

    gdk_draw_line(GTK_PIXMAP(gradient)->pixmap, gc, 0, y, WIDTH, y);

    y++;
  }

  gdk_gc_set_foreground(gc, &iconlist1->style->black);
  gdk_draw_rectangle(GTK_PIXMAP(gradient)->pixmap, gc, FALSE,
                        0, 0, WIDTH - 1, HEIGHT - 1); 

  y = 0;
  step = HEIGHT / data->gradient.nmajorticks;
  for(level = data->gradient.nmajorticks; level >= 0; level--){
    h = data->gradient.major_values[level];
    gdk_draw_line(GTK_PIXMAP(gradient)->pixmap, gc,
                     0, y,
                     4, y);
    gdk_draw_line(GTK_PIXMAP(gradient)->pixmap, gc,
                     WIDTH - 5, y,
                     WIDTH - 1, y);
    y += step;
  }
  gtk_widget_draw(gradient, NULL);

  color_max = data->color_max;
  gdk_gc_set_foreground(gc, &color_max);
  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(button_max)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(button_max)->child, NULL);

  color_min = data->color_min;
  gdk_gc_set_foreground(gc, &color_min);
  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(button_min)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(button_min)->child, NULL);

  gdk_gc_unref(gc);
}

static void
build_dialog()
{
  notebook = gtk_notebook_new();
  gtk_box_pack_start(GTK_BOX(main_box), notebook, FALSE, FALSE, 0);

  if(active_dataset->type != SG_DATA_SURFACE &&
     active_dataset->type != SG_DATA_CSURFACE &&
     active_dataset->type != SG_DATA_CONTOUR) {
         build_default_dialog();
         init_default_dialog();
  } else {
         build_surface_dialog();
         init_surface_dialog();
  }

  if(active_dataset->type == SG_DATA_CSURFACE ||
     active_dataset->type == SG_DATA_CONTOUR || 
     active_dataset->type == SG_DATA_COLORS || 
     active_dataset->type == SG_DATA_FLUX){ 
         build_gradient_dialog();
         init_gradient_dialog();
  }

  gtk_widget_show_all(notebook);
}

static void
build_default_dialog()
{
  GtkWidget *frame;
  GtkWidget *table, *table1;
  GtkWidget *label;
  GtkWidget *box;
  SGdataset *dataset;
  GtkRequisition req;
  GtkAdjustment *adj;
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  table = gtk_table_new(3, 4, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                           gtk_label_new("Symbols"));

  req.height = 16;
  req.width = 75;

  /* Lines */

  if(active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_CONTOUR){
      frame = gtk_frame_new("Line");
      gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
      gtk_table_attach_defaults(GTK_TABLE(table),frame,0,1,1,2);

      table1 = gtk_table_new(6, 2, FALSE);
      gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
      gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
      gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
      gtk_container_add (GTK_CONTAINER (frame), table1);

      label = gtk_label_new("Connect:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);

      connector_combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),connector_combo,1,2,0,1);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(connector_combo)->entry), FALSE);
      gtk_widget_size_request(GTK_WIDGET(connector_combo), &req);
      req.width /= 2;

      sg_combo_set_items(GTK_COMBO(connector_combo), connectors);

      label = gtk_label_new("Type:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);

      line_style_combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),line_style_combo,1,2,1,2);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(line_style_combo)->entry), FALSE);

      sg_combo_set_items(GTK_COMBO(line_style_combo), line_styles);

      label = gtk_label_new("Width:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,2,3);

      adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., .5, 1., 0.);
      line_width_spin = gtk_spin_button_new(adj, 0.5, 1);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(line_width_spin), TRUE);
      gtk_table_attach_defaults(GTK_TABLE(table1),line_width_spin,1,2,2,3);

      label = gtk_label_new("Color:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,3,4);

      line_color_combo = gtk_color_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),line_color_combo,1,2,3,4);

      fill_check = gtk_check_item_new_with_label("Fill area");
      gtk_table_attach_defaults(GTK_TABLE(table1),fill_check,0,2,4,5);

      gtk_widget_set_usize(connector_combo, req.width, req.height);
      gtk_widget_set_usize(line_width_spin, req.width, req.height);
      gtk_widget_set_usize(line_style_combo, req.width, req.height);
  }

  /* Symbols */

  if(active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_CONTOUR &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_HBARS){

      frame = gtk_frame_new("Symbol");
      gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
      gtk_table_attach_defaults(GTK_TABLE(table),frame,1,2,1,2);

      table1 = gtk_table_new(6, 2, FALSE);
      gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
      gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
      gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
      gtk_container_add (GTK_CONTAINER (frame), table1);

      if(active_dataset->type != SG_DATA_VBOXES &&
         active_dataset->type != SG_DATA_HBOXES){
           label = gtk_label_new("Shape:");
           gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
           gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);

           symbol_type_combo = sg_toggle_symbol_new();
           gtk_table_attach_defaults(GTK_TABLE(table1),symbol_type_combo,1,2,0,1);
           gtk_widget_set_usize(symbol_type_combo, req.width, req.height);
      }

      label = gtk_label_new("Style:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);

      symbol_style_combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),symbol_style_combo,1,2,1,2);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(symbol_style_combo)->entry), FALSE);

      sg_combo_set_items(GTK_COMBO(symbol_style_combo), symbol_styles);

      label = gtk_label_new("Size:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,2,3);

      adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 20., 1., 1., 0.);
      symbol_size_spin = gtk_spin_button_new(adj, 1., 0);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(symbol_size_spin), TRUE);
      gtk_spin_button_set_digits(GTK_SPIN_BUTTON(symbol_size_spin), 0);
      gtk_table_attach_defaults(GTK_TABLE(table1),symbol_size_spin,1,2,2,3);

      label = gtk_label_new("Thickness:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,3,4);

      adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
      symbol_width_spin = gtk_spin_button_new(adj, 0.5, 1);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(symbol_width_spin), TRUE);
      gtk_table_attach_defaults(GTK_TABLE(table1),symbol_width_spin,1,2,3,4);

      label = gtk_label_new("Color:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,4,5);

      symbol_color_combo = gtk_color_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),symbol_color_combo,1,2,4,5);

      label = gtk_label_new("Border:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,5,6);

      border_color_combo = gtk_color_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),border_color_combo,1,2,5,6);

      gtk_widget_set_usize(symbol_size_spin, req.width, req.height);
      gtk_widget_set_usize(symbol_width_spin, req.width, req.height);
      gtk_widget_set_usize(symbol_style_combo, req.width, req.height);
  }

  /* X-Y-Z lines */

  if(the_layer->type != SG_PLOT_POLAR &&
     active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_CONTOUR){

      if(the_layer->type == SG_PLOT_3D)
        frame = gtk_frame_new("X-Y-Z lines");
      else
        frame = gtk_frame_new("X-Y lines");
      
      gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

      table1 = gtk_table_new(6, 2, FALSE);
      gtk_table_attach_defaults(GTK_TABLE(table),frame,2,3,1,2);

      gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
      gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
      gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
      gtk_container_add (GTK_CONTAINER (frame), table1);

      label = gtk_label_new("X line:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);

      xline_combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),xline_combo,1,2,0,1);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(xline_combo)->entry), FALSE);
      sg_combo_set_items(GTK_COMBO(xline_combo), line_styles);

      label = gtk_label_new("Y line:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);

      yline_combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),yline_combo,1,2,1,2);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(yline_combo)->entry), FALSE);
      sg_combo_set_items(GTK_COMBO(yline_combo), line_styles);

      if(the_layer->type == SG_PLOT_3D){
        label = gtk_label_new("Z line:");
        gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
        gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,2,3);
  
        zline_combo = gtk_combo_new();
        gtk_table_attach_defaults(GTK_TABLE(table1),zline_combo,1,2,2,3);
        gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(zline_combo)->entry), FALSE);
        sg_combo_set_items(GTK_COMBO(zline_combo), line_styles);
        gtk_widget_set_usize(zline_combo, req.width, req.height);
      } 

      label = gtk_label_new("Width:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,3,4);

      adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
      xline_width_spin = gtk_spin_button_new(adj, 0.5, 1);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(xline_width_spin), TRUE);
      gtk_table_attach_defaults(GTK_TABLE(table1),xline_width_spin,1,2,3,4);

      label = gtk_label_new("Color:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,4,5);

      xline_color_combo = gtk_color_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),xline_color_combo,1,2,4,5);

      gtk_widget_set_usize(xline_width_spin, req.width, req.height);
      gtk_widget_set_usize(xline_combo, req.width, req.height);
      gtk_widget_set_usize(yline_combo, req.width, req.height);
  }

  /* Error bars */

  if(active_dataset->type != SG_DATA_FUNCTION &&
     active_dataset->type != SG_DATA_FLUX &&
     active_dataset->type != SG_DATA_VBARS &&
     active_dataset->type != SG_DATA_HBARS &&
     active_dataset->type != SG_DATA_BUBBLES &&
     active_dataset->type != SG_DATA_CONTOUR){

        frame = gtk_frame_new("Error bars");
        gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
        gtk_table_attach_defaults(GTK_TABLE(table),frame,3,4,1,2);

        if(the_layer->type == SG_PLOT_3D)
          table1 = gtk_table_new(4, 2, FALSE);
        else
          table1 = gtk_table_new(6, 2, FALSE);

        gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
        gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
        gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
        gtk_container_add (GTK_CONTAINER (frame), table1);
      
        label = gtk_label_new("Width:");
        gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
        gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);

        label = gtk_label_new("Caps:");
        gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
        gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);

        adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 1., 0.);
        err_width_spin = gtk_spin_button_new(adj, 0, 0);
        gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(err_width_spin), TRUE);
        gtk_spin_button_set_digits(GTK_SPIN_BUTTON(err_width_spin), 0);
        gtk_table_attach_defaults(GTK_TABLE(table1),err_width_spin,1,2,0,1);

        adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 20., 1., 1., 0.);
        err_caps_spin = gtk_spin_button_new(adj, 0, 0);
        gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(err_caps_spin), TRUE);
        gtk_spin_button_set_digits(GTK_SPIN_BUTTON(err_caps_spin), 0);
        gtk_table_attach_defaults(GTK_TABLE(table1),err_caps_spin,1,2,1,2);
      
        if(the_layer->type != SG_PLOT_3D){
          if(active_dataset->type != SG_DATA_VBOXES){
            xerr_check = gtk_check_item_new_with_label("show xerrbars");
            gtk_table_attach_defaults(GTK_TABLE(table1),xerr_check,0,3,2,3);
          }
          if(active_dataset->type != SG_DATA_HBOXES){
            yerr_check = gtk_check_item_new_with_label("show yerrbars");
            gtk_table_attach_defaults(GTK_TABLE(table1),yerr_check,0,3,3,4);
          }

          gtk_widget_set_usize(err_width_spin, req.width, req.height);
          gtk_widget_set_usize(err_caps_spin, req.width, req.height);

        } else {
          zerr_check = gtk_check_item_new_with_label("show zerrbars");
          gtk_table_attach_defaults(GTK_TABLE(table1),zerr_check,0,3,2,3);
        } 
  }

  /* Arrow */

  if(active_dataset->type == SG_DATA_FLUX){

      frame = gtk_frame_new("Arrows");
      gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
      gtk_table_attach_defaults(GTK_TABLE(table),frame,3,4,1,2);

      table1 = gtk_table_new(3, 2, FALSE);
      gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
      gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
      gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
      gtk_container_add (GTK_CONTAINER (frame), table1);

      label = gtk_label_new("Style:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1), label , 0, 1, 0, 1);
      arrow_combo = gtk_combo_new();
      sg_combo_set_items(GTK_COMBO(arrow_combo), symbol_styles);
      gtk_table_attach_defaults(GTK_TABLE(table1), arrow_combo, 1, 2, 0, 1);

      label = gtk_label_new("Length:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1), label, 0, 1, 1, 2);
      adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 40., 1., 1., 0.);
      arrow_length_spin = gtk_spin_button_new(adj, 0, 0);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(arrow_length_spin), TRUE);
      gtk_spin_button_set_digits(GTK_SPIN_BUTTON(arrow_length_spin), 0);
      gtk_table_attach_defaults(GTK_TABLE(table1), arrow_length_spin, 1, 2, 1, 2);

      label = gtk_label_new("Width:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1), label, 0, 1, 2, 3);
      adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 40., 1., 1., 0.);
      arrow_width_spin = gtk_spin_button_new(adj, 0, 0);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(arrow_width_spin), TRUE);
      gtk_spin_button_set_digits(GTK_SPIN_BUTTON(arrow_width_spin), 0);
      gtk_table_attach_defaults(GTK_TABLE(table1), arrow_width_spin, 1, 2, 2, 3);
      arrow_center_check = gtk_check_item_new_with_label("Center");
      gtk_table_attach_defaults(GTK_TABLE(table1), arrow_center_check, 0, 2, 3, 4);

      gtk_widget_set_usize(arrow_combo, req.width, req.height);
      gtk_widget_set_usize(arrow_width_spin, req.width, req.height);
      gtk_widget_set_usize(arrow_length_spin, req.width, req.height);

      gtk_list_select_item(GTK_LIST(GTK_COMBO(arrow_combo)->list),
                           GTK_PLOT_FLUX(active_dataset->real_data)->arrow_style);

      gtk_spin_button_set_value(GTK_SPIN_BUTTON(arrow_length_spin),
                                GTK_PLOT_FLUX(active_dataset->real_data)->arrow_length);

      gtk_spin_button_set_value(GTK_SPIN_BUTTON(arrow_width_spin),
                                GTK_PLOT_FLUX(active_dataset->real_data)->arrow_width);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(arrow_center_check),
                                   GTK_PLOT_FLUX(active_dataset->real_data)->centered);
  }


  /* Bars */

  if(active_dataset->type == SG_DATA_HBARS ||
     active_dataset->type == SG_DATA_VBARS){
      frame = gtk_frame_new("Bars");
      gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
      gtk_table_attach_defaults(GTK_TABLE(table),frame,0,1,1,2);

      table1 = gtk_table_new(7, 2, FALSE);
      gtk_container_set_border_width(GTK_CONTAINER(table1), 10);
      gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
      gtk_table_set_row_spacings(GTK_TABLE(table1), 10);
      gtk_container_add (GTK_CONTAINER (frame), table1);

      label = gtk_label_new("Width:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);

      width_entry = gtk_entry_new();
      sg_entry_set_numeric(GTK_ENTRY(width_entry), 15);
      gtk_table_attach_defaults(GTK_TABLE(table1), width_entry,1,2,0,1);
      gtk_widget_size_request(width_entry, &req);
      req.width /= 2;

      label = gtk_label_new("Style:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);

      symbol_style_combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),symbol_style_combo,1,2,1,2);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(symbol_style_combo)->entry), FALSE);

      sg_combo_set_items(GTK_COMBO(symbol_style_combo), symbol_styles);

      label = gtk_label_new("Thickness:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,2,3);

      adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
      symbol_width_spin = gtk_spin_button_new(adj, 0.5, 1);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(symbol_width_spin), TRUE);
      gtk_table_attach_defaults(GTK_TABLE(table1),symbol_width_spin,1,2,2,3);

      label = gtk_label_new("Border:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,3,4);

      border_color_combo = gtk_color_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),border_color_combo,1,2,3,4);

      label = gtk_label_new("Color:");
      gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,4,5);

      symbol_color_combo = gtk_color_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table1),symbol_color_combo,1,2,4,5);

      gtk_widget_set_usize(width_entry, req.width, req.height);
      gtk_widget_set_usize(symbol_width_spin, req.width, req.height);
      gtk_widget_set_usize(symbol_style_combo, req.width, req.height);
  }
 
  /* Labels */

  if(active_dataset->type == SG_DATA_LPOINTS ||
     active_dataset->type == SG_DATA_CONTOUR){
    frame = gtk_frame_new("Labels");
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
    gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 4, 2, 3);

    font_combo = gtk_font_combo_new();
    gtk_container_add(GTK_CONTAINER(frame), font_combo);
    gtk_container_set_border_width(GTK_CONTAINER(font_combo), 5);
    gtk_toolbar_append_space(GTK_TOOLBAR(font_combo));

    font_color_combo = gtk_color_combo_new();
    gtk_container_add(GTK_CONTAINER(font_combo), font_color_combo);

    gtk_toolbar_append_space(GTK_TOOLBAR(font_combo));
    gtk_toolbar_append_space(GTK_TOOLBAR(font_combo));
    gtk_toolbar_append_space(GTK_TOOLBAR(font_combo));

    labels_check = gtk_check_item_new_with_label("show labels");
    gtk_container_add(GTK_CONTAINER(font_combo), labels_check); 
  }

}

static void
build_surface_dialog()
{
  GtkWidget *frame;
  GtkWidget *table, *table1;
  GtkWidget *label;
  SGdataset *dataset;
  GtkAdjustment *adj;

  table = gtk_table_new(2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                           gtk_label_new("Colors & mesh"));

  if(active_dataset->type == SG_DATA_SURFACE){
    frame = gtk_frame_new("Colors");
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
    gtk_table_attach_defaults(GTK_TABLE(table),frame,0,1,0,1);

    table1 = gtk_table_new(2, 2, FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
    gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
    gtk_container_add (GTK_CONTAINER (frame), table1);

    label = gtk_label_new("Foreground:");
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);

    fg_color_combo = gtk_color_combo_new();  
    gtk_table_attach_defaults(GTK_TABLE(table1), fg_color_combo,1,2,0,1);

    label = gtk_label_new("Background:");
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);

    bg_color_combo = gtk_color_combo_new();  
    gtk_table_attach_defaults(GTK_TABLE(table1),bg_color_combo,1,2,1,2);
  }

  frame = gtk_frame_new("Lines");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  if(active_dataset->type == SG_DATA_SURFACE)
    gtk_table_attach_defaults(GTK_TABLE(table),frame,1,2,0,1);
  else
    gtk_table_attach_defaults(GTK_TABLE(table),frame,0,2,0,1);

  table1 = gtk_table_new(6, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
  gtk_container_add (GTK_CONTAINER (frame), table1);

  label = gtk_label_new("Foreground:");
  gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);

  line_color_combo = gtk_color_combo_new();  
  gtk_table_attach_defaults(GTK_TABLE(table1),line_color_combo,1,2,0,1);

  if(active_dataset->type == SG_DATA_SURFACE){
    label = gtk_label_new("Background:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);

    xline_color_combo = gtk_color_combo_new();  
    gtk_table_attach_defaults(GTK_TABLE(table1),xline_color_combo,1,2,1,2);
  }

  grid_check = gtk_check_item_new_with_label("Show grid");
  gtk_table_attach_defaults(GTK_TABLE(table1),grid_check,0,2,2,3);
  mesh_check = gtk_check_item_new_with_label("Show mesh");
  gtk_table_attach_defaults(GTK_TABLE(table1),mesh_check,0,2,3,4);
  
  if(active_dataset->type == SG_DATA_CSURFACE ||
     active_dataset->type == SG_DATA_CONTOUR){
      lines_check = gtk_check_item_new_with_label("Show lines");
      gtk_table_attach_defaults(GTK_TABLE(table1),lines_check,0,2,4,5);
      project_check = gtk_check_item_new_with_label("Show projections");
      gtk_table_attach_defaults(GTK_TABLE(table1),project_check,0,2,5,6);
  }

}

static void
build_gradient_dialog()
{
  GtkWidget *frame;
  GtkWidget *table, *table1;
  GtkWidget *label;
  GtkWidget *box;
  SGdataset *dataset;
  GtkRequisition req;
  GtkAdjustment *adj;
  GdkPixmap *pixmap;
  GdkBitmap *mask;


  /* Gradient */

  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                           gtk_label_new("Gradient"));

  frame = gtk_frame_new("Levels");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 10);
  gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 0, 2);

  table1 = gtk_table_new (7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table1), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
  gtk_container_add(GTK_CONTAINER(frame), table1);

  button_min = gtk_button_new();
  button_max = gtk_button_new();
  gtk_widget_size_request(dataset_combo, &req);
  req.width /= 2;
  gtk_widget_set_usize(button_min, WIDTH, req.height);
  gtk_widget_set_usize(button_max, WIDTH, req.height);

  pixmap = gdk_pixmap_new (iconlist1->window, WIDTH, HEIGHT, -1);
  gradient = gtk_pixmap_new(pixmap, NULL);

  gtk_table_attach(GTK_TABLE(table1), button_max, 0, 1, 0, 1, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table1), gradient, 0, 1, 1, 4, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table1), button_min, 0, 1, 4, 5, 0, 0, 0, 0);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 1, 2, 0, 1, 0, 0, 0, 0);
  
  adj = (GtkAdjustment *)gtk_adjustment_new(0., -1000000, 1000000., .1, 1., 1.);
  max_spin = gtk_spin_button_new(adj, 0.1, 3);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(max_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(max_spin), 3);
  gtk_box_pack_start(GTK_BOX(box), max_spin, 0, 0, 0);

  label = gtk_label_new("MAX");
  gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
  gtk_box_pack_start(GTK_BOX(box), label, 0, 0, 0);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 1, 2, 4, 5, 0, 0, 0, 0);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., -1000000, 1000000., .1, 1., 1.);
  min_spin = gtk_spin_button_new(adj, 0.1, 3);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(min_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(min_spin), 3);
  gtk_box_pack_start(GTK_BOX(box), min_spin, 0, 0, 0);

  label = gtk_label_new("MIN");
  gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
  gtk_box_pack_start(GTK_BOX(box), label, 0, 0, 0);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 1, 2, 2, 3, 0, 0, 0, 0);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 100., 1., 1., 1.);
  levels_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(levels_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(levels_spin), 0);
  gtk_box_pack_start(GTK_BOX(box), levels_spin, 0, 0, 0);

  label = gtk_label_new("Levels");
  gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
  gtk_box_pack_start(GTK_BOX(box), label, 0, 0, 0);

  gtk_widget_set_usize(max_spin, req.width, req.height);
  gtk_widget_set_usize(min_spin, req.width, req.height);
  gtk_widget_set_usize(levels_spin, req.width, req.height);

  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               color_xpm);
  color_pixmap = gtk_pixmap_new(pixmap, mask);
  gtk_container_add(GTK_CONTAINER(button_max), color_pixmap);

  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               color_xpm);
  color_pixmap = gtk_pixmap_new(pixmap, mask);
  gtk_container_add(GTK_CONTAINER(button_min), color_pixmap);


  frame = gtk_frame_new("Increment");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 10);
  gtk_table_attach_defaults(GTK_TABLE(table), frame, 1, 2, 0, 1);

  table1 = gtk_table_new (7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table1), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
  gtk_container_add(GTK_CONTAINER(frame), table1);

  h_check = gtk_check_item_new_with_label("Hue"); 
  s_check = gtk_check_item_new_with_label("Saturation"); 
  v_check = gtk_check_item_new_with_label("Value"); 

  gtk_table_attach_defaults(GTK_TABLE(table1), h_check, 0, 1, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table1), s_check, 0, 1, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(table1), v_check, 0, 1, 5, 6);

  gradient_check = gtk_check_item_new_with_label("show gradient");
  gtk_table_attach_defaults(GTK_TABLE(table), gradient_check, 1, 2, 1, 2);

  /* connect signals */
  gtk_signal_connect(GTK_OBJECT(button_max),"clicked",
                     GTK_SIGNAL_FUNC(open_color_selection), button_max);

  gtk_signal_connect(GTK_OBJECT(button_min),"clicked",
                     GTK_SIGNAL_FUNC(open_color_selection), button_min);

}

static void
pick_color(GtkWidget *widget, gpointer data)
{
  GtkWidget *button;
  GdkColor color;
  GdkGC *gc;
  gdouble values[4];

  button = GTK_WIDGET(data);

  gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(csd)->colorsel), values);

  color.red = 0xffff * values[0];
  color.green = 0xffff * values[1];
  color.blue = 0xffff * values[2];
  gdk_color_alloc(gdk_colormap_get_system(), &color);

  if(button == button_max) 
       color_max = color;
  else
       color_min = color;

  gc = gdk_gc_new(iconlist1->window);
  gdk_gc_set_foreground(gc, &color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(button)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(button)->child, NULL);

  gtk_widget_destroy(csd);
  gdk_gc_unref(gc);
}

static void
open_color_selection(GtkWidget *widget, gpointer data)
{
  GtkWidget *button;
  GdkColor color;
  gdouble values[4];

  button = GTK_WIDGET(data);
  if(button == button_max) 
    color = color_max;
  else
    color = color_min;


  csd = gtk_color_selection_dialog_new ("Pick a color");

  /* Set as modal */
  gtk_window_set_modal (GTK_WINDOW(csd),TRUE);

  sg_dialog_new(csd);

  gtk_signal_connect (GTK_OBJECT(csd), "destroy",
                      GTK_SIGNAL_FUNC(mw_destroy),NULL);

  gtk_signal_connect (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->ok_button),
                      "clicked", GTK_SIGNAL_FUNC(pick_color),
                      data);

  gtk_signal_connect_object (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->cancel_button),
                             "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                             GTK_OBJECT (csd));

  values[0] = color.red / 0xffff;
  values[1] = color.green / 0xffff;
  values[2] = color.blue / 0xffff;

  gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(csd)->colorsel), values);

  gtk_widget_show(csd);
  gtk_main();
}

static void
update_dialog(GtkWidget *widget, gpointer data)
{
  GList *list;
  gchar *text;
  SGdataset *dataset = NULL;

  text = gtk_entry_get_text(GTK_ENTRY(widget));
  
  if(notebook)
    gtk_container_remove(GTK_CONTAINER(main_box), notebook);

  list = the_layer->datasets;
  while(list){
    dataset = (SGdataset *)list->data;
    if(strcmp(text, dataset->real_data->name) == 0) break;
    list = list->next;
  }

  active_dataset = dataset;
  build_dialog();

  /* Legends */
  if(active_dataset->real_data->legend)
    gtk_entry_set_text(GTK_ENTRY(text_entry), active_dataset->real_data->legend);
}

GtkWidget *
sg_style_dialog_new (SGlayer *layer)
{
  GtkWidget *label;
  GtkWidget *box;
  GtkWidget *frame;
  GList *list;
  GtkWidget *item;

  the_layer = layer;

  /* Create widgets */
  main_box = gtk_vbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);

  /* Datasets combo */
  box = gtk_hbox_new (FALSE,5);
  gtk_box_pack_start(GTK_BOX(main_box), box, FALSE, FALSE, 0);

  label = gtk_label_new("Select Dataset:");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);

  gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

  dataset_combo = gtk_combo_new();
  gtk_box_pack_start(GTK_BOX(box), dataset_combo, FALSE, FALSE, 0);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dataset_combo)->entry), FALSE);

  list = layer->datasets;
  while(list){
   SGdataset *dataset; 

   dataset = (SGdataset *)list->data;
   item = gtk_list_item_new_with_label(dataset->real_data->name);
   gtk_widget_show(item);
   gtk_container_add(GTK_CONTAINER(GTK_COMBO(dataset_combo)->list), item);
   list = list->next;
  }

  active_dataset = (SGdataset *)layer->datasets->data;

  build_dialog();

  /* Legends */

  frame = gtk_frame_new("Legend");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_end(GTK_BOX(main_box), frame, FALSE, FALSE, 0);

  text_entry = gtk_entry_new();
  gtk_container_add(GTK_CONTAINER(frame), text_entry);

  if(active_dataset->real_data->legend)
    gtk_entry_set_text(GTK_ENTRY(text_entry), active_dataset->real_data->legend);

  /* Connect signals */

  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(dataset_combo)->entry),
                     "changed",
                     (GtkSignalFunc) update_dialog, NULL);

  return main_box;
}


