/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <signal.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"

/* #ifdef-check needed for Linux (/) vs. DOS (\) ? */
#define SLASH "/"

#define AUTOSAVE_PREFIX ".#autosave#"
#define RESCUE_PREFIX "#rescue#"
#define SG_WWW   "http://scigraphica.sourceforge.net"
#define SG_EMAIL "scigraphica-devel@lists.sourceforge.net"

static void bailout(int sig);
static void bailout_yes_pressed(GtkWidget *dialog);

static void rescue(int sig);
static void rescue_button_pressed(GtkWidget *dialog);

void
sg_project_rescue_init( void )
{
 /* connect signals to handlers */

 /* Program Error Signals
  * These signals are generated when a serious program error is detected by the operating
  * system or the computer itself. In general, all of these signals are indications that
  * the program is seriously broken in some way, and there's usually no way to continue
  * the computation which encountered the error.
  * The default action for all of these signals is to cause the process to terminate.
  */
 #ifdef SIGFPE
  signal( SIGFPE, rescue );
 #endif
 #ifdef SIGILL
  signal( SIGILL, rescue );
 #endif
 #ifdef SIGSEGV
  signal( SIGSEGV, rescue );
 #endif
 #ifdef SIGBUS
  signal( SIGBUS, rescue );
 #endif
 #ifdef SIGABRT
  signal( SIGABRT, rescue );
 #endif
 #ifdef SIGTOT
  signal( SIGTOT, rescue );
 #endif
 #ifdef SIGTRAP
  signal( SIGTRAP, rescue );
 #endif
 #ifdef SIGSYS
  signal( SIGSYS, rescue );
 #endif

 /* Termination Signals
  * These signals are all used to tell a process to terminate, in one way or another.
  */
 #ifdef SIGTERM
  signal( SIGTERM, bailout );
 #endif
 #ifdef SIGINT
  signal( SIGINT, bailout );
 #endif
 #ifdef SIGQUIT
  signal( SIGQUIT, bailout );
 #endif
 #ifdef SIGKILL
  signal( SIGKILL, bailout );
 #endif
 #ifdef SIGHUP
  signal( SIGHUP, bailout );
 #endif

 /* Alarm Signals
  * These signals are used to indicate the expiration of timers.
  * The default behavior for these signals is to cause program termination,
  */
 #ifdef SIGALRM
  signal( SIGALRM, SIG_DFL );
 #endif
 #ifdef SIGVTALRM
  signal( SIGVTALRM, SIG_DFL );
 #endif
 #ifdef SIGPROF
  signal( SIGPROF, SIG_DFL );
 #endif

 /* Asynchronous I/O Signals
  * These signals are used in conjunction with asynchronous I/O facilities.
  * The default action for these signals is to ignore them.
  */
 #ifdef SIGPOLL
  signal( SIGPOLL, SIG_DFL );
 #endif
 #ifdef SIGIO
  signal( SIGIO, SIG_DFL );
 #endif
 #ifdef SIGURG
  signal( SIGURG, SIG_DFL );
 #endif

 /* Job Control Signals
  * These signals are used to support job control. If your system
  * doesn't support job control, then these macros are defined but
  * the signals themselves can't be raised or handled.
  */
 #ifdef SIGCLD
  signal( SIGCLD, SIG_DFL );
 #endif
 #ifdef SIGCHLD
  signal( SIGCHLD, SIG_DFL );
 #endif
 #ifdef SIGCONT
  signal( SIGCONT, SIG_DFL );
 #endif
 #ifdef SIGSTOP
  signal( SIGSTOP, SIG_DFL );
 #endif
 #ifdef SIGTSTP
  signal( SIGTSTP, SIG_DFL );
 #endif
 #ifdef SIGTTIN
  signal( SIGTTIN, SIG_DFL );
 #endif
 #ifdef SIGTTOU
  signal( SIGTTOU, SIG_DFL );
 #endif

 /* Operation Error Signals
  * These signals are used to report various errors generated by an operation done by
  * the program. They do not necessarily indicate a programming error in the program,
  * but an error that prevents an operating system call from completing.
  * The default action for all of them is to cause the process to terminate.
  */
 #ifdef SIGPIPE
  signal( SIGPIPE, bailout );
 #endif
 #ifdef SIGLOST
  signal( SIGLOST, bailout );
 #endif
 #ifdef SIGXCPU
  signal( SIGXCPU, bailout );
 #endif
 #ifdef SIGXFSZ
  signal( SIGXFSZ, bailout );
 #endif

 /* Miscellaneous Signals These signals are used for various other purposes.
  * In general, they will not affect your program unless it explicitly uses them for something.
  */
 #ifdef SIGUSR1
  signal( SIGUSR1, SIG_DFL );
 #endif
 #ifdef SIGUSR2
  signal( SIGUSR2, SIG_DFL );
 #endif
 #ifdef SIGWINCH
  signal( SIGWINCH, SIG_DFL );
 #endif
 #ifdef SIGINFO
  signal( SIGINFO, SIG_DFL );
 #endif

 /* Other Signals
  * Stack fault on coprocessor / Power fail
  * No information available.
  */
 #ifdef SIGSTKFLT
  signal( SIGSTKFLT, SIG_DFL );
 #endif
 #ifdef SIGPWR
  signal( SIGPWR, SIG_DFL );
 #endif
}

static void
bailout_yes_pressed(GtkWidget *dialog)
{
 gtk_widget_destroy(dialog);
 sg_dialog_kill(dialog);
 gtk_exit(0);
}

static void
rescue_button_pressed(GtkWidget *dialog)
{
 gtk_widget_destroy(dialog);
 sg_dialog_kill(dialog);
 abort();
}

static void
bailout( int sig )
{/* exit gracefully, otherwise ignore signal and keep running */
 GtkWidget *dialog;
 gchar message[250];

 if (project_changed == FALSE) gtk_exit(0);

 g_snprintf(message, 250, "%s\nExit losing unsaved changes?", g_strsignal(sig));

#ifdef WITH_GNOME
 dialog=sg_accept_dialog_loop(message, 1);
 sg_dialog_new(dialog);
 gnome_dialog_button_connect(GNOME_DIALOG(dialog),0,(GtkSignalFunc) bailout_yes_pressed, NULL);
#else
 if (sg_accept_dialog(message, 1) == YES_CLICKED) gtk_exit(0);
#endif
}

static void
rescue( int sig )
{/* memory is possibly corrupted now, so rely on static storage only */
 static gint interrupts = -1; /* number of interrupts during rescue */
 static gchar message[250];
 static GtkWidget *dialog;

 if (interrupts < 0)
    {/* start the rescue */
     static gboolean rescue_OK = FALSE;
     static gchar rescue_file[250];

     interrupts = 0;

     /* prepare for the worst and keep your fingers crossed */
     g_snprintf(message, 250, "%s\n%s\n%s\n%s\n%s", g_strsignal(sig),
                                                    "\nSorry, failed to save project",
                                                    "\nPlease submit a bug report to",
                                                    SG_EMAIL, "or  " SG_WWW);

     /* save project to its original location */
      g_snprintf(rescue_file, 250, "%s" SLASH RESCUE_PREFIX "%s", last_project_path, last_project_filename);
      rescue_OK = sg_project_file_export_xml(rescue_file);
     if (!rescue_OK)
     {/* failed! then save project to the current working directory */
      g_snprintf(rescue_file, 250, RESCUE_PREFIX "%s", last_project_filename);
      rescue_OK = sg_project_file_export_xml(rescue_file);
     }
     if (!rescue_OK && getenv("HOME"))
     {/* failed again; as a last resort save to home directory */
      g_snprintf(rescue_file, 250, "%s" SLASH RESCUE_PREFIX "%s", getenv("HOME"), last_project_filename);
      rescue_OK = sg_project_file_export_xml(rescue_file);
     }

     if (rescue_OK) /* were we lucky ? */
        g_snprintf(message, 250, "%s\n%s\n%c%s%c\n%s\n%s\n%s", g_strsignal(sig),
                                                               "Project successfully saved to",
                                                               '"', rescue_file, '"',
                                                               "\nPlease submit a bug report to ",
                                                               SG_EMAIL, "or  " SG_WWW);
    }
 else /* this interrupt occurred while rescuing the project */
    if (++interrupts < 10) return; /* ignore up to 10 interrupts */

#ifdef SIGABRT
 /* reconnect abort signal to avoid looping back into rescue */
 signal( SIGABRT, SIG_DFL );
#endif

#ifdef WITH_GNOME
 dialog=sg_message_dialog_loop(message, 0);
 sg_dialog_new(dialog);
 gnome_dialog_button_connect(GNOME_DIALOG(dialog),0,(GtkSignalFunc) rescue_button_pressed, NULL);
#else
 sg_message_dialog(message, 0);
 abort();
#endif
}
