/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"

static gboolean custom_tick_labels		(GtkPlotAxis *axis, 
						 gdouble *tick_value, 
						 gchar *label,
						 gpointer data);
static void plot_changed			(GtkPlot *plot, gpointer data);

SGlayer *
sg_layer_new (gint plot_type, gdouble width, gdouble height)
{
  SGlayer *layer;
  GtkPlot *plot;

  layer = g_new0(SGlayer, 1);

  switch (plot_type){
     case SG_PLOT_2D:
           layer->real_plot = gtk_plot_new_with_size(NULL, width, height);
           break;
     case SG_PLOT_3D:
           layer->real_plot = gtk_plot3d_new_with_size(NULL, width, height);
           break;
     case SG_PLOT_POLAR:
           layer->real_plot = gtk_plot_polar_new_with_size(NULL, width, height);
           break;
  }

  layer->type = plot_type;
  layer->datasets = NULL;

  layer->left_axis_labels = NULL;
  layer->right_axis_labels = NULL;
  layer->top_axis_labels = NULL;
  layer->bottom_axis_labels = NULL;

  layer->symbol = 1;
  layer->symbol_style = 0;
  layer->line_style = 1;
  layer->connector = 1;
  gdk_color_black(gdk_colormap_get_system(), &layer->symbol_color);
  gdk_color_black(gdk_colormap_get_system(), &layer->line_color);

  plot = GTK_PLOT(layer->real_plot);
  
  plot->clip_data = TRUE;

  gtk_signal_connect(GTK_OBJECT(plot), "changed",
                     GTK_SIGNAL_FUNC(plot_changed), NULL);

  gtk_signal_connect(GTK_OBJECT(plot->left), "tick_label",
                     GTK_SIGNAL_FUNC(custom_tick_labels), layer);
  gtk_signal_connect(GTK_OBJECT(plot->right), "tick_label",
                     GTK_SIGNAL_FUNC(custom_tick_labels), layer);
  gtk_signal_connect(GTK_OBJECT(plot->top), "tick_label",
                     GTK_SIGNAL_FUNC(custom_tick_labels), layer);
  gtk_signal_connect(GTK_OBJECT(plot->bottom), "tick_label",
                     GTK_SIGNAL_FUNC(custom_tick_labels), layer);

  layer->button = gtk_toggle_button_new_with_label(" ");
  gtk_widget_set_usize(layer->button, 24, 24);

  gtk_plot_set_transparent(plot, TRUE);
  return layer;
}

static gboolean
custom_tick_labels(GtkPlotAxis *axis, gdouble *tick_value, gchar *label, gpointer data)
{
  SGlayer *layer;
  SGtick *tick;
  GtkPlot *plot;
  GList *list = NULL, *labels = NULL;

  layer = (SGlayer *)data;
  plot = GTK_PLOT(layer->real_plot);

  if(axis == plot->left) labels = layer->left_axis_labels;
  if(axis == plot->right) labels = layer->right_axis_labels;
  if(axis == plot->top) labels = layer->top_axis_labels;
  if(axis == plot->bottom) labels = layer->bottom_axis_labels;

  if(!labels) return FALSE;

  sprintf(label, "");

  list = labels;
  while(list){
   tick = (SGtick *)list->data;

   if(*tick_value == tick->value){
     g_snprintf(label, 100, tick->label);
     break;
   }

   list = list->next;
  }

  return TRUE; 
}

void
sg_layer_set_tick_labels(SGlayer *layer, 
                         gint axis_pos, 
                         gchar *sheet, 
                         gint values_col, 
                         gint labels_col)
{
  SGworksheet *worksheet;
  GList *list, *labels;
  GtkPlotAxis *axis;
  SGtick *tick;
  gchar *text,*vtext;
  gint i;
  gchar *default_label = "";

  worksheet = sg_project_get_worksheet(sheet);

  switch(axis_pos){
   case GTK_PLOT_AXIS_LEFT:
     labels = layer->left_axis_labels;
     break;
   case GTK_PLOT_AXIS_RIGHT:
     labels = layer->right_axis_labels;
     break;
   case GTK_PLOT_AXIS_TOP:
     labels = layer->top_axis_labels;
     break;
   case GTK_PLOT_AXIS_BOTTOM:
     labels = layer->bottom_axis_labels;
     break;
  }
 
  list = labels;
  while(list){ 

     labels = g_list_remove_link(labels, list);

     tick = (SGtick *)list->data;
     g_free(tick->label);
     g_free(list->data);
     g_list_free_1(list);
     list = labels;
  }

  labels = NULL;

  for(i = 0; i <= GTK_SHEET(worksheet->sheet)->maxallocrow; i++){
     text = g_strdup(sg_worksheet_cell_get_text(worksheet, i, labels_col));

     tick = g_new0(SGtick, 1);

     vtext = sg_worksheet_cell_get_text(worksheet, i, values_col);

     if (values_col>=0)
      { 
        if(vtext) 
          tick->value = atof(vtext);
        else
          tick->value = 0.0;
      }
     else 
      tick->value=(gdouble)i;
      
     if(text)
       tick->label = text;
     else
       { 
         if (vtext) 
           tick->label = default_label;
         else break; /* text==vtext==NULL && values_col<0 ==> end of the column */
       }

     labels = g_list_append(labels, tick);
  }

  switch(axis_pos){
   case GTK_PLOT_AXIS_LEFT:
     layer->left_axis_labels = labels;
     break;
   case GTK_PLOT_AXIS_RIGHT:
     layer->right_axis_labels = labels;
     break;
   case GTK_PLOT_AXIS_TOP:
     layer->top_axis_labels = labels;
     break;
   case GTK_PLOT_AXIS_BOTTOM:
     layer->bottom_axis_labels = labels;
     break;
  }
 
}

void
sg_layer_add_dataset(SGlayer *layer, SGdataset *dataset)
{
  gtk_plot_add_data(GTK_PLOT(layer->real_plot), dataset->real_data);
  gtk_widget_show(GTK_WIDGET(dataset->real_data));

  layer->datasets = g_list_append(layer->datasets, dataset);
}

void
sg_layer_add_dataset_default(SGlayer *layer, SGdataset *dataset)
{
  dataset->real_data->symbol.symbol_type = layer->symbol;
  dataset->real_data->symbol.symbol_style = layer->symbol_style;
  dataset->real_data->symbol.color = layer->symbol_color;
  dataset->real_data->symbol.border.color = layer->symbol_color;
  dataset->real_data->line.line_style = layer->line_style;
  dataset->real_data->line_connector = layer->connector;
  dataset->real_data->line.color = layer->line_color;

  gtk_plot_add_data(GTK_PLOT(layer->real_plot), dataset->real_data);
  gtk_widget_show(GTK_WIDGET(dataset->real_data));

  layer->datasets = g_list_append(layer->datasets, dataset);
}

void
sg_layer_clear(SGlayer *layer)
{
  GList *list;

  list = layer->datasets;
  while(list){
    sg_layer_remove_dataset(layer, (SGdataset *)list->data);
    list = layer->datasets;
  }

  layer->datasets = NULL;
}

void
sg_layer_remove_dataset(SGlayer *layer, SGdataset *dataset)
{
  GList *list;

  list = layer->datasets;

  while(list) {
     if(list->data == dataset){
         gtk_plot_remove_data(GTK_PLOT(layer->real_plot),
                              dataset->real_data); 

         sg_dataset_destroy(dataset);
         layer->datasets = g_list_remove_link(layer->datasets, list);
         g_list_free_1(list);
         break;
     }
     list = list->next;
  }

}

void
sg_layer_button_set_label(SGlayer *layer, gint num)
{
  gchar title[10];
  
  sprintf(title,"%i\n",num);

  gtk_misc_set_alignment(GTK_MISC(GTK_BIN(layer->button)->child), .5, .0);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(layer->button)->child), title);
}


static void 
plot_changed (GtkPlot *plot, gpointer data)
{
  sg_project_changed(TRUE);
}
