/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"

typedef struct{
  SGlayer *layer;

  GtkPlotAxis *axis;

  GtkPlotLine *major_grid;
  GtkPlotLine *minor_grid;

  GtkWidget *style_combo1; 
  GtkWidget *style_combo2; 
  GtkWidget *thick_spin1; 
  GtkWidget *thick_spin2; 
  GtkWidget *color_combo1; 
  GtkWidget *color_combo2; 
  GtkWidget *show_check1;
  GtkWidget *show_check2;
} SGgridsdialog;

static SGgridsdialog grids_dialog[3];
static GtkWidget *grid_ontop_check;

void
sg_grids_dialog_update_plot(GtkWidget *widget)
{
  GtkWidget *notebook = NULL;
  GtkPlot *plot;
  GtkPlotLine major_grid, minor_grid;
  GtkColorCombo *combo;
  GdkColor color;
  SGplot *parent;
  SGgridsdialog *dialog;
  GtkWidget *child;
  gchar *color_name;
  gint page;
  GList *list;

  list = GTK_BOX(widget)->children;
  while(list){
    GtkBoxChild *child;

    child = list->data;
    if(GTK_IS_NOTEBOOK(child->widget)){
       notebook = GTK_WIDGET(child->widget);
       break;
    }
    list = list->next;
  }

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
  dialog = &grids_dialog[page];
  plot = GTK_PLOT(dialog->layer->real_plot);

  combo = GTK_COLOR_COMBO(dialog->color_combo1);
  color_name = gtk_color_combo_get_color_at(combo, combo->row, combo->column);
  gdk_color_parse(color_name, &color);
  gdk_color_alloc(GTK_WIDGET(combo)->style->colormap, &color);

  major_grid.color = color;
  major_grid.line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->thick_spin1));

  child = GTK_LIST(GTK_COMBO(dialog->style_combo1)->list)->selection->data;
  major_grid.line_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo1)->list), child);

  combo = GTK_COLOR_COMBO(dialog->color_combo2);
  color_name = gtk_color_combo_get_color_at(combo, combo->row, combo->column);
  gdk_color_parse(color_name, &color);
  gdk_color_alloc(GTK_WIDGET(combo)->style->colormap, &color);

  minor_grid.color = color;
  minor_grid.line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->thick_spin2));

  child = GTK_LIST(GTK_COMBO(dialog->style_combo2)->list)->selection->data;
  minor_grid.line_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo2)->list), child);

  *dialog->major_grid = major_grid;
  *dialog->minor_grid = minor_grid;

  dialog->axis->show_major_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->show_check1));
  dialog->axis->show_minor_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->show_check2));

#if GTKEXTRA_MICRO_VERSION >= 16
    plot->grids_on_top = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(grid_ontop_check));
#endif

  parent = (SGplot *)dialog->layer->parent;
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(parent->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(parent->real_canvas));
}


static void
init_dialog(SGgridsdialog *dialog)
{
  GtkPlotLine *major_grid, *minor_grid;
  GdkColor line_color;
  gint line_style;
  gfloat line_width;

  major_grid = dialog->major_grid;
  minor_grid = dialog->minor_grid;

  line_style = major_grid->line_style;
  line_width = major_grid->line_width;
  line_color = major_grid->color;

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->style_combo1)->list), line_style);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->style_combo1)->entry), FALSE);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->thick_spin1), line_width);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->color_combo1), line_color);

  line_style = minor_grid->line_style;
  line_width = minor_grid->line_width;
  line_color = minor_grid->color;

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->style_combo2)->list), line_style);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->style_combo2)->entry), FALSE);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->thick_spin2), line_width);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->show_check1), dialog->axis->show_major_grid);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->show_check2), dialog->axis->show_minor_grid);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->color_combo2), line_color);

}


static GtkWidget *
build_grids_dialog (SGlayer *layer, GtkPlotAxis *axis, SGgridsdialog *dialog)
{
  GtkPlot *plot;
  GtkWidget *frame;
  GtkWidget *main_box, *main_table;
  GtkWidget *table;
  GtkRequisition req;
  GtkWidget *label;
  GtkAdjustment *adj;

  dialog->layer = layer;
  dialog->axis = axis;
  plot = GTK_PLOT(layer->real_plot);

  dialog->major_grid = &axis->major_grid; 
  dialog->minor_grid = &axis->minor_grid; 

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);

  main_table = gtk_table_new(3,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);

  /* Major Grids */

  frame = gtk_frame_new("Major grids");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 0, 1, 0, 1);

  table=gtk_table_new(4, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new("Style ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  label = gtk_label_new("Thickness(pts) ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  label = gtk_label_new("Color ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);


  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  dialog->thick_spin1 = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->thick_spin1), TRUE);
  dialog->style_combo1 = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->style_combo1), line_styles);
  dialog->color_combo1 = gtk_color_combo_new();

  dialog->show_check1 = gtk_check_item_new_with_label("Show major grids");

  gtk_widget_size_request(dialog->style_combo1, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->style_combo1, req.width, req.height);
  gtk_widget_set_usize(dialog->thick_spin1, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), dialog->style_combo1, 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->thick_spin1, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->color_combo1, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->show_check1, 0, 2, 3, 4);

  /* Minor Grids */

  frame = gtk_frame_new("Minor grids");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 1, 2, 0, 1);

  table=gtk_table_new(4, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new("Style ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  label = gtk_label_new("Thickness(pts) ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  label = gtk_label_new("Color ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  dialog->thick_spin2 = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->thick_spin2), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->thick_spin2), 0);
  dialog->style_combo2 = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->style_combo2), line_styles);
  dialog->color_combo2 = gtk_color_combo_new();

  dialog->show_check2 = gtk_check_item_new_with_label("Show minor grids");

  gtk_widget_set_usize(dialog->style_combo2, req.width, req.height);
  gtk_widget_set_usize(dialog->thick_spin2, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), dialog->style_combo2, 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->thick_spin2, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->color_combo2, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->show_check2, 0, 2, 3, 4);

  /* connect signals */

  gtk_widget_show_all(main_box);
  init_dialog(dialog);

  return (main_box);
}

GtkWidget *
sg_grids_dialog_new (SGlayer *layer)
{
  GtkWidget *notebook;
  GtkWidget *dialog;
  GtkWidget *box;
  GtkPlot *plot;

  box = gtk_vbox_new(FALSE, 5);
  notebook = gtk_notebook_new();
  gtk_box_pack_start(GTK_BOX(box), notebook, TRUE, TRUE, 0);

  plot = GTK_PLOT(layer->real_plot);

  dialog = build_grids_dialog(layer, plot->bottom, &grids_dialog[0]);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                           gtk_label_new("X axis"));
  dialog = build_grids_dialog(layer, plot->left, &grids_dialog[1]);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                           gtk_label_new("Y axis"));

  if(GTK_IS_PLOT3D(plot)){
    dialog = build_grids_dialog(layer, GTK_PLOT3D(plot)->az, &grids_dialog[2]);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                             gtk_label_new("Z axis"));
  }

#if GTKEXTRA_MICRO_VERSION >= 16
    grid_ontop_check = gtk_check_item_new_with_label("Show grids on top of data");
    gtk_box_pack_start(GTK_BOX(box), grid_ontop_check, TRUE, TRUE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(grid_ontop_check), plot->grids_on_top);
#endif
    return box;
}
