/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"

#define DATASET_NAME_LENGTH 100

static SGlayer *the_layer;

enum 
{
      SG_DATA_HAS_X    = 1 << 0,
      SG_DATA_HAS_Y    = 1 << 1,
      SG_DATA_HAS_Z    = 1 << 2,
      SG_DATA_HAS_A    = 1 << 3,
      SG_DATA_HAS_DX   = 1 << 4,
      SG_DATA_HAS_DY   = 1 << 5,
      SG_DATA_HAS_DZ   = 1 << 6,
      SG_DATA_HAS_DA   = 1 << 7,
      SG_DATA_HAS_LBLS = 1 << 8,
};

#define  SG_DATA_LPOINTS_MASK   SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y | \
                                SG_DATA_HAS_DX | \
                                SG_DATA_HAS_DY | \
                                SG_DATA_HAS_LBLS
#define  SG_DATA_VBARS_MASK     SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y
#define  SG_DATA_HBARS_MASK     SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y
#define  SG_DATA_VBOXES_MASK    SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y | \
                                SG_DATA_HAS_Z | \
                                SG_DATA_HAS_DZ
#define  SG_DATA_HBOXES_MASK    SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y | \
                                SG_DATA_HAS_Z | \
                                SG_DATA_HAS_DZ
#define  SG_DATA_BUBBLES_MASK   SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y | \
                                SG_DATA_HAS_A 
#define  SG_DATA_FLUX_MASK      SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y | \
                                SG_DATA_HAS_DX | \
                                SG_DATA_HAS_DY | \
                                SG_DATA_HAS_DA 
#define  SG_DATA_COLORS_MASK  	SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y | \
                                SG_DATA_HAS_A | \
                                SG_DATA_HAS_DA
#define  SG_DATA_CONTOUR_MASK   SG_DATA_HAS_X | \
                                SG_DATA_HAS_Y | \
                                SG_DATA_HAS_Z


enum{
      DATA_DIALOG,
      FUNCTIONS_DIALOG,
      PYTHON_DIALOG
};

static gint default_nargs[] = {0, 0, 2, 2, 2, 2, 2, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3};
static gint nargs[] = {0, 0, 2, 2, 2, 2, 2, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3};

static gchar *column_labels[][10] = {
  {"", "", "", "", "", "", "", "", ""},
  {"", "", "", "", "", "", "", "", ""},
  {"X", "Y", "Z", "", "Xerr", "Yerr", "Zerr", "", "Lbls"},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "DX", "DY", "DZ", "Amp", ""},
  {"X", "Y", "Z", "Size", "", "", "", "Amp", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "", "", "", "", ""},};

static gchar *dialog_names[] = {"Worksheet",
                                "Functions",
                                "Python variable/expression",
                                NULL};



static GtkWidget *column_entry[9];
static GtkWidget *left_button[9];
static GtkWidget *right_button[9];
static GtkWidget *entries_table = NULL;
static gint toggle_selection = 0;
static gint data_type;
static gint num_entries;
static gint default_columns_mask[] = {0,
                              0, 
                              SG_DATA_LPOINTS_MASK,
                              SG_DATA_VBARS_MASK,
                              SG_DATA_HBARS_MASK,
                              SG_DATA_VBOXES_MASK,
                              SG_DATA_HBOXES_MASK,
                              SG_DATA_BUBBLES_MASK,
                              SG_DATA_FLUX_MASK,
                              SG_DATA_COLORS_MASK,
                              SG_DATA_CONTOUR_MASK,
                              SG_DATA_CONTOUR_MASK,
                              SG_DATA_CONTOUR_MASK,
                              SG_DATA_CONTOUR_MASK };
static gint columns_mask[] = {0,
                              0, 
                              SG_DATA_LPOINTS_MASK,
                              SG_DATA_VBARS_MASK,
                              SG_DATA_HBARS_MASK,
                              SG_DATA_VBOXES_MASK,
                              SG_DATA_HBOXES_MASK,
                              SG_DATA_BUBBLES_MASK,
                              SG_DATA_FLUX_MASK,
                              SG_DATA_COLORS_MASK,
                              SG_DATA_CONTOUR_MASK,
                              SG_DATA_CONTOUR_MASK,
                              SG_DATA_CONTOUR_MASK,
                              SG_DATA_CONTOUR_MASK };
static SGdataset *dataset = NULL;
static GtkWidget *window = NULL;
static GtkWidget *main_table;
static GtkWidget *worksheet_combo, *goto_combo;
static GtkWidget *functions_list, *columns_list;
static SGplot *plot;
static GList *old_datasets = NULL;
static gint active_dialog = DATA_DIALOG;
static GtkWidget* dialog = NULL;
static GtkWidget *exp_entry = NULL;
static GtkWidget *toggle_combo;

static GtkWidget *entries[9];



static GtkWidget *sg_dataset_dialog                     (SGlayer *layer);
static GtkWidget *sg_functions_dialog                   (SGlayer *layer);
static GtkWidget *sg_dataset_exp_dialog 		(SGlayer *layer);
static GtkWidget *sg_pick_dialog                        (SGlayer *layer);

static gint goto_dialog                                 (SGlayer *layer,
                                                         gint dialog);
static void goto_pressed                                (GtkWidget *widget,
                                                         gpointer data);
static void real_select_columns				(GtkWidget *widget, 
							 gpointer data);
static void select_columns				(GtkWidget *widget, 
							 gint row, gint col,
							 gpointer data);
static void init_columns				(GtkWidget *widget);
static void select_exp                          	(GtkWidget *widget,
                                                         gint row, gint col,
                                                         gpointer data);
static void init_exp					(GtkWidget *widget);
static gint focus_next					(GtkWidget *widget, 
							 GtkWidget *next);
static void set_autosymbol				(SGdataset *dataset,
							 gint data_type);


static gint
get_column_from_name(gchar *name)
{
  SGworksheet *worksheet;
  gchar *worksheet_name = NULL;
  gint n = 0;

  worksheet_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

  worksheet = sg_project_get_worksheet(worksheet_name);
  for(n = 0; n <= GTK_SHEET(worksheet->sheet)->maxcol; n++){
    if(strcmp(name, GTK_SHEET(worksheet->sheet)->column[n].name) == 0) return n;
  }

  return -1;
}

static gboolean
mw_destroy(GtkWidget *widget)
{
  /* This is needed to get out of gtk_main */
  gtk_main_quit ();

  entries_table = NULL;

  return FALSE;
}

static gboolean
cancel_pressed(GtkWidget *widget)
{
  dataset = NULL;

  gtk_widget_destroy(window);
  return FALSE;
}

static gboolean
ok_pressed(GtkWidget *widget)
{
  SGworksheet *worksheet;
  SGdataset *function = NULL;
  GList *selection = NULL;
  gchar name[DATASET_NAME_LENGTH] = "";
  gchar *worksheet_name;
  gchar default_text[] = "";
  gint i, row = 0, narg = 0;
  gint dataset_columns[9];
  gchar column_text[9][100];
  gchar *column_name[9];

  switch(active_dialog){

   case DATA_DIALOG:
    for(i = 0; i < 9; i++){
      dataset_columns[i] = -1;
      if(columns_mask[data_type] & (1 << i)){
         column_name[i] = gtk_entry_get_text(GTK_ENTRY(column_entry[i]));
         dataset_columns[i] = get_column_from_name(column_name[i]);
         if(dataset_columns[i] != -1) narg++;
      }
    }

    if(narg < nargs[data_type]) return FALSE;

    worksheet_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

    worksheet = sg_project_get_worksheet(worksheet_name);

    if(worksheet){
      dataset = sg_dataset_new(data_type);
      gtk_widget_show(GTK_WIDGET(dataset->real_data));

      sg_dataset_set_worksheet(dataset, 
                               worksheet,
                               dataset_columns[0],
                               dataset_columns[1],
                               dataset_columns[2],
                               dataset_columns[3],
                               dataset_columns[4],
                               dataset_columns[5],
                               dataset_columns[6],
                               dataset_columns[7],
                               dataset_columns[8]);

      set_autosymbol(dataset, toggle_selection);
    }

    break;

  case FUNCTIONS_DIALOG:
    selection = GTK_CLIST(functions_list)->selection;
    if(!selection) return FALSE;

    row = GPOINTER_TO_INT(selection->data);

    function = (SGdataset *)gtk_clist_get_row_data(GTK_CLIST(functions_list), row);
    dataset = sg_dataset_new_function(function->exp);
    gtk_widget_show(GTK_WIDGET(dataset->real_data));

    sg_dataset_set_name(dataset, function->real_data->name);

    break;

  case PYTHON_DIALOG:
    narg=0;
    for(i = 0; i < 9; i++){
      dataset_columns[i] = -1;
      if(columns_mask[data_type] & (1 << i)){
#ifdef WITH_GNOME
         column_name[i] = gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entries[i]))));
#else
         column_name[i] = gtk_entry_get_text(GTK_ENTRY(entries[i]));
#endif
         narg++;
         if (strlen(column_name[i])>0)
         { 
           if (i>0) 
             g_snprintf(column_text[i], 100,",%s",column_name[i]);
           else 
             g_snprintf(column_text[i], 100,"%s",column_name[i]);
         }
         else
         { 
           column_name[i]=NULL;
           sprintf(column_text[i],"");
         }

      }
      else
      { 
          column_name[i]=NULL;
          sprintf(column_text[i],"");
      }
    }

    g_snprintf(name,100,"EXPR->(%s%s%s%s%s%s%s)",
                                       column_text[0],
                                       column_text[1],
                                       column_text[2],
                                       column_text[3],
                                       column_text[4],
                                       column_text[5],
                                       column_text[6]);

    if(strlen(column_name[0])>0 && strlen(column_name[1])>0){
      dataset = sg_dataset_new(data_type);
      gtk_widget_show(GTK_WIDGET(dataset->real_data));

      sg_dataset_set_name(dataset, name);

      sg_dataset_set_python(dataset, column_name);

      set_autosymbol(dataset, toggle_selection);
    }

    break;

  }

  gtk_widget_destroy(widget);

  return FALSE;
}

static void
set_autosymbol(SGdataset *dataset, gint data_style)
{
  if(active_dataset){
      dataset->real_data->symbol = active_dataset->real_data->symbol;
      dataset->real_data->line = active_dataset->real_data->line;
  } else {
    switch(toggle_selection){
      case 3:
        dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_IMPULSE;
        break;
      case 1:
        dataset->real_data->line.line_style = the_layer->line_style;
        the_layer->line_style++;
        if(the_layer->line_style >= NUM_LINE_STYLES)
              the_layer->line_style = 1;
        break;
      case 2:
        dataset->real_data->symbol.symbol_type = the_layer->symbol;
        dataset->real_data->symbol.symbol_style = the_layer->symbol_style;
        dataset->real_data->line.line_style = the_layer->line_style;
        the_layer->line_style++;
        if(the_layer->line_style >= NUM_LINE_STYLES)
              the_layer->line_style = 1;
        the_layer->symbol++;
        if(the_layer->symbol >= NUM_SYMBOLS){
           the_layer->symbol = 1;
           the_layer->symbol_style++;
           if(the_layer->symbol_style >= NUM_SYMBOL_STYLES)
              the_layer->symbol_style = 0;
        }
        break;
      case 0:
      default:
        dataset->real_data->symbol.symbol_type = the_layer->symbol;
        dataset->real_data->symbol.symbol_style = the_layer->symbol_style;
        dataset->real_data->line.line_style = GTK_PLOT_LINE_NONE;
        the_layer->symbol++;
        if(the_layer->symbol >= NUM_SYMBOLS){
           the_layer->symbol = 1;
           the_layer->symbol_style++;
           if(the_layer->symbol_style >= NUM_SYMBOL_STYLES)
              the_layer->symbol_style = 0;
        }
    }
  }

}


static void
update_columns(GtkEntry *entry)
{
  gint i, nrows;
  SGworksheet *worksheet = NULL;
  GList *list;
  gchar name[DATASET_NAME_LENGTH];
  gchar *text;

  nrows = GTK_CLIST(columns_list)->rows;
  for(i = 0; i <= nrows; i++){
    gtk_clist_remove(GTK_CLIST(columns_list), 0);
  }

  text = gtk_entry_get_text(entry);
  list = worksheets;
  while(list){
    worksheet = (SGworksheet *)list->data;
    if(strcmp(text, worksheet->name) == 0) break;
    list = list->next;
  }

  for(i = 0; i <= GTK_SHEET(worksheet->sheet)->maxcol; i++){
     if(!GTK_SHEET(worksheet->sheet)->column[i].name)
        sprintf(name,"%d",i);
     else
        sprintf(name,"%s",GTK_SHEET(worksheet->sheet)->column[i].name);
     text=g_strdup(name);
     gtk_clist_append(GTK_CLIST(columns_list), &text);
  };

  for(i = 0; i < num_entries; i++)
    gtk_entry_set_text(GTK_ENTRY(column_entry[i]), "");

}

SGdataset *
sg_edit_data_dialog (SGlayer *layer)
{
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *box;
  GtkWidget *ok_button, *cancel_button;
  GtkWidget *action_area;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *worksheet_combo;
  GList *list;
  SGplot *plot;
  gchar text[DATASET_NAME_LENGTH];
  gint i;

  plot = (SGplot *)layer->parent;
  the_layer = layer;

  window=gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);

  if(active_dataset)
    gtk_window_set_title (GTK_WINDOW(window),"Edit dataset");
  else
    gtk_window_set_title (GTK_WINDOW(window),"Add dataset");

  gtk_window_set_modal (GTK_WINDOW(window),TRUE);

  /* Create widgets */

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_add(GTK_CONTAINER (window), frame);

/**********************************************************/
  main_table = gtk_table_new(3, 1, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 10);
  gtk_container_add(GTK_CONTAINER(frame), main_table);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);

  gtk_table_attach_defaults(GTK_TABLE(main_table), sg_pick_dialog (layer), 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(main_table), gtk_hseparator_new(), 0, 1, 2, 3);

  main_box = gtk_vbox_new (FALSE, 5);
  gtk_table_attach_defaults(GTK_TABLE(main_table), main_box, 0, 1, 3, 4);

  dialog = NULL;
  for(i = 0; i < 14; i++){
     columns_mask[i] = default_columns_mask[i];
     nargs[i] = default_nargs[i];
  }

  if(layer->type == SG_PLOT_3D){
     columns_mask[SG_DATA_LPOINTS] |= SG_DATA_HAS_Z | SG_DATA_HAS_DZ;
     columns_mask[SG_DATA_LPOINTS] -= (SG_DATA_HAS_LBLS + 
                                      SG_DATA_HAS_DX + SG_DATA_HAS_DY);
     columns_mask[SG_DATA_BUBBLES] |= SG_DATA_HAS_Z;
     columns_mask[SG_DATA_COLORS] |= SG_DATA_HAS_Z;
     columns_mask[SG_DATA_FLUX] |= SG_DATA_HAS_Z | SG_DATA_HAS_DZ;

     nargs[SG_DATA_LPOINTS]++;
     nargs[SG_DATA_BUBBLES]++;
     nargs[SG_DATA_COLORS]++;
     nargs[SG_DATA_FLUX] += 2;
  }

  if(active_dataset){
     if(active_dataset->exp != NULL)
                    goto_dialog(layer, FUNCTIONS_DIALOG);
     else if(active_dataset->p_exp[0] != NULL)
                    goto_dialog(layer, PYTHON_DIALOG);
     else
                    goto_dialog(layer, DATA_DIALOG);
  } else 
                    goto_dialog(layer, active_dialog);

  gtk_table_attach_defaults(GTK_TABLE(main_table), sg_pick_dialog (layer), 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(main_table), gtk_hseparator_new(), 0, 1, 2, 3);

/**********************************************************/

  /* Action Area */
  action_area = gtk_hbutton_box_new ();
  gtk_container_set_border_width(GTK_CONTAINER(action_area), 5);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_end (GTK_BOX (main_box), action_area, FALSE, FALSE, 0);
  gtk_widget_show (action_area);

  ok_button = gtk_button_new_with_label ("Add");
  GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (ok_button);
  gtk_widget_show (ok_button);

  cancel_button = gtk_button_new_with_label ("Close");
  GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (cancel_button);

  /* connect signals */

  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
                             GTK_SIGNAL_FUNC (cancel_pressed),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
                             GTK_SIGNAL_FUNC (ok_pressed),
                             GTK_OBJECT (window));

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (mw_destroy),NULL);

  /* Show widgets */

  gtk_widget_show_all (window);

  /* wait until dialog get destroyed */
  gtk_main();

  return dataset;
}


static GtkWidget *
sg_dataset_dialog (SGlayer *layer)
{
  GtkWidget *frame, *vbox;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *box, *sw;
  GtkWidget *action_area;
  GtkWidget *goto_button;
  GList *list;
  GtkWidget *item;
  SGworksheet *worksheet;

  active_dialog = DATA_DIALOG;

  dialog = vbox = gtk_vbox_new(FALSE, 5);

  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_start (GTK_BOX (vbox), action_area, FALSE, FALSE, 0);

/*********************************************************************/

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new("Select Worksheet:");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
  worksheet_combo = gtk_combo_new();
  gtk_box_pack_end (GTK_BOX (box), worksheet_combo, FALSE, FALSE, 0);
  gtk_widget_show(box);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry), FALSE);
  list = worksheets;
  while(list){
     worksheet = (SGworksheet *)list->data;
     item = gtk_list_item_new_with_label(worksheet->name);
     gtk_widget_show(item);
     gtk_container_add(GTK_CONTAINER(GTK_COMBO(worksheet_combo)->list), item);
     list = list->next;
  }

  if(active_dataset)
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry), 
                       active_dataset->worksheet->name);

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new("Dataset Style:");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);

  toggle_combo = sg_toggle_data_new(layer->type);
  gtk_box_pack_start (GTK_BOX (box), toggle_combo, FALSE, FALSE, 0);
  sg_toggle_data_init(toggle_combo, 0);

/*********************************************************************/

  frame = gtk_frame_new("Columns");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  box = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(box), 5);
  gtk_container_add (GTK_CONTAINER (frame), box);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(sw, 180, 160);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (box), sw, FALSE, FALSE, 0);

  columns_list = gtk_clist_new(1);
  gtk_container_add(GTK_CONTAINER(sw), columns_list);

  if(active_dataset)
     init_columns(box);
  else
     select_columns(toggle_combo, 0, 0, box);

/********************************************************************/
  gtk_signal_connect(GTK_OBJECT(toggle_combo), "changed",
		     GTK_SIGNAL_FUNC(select_columns), box);

  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(worksheet_combo)->entry), 
                     "changed", GTK_SIGNAL_FUNC(real_select_columns), box);

  return vbox;
}

static gint
set_column(GtkWidget *widget, gpointer data)
{
  GList *selection;
  gchar *text;
  gint row;

  selection = GTK_CLIST(columns_list)->selection;
  if(!selection) return TRUE;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(strlen(text) > 0) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  gtk_clist_get_text(GTK_CLIST(columns_list), row, 0, &text);

  gtk_entry_set_text(GTK_ENTRY(data), text);
  gtk_clist_remove(GTK_CLIST(columns_list), row);
  return TRUE;
}

static gint
restore_column(GtkWidget *widget, gpointer data)
{
  gchar *text;
  gint col;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(strlen(text) == 0) return TRUE;

  col = get_column_from_name(text);
  gtk_clist_insert(GTK_CLIST(columns_list), col, &text);

  gtk_entry_set_text(GTK_ENTRY(data), "");
  return TRUE;
}

static void
init_columns(GtkWidget *widget)
{
  GtkWidget *column_label;
  GtkWidget *box;
  GtkWidget *button_box;
  gint i, nrows = 0;
  gchar label[9][100];
  gint col_index[9];

  col_index[0] = active_dataset->x;
  col_index[1] = active_dataset->y;
  col_index[2] = active_dataset->z;
  col_index[3] = active_dataset->a;
  col_index[4] = active_dataset->dx;
  col_index[5] = active_dataset->dy;
  col_index[6] = active_dataset->dz;
  col_index[7] = active_dataset->da;
  col_index[8] = active_dataset->l;
  data_type = active_dataset->type;

  box = widget;

  update_columns(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(box), entries_table);

  toggle_selection = 0;
 
  switch(the_layer->type){ 
    case SG_PLOT_3D:
      switch(data_type){
        case SG_DATA_LPOINTS:
          toggle_selection = 0;
          break;
        case SG_DATA_BUBBLES:
          toggle_selection = 1;
          break;
        case SG_DATA_COLORS:
          toggle_selection = 2;
          break;
        case SG_DATA_FLUX:
          toggle_selection = 3;
          break;
        case SG_DATA_SURFACE:
          toggle_selection = 4;
          break;
        case SG_DATA_CSURFACE:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_PLOT_POLAR:
      switch(data_type){
        case SG_DATA_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_DATA_BUBBLES:
          toggle_selection = 4;
          break;
        case SG_DATA_COLORS:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_PLOT_2D:
      switch(data_type){
        case SG_DATA_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_DATA_VBARS:
          toggle_selection = 4;
          break;
        case SG_DATA_HBARS:
          toggle_selection = 5;
          break;
        case SG_DATA_VBOXES:
          toggle_selection = 6;
          break;
        case SG_DATA_HBOXES:
          toggle_selection = 7;
          break;
        case SG_DATA_BUBBLES:
          toggle_selection = 8;
          break;
        case SG_DATA_COLORS:
          toggle_selection = 9;
          break;
        case SG_DATA_FLUX:
          toggle_selection = 10;
          break;
        case SG_DATA_CONTOUR:
          toggle_selection = 11;
          break;
      }
    break;
  }

  sg_toggle_data_init(toggle_combo, toggle_selection);

  entries_table = gtk_table_new(num_entries, 3, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_type][i]);
    if(columns_mask[data_type] & (1 << i)){
      button_box = gtk_hbox_new(TRUE, 0);
      gtk_table_attach_defaults(GTK_TABLE (entries_table), button_box, 
                                0, 1, nrows, nrows + 1);

      left_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(left_button[i]), 
                        gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(left_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), left_button[i], FALSE, FALSE, 0);
 
      right_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(right_button[i]), 
                        gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(right_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), right_button[i], FALSE, FALSE, 0);

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label, 
                                1, 2, nrows, nrows + 1);    
      column_entry[i] = gtk_entry_new();
      gtk_entry_set_editable(GTK_ENTRY(column_entry[i]), FALSE);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_entry[i], 
                                2, 3, nrows, nrows + 1);    

      if(col_index[i] != -1){
        gint row;
        gchar *text = GTK_SHEET(active_dataset->worksheet->sheet)->column[col_index[i]].name;
        gchar *ctext;

        gtk_entry_set_text(GTK_ENTRY(column_entry[i]), text);

        for(row = 0; row < GTK_CLIST(columns_list)->rows; row++){
          gtk_clist_get_text(GTK_CLIST(columns_list), row, 0, &ctext);
          if(strcmp(text, ctext) == 0)
            gtk_clist_remove(GTK_CLIST(columns_list), row);
        }
      }


      gtk_signal_connect(GTK_OBJECT(left_button[i]), "clicked",
                         (GtkSignalFunc) restore_column, column_entry[i]);
      gtk_signal_connect(GTK_OBJECT(right_button[i]), "clicked",
                         (GtkSignalFunc) set_column, column_entry[i]);

      nrows++;
    }
  }

  gtk_box_pack_start(GTK_BOX(box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}


static void
select_columns(GtkWidget *widget, gint row, gint col, gpointer data)
{
  GtkWidget *box;

  box = GTK_WIDGET(data);

  toggle_selection = sg_toggle_data_get_selection(widget);
  
  switch(toggle_selection){
    case 0:
      if(the_layer->type == SG_PLOT_3D){
         data_type = SG_DATA_LPOINTS;
         break;
      }
    case 1:
      if(the_layer->type == SG_PLOT_3D){
         data_type = SG_DATA_BUBBLES;
         break;
      }
    case 2:
      if(the_layer->type == SG_PLOT_3D){
         data_type = SG_DATA_COLORS;
         break;
      }
    case 3:
      if(the_layer->type == SG_PLOT_3D){
         data_type = SG_DATA_FLUX;
         break;
      }
      data_type = SG_DATA_LPOINTS;
      break;
    case 4:
      if(the_layer->type == SG_PLOT_3D){
         data_type = SG_DATA_SURFACE;
         break;
      }
      if(the_layer->type == SG_PLOT_POLAR){
         data_type = SG_DATA_BUBBLES;
         break;
      }
      data_type = SG_DATA_VBARS;
      break;
    case 5:
      if(the_layer->type == SG_PLOT_3D){
         data_type = SG_DATA_CSURFACE;
         break;
      }
      if(the_layer->type == SG_PLOT_POLAR){
         data_type = SG_DATA_COLORS;
         break;
      }
      data_type = SG_DATA_HBARS;
      break;
    case 6:
      data_type = SG_DATA_VBOXES;
      break;
    case 7:
      data_type = SG_DATA_HBOXES;
      break;
    case 8:
      data_type = SG_DATA_BUBBLES;
      break;
    case 9:
      data_type = SG_DATA_COLORS;
      break;
    case 10:
      data_type = SG_DATA_FLUX;
      break;
    case 11:
      data_type = SG_DATA_CONTOUR;
      break;
  }    

  real_select_columns(widget, data);
}

static void
real_select_columns(GtkWidget *widget, gpointer data)
{
  GtkWidget *box;
  GtkWidget *column_label;
  GtkWidget *button_box;
  gint i, nrows = 0;
  gchar label[9][100];

  box = GTK_WIDGET(data);

  update_columns(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(box), entries_table);

  entries_table = gtk_table_new(num_entries, 3, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_type][i]);
    if(columns_mask[data_type] & (1 << i)){
      button_box = gtk_hbox_new(TRUE, 0);
      gtk_table_attach_defaults(GTK_TABLE (entries_table), button_box, 
                                0, 1, nrows, nrows + 1);

      left_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(left_button[i]), 
                        gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(left_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), left_button[i], FALSE, FALSE, 0);
 
      right_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(right_button[i]), 
                        gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(right_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), right_button[i], FALSE, FALSE, 0);

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label, 
                                1, 2, nrows, nrows + 1);    
      column_entry[i] = gtk_entry_new();
      gtk_entry_set_editable(GTK_ENTRY(column_entry[i]), FALSE);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_entry[i], 
                                2, 3, nrows, nrows + 1);    

      gtk_signal_connect(GTK_OBJECT(left_button[i]), "clicked",
                         (GtkSignalFunc) restore_column, column_entry[i]);
      gtk_signal_connect(GTK_OBJECT(right_button[i]), "clicked",
                         (GtkSignalFunc) set_column, column_entry[i]);

      nrows++;
    }
  }

  gtk_box_pack_start(GTK_BOX(box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}


static GtkWidget *
sg_functions_dialog (SGlayer *layer)
{
  GtkWidget *frame, *vbox, *sw, *goto_button, *action_area;
  GList *list;
  SGdataset *dataset;
  gchar *text[1];
  gchar *titles[] = {"  Functions   "};

  active_dialog = FUNCTIONS_DIALOG;

  dialog = vbox = gtk_vbox_new(FALSE, 5);

  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_start (GTK_BOX (vbox), action_area, FALSE, FALSE, 0);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_widget_set_usize(sw, 120, 160);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(frame), sw);
  functions_list = gtk_clist_new_with_titles(1, titles);
  gtk_container_add(GTK_CONTAINER(sw), functions_list);

  list = functions;
  while(list){
    dataset = (SGdataset *)list->data;
    if(dataset->real_data->is_function){
        gint i;

        text[0] = dataset->real_data->name;
        i = GTK_CLIST(functions_list)->rows;
        gtk_clist_append(GTK_CLIST(functions_list), text);
        gtk_clist_set_row_data(GTK_CLIST(functions_list), i, (SGdataset *)list->data);
    }
    list = list->next;
  }

  return vbox;
}

static gint
focus_next(GtkWidget *widget, GtkWidget *next)
{
    gtk_widget_grab_focus(next);
    return TRUE;
}

static gint
set_exp(GtkWidget *widget, gpointer data)
{
  GList *selection;
  gchar *text;
  gint row;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(strlen(text) > 0) return TRUE;

  text = gtk_entry_get_text(GTK_ENTRY(exp_entry));
  gtk_signal_emit_by_name(GTK_OBJECT(exp_entry),"activate",NULL);
  if(!strlen(text)) return TRUE;
  gtk_entry_set_text(GTK_ENTRY(data), text);
  gtk_editable_delete_text(GTK_EDITABLE(exp_entry),0,-1);
  return TRUE;
}

static gint
restore_exp(GtkWidget *widget, gpointer data)
{
  GList *selection;
  gchar *text;
  gint row;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(!strlen(text)) return TRUE;

  gtk_editable_delete_text(GTK_EDITABLE(data),0,-1);
  return TRUE;
}

static GtkWidget *
sg_dataset_exp_dialog (SGlayer *layer)
{
  GtkWidget *frame, *vbox;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *box, *sw;
  GtkWidget *action_area;
  GtkWidget *goto_button;
  GList *list;
  GtkWidget *item;
  SGworksheet *worksheet;

  active_dialog = PYTHON_DIALOG;

  dialog = vbox = gtk_vbox_new(FALSE, 5);

  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_start (GTK_BOX (vbox), action_area, FALSE, FALSE, 0);

/*********************************************************************/

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new("Dataset Style:");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);

  toggle_combo = sg_toggle_data_new(layer->type);
  gtk_box_pack_start (GTK_BOX (box), toggle_combo, FALSE, FALSE, 0);
  sg_toggle_data_init(toggle_combo, 0);

/*********************************************************************/

  frame = gtk_frame_new("Expressions");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  box = gtk_hbox_new(TRUE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(box), 5);
  gtk_container_add (GTK_CONTAINER (frame), box);

  if(active_dataset)
    init_exp(box);
  else
    select_exp(toggle_combo, 0, 0, box);

  gtk_signal_connect(GTK_OBJECT(toggle_combo), "changed",
                     GTK_SIGNAL_FUNC(select_exp), box);

/********************************************************************/


  return vbox;
}

static void 
init_exp(GtkWidget *widget)
{
  GtkWidget *column_label;
  GtkWidget *box;
  GtkWidget *button_box;
  gint i, nrows = 0, j, k;
  gchar label[9][100], ident[80];

  box = widget;

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(box), entries_table);

  toggle_selection = 0;

  switch(the_layer->type){ 
    case SG_PLOT_3D:
      switch(data_type){
        case SG_DATA_LPOINTS:
          toggle_selection = 0;
          break;
        case SG_DATA_BUBBLES:
          toggle_selection = 1;
          break;
        case SG_DATA_COLORS:
          toggle_selection = 2;
          break;
        case SG_DATA_FLUX:
          toggle_selection = 3;
          break;
        case SG_DATA_SURFACE:
          toggle_selection = 4;
          break;
        case SG_DATA_CSURFACE:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_PLOT_POLAR:
      switch(data_type){
        case SG_DATA_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_DATA_BUBBLES:
          toggle_selection = 4;
          break;
        case SG_DATA_COLORS:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_PLOT_2D:
      switch(data_type){
        case SG_DATA_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_DATA_VBARS:
          toggle_selection = 4;
          break;
        case SG_DATA_HBARS:
          toggle_selection = 5;
          break;
        case SG_DATA_VBOXES:
          toggle_selection = 6;
          break;
        case SG_DATA_HBOXES:
          toggle_selection = 7;
          break;
        case SG_DATA_BUBBLES:
          toggle_selection = 8;
          break;
        case SG_DATA_COLORS:
          toggle_selection = 9;
          break;
        case SG_DATA_FLUX:
          toggle_selection = 10;
          break;
        case SG_DATA_CONTOUR:
          toggle_selection = 11;
          break;
      }
    break;
  }

  sg_toggle_data_init(toggle_combo, toggle_selection);

  entries_table = gtk_table_new(num_entries, 2, FALSE);
  gtk_table_set_row_spacings(GTK_TABLE(entries_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_type][i]);
    if(columns_mask[data_type] & (1 << i)){

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label,
                                0, 1, nrows, nrows + 1);
#ifdef WITH_GNOME
      g_snprintf(ident,80,"sg_dataset_exp_%s",label[i]);
      entries[i]=gnome_entry_new(ident);
#else
      entries[i] = gtk_entry_new();
#endif

      gtk_table_attach_defaults(GTK_TABLE (entries_table), entries[i], 
                                1, 2, nrows, 1+nrows);

      if(active_dataset){
        if(active_dataset->p_exp[i])
#ifdef WITH_GNOME
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), active_dataset->p_exp[i]); 
#else
          gtk_entry_set_text(GTK_ENTRY(entries[i]), active_dataset->p_exp[i]); 
#endif
      } else {
#ifdef WITH_GNOME
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), ""); 
#else
          gtk_entry_set_text(GTK_ENTRY(entries[i]), ""); 
#endif
      }

      nrows++;
    }
    else entries[i]=NULL;
  }

  for (j=0;j<nrows-1;j++)
  {   
    if (!entries[j]) continue;
    for (k=j+1;k<nrows;k++)
       if (entries[k]) break;
    if (k>=nrows) break;

#ifdef WITH_GNOME
      gtk_signal_connect(GTK_OBJECT(gnome_entry_gtk_entry(GNOME_ENTRY(entries[j]))),
                         "activate",(GtkSignalFunc) focus_next,
                         (gpointer)gnome_entry_gtk_entry(GNOME_ENTRY(entries[k])));
#else
      gtk_signal_connect(GTK_OBJECT(entries[j]), "activate",
                         (GtkSignalFunc) focus_next, (gpointer)entries[k]);
#endif
  }

  gtk_box_pack_start(GTK_BOX(box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}


static void 
select_exp(GtkWidget *widget, gint row, gint col, gpointer data)
{
  GtkWidget *column_label;
  GtkWidget *box;
  GtkWidget *button_box;
  gint i, nrows = 0, j, k;
  gchar label[9][100], ident[80];

  box = GTK_WIDGET(data);

  toggle_selection = sg_toggle_data_get_selection(widget);

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(box), entries_table);

  switch(toggle_selection){
    case 0:
    case 1:
    case 2:
    case 3:
      data_type = SG_DATA_LPOINTS;
      break;
    case 4:
      data_type = SG_DATA_VBARS;
      break;
    case 5:
      data_type = SG_DATA_HBARS;
      break;
    case 6:
      data_type = SG_DATA_VBOXES;
      break;
    case 7:
      data_type = SG_DATA_HBOXES;
      break;
    case 8:
      data_type = SG_DATA_BUBBLES;
      break;
    case 9:
      data_type = SG_DATA_COLORS;
      break;
    case 10:
      data_type = SG_DATA_FLUX;
      break;
    case 11:
      data_type = SG_DATA_CONTOUR;
      break;
  }   

  entries_table = gtk_table_new(num_entries, 2, FALSE);
  gtk_table_set_row_spacings(GTK_TABLE(entries_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_type][i]);
    if(columns_mask[data_type] & (1 << i)){

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label,
                                0, 1, nrows, nrows + 1);
#ifdef WITH_GNOME
      g_snprintf(ident,80,"sg_dataset_exp_%s",label[i]);
      entries[i] = gnome_entry_new(ident);
#else
      entries[i] = gtk_entry_new();
#endif
      gtk_table_attach_defaults(GTK_TABLE (entries_table), entries[i], 
                                1, 2, nrows, 1+nrows);

      if(active_dataset){
#ifdef WITH_GNOME
         gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), active_dataset->p_exp[i]); 
#else
         gtk_entry_set_text(GTK_ENTRY(entries[i]), active_dataset->p_exp[i]); 
#endif
      } else {
#ifdef WITH_GNOME
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), ""); 
#else
          gtk_entry_set_text(GTK_ENTRY(entries[i]), ""); 
#endif
      }

      nrows++;
    }
    else entries[i]=NULL;
  }

  for (j=0;j<nrows-1;j++)
  {   
    if (!entries[j]) continue;
    for (k=j+1;k<nrows;k++)
       if (entries[k]) break;
    if (k>=nrows) break;

#ifdef WITH_GNOME
      gtk_signal_connect(GTK_OBJECT(gnome_entry_gtk_entry(GNOME_ENTRY(entries[j]))),
                         "activate",(GtkSignalFunc) focus_next,
                         (gpointer)gnome_entry_gtk_entry(GNOME_ENTRY(entries[k])));
#else
      gtk_signal_connect(GTK_OBJECT(entries[j]), "activate",
                         (GtkSignalFunc) focus_next, (gpointer)entries[k]);
#endif
  }

  gtk_box_pack_start(GTK_BOX(box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}



static gint
goto_dialog(SGlayer *layer, gint new_dialog)
{
  if(GTK_IS_WIDGET(dialog)) gtk_container_remove(GTK_CONTAINER(main_table), dialog);

  switch(new_dialog){
    case DATA_DIALOG:
      entries_table = NULL;
      gtk_table_attach_defaults(GTK_TABLE(main_table), sg_dataset_dialog(layer), 0, 1, 1, 2);
      break;
    case FUNCTIONS_DIALOG:
      entries_table = NULL;
      gtk_table_attach_defaults(GTK_TABLE(main_table), sg_functions_dialog(layer), 0, 1, 1, 2);
      break;
    case PYTHON_DIALOG:
      entries_table = NULL;
      gtk_table_attach_defaults(GTK_TABLE(main_table), sg_dataset_exp_dialog(layer), 0, 1, 1, 2);
      break;
  }

  gtk_widget_show_all(dialog);
  return TRUE;
}

static GtkWidget *
sg_pick_dialog (SGlayer *layer)
{
  GtkWidget *hbox,*vbox;

  hbox = gtk_hbox_new(FALSE, 5);
  vbox = gtk_vbox_new(FALSE, 5);

  gtk_box_pack_start (GTK_BOX (vbox), hbox,
                      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new(),
                      TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX (hbox), gtk_label_new("Get data from"),
                      TRUE, TRUE, 0);

  goto_combo = gtk_combo_new();
  gtk_box_pack_start (GTK_BOX (hbox), goto_combo, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(goto_combo)->entry), FALSE);
  sg_combo_set_items(GTK_COMBO(goto_combo), dialog_names);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(goto_combo)->list),
                       active_dialog);

  if(active_dataset){
    gtk_widget_set_sensitive(goto_combo, FALSE);
  } else {
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(goto_combo)->entry),
                       "changed",
                       (GtkSignalFunc) goto_pressed, (gpointer)layer);
  }

  gtk_widget_show_all(hbox);
  return(vbox);
}

static void
goto_pressed(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gint i;
  gchar *text;

  text=gtk_entry_get_text(GTK_ENTRY(widget));
  for (i=0;i<3;i++)
      if (!strcmp(text,dialog_names[i])) break;

  if (i>=3) i=0;

  if (i==active_dialog) return;
  active_dialog=i;
  goto_dialog((SGlayer *)data, active_dialog);
}
