/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"
#include "../pixmaps/pixmaps.h"

static gchar **data_pixmaps[13];

static SGdataset *dataset = NULL;
static GtkWidget *worksheet_combo;
static GtkWidget *datasets_list;
static SGplot *plot;
static GList *old_datasets = NULL;
static GList *new_datasets = NULL;
static GtkWidget *dialog = NULL;
static SGlayer *the_layer;

static gint
show_dataset(GtkWidget *widget, gpointer data)
{
  SGdataset *aux_dataset;
  GdkColormap *colormap;
  GdkPixmap *eye_pixmap;
  GdkBitmap *eye_mask;
  GList *selection;
  gint row;

  colormap = gdk_colormap_get_system();

  selection = GTK_CLIST(datasets_list)->selection;
  if(!selection) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  aux_dataset = (SGdataset *)gtk_clist_get_row_data(GTK_CLIST(datasets_list), row);

  if(aux_dataset){
    if(GTK_WIDGET_VISIBLE(aux_dataset->real_data)){
       gtk_widget_hide(GTK_WIDGET(aux_dataset->real_data));
       gtk_clist_set_text(GTK_CLIST(datasets_list), row, 0, NULL);
    }else{
       gtk_widget_show(GTK_WIDGET(aux_dataset->real_data));
       eye_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, 
                                                    &eye_mask, NULL, eye_xpm);
       gtk_clist_set_pixmap(GTK_CLIST(datasets_list), row, 0, eye_pixmap, eye_mask);
       gdk_pixmap_unref(eye_pixmap);
       gdk_bitmap_unref(eye_mask);
    }

    gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
    gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
  }

  return TRUE;
}


static gint
free_auxiliary_list(GList **aux_list)
{
  GList *list;
  GList *old_list = *aux_list;
 
  while(old_list){
     SGdataset *aux_data;
     gboolean found = FALSE;
   
     aux_data = (SGdataset *)old_list->data;

     list = the_layer->datasets;
     while(list){
        dataset = (SGdataset *)list->data;

        if(dataset == aux_data) found = TRUE;
        list = list->next;
     }

     if(!found) sg_dataset_destroy(aux_data);

     *aux_list = g_list_remove_link(new_datasets, old_list);
     g_list_free_1(old_list);
     old_list = *aux_list;
  }

  *aux_list = NULL;

  return TRUE;
}

void 
sg_data_dialog_update_plot(GtkWidget *widget)
{
  GList *list = NULL;
  gint i;

  old_datasets = the_layer->datasets;

  the_layer->datasets = NULL;

  /* REMOVE OLD DATASETS */

  list = GTK_PLOT(the_layer->real_plot)->data_sets;
  while(list){
    gtk_plot_remove_data(GTK_PLOT(the_layer->real_plot), 
                         GTK_PLOT_DATA(list->data));
    list = GTK_PLOT(the_layer->real_plot)->data_sets;
  }

  GTK_PLOT(the_layer->real_plot)->data_sets = NULL;


  /* ADD NEW DATASETS */

  for(i = 0; i < GTK_CLIST(datasets_list)->rows; i++){
    SGdataset *row_dataset;
    gboolean visible;

    row_dataset = gtk_clist_get_row_data(GTK_CLIST(datasets_list), i);
    visible = GTK_WIDGET_VISIBLE(GTK_WIDGET(row_dataset->real_data));

    sg_layer_add_dataset(the_layer, row_dataset);
    if(!visible) gtk_widget_hide(GTK_WIDGET(row_dataset->real_data));
  }


  /* FREE AUXILIARY LISTS */

  free_auxiliary_list(&new_datasets);
  free_auxiliary_list(&old_datasets);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}


static gint
add_dataset(GtkWidget *widget, gpointer data)
{
  SGdataset *row_dataset = NULL;
  GdkColormap *colormap;
  GdkPixmap *pixmap, *eye_pixmap;
  GdkBitmap *mask, *eye_mask;
  gchar *text[1];
  gint i = 0;
  gint nrows;

  colormap = gdk_colormap_get_system();

  active_dataset = NULL;

  dataset = sg_edit_data_dialog(the_layer);

  if(!dataset) return FALSE;

  nrows = GTK_CLIST(datasets_list)->rows;

  for(i = 0; i < nrows; i++){
    row_dataset = (SGdataset *)gtk_clist_get_row_data(GTK_CLIST(datasets_list), i);

    if(strcmp(dataset->real_data->name, row_dataset->real_data->name) == 0 &&
       dataset->type == row_dataset->type){
        sg_dataset_destroy(dataset);
        return TRUE;
    }
  }

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                                 data_pixmaps[dataset->type]);
  eye_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &eye_mask,
                                                     NULL, eye_xpm);

  text[0] = NULL;
  gtk_clist_append(GTK_CLIST(datasets_list), text);
  text[0] = dataset->real_data->name;
  gtk_clist_set_pixtext(GTK_CLIST(datasets_list), nrows, 1, text[0], 5, pixmap, mask);
  gtk_clist_set_pixmap(GTK_CLIST(datasets_list), nrows, 0, eye_pixmap, eye_mask);
  gtk_clist_set_row_data(GTK_CLIST(datasets_list), i, dataset);
  new_datasets = g_list_append(new_datasets, dataset);

  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
  gdk_pixmap_unref(eye_pixmap);
  gdk_bitmap_unref(eye_mask);

  return TRUE;
}

static gint
remove_dataset(GtkWidget *widget, gpointer data)
{
  GList *selection;
  gint row;

  selection = GTK_CLIST(datasets_list)->selection;
  if(!selection) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  gtk_clist_remove(GTK_CLIST(datasets_list), row);

  return TRUE;
}

static gint
destroy_dialog(GtkWidget *widget, gpointer data)
{
  free_auxiliary_list(&new_datasets);

  return TRUE;
}

static gint
refresh_dataset(GtkWidget *widget, gpointer data)
{
  SGdataset *aux_dataset;
  GList *selection;
  gint row;

  selection = GTK_CLIST(datasets_list)->selection;
  if(!selection) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  aux_dataset = (SGdataset *)gtk_clist_get_row_data(GTK_CLIST(datasets_list), row);

  if(aux_dataset)
     sg_dataset_refresh(aux_dataset);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));

  return TRUE; 
}

static gint
edit_dataset(GtkWidget *widget, gpointer data)
{
  SGdataset *row_dataset = NULL;
  GdkColormap *colormap;
  GdkPixmap *pixmap, *eye_pixmap;
  GdkBitmap *mask, *eye_mask;
  gchar *text[1];
  gint i = 0;
  gint nrows;
  GList *selection;
  gint row;

  colormap = gdk_colormap_get_system();

  selection = GTK_CLIST(datasets_list)->selection;
  if(!selection) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  row_dataset = (SGdataset *)gtk_clist_get_row_data(GTK_CLIST(datasets_list), row);

  active_dataset = row_dataset; 

  if(!active_dataset) return;

  dataset = sg_edit_data_dialog(the_layer);

  if(!dataset) return FALSE;

  gtk_clist_remove(GTK_CLIST(datasets_list), row);

  nrows = GTK_CLIST(datasets_list)->rows;

  for(i = 0; i < nrows; i++){
    row_dataset = (SGdataset *)gtk_clist_get_row_data(GTK_CLIST(datasets_list), i);

    if(strcmp(dataset->real_data->name, row_dataset->real_data->name) == 0 &&
       dataset->type == row_dataset->type){
        sg_dataset_destroy(dataset);
        return TRUE;
    }
  }

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                                 data_pixmaps[dataset->type]);
  eye_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &eye_mask,
                                                     NULL, eye_xpm);

  text[0] = NULL;
  gtk_clist_append(GTK_CLIST(datasets_list), text);
  text[0] = dataset->real_data->name;
  gtk_clist_set_pixtext(GTK_CLIST(datasets_list), nrows, 1, text[0], 5, pixmap, mask);
  gtk_clist_set_pixmap(GTK_CLIST(datasets_list), nrows, 0, eye_pixmap, eye_mask);
  gtk_clist_set_row_data(GTK_CLIST(datasets_list), i, dataset);
  new_datasets = g_list_append(new_datasets, dataset);

  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);

  return TRUE;
}


GtkWidget *
sg_data_dialog_new (SGlayer *layer)
{
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *sw;
  GtkWidget *action_area;
  GtkWidget *new_button, *remove_button, *refresh_button, *edit_button, *show_button;
  GdkColormap *colormap;
  GList *list;
  gchar *text[1];
  gchar *titles[] = {"","Datasets", ""};
  GdkPixmap *eye_pixmap;
  GdkBitmap *eye_mask;

  colormap = gdk_colormap_get_system();

  data_pixmaps[0] = g_python_xpm;
  data_pixmaps[1] = g_function_xpm;
  data_pixmaps[2] = g_lpoints_xpm;
  data_pixmaps[3] = g_bars_xpm;
  data_pixmaps[4] = g_hbars_xpm;
  data_pixmaps[5] = g_boxes_xpm;
  data_pixmaps[6] = g_hbox_xpm;
  data_pixmaps[7] = g_bubbles_xpm;
  data_pixmaps[8] = g_vectors_xpm;
  data_pixmaps[9] = g_colors_xpm;
  data_pixmaps[10] = g_density_xpm;
  data_pixmaps[11] = g_contour_xpm;
  data_pixmaps[12] = g_csurface_xpm;
  data_pixmaps[13] = g_surface_xpm;

  if(layer->type == SG_PLOT_3D){
    data_pixmaps[2] = g_scatter3d_xpm;
    data_pixmaps[7] = g_bubbles3d_xpm;
    data_pixmaps[8] = g_flux3d_xpm;
    data_pixmaps[9] = g_colors3d_xpm;
  }

  plot = (SGplot *)layer->parent;
  the_layer = layer;

  active_dataset = NULL;

  /* Create widgets */
  main_box = gtk_vbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 10);

/********************************************************************/
  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_start (GTK_BOX (main_box), action_area, FALSE, FALSE, 0);

  new_button = gtk_button_new_with_label ("Add");
  gtk_box_pack_start (GTK_BOX (action_area), new_button, TRUE, TRUE, 0);
  gtk_widget_show (new_button);

  remove_button = gtk_button_new_with_label ("Remove");
  gtk_box_pack_start (GTK_BOX (action_area), remove_button, TRUE, TRUE, 0);
  gtk_widget_show (remove_button);

  refresh_button = gtk_button_new_with_label ("Refresh");
  gtk_box_pack_start (GTK_BOX (action_area), refresh_button, TRUE, TRUE, 0);
  gtk_widget_show (refresh_button);

  edit_button = gtk_button_new_with_label ("Edit");
  gtk_box_pack_start (GTK_BOX (action_area), edit_button, TRUE, TRUE, 0);
  gtk_widget_show (edit_button);

  show_button = gtk_button_new_with_label ("Show/Hide");
  gtk_box_pack_start (GTK_BOX (action_area), show_button, TRUE, TRUE, 0);
  gtk_widget_show (show_button);

/********************************************************************/
  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_widget_set_usize(sw, 320, 160);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(main_box), sw, TRUE, TRUE, 0);
  datasets_list = gtk_clist_new_with_titles(2, titles);
  gtk_clist_set_row_height(GTK_CLIST(datasets_list), 20);
  gtk_clist_set_column_min_width(GTK_CLIST(datasets_list), 0, 20);
  gtk_container_add(GTK_CONTAINER(sw), datasets_list);

  /* connect signals */
  gtk_signal_connect(GTK_OBJECT(new_button), "clicked",
                     (GtkSignalFunc) add_dataset, NULL);

  gtk_signal_connect(GTK_OBJECT(remove_button), "clicked",
                     (GtkSignalFunc) remove_dataset, NULL);

  gtk_signal_connect(GTK_OBJECT(refresh_button), "clicked",
                     (GtkSignalFunc) refresh_dataset, NULL);

  gtk_signal_connect(GTK_OBJECT(edit_button), "clicked",
                     (GtkSignalFunc) edit_dataset, NULL);

  gtk_signal_connect(GTK_OBJECT(show_button), "clicked",
                     (GtkSignalFunc) show_dataset, NULL);

  gtk_signal_connect(GTK_OBJECT(main_box), "destroy",
                     (GtkSignalFunc) destroy_dialog, NULL);

  /* Show widgets */
  eye_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &eye_mask,
                                                     NULL, eye_xpm);
  list = layer->datasets;
  while(list){
    gint nrows;
    GdkPixmap *pixmap;
    GdkBitmap *mask;

    dataset = (SGdataset *)list->data;

    pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                                   data_pixmaps[dataset->type]);


    nrows = GTK_CLIST(datasets_list)->rows;

    text[0] = NULL; 
    gtk_clist_append(GTK_CLIST(datasets_list), text);
    text[0] = GTK_PLOT_DATA(dataset->real_data)->name;
    gtk_clist_set_pixtext(GTK_CLIST(datasets_list), nrows, 1, text[0], 5, pixmap, mask);
    gtk_clist_set_row_data(GTK_CLIST(datasets_list), nrows, dataset);
    gtk_clist_set_row_data(GTK_CLIST(datasets_list), nrows, dataset);

    if(GTK_WIDGET_VISIBLE(dataset->real_data))
      gtk_clist_set_pixmap(GTK_CLIST(datasets_list), nrows, 0, eye_pixmap, eye_mask);

    gdk_pixmap_unref(pixmap);
    gdk_bitmap_unref(mask);

    list = list->next;
  }

  gdk_pixmap_unref(eye_pixmap);
  gdk_bitmap_unref(eye_mask);
 
  return main_box;
}


