/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include "sg_worksheet_file_xml.h"
#include "sg.h"

static void write_header_xml			(SGworksheetfile *file); 
static void write_footer_xml			(SGworksheetfile *file); 
static void write_column_title_xml		(SGworksheetfile *file, 
                                                 gint col); 
static void write_cell_xml			(SGworksheetfile *file,  
						 gint row, gint col);

gboolean
sg_worksheet_file_export_xml (SGworksheet *worksheet, 
                              gchar *filename,
                              GtkSheetRange *range,
			      FILE *opened)
{  
  SGworksheetfile *file;
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet->sheet);
  file = sg_worksheet_file_new(worksheet, filename);

  if(!opened){
    file->file = sg_fopen(filename, "w");
    if (!file->file){
         g_warning("ERROR: Cannot write to file: %s", filename);
         sg_worksheet_file_destroy(file);
         return FALSE;
    }
    fprintf(file->file,"<?xml version=\"1.0\"?>\n");
  } else {
    file->file = opened;
  }

  file->write_header = write_header_xml; 
  file->write_footer = write_footer_xml; 
  file->write_col_title = write_column_title_xml; 
  file->write_cell = write_cell_xml; 

  file->titlecolor = g_strdup("\"BBBBBB\"");
  file->cellcolor = g_strdup("\"FFFFFF\"");

  sg_worksheet_file_export(file, range);

  if(!opened){
    sg_fclose(file->file);
  }

  sg_worksheet_file_destroy(file);

  return TRUE;
}

static void
write_header_xml(SGworksheetfile *file)
{
  SGworksheet *worksheet = file->worksheet;

  if(worksheet->mode == SG_MODE_MATRIX)
    fprintf(file->file,"<sgw:Matrix xmlns:sgw=\"http://scigraphica.sourceforge.net\">\n");
  else
    fprintf(file->file,"<sgw:Worksheet xmlns:sgw=\"http://scigraphica.sourceforge.net\">\n");

  fprintf(file->file, "  <sgw:Summary>\n"); 
  fprintf(file->file, "    <sgw:Item>\n"); 
  fprintf(file->file, "      <sgw:name>application</sgw:name>\n"); 
  fprintf(file->file, "      <sgw:val-string>scigraphica</sgw:val-string>\n"); 
  fprintf(file->file, "    </sgw:Item>\n"); 
  fprintf(file->file, "    <sgw:Item>\n"); 
  fprintf(file->file, "      <sgw:name>author</sgw:name>\n"); 
  fprintf(file->file, "      <sgw:val-string>%s</sgw:val-string>\n", g_get_real_name()); 
  fprintf(file->file, "    </sgw:Item>\n"); 
  fprintf(file->file, "  </sgw:Summary>\n"); 
  fprintf(file->file, "  <sgw:Geometry Width=\"%d\" Height=\"%d\"/>\n",
          worksheet->width, worksheet->height);


  if(worksheet->mode == SG_MODE_MATRIX){
    gchar *xml_text = NULL;

    if(file->worksheet->matrix.exp)
      xml_text = xml_get_string(file->worksheet->matrix.exp);
    else
      xml_text = g_strdup("");

    fprintf(file->file, "  <sgw:Range Xmin=\"%f\" Xmax=\"%f\" Ymin=\"%f\" Ymax=\"%f\"/>\n", worksheet->xmin, worksheet->xmax, worksheet->ymin, worksheet->ymax); 
    fprintf(file->file, "  <sgw:MatrixFormat Exp=\"%s\" Format=\"%d\" Internal=\"%d\" Precision=\"%d\"/>\n", 
          xml_text,
          file->worksheet->matrix.format,
          file->worksheet->matrix.internal,
          file->worksheet->matrix.precision);

    g_free(xml_text);
  }
        
  fprintf(file->file, "  <sgw:Name>%s</sgw:Name>\n", worksheet->name); 
  fprintf(file->file, "  <sgw:MaxCol>%d</sgw:MaxCol>\n", 
          GTK_SHEET(worksheet->sheet)->maxcol); 
  fprintf(file->file, "  <sgw:MaxRow>%d</sgw:MaxRow>\n", 
          GTK_SHEET(worksheet->sheet)->maxrow); 
  fprintf(file->file, "  <sgw:Begin>%d</sgw:Begin>\n", 
          worksheet->begin); 
  fprintf(file->file, "  <sgw:End>%d</sgw:End>\n", 
          worksheet->end); 
}

static void
write_footer_xml(SGworksheetfile *file)
{
  if(file->worksheet->mode == SG_MODE_MATRIX)
    fprintf(file->file,"</sgw:Matrix>\n");
  else
    fprintf(file->file,"</sgw:Worksheet>\n");
}

static void
write_column_title_xml(SGworksheetfile *file, gint col) 
{
  GtkSheetColumn *column;
  gchar *xml_text = NULL;
 
  column = &GTK_SHEET(file->worksheet->sheet)->column[col];

  fprintf(file->file, "  <sgw:Column No=\"%d\" Width=\"%d\" Title=\"%s\" Type=\"%d\" Format=\"%d\" Internal=\"%d\" Precision=\"%d\">\n", 
          col, column->width, column->name, 
          file->worksheet->column[col].type,
          file->worksheet->column[col].format,
          file->worksheet->column[col].internal,
          file->worksheet->column[col].precision);

  if(file->worksheet->column[col].exp){
    xml_text = xml_get_string(file->worksheet->column[col].exp);
    fprintf(file->file, "    <sgw:ColFormula>%s</sgw:ColFormula>\n", xml_text);
    g_free(xml_text);
  }
        
  fprintf(file->file, "  </sgw:Column>\n"); 
}

static void
write_cell_xml(SGworksheetfile *file, gint row, gint col)
{
  gchar *text, *formula;
  gchar *xml_text = NULL;

  formula = sg_worksheet_cell_get_formula(file->worksheet, row, col);

  text = sg_worksheet_cell_get_text(file->worksheet, row, col);


  if((!text || strlen(text) == 0) && (!formula || strlen(formula) == 0))
      return;

  fprintf(file->file, "  <sgw:Cell Row=\"%d\" Col=\"%d\">\n", row, col);

  if (text && strlen(text) >0)
  {
      xml_text = xml_get_string(text);
      fprintf(file->file, "    <sgw:Content>%s</sgw:Content>\n", xml_text);
  }

  if (formula && strlen(formula) >0)
  {
      xml_text = xml_get_string(formula);
      fprintf(file->file, "    <sgw:Formula>%s</sgw:Formula>\n", xml_text);
  }

  fprintf(file->file, "  </sgw:Cell>\n");

  g_free(xml_text);
}

