/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <signal.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_project_rescue.h"
#include "sg.h"
#include "sg_dialogs.h"
#include "python/python_main.h"

#undef RESCUE_DEBUG

#define SG_WWW   "http://scigraphica.sourceforge.net"
#define SG_EMAIL "scigraphica-devel@lists.sourceforge.net"

/* connect a signal action to a handler and set flags & mask 
 *  SIGNAL = signal number [int]
 *  FUNCTION = new signal handler [ void (*FUNCTION)(int) ]
 *  FLAGS = new signal flags [int] (don't touch if FLAGS < 0)
 *  MASK = new signal mask [sigset_t *] (don't touch if MASK == NULL)
 *
 * The default settings of flags/mask may differ among the
 * various unix flavours (e.g Linux vs. BSD).
 *
 * See sigaction(2) for info on possible signal flags to set.
 * See sigsetops(3) for manipulating the mask.
 */
#define CONNECT_SIGACTION(SIGNAL, FUNCTION, FLAGS, MASK) \
               {struct sigaction action; \
	        sigaction((SIGNAL),NULL,&action); \
                action.sa_handler = (FUNCTION); \
                if ((FLAGS) >= 0) action.sa_flags = (FLAGS); \
                if ((MASK) != NULL) \
                   {sigset_t *pmask = (MASK); \
                    action.sa_mask = *pmask; \
                   } \
                sigaction((SIGNAL),&action,NULL); \
               }

static void bailout(int signum);
static gint bailout_idle(gpointer data);
static void rescue(int signum, int arg);

#ifdef RESCUE_DEBUG
static void sigaction_info(void);
#endif

void
sg_project_rescue_init( void )
{
 /* connect signals to handlers, see signal(3) and
  * http://www.gnu.org/manual/glibc-2.2.3/html_chapter/libc_24.html
  */
 int rescue_flags = 0;
 int default_flags = 0;
 sigset_t default_mask;

#ifdef RESCUE_DEBUG
 printf("Before sg_project_rescue_init:\n");
 sigaction_info();
#endif

#ifdef SA_RESETHAND
 rescue_flags += SA_RESETHAND;
#endif
#ifdef SA_NODEFER
 rescue_flags += SA_NODEFER;
#endif
#ifdef SA_RESTART
 default_flags += SA_RESTART;
#endif

 sigemptyset(&default_mask);

#define CONNECT_SIGNAL_TO_RESCUE(SIGNAL)  CONNECT_SIGACTION(SIGNAL, (void(*)(int))rescue, rescue_flags, &default_mask)
#define CONNECT_SIGNAL_TO_BAILOUT(SIGNAL) CONNECT_SIGACTION(SIGNAL, bailout, default_flags, &default_mask)
#define CONNECT_SIGNAL_TO_DFL(SIGNAL) CONNECT_SIGACTION(SIGNAL, SIG_DFL, default_flags, &default_mask)
#define CONNECT_SIGNAL_TO_IGN(SIGNAL) CONNECT_SIGACTION(SIGNAL, SIG_IGN, default_flags, &default_mask)

 /* Program Error Signals
  * These signals are generated when a serious program error is detected
  * by the operating system or the computer itself. In general, all of
  * these signals are indications that the program is seriously broken
  * in some way, and there's usually no way to continue the computation
  * which encountered the error.
  * The default action for all of these signals is to cause the process
  * to terminate.
  */
 #ifdef SIGFPE
  CONNECT_SIGNAL_TO_RESCUE( SIGFPE )
 #endif
 #ifdef SIGILL
  CONNECT_SIGNAL_TO_RESCUE( SIGILL )
 #endif
 #ifdef SIGSEGV
  CONNECT_SIGNAL_TO_RESCUE( SIGSEGV )
 #endif
 #ifdef SIGBUS
  CONNECT_SIGNAL_TO_RESCUE( SIGBUS )
 #endif
 #ifdef SIGABRT
  CONNECT_SIGNAL_TO_RESCUE( SIGABRT )
 #endif
 #ifdef SIGIOT
  CONNECT_SIGNAL_TO_RESCUE( SIGIOT )
 #endif
 #ifdef SIGTRAP
  CONNECT_SIGNAL_TO_RESCUE( SIGTRAP )
 #endif
 #ifdef SIGEMT
  CONNECT_SIGNAL_TO_RESCUE( SIGEMT )
 #endif
 #ifdef SIGSYS
  CONNECT_SIGNAL_TO_RESCUE( SIGSYS )
 #endif

 /* Termination Signals
  * These signals are all used to tell a process to terminate, in one way
  * or another.
  * The (obvious) default action for all of these signals is to cause the
  * process to terminate.
  */
 #ifdef SIGTERM
  CONNECT_SIGNAL_TO_BAILOUT( SIGTERM )
 #endif
 #ifdef SIGINT
  CONNECT_SIGNAL_TO_BAILOUT( SIGINT )
 #endif
 #ifdef SIGQUIT
  CONNECT_SIGNAL_TO_BAILOUT( SIGQUIT )
 #endif
 #ifdef SIGKILL
  /* SIGKILL can not be caught or ignored */
 #endif
 #ifdef SIGHUP
  CONNECT_SIGNAL_TO_BAILOUT( SIGHUP )
 #endif

 /* Alarm Signals
  * These signals are used to indicate the expiration of timers.
  * The default behavior for these signals is to cause program termination.
  */
 #ifdef SIGALRM
  CONNECT_SIGNAL_TO_DFL( SIGALRM )
 #endif
 #ifdef SIGVTALRM
  CONNECT_SIGNAL_TO_DFL( SIGVTALRM )
 #endif
 #ifdef SIGPROF
  CONNECT_SIGNAL_TO_DFL( SIGPROF )
 #endif

 /* Asynchronous I/O Signals
  * These signals are used in conjunction with asynchronous I/O facilities.
  * The default action for these signals is to ignore them.
  */
 #ifdef SIGIO
  CONNECT_SIGNAL_TO_DFL( SIGIO )
 #endif
 #ifdef SIGURG
  CONNECT_SIGNAL_TO_DFL( SIGURG )
 #endif
 #ifdef SIGPOLL
  CONNECT_SIGNAL_TO_DFL( SIGPOLL )
 #endif

 /* Job Control Signals
  * These signals are used to support job control. If your system
  * doesn't support job control, then these macros are defined but
  * the signals themselves can't be raised or handled.
  */
 #ifdef SIGCHLD
  CONNECT_SIGNAL_TO_DFL( SIGCHLD )
 #endif
 #ifdef SIGCLD
  CONNECT_SIGNAL_TO_DFL( SIGCLD )
 #endif
 #ifdef SIGCONT
  CONNECT_SIGNAL_TO_DFL( SIGCONT )
 #endif
 #ifdef SIGSTOP
  /* SIGSTOP can not be caught or ignored */
 #endif
 #ifdef SIGTSTP
  CONNECT_SIGNAL_TO_DFL( SIGTSTP )
 #endif
 #ifdef SIGTTIN
  CONNECT_SIGNAL_TO_DFL( SIGTTIN )
 #endif
 #ifdef SIGTTOU
  CONNECT_SIGNAL_TO_DFL( SIGTTOU )
 #endif

 /* Operation Error Signals
  * These signals are used to report various errors generated by an
  * operation done by the program. They do not necessarily indicate a
  * programming error in the program, but an error that prevents an
  * operating system call from completing.
  * The default action for all of them is to cause the process to terminate.
  */
 #ifdef SIGPIPE
  CONNECT_SIGNAL_TO_BAILOUT( SIGPIPE )
 #endif
 #ifdef SIGLOST
  CONNECT_SIGNAL_TO_BAILOUT( SIGLOST )
 #endif
 #ifdef SIGXCPU
  CONNECT_SIGNAL_TO_BAILOUT( SIGXCPU )
 #endif
 #ifdef SIGXFSZ
  CONNECT_SIGNAL_TO_BAILOUT( SIGXFSZ )
 #endif

 /* Miscellaneous Signals These signals are used for various other purposes.
  * In general, they will not affect your program unless it explicitly uses
  * them for something.
  */
 #ifdef SIGUSR1
  CONNECT_SIGNAL_TO_DFL( SIGUSR1 )
 #endif
 #ifdef SIGUSR2
  CONNECT_SIGNAL_TO_DFL( SIGUSR2 )
 #endif
 #ifdef SIGWINCH
  CONNECT_SIGNAL_TO_DFL( SIGWINCH )
 #endif
 #ifdef SIGINFO
  CONNECT_SIGNAL_TO_DFL( SIGINFO )
 #endif

 /* Other Signals
  * Stack fault on coprocessor / Power fail
  * No information available.
  */
 #ifdef SIGSTKFLT
  CONNECT_SIGNAL_TO_DFL( SIGSTKFLT )
 #endif
 #ifdef SIGPWR
  CONNECT_SIGNAL_TO_DFL( SIGPWR )
 #endif

#ifdef RESCUE_DEBUG
 printf("After sg_project_rescue_init:\n");
 sigaction_info();
#endif

#undef CONNECT_SIGNAL_TO_RESCUE
#undef CONNECT_SIGNAL_TO_BAILOUT
#undef CONNECT_SIGNAL_TO_DFL
#undef CONNECT_SIGNAL_TO_IGN
}


/* bailout
 * Do the real bailout as an idle process, when gtk is not busy.
 * Bailout catches signals that are not fatal, so ask whether
 * to lose the unsaved changes or to keep the application running
 */

static gint bailout_idle_id = 0;

static void
bailout( int signum )
{
 static int static_signum;

#ifdef RESCUE_DEBUG
 printf("Inside bailout (%s):\n", strsignal(signum));
 sigaction_info();
#endif
  
 /* ignore consecutive bailout signals */
 if (bailout_idle_id) return;

 if (project_changed == FALSE) bailout_idle(NULL);

 static_signum = signum;
 bailout_idle_id = gtk_idle_add(bailout_idle, &static_signum);
}

static gint
bailout_idle( gpointer data )
{
 if (data)
    {char message[80];

     gtk_idle_remove(bailout_idle_id);

     snprintf(message, 80, "%s\nExit losing unsaved changes?", strsignal( *((int*)data)) );
     if (sg_accept_dialog(message, 1) != YES_CLICKED) return (bailout_idle_id = 0);
    } 

 /* remove autosave file */ 
 sg_project_autosave_set(0);
 
 /* kill python terminal */ 
 child_died(0);

 gtk_exit(0);
}


/* rescue
 * Try to save the current project and exit.
 * Memory is possibly corrupted, so use static storage only.
 * Can't use the gtk_idle_add for rescue, because that will try to
 * resume the broken code, before calling the idle function.
 * However, not using gtk_idle_add may cause a stream of GLib warnings:
 *    g_main_iterate(): main loop already active in another thread
 */

static void
rescue(int signum, int arg)
{
 static char message[250] = "";
 static char arg_msg[80] = "";
 static int signal_count = 0;

 /* do what SA_RESETHAND and SA_NODEFER flags supposed to do here
  * why are these signal flags ignored? Due to Python, Imlib, Gtk? */
 {static struct sigaction action;
  sigaction(signum, NULL, &action);
  #ifdef SA_RESETHAND
  if (action.sa_flags & SA_RESETHAND)
     action.sa_handler = SIG_DFL;
  else
     action.sa_handler = (void(*)(int))rescue;
  sigaction(signum, &action, NULL);
  #endif
  #ifdef SA_NODEFER
  sigemptyset(&action.sa_mask);
  sigaddset(&action.sa_mask, signum);
  if (action.sa_flags & SA_NODEFER)
     sigprocmask(SIG_UNBLOCK, &action.sa_mask, NULL);
  else
     sigprocmask(SIG_BLOCK, &action.sa_mask, NULL);
  #endif
 }

#ifdef RESCUE_DEBUG
     printf("Inside rescue (%s):\n", strsignal(signum));
     sigaction_info();
#endif

/* Some OS (e.g. BSD, Sun) provide the handler with an extra
 * argument that distinguishes various causes of the exception.*/

#define SET_ARG_MSG(ARG, MSG) if (arg == ARG && *arg_msg == '\0') sprintf(arg_msg, " : %s", MSG);

#ifdef SIGFPE
 if (signum == SIGFPE) {
#ifdef FPE_INTOVF
/* impossible in a C program unless you enable overflow trapping in a hardware-specific fashion */
    SET_ARG_MSG(FPE_INTOVF, "Integer overflow")
#endif
#ifdef FPE_INTDIV   
    SET_ARG_MSG(FPE_INTDIV, "Integer division by zero")
#endif
#ifdef FPE_FLTDIV
    SET_ARG_MSG(FPE_FLTDIV, "Floating point division by zero")
#endif
#ifdef FPE_FLTOVF
    SET_ARG_MSG(FPE_FLTOVF, "Floating point overflow")
#endif
#ifdef FPE_FLTUND
/* trapping on floating underflow is not normally enabled */
    SET_ARG_MSG(FPE_FLTUND, "Floating point underflow")
#endif
#ifdef FPE_FLTRES
    SET_ARG_MSG(FPE_FLTRES, "Floating point inexact result")
#endif
#ifdef FPE_FLTINV
    SET_ARG_MSG(FPE_FLTINV, "Invalid floating point operation")
#endif
#ifdef FPE_FLTSUB
/* something that C programs never check for */
    SET_ARG_MSG(FPE_FLTSUB, "Subscript out of range")
#endif
 }
#endif /* SIGFPE */

#ifdef SIGILL
 if (signum == SIGILL) {
#ifdef ILL_ILLOPC
    SET_ARG_MSG(ILL_ILLOPC, "Illegal opcode") 
#endif
#ifdef ILL_ILLOPN
    SET_ARG_MSG(ILL_ILLOPN, "Illegal operand")
#endif
#ifdef ILL_ILLADR
    SET_ARG_MSG(ILL_ILLADR, "Illegal addressing mode")
#endif
#ifdef ILL_ILLTRP
    SET_ARG_MSG(ILL_ILLTRP, "Illegal trap")
#endif
#ifdef ILL_PRVOPC
    SET_ARG_MSG(ILL_PRVOPC, "Privileged opcode")
#endif
#ifdef ILL_PRVREG
    SET_ARG_MSG(ILL_PRVREG, "Privileged register")
#endif
#ifdef ILL_COPROC
    SET_ARG_MSG(ILL_COPROC, "Co-processor")
#endif
#ifdef ILL_BADSTK
    SET_ARG_MSG(ILL_BADSTK, "Bad stack")
#endif
 }
#endif /* SIGILL */

#ifdef SIGSEGV
 if (signum == SIGSEGV) {
#ifdef SEGV_MAPERR
    SET_ARG_MSG(SEGV_MAPERR, "Address not mapped to object")
#endif
#ifdef SEGV_ACCERR
    SET_ARG_MSG(SEGV_ACCERR, "Invalid permissions")
#endif
   }
#endif /* SIGSEGV */ 

#ifdef SIGBUS
 if (signum == SIGBUS) {
#ifdef BUS_PAGE_FAULT
    SET_ARG_MSG(BUS_PAGE_FAULT, "Page fault protection base")
#endif
#ifdef BUS_SEGNP_FAULT
    SET_ARG_MSG(BUS_SEGNP_FAULT, "Segment not present")
#endif
#ifdef BUS_STK_FAULT
    SET_ARG_MSG(BUS_STK_FAULT, "Stack segment")
#endif
#ifdef BUS_SEGM_FAULT
    SET_ARG_MSG(BUS_SEGM_FAULT, "Segment protection base")
#endif
#ifdef BUS_ADRALN
    SET_ARG_MSG(BUS_ADRALN, "Invalid address alignment")
#endif
#ifdef BUS_ADRERR
    SET_ARG_MSG(BUS_ADRERR, "Non-existent physical address")
#endif
#ifdef BUS_OBJERR
    SET_ARG_MSG(BUS_OBJERR, "Object specific hardware error")  
#endif
 }
#endif /* SIGBUS */

#ifdef SIGTRAP
 if (signum == SIGTRAP) {
#ifdef TRAP_BRKPT
    SET_ARG_MSG(TRAP_BRKPT, "Breakpoint trap")
#endif
#ifdef TRAP_TRACE
    SET_ARG_MSG(TRAP_TRACE, "Trace trap")
#endif
#ifdef TRAP_RWATCH
    SET_ARG_MSG(TRAP_RWATCH, "Read access watchpoint trap")
#endif
#ifdef TRAP_WWATCH
    SET_ARG_MSG(TRAP_WWATCH, "Write access watchpoint trap")
#endif 
#ifdef TRAP_XWATCH
    SET_ARG_MSG(TRAP_XWATCH, "Execute access watchpoint trap")
#endif 
 }
#endif /* SIGTRAP */

#ifdef SIGEMT
 if (signum == SIGEMT) {
#ifdef EMT_TAGOVF
    SET_ARG_MSG(EMT_TAGOVF, "Tag overflow")
#endif
 }
#endif /* SIGEMT */

#undef SET_ARG_MSG

 if ( signal_count++ == 0 )
 {static char rescue_file[250];
  static int rescue_OK = FALSE;

  /* prepare for the worst and keep fingers crossed */
  snprintf(message, 250, "%s%s\n\n%s\n%s\n%s\n%s", strsignal(signum), arg_msg,
                                                   "\nSorry, failed to rescue the project",
                                                   "\nPlease submit a bug report to",
                                                   SG_EMAIL, "or  " SG_WWW);

  /* save project to its original location */
   snprintf(rescue_file, 250, "%s/" RESCUE_PREFIX "%s", last_project_path, last_project_filename);
   rescue_OK = sg_project_file_export_xml(rescue_file);
  if (rescue_OK == FALSE)
  {/* failed! try saving project to the current working directory */
   snprintf(rescue_file, 250, RESCUE_PREFIX "%s", last_project_filename);
   rescue_OK = sg_project_file_export_xml(rescue_file);
  }
  if (rescue_OK == FALSE && g_get_home_dir())
  {/* failed again; as a last resort save to home directory */
   snprintf(rescue_file, 250, "%s/" RESCUE_PREFIX "%s", g_get_home_dir(), last_project_filename);
   rescue_OK = sg_project_file_export_xml(rescue_file);
  }

  if (rescue_OK) /* were we lucky? */
     snprintf(message, 250, "%s%s\n\n%s\n%c%s%c\n%s\n%s\n%s", strsignal(signum), arg_msg,
                                                              "Project successfully saved to",
                                                              '"', rescue_file, '"',
                                                              "\nPlease submit a bug report to ",
                                                              SG_EMAIL, "or  " SG_WWW);
 }
 else if ( signal_count < 10 )
    return; /* ignore up to 10 consecutive rescue signals */

 sg_message_dialog(message, 0);

 /* kill python terminal */
 child_died(0);

 /* be sure to continue with the default action */
 signal(signum, SIG_DFL);
 raise(signum);
}

/* -------  Facilities for signal-debugging only ------- */
#ifdef RESCUE_DEBUG

static const struct {
      int value;
      char name[10];
      } flag[]={
              #ifdef SA_ONSTACK
                {SA_ONSTACK,   "  ONSTACK"},
              #endif
              #ifdef SA_RESTART
                {SA_RESTART,   "  RESTART"},
              #endif
              #ifdef SA_NODEFER
                {SA_NODEFER,   "  NODEFER"},
              #endif
              #ifdef SA_RESETHAND
                {SA_RESETHAND, "RESETHAND"},
              #endif
              #ifdef SA_NOCLDSTOP
                {SA_NOCLDSTOP, "NOCLDSTOP"},
              #endif
              #ifdef SA_NOCLDWAIT
                {SA_NOCLDWAIT, "NOCLDWAIT"},
              #endif
	      #ifdef SA_WAITSIG
	        {SA_WAITSIG,   "  WAITSIG"},
	      #endif
              #ifdef SA_SIGINFO
                {SA_SIGINFO,   "  SIGINFO"},
              #endif
	       };

static const int number_of_flags = sizeof(flag)/sizeof(flag[0]);

static char*
mask_string(sigset_t mask, char *result)
{
 int i;
 for (result[0]= '\0', i = 1; i < NSIG; i++)
     {if ((i-1)%5==0 && i!=1) strcat(result, ".");
      strcat(result, (sigismember(&mask, i) ? "1" : "0") );
     }
 return result;
}
    
static char*
flags_string(int flags, char *result)
{
 int i;
 for (result[0] = '\0', i = 0; i < number_of_flags; i++)
     strcat(result, (flags & flag[i].value ? " 1" : " 0") );
 return result;
}

static char*
handler_string(struct sigaction action, char *result)
{
 if (action.sa_flags & SA_SIGINFO)
    {if (action.sa_sigaction == NULL) sprintf(result, "NULL     ");
     else sprintf(result, "%p", action.sa_sigaction);
    }
 else
    {if (action.sa_handler == SIG_DFL)      sprintf(result, "SIG_DFL  ");
     else if (action.sa_handler == SIG_IGN) sprintf(result, "SIG_IGN  ");
     else if (action.sa_handler == (void (*)(int))bailout) sprintf(result, "bailout  ");
     else if (action.sa_handler == (void (*)(int))rescue)  sprintf(result, "rescue   ");
     else if (action.sa_handler == NULL) sprintf(result, "NULL     ");
     else sprintf(result, "%p", action.sa_handler);
    }
 return result;
}

static void
sigaction_info(void)
{
 int n, m;
 char mask[NSIG + NSIG/5];
 struct sigaction action;

 for (printf("\n"), n = 0; n < strlen(flag[0].name); n++, printf("\n"))
    for (printf("    "), m = 0; m < number_of_flags; m++)
       printf(" %c", flag[m].name[n]);

 for (n = 1; n < NSIG; n++)
     {char flags[1 + 2*number_of_flags], handler[10];
      sigaction(n, NULL, &action);
      printf("(%2d)%s | %s | %s | (%2d) %s\n", n,
                                               flags_string(action.sa_flags, flags),
                                               mask_string(action.sa_mask, mask),
	 				       handler_string(action, handler),
                                               n, strsignal(n));
     }

 sigprocmask(0, NULL, &action.sa_mask);
 for (n = 0; n < 4 + 2*number_of_flags - strlen("sigprocmask"); n++) printf(" ");
 printf("sigprocmask = %s\n", mask_string(action.sa_mask, mask));
}
#endif /* RESCUE_DEBUG */
