/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_project.h"
#include "sg.h"
#include "sg_dialogs.h"
#include "sg_misc_dialogs.h"
#include "sg_menus.h"
#include "sg_dataset.h"
#include "sg_clipboard.h"
#include "../pixmaps/sheet_icon2.xpm"
#include "../pixmaps/plot_icon2.xpm"
#include "../pixmaps/matrix_icon.xpm"


gint num_worksheets;
gint num_plots;
gint num_functions;
gint num_expressions;
gint last_worksheet;
gint last_matrix;
gint last_plot;
gint last_dataset;
gint last_function;
gint last_expression;

gint arrange_rows;
gint arrange_cols;
gdouble arrange_vgap;
gdouble arrange_hgap;
gdouble arrange_top;
gdouble arrange_bottom;
gdouble arrange_left;
gdouble arrange_right;

GList *worksheets;
GList *plots;
GList *functions;
GList *datasets;
GList *expressions;


static gboolean remove_worksheet		(SGworksheet *worksheet);
static gboolean remove_plot			(SGplot *plot);
static gboolean	remove_dataset			(SGdataset *dataset);
static gboolean rename_worksheet		(GtkIconList *iconlist, 
						 GtkIconListItem *item);
static gboolean rename_plot			(GtkIconList *iconlist, 
						 GtkIconListItem *item);
static SGdataset *find_data_by_id               (GList *list,
                                                 gint id);
static gboolean focus_in_event			(GtkWidget *widget, 
						 GdkEventFocus *event);
static gint     delete_event                    (GtkWidget *window);


void
sg_project_new()
{
 sg_project_init();

 if(gui_iconlist){
   gtk_signal_connect(GTK_OBJECT(iconlist2), "deactivate_icon",
                      (GtkSignalFunc)rename_plot, iconlist2);

   gtk_signal_connect(GTK_OBJECT(iconlist1), "deactivate_icon",
                      (GtkSignalFunc)rename_worksheet, iconlist1);
 }
}

void
sg_project_init()
{
 last_function = num_functions = 0;
 last_plot = num_plots = 0;
 last_worksheet = num_worksheets = 0;
 last_dataset = 0;
 last_expression = 0;
 last_matrix = 0;
 active_icon = NULL;
 active_worksheet = NULL;
 active_plot = NULL;
 active_layer = NULL;
 active_dataset = NULL;
 arrange_rows = 2;
 arrange_cols = 1;
 arrange_top = .15;
 arrange_bottom = .15;
 arrange_left = .15;
 arrange_right = .15;
 arrange_vgap = .15;
 arrange_hgap = .15;

 project_changed = FALSE;

 sg_clipboard_init();
 sg_dialogs_init();
 plots = NULL;
 worksheets = NULL;
 datasets = NULL;
 functions = NULL;
 expressions = NULL;

 if(last_project_filename){
    g_free(last_project_filename);
    last_project_filename = g_strdup("untitled.sg");
 }

 sg_project_set_title();
}

void
sg_project_set_title()
{
  gchar title[1000];
  gchar icon_title[1000];

  if(project_changed){
    g_snprintf(title, 1000, "SciGraphica: %s (changed)", last_project_filename);
    g_snprintf(icon_title, 1000, "%s (changed)", last_project_filename);
  } else {
    g_snprintf(title, 1000, "SciGraphica: %s", last_project_filename);
    g_snprintf(icon_title, 1000, "%s", last_project_filename);
  }

  if(!gui_iconlist) return;

  gtk_window_set_title(GTK_WINDOW(main_window), title);
  gdk_window_set_icon_name(GTK_WIDGET(main_window)->window, icon_title);
}

void
sg_project_changed(gboolean changed)
{
  if(project_changed != changed){
    project_changed = changed;
    sg_project_set_title();
  }
}
  
SGworksheet *
sg_project_new_worksheet()
{
 GtkIconList *iconlist;
 SGworksheet *worksheet;
 gchar *name;
 gchar label[20];

 sg_project_changed(TRUE);

 num_worksheets++;
 last_worksheet++;
 sprintf(label,"Data%02d",last_worksheet);
 name = g_strdup(label);

 worksheet = sg_worksheet_new(name, 20, 5);
 worksheets = g_list_append(worksheets, worksheet);

 active_worksheet = worksheet;

 gtk_signal_connect(GTK_OBJECT(worksheet->sheet), "button_press_event",
                    (GtkSignalFunc)sg_worksheet_menu_show, NULL);

 gtk_signal_connect(GTK_OBJECT(worksheet),"delete_event",
                    (GtkSignalFunc) delete_event, NULL);

 gtk_signal_connect(GTK_OBJECT(worksheet),"focus_in_event",
                    (GtkSignalFunc) focus_in_event, NULL);

 gtk_signal_connect(GTK_OBJECT(worksheet),"set_focus",
                    (GtkSignalFunc) focus_in_event, (gpointer)worksheet);

 if(!gui_iconlist){
    g_free(name);
    return worksheet;
 }

 iconlist = GTK_ICON_LIST(iconlist1);

 worksheet->iconlist = iconlist;
 worksheet->icon = gtk_icon_list_add_from_data(iconlist,
                                               sheet_icon2_xpm,
                                               name,
                                               worksheet);

 gtk_icon_list_set_active_icon(iconlist, worksheet->icon);

 gtk_entry_select_region(GTK_ENTRY(worksheet->icon->entry), 0, strlen(name));
 sg_entry_cancel_space(GTK_ENTRY(worksheet->icon->entry));

 gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 0);

 g_free(name);

 return worksheet;
}

SGworksheet *
sg_project_new_matrix(void)
{
 GtkIconList *iconlist;
 SGworksheet *worksheet;
 gchar *name;
 gchar label[20];
 GdkColor color;

 sg_project_changed(TRUE);

 num_worksheets++;
 last_matrix++;
 sprintf(label,"Matrix%02d",last_matrix);
 name = g_strdup(label);

 worksheet = sg_worksheet_new(name, 10, 10);
 worksheets = g_list_append(worksheets, worksheet);

 worksheet->mode = SG_MODE_MATRIX;

 gdk_color_parse("light yellow", &color);
 gdk_color_alloc(gdk_colormap_get_system(), &color);
 gtk_sheet_set_background(GTK_SHEET(worksheet->sheet), &color);
 gdk_color_parse("light blue", &color);
 gdk_color_alloc(gdk_colormap_get_system(), &color);
 gtk_sheet_set_grid(GTK_SHEET(worksheet->sheet), &color);


 active_worksheet = worksheet;

 gtk_signal_connect(GTK_OBJECT(worksheet->sheet), "button_press_event",
                    (GtkSignalFunc)sg_matrix_menu_show, NULL);

 gtk_signal_connect(GTK_OBJECT(worksheet),"delete_event",
                    (GtkSignalFunc) delete_event, NULL);

 gtk_signal_connect(GTK_OBJECT(worksheet),"focus_in_event",
                    (GtkSignalFunc) focus_in_event, NULL);

 gtk_signal_connect(GTK_OBJECT(worksheet),"set_focus",
                    (GtkSignalFunc) focus_in_event, (gpointer)worksheet);


 if(!gui_iconlist){
    g_free(name);
    return worksheet;
 }

 iconlist = GTK_ICON_LIST(iconlist1);

 worksheet->iconlist = iconlist;
 worksheet->icon = gtk_icon_list_add_from_data(iconlist,
                                               matrix_icon_xpm,
                                               name,
                                               worksheet);

 gtk_icon_list_set_active_icon(iconlist, worksheet->icon);

 gtk_entry_select_region(GTK_ENTRY(worksheet->icon->entry), 0, strlen(name));
 sg_entry_cancel_space(GTK_ENTRY(worksheet->icon->entry));

 gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 0);

 g_free(name);

 return worksheet;
}

SGplot *
sg_project_new_plot()
{
 GtkIconList *iconlist;
 SGplot *plot;
 gchar *name;
 gchar label[20];

 sg_project_changed(TRUE);

 num_plots++;
 last_plot++;
 sprintf(label,"Plot%02d",last_plot);
 name = g_strdup(label);


 plot = sg_plot_new(name);
 plots = g_list_append(plots, plot);

 active_plot = plot;

 gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "button_press_event",
                    (GtkSignalFunc)sg_plot_menu_show, NULL);

 gtk_signal_connect(GTK_OBJECT(plot),"delete_event",
                    (GtkSignalFunc) delete_event, NULL);

 gtk_signal_connect(GTK_OBJECT(plot),"focus_in_event",
                    (GtkSignalFunc) focus_in_event, NULL);

 if(!gui_iconlist){
    g_free(name);
    return plot;
 }

 iconlist = GTK_ICON_LIST(iconlist2);

 plot->iconlist = iconlist;
 plot->icon = gtk_icon_list_add_from_data(iconlist, plot_icon2_xpm, name, plot);

 gtk_icon_list_set_active_icon(iconlist, plot->icon);
 gtk_entry_select_region(GTK_ENTRY(plot->icon->entry), 0, strlen(name));
 sg_entry_cancel_space(GTK_ENTRY(plot->icon->entry));

 gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 1);

 sg_layer_control_refresh(label);

 g_free(name);

 return plot;
}


SGplot *
sg_project_new_plot_with_layer(SGlayerType layer_type)
{
 SGplot *plot;
 GtkIconList *iconlist;
 gchar *name;
 gchar label[20];

 sg_project_changed(TRUE);

 num_plots++;
 last_plot++;
 sprintf(label,"Plot%02d",last_plot);
 name = g_strdup(label);


 plot = sg_plot_new_with_layer(layer_type, name);
 plots = g_list_append(plots, plot);

 active_plot = plot;

 gtk_signal_connect(GTK_OBJECT(plot),"delete_event",
                    (GtkSignalFunc) delete_event, NULL);

 gtk_signal_connect(GTK_OBJECT(plot),"focus_in_event",
                    (GtkSignalFunc) focus_in_event, NULL);

 gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "button_press_event",
                    (GtkSignalFunc)sg_plot_menu_show, NULL);

 if(!gui_iconlist){
    g_free(name);
    return plot;
 }
 
 iconlist = GTK_ICON_LIST(iconlist2);

 plot->iconlist = iconlist;
 plot->icon = gtk_icon_list_add_from_data(iconlist, plot_icon2_xpm, name, plot);

 gtk_icon_list_set_active_icon(iconlist, plot->icon);
 gtk_entry_select_region(GTK_ENTRY(plot->icon->entry), 0, strlen(name));
 sg_entry_cancel_space(GTK_ENTRY(plot->icon->entry));

 gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 1);

 sg_layer_control_refresh(label);

 g_free(name);

 return plot;
}

SGdataset *
sg_project_new_dataset(SGdataStyle style)
{
  SGdataset *dataset;

  sg_project_changed(TRUE);

  dataset = sg_dataset_new(style);
  datasets = g_list_append(datasets, dataset);

  dataset->id = last_dataset;
  last_dataset++;

  return dataset;
}

SGdataset *
sg_project_new_function(gchar *exp)
{
  SGdataset *function;
  gchar text[20];

  sg_project_changed(TRUE);

  function = sg_dataset_new_function(exp);
  functions = g_list_append(functions, function);

  num_functions++;
  last_function++;

  function->id = last_dataset;
  last_dataset++;

  snprintf(text, 20,"f%d(x)", last_function);
  sg_dataset_set_name(function, text);

  return function;
}

SGdataset *
sg_project_new_expression(SGdataStyle style, gchar *exp[9])
{
  SGdataset *dataset;

  sg_project_changed(TRUE);

  dataset = sg_dataset_new_python(style, exp);

  expressions = g_list_append(expressions, dataset);

  num_expressions++;
  last_expression++;

  dataset->id = last_dataset;
  last_dataset++;

  return dataset;
}


gboolean
sg_project_remove_worksheet(SGworksheet *worksheet)
{
  gboolean veto;
  gboolean deleting = FALSE;
  GList *list;
 
  list = datasets; 
  while(list) {
    SGdataset *dataset;
    
    dataset = SG_DATASET(list->data);
    if(dataset->worksheet && dataset->worksheet == worksheet){
      if(!deleting){
        gchar text[1000];
        g_snprintf(text, 1000, "There are datasets linked to this worksheet.\n The datasets will also be deleted.\n Do you really want to remove\n%s ?", worksheet->name);
        veto = sg_accept_dialog(text, 1);
        if(!veto) return FALSE;
        deleting = TRUE;
      }
      if(deleting){
        sg_project_remove_dataset(dataset, TRUE); 
        list = datasets;
      } else {
        list = list->next;
      }
    } else {
      list = list->next;
    }
  }

  veto = remove_worksheet(worksheet);    
  if(!veto) return FALSE;

  if(gui_worksheet && !worksheets) gui_worksheet = FALSE;

  if(!gui_iconlist && !gui_plot && !gui_worksheet){
    main_quit(NULL);
  }

  return TRUE;
}

gboolean
sg_project_remove_dataset(SGdataset *dataset, gboolean force)
{
  SGplot *plot;
  GList *list;
  GList *aux_plots;
  GList *aux_layers;
  GList *aux_datasets;
  SGdataset *parent;
  gboolean deleting = force;
  gboolean veto = TRUE;
 
  parent = SG_DATASET(gtk_plot_data_get_link(dataset->real_data));

  aux_plots = plots; 
  while(aux_plots) {
    plot = SG_PLOT(aux_plots->data);
    aux_layers = plot->layers;
    while(aux_layers){
      aux_datasets = SG_LAYER(aux_layers->data)->datasets;
      while(aux_datasets){
        SGdataset *aux_data, *aux_parent;

        aux_data = SG_DATASET(aux_datasets->data);
        aux_parent = SG_DATASET(gtk_plot_data_get_link(aux_data->real_data)); 

        if(aux_parent == parent && !deleting){
          gchar text[1000];
          g_snprintf(text, 1000, "This dataset is being used in plots.\n Do you really want to delete\n%s ?", parent->real_data->name);
          veto = sg_accept_dialog(text, 1);
          if(!veto) return FALSE;
          deleting = TRUE;
        }
        if(deleting && aux_parent == parent){
          sg_layer_remove_dataset(SG_LAYER(aux_layers->data), aux_data);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
          aux_datasets = SG_LAYER(aux_layers->data)->datasets;
        }else{
          aux_datasets = aux_datasets->next;
        }
      } 
      aux_layers = aux_layers->next;
    }
    aux_plots = aux_plots->next;
  }
  veto = remove_dataset(parent);
  return veto;
}

static gboolean
remove_dataset(SGdataset *dataset)
{ 
  GList *list;

  if(dataset->type == SG_DATA_FUNCTION){
     list = g_list_find(functions, dataset);
     if(!list) return FALSE;
     functions = g_list_remove_link(functions, list);
     num_functions--;
  } else if(dataset->exp){
     list = g_list_find(expressions, dataset);
     if(!list) return FALSE;
     expressions = g_list_remove_link(expressions, list);
     num_expressions--;
  }else{
     list = g_list_find(datasets, dataset);
     if(!list) return FALSE;
     datasets = g_list_remove_link(datasets, list);
  }

  sg_project_changed(TRUE);

  if(GTK_IS_OBJECT(dataset))
    gtk_object_destroy(GTK_OBJECT(dataset));

  g_list_free_1(list);
  return TRUE;
}

gboolean
sg_project_remove_plot(SGplot *plot)
{
  gboolean veto;

  veto = remove_plot(plot);    
  if(!veto) return FALSE;

  if(gui_plot && !plots) gui_plot = FALSE;

  if(!gui_iconlist && !gui_plot && !gui_worksheet){
    main_quit(NULL);
  }
  return TRUE;
}

void
sg_project_close()
{
  GList *list;

  if(gui_iconlist){
#ifndef WITH_GNOME
    gtk_signal_disconnect_by_data(GTK_OBJECT(iconlist1), iconlist1);
    gtk_signal_disconnect_by_data(GTK_OBJECT(iconlist2), iconlist2);
#endif
    gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist1), NULL);
    gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist2), NULL);
  }

  list = plots;
  while(list){
    remove_plot((SGplot *)list->data);
    list = plots;
  }

  list = worksheets;
  while(list){
    remove_worksheet((SGworksheet *)list->data);
    list = worksheets;
  }

  list = datasets;
  while(list){
    datasets = g_list_remove_link(datasets, list);
    g_list_free_1(list);
    list = datasets;
  }

  list = functions;
  while(list){
    functions = g_list_remove_link(functions, list);
    g_list_free_1(list);
    list = functions;
  }

  list = expressions;
  while(list){
    expressions = g_list_remove_link(expressions, list);
    g_list_free_1(list);
    list = functions;
  }

  sg_layer_control_destroy();
  sg_clipboard_clear();
  sg_dialogs_destroy(NULL);
}

static gboolean 
remove_worksheet(SGworksheet *worksheet)
{
  GList *list;
  SGdataset *dataset = NULL;
  SGlayer *layer = NULL;
  SGplot *plot = NULL;
  gboolean found = FALSE;

  if(worksheet == NULL) return FALSE;

  list = plots;
  while(list){
    GList *layers;
    GList *layer_data;

    plot = (SGplot *)list->data;
    layers = plot->layers;

    while(layers){
      layer = (SGlayer *)layers->data;

      layer_data = layer->datasets;
      while(layer_data){
        dataset = (SGdataset *)layer_data->data;

        if(dataset->worksheet == worksheet){
            found = TRUE;
        }

        layer_data = layer_data->next;
      }

      layers = layers->next;
    }

    list = list->next;
  }

  if(found){
     gchar message[255];
     g_snprintf(message, 255,"The worksheet is connected to active datasets\nAre you sure you want to remove it?");
     if( sg_accept_dialog(message, 1) != YES_CLICKED ) return FALSE;
  }


  list = g_list_find(worksheets, worksheet);

  if(!list) return FALSE;

  sg_project_changed(TRUE);

  if(gui_iconlist){
    gtk_icon_list_remove(worksheet->iconlist, worksheet->icon);
    gtk_icon_list_update(worksheet->iconlist);
  }

  gtk_widget_destroy(GTK_WIDGET(worksheet));

  worksheets = g_list_remove_link(worksheets, list);
  g_list_free_1(list);

  num_worksheets--;

  if(num_worksheets == 0) worksheets = NULL;

  return TRUE;
}

static gboolean 
remove_plot(SGplot *plot)
{
  GList *list;

  if(plot == NULL) return FALSE;

  list = g_list_find(plots, plot);

  if(!list) return FALSE;

  sg_project_changed(TRUE);

  if(gui_iconlist){
    gtk_icon_list_remove(plot->iconlist, plot->icon);
    gtk_icon_list_update(plot->iconlist);
  }

  gtk_widget_destroy(GTK_WIDGET(plot));

  plots = g_list_remove_link(plots, list);
  g_list_free_1(list);

  num_plots--;

  if(num_plots == 0) plots = NULL;

  sg_layer_control_refresh(NULL);

  return TRUE;
}

static gboolean
rename_worksheet(GtkIconList *iconlist, GtkIconListItem *item)
{
  gchar *name;
  SGworksheet *worksheet;

  if(!gui_iconlist) return TRUE;

  worksheet = (SGworksheet *)gtk_icon_list_get_link(item);
  name = gtk_entry_get_text(GTK_ENTRY(item->entry));

  return (sg_project_rename_worksheet(worksheet, name));
}


gint
sg_project_rename_worksheet(SGworksheet *worksheet, gchar *name)
{
  GList *list;
  gint n;
  gint veto;
  gboolean accept = FALSE;
  gboolean free_name = FALSE;

  while(!name || strlen(name) == 0){
      if(free_name) g_free(name);
      free_name = FALSE;
      name = sg_text_dialog("The worksheet must have a name.\nPlease enter a name:", 1);
      free_name = TRUE;
  }

  list = worksheets;
  while(!accept){
    gboolean new_name = FALSE;
    while(list){
     SGworksheet *data;

     data = (SGworksheet *)list->data;

     if(data != worksheet && strcmp(data->name, name) == 0){
        if(free_name) g_free(name);
        name = NULL;
        free_name = FALSE;
        while(!name || strlen(name) == 0){
           name = sg_text_dialog("Another worksheet has the same name.\nPlease enter a new name for this worksheet:", 1);
        }
        free_name = TRUE;
        new_name = TRUE;
        break; 
     }
     new_name = FALSE;
     list = list->next;
    }
    if(!new_name) accept = TRUE;
  }

  for(n = 0; n < strlen(name); n++){
    if(name[n] < 'a' || name[n] > 'z')
     if(name[n] < 'A' || name[n] > 'Z')
      if(name[n] < '0' || name[n] > '9')
        if(name[n] != '.' && name[n] != '_' && name[n] != '-' && name[n] != '+' && name[n] != '='){
          name[n] = '_';
/*
          sg_message_dialog("The name contains invalid characters", 0);
          return FALSE;
*/
      }
  }

  veto = sg_worksheet_rename(worksheet, name);

  if(veto){
    sg_project_refresh_datasets();
    sg_project_changed(TRUE);
  }

  if(free_name) g_free(name);
  return TRUE;
}

static gboolean
rename_plot(GtkIconList *iconlist, GtkIconListItem *item)
{
  gchar *name;
  SGplot *plot;

  if(!gui_iconlist) return TRUE;

  plot = (SGplot *)gtk_icon_list_get_link(item);
  name = gtk_entry_get_text(GTK_ENTRY(item->entry));

  return(sg_project_rename_plot(plot, name));
}

gint
sg_project_rename_plot(SGplot *plot, gchar *name)
{
  GList *list;
  gint n;
  gint veto;
  gboolean accept = FALSE;
  gboolean free_name = FALSE;

  while(!name || strlen(name) == 0){
    if(free_name) g_free(name);
    free_name = FALSE;
    name = sg_text_dialog("The plot must have a name.\nPlease enter a name for this plot:", 1);
    free_name = TRUE;
  }

  list = plots;
  while(!accept){
    gboolean new_name = FALSE;
    while(list){
     SGplot *data;

     data = (SGplot *)list->data;

     if(data != plot && strcmp(data->name, name) == 0){
        if(free_name) g_free(name);
        name = NULL;
        free_name = FALSE;
        while(!name || strlen(name) == 0){
          name = sg_text_dialog("Another plot has the same name.\nPlease enter a new name for this plot:", 1);
        }
        free_name = TRUE;
        new_name = TRUE;
        break;
     }
     new_name = FALSE;
     list = list->next;
    }
    if(!new_name) accept = TRUE;
  }

  for(n = 0; n < strlen(name); n++){
    if(name[n] < 'a' || name[n] > 'z')
     if(name[n] < 'A' || name[n] > 'Z')
      if(name[n] < '0' || name[n] > '9')
        if(name[n] != '.' && name[n] != '_' && name[n] != '-' && name[n] != '+' && name[n] != '='){
          name[n] = '_';
/*
          sg_message_dialog("The name contains invalid characters", 0);
          return FALSE;
*/
      }
  }

  veto = sg_plot_rename(plot, name);

  if(veto) sg_project_changed(TRUE);

  if(free_name) g_free(name);
  return TRUE;
}

void
sg_project_open_worksheet(GtkWidget *widget, 
                          GtkIconListItem *item, GdkEvent *event,
                          gpointer data)
{
  GtkIconList *iconlist;
  SGworksheet *worksheet;
  GdkModifierType mods;
  iconlist = GTK_ICON_LIST(widget);

/*
  sg_layer_control_destroy();
*/

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);

  worksheet = (SGworksheet *)gtk_icon_list_get_link(item);

  if(!event) return;
  if((mods & GDK_BUTTON1_MASK) && event->type == GDK_2BUTTON_PRESS)
  {
                gtk_widget_show(GTK_WIDGET(worksheet));
                gdk_window_raise(GTK_WIDGET(worksheet)->window);
  }

  if(mods & GDK_BUTTON3_MASK)
                sg_worksheet_popup_show(item,(GdkEventButton *)event);

  active_worksheet = worksheet;

  return;

}

void
sg_project_open_plot(GtkWidget *widget, 
                     GtkIconListItem *item,
                     GdkEvent *event,
                     gpointer data)
{
  GtkIconList *iconlist;
  SGplot *plot;
  GdkModifierType mods;
  iconlist = GTK_ICON_LIST(widget);

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);

  plot = (SGplot *)gtk_icon_list_get_link(item);
  if(!event) return;
  if((mods & GDK_BUTTON1_MASK) && event->type == GDK_2BUTTON_PRESS){
                gtk_widget_show(GTK_WIDGET(plot));
                gdk_window_raise(GTK_WIDGET(plot)->window);
  }

  if(mods & GDK_BUTTON3_MASK)
                sg_plot_popup_show(item,(GdkEventButton *)event);

  active_plot = plot;

  return;

}

SGworksheet *
sg_project_get_worksheet (gchar *name)
{
  SGworksheet *worksheet;
  GList *list;

  if(!name || !strcmp(name,"current")) return active_worksheet;

  list = worksheets;
  while(list){
    worksheet = (SGworksheet *) list->data;

    if(strcmp(worksheet->name, name) == 0) return worksheet;
    list = list->next;
  }

  return NULL;
} 

SGplot *
sg_project_get_plot (gchar *name)
{
  SGplot *plot;
  GList *list;

  if(!name) return active_plot;

  list = plots;
  while(list){
    plot = (SGplot *) list->data;

    if(strcmp(plot->name, name) == 0) return plot;
    list = list->next;
  }

  return NULL;
}

void
sg_project_refresh_datasets()
{
  GList *list;

  list = plots;
  while(list){
    SGplot *plot;

    plot = (SGplot *)list->data;

    sg_plot_refresh_datasets(plot);
    list = list->next;
  }
}

static SGdataset *
find_data_by_id(GList *list, gint id)
{
  GList *aux = list; 
  SGdataset *dataset = NULL;

  while(aux){
    SGdataset *aux_data;

    aux_data = (SGdataset *)aux->data;
    if(aux_data && aux_data->id == id){
      dataset = aux_data;
      break;
    } 
    aux = aux->next;
  }

  return dataset;
}

SGdataset *
sg_project_dataset_get_by_id(gint id)
{
  return(find_data_by_id(datasets, id));
}

SGdataset *
sg_project_function_get_by_id(gint id)
{
  return(find_data_by_id(functions, id));
}

SGdataset *
sg_project_expression_get_by_id(gint id)
{
  return(find_data_by_id(expressions, id));
}

static gint
delete_event(GtkWidget *widget)
{
  if(GTK_IS_SG_PLOT(widget)){
    SGplot *plot = SG_PLOT(widget);

    sg_layer_control_destroy();

    gdk_window_get_position(widget->window,
                            &plot->x,
                            &plot->y);
    gdk_window_get_size(widget->window,
                        &plot->width,
                        &plot->height);

  } else if(GTK_IS_SG_WORKSHEET(widget)){
    SGworksheet *worksheet = SG_WORKSHEET(widget);

    gdk_window_get_position(widget->window,
                            &worksheet->x,
                            &worksheet->y);
    gdk_window_get_size(widget->window,
                        &worksheet->width,
                        &worksheet->height);
  }

  gtk_widget_hide(widget);

  return TRUE;
}

static gboolean
focus_in_event(GtkWidget *widget, GdkEventFocus *event)
{
  GList *list = NULL;
  gchar msg[80];

  if(GTK_IS_SG_PLOT(widget)){
    list = plots;
    while(list)
     {
       SGplot *plot;
       plot = SG_PLOT(list->data);

       if(GTK_WIDGET(plot) == widget){
          active_plot = plot;
          break;
       }

       list = list->next;
     }
#ifdef WITH_GNOME
    g_snprintf(msg,80,"Active plot is %s", active_plot->name);

    if(gui_iconlist) gnome_app_flash(GNOME_APP(main_window), msg);
#else
/* Do something equivalent for gtk!*/
#endif

    gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
  } else if(GTK_IS_SG_WORKSHEET(widget)){

    list = worksheets;
    while(list)
     {
       SGworksheet *worksheet;
       worksheet = SG_WORKSHEET(list->data);

       if(GTK_WIDGET(worksheet) == widget){
          active_worksheet = worksheet;
          break;
       }

       list = list->next;
     }
#ifdef WITH_GNOME
    g_snprintf(msg,80,"Active sheet is %s", active_worksheet->name);

    if(gui_iconlist) gnome_app_flash(GNOME_APP(main_window), msg);
#else
/* Do something equivalent for gtk!*/
#endif
  }

  return FALSE;
}

