#include <gdk/gdk.h>
#include <glib.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <string.h>
#include <Python.h>
#include <grammar.h>
#include <node.h>
#include <parsetok.h>
#include <errcode.h>
#include <compile.h>
#include <eval.h>
#include <marshal.h>
#include "python_main.h"
#include "../sg.h"
#include "../sg_wrap.h"

/* Function forward declarations */
static PyObject *get_cell(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *get_cell_double(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *get_col_list(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *get_row_list(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *get_col_double_array(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *get_row_double_array(PyObject *self, PyObject *args, PyObject *keywds);

PyObject *set_sheet(gint row, gint col, PyObject *object, SGworksheet *worksheet,
                     GtkOrientation orient);

static PyObject *set_cell(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *set_col(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *clear_col(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *set_row(PyObject *self, PyObject *args, PyObject *keywds);

static PyObject *_new_worksheet(PyObject *self, PyObject *args);
static PyObject *_remove_worksheet(PyObject *self, PyObject *args);
static PyObject *_rename_worksheet(PyObject *self, PyObject *args);
static PyObject *_update_worksheet(PyObject *self, PyObject *args);
static PyObject *_set_cell_value(PyObject *self, PyObject *args);
static PyObject *_set_cell_text(PyObject *self, PyObject *args);
static PyObject *_get_cell_value(PyObject *self, PyObject *args);
static PyObject *_get_cell_text(PyObject *self, PyObject *args);
static PyObject *_set_column_name(PyObject *self, PyObject *args);
static PyObject *_set_column_values(PyObject *self, PyObject *args);
static PyObject *_set_column_type(PyObject *self, PyObject *args);
static PyObject *_set_column_numbers(PyObject *self, PyObject *args);

static PyObject *_new_plot(PyObject *self, PyObject *args);
static PyObject *_remove_plot(PyObject *self, PyObject *args);
static PyObject *_rename_plot(PyObject *self, PyObject *args);
static PyObject *_clear_plot(PyObject *self, PyObject *args);

static PyObject *_new_layer(PyObject *self, PyObject *args);
static PyObject *_remove_layer(PyObject *self, PyObject *args);
static PyObject *_move_layer(PyObject *self, PyObject *args);
static PyObject *_resize_layer(PyObject *self, PyObject *args);
static PyObject *_clear_layer(PyObject *self, PyObject *args);
static PyObject *_show_legends(PyObject *self, PyObject *args);
static PyObject *_hide_legends(PyObject *self, PyObject *args);
static PyObject *_move_legends(PyObject *self, PyObject *args);

static PyObject *_set_xrange(PyObject *self, PyObject *args);
static PyObject *_set_yrange(PyObject *self, PyObject *args);
static PyObject *_set_zrange(PyObject *self, PyObject *args);
static PyObject *_set_xticks(PyObject *self, PyObject *args);
static PyObject *_set_yticks(PyObject *self, PyObject *args);
static PyObject *_set_zticks(PyObject *self, PyObject *args);
static PyObject *_set_xtitle(PyObject *self, PyObject *args);
static PyObject *_set_ytitle(PyObject *self, PyObject *args);
static PyObject *_set_ztitle(PyObject *self, PyObject *args);
static PyObject *_show_xgrids(PyObject *self, PyObject *args);
static PyObject *_show_ygrids(PyObject *self, PyObject *args);
static PyObject *_show_zgrids(PyObject *self, PyObject *args);
static PyObject *_hide_xgrids(PyObject *self, PyObject *args);
static PyObject *_hide_ygrids(PyObject *self, PyObject *args);
static PyObject *_hide_zgrids(PyObject *self, PyObject *args);

static PyObject *_set_symbol(PyObject *self, PyObject *args);
static PyObject *_set_symbol_style(PyObject *self, PyObject *args);
static PyObject *_set_symbol_color(PyObject *self, PyObject *args);
static PyObject *_set_line_style(PyObject *self, PyObject *args);
static PyObject *_set_line_color(PyObject *self, PyObject *args);
static PyObject *_set_connector(PyObject *self, PyObject *args);

static PyObject *_plot_exp_xy(PyObject *self, PyObject *args);
static PyObject *_plot_exp_xy_bars(PyObject *self, PyObject *args);
static PyObject *_plot_exp(PyObject *self, PyObject *args);

static PyObject *_read_file(PyObject *self, PyObject *args);

/*static PyObject *sg_import(PyObject *self, PyObject *args);*/
static PyObject *fregister(PyObject *self, PyObject *args);
static PyObject *new_plugin_group(PyObject *self, PyObject *args);
static PyObject *get_selection(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_describe_all(PyObject *self, PyObject *args);

static PyObject *_get_active_worksheet_name(PyObject *self, PyObject *args);
static PyObject *_get_worksheet_names(PyObject *self, PyObject *args);
static PyObject *_get_column_names(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_worksheet_add_columns(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_worksheet_get_column_exp(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_worksheet_get_column_precision(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_config_register(PyObject *self, PyObject *args);
static PyObject *_config_get_value(PyObject *self, PyObject *args);
static PyObject *_config_set_value(PyObject *self, PyObject *args);
static PyObject *_config_exec_default(PyObject *self, PyObject *args);
static PyObject *_config_exec_commit(PyObject *self, PyObject *args);

static PyObject *_plot_list(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_layers(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_active_layer(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_layer_dataset_list(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_layer_dataset(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_active(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_layer_refresh(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_layer_autoscale(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_refresh_canvas(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_freeze_canvas(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_thaw_canvas(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *_plot_get_size_pos(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *axis_text_properties(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *set_axis_label_pos(PyObject *self, PyObject *args, PyObject *keywds);

SGworksheet *sg_current_worksheet;
GSList *py_mod_list_head=NULL;

static PyObject *no_such_worksheet(gchar *sheet){
     gchar message[160];
     g_snprintf(message,160,"No such sheet '%s'",sheet);
     PyErr_SetString(PyExc_ValueError,message);
     return NULL;
}



static PyObject *
_set_column_numbers(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *col;
  gint format, internal, precision;

  if (!PyArg_ParseTuple(args, "ssiii:set_column_numbers", &name, &col, &internal, &format, &precision))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  set_column_numbers(name, col, format, internal, precision);

  Py_INCREF(Py_None);
  return Py_None;
}



static PyObject *
_set_column_type(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *col;
  gint type;

  if (!PyArg_ParseTuple(args, "ssi:set_column_type", &name, &col, &type))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  set_column_type(name, col, type);

  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
_set_column_values(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *col;
  gchar *exp;
  gint from = -1, to = -1;

  if (!PyArg_ParseTuple(args, "sss|i|i:set_column_values", &name, &col, &exp, &from, &to))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  set_column_values(name, col, exp, from, to);

  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
_set_column_name(PyObject *self, PyObject *args)
{
  PyObject *col_obj;
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *new_name = NULL;
  gchar *col;


  if (!PyArg_ParseTuple(args, "sOs:set_column_name", &name, &col_obj, &new_name))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  if (PyLong_Check(col_obj))
      sg_worksheet_set_column_name(worksheet,(int)PyLong_AsLong(col_obj),new_name);
  else if (PyInt_Check(col_obj))
      sg_worksheet_set_column_name(worksheet,(int)PyInt_AsLong(col_obj),new_name);
  else if (PyString_Check(col_obj))
      set_column_name(name, PyString_AsString(col_obj), new_name);
  else{
      PyErr_SetString(PyExc_TypeError,"Second argument must be either the column name "
                      "or the column number");
      return NULL;
  }

  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
_set_cell_value(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *col;
  gint row;
  gdouble value;

  if (!PyArg_ParseTuple(args, "ssid:set_cell_value", &name, &col, &row, &value))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  set_cell_value(name, col, row, value);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_cell_text(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *col;
  gint row;
  gchar *text;

  if (!PyArg_ParseTuple(args, "ssis:set_cell_text", &name, &col, &row, &text))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  set_cell_text(name, col, row, text);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_get_cell_value(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *col;
  gint row;
  gdouble value;
  PyObject *object;

  if (!PyArg_ParseTuple(args, "ssi:get_cell_value", &name, &col, &row))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  value = get_cell_value(name, col, row);

  object=Py_BuildValue("f", value);

  Py_INCREF(object);
  return object;
}

static PyObject *
_get_cell_text(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *col;
  gint row;
  gchar *text;
  PyObject *object;

  if (!PyArg_ParseTuple(args, "ssi:get_cell_text", &name, &col, &row))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  text = get_cell_text(name, col, row);

  object=Py_BuildValue("s", text);

  Py_INCREF(object);
  return object;
}




static PyObject *
_read_file(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;
  gchar *path = NULL;

  if (!PyArg_ParseTuple(args, "ss:read_file", &name, &path))
      return NULL;

  worksheet = sg_project_get_worksheet(name);
  if(!worksheet) return no_such_worksheet(name);

  if(read_file(name, path) == WRAP_ERROR){
      /* PRINT ERROR "Can't read file" */
    return NULL;
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_new_worksheet(PyObject *self, PyObject *args)
{
  SGworksheet *worksheet;
  gchar *name = NULL;

  if (!PyArg_ParseTuple(args, "|s:new_worksheet", &name))
      return NULL;

  worksheet = new_worksheet(name);

  if(!worksheet){
      /* PRINT ERROR "Another worksheet has the same name" */
  }

  Py_INCREF(Py_None);
  return Py_None;
}



static PyObject *
_new_plot(PyObject *self, PyObject *args)
{
  SGplot *plot;
  gint type;
  gchar *name = NULL,*tname="Plot01",*active_name;

  if (!PyArg_ParseTuple(args, "i|s:new_plot", &type, &name))
      return NULL;

  if (!name)
    name=get_active_plot_name();

  if (!name)
    plot = new_plot(strdup(tname), type);
  else 
  { plot=sg_project_get_plot(name);
    if (!plot)
      plot = new_plot(strdup(name), type);
    else{
       PyErr_SetString(PyExc_ValueError,"A plot with that name already exists");
       return NULL;
   }
  }
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"Unable to create new plot");
       return NULL;
   }
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_new_layer(PyObject *self, PyObject *args)
{
  SGlayer *layer;
  gint type;
  gchar *plot_name = NULL;

  if (!PyArg_ParseTuple(args, "si:new_layer", &plot_name, &type))
      return NULL;

  layer = new_layer(plot_name, type);

  if(!layer){
      /* PRINT ERROR "No plot with that name" */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_remove_layer(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:remove_layer", &plot_name, &nth))
      return NULL;

  change = remove_layer(plot_name, nth);
  
  if(change == WRAP_ERROR)
  { 
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_clear_layer(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:remove_layer", &plot_name, &nth))
      return NULL;

  change = clear_layer(plot_name, nth);
  
  if(change == WRAP_ERROR)
  { 
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
_move_layer(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble x, y;
  gint change;

  if (!PyArg_ParseTuple(args, "sidd:move_layer", &plot_name, &nth, &x, &y))
      return NULL;

  change = move_layer(plot_name, nth, x, y);

  if(change == WRAP_ERROR) {
      /* PRINT ERROR "No plot with that name" */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_resize_layer(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble x, y;
  gint change;

  if (!PyArg_ParseTuple(args, "sidd:move_layer", &plot_name, &nth, &x, &y))
      return NULL;

  change = resize_layer(plot_name, nth, x, y);

  if(change == WRAP_ERROR)  {
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_show_legends(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:show_legends", &plot_name, &nth))
      return NULL;

  change = show_legends(plot_name, nth);

  if(change == WRAP_ERROR)  {
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_hide_legends(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:hide_legends", &plot_name, &nth))
      return NULL;

  change = hide_legends(plot_name, nth);

  if(change == WRAP_ERROR)  {
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_move_legends(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;
  gdouble x, y;

  if (!PyArg_ParseTuple(args, "sidd:move_legends", &plot_name, &nth, &x, &y))
      return NULL;

  change = move_legends(plot_name, nth, x, y);

  if(change == WRAP_ERROR)  {
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}




static PyObject *
_set_xrange(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble min, max;
  gint change;

  if (!PyArg_ParseTuple(args, "sidd:set_xrange", &plot_name, &nth, &min, &max))
      return NULL;

  change = set_xrange(plot_name, nth, min, max);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_yrange(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble min, max;
  gint change;

  if (!PyArg_ParseTuple(args, "sidd:set_yrange", &plot_name, &nth, &min, &max))
      return NULL;

  change = set_yrange(plot_name, nth, min, max);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_zrange(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble min, max;
  gint change;

  if (!PyArg_ParseTuple(args, "sidd:set_zrange", &plot_name, &nth, &min, &max))
      return NULL;

  change = set_zrange(plot_name, nth, min, max);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_xticks(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble major;
  gint nminor;
  gint change;

  if (!PyArg_ParseTuple(args, "sidi:set_xticks", &plot_name, &nth, &major, &nminor))
      return NULL;

  change = set_xticks(plot_name, nth, major, nminor);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_yticks(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble major;
  gint nminor;
  gint change;

  if (!PyArg_ParseTuple(args, "sidi:set_yticks", &plot_name, &nth, &major, &nminor))
      return NULL;

  change = set_yticks(plot_name, nth, major, nminor);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_zticks(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gdouble major;
  gint nminor;
  gint change;

  if (!PyArg_ParseTuple(args, "sidi:set_zticks", &plot_name, &nth, &major, &nminor))
      return NULL;

  change = set_zticks(plot_name, nth, major, nminor);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_xtitle(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gchar *title;
  gint change;

  if (!PyArg_ParseTuple(args, "sis:set_xtitle", &plot_name, &nth, &title))
      return NULL;

  change = set_xtitle(plot_name, nth, title);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_ytitle(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gchar *title;
  gint change;

  if (!PyArg_ParseTuple(args, "sis:set_ytitle", &plot_name, &nth, &title))
      return NULL;

  change = set_ytitle(plot_name, nth, title);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_ztitle(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gchar *title;
  gint change;

  if (!PyArg_ParseTuple(args, "sis:set_ztitle", &plot_name, &nth, &title))
      return NULL;

  change = set_ztitle(plot_name, nth, title);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_show_xgrids(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:show_xgrids", &plot_name, &nth))
      return NULL;

  change = show_xgrids(plot_name, nth);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_show_ygrids(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:show_ygrids", &plot_name, &nth))
      return NULL;

  change = show_ygrids(plot_name, nth);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_show_zgrids(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:show_zgrids", &plot_name, &nth))
      return NULL;

  change = show_zgrids(plot_name, nth);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_hide_xgrids(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:hide_xgrids", &plot_name, &nth))
      return NULL;

  change = hide_xgrids(plot_name, nth);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_hide_ygrids(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:hide_ygrids", &plot_name, &nth))
      return NULL;

  change = hide_ygrids(plot_name, nth);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_hide_zgrids(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "si:hide_zgrids", &plot_name, &nth))
      return NULL;

  change = hide_zgrids(plot_name, nth);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_symbol(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;
  gint symbol;

  if (!PyArg_ParseTuple(args, "sii:set_symbol", &plot_name, &nth, &symbol))
      return NULL;

  change = set_symbol(plot_name, nth, symbol);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_symbol_style(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;
  gint style;

  if (!PyArg_ParseTuple(args, "sii:set_symbol_style", &plot_name, &nth, &style))
      return NULL;

  change = set_symbol_style(plot_name, nth, style);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_symbol_color(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;
  gchar *color = NULL;

  if (!PyArg_ParseTuple(args, "sis:set_symbol_color", &plot_name, &nth, &color))
      return NULL;

  change = set_symbol_color(plot_name, nth, color);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
_set_line_style(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;
  gint style;

  if (!PyArg_ParseTuple(args, "sii:set_line_style", &plot_name, &nth, &style))
      return NULL;

  change = set_line_style(plot_name, nth, style);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_set_line_color(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;
  gchar *color = NULL;

  if (!PyArg_ParseTuple(args, "sis:set_line_color", &plot_name, &nth, &color))
      return NULL;

  change = set_line_color(plot_name, nth, color);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
_set_connector(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gint change;
  gint connector;

  if (!PyArg_ParseTuple(args, "sii:set_connector", &plot_name, &nth, &connector))
      return NULL;

  change = set_connector(plot_name, nth, connector);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_plot_exp_xy(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gchar *exp_name = NULL;
  gchar *exp1, *exp2;
  gint change;

  if (!PyArg_ParseTuple(args, "siss|s:plot_exp_xy", &plot_name, &nth, &exp1, &exp2,&exp_name))
      return NULL;

  change = plot_exp_xy(plot_name, nth, exp1, exp2, exp_name);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_plot_exp_xy_bars(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gchar *exp_name = NULL;
  gchar *exp1, *exp2;
  gint change;
  gfloat width;

  if (!PyArg_ParseTuple(args, "sissf|s:plot_exp_xy_bars", &plot_name, &nth, &exp1, &exp2, &width, &exp_name))
      return NULL;
  change = plot_exp_xy_bars(plot_name, nth, exp1, exp2, exp_name,SG_STYLE_VBARS,width);

  if(change == WRAP_ERROR){
     /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
_plot_exp(PyObject *self, PyObject *args)
{
  gint nth;
  gchar *plot_name = NULL;
  gchar *exp_name = NULL;
  gchar *exp;
  gint change;

  if (!PyArg_ParseTuple(args, "sis|s:plot_exp", &plot_name, &nth, &exp,&exp_name))
      return NULL;

  change = plot_exp(plot_name, nth, exp,exp_name);

  if(change == WRAP_ERROR){
      /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
_remove_worksheet(PyObject *self, PyObject *args)
{
  gchar *name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "s:remove_worksheet", &name))
        return NULL;

  change = remove_worksheet(name);

  if(change == WRAP_ERROR){
    /* PRINT ERROR "No worksheet with that name" */
  }

  Py_INCREF(Py_None);
  return Py_None;
} 

static PyObject *
_remove_plot(PyObject *self, PyObject *args)
{
  gchar *name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "s:remove_plot", &name))
        return NULL;

  change = remove_plot(name);

  if(change == WRAP_ERROR) {
    /* PRINT ERROR "No plot with that name" */
  }

  Py_INCREF(Py_None);
  return Py_None;
} 

static PyObject *
_clear_plot(PyObject *self, PyObject *args)
{
  gchar *name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "s:remove_plot", &name))
        return NULL;

  change = clear_plot(name);

  if(change == WRAP_ERROR) {
    /* PRINT ERROR "No plot with that name" */
  }

  Py_INCREF(Py_None);
  return Py_None;
} 



static PyObject *
_rename_worksheet(PyObject *self, PyObject *args)
{
  gchar *name = NULL, *old_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "ss:rename_worksheet", &old_name, &name))
        return NULL;

  change = rename_worksheet(old_name, name);

  if(change == WRAP_ERROR){
    /* PRINT ERROR */;
  }

  Py_INCREF(Py_None);
  return Py_None;
} 

static PyObject *
_update_worksheet(PyObject *self, PyObject *args)
{
  gchar *name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "s:update_worksheet", &name))
        return NULL;

  change = update_worksheet(name);

  if(change == WRAP_ERROR){
    /* PRINT ERROR */;
  }

  Py_INCREF(Py_None);
  return Py_None;
} 


static PyObject *
_rename_plot(PyObject *self, PyObject *args)
{
  gchar *name = NULL, *old_name = NULL;
  gint change;

  if (!PyArg_ParseTuple(args, "ss:rename_plot", &old_name, &name))
        return NULL;

  change = rename_plot(old_name, name);

  if(change == WRAP_ERROR){
    /* PRINT ERROR */
  }

  Py_INCREF(Py_None);
  return Py_None;
} 


#ifdef WITH_WARNINGS
#warning What about METH_KEYWORDS when it does not use keywords to Parse?
#endif
static PyMethodDef SGMethods[] =
{
  {"new_worksheet", (PyCFunction)_new_worksheet, METH_VARARGS,
   "Create a new named sheet in Scigraphica.\n\n"
   "Takes the sheet's name as an argument.\n"},
  {"new_plot", (PyCFunction)_new_plot, METH_VARARGS,
   "Create a new names plot in Scigraphica.\n\n"
   "Takes the plot's type (integer) and name as an argument.\n"
   "Type 0 is SG_LAYER_2D, 1 is SG_LAYER_3D, 2 is SG_LAYER_POLAR.\n"},
  {"new_layer", (PyCFunction)_new_layer, METH_VARARGS,
   "Append a new drawing layer to a Scigraphica plot.\n\n"
   "Takes the plot's name (string) and the layer type (integer).\n"
   "Type 0 is SG_LAYER_2D, 1 is SG_LAYER_3D, 2 is SG_LAYER_POLAR.\n"},
  {"remove_worksheet", (PyCFunction)_remove_worksheet, METH_VARARGS,
   "Permanently remove a sheet and its data from scigraphica.\n\n"
   "Takes the sheet's name as an argument.\n"},
  {"remove_plot", (PyCFunction)_remove_plot, METH_VARARGS,
   "Permanently remove a plot from Scigraphica.\n\n"
   "Takes the plot's name as an argument.\n"},
  {"remove_layer", (PyCFunction)_remove_layer, METH_VARARGS,
   "Permanently remove a layer from a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer >=1).\n"},
  {"clear_layer", (PyCFunction)_clear_layer, METH_VARARGS,
   "Unassociates the layer from its dataset in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer >=1).\n"},
  {"move_layer", (PyCFunction)_move_layer, METH_VARARGS,
   "Relocates the position of the layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new position as x (double) and y (double).\n"},
  {"resize_layer", (PyCFunction)_resize_layer, METH_VARARGS,
   "Changes the size of a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new size as x (double) and y (double).\n"},
  {"rename_worksheet", (PyCFunction)_rename_worksheet, METH_VARARGS,
   "Change the name of a sheet in Scigraphica.\n\n"
   "Takes the sheet's current name (string) and the new name (string).\n"},
  {"update_worksheet", (PyCFunction)_update_worksheet, METH_VARARGS,
   "Recalculates formulas in a Scigraphica worksheet.\n\n"
   "Takes the sheet's current name (string).\n"},
  {"rename_plot", (PyCFunction)_rename_plot, METH_VARARGS,
   "Change the name of a plot in Scigraphica.\n\n"
   "Takes the plot's current name (string) and the new name (string).\n"},
  {"clear_plot", (PyCFunction)_clear_plot, METH_VARARGS,
   "Permanently removes all the layers of a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) as an argument.\n"},
  {"show_legends", (PyCFunction)_show_legends, METH_VARARGS,
   "Display a legend for a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1).\n"},
  {"hide_legends", (PyCFunction)_hide_legends, METH_VARARGS,
   "Do not display a legend for a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1).\n"},
  {"move_legends", (PyCFunction)_move_legends, METH_VARARGS,
   "Relocates a legend for a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new position as x (double) and y (double).\n"},
  {"set_xrange", (PyCFunction)_set_xrange, METH_VARARGS,
   "Manually set the x axis range of a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new range as minimum (double) and  maximum (double).\n"},
  {"set_yrange", (PyCFunction)_set_yrange, METH_VARARGS,
   "Manually set the y axis range of a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new range as minimum (double) and  maximum (double).\n"},
  {"set_zrange", (PyCFunction)_set_zrange, METH_VARARGS,
   "Manually set the z axis range of a layer in a 3D plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new range as minimum (double) and  maximum (double).\n"},
  {"set_xticks", (PyCFunction)_set_xticks, METH_VARARGS,
   "Manually set the ticks for the x axis of a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new ticks as the major increment (double) and number of minor ticks (integer).\n"},
  {"set_yticks", (PyCFunction)_set_yticks, METH_VARARGS,
   "Manually set the ticks for the y axis of a layer in a plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new ticks as the major increment (double) and number of minor ticks (integer).\n"},
  {"set_zticks", (PyCFunction)_set_zticks, METH_VARARGS,
   "Manually set the ticks for the z axis of a layer in a 3D plot in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new ticks as the major increment (double) and number of minor ticks (integer).\n"},
  {"set_xtitle", (PyCFunction)_set_xtitle, METH_VARARGS,
   "Set the title of the x axis of a layer in a plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new title (string).\n"},
  {"set_ytitle", (PyCFunction)_set_ytitle, METH_VARARGS,
   "Set the title of the y axis of a layer in a plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new title (string).\n"},
  {"set_ztitle", (PyCFunction)_set_ztitle, METH_VARARGS,
   "Set the title of the z axis of a layer in a 3D plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the new title (string).\n"},
  {"show_xgrids", (PyCFunction)_show_xgrids, METH_VARARGS,
   "Display the grid lines perpendicular to the x axis of a layer in a plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"},
  {"show_ygrids", (PyCFunction)_show_ygrids, METH_VARARGS,
   "Display the grid lines perpendicular to the y axis of a layer in a plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"},
  {"show_zgrids", (PyCFunction)_show_zgrids, METH_VARARGS,
   "Display the grid lines perpendicular to the z axis of a layer in a 3D plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"},
  {"hide_xgrids", (PyCFunction)_hide_xgrids, METH_VARARGS,
   "Do not display the grid lines perpendicular to the x axis of a layer in a plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"},
  {"hide_ygrids", (PyCFunction)_hide_ygrids, METH_VARARGS,
   "Do not display the grid lines perpendicular to the y axis of a layer in a plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"},
  {"hide_zgrids", (PyCFunction)_hide_zgrids, METH_VARARGS,
   "Do not display the grid lines perpendicular to the z axis of a layer in a 3D plot in Scigraphica.\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"},

  {"set_symbol", (PyCFunction)_set_symbol, METH_VARARGS,
   "Set default symbol for plotting in the current layer in Scigraphica.\n"
   "Takes the plot's name (string), the layer index (integer>=1), and the symbol(integer<10)\n"},
  {"set_symbol_style", (PyCFunction)_set_symbol_style, METH_VARARGS,
   "Set default symbol style for plotting in the current layer in Scigraphica.\n"
   "Takes the plot's name (string), the layer index (integer>=1), and the symbol style(integer<3)\n"},
  {"set_symbol_color", (PyCFunction)_set_symbol_color, METH_VARARGS,
   "Set default symbol color for plotting in the current layer in Scigraphica.\n"
   "Takes the plot's name (string), the layer index (integer>=1), and the color string\n"},
  {"set_line_style", (PyCFunction)_set_line_style, METH_VARARGS,
   "Set default line style for plotting in the current layer in Scigraphica.\n"
   "Takes the plot's name (string), the layer index (integer>=1), and the line style(integer<7)\n"},
  {"set_line_color", (PyCFunction)_set_line_color, METH_VARARGS,
   "Set default line color for plotting in the current layer in Scigraphica.\n"
   "Takes the plot's name (string), the layer index (integer>=1), and the color string\n"},
  {"set_connector", (PyCFunction)_set_connector, METH_VARARGS,
   "Set default connector for plotting in the current layer in Scigraphica.\n"
   "Takes the plot's name (string), the layer index (integer>=1), and the connector(integer<6)\n"},
  {"plot_exp_xy", (PyCFunction)_plot_exp_xy, METH_VARARGS,
   "Plots an XY plot in a layer in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the expressions (string) for the 'x' and 'y' variables.\n"},
  {"plot_exp_xy_bars", (PyCFunction)_plot_exp_xy_bars, METH_VARARGS,
   "Plot vertical bars in a layer in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the expressions (string) for the 'x' and 'y' variables.\n"},
   {"plot_exp", (PyCFunction)_plot_exp, METH_VARARGS,
   "Plots a python expression in a layer in Scigraphica.\n\n"
   "Takes the plot's name (string) and the layer index (integer>=1)\n"
   "and the expression (string) which has 'x' as the independent variable.\n"},
  {"read_file", (PyCFunction)_read_file, METH_VARARGS,
   "Takes the name (string) and path (string).\n\n"
   "XXX This looks broken at the moment.\n"},
  {"set_cell_value", (PyCFunction)_set_cell_value, METH_VARARGS,
   "Takes the worksheet name, column (string), row (int) and value (double).\n\n"
   "XXX This should take col as a string or int (get_worksheet_col).\n"},
  {"set_cell_text", (PyCFunction)_set_cell_text, METH_VARARGS,
   "Takes the worksheet name, column (string), row (int) and value (string).\n\n"},
  {"get_cell_value", (PyCFunction)_get_cell_value, METH_VARARGS,
   "Takes the worksheet name, column (string), row (int) returns the value (double).\n\n"},
  {"get_cell_text", (PyCFunction)_get_cell_text, METH_VARARGS,
   "Takes the worksheet name, column (string), row (int) returns the value (string).\n\n"},
  {"set_column_name", (PyCFunction)_set_column_name, METH_VARARGS,
   "Takes the worksheet name, column (string) and the new column name (string).\n\n"},
  {"set_column_values", (PyCFunction)_set_column_values, METH_VARARGS,
   "Takes the worksheet name, column (string), expression (string, of 'x'),\n"
   "and from (int) and to (int) row numbers.\n"},
  {"set_column_type", (PyCFunction)_set_column_type, METH_VARARGS,
   "Takes the worksheet name, column (string), and type (integer).\n"
   "Type is 0=SG_TYPE_NUMBER, 1=SG_TYPE_TEXT, 2=SG_TYPE_DATE, 3=SG_TYPE_TIME.\n"},
  {"set_column_numbers", (PyCFunction)_set_column_numbers, METH_VARARGS,
   "Takes the worksheet name, column (string), internal (int), format (int), and precision (int).\n"},
  {"cell",  (PyCFunction)get_cell_double, METH_VARARGS|METH_KEYWORDS,
   "Return the content (number or string) of a cell in a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'col'   (as integer or as label name string)\n"
   "                   'sheet' (string, will default to the current sheet)\n"},
  {"col",  (PyCFunction)get_col_double_array, METH_VARARGS|METH_KEYWORDS,
   "Return a PyArray (or None) of the column of the sheet in Scigraphica.\n\n"
   "Only the numeric values are returned, with no gaps. So indices of the\n"
   "returned array may not correspond to the original Scigraphica column.\n"
   "(I hope this changes to using masked arrays - chrisk@mit.edu)\n"
   "Takes the keywords 'col'   (as integer or as label name string)\n"
   "                   'sheet' (string, will default to the current sheet)\n"},
  {"row",  (PyCFunction)get_row_double_array, METH_VARARGS|METH_KEYWORDS,
   "Return a PyArray (or None) of the row of the sheet in Scigraphica.\n\n"
   "Only the numeric values are returned, with no gaps. So indices of the\n"
   "returned array may not correspond to the original Scigraphica row.\n"
   "(I hope this changes to using masked arrays - chrisk@mit.edu)\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'sheet' (string, will default to 'current')\n"},
  {"set_cell",  (PyCFunction)set_cell, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,col).\n"},
  {"set_column",  (PyCFunction)set_col, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to 'current')\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (0,col).\n"},
  {"clear_column",  (PyCFunction)clear_col, METH_VARARGS|METH_KEYWORDS,
   "Clears the contents of a column in sheet.\n\n"
   "Takes the keywords 'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to 'current')\n"},

   {"set_row",  (PyCFunction)set_row, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet')\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,0).\n"},
  {"setcell",  (PyCFunction)set_cell, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,col).\n"},
  {"setcol",  (PyCFunction)set_col, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (0,col).\n"},
  {"setrow",  (PyCFunction)set_row, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,0).\n"},
  {"read_block", (PyCFunction)python_read_table_block, METH_VARARGS|METH_KEYWORDS,
   "Return an array with data from the given block in a file (Scigraphica).\n\n"
   "Takes the keywords 'filename'   (string)\n"
   "                   'block'      (integer>=1)\n"
   "                   'comment'    (string, defaults to '#')\n"
   "                   'delimiter'  (string, defaults to ' \\t\\n')\n"
   "                   'blockstart' (string, defaults to '#')\n"},
  {"read_lines", (PyCFunction)python_read_table_lines, METH_VARARGS|METH_KEYWORDS,
   "Return an array with data from a given range of lines in a file (Scigraphica).\n\n"
   "Takes the keywords 'filename'  (string)\n"
   "                   'beginline' (integer>=1)\n"
   "                   'endline'   (integer, may be zero to mean read until EOF)\n"
   "                   'comment'   (string, defaults to '#')\n"
   "                   'delimiter' (string, defaults to ' \\t\\n')\n"},
  {"read_block_string", (PyCFunction)python_read_table_block_string, METH_VARARGS|METH_KEYWORDS,
   "Return a list of list of string from the given block in a file (Scigraphica).\n\n"
   "Takes the keywords 'filename'   (string)\n"
   "                   'block'      (integer>=1)\n"
   "                   'comment'    (string, defaults to '#')\n"
   "                   'delimiter'  (string, defaults to ' \\t\\n')\n"
   "                   'blockstart' (string, defaults to '#')\n"},
  {"read_lines_string", (PyCFunction)python_read_table_lines_string, METH_VARARGS|METH_KEYWORDS,
   "Return a list of list of string from the given block in a file (Scigraphica).\n\n"
   "Takes the keywords 'filename'  (string)\n"
   "                   'beginline' (integer>=1)\n"
   "                   'endline'   (integer, may be zero to mean read until EOF)\n"
   "                   'comment'   (string, defaults to '#')\n"
   "                   'delimiter' (string, defaults to ' \\t\\n')\n"},
  {"register_plugin", (PyCFunction)fregister, METH_VARARGS,
   "Register a new plugin.\n\n"
   "Takes the keywords 'name'      (string)\n"
   "                   'group'     (string) Colon separated group path\n"
   "                   'endline'   (string) Name of the function to be called\n"},
  {"new_plugin_group", (PyCFunction)new_plugin_group, METH_VARARGS,"Create a new plugin group.\n\n"
   "Takes the keywords 'name'      (string) Name of the group, which is a colon separated path\n"},
  {"get_selection", (PyCFunction)get_selection, METH_VARARGS,"Create a new plugin group.\n\n"
   "Takes the keywords 'name'      (string) Name of the group, which is a colon separated path\n"},
/*   {"__import__", sg_import, METH_VARARGS,
   "Calls PyImport_ImportModuleEx (Scigraphica).\n\n"
   "Takes a module name (string) and optionally globals, locals, and fromlist (objects).\n"},*/
  {"get_help_text", (PyCFunction)_describe_all,METH_VARARGS,
  "Returns a multi-line string listing all the exported scrigraphica functions  their documentation strings.\n"},
  {"active_worksheet_name", (PyCFunction)_get_active_worksheet_name,METH_VARARGS,
   "Reurns the name of the active worksheet.\n"},
  {"worksheet_names", (PyCFunction)_get_worksheet_names,METH_VARARGS,
   "Returns the list of worksheet names in this project.\n"},
  {"column_names", (PyCFunction)_get_column_names,METH_VARARGS|METH_KEYWORDS,
   "Reurns the list of column names in a worksheet.\n"},
  {"add_columns", (PyCFunction)_worksheet_add_columns,METH_VARARGS|METH_KEYWORDS,
  "Add the number of columns to the current sheet, or the sheet specified in the "
  "'sheet' argument.\n"},
  {"get_column_exp", (PyCFunction)_worksheet_get_column_exp,METH_VARARGS|METH_KEYWORDS,
  "Returns the expression of the named (or numbered) column specified in the current sheet,"
   "or the sheet specified in the 'sheet' argument.\n"},
  {"get_column_precision", (PyCFunction)_worksheet_get_column_precision,METH_VARARGS|METH_KEYWORDS,
  "Returns the precision of the named (or numbered) column specified in the current sheet,"
   "or the sheet specified in the 'sheet' argument.\n"},
  {"get_column_precision", (PyCFunction)_worksheet_get_column_precision,METH_VARARGS|METH_KEYWORDS,
  "Returns the precision of the named (or numbered) column specified in the current sheet,"
   "or the sheet specified in the 'sheet' argument.\n"},
  {"config_register", (PyCFunction)_config_register,METH_VARARGS|METH_KEYWORDS,
  "Registers a configuration variable. Requires four compulsory aruments:\n"
  "name: the name of the variable.\n"
  "group: the group the place the variable in.\n"
  "set_default: A callable object which sets the default value of the variable.\n"
  "commit_value: A callable object which registers side-effects implied by a change in the variable.\n"},

  {"config_get_value", (PyCFunction)_config_get_value,METH_VARARGS|METH_KEYWORDS,
  "Returns the configuration variable stored in:\n"
  "name: the name of the variable, and\n"
  "group: the group of the variable.\n"},
  {"config_set_value", (PyCFunction)_config_set_value,METH_VARARGS|METH_KEYWORDS,
  "Returns the configuration variable stored in:\n"
  "name: the name of the variable, and\n"
  "group: the group of the variable.\n"},

  {"config_exec_default", (PyCFunction)_config_exec_default,METH_VARARGS|METH_KEYWORDS,
  "Executes the function to set the default of the configuration variable stored in:\n"
  "name: the name of the variable, and\n"
  "group: the group of the variable.\n"},
  {"config_exec_commit", (PyCFunction)_config_exec_commit,METH_VARARGS|METH_KEYWORDS,
  "Executes the function to commit of the configuration variable stored in:\n"
  "name: the name of the variable, and\n"
  "group: the group of the variable.\n"},

  {"plot_list", (PyCFunction)_plot_list,METH_VARARGS,
  "Returns a list of plot names in the current project"},
  {"active_plot", (PyCFunction)_plot_active,METH_VARARGS,
  "Returns the name of the currently active plot"},

  {"plot_layers", (PyCFunction)_plot_layers,METH_VARARGS,
  "Returns the number of layers in the named plot. If no name is given,\n"
  "The currently selected plot is assumed."},
  {"plot_active_layer", (PyCFunction)_plot_active_layer,METH_VARARGS,
  "Returns the number of the active layer in the named plot. If no name is given\n"
  "The currently selected plot is assumed."},
  {"plot_layer_dataset_list", (PyCFunction)_plot_layer_dataset_list,METH_VARARGS,
  "Returns a list of the datasets in the specified layer, in the named plot. If no name or layer number is given\n"
  "The active layer in the currently selected plot is assumed."},
  {"plot_layer_dataset", (PyCFunction)_plot_layer_dataset,METH_VARARGS,
  "Returns a structure of the dataset's elements"},
  {"plot_layer_refresh", (PyCFunction)_plot_layer_refresh,METH_VARARGS,
  "Refreshes the datasets in the specified layer"},
  {"plot_layer_autoscale", (PyCFunction)_plot_layer_autoscale,METH_VARARGS,
  "Autoscales the specified layer"},
  {"plot_refresh_canvas", (PyCFunction)_plot_refresh_canvas,METH_VARARGS,
  "Redraws the canvas"},
  {"plot_freeze_canvas", (PyCFunction)_plot_freeze_canvas,METH_VARARGS,
  "Freezes the canvas"},
  {"plot_thaw_canvas", (PyCFunction)_plot_thaw_canvas,METH_VARARGS,
  "Thaws the canvas"},

  
  
  {"set_axis_text_properties", (PyCFunction)axis_text_properties,METH_VARARGS|METH_KEYWORDS,
   "Sets the following axis properties:\n"
   "show_labels=[TRUE|FALSE]\n"
   "show_titles=[TRUE|FALSE]\n"
   "titlename=[STRING]\n"
   "The method takes three compulsory arguments:\n"
   "plot_name=[STRING]\n"
   "layer_num=[INTEGER]\n"
   "axis_num =[0|1|2|3] for bottom, left, top, right axes"},
  {"set_axis_label_pos", (PyCFunction)set_axis_label_pos,METH_VARARGS|METH_KEYWORDS,
   "The method takes three compulsory arguments:\n"
   "plot_name=[STRING]\n"
   "layer_num=[INTEGER]\n"
   "axis_num =[0|1|2|3] for bottom, left, top, right axes"
    "Sets the following axis properties:\n"
   "offset Offset of the title from the axis in normalized page coordinates\n"
   "justify_axis A number between 0 and 1 indicating the position of a reference point"
   "             along the axis\n"
   "justify The position of this reference point along the length of the string:"
   "        0=left\n"
   "        1=right\n"
   "        2=center[default]\n"},
  {NULL, NULL, 0,NULL}        /* Sentinel */
};

void init_sg_python()
{ 
  (void) Py_InitModule("sg", SGMethods);
}

gint 
get_worksheet_col(SGworksheet *worksheet,PyObject *col_obj)
{ 
  gint col,num;
  gchar *col_name;

  if (PyString_Check(col_obj))
  { 
    num=gtk_sheet_get_columns_count(GTK_SHEET(worksheet->sheet));
    col_name=PyString_AsString(col_obj);
    for (col=0;col<num;col++)
        if (GTK_SHEET(worksheet->sheet)->column[col].name)
        { 
           if (!strcmp(col_name,GTK_SHEET(worksheet->sheet)->column[col].name))
               break;
        }
        else
        { if (col==atoi(col_name));
            return col;
        }
  }
  else if (PyInt_Check(col_obj)){
      col=(int)PyInt_AsLong(col_obj);
      printf("col_num 1 =%d\n",col);
  }

  /* Now check if row and col values are sane */
  if (col<0 || col>gtk_sheet_get_columns_count(GTK_SHEET(worksheet->sheet)))
  { 
    PyErr_SetString(PyExc_ValueError,"Column number out of range");
    return -1;
  }
  return(col);
}

gint get_worksheet_row(SGworksheet *worksheet,PyObject *row_obj)
{ 
  gint row,num;
  gchar *row_name;

  if (PyString_Check(row_obj))
      { 
        num=gtk_sheet_get_rows_count(GTK_SHEET(worksheet->sheet));
	row_name=PyString_AsString(row_obj);
	  { 
            for (row=0;row<num;row++)
             if (GTK_SHEET(worksheet->sheet)->row[row].name)
	      { if (!strcmp(row_name,GTK_SHEET(worksheet->sheet)->row[row].name))
	         break;
	      }
	     else
	      { if (row==atoi(row_name));
		return(row);
	      }
	  }
      }
  else if (PyInt_Check(row_obj))
      row=(int)PyInt_AsLong(row_obj);

  /* Now check if row and col values are sane */
  if (row<0 || row>gtk_sheet_get_rows_count(GTK_SHEET(worksheet->sheet)))
  { 
    PyErr_SetString(PyExc_ValueError,"Row number out of range");
    return -1;
  }
  return(row);
}

static PyObject *
get_cell_object(SGworksheet *worksheet, gint row, gint col)
{ 
  gdouble value;
  PyObject *object=NULL;
  gchar *sheet=NULL,*content;
  SGhiddencell *link;

/* Get sheet pointer */
  worksheet= sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet->sheet),
                                            row, col);
  if(link)
  switch(link->type){
      case SG_TYPE_NUMBER:
      switch (link->internal){
          case SG_INTERNAL_INTEGER: object=Py_BuildValue("l", link->value.val_long);
          case SG_INTERNAL_DOUBLE:  object=Py_BuildValue("f", link->value.val_double);
          default: object=Py_BuildValue("f",0.0);
      }
      case SG_TYPE_TEXT:        
      case SG_TYPE_TIME:        
      case SG_TYPE_DATE:        
      default:
          content=sg_worksheet_cell_get_formula(worksheet,row,col);
          if (!content) content=sg_worksheet_cell_get_text(worksheet,row,col);
          if (content)
              object=Py_BuildValue("s",sg_worksheet_cell_get_formula(worksheet,row,col));

  }


  if (!object)
      object=Py_None;

  Py_INCREF(object);
  return object;
}


/* Cell/Col/Row query functions */
#ifdef WITH_WARNINGS
#warning TODO take keyword logic and put in _get_cell_text and _get_cell_calue
#warning and then remove this function. NO! Please do not! - CDS
#endif
static PyObject *
get_cell(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object=NULL,*row_obj,*col_obj;
  gboolean restore_active=FALSE;
  gchar *sheet=NULL,*content;
  gint row=0,col=0,num,arow=-1,acol=-1;
  static gchar *kwlist[]={"col","row","sheet",NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &col_obj, &row_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet= sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  /* Now check if row and col values are sane */
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0 || row >GTK_SHEET(worksheet->sheet)->maxallocrow) return NULL;
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0 || col >GTK_SHEET(worksheet->sheet)->maxalloccol) return NULL;

  return get_cell_object(worksheet,row,col);

}

static PyObject *
get_cell_double(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object=NULL,*row_obj,*col_obj;
  gboolean error=FALSE;
  gchar *sheet=NULL,*content;
  gint row=0,col=0,num,arow=-1,acol=-1;
  static gchar *kwlist[]={"col","row","sheet",NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &col_obj, &row_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet= sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  /* Now check if row and col values are sane */
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0 || row >GTK_SHEET(worksheet->sheet)->maxallocrow) return NULL;
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0 || col >GTK_SHEET(worksheet->sheet)->maxalloccol) return NULL;

  object=Py_BuildValue("f", sg_worksheet_cell_get_double(worksheet, row, col,&error));

  if (!object || error)
      object=Py_None;

  Py_INCREF(object);
  return object;

}


#ifdef WITH_WARNINGS
#warning I would consider silently deleting non numeric values a bug
#warning Why not use the Masked Array capabilities of Numeric Python?
#warning This is exactly what they were designed to handle - chrisk@mit.edu
#endif
static PyObject *
get_col_double_array(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet=NULL,*content;
  static gchar *kwlist[]={"col","sheet",NULL},*col_name;
  gint row=0,num,col=0,nd=1,dims[1],i,j=0,arow=-1,acol=-1;
  SGworksheet *worksheet=NULL;
  gdouble value,*data,*rdata;
  PyObject *object,*col_obj;
  gboolean error=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist,
                                   &col_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  col=get_worksheet_col(worksheet,col_obj);
  if (col<0 || col >GTK_SHEET(worksheet->sheet)->maxalloccol) {
     PyErr_SetString(PyExc_ValueError,"No such column");
     return NULL;
}

/* See how many elements we need */
  data=g_new(gdouble,GTK_SHEET(worksheet->sheet)->maxallocrow+1);
  for (i=0;i<=GTK_SHEET(worksheet->sheet)->maxallocrow;i++){
      value=sg_worksheet_cell_get_double(worksheet, i, col,&error);
      if (error) break;
      data[j++]=value;
  }
  if (i==j)
      rdata=data;
  else
      rdata=g_renew(gdouble,data,j);


  if (j>0)
  {
      dims[0]=j;
#ifdef WITH_WARNINGS
#warning rdata is never freed. Says who? Go do your homework - CDS
#endif
      object=PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,(gchar *)rdata);
  }
  else{
      object=Py_None;
      g_free(rdata);
  }


  Py_INCREF(object);
  return object;
}

#ifdef WITH_WARNINGS
#warning I would consider silently deleting non numeric values a bug
#warning Why not use the Masked Array capabilities of Numeric Python?
#warning This is exactly what they were designed to handle - chrisk@mit.edu
#endif
static PyObject *
get_row_double_array(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet=NULL,*content;
  static gchar *kwlist[]={"row","sheet",NULL};
  gint row=0,arow=-1,acol=-1,num,nd=1,dims[1],i,j=0;
  SGworksheet *worksheet=NULL;
  gdouble value,*data,*rdata;
  PyObject *object, *row_obj;
  gboolean error=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist,
                                   &row_obj, &sheet))
      return NULL;
  /* Get sheet pointer */
  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  /* Now check if row and col values are sane */
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0 || row >GTK_SHEET(worksheet->sheet)->maxallocrow) {
     PyErr_SetString(PyExc_ValueError,"No such row");
     return NULL;
}


/* See how many elements we need */
  data=g_new(gdouble,GTK_SHEET(worksheet->sheet)->maxalloccol+1);
  for (i=0;i<=GTK_SHEET(worksheet->sheet)->maxalloccol;i++){
      value=sg_worksheet_cell_get_double(worksheet, row, i,&error);
      if (error) break;
      data[j++]=value;
  }

  if (i==j) 
      rdata=data;
  else
  { 
      rdata=g_renew(gdouble,data,j);
  }

  if (j>0)
   {
       dims[0]=j;
       object=PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,(gchar *)rdata);
   }
  else{
      object=Py_None;
      g_free(rdata);
  }


  Py_INCREF(object);
  return object;
}

static PyObject *
get_col_list(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet=NULL,*content;
  static gchar *kwlist[]={"col","sheet",NULL},*col_name;
  gint row=0,num,col=0,nd=1,dims[1],i,j=0,arow=-1,acol=-1;
  SGworksheet *worksheet=NULL;
  gdouble value,*data,*rdata;
  PyObject *object,*col_obj,*list;
  gboolean restore_active=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist,
                                   &col_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);


  col=get_worksheet_col(worksheet,col_obj);
  if (col<0|| col >GTK_SHEET(worksheet->sheet)->maxalloccol) return NULL;


/* See how many elements we need */
  list=PyList_New(0);
  for (i=0;i<=GTK_SHEET(worksheet->sheet)->maxallocrow;i++)
    PyList_Append(list,get_cell_object(worksheet,i,col));

  Py_INCREF(list);
  return list;
}


static PyObject *
get_row_list(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet="current",*content;
  static gchar *kwlist[]={"row","sheet",NULL};
  gint row=0,arow=-1,acol=-1,num,nd=1,dims[1],i,j=0;
  SGworksheet *worksheet=NULL;
  gdouble value,*data,*rdata;
  PyObject *object, *row_obj, *list;
  gboolean restore_active=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist,
                                   &row_obj, &sheet))
      return NULL;
  /* Get sheet pointer */
  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  /* Now check if row and col values are sane */
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0 || row >GTK_SHEET(worksheet->sheet)->maxallocrow) return NULL;

/* See how many elements we need */
  list=PyList_New(0);
  for (i=0;i<=GTK_SHEET(worksheet->sheet)->maxalloccol;i++)
      PyList_Append(list,get_cell_object(worksheet,row,i));

  Py_INCREF(list);
  return list;
}



/* Cell/Col/Row setting functions */

gint row_check(gint row,GtkSheet *sheet)
{
 if (row>gtk_sheet_get_rows_count(sheet))
  { 
    PyErr_SetString(PyExc_ValueError,"Row number out of range");
    return FALSE;
  }

 return TRUE;
}

gint col_check(gint col,GtkSheet *sheet)
{
 if (col>gtk_sheet_get_columns_count(sheet))
  { 
    PyErr_SetString(PyExc_ValueError,"Column number out of range");
    return FALSE;
  }

 return TRUE;
}

PyObject *
set_sheet(gint row, gint col, PyObject *object, SGworksheet *worksheet,
          GtkOrientation orient)
{ 
  gint retval,acol,arow;
  gboolean restore_active=FALSE;

  /* Now check if row and col values are sane */

      if (PyArray_Check(object))
       { 
         gtk_sheet_freeze(GTK_SHEET(worksheet->sheet));
         retval=python_array(worksheet,row,col,(PyArrayObject *)object,orient,FALSE);
         gtk_sheet_thaw(GTK_SHEET(worksheet->sheet));
       }
      else
      if (PySequence_Check(object))
       { 
         gtk_sheet_freeze(GTK_SHEET(worksheet->sheet));
         retval=python_sequence(worksheet,row,col,object,orient,FALSE,FALSE);
         gtk_sheet_thaw(GTK_SHEET(worksheet->sheet));
       }
      else
        retval=python_singleton(worksheet,row,col,object,FALSE,FALSE);


      if (!retval)
      { 
        if (PyErr_Occurred())
          {  
              create_python_term(NULL,NULL);
              PyErr_Print();
              if (Py_FlushLine()) PyErr_Clear();
              PyErr_Clear();
          }
        return NULL;
      }
      else
      {   
          Py_INCREF(Py_None);
          return Py_None;
      }
}

static PyObject *
set_cell(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet=NULL, *content;
  static gchar *kwlist[]={"col","row","object","sheet",NULL};
  gint row=0,col=0,num,arow=-1,acol=-1,retval;
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object,*row_obj,*col_obj;
  gboolean restore_active=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OOO|s", kwlist,
                                   &col_obj, &row_obj, &object, &sheet))
    return NULL;
  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  row=get_worksheet_row(worksheet,row_obj);
  if (row<0) return NULL;
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0) return NULL;

  return set_sheet(row,col,object,worksheet,GTK_ORIENTATION_VERTICAL);
}

static PyObject *
set_col(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet=NULL, *content;
  static gchar *kwlist[]={"col","object","sheet",NULL};
  gint row=0,col=0,num,arow=-1,acol=-1,retval;
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object,*col_obj;
  gboolean restore_active=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &col_obj, &object, &sheet))
      return NULL;
  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0) return NULL;

  return set_sheet(0,col,object,worksheet,GTK_ORIENTATION_VERTICAL);
}

static PyObject *
clear_col(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet=NULL, *content;
  static gchar *kwlist[]={"col","sheet",NULL};
  gint row=0,col=0,num,arow=-1,acol=-1,retval,j;
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object,*col_obj;
  gboolean restore_active=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist,
                                   &col_obj, &sheet))
      return NULL;
  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0) return NULL;
  row=gtk_sheet_get_rows_count(GTK_SHEET(worksheet->sheet));
  for (j=0;j<row;j++)
    sg_worksheet_cell_clear(worksheet, j, col);
  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *
set_row(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  gchar *sheet=NULL, *content;
  static gchar *kwlist[]={"row","object","sheet",NULL};
  gint row=0,col=0,num,arow=-1,acol=-1,retval;
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object,*row_obj;
  gboolean restore_active=FALSE;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &row_obj, &object, &sheet))
      return NULL;

  worksheet=sg_project_get_worksheet(sheet);
  if(!worksheet) return no_such_worksheet(sheet);

  row=get_worksheet_row(worksheet,row_obj);
  if (row<0) return NULL;

  return set_sheet(row,0,object,worksheet,GTK_ORIENTATION_HORIZONTAL);
}

int python_singleton(SGworksheet *worksheet, gint row, gint col,
                     PyObject *object, gboolean link, gboolean as_is)
{ 
  static PyObject *s_obj;
  GtkSheet *sheet;
  gchar *string = NULL;
  gchar fpnum[40],pspec[20];
  gint freemem=FALSE;
  SGhiddencell *hidden;
  gdouble value;

  sheet = GTK_SHEET(worksheet->sheet);
  hidden = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);
  if (!hidden)
  {
      hidden = g_new(SGhiddencell, 1);
      hidden->formula = NULL;
      hidden->updated = FALSE;
      hidden->type =  worksheet->column[col].type;
      hidden->format =  worksheet->column[col].format;
      hidden->internal =  worksheet->column[col].internal;
      hidden->precision = worksheet->column[col].precision;
      gtk_sheet_link_cell(sheet, row, col, hidden);
  }

  fpnum[0]='\0';
  pspec[0]='\0';

  if (as_is)
   s_obj=object;
  else
   s_obj=PyObject_Str(object);

  if (s_obj)
   { 
     Py_INCREF(s_obj);
     if (object==Py_None)
     {   
         string=strdup("");
     }
     else
      switch(hidden->type){
        case SG_TYPE_NUMBER:       
          if (PyFloat_Check(object) || PyLong_Check(object) || PyInt_Check(object)){
             switch(hidden->internal){
               case SG_INTERNAL_DOUBLE:
                   if(PyFloat_Check(object))
                     hidden->value.val_double=PyFloat_AsDouble (object);
                   else if(PyLong_Check(object))
                     hidden->value.val_double=PyLong_AsDouble (object);
                   else if(PyInt_Check(object))
                     hidden->value.val_double=(gdouble)PyInt_AsLong (object);
                   break;
               case SG_INTERNAL_INTEGER:
                   if(PyFloat_Check(object))
                     hidden->value.val_long=(glong)PyFloat_AsDouble (object);
                   else if(PyLong_Check(object))
                     hidden->value.val_long=PyLong_AsLong (object);
                   else if(PyInt_Check(object))
                     hidden->value.val_long=PyInt_AsLong (object);
                   break;
             }
          } else {
             string=PyString_AsString (s_obj);
             switch(hidden->internal){
               case SG_INTERNAL_DOUBLE:
                   hidden->value.val_double=atof(string);
                   break;
               case SG_INTERNAL_INTEGER:
                   hidden->value.val_long=atoi(string);
                   break;
             }
          }
          break;
        case SG_TYPE_TEXT:        
        case SG_TYPE_TIME:        
        case SG_TYPE_DATE:        
        default:
          string=PyString_AsString (s_obj);
          hidden->value.val_char=g_strdup(string);
      }
      sg_worksheet_cell_update_format(worksheet,row, col);
  }
  Py_XDECREF(s_obj);

  if(hidden->formula)
    g_free(hidden->formula);
    
  if(string)
    hidden->formula = g_strdup(string);
  else 
    hidden->formula = NULL;
  
  return 1;
}

int python_sequence(SGworksheet *worksheet,gint row, gint col, PyObject *object,
                    GtkOrientation orient,gboolean link,gboolean as_is)
{ 
  gint len,i;
  PyObject *seq_obj;
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet->sheet);

  

  /* Check to see if we need to resize the sheet */
  if (PyString_Check(object)||!PySequence_Check(object))
      python_singleton(worksheet,row,col,object,link,as_is);
  else
  {
     len=PySequence_Length(object);
     seq_obj=PySequence_GetItem(object,0);

     if (orient==GTK_ORIENTATION_VERTICAL && gtk_sheet_get_rows_count(sheet) < row+len)
              sg_worksheet_add_rows(worksheet,len-gtk_sheet_get_rows_count(sheet)+row);
     else if (orient==GTK_ORIENTATION_HORIZONTAL && gtk_sheet_get_columns_count(sheet) < col+len)
               sg_worksheet_add_columns(worksheet,len-gtk_sheet_get_columns_count(sheet)+col);
              
      for (i=0;i<len;i++)
       { 
         seq_obj=PySequence_GetItem(object,i);
         Py_INCREF(seq_obj);
         if (PySequence_Check(seq_obj) && !PyString_Check(seq_obj) )
         {  
            if (orient==GTK_ORIENTATION_VERTICAL)
                python_sequence(worksheet,row+i,col,seq_obj,GTK_ORIENTATION_HORIZONTAL,link,as_is);
            else
                python_sequence(worksheet,row,col+i,seq_obj,GTK_ORIENTATION_VERTICAL,link,as_is);
         }
         else
         { 
           if (orient==GTK_ORIENTATION_VERTICAL)
             python_singleton(worksheet,row+i,col,seq_obj,link,as_is);
           else
             python_singleton(worksheet,row,col+i,seq_obj,link,as_is);
         }
         Py_XDECREF(seq_obj);
       }
  }
  return 1;
}

int 
python_array_print(SGworksheet *worksheet,gint row, gint col,
                   PyArrayObject *object,gint dim, gchar *data,
                   GtkOrientation orient,gboolean link)
{ 
  gchar *array;
  gchar singleton[80];
  gchar CHAR[]="%%c",
        UBYTE[]="%%hi",
        SHORT[]="%%hd",
        INT[]="%%d",
        LONG[]="%%ld",
        FLOAT[]="%%1.%df",
        CFLOAT[]="%%Lg",
        UFLOAT[]="%%f";
  gchar format[20],*fpt,*apt;
  gint stride,i,j,vert,rowskip,curr_col;
  PyObject *single_object=NULL;
  
  array=data;
  if (dim>1)
   { 
     rowskip=1;
     for (i=dim;i>0;i--)
       rowskip*=object->dimensions[i];
     stride=object->strides[dim];
     for (i=0;i<object->dimensions[i];i++)
       { 
         python_array_print(worksheet,row+i*rowskip,col,object,dim-1,array,orient,link);
         array+=stride;
       }
   }
  else
    { if (dim>0)
       { vert=object->dimensions[1];
         stride=object->strides[1];
       }
      else
       { vert=1;
         stride=0;
       }

    /* Now insert the cell values */
      array=data;

      for (j=0;j<vert;j++)
        { 
          for (i=0;i<object->dimensions[0];i++)
            { 
              apt=array+i*object->strides[0];
              if (orient==GTK_ORIENTATION_VERTICAL)
                  curr_col=col+j;
              else
                  curr_col=col+i;
              switch (object->descr->type_num)
               { 
                 case PyArray_CHAR:
                 case PyArray_UBYTE:
                 case PyArray_SBYTE:
                 case PyArray_SHORT:
                 case PyArray_INT:
                 case PyArray_LONG:
                     single_object=PyLong_FromLong (*(long*) (array+i*object->strides[0]));
                     Py_INCREF(single_object);
                     break;
                 case PyArray_FLOAT:
                 case PyArray_DOUBLE:
                 case PyArray_CFLOAT:
                 case PyArray_CDOUBLE:
                     single_object=PyFloat_FromDouble (*(double*) (array+i*object->strides[0]));
                     Py_INCREF(single_object);
                     break;
                 case PyArray_OBJECT:
                  python_sequence(worksheet,row+j,col+i,
                                  (PyObject *)(array+i*object->strides[0]),orient,link,FALSE);
                  break;
               }
               if (object->descr->type_num!=PyArray_OBJECT)
               { 
                 python_singleton(worksheet,row+i,col+j,
                                  single_object,link,FALSE);
                 Py_XDECREF(single_object);
               }
          }
          array+=stride;
        }
    }
  return 1;
}

int 
python_array(SGworksheet *worksheet,gint row, gint col,
             PyArrayObject *object, GtkOrientation orient,gboolean link)
{ 
  gint len,i;
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet->sheet);
  if (object->nd>2) return 0; /* No nore than 2 dimensions for now */

  /* Check to see if we need to resize the sheet */
  if (orient==GTK_ORIENTATION_VERTICAL)
  {
      if (object->nd>1 && 
          gtk_sheet_get_columns_count(sheet)< object->dimensions[1]+col+1)
         sg_worksheet_add_columns(worksheet,object->dimensions[1]-
                gtk_sheet_get_columns_count(sheet)+col);
      len=1;
      for (i=0;i<object->nd;i++)
        if (i!=1) len=len*object->dimensions[i];
    
      if (gtk_sheet_get_rows_count(sheet) < row+len)
          sg_worksheet_add_rows(worksheet,len-gtk_sheet_get_rows_count(sheet)+row);
  }
  else
  {
      if (object->nd>1 && 
          gtk_sheet_get_rows_count(sheet)< object->dimensions[1]+row+1)
         sg_worksheet_add_rows(worksheet,object->dimensions[1]-
                gtk_sheet_get_rows_count(sheet)+row);
      len=1;
      for (i=0;i<object->nd;i++)
        if (i!=1) len=len*object->dimensions[i];
    
      if (gtk_sheet_get_columns_count(sheet) < col+len)
          sg_worksheet_add_columns(worksheet,len-gtk_sheet_get_columns_count(sheet)+col);

  }

  python_array_print(worksheet,row,col,object,object->nd-1,object->data,
                     orient,link);

  return 1;
}


int 
python_insert_object(SGworksheet *worksheet,gint row, gint col,
                     PyObject *object,GtkOrientation orient,
                     gboolean link,gboolean as_is)
{ 
  gchar errmsg[]="Error!";
  gchar *msg;
  GtkSheet *sheet;
  SGworksheet *temp_sheet;
  gint retval,acol,arow;
  gboolean restore_active=FALSE;

  temp_sheet=active_worksheet;
  active_worksheet=worksheet;
  sheet = GTK_SHEET(worksheet->sheet);

  if (object)
  {
#ifdef WITH_NUMERIC_PYTHON
      if (PyArray_Check(object))
       { 
         gtk_sheet_freeze(sheet);
         retval=python_array(worksheet,row,col,(PyArrayObject *)object,orient,link);
         gtk_sheet_thaw(sheet);
       }
      else
#endif
      if (PySequence_Check(object))
       { 
         gtk_sheet_freeze(sheet);
         retval=python_sequence(worksheet,row,col,object,orient,link,as_is);
         gtk_sheet_thaw(sheet);
       }

      else
         retval=python_singleton(worksheet,row,col,object,link,as_is);

   }
  else
      retval=0;

  return retval;
}

PyObject *
python_eval_expr(gchar *command)
{ 
  PyObject *object;

  object=PyRun_String (command, Py_eval_input, main_dict, sg_dict);

  if (object)
    return object;
  if (PyErr_Occurred()) PyErr_Clear();

  object=PyRun_String (command, Py_single_input, main_dict, sg_dict);
  if (object)
    return object;
  if (python_error_report(object))
        return NULL;

  Py_INCREF(Py_None);
  return(Py_None);
}

int 
python_sheet(SGworksheet *worksheet, gint row, gint col, gchar *command,
             GtkOrientation orient)
{ 
  PyObject *object=NULL;
  gchar errmsg[]="Error!";
  gchar *msg;
  GtkSheet *sheet;
  SGworksheet *temp_sheet;
  gint retval;

  temp_sheet=active_worksheet;
  active_worksheet=worksheet;
  sheet = GTK_SHEET(worksheet->sheet);
  object=PyRun_String (command, Py_eval_input, main_dict, sg_dict);
  if (!object)
   { 
     if (PyErr_Occurred()) PyErr_Clear();
     PyRun_String (command, Py_single_input, main_dict, sg_dict);
   }


  if (object)
   {  
      Py_INCREF(object);
      retval=python_insert_object(worksheet, row, col, object, orient, FALSE, FALSE);
      Py_XDECREF(object);
   }
  else
   retval=0;

  active_worksheet=temp_sheet;
  if (!retval)
   { 
     if (PyErr_Occurred())
      { 
        msg=errmsg;
        create_python_term(NULL,NULL);
        PyErr_Print();
        if (Py_FlushLine()) PyErr_Clear();
        PyErr_Clear();
      }
     else {
         SGhiddencell *link;
         link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet->sheet), row, col);
         if (link && link->formula){
             g_free(link->formula);
             link->formula=NULL;
         }
             msg=command;
     }

     sg_worksheet_cell_set_text(worksheet,row,col,msg);
     return 0;
   }
  return 1;
}

/* This function is disabled due to a lack of interest in sandboxing 
   the python interpreter.

static PyObject *
sg_import(PyObject *self, PyObject *args)
{
  char *name;
  PyObject *globals = NULL; 
  PyObject *locals = NULL;  
  PyObject *fromlist = NULL;
  PyObject *object = NULL;
  PyObject *mod_object = NULL;
  char message[160];
  char *basename;
  int add_trusted=0;


  if (!PyArg_ParseTuple(args, "s|OOO:__import__",
 	                &name, &globals, &locals, &fromlist))
      return NULL;

  basename=strtok(name,".");
  /* Add code to check against safe module list here 
  object=PyDict_GetItemString(SG_SAFE_MODULES,basename);
  if (!object)
    object=PyDict_GetItemString(SG_IMPORT_MODULES,basename);

  if (!object){
   g_snprintf(message,160,"Attempt to load module named '%s'.\nPermanently add to list of trusted modules?",basename);

   add_trusted=sg_accept_dialog(message,0);
   if (add_trusted==2) {
      g_snprintf(message,160,"Attempt to load untrusted module '%s'",name);
      PyErr_SetString(PyExc_ValueError,message);
      return NULL;
   }
   if (add_trusted==1)
       PyDict_SetItemString(SG_SAFE_MODULES,basename,Py_None);
  }

  mod_object=PyImport_ImportModuleEx (name, globals, locals, fromlist);
  Py_INCREF(mod_object);
  return mod_object;
}
*/

static PyObject *fregister(PyObject *self, PyObject *args){
    char *name,*group;
    SGplugin *plugin;
    PyObject *callable;

  if (!PyArg_ParseTuple(args, "ssO", &name, &group, &callable))
      return NULL;
  if (!PyCallable_Check(callable)){
      char message[160];
      g_snprintf(message,160,"The third argument must be a callable object.");
      PyErr_SetString(PyExc_ValueError,message);
      return NULL;
  }


  plugin=sg_plugin_new_python(name, group, callable);
  if (!plugin){
      char message[160];
      g_snprintf(message,160,"Error registering plugin '%s'",name);
      PyErr_SetString(PyExc_ValueError,message);
      return NULL;
  }

  Py_INCREF(Py_None);
  return(Py_None);
}

static PyObject *new_plugin_group(PyObject *self, PyObject *args){
    char *name;

    if (!PyArg_ParseTuple(args, "s", &name))
        return NULL;
  
    sg_plugin_group_new(name);
  
    Py_INCREF(Py_None);
    return(Py_None);
}


static PyObject *
_describe_all(PyObject *self, PyObject *args)
{
    PyMethodDef *method=SGMethods;
    const gchar *eol="\n";
    const gchar *sep="####";
    gchar *prev=NULL;
    gchar *next=NULL;
    gchar *final=NULL;
    PyObject *object;

    /*    if (!PyArg_ParseTuple(args, "|O:_describe_all",&object))
       return NULL;
    */

    if ( (NULL!=method) && (NULL!=method->ml_name) )
    {
        prev=g_strjoin(eol,method->ml_name,method->ml_doc,NULL);
        ++method;
        for (; NULL!=method->ml_name; ++method)
        {
            next=g_strjoin(eol,prev,sep,method->ml_name,method->ml_doc,NULL);
            g_free(prev);
            prev=next;
            next=NULL;
        }
        final=g_strconcat(prev,eol,NULL);
        g_free(prev);
        prev=NULL;
    }
    else
    {
        final=g_strdup("No methods found\n");
    }

    object = Py_BuildValue("s",final);

    g_free(final);
    Py_INCREF(object);
    return object;

}

static PyObject *_get_worksheet_names(PyObject *self, PyObject *args){
  SGworksheet *worksheet;
  gchar *name;
  PyObject *pname,*plist;
  GList *list;

  if (!PyArg_ParseTuple(args, ":worksheets"))
      return NULL;

  list=get_worksheet_names();

  if (!list) {
    Py_INCREF(Py_None);
    return Py_None;  
  }

  plist=PyList_New(0);

  while (list){
      name=(gchar*)list->data;
      pname=PyString_FromString(name);
      Py_INCREF(pname);
      PyList_Append(plist,pname);
      list=list->next;
  }
  Py_INCREF(plist);
  return plist;
}

static PyObject *_get_active_worksheet_name(PyObject *self, PyObject *args){
  SGworksheet *worksheet;
  gchar *name;
  PyObject *pname;

  if (!PyArg_ParseTuple(args, ":get_worksheets"))
      return NULL;

  if (active_worksheet){
      name=active_worksheet->name;
      pname=PyString_FromString(active_worksheet->name);
      if (pname){

          Py_INCREF(pname);
          return pname;
      }
  }
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *_get_column_names(PyObject *self, PyObject *args, PyObject *keywds){
  SGworksheet *worksheet;
  PyObject *pname,*plist;
  GList *list;
  gchar *sheet=NULL,*name;
  static gchar *kwlist[]={"sheet",NULL},*col_name;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "|s", kwlist, &sheet))
      return NULL;

  
//  if (!PyArg_ParseTuple(args, "s", &name))
//      return NULL;
  worksheet = sg_project_get_worksheet(sheet);

  list=get_column_names(worksheet);
  if (!list) {
    Py_INCREF(Py_None);
    return Py_None;  
  }

  plist=PyList_New(0);

  while (list){
      name=(gchar*)list->data;
      pname=PyString_FromString(name);
      Py_INCREF(pname);
      PyList_Append(plist,pname);
      list=list->next;
  }
  Py_INCREF(plist);
  return plist;
}

static PyObject *_worksheet_add_columns(PyObject *self, PyObject *args, PyObject *keywds){
  SGworksheet *worksheet;
  PyObject *pname,*plist;
  GList *list;
  gchar *sheet=NULL,*name;
  gint num;
  static gchar *kwlist[]={"sheet",NULL},*col_name;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "i|s", kwlist, &num,&sheet))
      return NULL;

  worksheet = sg_project_get_worksheet(sheet);

  sg_worksheet_add_columns(worksheet,num);
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *_worksheet_get_column_exp(PyObject *self, PyObject *args, PyObject *keywds){
  SGworksheet *worksheet;
  PyObject *col_obj,*p_exp;
  GList *list;
  gchar *sheet=NULL,*exp;
  gint num;
  static gchar *kwlist[]={"sheet",NULL},*col_name;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist, &col_obj,&sheet))
      return NULL;

  worksheet = sg_project_get_worksheet(sheet);
  num=get_worksheet_col(worksheet,col_obj);
   if (num<GTK_SHEET(worksheet->sheet)->maxcol){
    exp=g_strdup(worksheet->column[num].exp);
    if (exp) p_exp=PyString_FromString(exp);
    else (p_exp)=Py_None;
    Py_INCREF(p_exp);
    return p_exp;
  }
  PyErr_SetString(PyExc_ValueError, "Column number out of range or name not found");
  return NULL;
}

static PyObject *_worksheet_get_column_precision(PyObject *self, PyObject *args, PyObject *keywds){
  SGworksheet *worksheet;
  PyObject *col_obj,*p_prec;
  GList *list;
  gchar *sheet=NULL,*exp;
  gint num,prec;
  static gchar *kwlist[]={"sheet",NULL},*col_name;

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist, &col_obj,&sheet))
      return NULL;

  worksheet = sg_project_get_worksheet(sheet);
  num=get_worksheet_col(worksheet,col_obj);

  if (num<GTK_SHEET(worksheet->sheet)->maxcol){
    prec=worksheet->column[num].precision;
    p_prec=PyInt_FromLong((long)prec);
    Py_INCREF(p_prec);
    return p_prec;
  }
  PyErr_SetString(PyExc_ValueError, "Column number out of range or name not found");
  return NULL;
}


static PyObject *
get_selection(PyObject *self, PyObject *args, PyObject *keywds)
{ 
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object=NULL,*dict;
  gboolean error=FALSE;
  gchar *sheetname=NULL,*content;
  gint row=0,col=0,num,arow=-1,acol=-1;
  GtkSheet *sheet;
  static gchar *kwlist[]={"sheet",NULL};

  if (!PyArg_ParseTuple(args, "|s:get_selection", &sheetname))
      return NULL;

  /* Get sheet pointer */
  worksheet= sg_project_get_worksheet(sheetname);
  if(!worksheet) {
      char message[160];
      g_snprintf(message,160,"No such worksheet '%s'",sheetname);
      PyErr_SetString(PyExc_ValueError,message);
      return NULL;

  }

  sheet=GTK_SHEET(worksheet->sheet);
  dict=PyDict_New();

  PyDict_SetItemString(dict, "col0", PyInt_FromLong(sheet->range.col0));
  PyDict_SetItemString(dict, "row0", PyInt_FromLong(sheet->range.row0));
  PyDict_SetItemString(dict, "coli", PyInt_FromLong(sheet->range.coli));
  PyDict_SetItemString(dict, "rowi", PyInt_FromLong(sheet->range.rowi));

  Py_INCREF(dict);
  return dict;

}

static PyObject *
_config_register(PyObject *self, PyObject *args)
{
  gchar *name, *group;
  PyObject *def,*commit;

  if (!PyArg_ParseTuple(args, "ssOO", &name, &group, &def, &commit))
      return NULL;
  if (!PyCallable_Check(def) || !PyCallable_Check(commit)){
     PyErr_SetString(PyExc_ValueError,"Third and fourth arguments must be callable objects.");
     return NULL;
  }

  sg_config_new_python(name, group, def, commit);

  Py_INCREF(Py_None);
  return Py_None;

}

static PyObject *
_config_get_value(PyObject *self, PyObject *args)
{
  gchar *name, *group;
  PyObject *value;

  if (!PyArg_ParseTuple(args, "ss", &name, &group))
      return NULL;
  value=sg_config_get_value(name, group);
  if (!value){
     PyErr_SetString(PyExc_ValueError,"No such name or group registered");
     return NULL;
  }

  Py_INCREF(value);
  return value;

}

static PyObject *
_config_set_value(PyObject *self, PyObject *args)
{
  gchar *name, *group;
  PyObject *value;
  gint overwrite=TRUE;

  if (!PyArg_ParseTuple(args, "ssO|i:overwrite", &name, &group,&value,&overwrite))
      return NULL;
  if (!value){
     PyErr_SetString(PyExc_ValueError,"No such name or group registered");
     return NULL;
  }
  sg_config_set_value(name, group,value,overwrite);
  Py_INCREF(Py_None);
  return Py_None;

}

static PyObject *
_config_exec_default(PyObject *self, PyObject *args)
{ SGconfig *config;
  gchar *name, *group;

  if (!PyArg_ParseTuple(args, "ss", &name, &group))
      return NULL;
  config=sg_config_get_config(name, group);
  
  if (!config){
     PyErr_SetString(PyExc_ValueError,"No such name or group registered");
     return NULL;
  }

  sg_config_exec_default(config);
  Py_INCREF(Py_None);
  return Py_None;

}

static PyObject *
_config_exec_commit(PyObject *self, PyObject *args)
{ SGconfig *config;
  gchar *name, *group;

  if (!PyArg_ParseTuple(args, "ss", &name, &group))
      return NULL;

  config=sg_config_get_config(name, group);
  
  if (!config){
     PyErr_SetString(PyExc_ValueError,"No such name or group registered");
     return NULL;
  }

  sg_config_exec_commit(config);
  Py_INCREF(Py_None);
  return Py_None;

}

static PyObject *_plot_active(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL;
  PyObject *pname;
  SGplot *plot;

  if (!PyArg_ParseTuple(args, ""))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot || !plot->name){
       PyErr_SetString(PyExc_ValueError,"No active plot, or no plots in project");
       return NULL;
    }

  pname=PyString_FromString(plot->name);

  Py_INCREF(pname);
  return pname;
}


static PyObject *_plot_list(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name;
  PyObject *pname,*plist;
  GList *list;

  if (!PyArg_ParseTuple(args, ""))
      return NULL;

  plist=PyList_New(0);

  list=plots;
  while (list){
      SGplot *plot;
      plot= (SGplot *)list->data;
      name=plot->name;
      pname=PyString_FromString(name);
      Py_INCREF(pname);
      PyList_Append(plist,pname);
      list=list->next;
  }
  Py_INCREF(plist);
  return plist;
}

static PyObject *_plot_layers(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL;
  PyObject *pint;
  SGplot *plot;

  if (!PyArg_ParseTuple(args, "|s",&name))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
    }

  pint=PyInt_FromLong((long)plot->nlayers);

  Py_INCREF(pint);
  return pint;
}

static PyObject *_plot_active_layer(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL;
  PyObject *pint;
  SGplot *plot;
  GList *layers;
  gint num=-1;

  if (!PyArg_ParseTuple(args, "|s",&name))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
    }

  layers=plot->layers;
  while (layers){
      num++;
      if (layers->data==plot->active_layer) break;
      layers=layers->next;
  }

  pint=PyInt_FromLong((long)num+1);

  Py_INCREF(pint);
  return pint;
}

static PyObject *_plot_layer_dataset_list(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL;
  PyObject *plist,*pname;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1;
  SGlayer *slayer=NULL;

  if (!PyArg_ParseTuple(args, "|si",&name,&lnum))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  if (plot->nlayers<1){
       PyErr_SetString(PyExc_ValueError,"No layers in plot");
       return NULL;
   }


  layers=plot->layers;

  lnum--;
  if (lnum>=0 && lnum<plot->nlayers){
    while (layers){
        num++;
        if (num==lnum) break;
        layers=layers->next;
    }
  }
  else{
    while (layers){
        if (layers->data==plot->active_layer) break;
        layers=layers->next;
    }
    
  }

  if (layers)
      slayer=(SGlayer *)layers->data;
  
  if (!slayer){
       PyErr_SetString(PyExc_ValueError,"Specified layer not found");
       return NULL;
  }
  
  plist=PyList_New(0);

  list=slayer->datasets;
  while (list){

      SGdataset *plotdata;

      plotdata= (SGdataset *)list->data;
      name=plotdata->real_data->name;
      pname=PyString_FromString(name);
      Py_INCREF(pname);
      PyList_Append(plist,pname);
      list=list->next;
  }
  Py_INCREF(plist);
  return plist;

}


static PyObject *_plot_layer_dataset(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL,*dataname=NULL;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1];
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;

  if (!PyArg_ParseTuple(args, "sis",&name,&lnum,&dataname))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  if (plot->nlayers<1){
       PyErr_SetString(PyExc_ValueError,"No layers in plot");
       return NULL;
   }


  layers=plot->layers;

  lnum--;
  if (lnum>=0 && lnum<plot->nlayers){
    while (layers){
        num++;
        if (num==lnum) break;
        layers=layers->next;
    }
  }
  else{
    while (layers){
        if (layers->data==plot->active_layer) break;
        layers=layers->next;
    }
    
  }

  if (layers)
      slayer=(SGlayer *)layers->data;
  
  if (!slayer){
       PyErr_SetString(PyExc_ValueError,"Specified layer not found");
       return NULL;
  }
  
  plist=PyList_New(0);

  list=slayer->datasets;
  while (list){
      plotdata= (SGdataset *)list->data;
      if (!strcmp(plotdata->real_data->name,dataname)) break;
  }

  if (plotdata && list) /* dataset found */
  { gint i;
    cobject=PyDict_New();
    plist=PyList_New(0);

    for (i=0;i<9;i++){
        if (plotdata->p_exp[i])
            PyList_Append (plist, PyString_FromString (plotdata->p_exp[i]));
        else
            PyList_Append (plist, Py_None);
    }
    PyDict_SetItemString(cobject,"p_exp",plist);
    if (plotdata->exp)
        PyDict_SetItemString(cobject,"exp",PyString_FromString (plotdata->exp));
    else
        PyDict_SetItemString(cobject,"exp", Py_None);

    PyDict_SetItemString(cobject,"type",PyInt_FromLong ((long) plotdata->type));

    dims[0]=plotdata->real_data->num_points;
    
    if (plotdata->real_data->num_points>0)
    {  gboolean show_labels,error;
       gchar **labels, *label;
       gint num_points;
       gdouble *data[8], x,y,z,a,dx,dy,dz,da;
       GtkPlotIterator *iterator;

       /*iterator=(GtkPlotIterator *)plotdata->real_data->iterator;*/
       num_points=plotdata->real_data->num_points;
       if (plotdata->x>=0)
	 data[0]=g_new(gdouble,num_points);

       if (plotdata->y>=0)
	 data[1]=g_new(gdouble,num_points);

       if (plotdata->z>=0)
	 data[2]=g_new(gdouble,num_points);

       if (plotdata->a>=0)
	 data[3]=g_new(gdouble,num_points);

       if (plotdata->dx>=0)
	 data[4]=g_new(gdouble,num_points);

       if (plotdata->dy>=0)
	 data[5]=g_new(gdouble,num_points);

       if (plotdata->dz>=0)
	 data[6]=g_new(gdouble,num_points);

       if (plotdata->da>=0)
	 data[7]=g_new(gdouble,num_points);

       for (i=0;i<num_points;i++)
       {  gtk_plot_data_get_point(plotdata->real_data, i,
                                  &x, &y, &z, &a, &dx, &dy, &dz, &da,&label,&error);
	 if (plotdata->x>=0)
	   data[0][i]=x;
  
	 if (plotdata->y>=0)
	   data[1][i]=y;
  
	 if (plotdata->z>=0)
	   data[2][i]=z;
  
	 if (plotdata->a>=0)
	   data[3][i]=a;
  
	 if (plotdata->dx>=0)
	   data[4][i]=dx;
  
	 if (plotdata->dy>=0)
	   data[5][i]=dy;
  
	 if (plotdata->dz>=0)
	   data[6][i]=dz;
  
	 if (plotdata->da>=0)
	   data[7][i]=da;

       }

	 if (plotdata->x>=0)
		PyDict_SetItemString(cobject,"x",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[0]));
        else
         PyDict_SetItemString(cobject,"x",PyList_New(0));

	 if (plotdata->y>=0)
		PyDict_SetItemString(cobject,"y",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[1]));
        else
         PyDict_SetItemString(cobject,"y",PyList_New(0));

	 if (plotdata->z>=0)
		PyDict_SetItemString(cobject,"z",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[2]));
        else
         PyDict_SetItemString(cobject,"z",PyList_New(0));

	if (plotdata->a>=0)
		PyDict_SetItemString(cobject,"a",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[3]));
        else
         PyDict_SetItemString(cobject,"a",PyList_New(0));

	 if (plotdata->dx>=0)
		PyDict_SetItemString(cobject,"dx",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[4]));
        else
         PyDict_SetItemString(cobject,"dx",PyList_New(0));

	 if (plotdata->dy>=0)
		PyDict_SetItemString(cobject,"dy",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[5]));
        else
         PyDict_SetItemString(cobject,"dy",PyList_New(0));

	 if (plotdata->dz>=0)
		PyDict_SetItemString(cobject,"dz",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[6]));
        else
         PyDict_SetItemString(cobject,"dz",PyList_New(0));

	if (plotdata->da>=0)
		PyDict_SetItemString(cobject,"da",PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,
               (char *)data[7]));
        else
         PyDict_SetItemString(cobject,"da",PyList_New(0));


       plist=PyList_New(0);
       Py_INCREF(plist);
       PyDict_SetItemString(cobject,"l",plist);
       labels=gtk_plot_data_get_labels(plotdata->real_data,&show_labels);
         for (i=0;i<plotdata->l;i++)
         {  pname=PyString_FromString(labels[i]);
            PyList_Append(plist,pname);
         }

    } 
    Py_INCREF(cobject);
    return cobject;
  
  }
  else
    PyErr_SetString(PyExc_ValueError,"Specified dataset not found");
       return NULL;
}


static PyObject *_plot_layer_refresh(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL,*dataname=NULL;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1];
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;

  if (!PyArg_ParseTuple(args, "si",&name,&lnum))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  if (plot->nlayers<1){
       PyErr_SetString(PyExc_ValueError,"No layers in plot");
       return NULL;
   }


  layers=plot->layers;

  lnum--;
  if (lnum>=0 && lnum<plot->nlayers){
    while (layers){
        num++;
        if (num==lnum) break;
        layers=layers->next;
    }
  }
  else{
    while (layers){
        if (layers->data==plot->active_layer) break;
        layers=layers->next;
    }
    
  }

  if (layers)
      slayer=(SGlayer *)layers->data;
  
  if (!slayer){
       PyErr_SetString(PyExc_ValueError,"Specified layer not found");
       return NULL;
  }
  
  sg_layer_refresh_datasets(slayer);
  
  Py_INCREF(Py_None);
  return (Py_None);

}

static PyObject *_plot_layer_autoscale(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL,*dataname=NULL;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1];
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;
  GtkPlot *real_plot;

  if (!PyArg_ParseTuple(args, "si",&name,&lnum))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  if (plot->nlayers<1){
       PyErr_SetString(PyExc_ValueError,"No layers in plot");
       return NULL;
   }


  layers=plot->layers;

  lnum--;
  if (lnum>=0 && lnum<plot->nlayers){
    while (layers){
        num++;
        if (num==lnum) break;
        layers=layers->next;
    }
  }
  else{
    while (layers){
        if (layers->data==plot->active_layer) break;
        layers=layers->next;
    }
    
  }

  if (layers)
      slayer=(SGlayer *)layers->data;
  
  if (!slayer){
       PyErr_SetString(PyExc_ValueError,"Specified layer not found");
       return NULL;
  }
  
  real_plot = GTK_PLOT(slayer->real_plot);

  if(GTK_IS_PLOT3D(real_plot))
    gtk_plot3d_autoscale(GTK_PLOT3D(real_plot));
  else
    sg_layer_autoscale(slayer,sg_autoscale_left,sg_autoscale_right,sg_autoscale_top,
                      sg_autoscale_bottom);

  Py_INCREF(Py_None);
  return (Py_None);

}

static PyObject *_plot_refresh_canvas(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL,*dataname=NULL;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1];
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;
  GtkPlot *real_plot;

  if (!PyArg_ParseTuple(args, "s",&name))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));

  
  Py_INCREF(Py_None);
  return (Py_None);

}

static PyObject *_plot_get_size_pos(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL,*dataname=NULL;
  PyObject *dict=NULL,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1];
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;
  GtkPlot *real_plot;

  if (!PyArg_ParseTuple(args, "s",&name))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }
  
   dict=PyDict_New();
   PyDict_SetItemString(dict,"x",PyFloat_FromDouble(plot->x));
   PyDict_SetItemString(dict,"y",PyFloat_FromDouble(plot->y));
   PyDict_SetItemString(dict,"width",PyFloat_FromDouble(plot->width));
   PyDict_SetItemString(dict,"height",PyFloat_FromDouble(plot->height));
  
  Py_INCREF(dict);
  return (dict);

}


static PyObject *axis_text_properties(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *plotname=NULL,*titlename=NULL;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1],show_labels=-1,show_titles=-1,axisnum=0,label_style=-1,
       label_prec=-1;
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;
  GtkPlot *real_plot;
  GtkPlotAxis *axis;
  GtkPlotAxisPos axispos;
  static gchar *kwlist[]={"plot","layer","axis","show_labels","show_title","title","label_style",
                          "label_prec",NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "sii|iisii", kwlist,
                                   &plotname, &lnum, &axisnum, &show_labels,&show_titles,&titlename,&label_style,
                                   &label_prec))
      return NULL;

  plot=sg_project_get_plot(plotname);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  if (plot->nlayers<1){
       PyErr_SetString(PyExc_ValueError,"No layers in plot");
       return NULL;
   }


  layers=plot->layers;

  lnum--;
  if (lnum>=0 && lnum<plot->nlayers){
    while (layers){
        num++;
        if (num==lnum) break;
        layers=layers->next;
    }
  }
  else{
    while (layers){
        if (layers->data==plot->active_layer) break;
        layers=layers->next;
    }
    
  }

  if (layers)
      slayer=(SGlayer *)layers->data;
  
  if (!slayer){
       PyErr_SetString(PyExc_ValueError,"Specified layer not found");
       return NULL;
  }
  
  real_plot = GTK_PLOT(slayer->real_plot);

  if(GTK_IS_PLOT3D(real_plot))
      switch(axisnum)
     { case 0: axis=&GTK_PLOT3D(real_plot)->xy;
               axispos=GTK_PLOT_AXIS_LEFT;
               break;
       case 1: axis=&GTK_PLOT3D(real_plot)->yx;
               axispos=GTK_PLOT_AXIS_RIGHT;
               break;
       case 2: axis=&GTK_PLOT3D(real_plot)->xz;
               axispos=GTK_PLOT_AXIS_TOP;
               break;
       case 3: axis=&GTK_PLOT3D(real_plot)->yz;
               axispos=GTK_PLOT_AXIS_BOTTOM;
               break;
       case 4: axis=&GTK_PLOT3D(real_plot)->zx;
               axispos=GTK_PLOT_AXIS_Z;
               break;
       case 5: axis=&GTK_PLOT3D(real_plot)->zy;
               axispos=GTK_PLOT_AXIS_Z;
               break;

       default: PyErr_SetString(PyExc_ValueError,"Specified axis not found");
               return NULL;
     }

  else
    switch(axisnum)
     { case 0: axis=real_plot->bottom;
               axispos=GTK_PLOT_AXIS_BOTTOM;
               break;
       case 1: axis=real_plot->left;
               axispos=GTK_PLOT_AXIS_LEFT;
               break;
       case 2: axis=real_plot->top;
               axispos=GTK_PLOT_AXIS_TOP;
               break;
       case 3: axis=real_plot->right;
               axispos=GTK_PLOT_AXIS_RIGHT;
               break;
       default: PyErr_SetString(PyExc_ValueError,"Specified axis not found");
               return NULL;
     }
  
  if (show_labels==TRUE || show_labels==FALSE)
    axis->label_mask = show_labels*GTK_PLOT_LABEL_OUT;

  if (show_titles==TRUE || show_titles==FALSE)
    axis->title_visible = show_titles;

  if (label_style>=0 && label_style<3)
    axis->label_style = label_style;

  if (label_prec>=0 && label_prec<10)
    axis->label_precision = label_prec;

  
  if (titlename)
    gtk_plot_axis_set_title(GTK_PLOT(slayer->real_plot), axispos, g_strdup(titlename));
  
  Py_INCREF(Py_None);
  return (Py_None);

}

static PyObject *set_axis_label_pos(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *plotname=NULL,*titlename=NULL;
  gdouble justify_axis,offset,dir,x_pos,y_pos,x,y,width,height,angle;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1],show_labels=-1,show_titles=-1,axisnum=0,label_style=-1,
       label_prec=-1, justify=GTK_JUSTIFY_CENTER;
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;
  GtkPlot *real_plot;
  GtkPlotAxis *axis;
  GtkPlotAxisPos axispos;
  static gchar *kwlist[]={"plot","layer","axis","offset","justify_axis","angle","justify",NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "siiddd|i", kwlist,
                                   &plotname, &lnum, &axisnum, &offset, &justify_axis, &angle, &justify))
      return NULL;

  plot=sg_project_get_plot(plotname);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  if (plot->nlayers<1){
       PyErr_SetString(PyExc_ValueError,"No layers in plot");
       return NULL;
   }


  layers=plot->layers;

  lnum--;
  if (lnum>=0 && lnum<plot->nlayers){
    while (layers){
        num++;
        if (num==lnum) break;
        layers=layers->next;
    }
  }
  else{
    while (layers){
        if (layers->data==plot->active_layer) break;
        layers=layers->next;
    }
    
  }

  if (layers)
      slayer=(SGlayer *)layers->data;
  
  if (!slayer){
       PyErr_SetString(PyExc_ValueError,"Specified layer not found");
       return NULL;
  }
  
  real_plot = GTK_PLOT(slayer->real_plot);
  gtk_plot_get_position(real_plot,&x,&y);
  gtk_plot_get_size(real_plot,&width,&height);

  if(GTK_IS_PLOT3D(real_plot))
      switch(axisnum)
     { case 0: axis=&GTK_PLOT3D(real_plot)->xy;
               axispos=GTK_PLOT_AXIS_LEFT;
               break;
       case 1: axis=&GTK_PLOT3D(real_plot)->yx;
               axispos=GTK_PLOT_AXIS_RIGHT;
               break;
       case 2: axis=&GTK_PLOT3D(real_plot)->xz;
               axispos=GTK_PLOT_AXIS_TOP;
               break;
       case 3: axis=&GTK_PLOT3D(real_plot)->yz;
               axispos=GTK_PLOT_AXIS_BOTTOM;
               break;
       case 4: axis=&GTK_PLOT3D(real_plot)->zx;
               axispos=GTK_PLOT_AXIS_Z;
               break;
       case 5: axis=&GTK_PLOT3D(real_plot)->zy;
               axispos=GTK_PLOT_AXIS_Z;
               break;

       default: PyErr_SetString(PyExc_ValueError,"Specified axis not found");
               return NULL;
     }

  else
    switch(axisnum)
     { case 0: axis=real_plot->bottom;
               axispos=GTK_PLOT_AXIS_BOTTOM;
               y_pos=y+offset+height;
               x_pos=x+width*justify_axis;
               break;
       case 1: axis=real_plot->left;
               axispos=GTK_PLOT_AXIS_LEFT;
               x_pos=x-offset;
               y_pos=y+height*justify_axis;
               break;
       case 2: axis=real_plot->top;
               axispos=GTK_PLOT_AXIS_TOP;
               y_pos=y-offset;
               x_pos=x+width*justify_axis;
               break;
       case 3: axis=real_plot->right;
               axispos=GTK_PLOT_AXIS_RIGHT;
               x_pos=x+offset+width;
               y_pos=y+height*justify_axis;
               break;
       default: PyErr_SetString(PyExc_ValueError,"Specified axis not found");
               return NULL;
     }
    if (offset>0.5 || offset<-0.5){
       PyErr_SetString(PyExc_ValueError,"Offset value too large, must be in the range -0.5 -> 0.5");
       return NULL;
   }
   if (justify_axis>1. || justify_axis<0.){
       PyErr_SetString(PyExc_ValueError,"Axis justify value too large, must be in the range 0 -> 1");
       return NULL;
   }

   if (angle!=0. && angle!=90 && angle!=180 && angle!=270){
       PyErr_SetString(PyExc_ValueError,"Angle value should be one of 0, 90, 180 or 270 degrees");
       return NULL;
   }

   if (justify<0 || justify>2){
       PyErr_SetString(PyExc_ValueError,"Justify value should be 0=left, 1=right or 2=center[default]");
       return NULL;
   }


  gtk_plot_axis_move_title(real_plot, axispos, angle, x_pos, y_pos);
  gtk_plot_axis_justify_title(real_plot, axispos, justify);

  Py_INCREF(Py_None);
  return (Py_None);

}

static PyObject *_plot_freeze_canvas(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL,*dataname=NULL;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1];
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;
  GtkPlot *real_plot;

  if (!PyArg_ParseTuple(args, "s",&name))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  gtk_plot_canvas_freeze(GTK_PLOT_CANVAS(plot->real_canvas));

  
  Py_INCREF(Py_None);
  return (Py_None);

}

static PyObject *_plot_thaw_canvas(PyObject *self, PyObject *args, PyObject *keywds){
  gchar *name=NULL,*dataname=NULL;
  PyObject *plist,*pname,*cobject=NULL,*pint,*ptemp;
  SGplot *plot;
  GList *layers,*list;
  gint lnum=-1,num=-1,dims[1];
  SGlayer *slayer=NULL;
  SGdataset *plotdata=NULL;
  GtkPlot *real_plot;

  if (!PyArg_ParseTuple(args, "s",&name))
      return NULL;

  plot=sg_project_get_plot(name);
  if (!plot){
       PyErr_SetString(PyExc_ValueError,"No such plot, or no plots in project");
       return NULL;
   }

  gtk_plot_canvas_thaw(GTK_PLOT_CANVAS(plot->real_canvas));

  
  Py_INCREF(Py_None);
  return (Py_None);

}
