/* rtp_math.c 
 *
 * Math fuunctions for the rtp (real-time-plotter) program.
 *
 * Copyright (c) 1999, David Watt <wattd@elcsci.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static char rtp_math_c_id[] = "$Id: rtp_math.c,v 1.8 2000/03/16 06:46:50 davidw Exp $";

#include <limits.h>
#include <math.h>
#include <string.h>
#include <stdio.h>
#include <deque>
using namespace std;

#include "rtp.h"

#define MIN(x,y)  ((x)<(y) ? (x) : (y))
#define MAX(x,y)  ((x)>(y) ? (x) : (y))
#define ROUND(x)  (int(floor((x) + 0.5)))

/* rtpClip
 *
 * Clips a line segment to a given closed bounding box.  Uses the
 * Liang-Barsky algorithm to take 1 half-space at a time.  Note that
 * the points *r1 and *r2 will be modified directly if clipping is done.
 *
 * r1 and r2 are the line segment endpoints used, and box is the bounding
 * rectangle.  The return code is true if clipping was successful, or false
 * if the line segment must be rejected.
 */

inline static bool rtpClipGt(double x, double y) { return x > y; }
inline static bool rtpClipLt(double x, double y) { return x < y; }
inline static bool rtpClipGeq(double x, double y) { return x >= y; }
inline static bool rtpClipLeq(double x, double y) { return x <= y; }

#define RTP_CLIP_HALFSPACE(cmp_out, cmp_in, edge, cmp_axis, other_axis)    \
  if (cmp_out(r1->cmp_axis, edge) && cmp_out(r2->cmp_axis, edge))          \
    return false;                                                          \
  else if (cmp_in(r1->cmp_axis, edge) && cmp_in(r2->cmp_axis, edge))       \
    ;                                                                      \
  else {                                                                   \
    if (cmp_out(r1->cmp_axis, r2->cmp_axis))                               \
      boundPt = r1;                                                        \
    else                                                                   \
      boundPt = r2;                                                        \
    t = (edge - r1->cmp_axis) / (r2->cmp_axis - r1->cmp_axis);             \
    boundPt->other_axis = r1->other_axis * (1.0 - t) + r2->other_axis * t; \
    boundPt->cmp_axis = edge;                                              \
  }

bool rtpClip(BBox& box, DoubPt *r1, DoubPt *r2)
{
  double t;
  DoubPt *boundPt;

  RTP_CLIP_HALFSPACE(rtpClipLt, rtpClipGeq, box.minX, x, y)
  RTP_CLIP_HALFSPACE(rtpClipGt, rtpClipLeq, box.maxX, x, y)
  RTP_CLIP_HALFSPACE(rtpClipLt, rtpClipGeq, box.minY, y, x)
  RTP_CLIP_HALFSPACE(rtpClipGt, rtpClipLeq, box.maxY, y, x)

  return true;
}

/* rtpMakeLabels
 * 
 * Generates a deque of strings containing locations and labels for plot 
 * scale ticks.  There are often more ticks than labels, so some of the
 * items in the output deque may have null label pointers.
 * Inputs are the range [minVal, maxVal], and the desired number of cells.
 * The delta per each tick will be a multiple of 2, 5, or 10, whichever
 * yields the closest number of cells, and with whatever
 * exponent is necessary.  Each of the values returned should both be
 * machine convertable via scanf("%lf") and nice looking enough to display
 * on the plot.  The label members of ticks must be deleted by calling
 * code.
 */

void rtpMakeLabels(deque<RTP_TICK_LABEL> *ticks, double minVal, double maxVal,
                   double desiredDeltaTick, int nSpacesPerLabel, 
                   int maxCharLabels)
{
  int expVal, nDigs, nDec, subSample, nLabels;
  int labelSize;
  double deltaTick, bestDeltaTick = 0.0, minTickVal, maxTickVal, dtErr;
  double deltaLabel, minLabelVal, maxLabelVal;
  double dum1, dum2;
  char fmtStr[256], fmtStr2[256];

  /* Compute deltaTick = 10 ^ floor(log10(desiredDeltaTick)).
   * Then look at 1, 2, 5, 10 times this value.
   * Take whichever value is closest to desiredDeltaTick.
   */

  deltaTick = pow(10.0, floor(log10(desiredDeltaTick)));
  dtErr = HUGE_VAL; /* start with some ridiculous big number */ 
  
#define CHECK_NUM_TICKS(x)                                                 \
  if (fabs((x) - desiredDeltaTick) < dtErr) {                              \
    dtErr = fabs((x) - desiredDeltaTick);                                  \
    bestDeltaTick = (x);                                                   \
  }

  CHECK_NUM_TICKS(deltaTick)
  CHECK_NUM_TICKS(deltaTick*2.0)
  CHECK_NUM_TICKS(deltaTick*5.0)
  CHECK_NUM_TICKS(deltaTick*10.0)
  deltaTick = bestDeltaTick;

#define MIN_MAX_TICKS(delta, minTick, maxTick)                             \
  dum1 = floor(minVal / (delta));                                          \
  if (dum1 * (delta) < minVal)                                             \
    dum1 += 1.0;                                                           \
  (minTick) = dum1 * (delta);                                              \
  dum1 = floor(maxVal / (delta));                                          \
  dum1 += 1.0;                                                             \
  if (dum1 * (delta) > maxVal)                                             \
    dum1 -= 1.0;                                                           \
  (maxTick) = dum1 * (delta);
 
  /* Find the minTickVal and maxTickVal.  If the delta is such that there
   * are no ticks, return empty-handed. */
  MIN_MAX_TICKS(deltaTick, minTickVal, maxTickVal)
  if (minTickVal > maxTickVal)
    return;

  /* Store tick positions in the deque. */
  for (dum1 = minTickVal; dum1 <= maxVal; dum1 += deltaTick) {
    RTP_TICK_LABEL tick;
    tick.pos = dum1;
    tick.label = NULL;
    ticks->push_back(tick);
  }

  /* Now we find what density labels should be placed amongst these ticks.
   * To do this we scale deltaLabel as an integral multiple of deltaTick
   * and see how many characters are required to display the largest
   * magnitude number.  We keep making deltaLabel bigger until
   * nLabels * maxChars < maxCharLabels or nLabels <= 1 */
  
  subSample = 0;
  do {
    ++subSample;
    deltaLabel = double(subSample) * deltaTick;
    MIN_MAX_TICKS(deltaLabel, minLabelVal, maxLabelVal)
    // Check for min/max screw-up.  If detected, return without filling
    // any labels.
    if (minLabelVal > maxLabelVal)
      return;
    nLabels = ROUND((maxLabelVal - minLabelVal) / deltaLabel) + 1;

    if (maxLabelVal >= 0 && minLabelVal <= 0)
      expVal = int(floor(log10(deltaLabel)));
    else
      expVal = MIN(int(floor(log10(fabs(minLabelVal)))),
                   int(floor(log10(fabs(maxLabelVal)))));


    /* Compute how many digits are necessary. Allow the special case of
     * both minLabelVal and maxLabelVal equal to zero.
     */

    if (   fabs(minLabelVal) < 0.5 * deltaTick
        && fabs(maxLabelVal) < 0.5 * deltaTick )
    {
      nDigs = 1;   // Special handling for zero.
    }
    else
    {
      dum1 = MAX(fabs(minLabelVal), fabs(maxLabelVal));
      nDigs = int(floor(log10(dum1))) - int(floor(log10(deltaTick))) + 1;
    }

    /* Is a decimal point necessary, and if so, how many digits on the
     * right side? */
    nDec = expVal - int(floor(log10(deltaTick)));
  
    /* Bring expVal down to a multiple of 3 (engineering notation).
     * Shift decimal point and add digits as necessary. */
    while (abs(expVal) % 3 != 0) {
      expVal--;
      if (nDec > 0)
        nDec--;
      else
        nDigs++;
    }

    /* Compute largest field size to support. */
    labelSize = nDigs;
    if (minLabelVal < 0.0)
      ++labelSize;           // Need minus sign.
    if (nDec > 0)
      ++labelSize;           // Need decimal point.
    if (expVal != 0) {
      sprintf(fmtStr2, "E%d", expVal);
      labelSize += strlen(fmtStr2);
    }
    else
      fmtStr2[0] = '\0';
  } while (   nLabels * (labelSize + nSpacesPerLabel) > maxCharLabels
           && nLabels > 1);
    
  /* Build a format string. */
  sprintf(fmtStr, "%%.%df", nDec);
  strcat(fmtStr, fmtStr2);

  /* Make the labels for each deltaLabel value. */
  deque<RTP_TICK_LABEL>::iterator ptr = ticks->begin();
  while (fabs((*ptr).pos - minLabelVal) > 0.5 * deltaTick)
    ptr++;
 
  dum2 = pow(10.0, (double)expVal);
  while (((*ptr).pos <= maxLabelVal + 0.5 * deltaTick)
         && (ptr != ticks->end())) {
    (*ptr).label = new char[labelSize+1];   // +1 for '\0'
    if (fabs((*ptr).pos) < 0.5 * deltaTick) {   // Special handling for zero.
      (*ptr).label[0] = '0';
      (*ptr).label[1] = '\0';
#if 0
      fprintf(stderr,"0\n");
#endif
    }
    else {
#if 0
      fprintf(stderr, "%s\n", fmtStr);
      fprintf(stderr, fmtStr, (*ptr).pos/dum2);
      fprintf(stderr, "\n");
#endif
      sprintf((*ptr).label, fmtStr, (*ptr).pos/dum2);
    }
    for (int ii = 0; (ii < subSample) && (ptr != ticks->end()); ++ii)
      ptr++;
  }
}

/* rtpResOk
 *
 * Checks that there is enough resolution to draw into passed BBox.
 * There are two checks.  The first is that the delta for each axis is
 * at least some multiple of the minimum double epsilon.  The second is
 * that the delta divided by the max magnitude boundary is at least some
 * multiple of the minimum double epsilon.
 *
 * The passed box will be expanded to pass the checks if necessary.
 * The return value is true if the box has been expanded.
 */

bool rtpResCheck(BBox *b)
{
  bool modified = false;
  double rangeX, rangeY;
  rangeX = b->maxX - b->minX;
  rangeY = b->maxY - b->minY;

  if (rangeX < RTP_MIN_RANGE) {
    b->minX -= (RTP_MIN_RANGE - rangeX) / 2.0;
    b->maxX += (RTP_MIN_RANGE - rangeX) / 2.0;
    rangeX = b->maxX - b->minX;
    modified = true;
  }
  if (rangeY < RTP_MIN_RANGE) {
    b->minY -= (RTP_MIN_RANGE - rangeY) / 2.0;
    b->maxY += (RTP_MIN_RANGE - rangeY) / 2.0;
    rangeY = b->maxY - b->minY;
    modified = true;
  }

  if (   (b->maxX > 0) && (b->minX > 0)
      && (rangeX / b->maxX < RTP_MIN_RES)) {
    b->minX = b->maxX - RTP_MIN_RES * b->maxX;
    modified = true;
  }
  else if (   (b->maxX < 0) && (b->minX < 0)
	   && (rangeX / b->minX > RTP_MIN_RES)) {
    b->maxX = b->minX - RTP_MIN_RES * b->minX;
    modified = true;
  }
  if (   (b->maxY > 0) && (b->minY > 0)
      && (rangeY / b->maxY < RTP_MIN_RES)) {
    b->minY = b->maxY - RTP_MIN_RES * b->maxY;
    modified = true;
  }
  else if (   (b->maxY < 0) && (b->minY < 0)
	   && (rangeY / b->minY > RTP_MIN_RES)) {
    b->maxY = b->minY - RTP_MIN_RES * b->minY;
    modified = true;
  }

  return modified;
}
