#include <qobject.h>

RCSTAG( "$Id: ksciplot.cpp,v 3.8 1998/02/15 15:07:14 kde Exp $" );

#include <qfiledlg.h>
#include <qkeycode.h>
#include <qpainter.h>
#include <qprinter.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "differentiate.h"
#include "expression.h"
#include "fileoperations.h"
#include "koverview.h"
#include "ksciplot.moc"
#include "kprintpreview.h"
#include "alldf.h"
#include "datafile.h"
#include "version.h"

// #define __DEBUG

QList<KFunction>* functions;
bool askOnDelete;

KAllDF *g_kadf;


double
xlog( double x, double base )
{
  return log( x ) / log( base );
} // xlog

double
log2( double x )
{
  return log( x ) / log( 2 );
} // log2


KPlot::KPlot( int& argc, char** argv ) : KApplication( argc, argv )
{
  KPlotMainWindow* mainWindow;
  if ( isRestored() && KTopLevelWidget::canBeRestored( 1 ) ) {
    mainWindow= new KPlotMainWindow();
    CHECK_PTR( mainWindow );
    mainWindow->restore( 1 );
  } // if
  else {
    mainWindow = new KPlotMainWindow();
    CHECK_PTR( mainWindow );
  } // else
  setMainWidget( mainWindow );
} // KPlot constructor


void
KPlot::print()
{
#ifdef __DEBUG
  debug( "KPlot::print" );
#endif

  QPrinter* printer = new QPrinter;
  CHECK_PTR( printer );

  QPainter* p = NULL;
  // initializes p:
  KPrintPreviewDlg* dlg = new KPrintPreviewDlg( mainWidget(), printer, &p );
  CHECK_PTR( dlg );
  if ( dlg->exec() ) paintExpressions( printer, p ); // deletes p
  else printer->abort();

  delete dlg;
  delete printer;
} // KPlot::print


void
KPlot::newExpression()
{
  KFunction* f = new KFunction;
  CHECK_PTR( f );
  f->text = "";
  f->pen = black;

  KSettingsDialog* dlg = new KSettingsDialog( f, mainWidget() );
  CHECK_PTR( dlg );
  if ( dlg->exec() == 1 ) {
    functions->append( f );

    emit expressionAdded();

    mainWidget()->repaint( TRUE );
  } // if
  else delete f;
} // KPlot::newExpression


void
KPlot::delExpression( uint index )
{
  int Delete = 1;
  if ( askOnDelete ) Delete = KMsgBox::yesNo( mainWidget(),
    klocale->translate( "delete expression" ),
    klocale->translate(
      "Are you sure that you want to delete this expression?" ) );

#ifdef __DEBUG
  debug( "Delete = %i\n", Delete );
#endif

  if ( Delete != 1 ) return;

  functions->remove( index );
  emit expressionDeleted( index );
  mainWidget()->repaint( TRUE );
} // KPlot::delExpression


void
KPlot::addFromFile()
{
  QString s = QFileDialog::getOpenFileName( 0, "*.kpd", mainWidget() );
  if ( !s.isEmpty() ) addFunctionsFromFile( s );
} // KPlot::addFromFile


KPlotMainWindow::KPlotMainWindow() : KTopLevelWidget( "KPlot main window" )
{
  setCaption( kapp->getCaption() );

  KMenuBar* mb = new KMenuBar( this, "menubar" );
  CHECK_PTR( mb );

  /* the file menu */
  QPopupMenu* qpmFile = new QPopupMenu;
  CHECK_PTR( qpmFile );
  qpmFile->insertItem( klocale->translate( "&Print" ), kapp,
		       SLOT( print() ), CTRL+Key_P );
  qpmFile->insertSeparator();
  qpmFile->insertItem( klocale->translate( "&Quit" ), kapp, SLOT( quit() ),
		       CTRL+Key_Q );
  
  mb->insertItem( klocale->translate( "&File" ), qpmFile );

  QPopupMenu* qpmExpr = new QPopupMenu;
  CHECK_PTR( qpmExpr );
  qpmExpr->insertItem( klocale->translate( "&New" ), kapp,
                       SLOT( newExpression() ), CTRL+Key_N );
  qpmExpr->insertItem( klocale->translate( "&Overview" ), this,
		       SLOT( overview() ) );
  qpmExpr->insertSeparator();
  qpmExpr->insertItem( klocale->translate( "&add from file" ), kapp,
		       SLOT( addFromFile() ) );
  mb->insertItem( klocale->translate( "&Expressions" ), qpmExpr );

  qpmOptions = new QPopupMenu;
  CHECK_PTR( qpmOptions );
  deleteID = qpmOptions->insertItem( klocale->translate( "&ask on delete" ),
				     this, SLOT ( askDelete() ) );
  qpmOptions->insertSeparator();
  statusID = qpmOptions->insertItem( klocale->translate( "show &status bar" ),
				     this, SLOT( toggleStatusBar() ) );
  toolID = qpmOptions->insertItem( klocale->translate( "show &tool bar" ),
				   this, SLOT( toggleToolBar() ) );

  mb->insertItem( klocale->translate( "&Options" ), qpmOptions );


  // add a standard help menu
  float f = APP_VERSION;
  QString s1 = "ksciplot V";
  QString s2;
  s2.setNum( f );
  s1 += s2;
  s1 += "\nMichael Ritzert <Ritzert@T-Online.de>";
  s1 += "\nDavid Sweet <dsweet@Glue.umd.edu>";

  QPopupMenu* qpmHelp = kapp->getHelpMenu( FALSE, s1 );
  CHECK_PTR( qpmHelp );
  mb->insertSeparator();
  mb->insertItem( klocale->translate( "&Help" ), qpmHelp );
  
  setMenu( mb );

  toolbar = new KFunctionToolBar( this );
  CHECK_PTR( toolbar );
  addToolBar( toolbar );
  showToolBar = TRUE;

  plotWin = new KPlotWindow( this );
  CHECK_PTR( plotWin );
  setView( plotWin, TRUE );
  // to show the current cursor position in the status bar
  connect( plotWin, SIGNAL( mouseAt( double, double ) ),
	   SLOT( mouseAt( double, double ) ) );
  // the toolbar connects a signal to this signal
  connect( this, SIGNAL( unzoom() ), plotWin, SLOT( unzoom() ) );
  connect( this, SIGNAL( addLabel() ), plotWin, SLOT( addLabel() ) );

  statusBar = new KStatusBar( this );
  CHECK_PTR( statusBar );
  setStatusBar( statusBar );
  showStatusBar = TRUE;
  statusBar->insertItem( "x = 0.000; y = 0.000", 1 );

  qpmOptions->setItemChecked( statusID, showStatusBar );
  qpmOptions->setItemChecked( toolID, showToolBar );
  qpmOptions->setItemChecked( deleteID, askOnDelete );

  show();
} // KPlotMainWindow constructor


void
KPlotMainWindow::overview()
{
  KOverviewDlg* dlg = new KOverviewDlg( this );
  CHECK_PTR( dlg );
  dlg->exec();
  delete dlg;
} // KPlotMainWindow::overview


void
KPlotMainWindow::mouseAt( double x, double y )
{
  static QString sX, sY;
  sX.setNum( x );
  sY.setNum( y );
  statusBar->changeItem( "x = " + sX + "; y = " + sY, 1 );
} // KPlotMainWindow::mouseAt


void
KPlotMainWindow::askDelete()
{
  askOnDelete = !askOnDelete;
  qpmOptions->setItemChecked( deleteID, askOnDelete );
} // KPlotMainWindow


void
KPlotMainWindow::toggleStatusBar()
{
  showStatusBar = !showStatusBar;
  if ( showStatusBar ) enableStatusBar( KStatusBar::Show );
  else enableStatusBar( KStatusBar::Hide );

  qpmOptions->setItemChecked( statusID, showStatusBar );
} // KPlotMainWindow::toggleStatusBar


void
KPlotMainWindow::toggleToolBar()
{
  showToolBar = !showToolBar;
  if ( showToolBar ) enableToolBar( KToolBar::Show, toolbar1 );
  else enableToolBar( KToolBar::Hide, toolbar1 );

  qpmOptions->setItemChecked( toolID, showToolBar );
} // KPlotMainWindow::toggleToolBar


void
KPlotMainWindow::paintEvent( QPaintEvent* e )
{
  KTopLevelWidget::paintEvent( e );
  plotWin->repaint( e->rect(), TRUE );
} // KPlowMainWindow::paintEvent


void
KPlotMainWindow::saveProperties( KConfig* c )
{
  KTopLevelWidget::saveProperties( c );

  QString s;

  s.setNum( minX );
  c->writeEntry( "minX", s );
  s.setNum( maxX );
  c->writeEntry( "maxX", s );
  s.setNum( minY );
  c->writeEntry( "minY", s );
  s.setNum( maxY );
  c->writeEntry( "maxY", s );

  c->writeEntry( "showGrid", showGrid );
  s.setNum( gridX );
  c->writeEntry( "gridX", s );
  s.setNum( gridY );
  c->writeEntry( "gridY", s );
  c->writeEntry( "gridColor", gridColor );
  c->writeEntry( "gridPenStyle", static_cast<int>( gridPenStyle ) );

  QListIterator<KFunction> it( *functions );

  // save the number of functions
  int n = it.count();
  c->writeEntry( "functionCount", n );

  // save the functions itself
  KFunction* f;
  int x = 0;  // The iterator doesn't provide the index of the current item,
              // so we have to count on our own
  QString sX; // will keep the current function number as a string

  QString sPrefixFunction = "function";   // key name for the function term
  QString sPrefixLineColor = "linecolor"; // key name for the line color
  QString sPrefixLineStyle = "linestyle"; // key name for the line style

  for( ; ( f = it.current() ); ++it ) {
    sX.setNum( x );

    // write the function term
    s = sPrefixFunction + sX;
    c->writeEntry( s, f->text );

    // the line color
    s = sPrefixLineColor + sX;
    c->writeEntry( s, f->pen.color() );

    // the line style
    s = sPrefixLineStyle + sX;
    c->writeEntry( s, static_cast<int>( f->pen.style() ) );

    // increment the counter
    x++;
  } // for


  QListIterator<KTextLabel> itLabels( Labels );
  // save the labels
  n = itLabels.count();
  c->writeEntry( "labelCount", n );
  KTextLabel* l;
  x = 0;
  QString sPrefixText = "labelText";
  QString sPrefixX = "labelX";
  QString sPrefixY = "labelY";
  QString sPrefixColor = "labelColor";
  QString sPrefixPos = "labelCrossPosition";

  for( ; ( l = itLabels.current() ); ++itLabels ) {
    sX.setNum( x );
    s = sPrefixText + sX;
    c->writeEntry( s, l->text );
    s = sPrefixX + sX;
    QString sTemp;
    sTemp.setNum( l->x );
    c->writeEntry( s, sTemp );
    s = sPrefixY + sX;
    sTemp.setNum( l->y );
    c->writeEntry( s, sTemp );
    s = sPrefixColor + sX;
    c->writeEntry( s, l->color );
    s = sPrefixPos + sX;
    c->writeEntry( s, static_cast<int>( l->showCross ) );

    // increment the counter
    x++;
  } // for

  c->writeEntry( "showStatusBar", showStatusBar ? "TRUE" : "FALSE" );
  c->writeEntry( "showToolBar", showToolBar ? "TRUE" : "FALSE" );
  c->writeEntry( "askOnDelete", askOnDelete ? "TRUE" : "FALSE" );
} // KPlotMainWindow::saveProperties


void
KPlotMainWindow::readProperties( KConfig* c )
{
#ifdef __DEBUG
  debug( "KPlotMainWindow::readProperties()" );
#endif

  KTopLevelWidget::readProperties( c );

  QString s;
  
  s = c->readEntry( "minX" );
  minX = s.toDouble();
  s = c->readEntry( "maxX" );
  maxX = s.toDouble();
  s = c->readEntry( "minY" );
  minY = s.toDouble();
  s = c->readEntry( "maxY" );
  maxY = s.toDouble();

  showGrid = c->readNumEntry( "showGrid" );
  s = c->readEntry( "gridX" );
  gridX = s.toDouble();
  s = c->readEntry( "gridY" );
  gridY = s.toDouble();
  gridColor= c->readColorEntry( "gridColor" );
  gridPenStyle = static_cast<PenStyle>( c->readNumEntry( "gridPenStyle" ) );

  // get the number of functions to read
  int n;
  n = c->readNumEntry( "functionCount" );

  // get the functions itself
  KFunction* f;
  int x = 0;  // counter for the function number
  QString sX; // x as a string

  QString sPrefixFunction = "function";   // key name for the function term
  QString sPrefixLineColor = "linecolor"; // key name for the line color
  QString sPrefixLineStyle = "linestyle"; // key name for the line style

  for ( x = 0; x < n; x++ ) {
    sX.setNum( x );

    f = new KFunction;
    CHECK_PTR( f );

    // read the function term
    s = sPrefixFunction + sX;
    f->text = c->readEntry( s );
    QStrList slist;
    slist.append( "x" );
    f->expression.parse( f->text, &slist );

#ifdef __DEBUG
  debug( "reading function %s", static_cast<const char*>( f->text ) );
#endif

    // read the line color
    s = sPrefixLineColor + sX;
    QColor col = c->readColorEntry( s );
    f->pen.setColor( col );

    // read the line style
    s = sPrefixLineStyle + sX;
    int nLineStyle = c->readNumEntry( s );
    f->pen.setStyle( static_cast<PenStyle>( nLineStyle ) );

    functions->append( f );
  } // for


  QString sPrefixText = "labelText";
  QString sPrefixX = "labelX";
  QString sPrefixY = "labelY";
  QString sPrefixColor = "labelColor";
  QString sPrefixPos = "labelCrossPosition";

  n = c->readNumEntry( "labelCount" );

  KTextLabel* l;
  x = 0;

  for ( x = 0; x < n; x++ ) {
    sX.setNum( x );

    l = new KTextLabel;
    CHECK_PTR( l );

    s = sPrefixText + sX;
    l->text = c->readEntry( s );
    s = sPrefixX + sX;
    QString sTemp = c->readEntry( s );
    l->x = sTemp.toDouble();
    s = sPrefixY + sX;
    sTemp = c->readEntry( s );
    l->y = sTemp.toDouble();
    s = sPrefixColor + sX;
    l->color = c->readColorEntry( s );
    s = sPrefixPos + sX;
    l->showCross = static_cast<KTextLabel::crossPos>( c->readNumEntry( s ) );

    Labels.append( l );
  } // for


  s = c->readEntry( "showStatusBar" );
  showStatusBar = ( s == "TRUE" );
  qpmOptions->setItemChecked( statusID, showStatusBar );

  s = c->readEntry( "showToolBar" );
  showToolBar = ( s == "TRUE" );
  qpmOptions->setItemChecked( toolID, showToolBar );

  s = c->readEntry( "askOnDelete" );
  askOnDelete = ( s == "TRUE" );
  qpmOptions->setItemChecked( deleteID, askOnDelete );
} // KPlotMainWindow::readProperties


void
KPlotMainWindow::keyReleaseEvent( QKeyEvent* e )
{
  kapp->sendEvent( plotWin, e );
  if( !e->isAccepted() ) KTopLevelWidget::keyReleaseEvent( e );
} // KPlotMainWindow::keyReleaseEvent


int
main( int argc, char** argv )
{
  dictConstants = new QDict<double>;
  CHECK_PTR( dictConstants );
  dictConstants->setAutoDelete( TRUE );

  lstfunctions = new QStrList;

  dictConstants->insert( "e", new double( M_E ) );
  dictConstants->insert( "pi", new double( M_PI ) );


  //  dictFunctions = new QDict<funcCall>;
  dictFunctions = new KPDict;
  CHECK_PTR( dictFunctions );
  dictFunctions->setAutoDelete( TRUE );
  
  funcCall* fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = sin;
  fc->description="x represents one angle in a right triangle\n (not the 90-degree angle).  sin(x) returns the ratio of\n the length of the leg opposite x to the\n hypotenuse.";

  fc->differentiateFunc = differentiate_sin;
  dictFunctions->insert( "sin", fc );
  
  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = cos;
  fc->differentiateFunc = differentiate_cos;
  dictFunctions->insert( "cos", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = exp;
  fc->differentiateFunc = differentiate_exp;
  dictFunctions->insert( "exp", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = sqrt;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "sqrt", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = fabs;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "abs", fc );

  // natural logarithm
  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = log;
  fc->differentiateFunc = differentiate_ln;
  dictFunctions->insert( "ln", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = tan;
  fc->differentiateFunc = differentiate_tan;
  dictFunctions->insert( "tan", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = asin;
  fc->differentiateFunc = differentiate_asin;
  dictFunctions->insert( "asin", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = acos;
  fc->differentiateFunc = differentiate_acos;
  dictFunctions->insert( "acos", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = atan;
  fc->differentiateFunc = differentiate_atan;
  dictFunctions->insert( "atan", fc );
 
  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = cosh;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "cosh", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = sinh;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "sinh", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = tanh;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "tanh", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = acosh;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "acosh", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = asinh;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "asinh", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = atanh;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "atanh", fc );

  // logarithm to the base of 10
  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = log10;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "lg", fc );

  // cube root [x^(1/3)]
  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = cbrt;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "cbrt", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = ceil;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "ceil", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = floor;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "floor", fc );

  // log( a, b ) logarithm of a to the base of b
  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 2;
  fc->twoParams = xlog;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "log", fc );

  fc = new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 1;
  fc->oneParam = log2;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "ld", fc );

  //add a function to plot (function) data from files
  //datafile2D (x, "filename.dat", options)
  
  fc= new funcCall;
  CHECK_PTR( fc );
  fc->numParameters = 2;
  fc->twoParams=datafile2D;
  fc->differentiateFunc = NULL;
  dictFunctions->insert( "datafile2D", fc);

  // create the functions list with the functions to plot
  functions = new QList<KFunction>;
  CHECK_PTR( functions );
  functions->setAutoDelete( TRUE );

  minX = -5; minY = -5; maxX = 5; maxY = 5;

  //We want this single global class to manage the data for all data files.
  //We can avoid sending non-number parameters to parse this way
  // and avoid reloading data when replotting.
  g_kadf = new KAllDF;

  gridX = 1;
  gridY = 1;
  showGrid = grid_Dots;
  gridPenStyle = SolidLine;
  gridColor = gray;

  KPlot* app = new KPlot( argc, argv );
  CHECK_PTR( app );

  // actually run the app
  int nReturn = app->exec();

  // clean up
  delete app;
  return nReturn;
} // main

