#ifndef _EXPRESSION_H
#define _EXPRESSION_H

#include <math.h>
#include <qpaintd.h>
#include <qdict.h>
#include <qstring.h>
#include <qstrlist.h>

#include "kfndialog.h"
#include "kpdict.h"
#include "funccall.h"

const int err_Ok                      =  0;
const int err_TooManyClosingBrackets  = -1;
const int err_ClosingBracketMissing   = -2;
const int err_funcCallUnknown         = -3;
const int err_wrongNumberOfParameters = -4;
const int err_constantUnknown         = -5;
const int err_notANumber              = -6;
const int err_expressionEmpty         = -7;
const int err_badDataFile             = -8;
const int err_noMem                   = -9;

class KExpression;
class KNode;
class KNodePlus;
class KNodeMinus;
class KNodeMult;
class KNodeDiv;
class KNodePow;
class KNodeVar;
class KNodeFuncCall;


int parse( QString expression, QStrList* slist, KNode*& exp );


class KNode
{
public:
  KNode* left, * right;
  KNode();
  virtual inline double getValue( QDict<double>* );
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString print( int lastLevel ) const;
  virtual QString type() const { return "KNode"; }
  virtual int level() const { return 0; }
  virtual bool isConst() const { return left->isConst() && right->isConst(); };
  virtual KNode* simplify();
  virtual ~KNode();
}; // class KNode


class KNodePlus : public KNode
{
public:
  virtual inline double getValue( QDict<double>* x ) 
    { return left->getValue( x ) + right->getValue( x ); };
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString type() const { return "+"; }
  virtual int level() const { return 1; }
  virtual KNode* simplify();
}; // class KNodePlus

class KNodeMinus : public KNode
{
public:
  virtual inline double getValue( QDict<double>* x ) 
    { return left->getValue( x ) - right->getValue( x ); };
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString type() const { return "-"; }
  virtual int level() const { return 1; }
  virtual KNode* simplify();
}; // class KNodeMinus

class KNodeMult : public KNode
{
public:
  virtual inline double getValue( QDict<double>* x ) 
    { return left->getValue( x ) * right->getValue( x ); };
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString type() const { return "*"; }
  virtual int level() const { return 2; }
  virtual bool isConst() const;
  virtual KNode* simplify();
}; // class KNodeMult

class KNodeDiv : public KNode
{
public:
  virtual inline double getValue( QDict<double>* x ) 
    { return left->getValue( x ) / right->getValue( x ); };
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString type() const { return "/"; }
  virtual int level() const { return 2; }
  virtual bool isConst() const;
  virtual KNode* simplify();
}; // class KNodeDiv

class KNodePow : public KNode
{
public:
  virtual inline double getValue( QDict<double>* x ) 
    { return pow( left->getValue( x ), right->getValue( x ) ); };
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString type() const { return "^"; }
  virtual int level() const { return 3; }
  virtual bool isConst() const;
  virtual KNode* simplify();
}; // class KNodePow

class KNodeValue : public KNode
{
public:
  KNodeValue( double val = 0.0 ) : KNode() { value = val; };
  virtual inline double getValue( QDict<double>* ) 
    { return value; };
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString print( int lastLevel ) const;
  virtual QString type() const { return "value"; }
  virtual int level() const { return 5; }
  virtual bool isConst() const { return TRUE; };

  double value;
}; // class KNodeValue

class KNodeVar : public KNode
{
public:
  KNodeVar( QString _sVarName );
  virtual inline double getValue( QDict<double>* x ) 
    { return *(*x)[ sVarName ]; }
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString print( int ) const { return sVarName; };
  virtual QString type() const { return "var"; }
  virtual int level() const { return 5; }
  virtual bool isConst() const { return FALSE; };

  QString sVarName;
}; // class KNodeVar


class KExpression
{
public:
  KExpression();
  KExpression( QString exp );
  KExpression( QString exp, QStrList* vars );
  KExpression( KNode* _expression ) { expression = _expression; }
 ~KExpression();

  int parse( QString exp, QStrList* slist, QWidget* messageParent = 0 );
  QString getErrorText( int nError );

  inline double getValue( QDict<double>* x ) 
    { return expression->getValue( x ); };

  void plot( QPaintDevice* pd, QPainter* p,
	     QPen pen, double minX, double minY, double maxX, double maxY );
  inline KExpression* differentiate( QString var ) const {
    KNode* n = expression->differentiate( var );
    if ( !n ) return static_cast<KExpression*>( NULL );
    return new KExpression( n );
  } // differentiate
  QString print() const { return expression->print( 0 ); };
  void simplify();

protected:
  KNode* expression;
}; // class KExpression


class userFuncCall : public funcCall {
public:
  userFuncCall() : funcCall() { userFunction = 0; }
  virtual ~userFuncCall() { if ( userFunction ) delete userFunction; };

  KExpression* userFunction;
  QStrList vars;

  virtual double getValue( QDict<double>* x, KNode** param );
  virtual KNode* differentiate( QString, const KNodeFuncCall* ) const {
    return static_cast<KNode*>( NULL ); };
}; // userFuncCall


extern QDict<double>*   dictConstants;
extern KPDict *dictFunctions;

class KNodeFuncCall : public KNode
{
public:
  KNodeFuncCall() : KNode() {
    param = new KNode*[ 3 ];
    int n; for( n = 0; n < 3; n++ ) param[ n ] = static_cast<KNode*>( NULL );
  } // KNodeFuncCall constructor
  virtual ~KNodeFuncCall() {
    int n; for( n = 0; n < 3; n++ ) if( param[ n ] ) delete param[ n ];
    delete param;
  } // KNodeFuncCall destructor

  virtual double getValue( QDict<double>* x );
  virtual KNode* copy();
  virtual KNode* differentiate( QString var );
  virtual QString print( int lastLevel ) const;
  virtual QString type() const { return "funccall"; }
  virtual int level() const { return 4; }
  virtual bool isConst() const;

  funcCall*  function;
  KNode** param;
}; // class KNodeFuncCall


#endif // _EXPRESSION_H
