/*
   fofx: function grapher for character-cell terminals
   Copyright (C) 1997 Tom Stepleton

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
   ParseCmdLine: parses the command line arguments
   DoEquation: recursive-descent parser, parses the equation
   Other stuff: DoEquation's helpers.
*/

/*
   Expected Syntax:
     am <args> <equation>
   Equation must be ONE argument
   Example:
     fofx -iv .1 -tlx -1 -tly 1 -brx 1 -bry -1 "asin (x)"
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "fofx.h"       
#include "fofx_err.h"

void ParseCmdLine(int argc, unsigned char *argv[])
{
  int i,j=0;
  double temp;
  unsigned char *temp_func_tok;

  progname = argv[0];

  if(argc==1) fatal_error(NO_EQN_ERROR,HELP_MESSAGE);

  i = argc;
  do {
    i--;
    if((!(strcmp("-?",argv[i]))) || (!(strcmp("--help",argv[i]))))
      fatal_error(NO_ERROR,HELP_MESSAGE);
    if((!(strcmp("-v",argv[i]))) || (!(strcmp("--version",argv[i]))))
      fatal_error(NO_ERROR,VERSION_MESSAGE);
  } while(i>1);

  /* Convert the function to lowercase, tokenize, and point to it with
    *function */

  function = all_lcase(argv[argc-1]);

  temp_func_tok = function_tokens;

  i=0;
  while(*temp_func_tok) {
    preparse(function_phrases[i],*temp_func_tok);
    temp_func_tok++;
    i++;
  }

  /* This could be buggy */

  iv = HUGE_VAL;

  for(i=0;i<=(argc-2);i++) {

    if(!(strcmp(argv[i],"-axes"))) {
      xaxis = yaxis = 1;
      continue;
    }

    if(!(strcmp(argv[i],"-cols"))) {
      if(i+1 < argc) {
        i++;

        while(argv[i][j]) {
          if(!(isdigit(argv[i][j])))
            fatal_error(COLS_ERROR,COLS_BAD_ERROR);
          j++;
        }

        cols = atoi(argv[i]);

        if(cols <= MIN_COLUMNS || cols >= MAX_COLUMNS)
          fatal_error(COLS_ERROR,RANGE_ERROR);
      }
      continue;
    }

    if(!(strcmp(argv[i],"-tlx"))) {
      if(i+1 < argc) {
        temp = atof(argv[++i]);
        if(temp >= brx) fatal_error(TLX_ERROR,RANGE_ERROR);
        tlx = temp;
      }
      continue;
    }

    if(!(strcmp(argv[i],"-tly"))) {
      if(i+1 < argc) {
        temp = atof(argv[++i]);
        if(temp <= bry) fatal_error(TLY_ERROR,RANGE_ERROR);
        tly = temp;
      }
      continue;
    }

    if(!(strcmp(argv[i],"-brx"))) {
      if(i+1 < argc) {
        temp = atof(argv[++i]);
        if(temp <= tlx) fatal_error(BRX_ERROR,RANGE_ERROR);
        brx = temp;
      }
      continue;
    }

    if(!(strcmp(argv[i],"-bry"))) {
      if(i+1 < argc) {
        temp = atof(argv[++i]);
        if(temp >= tly) fatal_error(BRY_ERROR,RANGE_ERROR);
        bry = temp;
      }
      continue;
    }

    if(!(strcmp(argv[i],"-iv"))) {
      if(i+1 < argc) {
        temp = atof(argv[++i]);
        if(temp <= 0) fatal_error(IV_ERROR,IV_BAD_ERROR);
        if(temp >= (brx - bry)) fatal_error(IV_ERROR,RANGE_ERROR);
                                     /* bug check */
        iv = temp;
      }
      continue;
    }
  }
  if(iv==HUGE_VAL) iv=(brx-tlx)/20;
}

double DoEquation(void)
{
  double the_answer;

  func_ptr = function;

  NextToken();
  if(!*token) fatal_error(EQN_ERROR,NO_EQN_ERROR);

  DoAddition(&the_answer);

  return(the_answer);
}

void DoAddition(double *answer)
{
  double temp;
  unsigned char operation;

  DoMultiplication(answer);
  while((operation = *token) == '+' || operation == '-') {
    NextToken();
    DoMultiplication(&temp);
    if(operation == '+') *answer = *answer + temp;
    else if(operation == '-') *answer = *answer - temp;    
  }
}

void DoMultiplication(double *answer)
{
  double temp;
  unsigned char operation;

  DoExponent(answer);
  while((operation = *token) == '*' || operation == '/' || operation == '%') {
    NextToken();
    DoExponent(&temp);
    if(operation == '*') *answer = *answer * temp;
    else if(operation == '/') *answer = *answer / temp;
    else if(operation == '%') *answer = (int) *answer % (int) temp;
  }
}

void DoExponent(double *answer)
{
  double temp;

  DoUnaries(answer);
  while(*token == '^') {
    NextToken();
    DoExponent(&temp);
    *answer = pow(*answer,temp);
  }
}

void DoUnaries(double *answer)
{
  unsigned char operation=0;

  if(isunary(operation = *token)) NextToken();
  DoParentheses(answer);
  switch(operation) {
    case '-':
      *answer = -(*answer);
      break;
    case 0x80:
      *answer = sin(*answer);
      break;
    case 0x81:
      *answer = cos(*answer);
      break;
    case 0x82:
      *answer = tan(*answer);
      break;
    case 0x83:
      *answer = asin(*answer);
      break;
    case 0x84:
      *answer = acos(*answer);
      break;
    case 0x85:
      *answer = atan(*answer);
      break;
    case 0x86:
      *answer = sinh(*answer);
      break;
    case 0x87:
      *answer = cosh(*answer);
      break;
    case 0x88:
      *answer = tanh(*answer);
      break;
    case 0x89:
      *answer = log10(*answer);
      break;
    case 0x8a:
      *answer = log(*answer);
      break;
    case 0x8b:
      *answer = sqrt(*answer);
      break;
    case 0x8c:
      *answer = fabs(*answer);
      break;
  }
}

void DoParentheses(double *answer)
{
  if(*token == '(') {
    NextToken();
    DoAddition(answer);
    if(*token != ')') {
      printf("Parentheses error!\n");
      exit(3478);
    }
    NextToken();
  } else GetNumber(answer);
}

void NextToken(void)
{
  unsigned char *token_ptr;
  int i=0;

  token_ptr = token;
  token_type = 0;

  *token = '\0';

  if(!*func_ptr) return;
  while(isspace(*func_ptr)) func_ptr++;

  if(isdelim(*func_ptr)) {
    *token_ptr++ = *func_ptr++;
    *token_ptr = '\0';
    return;
  }
  if(*func_ptr == 'x') {
    token_type = VARIABLE;
    *token_ptr++ = *func_ptr++;
    *token_ptr = '\0';
    return;
  }
  if(isdigit(*func_ptr) || *func_ptr == DECIMAL_SEPARATOR) {
    token_type = NUMBER;
    while(isdigit(*func_ptr) || *func_ptr == DECIMAL_SEPARATOR || *func_ptr == THOUSAND_SEPARATOR) {
      if(i>70) fatal_error(EQN_ERROR,LONG_CONSTANT_ERROR);
      if(!(*func_ptr == THOUSAND_SEPARATOR)) *token_ptr++ = *func_ptr++;
      i++;
    }
    *token_ptr = '\0';
    return;
  }
  fatal_error(EQN_ERROR,ALIEN_ERROR);
}

void GetNumber(double *answer)
{
  switch(token_type) {
    case NUMBER:
      *answer = atof(token);
      NextToken();
      return;
    case VARIABLE:
      *answer = Variable;
      NextToken();
      return;
    default:
      fatal_error(EQN_ERROR,BAD_TOKEN_ERROR);
  }
}

int isdelim(unsigned char thechar)
{
  unsigned char temp_str[80] = "+-*/%^() ";

  strcat(temp_str,function_tokens);

  if(strchr(temp_str,thechar)) return(1);
    else return(0);
}

int isunary(unsigned char thechar)
{
  unsigned char temp_str[80] = "+-";

  strcat(temp_str,function_tokens);

  if(strchr(temp_str,thechar)) return(1);
    else return(0);
}

