/*  pdnmesh - a 2D finite element solver
 *
    Copyright (C) 2001-2004 Sarod Yatawatta <sarod@users.sf.net>  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  $Id: types.h,v 1.51 2004/11/17 01:34:01 sarod Exp $
*/

#ifndef TYPES_H 
#define TYPES_H 1
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkgl.h>
#include <GL/gl.h>
#include <GL/glu.h>


#define ONE_OVER_TWELVE 0.0833333333333333333333f
#define ONE_OVER_SIX    0.166666666666666666667f
#define ONE_OVER_THREE 0.33333333333333333333f

#ifndef M_PI
#define M_PI  3.14159265358979323846  /* pi */
#endif /* M_PI */

#define TOL 0.000000001f
/*
 *            bitree.c 
 */
#define RESIZE 100 /* number to resize the mesh by realloc() */

typedef enum { RED, BLACK } nodecolour; /* red-black tree */
typedef enum { VAR, /* varible point */
				         FX, /* fixed */ 
             P_UNKNOWN /* unknown - initial imported point */ 
             } pointvalue; /* point type */
typedef enum { NU, /* Neumann edge */
               DR, /* Dirichlet boundary */
               E_UNKNOWN } etype;
/* coordinate precision */
#define MY_DOUBLE double
#ifndef MY_DOUBLE
#define MY_DOUBLE  float
#endif
/* format for MY_DOUBLE */
#define MDF "%lf"
#ifndef MY_INT
#define MY_INT int
#endif
#define MIF "%d"

#define DEFAULT_TITLE "pdnmesh"
#define DEFAULT_WIDTH 400
#define DEFAULT_HEIGHT 400

/* for the point tree */
typedef struct Node_x_ {
	struct Node_x_ * left;
	struct Node_x_ * right;
	struct Node_x_ * parent;
	struct Node_y_ * Ytree;
	nodecolour colour;

	/* user data */
	MY_DOUBLE x;  /* X coordinate */
} node_x;

typedef struct Node_y_ {
  struct Node_y_ * left;
  struct Node_y_ * right;
	struct Node_y_ * parent;

	unsigned int n; /* point number */
	nodecolour colour;

	/* user data */
	MY_DOUBLE *xp; /* pointer to X coordinate */
	MY_DOUBLE y; /* Y coordinate */
	MY_DOUBLE *z; /* potential, etc, etc. array, length=degree_of_freedom*/
	pointvalue val; /*  VAR or FX */
} node_y;

typedef struct Mesh_{
	node_x * Xtree;
	node_y ** Narray; /* we keep an array for fast access */
	unsigned int count;
	unsigned int maxpoints; /* current size of Narray */
} Mesh;

typedef struct point_ {
	MY_DOUBLE x;
	MY_DOUBLE y;
	MY_DOUBLE *z; /* potential, etc. etc. array, length=degree_of_freedom*/
	/* any other problem data */
	pointvalue val; /* fixed or variable? */
} point;

/*
 *  * lexer.l
 *   *
 *    */
/* parse equation in string */
extern int
equation_parse(char *string);



/*
 *  parser.y
 *
 */
/* data structures for the parsing part */
/* node types of the syntax tree */
typedef enum { CONST, VARI, OPR } symtype;
/* symbol types */

/* constants - double */
typedef struct {
	symtype type; /* type */
	MY_DOUBLE value; /* value */
} node_const;

/* variables - lower case letters */
typedef struct {
	symtype type; /* type */
	int d; /* identifier (ASCII) of variable to point to array symval[] */
} node_var;

/* operators */
typedef struct {
	symtype type; /* type */
	int operatr;  /* type of operator */
	int nops; /* no of operands */
	union exp_nodetype_* operand[1]; /* array of operands - similar nodes */
} node_opr;

typedef union exp_nodetype_ {
	symtype type; /* either const, var or operand */
	node_const constnt;
	node_var var;
	node_opr opr;
} exp_nodetype;


/* function to evaluate parse treee */
/* input are triangle coordinates and root pf parse tree */
extern MY_DOUBLE
ex( exp_nodetype *p, MY_INT p1 ); 

/*
 *            poly.c 
 */

/*******************/
/* edge */
typedef struct edge_ {
		int p1; /* point 1 */
		int p2; /* point 2 */
		/* more data */
		etype type; /* Dirichlet or Neumann edge */
} poly_edge;

/* doubly connected edge list for polygon */
typedef struct elist_{
     struct elist_* next;
     struct elist_* prev;
		 poly_edge data;
} poly_elist;


/* polygon */
typedef struct poly_{
		poly_elist* head;
		int nedges; /* no of edges */
		/* more data */
		int hollow; /* 0: hollow boundary- remove triangles */
		MY_DOUBLE mu; /* permeability */
		MY_DOUBLE rho; /* current density */
		/* we can use an expression for rho as well */
		/* however if rhoexp=0 we use the static value */
		exp_nodetype *rhoexp; /* expression for rho */
    MY_DOUBLE epsilon; /* permittivity (we keep this seperate from mu) */
} polygon;

/* all boundaries */
/* boundaries are closed polygons */
/* we can have open boundaries, but for them,
	* only the edges exist in edge array */
typedef struct bound_{
				polygon *poly_array; /* array of polygons */
				int npoly; /* number of polygons */
} boundary;

/* the following are for dxf.c */

typedef struct dxf_int_{
	  int n;
		struct dxf_int_ *next;
		struct dxf_int_ *prev;
} dxf_int;

/* dxf polygon: edges as integers */
typedef struct dxf_poly_{
		dxf_int* head; /* doubly linked list,sorted in number order */
		int nedges; /* no of edges */
		/* more data */
		int hollow; /* 1: hollow boundary- remove triangles */
		MY_DOUBLE mu; /* 1/permeability */
		MY_DOUBLE rho; /* current density */
		/* we can use an expression for rho as well */
} dxf_polygon;


/* list of polygons */
typedef struct poly_list_{
			dxf_polygon *data;
			struct poly_list_ *next;
			struct poly_list_ *prev;
			int number; /* 0 is the outer boundary */
} poly_list;

/* boundary as a list of polygons */
typedef struct bound_list_{
		 poly_list * head;
		 int count;
} boundary_list;

/*
 *           dag.c 
 */
#define DAG_STATUS_OK 1

typedef struct dag_node_ {
				struct dag_node_list_* list;
				void *rec; /* data record */
} DAG_node;

typedef struct dag_node_list_ {
				struct dag_node_list_ *next;
				struct dag_node_ *child;
} DAG_node_list; 

typedef struct dag_node_queue_ {
				struct dag_node_queue_ *next;
				struct dag_node_queue_ *prev;
				struct dag_node_ *child;
} DAG_node_queue; 

typedef struct dag_tree_ {
		/* function for testing inclusion */
		int (*comp_INC)(const void *rec1,const void *rec2);
		/* function for printing record data */
		void (*print_record)(const void *rec);


		/* root node */
		DAG_node *root;

		/* the following used for the queue of leaves */
		/* head of linked list of leaf nodes */
		DAG_node_queue *qhead;
		/* auxilliary poiter used to traverse the list */
		DAG_node_queue *qtail;
		/* sentinel in the queue */
		DAG_node_queue *sentinel;
} DAG_tree;

typedef enum {LIVE, HIDDEN,DEAD} triangle_status;
/* LIVE - leaf in dag, active triangle
   HIDDEN - leaf in dag, but belongs to outer plane, not plotted 
	 DEAD - internal node in dag
*/

typedef struct triag_ {
		MY_INT p0,p1,p2; /* points in ccw order */
		MY_INT n; /* unique triangle number */
    DAG_node *dag_p; /* pointer to host node in DAG */
		/* neighbouring triangles */
		int t0,t1,t2; /* t0->p0, t1->p2, t2->p2 */
		triangle_status status; /* status flag to indicate triangle still in mesh */
		/* boundary */
		int boundary;
} triangle;


/*
 *           rbt.c 
 */


/* colors */
typedef enum { rbtBLACK, rbtRED } RBT_node_color;

/* error codes */
#define RBT_STATUS_OK 0
#define RBT_STATUS_KEY_FOUND 1
#define RBT_STATUS_KEY_NOT_FOUND 2

/* generic tree node */
typedef struct node_ {
	struct node_* left;
	struct node_* right;
	struct node_* parent;
	RBT_node_color color;
	void *rec; /* record for data, including the rec for order */
} RBT_node_type;



/* generic tree */
typedef struct tree_ {
	/* function for comparison of recs == */
        int (*comp_EQ) (const void *rec1, const void *rec2);
	/* function for comparison of recs < */
        int (*comp_LT) (const void *rec1, const void *rec2);
         /* function to print node data */
	void (*print_record)(const void *rec);
	/* function to free memory used by records */
	  void (*free_record)(void *rec);


	/* root node */
	RBT_node_type *root;
} RBT_tree;




/*
 *            bitree.c 
 */

/* initialise the mesh */
extern void 
BIT_init(Mesh *Ms, int maxpoints);

/* the following will insert the given point into the mesh if is already is */
/* not there, else it will do nothing but in both cases, the point number will be returned */
extern unsigned int 
BIT_insert(Mesh *Ms, point p);

/* free the mesh */
void 
BIT_free(Mesh *Ms);

extern Mesh M;
extern int maxpoints;
/* macros for accessing nodes */
/* first to get x cordinate */
#define Mx(i)  *((M.Narray[(i)])->xp)
/* to get y coordinate */
#define My(i) (M.Narray[(i)])->y
/* to get potential */
#define Mz(i) (M.Narray[(i)])->z[0]
/* to get any value of z */
#define Mzz(i,j) (M.Narray[(i)])->z[(j)]

/* get fixed or variable property */
#define Mval(i) (M.Narray[(i)])->val
/* to get position relative to a boundary */
#define Mpos(i) (M.Narray[(i)])->pos


#define DETERM(x1,y1,x2,y2,x3,y3) \
   ((x1)*(y2-y3)+(x2)*(y3-y1)+(x3)*(y1-y2))

#ifndef ABS
#define ABS(x) \
					((x)>=0? (x):-(x))
#endif

#ifndef LENGTH
#define LENGTH(p1,p2) \
				((Mx(p1)-Mx(p2))*(Mx(p1)-Mx(p2))+(My(p1)-My(p2))*(My(p1)-My(p2)))
#endif

#ifndef LENGTH1
#define LENGTH1(p,x,y) \
				((Mx(p)-(x))*(Mx(p)-(x))+(My(p)-(y))*(My(p)-(y)))
#endif


#define IS_ZERO(x) \
     (ABS(x) <= TOL)

/* pdnmesh.c */

extern int degree_of_freedom; /* degree of freedom for a point */

extern int requested_degree_of_freedom; /* user request, for eigenvalue problems */

extern int max_iteration_limit; /* limit for iterative solution */

/* input cord file name */
extern char *  input_cord_filename;

extern int
insert_point_to_mesh(point p);

/* initial buildup of mesh */
extern int 
solve_problem_from_scratch(char *filename);

/* free all allocated variables */
extern void
free_everything(void);

extern void 
intialize_global_variables(int argc, char *argv[]);



/*********************/
/*
 *           input.c 
 */
/* globals */
extern poly_edge *edge_array;
extern int nedges;
extern boundary bound; /* boundaries */
extern MY_DOUBLE g_xoff,g_yoff; /* x,y offsets */
extern MY_DOUBLE g_xscale,g_yscale; /* x,y scales */
extern MY_DOUBLE g_xrange,g_yrange; /* x,y ranges, used for contour plot */

/* global no for current boundary */
extern MY_INT  current_boundary_number;

/* read input file */
extern int
read_input_file(point **input_points, char *infile);

/* inserts the expression parsed into boundary array */
/* given by bondary number 'boundary' */
extern int
insert_expression(exp_nodetype *p, MY_INT boundary_no);


/*
 *           poly.c 
 */

/* poly.c */
/* initialize a polygon */
extern void
init_poly(polygon *poly);
/* insert an edge to a polygon */
extern int
insert_edge_to_poly(polygon *poly,poly_edge *e);
/* check point is inside the polygon */
/* returns 1 if true */
/* note the points, and the edges has to be in ccw order */
extern int
inside_poly(polygon *poly,int p);
/* check if point p is inside polygon poly */
int
inside_poly_point(polygon *poly,point p);
/* sanity check polygon */
/* arrange edges in correct adjacent order */
extern void
arrange_polgon_edges(polygon *poly);

/* function to remove intersection of the triangles 
   with polygon edges */
extern int
remove_polygon_mesh_intersections(polygon *poly);

/* copies polygon a to b, returns edges copied */
extern int
copy_polygon(polygon *a,polygon *b);

/* destroy a polygon */
void
destroy_polygon(polygon *a);


/* routines to handle a list of polygons */
extern void
init_boundary_list(boundary_list *b);
/* insert a polygon */
extern int
insert_poly_to_boundary(boundary_list *b, dxf_polygon *p);
/* remove a polygon from list */
extern void
remove_poly_from_boundary(boundary_list *b, int number);
/* get polygon from list, given by number */
/* returns 0 if not found */
extern dxf_polygon *
lookup_poly_from_boundary(boundary_list *b, int number);

/* insert edge if it is not included in the polygon
 * else remove it from the polygon */
extern int
insert_remove_from_polygon(dxf_polygon *p,int edge_number);
/* sort all edges in polygons of the
 * boundaries so that they are adjacent */
extern int 
sort_edges_in_polygons(boundary_list *b);

/*
 *           dag.c 
 */

extern int
DAG_init(DAG_tree *t, int (*comp_INC)(const void *rec1,const void *rec2),
								void (*print_record)(const void *rec));
/* we return the dag node with piggyback data record */
/* because we need it for cross referencing */
extern void *
DAG_insert(DAG_tree *t, DAG_node *parent, void *rec);
/* returns node pointer where data is found */
/* assertion - nodes at same level cover disjoint regions */
/* nodes at higher level cover same region as nodes at lower level */
extern void *
DAG_find(DAG_tree *t, void *datarec);
/* traverse and print */
extern void 
DAG_traverse(DAG_tree *t);
/* add a cross link between two nodes */
extern void *
DAG_link(DAG_tree *t, DAG_node *parent, DAG_node *child);
/* traverse the leaf list and prune it */
/* that is, remove nodes that are not leaves */
/* returns the next leaf node data pointer */
extern void *
DAG_traverse_prune_list(DAG_tree *t);

/* reset queue for a new traversal */
extern void
DAG_traverse_list_reset(DAG_tree *t);

/* BIG NOTE: we do not free the record here
 * * it has to be freed using the RBT */
/* free DAG */
void
DAG_free(DAG_tree *t);


/* these are for reverse traversal, from tail to head */
extern void *
DAG_reverse_traverse_prune_list(DAG_tree *t);

extern void
DAG_reverse_traverse_list_reset(DAG_tree *t);

/*
 *           graphics.c 
 */
/* globals needed */
extern double zoom_x1,zoom_x2,zoom_y1,zoom_y2;
/* current transformation matrix */
extern GLdouble current_mat[16];
extern int ntriangles,mouse_responce_flag,windowid;
extern DAG_tree dt;
extern triangle *chosen_triangle;
extern GtkWidget *global_status_bar,*global_status_label;


/* global to update status bar */
extern void
update_status_bar(char *msg);

/* option flags for plotting */
extern int plot_mesh, plot_cont, plot_fill, plot_grad, plot_legend;

/* menu items */
#define MENU_0 0
#define MENU_FULL 1
#define MENU_ORIG 2
#define MENU_EXIT 3
#define MENU_INFO 4
#define MENU_ZOOM_START 5
#define MENU_ZOOM_END 6
#define MENU_ZOOM_ALL 7
#define MENU_ZOOM_BACK 8
#define MENU_SPLIT 9
/* the following to modify point/edge properties */
#define MENU_EDIT_POINT 10
#define MENU_EDIT_EDGE 11
#define MENU_ADD_EDGE 12
/* followind to spint triangles */
#define MENU_SPLIT_TRIANGLE 13

/* flags for plotting */
#define PLOT_MESH 0
#define PLOT_CONT 1
#define PLOT_GRAD 2
#define PLOT_FILL 3

/* flags for drawing modes */
#define DRAW_OUTPUT 0
#define DRAW_INPUT 1
#define DRAW_DXF 3

/* macro to update gui under long computations */
#define UPDATE_GUI\
 while(gtk_events_pending())\
		gtk_main_iteration();

/*
 *           graphics.c 
 */
/* main window creation */
extern GtkWidget*
create_window(GdkGLConfig *glconfig);

/* openGL framebuffer configuratiion */
extern GdkGLConfig *
configure_gl(void);


/*
 *           rbt.c 
 */

extern RBT_tree rt;
/* intialize tree */
/* returns RBT_STATUS_OK if no error */
extern int
RBT_init(RBT_tree *t, int (*comp_EQ) (const void *rec1, const void *rec2),
		int (*comp_LT) (const void *rec1, const void *rec2),
	        void (*print_record) (const void *rec),
				 void (*free_record)(void *rec));

/* insert */
/* returns RBT_STATUS_OK if inserted, RBT_STATUS_KEY_FOUND if key exists */
extern int 
RBT_insert(void *rec, RBT_tree *t);
/* delete */
/* returns RBT_STATUS_OK if key deleted, RBT_STATUS_KEY_NOT_FOUND if not found */
extern int 
RBT_delete(void * rec,RBT_tree *t);
/* search */
/* returns RBT_STATUS_KEY_FOUND if found, RBT_STATUS_KEY_NOT_FOUND if key not found*/
extern void *
RBT_find(void *rec, RBT_tree *t);
/* traverse */
extern void
RBT_traverse(RBT_tree *t);

/* free memory used by a  tree */
/* makes t=NIL */
void
RBT_free(RBT_tree *t);


/* 
	* subdivide.c 
	*/
/* subdivision of a triangle */
extern int
subdivide_this_triangle(triangle *tg);

/* subdivides all live triangles */
extern int
subdivide_all_triangles(void);

extern int
refine_mesh_with_iterations(MY_DOUBLE priority);
extern int
refine_this_triangle(triangle *tg, int override, MY_DOUBLE priority);

/* parameters for modifying the mesh */
extern MY_DOUBLE g_badness_limit;
extern MY_DOUBLE g_area_floor;
extern MY_DOUBLE g_area_ceil;
extern MY_DOUBLE g_gradient_limit;

/* 
 * solve.c 
 */
typedef enum { POISSON, POISSON_SPARSE, HELMHOLTZ, HELMHOLTZ_INHOMO, HELMHOLTZ_FREQ, HELMHOLTZ_BETA } eq_type; /* type of equation to solve */
/* HELMHOLTZ: homogeneous wave eqn. HELMHOLTZ_INHOMO: inhomogeneous
  HELMHOLTZ_FREQ: given beta, find cutoff 
  HELMHOLTZ_BETA: given freq. find beta */
extern MY_DOUBLE g_maxpot,g_minpot;

/*extern MY_INT *renumber,*re_renumber; */

extern int 
re_number_and_solve_helmholtz(void);

extern int 
re_number_and_solve_poisson(void);


/* 
 * contour.c 
 */
/* function to plot the contour */
extern void 
plot_contour_all(void);
extern void 
plot_contour_all_in_3d(void);
/* printing contour plot */
void
print_contour_all(const char* filename,int color_flag);
/* printing mesh as eps*/
extern void
print_mesh_eps(const char* filename);
/* printing mesh as an ASCII file */
extern void
print_mesh_ascii(const char* filename);
/* printing potentials as ACII file*/
extern void
print_potential(const char* filename);
/* plotting gradient */
extern void
plot_gradient(void);
/* plotting legend */
extern void
display_legend_window(void);
/* a function to print the gradient of potential */
extern void
print_gradient(const char *filename);

/* to get a color to plot */
extern int 
get_colour(float* redp, float* greenp, float *bluep, int level, int max_levels);

extern int current_plotting_contour; /* number in z array */
extern int contour_levels; /* no of contour levels */
/*
 * eig.c
 */
/* inverse iteration for finding eigenvectors
 * given the eigenvalues */
#define EIGENVECTOR_ITERATION_LIMIT 5
/* type of equation POISSON, HELMHOLTZ */
extern eq_type solve_equation;
extern MY_DOUBLE *eig_array; /* array to store solved eigenvalues */
/* find the eigenvector(s) given (an) eigenvalue(s) */
extern void
find_eigenvectors(MY_DOUBLE **a, MY_DOUBLE **x, MY_DOUBLE *ev,  MY_INT m, MY_INT n);


/* 
 * dxf.c 
 */
/* read DXF file */
extern int
read_dxf_file_and_triangulate(const char *filename);

/* display edit window */
extern void
display_edit_window(const char* title);

/* generic dialog function */
extern void
generic_dialog(const gchar *msg,const gchar *title);

/* array of poins in the DXF file */
extern point * dxf_point_array;
extern int dxf_points;


/* 
 * output.c
 */
extern void
generate_output_files(void);


/* 
 * eig_lapack.c 
 */

/* solves the symmetrix, generalized eigenvalue problem 
* (P-lambda. T)x= 0
* P, T = N by N matrices, only lower triangle
* x = eigenvectors to be found size (N x M)
* using LAPACK 
*/
#ifdef USE_LAPACK
/* symmetric definite eigensolver */
extern void
solve_helmholtz_lapack(MY_DOUBLE **P,MY_DOUBLE **T, MY_DOUBLE **x,  MY_DOUBLE *ev, MY_INT  N, MY_INT G);
/* symmetric non definite eigensolver */
extern void
solve_generalized_eig_lapack(MY_DOUBLE **P,MY_DOUBLE **T, MY_DOUBLE **x, MY_DOUBLE *ev, MY_INT  N, MY_INT G);
#endif /* !USE_LAPACK */



/*
 *  hash.c 
 */
/* cutoff frequency and propagation constant used 
  in waveguide problems */
extern MY_DOUBLE global_wave_k0, global_wave_beta;

extern int
buildup_hash_table_and_equation(void);

/*
 * integrate.c
 */
/* the following function was sent by Werner Hoch */
/* calc the integral seperate for each dirichlet edge */
/* input int con - eigenmode of potential */
extern MY_DOUBLE
calc_integral(int con);

/* 
 * spmat.h
 */
/* sparse matrix routines */
typedef struct SPMat_ {
   int N; /* size is N by N */
   RBT_tree **rt;
} SPMat;

/* intialize a matrix */
extern int
SPM_init( SPMat *sM, int N ); /* N - N by N sparse matrix M */

/* add z to i,j location of M */
/* initially assumed zero */
/* i,j has to be within [0,N-1] */
extern int
SPM_add( SPMat *sM, int i, int j,  MY_DOUBLE z);

/* myltiply i,j location of M  by z */
/* initially assumed zero */
/* i,j has to be within [0,N-1] */
extern int
SPM_mult( SPMat *sM, int i, int j,  MY_DOUBLE z);

/* returns size of M, or value of N */
extern int
SPM_dim(SPMat *sM);

/* delete matrix M */
extern int
SPM_destroy(SPMat *sM);


/* copy row i of M to vector v */
/* v size N by 1 */
extern int 
SPM_row(SPMat *sM, MY_INT i, MY_DOUBLE *v);

/* copy column i of M to vector v */
/* v size N by 1 */
/* note column access is more expensive than row access */
extern int 
SPM_col(SPMat *sM, MY_INT i, MY_DOUBLE *v);

/* returns the value at location (i,j)
 of the matrix or zero if none found */
extern MY_DOUBLE 
SPM_e(SPMat *sM, MY_INT i, MY_INT j);

/* replaces the value at location (i,j)
 of the matrix with z */
extern int
SPM_eq(SPMat *sM, MY_INT i, MY_INT j,MY_DOUBLE z);

/* sparse solver using conjugate gradients for
 Poisson equation */
extern int 
re_number_and_solve_poisson_sparse(void);

#endif /* ! TYPES_H*/
